/*
 * Grace - Graphics for Exploratory Data Analysis
 * 
 * Home page: http://plasma-gate.weizmann.ac.il/Grace/
 * 
 * Copyright (c) 1991-95 Paul J Turner, Portland, OR
 * Copyright (c) 1996-98 GRACE Development Team
 * 
 * Maintained by Evgeny Stambulchik <fnevgeny@plasma-gate.weizmann.ac.il>
 * 
 * 
 *                           All Rights Reserved
 * 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 * 
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 * 
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * Contents:
 *     arrange graphs popup
 *     overlay graphs popup
 *     autoscaling popup
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>

#include <Xm/Xm.h>
#include <Xm/BulletinB.h>
#include <Xm/DialogS.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/Separator.h>
#include <Xm/Text.h>

#include "globals.h"
#include "protos.h"
#include "motifinc.h"


static Widget arrange_frame;
static Widget arrange_panel;

static Widget overlay_frame;
static Widget overlay_panel;

/*
 * Panel item declarations
 */

static Widget *arrange_rows_item;
static Widget *arrange_cols_item;
static Widget arrange_vgap_item;
static Widget arrange_hgap_item;
static Widget arrange_startx_item;
static Widget arrange_starty_item;
static Widget arrange_widthx_item;
static Widget arrange_widthy_item;
static Widget *arrange_packed_item;

static Widget *graph_overlay1_choice_item;
static Widget *graph_overlay2_choice_item;
static Widget *graph_overlaytype_item;

static Widget but1[2];

int maxmajorticks = 500;
int maxminorticks = 1000;

static void define_arrange_proc(Widget w, XtPointer client_data, XtPointer call_data);
static void define_overlay_proc(Widget w, XtPointer client_data, XtPointer call_data);
static void define_autos_proc(Widget w, XtPointer client_data, XtPointer call_data);


/*
 * Arrange graphs popup routines
 */
static void define_arrange_proc(Widget w, XtPointer client_data, XtPointer call_data)
{
    int nrows, ncols, pack;
    double vgap, hgap, sx, sy, wx, wy;

    nrows = GetChoice(arrange_rows_item) + 1;
    ncols = GetChoice(arrange_cols_item) + 1;
    if (nrows == 1 && ncols == 1) {
	errwin("For a single graph use View/Viewport...");
	return;
    }
    pack = GetChoice(arrange_packed_item);
	xv_evalexpr(arrange_vgap_item, &vgap);
	xv_evalexpr(arrange_hgap_item, &hgap);
	xv_evalexpr(arrange_startx_item, &sx);
	xv_evalexpr(arrange_starty_item, &sy);
	xv_evalexpr(arrange_widthx_item, &wx);
	xv_evalexpr(arrange_widthy_item, &wy);
    if (wx <= 0.0) {
	errwin("Graph width must be > 0.0");
	return;
    }
    if (wy <= 0.0) {
	errwin("Graph height must be > 0.0");
	return;
    }
    define_arrange(nrows, ncols, pack, vgap, hgap, sx, sy, wx, wy);
}

void update_arrange(void)
{
    xv_setstr(arrange_vgap_item, "0.05");
    xv_setstr(arrange_hgap_item, "0.05");
    xv_setstr(arrange_startx_item, "0.1");
    xv_setstr(arrange_starty_item, "0.1");
    xv_setstr(arrange_widthx_item, "0.8");
    xv_setstr(arrange_widthy_item, "0.8");
}

void row_arrange_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
    int nrow = (int)client_data,pack;
    double width, vgap, starty;

    xv_evalexpr(arrange_starty_item, &starty);
    if ( (pack = GetChoice(arrange_packed_item)) == 2 || pack == 3 ) {
        vgap = 0.;
    } else {
        xv_evalexpr(arrange_vgap_item, &vgap);
    }
    width = (1. - 2*starty - (nrow-1)*vgap)/nrow;
    sprintf( buf, "%g", width );
    xv_setstr(arrange_widthy_item, buf );
}

void col_arrange_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
    int ncol = (int)client_data, pack;
    double height, hgap, startx;

    xv_evalexpr(arrange_startx_item, &startx);
    if ( (pack = GetChoice(arrange_packed_item)) == 1 || pack == 3 ) {
        hgap = 0.;
    } else {
        xv_evalexpr(arrange_hgap_item, &hgap);
    }
    height = (1. - 2*startx - (ncol-1)*hgap)/ncol;
    sprintf( buf, "%g", height );
    xv_setstr(arrange_widthx_item, buf );
}


void create_arrange_frame(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget rc;
    int i;
   	set_wait_cursor();
    if (arrange_frame == NULL) {
	char *label1[2];
	label1[0] = "Accept";
	label1[1] = "Close";
	arrange_frame = XmCreateDialogShell(app_shell, "Arrange graphs", NULL, 0);
	handle_close(arrange_frame);
	arrange_panel = XmCreateRowColumn(arrange_frame, "arrange_rc", NULL, 0);

	rc = XtVaCreateWidget("rc", xmRowColumnWidgetClass, arrange_panel,
			      XmNpacking, XmPACK_COLUMN,
			      XmNnumColumns, 9,	/* nitems / 2 */
			      XmNorientation, XmHORIZONTAL,
			      XmNisAligned, True,
			      XmNadjustLast, False,
			      XmNentryAlignment, XmALIGNMENT_END,
			      NULL);

	XtVaCreateManagedWidget("Rows: ", xmLabelWidgetClass, rc, NULL);
	arrange_rows_item = CreatePanelChoice(rc, " ",
					      11,
			  "1", "2", "3", "4", "5", "6", "7", "8", "9", "10",
					      NULL, NULL);
	for( i=2; i<12; i++ )
	XtAddCallback(arrange_rows_item[i], XmNactivateCallback, 
			(XtCallbackProc) row_arrange_cb, (XtPointer) (i-1));
			
	XtVaCreateManagedWidget("Columns: ", xmLabelWidgetClass, rc, NULL);
	arrange_cols_item = CreatePanelChoice(rc, " ",
					      11,
			  "1", "2", "3", "4", "5", "6", "7", "8", "9", "10",
					      NULL, NULL);
	for( i=2; i<12; i++ )
	XtAddCallback(arrange_cols_item[i], XmNactivateCallback, 
			(XtCallbackProc) col_arrange_cb, (XtPointer) (i-1));

	XtVaCreateManagedWidget("Packing: ", xmLabelWidgetClass, rc, NULL);
	arrange_packed_item = CreatePanelChoice(rc, " ",
						5,
				   "None", "Horizontal", "Vertical", "Both",
						NULL, NULL);

	arrange_vgap_item = CreateTextItem4(rc, 10, "Vertical gap:");
	arrange_hgap_item = CreateTextItem4(rc, 10, "Horizontal gap:");
	arrange_startx_item = CreateTextItem4(rc, 10, "Start at X =");
	arrange_starty_item = CreateTextItem4(rc, 10, "Start at Y =");
	arrange_widthx_item = CreateTextItem4(rc, 10, "Graph width:");
	arrange_widthy_item = CreateTextItem4(rc, 10, "Graph height:");
	XtManageChild(rc);

	XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, arrange_panel, NULL);

	CreateCommandButtons(arrange_panel, 2, but1, label1);
	XtAddCallback(but1[0], XmNactivateCallback, (XtCallbackProc) define_arrange_proc, (XtPointer) NULL);
	XtAddCallback(but1[1], XmNactivateCallback, (XtCallbackProc) destroy_dialog, (XtPointer) arrange_frame);

	XtManageChild(arrange_panel);
	update_arrange();
    }
/*	update_arrange(); */
    XtRaise(arrange_frame);
    unset_wait_cursor();
}

/*
 * Overlay graphs popup routines
 */
static void define_overlay_proc(Widget w, XtPointer client_data, XtPointer call_data)
{
    int i;
    int g1 = GetChoice(graph_overlay1_choice_item);
    int g2 = GetChoice(graph_overlay2_choice_item);
    int type = GetChoice(graph_overlaytype_item);
    if (g1 == g2) {
	errwin("Can't overlay a graph onto itself");
	return;
    }
    if (!isactive_graph(g1)) {
	set_graph_active(g1);
    }
    if (!isactive_graph(g2)) {
	set_graph_active(g2);
    }
    /* set identical viewports g2 is the controlling graph */
    g[g1].v = g[g2].v;
    switch (type) {
    case 0:
	g[g1].w = g[g2].w;
	for (i = 0; i < MAXAXES; i++) {
/*
	    g[g1].t[i] = g[g2].t[i];
*/
	    g[g1].t[i].active = FALSE;
	    g[g2].t[i].active = TRUE;
	}
	g[g1].t[1].tl_op = PLACE_LEFT;
	g[g2].t[1].tl_op = PLACE_LEFT;
	g[g1].t[1].t_op = PLACE_BOTH;
	g[g2].t[1].t_op = PLACE_BOTH;

	g[g1].t[0].tl_op = PLACE_BOTTOM;
	g[g2].t[0].tl_op = PLACE_BOTTOM;
	g[g1].t[0].t_op = PLACE_BOTH;
	g[g2].t[0].t_op = PLACE_BOTH;
	break;
    case 1:
	g[g1].w.xg1 = g[g2].w.xg1;
	g[g1].w.xg2 = g[g2].w.xg2;
	for (i = 0; i < MAXAXES; i++) {
	    if (i % 2 == 0) {
		g[g1].t[i].active = FALSE;
	    } else {
		g[g1].t[i].active = TRUE;
	    }
	}
	g[g2].t[1].tl_op = PLACE_LEFT;
	g[g1].t[1].tl_op = PLACE_RIGHT;
	g[g2].t[1].t_op = PLACE_LEFT;
	g[g1].t[1].t_op = PLACE_RIGHT;

	g[g2].t[0].tl_op = PLACE_BOTTOM;
	g[g1].t[0].tl_op = PLACE_BOTTOM;
	g[g2].t[0].t_op = PLACE_BOTH;
	g[g1].t[0].t_op = PLACE_BOTH;

	break;
    case 2:
	g[g1].w.yg1 = g[g2].w.yg1;
	g[g1].w.yg2 = g[g2].w.yg2;
	for (i = 0; i < MAXAXES; i++) {
	    if (i % 2 == 1) {
		g[g1].t[i].active = FALSE;
	    } else {
		g[g1].t[i].active = TRUE;
	    }
/*
	    g[g1].t[i] = g[g2].t[i];
*/
	}
	g[g2].t[0].tl_op = PLACE_BOTTOM;
	g[g1].t[0].tl_op = PLACE_TOP;
	g[g2].t[0].t_op = PLACE_BOTTOM;
	g[g1].t[0].t_op = PLACE_TOP;

	g[g2].t[1].tl_op = PLACE_LEFT;
	g[g1].t[1].tl_op = PLACE_LEFT;
	g[g2].t[1].t_op = PLACE_BOTH;
	g[g1].t[1].t_op = PLACE_BOTH;
	break;
    case 3:
	for (i = 0; i < MAXAXES; i++) {
	    g[g1].t[i].active = TRUE;
	    g[g2].t[i].active = TRUE;
	}
	g[g2].t[1].tl_op = PLACE_LEFT;
	g[g1].t[1].tl_op = PLACE_RIGHT;
	g[g2].t[0].tl_op = PLACE_BOTTOM;
	g[g1].t[0].tl_op = PLACE_TOP;
	g[g2].t[1].t_op = PLACE_LEFT;
	g[g1].t[1].t_op = PLACE_RIGHT;
	g[g2].t[0].t_op = PLACE_BOTTOM;
	g[g1].t[0].t_op = PLACE_TOP;
	break;
    }
    update_all(cg);
    drawgraph();
    set_dirtystate();
}

void create_overlay_frame(Widget w, XtPointer client_data, XtPointer call_data)
{
    set_wait_cursor();
    if (overlay_frame == NULL) {
	char *label1[2];
	label1[0] = "Accept";
	label1[1] = "Close";
	overlay_frame = XmCreateDialogShell(app_shell, "Overlay graphs", NULL, 0);
	handle_close(overlay_frame);
	overlay_panel = XmCreateRowColumn(overlay_frame, "overlay_rc", NULL, 0);
	graph_overlay1_choice_item = CreateGraphChoice(overlay_panel, "Overlay graph: ", maxgraph, 0);
	graph_overlay2_choice_item = CreateGraphChoice(overlay_panel, "Onto graph: ", maxgraph, 0);
	graph_overlaytype_item = CreatePanelChoice(overlay_panel, "Overlay type:",
						   5,
					  "Same axes scaling along X and Y",
					   "X-axes same, Y-axes different:",
					   "Y-axes same, X-axes different:",
						   "X and Y axes different:",
						   NULL, NULL);

	XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, overlay_panel, NULL);

	CreateCommandButtons(overlay_panel, 2, but1, label1);
	XtAddCallback(but1[0], XmNactivateCallback, (XtCallbackProc) define_overlay_proc, (XtPointer) NULL);
	XtAddCallback(but1[1], XmNactivateCallback, (XtCallbackProc) destroy_dialog, (XtPointer) overlay_frame);

	XtManageChild(overlay_panel);
    }
/*	update_overlay(); */
    XtRaise(overlay_frame);
    unset_wait_cursor();
}

/*
 * autoscale popup
 */
typedef struct _Auto_ui {
    Widget top;
    SetChoiceItem sel;
    Widget *on_item;
    Widget *method_item;
    Widget *nticksx_item;
    Widget *nticksy_item;
    Widget *applyto_item;
} Auto_ui;

static Auto_ui aui;

static void define_autos_proc(Widget w, XtPointer client_data, XtPointer call_data)
{
    int aon, ameth, antx, anty, au, ap;
    Auto_ui *ui = (Auto_ui *) client_data;
    aon = GetChoice(ui->on_item) - 4;
    au = GetSelectedSet(ui->sel);
    if (au == SET_SELECT_ERROR) {
        errwin("No set selected");
        return;
    }
    if (au == SET_SELECT_ALL) {
	au = -1;
    } 
    ap = GetChoice(ui->applyto_item);
    ameth = GetChoice(ui->method_item);
    antx = GetChoice(ui->nticksx_item);
    anty = GetChoice(ui->nticksy_item);
    define_autos(aon, au, ap, ameth, antx, anty);
}

void update_autos(int gno)
{
    if (aui.top) {
	SetChoice(aui.method_item, g[gno].auto_type == TYPE_SPEC);
	SetChoice(aui.nticksx_item, g[gno].t[0].t_autonum - 2);
	SetChoice(aui.nticksy_item, g[gno].t[1].t_autonum - 2);
    }
}

void create_autos_frame(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget panel, rc;

    set_wait_cursor();
    if (aui.top == NULL) {
	char *label1[2];
	label1[0] = "Accept";
	label1[1] = "Close";
	aui.top = XmCreateDialogShell(app_shell, "Autoscale graphs", NULL, 0);
	handle_close(aui.top);
	panel = XmCreateRowColumn(aui.top, "autos_rc", NULL, 0);

	rc = XtVaCreateWidget("rc", xmRowColumnWidgetClass, panel,
			      XmNpacking, XmPACK_COLUMN,
			      XmNnumColumns, 5,
			      XmNorientation, XmHORIZONTAL,
			      XmNisAligned, True,
			      XmNadjustLast, False,
			      XmNentryAlignment, XmALIGNMENT_END,
			      NULL);
	XtVaCreateManagedWidget("Autoscale axis:", xmLabelWidgetClass, rc, NULL);
	aui.on_item = CreatePanelChoice(rc, " ",
					 9,
				  	 "None",
				  	 "All",
				  	 "All X-axes",
				  	 "All Y-axes",
				  	 "X-axis",
				  	 "Y-axis",
				  	 "Zero X-axis",
				  	 "Zero Y-axis",
				  	 NULL,
				  	 NULL);

	XtVaCreateManagedWidget("Autoscale type:", xmLabelWidgetClass, rc, NULL);
	aui.method_item = CreatePanelChoice(rc, " ",
					    3,
					    "Auto",
					    "Fixed",
					    NULL,
					    NULL);
	XtVaCreateManagedWidget("Number of ticks in X:", xmLabelWidgetClass, rc, NULL);
	aui.nticksx_item = CreatePanelChoice(rc, " ",
					       12,
		   "2", "3", "4", "5", "6", "7", "8", "9", "10", "11", "12",
					       NULL,
					       NULL);
	XtVaCreateManagedWidget("Number of ticks in Y:", xmLabelWidgetClass, rc, NULL);
	aui.nticksy_item = CreatePanelChoice(rc, " ",
					       12,
		   "2", "3", "4", "5", "6", "7", "8", "9", "10", "11", "12",
					       NULL,
					       NULL);
	XtVaCreateManagedWidget("Apply to:", xmLabelWidgetClass, rc, NULL);
	aui.applyto_item = CreatePanelChoice(rc, " ",
					       3,
					       "Current graph",
					       "All active graphs",
					       NULL,
					       NULL);
	XtManageChild(rc);

	aui.sel = CreateSetSelector(panel, "Use set:",
                                    SET_SELECT_ALL,
                                    FILTER_SELECT_NONE,
                                    GRAPH_SELECT_CURRENT,
                                    SELECTION_TYPE_MULTIPLE);

	XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, panel, NULL);

	CreateCommandButtons(panel, 2, but1, label1);
	XtAddCallback(but1[0], XmNactivateCallback, 
	              (XtCallbackProc) define_autos_proc, (XtPointer) &aui);
	XtAddCallback(but1[1], XmNactivateCallback,
	              (XtCallbackProc) destroy_dialog, (XtPointer) aui.top);

	XtManageChild(panel);
    }
    XtRaise(aui.top);
    update_autos(cg);
    unset_wait_cursor();
}

void define_autos(int aon, int au, int ap, int ameth, int antx, int anty)
{
    int i, ming, maxg;

    if (au >= 0 && !isactive_set(cg, au)) {
	errmsg("Set not active");
	return;
    }
    if (ap) {
	ming = 0;
	maxg = maxgraph - 1;
    } else {
	ming = cg;
	maxg = cg;
    }
    if (ming == cg && maxg == cg) {
	if (!isactive_graph(cg)) {
	    errmsg("Current graph is not active!");
	    return;
	}
    }
    for (i = ming; i <= maxg; i++) {
	if (isactive_graph(i)) {
	    if (ameth == 0) {
		g[i].auto_type = TYPE_AUTO;
	    } else {
		g[i].auto_type = TYPE_SPEC;
	    }
	    g[i].t[0].t_autonum = antx + 2;
	    g[i].t[1].t_autonum = anty + 2;
	    if (aon > -4) {	/* -4 == just set parameters but don't
				 * autoscale */
		if (au >= 0 && isactive_set(i, au)) {
		    autoscale_set(i, au, aon);
		} else {
		    autoscale_graph(i, aon);
		}
	    }
	}
    }
    if (aon != -4) {
	drawgraph();
    }
}

