;;; mew-refile.el --- Refile for Mew

;; Author:  Yoshinari NOMURA <nom@csce.kyushu-u.ac.jp>
;;          Kazu Yamamoto <Kazu@Mew.org>
;; Created: Jun 11, 1994
;; Revised: Sep  1, 1998

;;; Code:

(defconst mew-refile-version "mew-refile.el version 0.65")

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Variables
;;;

(defvar mew-refile-msgid-alist nil)
(defvar mew-refile-msgid-file-name ".mew-refile-msgid-alist")
(defvar mew-refile-from-alist nil)
(defvar mew-refile-from-file-name ".mew-refile-from-alist")

(defvar mew-refile-alist-max-length 1000
  "*Max length of mew-refile-from-alist and mew-refile-msgid-alist.")

(defvar mew-refile-last-folder nil
  "Folder name previously you refiled")

(defvar mew-refile-ctrl-multi t
  "*If *non-nil*, guess functions guess multi folders.")

(defvar mew-refile-ctrl-throw nil
  "*If *non-nil*, guess function doesn't guess any more if some other
functions were successful. This variable is set in guess control
functions. So this is not a user customize variable.")

(defvar mew-refile-ctrl-auto-boundary nil
  "*If *non-nil*, auto refile function stops at this point. This variable
is set in guess control functions. So this is not a user customize
variable.")

(defvar mew-refile-guess-alist nil
  "*If non-nil, mew guesses destination folder by using this hint.
The format is like this:

    (setq mew-refile-guess-alist
          '((\"To:\" 
              (\"wide@wide\" . \"+wide/wide\")
              (\"adam\"      . \"+labo/adam\"))
            (\"Newsgroups:\"
              (\"^nifty\\\\.\\\\([^ ]+\\\\)\" . \"+Nifty/\\\\1\"))
            (\"From:\" 
              (\"uucp\" . \"+adm/uucp\")
              (\".*\"   . \"+misc\"))
            ))
")

(defvar mew-refile-guess-key-list mew-destination:-list
  "*A list of field key used by mew-refile-guess-by-folder.")

(defvar mew-refile-guess-control
  '(mew-refile-guess-by-alist
    mew-refile-guess-by-newsgroups
    mew-refile-guess-by-folder
    mew-refile-ctrl-throw
    mew-refile-ctrl-auto-boundary
    mew-refile-guess-by-msgid
    mew-refile-guess-by-from
    mew-refile-guess-by-default))

(defvar mew-refile-auto-refile-skip-any-mark nil
  "*If *non-nil*, mew-summary-auto-refile doesn't touch
any alredy marked message.")

;;
;; initialize function
;;
(defun mew-refile-init ()
  ;; load message id alist
  (or mew-refile-msgid-alist
      (setq mew-refile-msgid-alist
	    (mew-refile-alist-load mew-refile-msgid-file-name)))
  ;; load from alist
  (or mew-refile-from-alist
      (setq mew-refile-from-alist
	    (mew-refile-alist-load mew-refile-from-file-name))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Guess functions 
;;;

;; We have two types of functions in mew-refile-guess-control,
;; guess function and ctrl function.
;; guess function must return a folder list or folder string or nil.
;; guess function must not have a string "ctrl" in its symbol name.
;; ctrl function must have a string "ctrl" in its symbol name.

;; dispatcher returns: ((guess1 guess2 ..) info1 info2 ...) multi  guess mode
;;                     ((guess1)           info1 info2 ...) single guess mode
;;            info1:   ('guess-func-name guess1 guess2 ...)
;;
;; that is, 'car' is a list of judged  folders.
;;          'cdr' is a alist of opinions by guess functions.
;;
(defun mew-refile-guess (&optional auto)
  (let ((funcs mew-refile-guess-control) ret guess info)
    (mew-refile-ctrl-throw-off)
    (mew-refile-ctrl-auto-boundary-off)
    (while funcs
      (if (string-match "ctrl" (symbol-name (car funcs)))
	  ;; func is control function
	  (funcall (car funcs))
	;; func is guess function
	(setq ret (funcall (car funcs))
	      ret (or (and (listp ret) ret)
		      (and ret (list ret))))
	(if (not (or (and guess mew-refile-ctrl-throw)
		     (and auto mew-refile-ctrl-auto-boundary)))
	    (setq guess (append guess ret)))
	(setq info (cons (cons (car funcs) ret) info)))
      (setq funcs (cdr funcs)))
    (setq info (nreverse info))
    (if mew-refile-ctrl-multi
	(cons (mew-uniq-list guess) info)
      (cons (list (car guess)) info))))

;;
;; guess control functions
;;
(defun mew-refile-ctrl-auto-boundary ()
  (setq mew-refile-ctrl-auto-boundary t))

(defun mew-refile-ctrl-auto-boundary-on ()
  (setq mew-refile-ctrl-auto-boundary t))

(defun mew-refile-ctrl-auto-boundary-off ()
  (setq mew-refile-ctrl-auto-boundary nil))

(defun mew-refile-ctrl-throw ()
  (setq mew-refile-ctrl-throw t))

(defun mew-refile-ctrl-throw-on ()
  (setq mew-refile-ctrl-throw t))

(defun mew-refile-ctrl-throw-off ()
  (setq mew-refile-ctrl-throw nil))

;;
;; by alist returns: (guess1 guess2 ...) or nil
;;
(defun mew-refile-guess-by-alist ()
  (mew-refile-guess-by-alist1 mew-refile-guess-alist))

(defun mew-refile-guess-by-alist1 (alist)
  (let (name header sublist key val ent ret)
    (while alist
      (setq name (car (car alist)))
      (setq sublist (cdr (car alist)))
      (cond
       ((eq name t)
	(setq ret (cons sublist ret)))
       ((eq name nil)
	(or ret (setq ret (cons sublist ret))))
       (t
	(setq header (mew-header-get-value name))
	(if header
	    (while sublist
	      (setq key (car (car sublist)))
	      (setq val (cdr (car sublist)))
	      (if (and (stringp key) (string-match key header))
		  (cond
		   ((stringp val)
		    (setq ent (mew-refile-guess-by-alist2 key header val)))
		   ((listp val)
		    (setq ent (mew-refile-guess-by-alist1 val)))))
	      (if ent
		  (if (listp ent)
		      (setq ret (append ent ret))
		    (setq ret (cons ent ret))))
	      (setq sublist (cdr sublist))))))
      (setq alist (cdr alist)))
    (mew-uniq-list (nreverse ret))))

(defun mew-refile-guess-by-alist2 (regexp field string)
  (let ((p 1) match-list b e)
    (string-match regexp field)
    (while (<= p 9)
      (setq b (or (match-beginning p) 0))
      (setq e (or (match-end p) 0))
      (setq match-list (cons (substring field b e) match-list))
      (setq p (1+ p)))
    (setq  p 1)
    (while (<= p 9)
      (if (string-match (concat "\\\\" (int-to-string p)) string)
	  (setq string
		(concat (substring string 0 (match-beginning 0))
			(nth (- 9 p) match-list)
			(substring string (match-end 0))))
	(setq p (1+ p))))
    string))

;;
;; by newsgroups returns (guess1 guess2 ...) or nil
;;
(defun mew-refile-guess-by-newsgroups ()
  (let ((newsgroups (mew-addrstr-parse-value-list2 
		     (mew-header-get-value mew-newsgroups:)))
	ent ret)
    (if (not newsgroups)
	()
      (while newsgroups
	(setq ent (mew-assoc-case-equal (car newsgroups) mew-folder-alist 1))
	(if ent (setq ret (cons (nth 0 ent) ret)))
	(setq newsgroups (cdr newsgroups)))
      (mew-uniq-list (nreverse ret)))))

;;
;; by folder returns: (guess1 guess2 ...) or nil
;;
(defun mew-refile-guess-by-folder ()
  (let ((to-cc (mew-header-parse-address-list mew-refile-guess-key-list))
	ent ret ml-name)
    (while to-cc
      (setq ml-name (mew-addrstr-extract-user (or (car to-cc) "")))
      (setq ent (mew-assoc-case-equal ml-name mew-folder-alist 1))
      (if (not ent)
	  ()
	(setq ret (cons (nth 0 ent) ret))
	(mew-refile-treat-alias-insert (car to-cc)))
      (setq to-cc (cdr to-cc)))
    (mew-uniq-list (nreverse ret))))

;;
;; by message-id returns: guess1 or nil
;;
(defun mew-refile-guess-by-msgid ()
  (let ((msgid (or (mew-header-get-value mew-references:)
		   (mew-header-get-value mew-in-reply-to:))))
    ;; search for msgid
    (if (and msgid 
	     (string-match "\\(<[^ \t\n]*>\\)[^>]*\0" (concat msgid "\0")))
	(nth 1 (assoc (substring msgid 
				 (match-beginning 1)
				 (match-end 1))
		      mew-refile-msgid-alist)))))

;;
;; by from returns: guess1
;;
(defun mew-refile-guess-by-from (&optional addr)
  (let ((from (or addr (mew-header-parse-address mew-from:) "")) default)
    ;; search from alist
    (setq default (mew-refile-guess-by-default from))
    (if (file-exists-p (mew-expand-folder default))
	default
      (cdr (assoc from mew-refile-from-alist)))))

;;
;; by To: or Cc: when From: is mine. (Undocumented)
;;
(defun mew-refile-guess-by-from2 ()
  (let ((to-cc (mew-header-parse-address-list mew-refile-guess-key-list))
	(from  (mew-header-parse-address mew-from:)))
    (if (catch 'my-addr
	  (mapcar (lambda (a)
		    (if (string-match a from)
			(throw 'my-addr t)))
		  (cons (concat "^" (regexp-quote (user-login-name)) "$")
			(cons (concat "^" mew-mail-address "$")
			      mew-mail-address-list)))
	  nil)
	(let (ent ret)
	  (while to-cc
	    (setq ent (mew-refile-guess-by-from (car to-cc)))
	    (if ent (setq ret (cons ent ret)))
	    (setq to-cc (cdr to-cc)))
	  (mew-uniq-list (nreverse ret)))
      ())))

;;
;; by default returns: guess1
;;
(defun mew-refile-guess-by-default (&optional addr)
  (let ((from (or addr (mew-header-parse-address mew-from:) "")))
    (if (and mew-folders-default-folder
	     (not (equal "" mew-folders-default-folder)))
	(concat (file-name-as-directory mew-folders-default-folder)
		(downcase (mew-addrstr-extract-user from)))
      (concat "+" (downcase (mew-addrstr-extract-user from))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Learning functions
;;;

;; dispatcher
;;
;; mew-refile-guess-learn (buf result)
;;
;; buf is message buffer.
;;
;; result is ((chosen1 chosen2 ...)
;;           (guess-func-name1 guess1 guess2...)
;;           (guess-func-name2 guess1 guess2...))
;;
;; that is, 'car' is a list of user chosen folders.
;;          'cdr' is a list of opinions by guess functions.
;;
(defun mew-refile-guess-learn (buf result)
  (let ((chosen (car result))  ;; (folder1 folder2 ...)
	(info   (cdr result))) ;; (guess-func-name guess1 guess2...)
    (save-excursion
      (set-buffer buf)
      (if (member 'mew-refile-guess-by-from mew-refile-guess-control)
	  (mew-refile-guess-by-from-learn chosen info))
      (if (member 'mew-refile-guess-by-msgid mew-refile-guess-control)
	  (mew-refile-guess-by-msgid-learn chosen info))
      (mew-refile-treat-alias-insert (mew-header-parse-address mew-from:)))))

;;
;; learn from msgid
;;
(defun mew-refile-guess-by-msgid-learn (chosen info)
  (let* ((msgid  (mew-header-get-value mew-message-id:)) 
	 (folder (car chosen))
	 ;; ohter people's honest opinion and my honest opinion.
	 (oho    (apply 'append info))
	 (mho    (cdr (assoc 'mew-refile-guess-by-msgid info))))
    (if (and msgid (string-match "<[^ \n>]*>" msgid))
	(setq msgid (substring msgid (match-beginning 0) (match-end 0))))
    (if (or (not msgid) (not chosen))
	()
      ;; if my opninion was right, I learn it.
      ;; or a folder was not in other people's opinion,
      ;; I accept it.
      (catch 'match
	(while chosen
	  (if (or (member (car chosen) mho)
		  (not (member (car chosen) oho)))
	      (throw 'match (setq folder (car chosen))))
	  (setq chosen (cdr chosen))))
      (setq mew-refile-msgid-alist
	    (cons (list msgid folder "??")
		  (mew-refile-alist-purge msgid mew-refile-msgid-alist))))))

;;
;; learn from "From:" field
;;
(defun mew-refile-guess-by-from-learn (chosen info)
  (let ((from (mew-header-parse-address mew-from:))
	(folder nil)
	;; ohter people's honest opinion and my honest opinion.
	(oho    (apply 'append info))
	(mho    (cdr (assoc 'mew-refile-guess-by-from info))))
    (if (or (not from) (not chosen))
	()
      ;; if my opninion was right, I learn it.
      ;; or a folder was not in other people's opinion,
      ;; I accept it.
      (catch 'match
	(while chosen
	  (if (or (member (car chosen) mho)
		  (not (member (car chosen) oho)))
	      (throw 'match (setq folder (car chosen))))
	  (setq chosen (cdr chosen))))
      (if  folder
	  (progn
	    (setq mew-refile-from-alist
		  (mew-refile-alist-purge from mew-refile-from-alist))
	    (if mew-refile-from-alist
		(nconc mew-refile-from-alist (list (cons from folder)))
	      (setq mew-refile-from-alist (list (cons from folder)))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Aliases
;;;

;;
;; Address completion in Draft mode.
;; It should not be here...
;;
(defun mew-refile-treat-alias-insert (address)
  (if (and (stringp address) (string-match "@" address))
      (let (user)
	;; Add empty information ("nom@mew.org" . nil) if key is absent.
	(if (assoc address mew-refile-from-alist)
	    ()
	  (if mew-refile-from-alist
	      (nconc mew-refile-from-alist (list (cons address nil)))
	    (setq mew-refile-from-alist (list (cons address nil)))))
	;; also Add to mew-alias-alist
	(setq user (mew-addrstr-extract-user address))
	(if (assoc user mew-alias-alist)
	    ()
	  (if mew-alias-alist
	      (nconc mew-alias-alist (list (cons user address)))
	    (setq mew-alias-alist (list (cons user address))))))))

;;
;; append alias items made from from-alist to alias-alist
;; in reverse order.
;;

(defun mew-refile-alist-append-alias (alias-alist)
  ;; must be uniqfy for alias completion
  (let ((loop-alist mew-refile-from-alist) 
	ret addr user)
    (while loop-alist
      (setq addr (car (car loop-alist)))
      (setq user (mew-addrstr-extract-user addr))
      (if (assoc user alias-alist)
	  ()
	(setq ret (cons (cons user addr) ret)))
      (setq loop-alist (cdr loop-alist)))
    (if alias-alist
	(nconc alias-alist (nreverse ret))
      (nreverse ret))))

;;
;; Alias maintenance function (purge)
;;
(defun mew-refile-delete-alias ()
  "Delete an alias from mew-alias-alist and mew-refile-from-alist."
  (interactive)
  (let* ((addr (or (mew-delete-backward-char) ""))
	 (user (mew-addrstr-extract-user addr))
	 ent)
    (insert addr)
    (if (not (assoc addr mew-refile-from-alist))
	(message "<%s> is not derived from auto learning." addr)
      (if (not (y-or-n-p (format "Delete alias for %s ? " addr)))
	  (message "...")
	;; delete from alias-alist and from-alist
	(setq mew-refile-from-alist
	      (delete (assoc addr mew-refile-from-alist) mew-refile-from-alist))
	(setq ent (assoc user mew-alias-alist))
	(if (equal addr (cdr ent))
	    (setq mew-alias-alist (delete ent mew-alias-alist)))
	;; save from-alist for safety
	(mew-refile-alist-save mew-refile-from-file-name mew-refile-from-alist)
	;; delete address in screen.
	(mew-delete-backward-char)
	(message "Alias for %s is deleted." addr)))))
;;
;; Alias maintenance function (add)
;;
(defun mew-refile-add-alias ()
  "Add an alias to mew-alias-alist and mew-refile-from-alist."
  (interactive)
  (let* ((addr (or (mew-delete-backward-char) ""))
	 (key  (mew-addrstr-extract-user addr))
	 value)
    (insert addr)
    (if (not (string-match "@" addr))
	(message "<%s> is not complete address notation.")
      (if (not (y-or-n-p (format "Add alias for %s ? " addr)))
	  ()
        ;; if key is absent, add empty information ("nom@mew.org" . nil).
        ;; if key is already exists, move it to the top of the alist.
	(if (setq value (assoc addr mew-refile-from-alist))
	    (setq mew-refile-from-alist
		  (cons value (delete value mew-refile-from-alist)))
	  (setq mew-refile-from-alist
		(cons (cons addr nil) mew-refile-from-alist)))
	;; also Add to mew-alias-alist
	(setq mew-alias-alist
	      (cons 
	       (cons key addr)
	       (delete (assoc key mew-alias-alist) mew-alias-alist)))
	;; save from-alist for safety
	(mew-refile-alist-save mew-refile-from-file-name mew-refile-from-alist)
	(message "Alias for %s is added." addr)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; common routines for (a)list
;;;

(defun mew-refile-alist-purge (key alist)
  (let ((a alist))
    (delete (assoc key a) a)))

(defun mew-refile-alist-load (filename)
  (let ((alist nil)
	(fullname (expand-file-name filename mew-mail-path)))
    (save-excursion
      (if (not (file-readable-p fullname))
	  ()
	(mew-set-buffer-tmp)
	(insert-file-contents fullname)
	(setq alist 
	      (condition-case nil
		  (read (current-buffer))
		(error ())))
	alist))))

(defun mew-refile-alist-save (filename alist)
  (save-excursion
    (let* ((fullname (expand-file-name filename mew-mail-path))
	   (tmp-buf  (set-buffer (create-file-buffer fullname)))
	   (n mew-refile-alist-max-length)
	   (pointer (cons nil alist)))
      (while pointer
	(if (> n 0)
	    (progn
	      (setq pointer (cdr pointer))
	      (setq n (1- n)))
	  (setcdr alist nil)
	  (setq pointer nil)))
      (mew-erase-buffer)
      (prin1 alist tmp-buf)
      (princ "\n" tmp-buf)
      (write-region (point-min) (point-max) fullname nil 'no-msg)
      (kill-buffer tmp-buf))))

(defun mew-refile-guess-save ()
  (if (and mew-refile-from-alist
	   (member 'mew-refile-guess-by-from mew-refile-guess-control))
      (mew-refile-alist-save mew-refile-from-file-name mew-refile-from-alist))
  (if (and mew-refile-msgid-alist
	   (member 'mew-refile-guess-by-msgid mew-refile-guess-control))
      (mew-refile-alist-save mew-refile-msgid-file-name
			     mew-refile-msgid-alist)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Summary mode
;;;
(defun mew-refile-set (msg folder)
  (let* ((msg-folders (assoc msg mew-summary-buffer-refile))
	 (folders (cdr msg-folders)))
    (if folders
	(if (not (mew-folder-member folder folders))
	    (setq mew-summary-buffer-refile 
		  (cons (append (list msg) folders (list folder))
			(delete msg-folders mew-summary-buffer-refile))))
      (setq mew-summary-buffer-refile 
	    (cons (list msg folder) mew-summary-buffer-refile)))))

(defun mew-refile-reset (msg)
  (setq mew-summary-buffer-refile
	(delete (assoc msg mew-summary-buffer-refile)
		mew-summary-buffer-refile)))

;; mew-refile-decide-folders returns: ((input1 input2...) 
;;                                     (guess-func-name1 guess1 guess2...)
;;                                     (guess-func-name2 guess1 guess2...))
;; that is, 'car' is a list of user chosen folders.
;;          'cdr' is a alist of opinions by guess functions.
;; cdr is needed for learning functions.
;;
(defun mew-refile-decide-folders (buf msg mark &optional auto)
  (let (learn-info folders ret)
    (save-excursion
      (set-buffer buf)
      (setq learn-info (mew-refile-guess auto)))
    (setq 
     folders
     (cond
      ;; if auto is set, simply use the guess.
      (auto (car learn-info))
      ;; add new folder 
      ((equal mew-mark-refile mark)
       (mew-input-folders 
	nil (mew-join "," 
		      (cdr (assoc msg mew-summary-buffer-refile)))))
      ;; multi guess
      ((nth 1 (car learn-info))
       (mew-input-folders nil (mew-join "," (car learn-info))))
      ;; single guess
      (t 
       (mew-input-folders (nth 0 (car learn-info))))))
    ;; check folder existence.
    (while folders
      (if (mew-folder-check (car folders))
	  (setq ret (cons (car folders) ret)))
      (setq folders (cdr folders)))
    (cons (nreverse ret) (cdr learn-info)) ;; return value
    ))

(defun mew-summary-refile (&optional exp-flds auto)
  "Put the refile mark(default is 'o') on this message. 
If already marked with 'o', it prints where this message 
will be refiled. This can overlay other marks. When it overlays, 
the cursor stays on the message. If it marks newly, displays 
the next message."
  (interactive)
  (mew-summary-only
   (mew-summary-msg-or-part
    (let (msg folders mark buf learn-info)
      (save-excursion
	;; save the cursor position anyway
	(mew-summary-goto-message)
	;; on the message
	(setq msg (mew-summary-message-number)) ;; msg is never nil
	(setq mark (mew-summary-get-mark)) ;; any mark
	(if exp-flds
	    (setq folders exp-flds)
	  ;; show message if not displayed
	  (mew-summary-display t) ;; not force
	  (setq buf (or (mew-cache-hit (cons (buffer-name) msg))
			;; non analysis
			(mew-buffer-message)))
	  (setq learn-info (mew-refile-decide-folders buf msg mark auto))
	  (setq folders (car learn-info)))
 	;; mark refile
	(if (null folders)
	    ()
	  (or exp-flds (mew-refile-guess-learn buf learn-info))
	  (mew-refile-reset msg)
	  (mapcar (lambda (x) (mew-refile-set msg x)) folders)
	  (mew-mark-unmark)
	  (mew-summary-mark-as mew-mark-refile)))
      ;; memorize last-folder
      (setq mew-refile-last-folder folders)
      (if (or mark auto (not folders))
	  () ;; stay here
	(mew-summary-goto-message)
	;; on the message
	(mew-decode-syntax-delete)
	;; for C-x C-x
	(beginning-of-line)
	(let ((zmacs-regions nil))
	  (push-mark (point) t t))
	(mew-summary-display-next))
      (set-buffer-modified-p nil)
      folders)))) ;; return value

(defun mew-summary-refile-again ()
  "Put a refile mark on this message according to the previous 
refile folder."
  (interactive)
  (mew-summary-only
   (mew-summary-refile mew-refile-last-folder)))

(defun mew-summary-auto-refile ()
  "Refile each message in the folder automatically."
  (interactive)
  (mew-summary-only
   (let ((mew-analysis nil) 
	 (after-change-function nil)
	 (after-change-functions nil)
	 (mew-use-highlight-x-face nil)
	 (lines (count-lines (point-min) (point-max)))
	 (line 1)
	 (mark nil))
     (message "Auto refiling ...")
     (save-window-excursion
       (goto-char (point-min))
       (while (not (eobp))
	 (setq mark (mew-summary-get-mark))
	 (or (equal mark mew-mark-refile)
	     (equal mark mew-mark-delete)
	     (and mark mew-refile-auto-refile-skip-any-mark)
	     (mew-summary-refile nil t))
	 (forward-line)
	 (if (equal (% (/ (* 100 line) lines) 10) 0)
	     (message "Auto refiling ... %s%%"
		      (/ (* 100 line) lines)))
	 (setq line (1+ line)))
       (message "Auto refiling ... done")))))

;;
;; "mx" extension
;;
(defun mew-summary-mark-refile ()
  "Refile messages marked with '*'."
  (interactive)
  (mew-summary-only
   (let ((mew-analysis nil)
	 (after-change-function nil)
	 (after-change-functions nil)
	 (mew-use-highlight-x-face nil)
	 (last nil)
	 (regex (concat mew-summary-message-regex
			(regexp-quote (char-to-string mew-mark-review)))))
     (message "Mark refiling ...")
     (save-excursion
       (save-window-excursion
	 (goto-char (point-min))
	 (while (re-search-forward regex nil t)
	   (setq last (mew-summary-refile last))
	   (forward-line))
	 (message "Mark refiling ... done"))))))

(provide 'mew-refile)

;;; Copyright Notice:

;; Copyright (C) 1994, 1995, 1996, 1997, 1998 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-refile.el ends here
