/* A program to lock a file exactly as Exim would, for investigation of
interlocking problems.

Options:  -fcntl    use fcntl lock only
          -lockfile use lock file only

Argument: the name of the lock file
*/

#include "os.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <netdb.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/utsname.h>
#include <sys/stat.h>
#include <pwd.h>



#ifdef STRERROR_FROM_ERRLIST
/*************************************************
*     Provide strerror() for non-ANSI libraries  *
*************************************************/

/* Some old-fashioned systems still around (e.g. SunOS4) don't have strerror()
in their libraries, but can provide the same facility by this simple
alternative function. */

char *
strerror(int n)
{
if (n < 0 || n >= sys_nerr) return "unknown error number";
return sys_errlist[n];
}
#endif /* STRERROR_FROM_ERRLIST */


/*************************************************
*           The exim_lock program                *
*************************************************/

int main(int argc, char **argv)
{
int lock_retries = 10;
int lock_interval = 5;
int use_lockfile = 0;
int use_fcntl = 0;
int verbose = 0;
int quiet = 0;
int i, j, len, rc;
int fd = -1;
int hd = -1;
int yield = 0;
int now = time(NULL);
char *filename, *lockname, *hitchname;
char *primary_hostname, *command;
struct utsname s;
struct hostent *host;
char buffer[256];

/* Decode options */

for (i = 1; i < argc; i++)
  {
  char *arg = argv[i];
  if (*arg != '-') break;
  if (strcmp(arg, "-fcntl") == 0) use_fcntl = 1;
  else if (strcmp(arg, "-lockfile") == 0) use_lockfile = 1;
  else if (strcmp(arg, "-v") == 0) verbose = 1;
  else if (strcmp(arg, "-q") == 0) quiet = 1;
  else
    {
    printf("usage: exim_lock [-v] [-q] [-fcntl] [-lockfile] <file name> [command]\n");
    exit(1);
    }
  }
if (quiet) verbose = 0;

/* Default is to use both kinds of lock */

if (use_lockfile + use_fcntl == 0) use_lockfile = use_fcntl = 1;

/* A file name is required */

if (i >= argc)
  {
  printf("usage: exim_lock [-v] [-q] [-fcntl] [-lockfile] <file name> [command]\n");
  exit(1);
  }

filename = argv[i++];

/* Expand file names starting with ~ */

if (*filename == '~')
  {
  struct passwd *pw;

  if (*(++filename) == '/')
    pw = getpwuid(getuid());
  else
    {
    char *s = buffer;
    while (*filename != 0 && *filename != '/')
      *s++ = *filename++;
    *s = 0;
    pw = getpwnam(buffer);
    }

  if (pw == NULL)
    {
    printf("exim_lock: unable to expand file name %s\n", argv[i-1]);
    exit(1);
    }

  if ((int)strlen(pw->pw_dir) + (int)strlen(filename) + 1 > sizeof(buffer))
    {
    printf("exim_lock: expanded file name %s%s is too long", pw->pw_dir,
      filename);
    exit(1);
    }

  strcpy(buffer, pw->pw_dir);
  strcat(buffer, filename);
  filename = buffer;
  }

/* If using a lock file, prepare by creating the lock file name and
the hitching post name. */

if (use_lockfile)
  {
  if (uname(&s) < 0)
    {
    printf("exim_lock: failed to find host name using uname()");
    exit(1);
    }
  primary_hostname = s.nodename;

  len = (int)strlen(filename);
  lockname = malloc(len + 8);
  sprintf(lockname, "%s.lock", filename);
  hitchname = malloc(len + 32 + (int)strlen(primary_hostname));
  sprintf(hitchname, "%s.%s.%08x.%08x", lockname, primary_hostname,
    now, getpid());

  if (verbose)
    printf("exim_lock: lockname =  %s\n           hitchname = %s\n", lockname,
      hitchname);
  }

/* Locking retry loop */

for (j = 0; j < lock_retries; sleep(lock_interval), j++)
  {
  struct flock lock_data;
  struct stat statbuf;
  hd = -1;

  /* Try to build a lock file if so configured */

  if (use_lockfile)
    {
    int rc;
    if (verbose) printf("exim_lock: creating lock file\n");
    hd = open(hitchname, O_WRONLY | O_CREAT | O_EXCL, 0440);
    if (hd < 0)
      {
      printf("exim_lock: failed to create hitching post %s: %s\n", hitchname,
        strerror(errno));
      exit(1);
      }

    /* Apply hitching post algorithm. */

    if ((rc = link(hitchname, lockname)) != 0) fstat(hd, &statbuf);
    close(hd);
    unlink(hitchname);

    if (rc != 0 && statbuf.st_nlink != 2)
      {
      printf("exim_lock: failed to link hitching post to lock file ... retrying\n");
      continue;
      }

    if (!quiet) printf("exim_lock: lock file successfully created\n");
    }

  if (!use_fcntl) break;

  /* Open the file for writing, if not already open, and lock it using fcntl. */

  if (fd < 0)
    {
    fd = open(filename, O_WRONLY + O_APPEND);
    if (fd < 0)
      {
      printf("exim_lock: failed to open %s for writing: %s\n", filename,
        strerror(errno));
      yield = 1;
      goto CLEAN_UP_LOCKFILE;
      }
    }

  lock_data.l_type = F_WRLCK;
  lock_data.l_whence = lock_data.l_start = lock_data.l_len = 0;
  if (fcntl(fd, F_SETLK, &lock_data) >= 0)
    {
    if (!quiet) printf("exim_lock: fcntl() lock successfully applied\n");
    break;
    }
  else
    {
    printf("exim_lock: fcntl() failed: %s", strerror(errno));
    if (hd > 0)
      {
      unlink(lockname);
      printf("\nexim_lock: lock file removed");
      }
    printf(" ... retrying\n");
    }
  }

if (j >= lock_retries)
  {
  printf("exim_lock: locking failed\n");
  exit(1);
  }

if (!quiet) printf("exim_lock: locking %s succeeded: ", filename);

/* If there are no further arguments, run the user's shell; otherwise
the next argument is a command to run. */

if (i >= argc)
  {
  command = getenv("SHELL");
  if (command == NULL || *command == 0) command = "/bin/sh";
  if (!quiet) printf("running %s ...\n", command);
  }
else
  {
  command = argv[i];
  if (!quiet) printf("running the command ...\n");
  }

/* Run the command */

system(command);

/* Remove the locks and exit */

if (use_fcntl)
  {
  if (close(fd) < 0)
    printf("exim_lock: close failed: %s\n", strerror(errno));
  else
    if (!quiet) printf("exim_lock: file closed\n");
  }

CLEAN_UP_LOCKFILE:

if (use_lockfile && hd >= 0)
  {
  if (unlink(lockname) < 0)
    printf("exim_lock: unlink of %s failed: %s\n", lockname, strerror(errno));
  else
    if (!quiet) printf("exim_lock: lock file removed\n");
  }

exit(yield);
}

/* End */
