/*								-*- C++ -*-
 * $Id: IPC_socket.cpp,v 1.2 1997/11/18 15:54:09 wmglo Exp $
 *
 * Purpose: socket mamagment
 *
 * Authors: Markus Holzem and Julian Smart
 *
 * Copyright: (C) 1995, AIAI, University of Edinburgh (Julian)
 * Copyright: (C) 1995, GNU (Markus)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Additionally everyone using this library has to announce it with:
 *
 *   This software uses the wxWindows-Xt GUI library
 *   (C) Markus Holzem, available via
 *       ftp://ftp.aiai.ed.ac.uk/pub/packages/wxwin/ports/xt
 */

#include "wx_setup.h"

#include <ctype.h>
#include <fcntl.h>
#include <math.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <sys/errno.h>
#ifdef sgi
    #include <bstring.h>
#endif
#ifdef SVR4
//    #include <strings.h>
    #include <errno.h>
#endif
#include <sys/time.h>
#include <sys/wait.h>
#ifdef UNIX_ADDRESSING
    #include <sys/un.h>
#else
    #include <netdb.h>
#endif

#ifdef __GNUG__
#pragma implementation "IPC_socket.h"
#endif

#define  Uses_XtIntrinsic
#define  Uses_wxIPC
#include "wx.h"
#include "IPC_socket.h"

#ifdef UNIX_ADDRESSING
    typedef union {
	struct sockaddr addr;
	struct sockaddr_un unaddr;
	struct sockaddr_in inaddr;
    } sockaddr_union;
    #define wxNAMELEN(s)\
	(wxFAMILY(s) == AF_INET ? \
	 sizeof(s.inaddr) : \
	 strlen(s.unaddr.sun_path)+sizeof(s.unaddr)-sizeof(s.unaddr.sun_path))
#else
    typedef union {
	struct sockaddr addr;
	struct sockaddr_in inaddr;
    } sockaddr_union;
    #define wxNAMELEN(s) sizeof(s.inaddr)
#endif
#define wxFAMILY(s) s.addr.sa_family

//-----------------------------------------------------------------------------
// MiniStopWatch
//-----------------------------------------------------------------------------

MiniStopwatch::MiniStopwatch(void) : running(FALSE)
{
}

MiniStopwatch::~MiniStopwatch(void)
{
}

void MiniStopwatch::start(void)
{
    running = TRUE;
    gettimeofday(&startTV, 0);
}

void MiniStopwatch::stop(void)
{
    if (!running)
	return;
    running = FALSE;
    gettimeofday(&stopTV, 0);
}

long MiniStopwatch::howLong(void)
{
    timeval tv;  // Holds current time (if running) or stop time.

    if (running)
	gettimeofday(&tv, 0);
    else  // Client presumably has done a start/stop.
	tv = stopTV;
    return (tv.tv_usec - startTV.tv_usec) / 1000
	 + (tv.tv_sec - startTV.tv_sec) * 1000;
}

Bool MiniStopwatch::past(long msec)
{
    return howLong() >= msec;
}

Bool MiniStopwatch::isRunning(void)
{
    return running;
}

// Sleep for 1 millisecond.
void MiniStopwatch::tick(void)
{
#ifdef SVR4
    sleep(1);
#else
    // Only defined for BSD platforms!
    usleep(1000);
#endif
}
// If not already past(), tick() and then see whether we are past().
Bool MiniStopwatch::slowPast(long msec)
{
    if (past(msec))
	return TRUE;
    tick();
    return past(msec);
}

//-----------------------------------------------------------------------------
// SafeSock
//-----------------------------------------------------------------------------

const char* SafeSock::MAGIC     = "ROGR";	// No digits allowed (dec or hex).
const int   SafeSock::COUNT_LEN = 8;		// Exact # of hex digits in count.
const char  SafeSock::SEP_CHAR  = '|';		// No digits allowed (dec or hex).
const int   SafeSock::MAGIC_LEN = 4;
const int   SafeSock::OVRHD_LEN = 13;

SafeSock::SafeSock(int fd_, Bool ownFd_) : fd(fd_)
{
    ownFd    = ownFd_;
    inBox    = wxNEW char[1];
    inBoxLen = 0;
    timeout  = -1;
    timedOut = FALSE;
    *inBox  = 0;
}

SafeSock::~SafeSock(void)
{
    delete [] inBox;  inBox = 0;
    if (ownFd)
	close(fd);
}

void SafeSock::setTimeout(int msec)
{
    timeout = msec;
    if (timeout < 0) { // Set to synchronous mode; use blocking I/O.
	if (fcntl(fd, F_SETFL, O_SYNC) < 0)
	    perror("fcntl F_SETFL, O_SYNC");
    } else {           // Set asynchronous mode so we can deal with timeouts.
	if (fcntl(fd, F_SETFL, O_NDELAY) < 0)
	    perror("fcntl F_SETFL, O_NDELAY");
    }
}

void SafeSock::clearTimedOut(void)
{
    timedOut = FALSE; // Clear the flag.
}

Bool SafeSock::lastTimedOut(void)
{
    return timedOut;
}

int SafeSock::write(char * buf, int nBytes)
{
    Bool async = (timeout >= 0);

    if (async) { // Ensure data can be written within timeout.
	fd_set fdSet;

	FD_ZERO(&fdSet);
	FD_SET(fd, &fdSet);

	timeval tv;  // Calc sec/usec from timeout (which is msec).
	tv.tv_sec = timeout / 1000;
	tv.tv_usec = (timeout % 1000) * 1000;

	int sel = select(fd + 1, 0, &fdSet, 0, &tv);

	timedOut = !sel;
	if (timedOut)
	    return -1;

	// On error (-1) from select(), FALL THROUGH and return error
	// from write().
    }

    // Prepare a real buf with our wrapper info.
    int realLen = nBytes + OVRHD_LEN;
    char * realBuf = new char [realLen+1];

    // Begin with byte count + separation char ....
    sprintf(realBuf, "%08lx%c", (unsigned long) nBytes, SEP_CHAR);

    // Now copy the "real" data.
    char * p = realBuf + COUNT_LEN + 1;
    memcpy(p, buf, nBytes);

    // Finish off with magic sequence.
    strcpy(p + nBytes, MAGIC);

    // Write the real buf and free its memory.
    int nWritten = 0;

    if (async)
	stopwatch.start();

    int nWrittenThis = 0;  // Up to 4K may be written per write() call.
    Bool sawError = FALSE;

    do {
	// If synchronous (no timeout), just block; else poll at
	// (approx.) 1-msec intervals until success or timeout.
	do {
	    int nWrittenThis = ::write(fd, realBuf + nWritten, realLen - nWritten);

	    if (nWrittenThis < 0) {
		if (errno != EWOULDBLOCK)
		    sawError = TRUE;
	    } else
		nWritten += nWrittenThis;
#if 0
	    cerr << "LoopWrite: " << nWritten
		 << '\t' << sawError
		 << '\t' << errno
		 << endl;
#endif
	} while (async && !sawError && (nWrittenThis < 0) &&
		 !stopwatch.slowPast(timeout));
	if (async && (sawError || stopwatch.past(timeout)))
	    break;
    } while (nWritten < nBytes);

    delete [] realBuf;  realBuf = 0;

    if (async) {
	stopwatch.stop();
#if 0
	cerr << "    Write: " << nWritten
	     << '\t' << sawError
	     << '\t' << errno
	     << endl;
#endif
	if ((nWritten < 0) || sawError)
	    timedOut = TRUE;
	else if (!nWritten && !sawError && errno)
	    timedOut = TRUE;
    }
    return nWritten;
}

int SafeSock::read(char * buf, int nBytes, Bool & isWhole, Bool & isMore)
{
    int nRead = 0;

    // If isMore == TRUE, caller is getting second (or later) message from
    // earlier read.  Otherwise, read from fd.
    if (!isMore) {
	Bool async = (timeout >= 0);

	if (async) { // Ensure data can be read within timeout.
	    fd_set fdSet;

	    FD_ZERO(&fdSet);
	    FD_SET(fd, &fdSet);

	    timeval tv;  // Calc sec/usec from timeout (which is msec).
	    tv.tv_sec = timeout / 1000;
	    tv.tv_usec = (timeout % 1000) * 1000;

	    int sel = select(fd + 1, 0, &fdSet, 0, &tv);
	    
	    timedOut = !sel;
	    if (timedOut)
		return -1;

	    // On error (-1) from select(), FALL THROUGH and return error
	    // from read().
	}
	if (async)
	    stopwatch.start();
	// If synchronous (no timeout), just block; else poll at
	// (approx.) 1-msec intervals until success or timeout.
	Bool sawError = FALSE;
	do {
	    nRead = ::read(fd, buf, nBytes);

	    if ( (nRead < 0) && (errno != EWOULDBLOCK) )
		sawError = TRUE;
#if 0
	    cerr << " LoopRead: " << nRead
		 << '\t' << sawError
		 << '\t' << errno
		 << endl;
#endif
	} while (async && !sawError && (nRead < 0)
		 && !stopwatch.slowPast(timeout));

	if (async) {
	    stopwatch.stop();
#if 0
	    cerr << "     Read: " << nRead
		 << '\t' << sawError
		 << '\t' << errno
		 << endl;
#endif
	    if ((nRead < 0) || sawError)
		timedOut = TRUE;
	    else if (!nRead && !sawError && errno)
		timedOut = TRUE;
	}
	if (nRead <= 0) {
	    isWhole = TRUE;
	    isMore = FALSE;
	    return nRead;
	}
	// Append buf to inBox.
	int newLen = inBoxLen + nRead;
	char * newBox = new char [newLen];
// Causes a strange error message for Solaris compiler
// 	bcopy(inBox, newBox, inBoxLen);
// 	bcopy(buf, newBox + inBoxLen, nRead);
	int i;
	for (i = 0; i < int(inBoxLen); i++)
	    newBox[i] = inBox[i];
	for (i = 0; i < nRead; i++)
	    newBox[inBoxLen + i] = buf[i];
	delete [] inBox;  inBox = newBox;  newBox = 0;
	inBoxLen = newLen;
    }
    isWhole = isMore = FALSE;  // Assumptions.
    // Is a complete message in inBox?  If not, just return nRead;
    // caller is responsible for noticing that !isWhole.
    char * start = inBox, * end = start + inBoxLen;
    if (!findMsg(start, end))
	return nRead;
    // Good, we have at least one message.  Place it in buf.
    int msgLen = end - start;  // Should really be an unsigned long.
    isWhole = TRUE;
    memcpy(buf, start, msgLen);
    buf[msgLen] = 0;
    // Remove this message (and any preceding crud) from inBox.
    end += MAGIC_LEN;  // Point past MAGIC sequence.
    char * oldBox = inBox;
    inBoxLen -= end - inBox;  // NOT (end - start)!
    inBox = new char[inBoxLen];
    memcpy(inBox, end, inBoxLen);
    delete [] oldBox;  oldBox  = 0;
    // Is there another complete message in inBox?
    start = inBox;  end = start + inBoxLen;
    isMore = findMsg(start, end);
    return msgLen;
}

int SafeSock::getFd(void)
{
    return fd;
}


Bool SafeSock::findMsg(char * & start, char * & end)
{
    // Try to find beginning of count (first digit in pointed-to range).
    while ((start < end) && !isxdigit(*start))
	++start;
    if (start >= end)
	return FALSE;
    // start currently points at first digit.  Make sure we have
    // COUNT_LEN digits followed by SEP_CHAR.
    char * p = start + 1;

    while ((p != end) && isxdigit(*p) && ((p - start) < COUNT_LEN))
	++p;
    if (p == end)
	return FALSE;

    // Now p is pointing one past the count, which should contain
    // the sep char.  If not, try again with a later start.
    if ((*p != SEP_CHAR) || ((p - start) != COUNT_LEN))
	return findMsg(++start, end);

    *p = 0;
    unsigned long msgLen;
    sscanf(start, "%08lx", &msgLen);
    *p = SEP_CHAR;

    // Point start at first char of what we tentatively believe is a
    // message; try to find message end.
    start = p + 1;

    // See if there is enough room for this message.  If not, it may
    // not be all here or we may have been fooled by some crud; try again.
    if (int(end - start) < int(msgLen + MAGIC_LEN))
	return findMsg(start, end);

    // Insist on magic sequence to help ensure that we have a real message.
    if (strncmp(start + msgLen, MAGIC, MAGIC_LEN))
	return findMsg(start, end);

    end = start + msgLen;
    return TRUE;
}


//-----------------------------------------------------------------------------
// SafeSock
//-----------------------------------------------------------------------------

SockMgr* SockMgr::master = 0;

SockMgr::SockMgr(void)
{
    if (!master)  master = this;
}
SockMgr::~SockMgr(void)
{
    // Close any sockets remaining in the list, deleting the objects.
    for (wxNode * n = socks.First(); n; n = n->Next()) {
	SafeSock * sock = (SafeSock *) (n->Data());
	close(sock->getFd());
	delete sock;  sock = 0;
    }
    master = 0;
}

// Add a SafeSock object managing the given fd to the list, unless such
// an object already exists (which it should not!).
void SockMgr::addSock(int fd)
{
    SafeSock * sock = getSock(fd);
    if (!sock)
	socks.Append((wxObject *) (new SafeSock(fd)));
}

// Return a reference to the master socket manager, creating if necessary.
SockMgr& SockMgr::create(void)
{
    return *(master  ?  master  :  new SockMgr);
}

//  Should be called when app terminates.  Since there is as yet no good way
//  to determine when the app is terminating, the mfn is commented out for now.
//  It would be nice to have a wxApp::AtExit(), which you could use to place
//  a fn on a list of fns to be called back during ~wxApp().

    // Destroy master SockMgr.  Note that this invalidates all existing refs!
//    static void destroy(void)  {  delete master;  master = 0;  }

// Attempt to create a socket connection on the given fd.
// Args and return value are the same as connect(2).
int SockMgr::connect(int fd, struct sockaddr * name, int nameLen)
{
    int rval = ::connect(fd, name, nameLen);
    if (!rval)
	addSock(fd);
    return rval;
}

// Attempt to accept a socket connection on the given fd.
// Args and return value are the same as connect(2).
int SockMgr::accept(int fd, struct sockaddr * addr, int * addrLen)
{
    int rval = ::accept(fd, addr, addrLen);
    if (rval > 0)
	addSock(rval);
    return rval;
}

// Set read/write timeout on the indicated SafeSock.  For
// consistency with the rest of wxWindos, the timeout value is in
// msec.  If timeout < 0, timeouts are disabled.
void SockMgr::setTimeout(int fd, int msec)
{
    SafeSock * sock = getSock(fd);
    
    if (sock)
	sock->setTimeout(msec);
}

Bool SockMgr::lastTimedOut(int fd)
{
    SafeSock * sock = getSock(fd);

    return sock && sock->lastTimedOut();
}

void SockMgr::clearTimedOut(int fd)
{
    SafeSock* sock = getSock(fd);
    
    if (sock)
	sock->clearTimedOut();
}

// Return ptr to SafeSock object corresponding to fd, or 0 if not found.
SafeSock* SockMgr::getSock(int fd)
{
    for (wxNode * n = socks.First(); n; n = n->Next()) {
	SafeSock * sock = (SafeSock *) (n->Data());
	if (sock->getFd() == fd)
	    return sock;
    }
    return 0;
}

// Close the given fd; if successful, delete the corresponding list node.
// Args and return value are the same as for close(2).
int SockMgr::close(int fd)
{
    int rval = ::close(fd);

    // If close succeeded, delete corresponding list node.
    // May be no corresponding node, if fd is STDI/O rather than socket.
    if (!rval) {
	for (wxNode * n = socks.First(); n; n = n->Next()) {
	    SafeSock * sock = (SafeSock *) (n->Data());
	    if (sock->getFd() == fd) {
		delete sock;  sock = 0;
		socks.DeleteNode(n);
		break;
	    }
	}
    }
    return rval;
}

// Read/write through the SockMgr.  For single reads/writes, this is
// convenient.  But if you have multiple reads/writes to do on the same
// fd, it is more efficient to call getSock() and then use the SafeSock
// directly (that way avoids a list search per operation).  On failure,
// these return -1 without setting errno!

int SockMgr::read(int fd, char * buf, int nBytes, Bool & isWhole, Bool & isMore)
{
    SafeSock * sock = getSock(fd);
    if (!sock)
	return -1;
    return sock->read(buf, nBytes, isWhole, isMore);
}

int SockMgr::write(int fd, char * buf, int nBytes)
{
    SafeSock * sock = getSock(fd);
    if (!sock)
	return -1;
    return sock->write(buf, nBytes);
}
