/*								-*- C++ -*-
 * $Id: DEV_windowdc.cpp,v 1.2 1997/07/14 09:38:24 wmglo Exp $
 *
 * Purpose: device context to draw drawables
 *          (windows and pixmaps, even if pixmaps are covered by wxMemoryDC)
 *
 * Authors: Markus Holzem and Julian Smart
 *
 * Copyright: (C) 1995, AIAI, University of Edinburgh (Julian)
 * Copyright: (C) 1995, GNU (Markus)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Additionally everyone using this library has to announce it with:
 *
 *   This software uses the wxWindows-Xt GUI library
 *   (C) Markus Holzem, available via
 *       ftp://ftp.aiai.ed.ac.uk/pub/packages/wxwin/ports/xt
 */

#ifdef __GNUG__
#pragma implementation "DEV_windowdc.h"
#endif

#define  Uses_XLib
#define  Uses_wxList
#define  Uses_wxMemoryDC
#define  Uses_wxWindowDC
#include "wx.h"

#include <X11/Xatom.h>
#include <math.h>
#include <string.h>
#ifndef PI
#define PI 3.1415926535
#endif

// This normalizes the graphics code to behave in a standard way when
#if WX_STANDARD_GRAPHICS
#define WX_GC_CF 1
#else
#define WX_GC_CF 0
#endif

// constant to convert radian to degree
#define RAD2DEG 57.2957795131

// translate defines from wxDefines.h to XLib constants
static int join_style[] = { JoinBevel, JoinMiter, JoinRound };
static int cap_style[]  = { CapRound, CapProjecting, CapButt, CapNotLast };
static int fill_rule[]  = { EvenOddRule, WindingRule };
static int function[]   = { GXand, GXandInverted, GXandReverse, GXclear, GXcopy,
			    GXequiv, GXinvert, GXnand, GXnor, GXnoop, GXor,
			    GXorInverted, GXorReverse, GXset, GXcopyInverted,
			    GXxor };

// hatches, used for stippling in any DC
#include <bdiag.xbm>
#include <fdiag.xbm>
#include <cdiag.xbm>
#include <horiz.xbm>
#include <verti.xbm>
#include <cross.xbm>
#define  num_hatches 6

static Pixmap  hatches[num_hatches];
static Pixmap* hatch_bitmap = NULL;

//-----------------------------------------------------------------------------
// X internal representation for device contexts
//-----------------------------------------------------------------------------

class wxWindowDC_Xintern { // X GDI data
public:
    GC           pen_gc, brush_gc, text_gc, bg_gc;
    Region       user_reg, expose_reg, current_reg;
    Display      *dpy;
    Screen       *scn;
    Drawable     drawable;
    Window       draw_window;
    unsigned int width, height, depth;
};

// easier access to private data (use protected methods for derived classes)
#define PEN_GC		(X->pen_gc)
#define BRUSH_GC	(X->brush_gc)
#define TEXT_GC		(X->text_gc)
#define BG_GC		(X->bg_gc)
#define USER_REG	(X->user_reg)
#define EXPOSE_REG	(X->expose_reg)
#define CURRENT_REG	(X->current_reg)
#define DPY		(X->dpy)
#define SCN		(X->scn)
#define DRAWABLE	(X->drawable)
#define DRAW_WINDOW	(X->draw_window)
#define WIDTH		(X->width)
#define HEIGHT		(X->height)
#define DEPTH		(X->depth)
#define CMAP            (current_cmap->GetColormap())

//-----------------------------------------------------------------------------
// create and destroy wxWindowDC
//-----------------------------------------------------------------------------

IMPLEMENT_DYNAMIC_CLASS(wxWindowDC, wxDC)

wxWindowDC::wxWindowDC(void) : wxDC()
{
    __type = wxTYPE_DC_CANVAS;

    device = wxDEVICE_CANVAS;

    X = wxNEW wxWindowDC_Xintern; // allocate space for X data

    PEN_GC = BRUSH_GC = TEXT_GC = BG_GC = NULL;
    USER_REG = EXPOSE_REG = CURRENT_REG = NULL;
    DPY = NULL;
    SCN = NULL;
    DRAWABLE = 0;
    DRAW_WINDOW = 0;
    WIDTH = HEIGHT = DEPTH = 0;

    if (!hatch_bitmap) {
	Display *dpy    = wxAPP_DISPLAY;
	Window  win     = RootWindow(dpy, DefaultScreen(dpy));
	hatch_bitmap    = hatches;
	hatch_bitmap[0] = XCreateBitmapFromData(dpy, win, bdiag_bits,
						bdiag_width, bdiag_height);
	hatch_bitmap[1] = XCreateBitmapFromData(dpy, win, cdiag_bits,
						cdiag_width, cdiag_height);
	hatch_bitmap[2] = XCreateBitmapFromData(dpy, win, fdiag_bits,
						fdiag_width, fdiag_height);
	hatch_bitmap[3] = XCreateBitmapFromData(dpy, win, cross_bits,
						cross_width, cross_height);
	hatch_bitmap[4] = XCreateBitmapFromData(dpy, win, horiz_bits,
						horiz_width, horiz_height);
	hatch_bitmap[5] = XCreateBitmapFromData(dpy, win, verti_bits,
						verti_width, verti_height);
    }
}

wxWindowDC::~wxWindowDC(void)
{
    Destroy();

    delete X; // free space for X data
}

//-----------------------------------------------------------------------------
// drawing methods
//-----------------------------------------------------------------------------

Bool wxWindowDC::Blit(float xdest, float ydest, float w, float h, wxDC *_src,
		      float xsrc, float ysrc, int rop)
{
    // I need drawables for this operation
    if (!DRAWABLE || !_src->IsKindOf(CLASSINFO(wxWindowDC)) || !((wxWindowDC*)_src)->DrawArea())
	return FALSE;

    wxWindowDC *src = (wxWindowDC*)_src;
    Bool ret_value = FALSE;

    // source and destination coordinates on devices,
    // adjusted for XGetImage keep the margins inside the pixmap/window
    // the root is used as XGetImage source, because a window must not be
    // off screen, and this may happen to a canvas.
    int _xsrc  = wxMax(0, src->XLOG2DEV(xsrc));
    int _ysrc  = wxMax(0, src->YLOG2DEV(ysrc));
    int _xdest = XLOG2DEV(xdest);
    int _ydest = YLOG2DEV(ydest);   
    int _wsrc  = wxMax(1, src->XLOG2DEVREL(w));
	_wsrc  = wxMin(_wsrc,  int(src->Width()  - _xsrc));
    int _hsrc  = wxMax(1, src->YLOG2DEVREL(h));
	_hsrc  = wxMin(_hsrc,  int(src->Height() - _ysrc));
    int _wdest = wxMax(1, XLOG2DEVREL(_wsrc));
	_wdest = wxMin(_wdest, DisplayWidth (DPY, DefaultScreen(DPY)));
    int _hdest = wxMax(1, YLOG2DEVREL(_hsrc));
	_hdest = wxMin(_wdest, DisplayHeight(DPY, DefaultScreen(DPY)));

    // set given logical function
    int old_logical_fkt = current_logical_fkt;
    if (current_logical_fkt != rop)
	SetLogicalFunction(rop);

    if (do_gdi_scaling && (_wsrc != _wdest || _hsrc != _hdest)) {
	// test for exposure since working with images is very SLOW
	if (!EXPOSE_REG || XRectInRegion(EXPOSE_REG, _xdest, _ydest, _wdest, _hdest)) {
	    // The device context is scaled => scale bitmap
	    XImage *img1=NULL, *img2=NULL;
	    if ((img1 = XGetImage(src->Dpy(), src->DrawArea(),
				  _xsrc, _ysrc, _wsrc, _hsrc, AllPlanes, ZPixmap))) {
		if ((img2 = XGetImage(DPY, XRootWindowOfScreen(SCN),
				      0, 0, _wdest, _hdest, AllPlanes, ZPixmap))) {
		    // Transfer images
		    for (int xx1 = 0; xx1 < _wdest; ++xx1)
			for (int yy1 = 0; yy1 < _hdest; ++yy1)
			    XPutPixel(img2, xx1, yy1,
				      XGetPixel(img1, xx1*_wsrc/_wdest, yy1*_hsrc/_hdest));
		    // Put image into drawable and destroy images
		    XPutImage(DPY, DRAWABLE, PEN_GC, img2, 0, 0,
			      _xdest, _ydest, _wdest, _hdest);
		    ret_value = TRUE; // DONE
		    XDestroyImage(img2);
		}
		XDestroyImage(img1);
	    }
	}
    } else if (src->Depth() == 1) {
	// mono bitmap
	XCopyPlane(DPY, src->DrawArea(), DRAWABLE, PEN_GC,
		   _xsrc, _ysrc, _wsrc, _hsrc, _xdest, _ydest, 1);
	ret_value = TRUE; // DONE
    } else if (src->Depth() == DEPTH) {
	// depth of bitmap == depth of screen
	XCopyArea(DPY, src->DrawArea(), DRAWABLE, PEN_GC,
		  _xsrc, _ysrc, _wsrc, _hsrc, _xdest, _ydest);
	ret_value = TRUE; // DONE
    }

    // reset old logical function
    if (current_logical_fkt != old_logical_fkt)
	SetLogicalFunction(old_logical_fkt);

    CalcBoundingBox(xdest, ydest);
    CalcBoundingBox(xdest + w, ydest + h);

    return ret_value;
}

void wxWindowDC::Clear(void)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    // clear means to clear the entire canvas without expose region clipping
    EXPOSE_REG = NULL;
    SetCanvasClipping();
    // clear canvas
    if (DRAW_WINDOW)
	XClearWindow(DPY, DRAW_WINDOW);
    else {
	float w; float h;
	GetSize(&w, &h);
	XFillRectangle(DPY, DRAWABLE, BG_GC,
		       0, 0, XLOG2DEVREL(w), YLOG2DEVREL(h));
    }
}

void wxWindowDC::CrossHair(float x, float y)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    if (!current_pen || current_pen->GetStyle() == wxTRANSPARENT)
	return;
    int xx = XLOG2DEV(x);
    int yy = YLOG2DEV(y);
    float w; float h;
    GetSize(&w, &h);
    XDrawLine(DPY, DRAWABLE, PEN_GC, 0, yy, XLOG2DEVREL(w), yy);
    XDrawLine(DPY, DRAWABLE, PEN_GC, xx, 0, xx, YLOG2DEVREL(h));
}

void wxWindowDC::DrawArc(float x1, float y1, float x2, float y2,
			 float xc, float yc)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    int xx1 = XLOG2DEV(x1); int yy1 = YLOG2DEV(y1);
    int xx2 = XLOG2DEV(x2); int yy2 = YLOG2DEV(y2);
    int xxc = XLOG2DEV(xc); int yyc = YLOG2DEV(yc);
    double dx = xx1 - xxc; double dy = yy1 - yyc;
    double radius = sqrt(dx*dx+dy*dy);
    int    r      = (int)radius;
    double radius1, radius2;

    if (xx1 == xx2 && yy1 == yy2) {
	radius1 = 0.0;
	radius2 = 360.0;
    } else if (radius == 0.0) {
	radius1 = radius2 = 0.0;
    } else {
	radius1 = (xx1 - xxc == 0) ?
	    (yy1 - yyc < 0) ? 90.0 : -90.0 :
	    -atan2(double(yy1-yyc), double(xx1-xxc)) * RAD2DEG;
	radius2 = (xx2 - xxc == 0) ?
	    (yy2 - yyc < 0) ? 90.0 : -90.0 :
	    -atan2(double(yy2-yyc), double(xx2-xxc)) * RAD2DEG;
    }
    int alpha1 = int(radius1 * 64.0);
    int alpha2 = int((radius2 - radius1) * 64.0);
    while (alpha2 <= 0)
	alpha2 += 360*64;
    while (alpha1 > 360*64)
	alpha1 -= 360*64;

    if (current_brush && current_brush->GetStyle() != wxTRANSPARENT)
	XFillArc(DPY,DRAWABLE,BRUSH_GC,xxc-r,yyc-r,2*r,2*r,alpha1,alpha2);
    if (current_pen && current_pen->GetStyle() != wxTRANSPARENT)
	XDrawArc(DPY,DRAWABLE,PEN_GC,xxc-r,yyc-r,2*r,2*r,alpha1,alpha2);
    CalcBoundingBox(x1, y1);
    CalcBoundingBox(x2, y2);
}

void wxWindowDC::DrawBitmap8(unsigned char *data, int w, int h,
			     wxColourMap *cmap, float x, float y)
{
    XWindowAttributes xwa;
    XVisualInfo *vinfo, vi_templ;
    XImage *ximage;
    char *ximage_data;
    int ni, bpp; // bits per pixel

    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    // Get the window's visual.
    XGetWindowAttributes(DPY, DRAWABLE, &xwa);
    vi_templ.visualid = XVisualIDFromVisual(xwa.visual);
    vinfo = XGetVisualInfo(DPY, VisualIDMask, &vi_templ, &ni);
    if (!vinfo) return;
    bpp = ((vinfo->depth+7)/8)*8;
    if (bpp == 24) bpp = 32;

    // Map the bitmap through the colormap.
    if (cmap) {
	ximage_data = new char[(long)w*h*(bpp/8)];
	if (!ximage_data) {
	    XFree(vinfo);
	    return;
	}
	if (!cmap->TransferBitmap8(data, (long)w*h,
				   ximage_data, bpp)) {
	    wxError("can't handle display depth", "wxWindowDC");
	    delete ximage_data;
	    XFree(vinfo);
	    return;
	}
    } else {
	if (bpp != 8) {
	    wxError("can't handle display depth", "wxWindowDC");
	    XFree(vinfo);
	    return;
	}
	ximage_data = (char *)data;
    }

    ximage = XCreateImage(DPY, vinfo->visual, vinfo->depth, ZPixmap, 0,
			  ximage_data, w, h, 8, /* pad in bits */
			  w*(bpp/8)); /* bytes per line */
    XFree(vinfo);
    if (!ximage) {
	delete ximage_data;
	return;
    }

    // Draw the XImage on the canvas.
    SetLogicalFunction(wxCOPY);
    XPutImage(DPY, DRAWABLE, PEN_GC,
	      ximage, 0, 0, // src
	      XLOG2DEV(x), YLOG2DEV(y), // dest
	      ximage->width, ximage->height);

    // Free the XImage structure.
    if (cmap) delete ximage->data;
    ximage->data = NULL;
    XDestroyImage(ximage);
}

void wxWindowDC::DrawEllipse(float x, float y, float w, float h,
			     int startAngle, int sizeAngle)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    if (w<0) { w=-w; x=x-w; }
    if (h<0) { h=-h; y=y-h; }

    if (current_brush && current_brush->GetStyle() != wxTRANSPARENT) {
	XFillArc(DPY, DRAWABLE, BRUSH_GC, XLOG2DEV(x), YLOG2DEV(y),
		 XLOG2DEVREL(w), YLOG2DEVREL(h), startAngle, sizeAngle);
    }
    if (current_pen && current_pen->GetStyle() != wxTRANSPARENT) {
	XDrawArc(DPY, DRAWABLE, PEN_GC, XLOG2DEV(x), YLOG2DEV(y),
		 XLOG2DEVREL(w) - WX_GC_CF, YLOG2DEVREL(h) - WX_GC_CF,
		 startAngle, sizeAngle);
	if (sizeAngle % (360*64)) {
	    float cx = x + w/2;
	    float cy = y + h/2;
	    XPoint pts[3]; 
	    pts[1].x = XLOG2DEV(cx); // center point
	    pts[1].y = YLOG2DEV(cy);
	    pts[0].x = XLOG2DEV(cx+cos(startAngle*PI/(64*180))*w/2);
	    pts[0].y = YLOG2DEV(cy-sin(startAngle*PI/(64*180))*h/2);
	    pts[2].x = XLOG2DEV(cx+cos((startAngle+sizeAngle)*PI/(64*180))*w/2);
	    pts[2].y = YLOG2DEV(cy-sin((startAngle+sizeAngle)*PI/(64*180))*h/2);
	    XDrawLines(DPY, DRAWABLE, PEN_GC, pts, 3, CoordModeOrigin);
	}
    }
    CalcBoundingBox(x, y);
    CalcBoundingBox(x+w, y+h);
}

void wxWindowDC::DrawIcon(wxBitmap *icon, float x, float y, Bool useMask)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;
    if ( !(icon->Ok()) ) // ensure correct icon
	return;

    int    ix   = XLOG2DEV(x);
    int    iy   = YLOG2DEV(y);
    int    w    = icon->GetWidth();
    int    h    = icon->GetHeight();
    Pixmap mask = icon->GetMask();
    
    //    if (XRectInRegion(CURRENT_REG, ix, iy, w, h)) {
    // set mask region
    if (useMask && mask) {
	XSetClipMask(DPY, PEN_GC, mask);
	XSetClipOrigin(DPY, PEN_GC, ix, iy);
    }
    // draw bitmap
    if (icon->GetDepth() == 1) {
	// mono bitmap
	XCopyPlane(DPY, icon->GetPixmap(), DRAWABLE, PEN_GC,
		   0, 0, icon->GetWidth(), icon->GetHeight(), ix, iy, 1);
    } else if (icon->GetDepth() == (int)DEPTH) {
	// set mask region
	// depth of bitmap == depth of screen
	XCopyArea(DPY, icon->GetPixmap(), DRAWABLE, PEN_GC,
		  0, 0, icon->GetWidth(), icon->GetHeight(), ix, iy);
    }
    // reset region
    if (useMask && mask)
	SetCanvasClipping(); //XSetRegion(DPY, PEN_GC, CURRENT_REG);
    // }

    CalcBoundingBox(x, y);
    CalcBoundingBox(x + XDEV2LOGREL(w), y + YDEV2LOGREL(h));
}

void wxWindowDC::DrawLine(float x1, float y1, float x2, float y2)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    if (current_pen && current_pen->GetStyle() != wxTRANSPARENT)
	XDrawLine(DPY, DRAWABLE, PEN_GC,
		  XLOG2DEV(x1), YLOG2DEV(y1), XLOG2DEV(x2), YLOG2DEV(y2));
    CalcBoundingBox(x1, y1);
    CalcBoundingBox(x2, y2);
}

void wxWindowDC::DrawLines(int n, wxPoint pts[], float xoff, float yoff)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    XPoint *xpts = wxNEW XPoint[n];
    for (int i=0; i<n; ++i)
	CalcBoundingBox(xpts[i].x = XLOG2DEV(pts[i].x + xoff),
			xpts[i].y = YLOG2DEV(pts[i].y + yoff));
    if (current_pen && current_pen->GetStyle() != wxTRANSPARENT)
	XDrawLines(DPY, DRAWABLE, PEN_GC, xpts, n, 0);
    delete[] xpts;
}

void wxWindowDC::DrawLines(int n, wxIntPoint pts[], int xoff, int yoff)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    XPoint *xpts = wxNEW XPoint[n];
    for (int i=0; i<n; ++i)
	CalcBoundingBox(xpts[i].x = XLOG2DEV(pts[i].x + xoff),
			xpts[i].y = YLOG2DEV(pts[i].y + yoff));
    if (current_pen && current_pen->GetStyle() != wxTRANSPARENT)
	XDrawLines(DPY, DRAWABLE, PEN_GC, xpts, n, 0);
    delete[] xpts;
}

void wxWindowDC::DrawLines(wxList *pts, float xoff, float yoff)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    int n = pts->Number();
    XPoint *xpts = wxNEW XPoint[n];
    int i = 0;
    for (wxNode *node=pts->First(); node; node=node->Next()) {
	wxPoint *point = (wxPoint*)node->Data();
	CalcBoundingBox(xpts[i].x = XLOG2DEV(point->x + xoff),
			xpts[i].y = YLOG2DEV(point->y + yoff));
	++i;
    }
    if (current_pen && current_pen->GetStyle() != wxTRANSPARENT)
	XDrawLines(DPY, DRAWABLE, PEN_GC, xpts, n, 0);
    delete[] xpts;
}

void wxWindowDC::DrawPoint(float x, float y)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    if (current_pen && current_pen->GetStyle() != wxTRANSPARENT)
	XDrawPoint(DPY, DRAWABLE, PEN_GC, XLOG2DEV(x), YLOG2DEV(y));
    CalcBoundingBox(x, y);
}

void wxWindowDC::DrawPolygon(int n, wxPoint pts[], float xoff, float yoff,
			     int fill)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    XPoint *xpts = wxNEW XPoint[n+1];
    for (int i=0; i<n; ++i)
	CalcBoundingBox(xpts[i].x = XLOG2DEV(pts[i].x + xoff),
			xpts[i].y = YLOG2DEV(pts[i].y + yoff));
    xpts[n].x = xpts[0].x; // close figure
    xpts[n].y = xpts[0].y;
    if (current_brush && current_brush->GetStyle() != wxTRANSPARENT) {
	XSetFillRule(DPY, PEN_GC, fill_rule[fill]);
	XFillPolygon(DPY, DRAWABLE, BRUSH_GC, xpts, n, Complex, 0);
    }
    if (current_pen && current_pen->GetStyle() != wxTRANSPARENT)
	XDrawLines(DPY, DRAWABLE, PEN_GC, xpts, n+1, 0);
    delete[] xpts;
}

void wxWindowDC::DrawPolygon(wxList *pts, float xoff, float yoff, int fill)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    int n = pts->Number();
    XPoint *xpts = wxNEW XPoint[n+1];
    int i=0;
    for(wxNode *node=pts->First(); node; node=node->Next()) {
	wxPoint *point = (wxPoint*)node->Data();
	CalcBoundingBox(xpts[i].x = XLOG2DEV(point->x + xoff),
			xpts[i].y = YLOG2DEV(point->y + yoff));
	++i;
    }
    xpts[n].x = xpts[0].x; // close figure
    xpts[n].y = xpts[0].y;
    if (current_brush && current_brush->GetStyle() != wxTRANSPARENT) {
	XSetFillRule(DPY, PEN_GC, fill_rule[fill]);
	XFillPolygon(DPY, DRAWABLE, BRUSH_GC, xpts, n, Complex, 0);
    }
    if (current_pen && current_pen->GetStyle() != wxTRANSPARENT)
	XDrawLines(DPY, DRAWABLE, PEN_GC, xpts, n+1, 0);
    delete[] xpts;
}

void wxWindowDC::DrawRectangle(float x, float y, float w, float h)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    if (w<0) { w=-w; x=x-w; }
    if (h<0) { h=-h; y=y-h; }

    if (current_brush && current_brush->GetStyle() != wxTRANSPARENT)
	XFillRectangle(DPY, DRAWABLE, BRUSH_GC, XLOG2DEV(x), YLOG2DEV(y),
		       XLOG2DEVREL(w), YLOG2DEVREL(h));
    if (current_pen && current_pen->GetStyle() != wxTRANSPARENT)
	XDrawRectangle(DPY, DRAWABLE, PEN_GC, XLOG2DEV(x), YLOG2DEV(y),
		       XLOG2DEVREL(w) - WX_GC_CF, YLOG2DEVREL(h) - WX_GC_CF);
    CalcBoundingBox(x, y);
    CalcBoundingBox(x+w, y+h);
}

void wxWindowDC::DrawRoundedRectangle(float x, float y, float w, float h,
				      float radius)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    if (w<0) { w=-w; x=x-w; }
    if (h<0) { h=-h; y=y-h; }

    if (radius < 0.0)
	radius = - radius * ((w < h) ? w : h);
    int xx = XLOG2DEV(x);    int yy = YLOG2DEV(y);
    int ww = XLOG2DEVREL(w); int hh = YLOG2DEVREL(h);
    int rr = XLOG2DEVREL(radius);
    int dd = 2 * rr;

    if (current_brush && current_brush->GetStyle() != wxTRANSPARENT) {
	XFillRectangle(DPY, DRAWABLE, BRUSH_GC, xx+rr, yy, ww-dd, hh);
	XFillRectangle(DPY, DRAWABLE, BRUSH_GC, xx, yy+rr, ww, hh-dd);
	XFillArc(DPY, DRAWABLE, BRUSH_GC, xx, yy, dd, dd, 90*64, 90*64);
	XFillArc(DPY, DRAWABLE, BRUSH_GC, xx+ww-dd, yy, dd, dd, 0, 90*64);
	XFillArc(DPY, DRAWABLE, BRUSH_GC, xx+ww-dd, yy+hh-dd, dd, dd,
		 270*64, 90*64);
	XFillArc(DPY, DRAWABLE, BRUSH_GC, xx, yy+hh-dd, dd, dd, 180*64, 90*64);
    }
    if (current_pen && current_pen->GetStyle() != wxTRANSPARENT){
        ww -= WX_GC_CF;
        hh -= WX_GC_CF;
	XDrawLine(DPY, DRAWABLE, PEN_GC, xx+rr, yy, xx+ww-rr, yy);
	XDrawLine(DPY, DRAWABLE, PEN_GC, xx+rr, yy+hh, xx+ww-rr, yy+hh);
	XDrawLine(DPY, DRAWABLE, PEN_GC, xx, yy+rr, xx, yy+hh-rr);
	XDrawLine(DPY, DRAWABLE, PEN_GC, xx+ww, yy+rr, xx+ww, yy+hh-rr);
	XDrawArc(DPY, DRAWABLE, PEN_GC, xx, yy, dd, dd, 90*64, 90*64);
	XDrawArc(DPY, DRAWABLE, PEN_GC, xx+ww-dd, yy, dd, dd, 0, 90*64);
	XDrawArc(DPY, DRAWABLE, PEN_GC, xx+ww-dd, yy+hh-dd, dd, dd,
		 270*64, 90*64);
	XDrawArc(DPY, DRAWABLE, PEN_GC, xx, yy+hh-dd, dd, dd, 180*64, 90*64);
    }
    CalcBoundingBox(x, y);
    CalcBoundingBox(x+w, y+h);
}

void wxWindowDC::FloodFill(float WXUNUSED(x), float WXUNUSED(y),
			   wxColour *WXUNUSED(col),int WXUNUSED(style))
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    // don't know how to do it for X11
}

void wxWindowDC::IntDrawLine(int x1, int y1, int x2, int y2)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    if (current_pen && current_pen->GetStyle() != wxTRANSPARENT)
	XDrawLine(DPY, DRAWABLE, PEN_GC,
		  XLOG2DEV(x1), YLOG2DEV(y1), XLOG2DEV(x2), YLOG2DEV(y2));
}

void wxWindowDC::IntDrawLines(int n, wxIntPoint pts[], int xoff, int yoff)
{
    if (!DRAWABLE) // ensure that a drawable has been associated
	return;

    XPoint *xpts = wxNEW XPoint[n];
    for (int i=0; i<n; ++i)
	CalcBoundingBox(xpts[i].x = XLOG2DEV(pts[i].x + xoff),
			xpts[i].y = YLOG2DEV(pts[i].y + yoff));
    if (current_pen && current_pen->GetStyle() != wxTRANSPARENT)
	XDrawLines(DPY, DRAWABLE, PEN_GC, xpts, n, 0);
    delete[] xpts;
}

//-----------------------------------------------------------------------------
// drawing tools
//-----------------------------------------------------------------------------

void wxWindowDC::SetBackground(wxBrush *brush)
{
    if (!(current_background_brush = brush)) // nothing to do without brush
	return;
    
    unsigned long pixel = brush->GetColour().GetPixel(current_cmap);

    if (DRAW_WINDOW)
	XSetWindowBackground(DPY, DRAW_WINDOW, pixel);
    else
	XSetForeground(DPY, BG_GC, pixel);
    XSetBackground(DPY, PEN_GC, pixel);
    XSetBackground(DPY, BRUSH_GC, pixel);

    if (current_logical_fkt == wxXOR) { // use the correct pixel values for XOR
	SetPen(current_pen);
	SetBrush(current_brush);
    }
}

void wxWindowDC::SetBrush(wxBrush *brush)
{
    if (!(current_brush = brush)) // nothing to do without brush
	return;
    // for XChangeGC
    XGCValues     values;
    unsigned long mask = GCFillStyle | GCForeground;

    values.fill_style = FillSolid;
    // wxXOR shall work the correct way
    unsigned long pixel = brush->GetColour().GetPixel(current_cmap);
    if (current_logical_fkt == wxXOR) {
	XGCValues values_req;
	XGetGCValues(DPY, BRUSH_GC, GCBackground, &values_req);
	values.foreground = pixel ^ values_req.background;
    } else
	values.foreground = pixel;

    int style = brush->GetStyle();
    switch (style) {
    case wxSOLID: case wxTRANSPARENT:
	break;
    default:
	if (wxIS_HATCH(style) || style==wxSTIPPLE) {
	    Pixmap stipple = (Pixmap)0; // for FillStippled
	    Pixmap tile    = (Pixmap)0; // for FillTiled
	    if (style==wxSTIPPLE) {
		if (brush->GetStipple()->Ok()) {
		    if (brush->GetStipple()->GetDepth() == 1) {
			stipple = brush->GetStipple()->GetPixmap();
			values.fill_style = FillStippled;
		    } else if (brush->GetStipple()->GetDepth() == (signed)DEPTH) {
			tile = brush->GetStipple()->GetPixmap();
			values.fill_style = FillTiled;
		    } // else wrong depth
		}
	    } else {
		stipple = hatch_bitmap[style-wxFIRST_HATCH];
		values.fill_style = FillStippled;
	    }
	    if (stipple) XSetStipple(DPY, BRUSH_GC, stipple);
	    if (tile)    XSetTile(DPY, BRUSH_GC, tile);
	}
	break;
    }
    XChangeGC(DPY, BRUSH_GC, mask, &values);
}

void wxWindowDC::SetColourMap(wxColourMap *new_cmap)
{
    current_cmap = new_cmap ? new_cmap : wxAPP_COLOURMAP;

    if (DRAW_WINDOW)
	XSetWindowColormap(DPY, DRAW_WINDOW, CMAP);
}

void wxWindowDC::SetLogicalFunction(int fkt)
{
    if (current_logical_fkt == fkt)
	return;
    Bool must_reset_tools = ((fkt==wxXOR) || (current_logical_fkt==wxXOR));
    XSetFunction(DPY, PEN_GC, function[fkt]);
    XSetFunction(DPY, BRUSH_GC, function[fkt]);
    current_logical_fkt = fkt;
    if (must_reset_tools) {
	// function has changed from or to XOR -> use the correct pixel values
	SetPen(current_pen);
	SetBrush(current_brush);
    }
}

void wxWindowDC::SetPen(wxPen *pen)
{
    if (!(current_pen = pen)) // nothing to do without pen
	return;
    // for XChangeGC
    XGCValues     values;
    unsigned long mask = GCCapStyle  | GCFillStyle | GCForeground |
	                 GCJoinStyle | GCLineStyle | GCLineWidth;

    values.cap_style  = cap_style[pen->GetCap()];
    values.fill_style = FillSolid;
    values.join_style = join_style[pen->GetJoin()];
    values.line_style = LineSolid;
    values.line_width = (do_gdi_scaling ? XLOG2DEVREL(pen->GetWidth()) : pen->GetWidth());
    // wxXOR shall work the correct way
    unsigned long pixel = pen->GetColour().GetPixel(current_cmap);
    if (current_logical_fkt == wxXOR) {
	XGCValues values_req;
	XGetGCValues(DPY, PEN_GC, GCBackground, &values_req);
	values.foreground = pixel ^ values_req.background;
	if (values.foreground == 0)
	    values.foreground = 1; // solve problem with XOR 0
    } else
	values.foreground = pixel;

    int style = pen->GetStyle();
    switch (style) {
    case wxSOLID: case wxTRANSPARENT:
	break;
    default:
	if (wxIS_DASH(style) || style==wxUSER_DASH) {
	    static wxDash dashdefs[4][4] = {
		{ 2, 5, 0, 0 }, // wxDOT
		{ 4, 8, 0, 0 }, // wxLONG_DASH
		{ 4, 4, 0, 0 }, // wxSHORT_DASH
		{ 6, 6, 2, 6 }  // wxDOT_DASH
	    };
	    static int    num_dashes[] = { 2, 2, 2, 4 };
	    int           num_dash;
	    wxDash        *dashdef, *scaleddef;
	    if (style==wxUSER_DASH) {
		num_dash = pen->GetDashes(&dashdef);
	    } else {
		num_dash = num_dashes[style-wxFIRST_DASH];
		dashdef  = dashdefs[style-wxFIRST_DASH];
	    }
	    if (do_gdi_scaling && (scaleddef = wxNEW wxDash[num_dash])) {
		for (int i=0; i<num_dash; ++i)
		    scaleddef[i] = wxDash(scale_x * dashdef[i] + 0.5);
		XSetDashes(DPY, PEN_GC, 0, (char*)scaleddef, num_dash);
		delete[] scaleddef;
	    } else { // don't scale or not enough memory to scale
		XSetDashes(DPY, PEN_GC, 0, (char*)dashdef, num_dash);
	    }
	    values.line_style = LineOnOffDash; // for XChangeGC
	}
	if (wxIS_HATCH(style) || style==wxSTIPPLE) {
	    Pixmap stipple = (Pixmap)0; // for FillStippled
	    Pixmap tile    = (Pixmap)0; // for FillTiled
	    if (style==wxSTIPPLE) {
		if (pen->GetStipple()->Ok()) {
		    if (pen->GetStipple()->GetDepth() == 1) {
			stipple = pen->GetStipple()->GetPixmap();
			values.fill_style = FillStippled;
		    } else if (pen->GetStipple()->GetDepth() == (signed)DEPTH) {
			tile = pen->GetStipple()->GetPixmap();
			values.fill_style = FillTiled;
		    } // else wrong depth
		}
	    } else {
		stipple = hatch_bitmap[style-wxFIRST_HATCH];
		values.fill_style = FillStippled;
	    }
	    if (stipple) XSetStipple(DPY, PEN_GC, stipple);
	    if (tile)    XSetTile(DPY, PEN_GC, tile);
	}
	break;
    }
    XChangeGC(DPY, PEN_GC, mask, &values);
}

//-----------------------------------------------------------------------------
// text and font methods
//-----------------------------------------------------------------------------

void wxWindowDC::DrawText(const char *text, float x, float y, Bool use16Bit)
{
    if (!DRAWABLE || !text) // ensure that a drawable has been associated
	return;

    if (!current_font) // a font must be associated for drawing
	return;

    XFontStruct *fontinfo
	= (XFontStruct*)current_font->GetInternalFont(do_gdi_scaling ? scale_y : 1.0);
    int         direction;
    int         ascent, descent;
    int         dev_x = XLOG2DEV(x);
    int         dev_y = YLOG2DEV(y);
    int         textlen = use16Bit ? str16len(text) : strlen(text);
    XCharStruct overall;

    // XDrawString draws on the baseline. make adaptions depending
    // on current_text_alignment;
    if (use16Bit)
	XTextExtents16(fontinfo, (XChar2b*)text, textlen, &direction, &ascent, &descent, &overall);
    else
	XTextExtents(fontinfo, text, textlen, &direction, &ascent, &descent, &overall);
    if (current_text_alignment & wxALIGN_TOP) {
	dev_y += ascent;
    } else if (current_text_alignment & wxALIGN_BOTTOM) {
	dev_y -= descent; y -= YLOG2DEVREL(descent+ascent);
    } else {
	y -= YLOG2DEVREL(ascent);
    }
    if (current_text_alignment & wxALIGN_CENTRE) {
	dev_x -= overall.width/2; x -= XLOG2DEVREL(overall.width/2);
    } else if(current_text_alignment & wxALIGN_RIGHT) {
	dev_x -= overall.width;   x -= XLOG2DEVREL(overall.width);
    }
    // Draw String
    if (use16Bit) {
	if (current_text_bgmode != wxTRANSPARENT)
	    XDrawImageString16(DPY, DRAWABLE, TEXT_GC, dev_x, dev_y, (XChar2b*)text, textlen);
	else
	    XDrawString16(DPY, DRAWABLE, TEXT_GC, dev_x, dev_y, (XChar2b*)text, textlen);
    } else {
	if (current_text_bgmode != wxTRANSPARENT)
	    XDrawImageString(DPY, DRAWABLE, TEXT_GC, dev_x, dev_y, text, textlen);
	else
	    XDrawString(DPY, DRAWABLE, TEXT_GC, dev_x, dev_y, text, textlen);
    }
    // Draw underline
    if (current_font->GetUnderlined()) {
	unsigned long upos, uthick;
	if (!XGetFontProperty(fontinfo, XA_UNDERLINE_POSITION, &upos)) {
	    // not found property, use descent as default
	    upos = descent;
	}
	if (!XGetFontProperty(fontinfo, XA_UNDERLINE_THICKNESS, &uthick)) {
	    // not found property, use 1 (probably scaled) as default
	    uthick = (do_gdi_scaling ? int(scale_y+0.5) : 1);
	}
	if (uthick >= 1 && overall.width >= 1)
	    XFillRectangle(DPY, DRAWABLE, TEXT_GC,
			   dev_x,         dev_y + upos,
			   overall.width, uthick);
    }
    // calculate boundling box
    CalcBoundingBox(x, y);
    CalcBoundingBox(x+XDEV2LOGREL(overall.width), y+YDEV2LOGREL(ascent+descent));
}

float wxWindowDC::GetCharHeight(void)
{
    if (!current_font) // no font
	return YDEV2LOGREL(12);

    int         direction, ascent, descent;
    XCharStruct overall;
    XTextExtents ((XFontStruct*)current_font->GetInternalFont(do_gdi_scaling ? scale_y : 1.0),
		  "x", 1, &direction, &ascent, &descent, &overall);
    return YDEV2LOGREL(ascent + descent);
}

float wxWindowDC::GetCharWidth(void)
{
    if (!current_font)
	return XDEV2LOGREL(16);

    int         direction, ascent, descent;
    XCharStruct overall;
    XTextExtents ((XFontStruct*)current_font->GetInternalFont(do_gdi_scaling ? scale_y : 1.0),
		  "x", 1, &direction, &ascent, &descent, &overall);
    return XDEV2LOGREL(overall.width);
}

void wxWindowDC::GetTextExtent(const char *s, float *_w, float *_h, float *_descent,
			       float *_ext_leading, wxFont *_font, Bool use16bit)
{
    if (s == NULL) {
	*_w = *_h = 0.0;
	return;
    }

    wxFont *font_to_use = _font ? _font : current_font;
    if (!font_to_use) {
	wxError("set a font before calling GetTextExtent", "wxWindowDC");
	*_w = *_h = -1.0;
	return;
    }

    int         direction, ascent, descent;
    XCharStruct overall;
    XFontStruct *fontinfo = (XFontStruct*)font_to_use->GetInternalFont(do_gdi_scaling ? scale_y : 1.0);
    if (use16bit)
	XTextExtents16(fontinfo, (XChar2b*)s, str16len(s), &direction, &ascent, &descent, &overall);
    else
	XTextExtents(fontinfo, s, strlen(s), &direction, &ascent, &descent, &overall);
    *_w = XDEV2LOGREL(overall.width);
    *_h = YDEV2LOGREL(ascent + descent);
    if (_descent)
	*_descent = YDEV2LOGREL(descent);
    if (_ext_leading)
	*_ext_leading = 0.0;
}

void wxWindowDC::SetFont(wxFont *font)
{
    if (!(current_font = font)) // nothing to do without a font
	return;

    XSetFont(DPY, TEXT_GC,
	     ((XFontStruct*)font->GetInternalFont(do_gdi_scaling ? scale_y : 1.0))->fid);
}

void wxWindowDC::SetTextForeground(wxColour *col)
{
    if (!col)
	return;
    current_text_fg = *col;
    XSetForeground(DPY, TEXT_GC, current_text_fg.GetPixel(current_cmap));
}

void wxWindowDC::SetTextBackground(wxColour *col)
{
    if (!col)
	return;
    current_text_bg = *col;
    XSetBackground(DPY, TEXT_GC, current_text_bg.GetPixel(current_cmap));
}

//-----------------------------------------------------------------------------
// clipping region
//-----------------------------------------------------------------------------

void wxWindowDC::GetClippingRegion(float *x, float *y, float *w, float *h)
{
    if (USER_REG) {
	XRectangle r;
	XClipBox(USER_REG, &r);
	*x = XDEV2LOG(r.x);
	*y = YDEV2LOG(r.y);
	*w = XDEV2LOGREL(r.width);
	*h = YDEV2LOGREL(r.height);
    } else {
	*x = *y = 0;
	*w = *h = -1;
    }
}

void wxWindowDC::SetClippingRegion(float x, float y, float w, float h)
{
    XRectangle r;

    if (USER_REG)
	XDestroyRegion(USER_REG);
    USER_REG = XCreateRegion();
    r.x      = XLOG2DEV(x);
    r.y      = YLOG2DEV(y);
    r.width  = XLOG2DEVREL(w);
    r.height = YLOG2DEVREL(h);
    XUnionRectWithRegion(&r, USER_REG, USER_REG);
    SetCanvasClipping();
}

void wxWindowDC::DestroyClippingRegion(void)
{
    if (USER_REG)
	XDestroyRegion(USER_REG);
    USER_REG = NULL;
    SetCanvasClipping();
}

//-----------------------------------------------------------------------------
// ask, if a part of the DC is exposed
//-----------------------------------------------------------------------------

int wxWindowDC::IsExposed(float x, float y)
{
    if (!EXPOSE_REG)
	return wxEXPOSED_FULL;
    if (XPointInRegion(EXPOSE_REG, XLOG2DEV(x), YLOG2DEV(y)))
	return wxEXPOSED_FULL;
    return wxEXPOSED_NOT;
}

int wxWindowDC::IsExposed(float x, float y, float w, float h)
{
    if (!EXPOSE_REG)
	return wxEXPOSED_FULL;

    int          ix = XLOG2DEV(x);
    int          iy = YLOG2DEV(y);
    unsigned int iw = XLOG2DEVREL(w);
    unsigned int ih = YLOG2DEVREL(h);

    switch (XRectInRegion(EXPOSE_REG, ix, iy, iw, ih)) {
    case RectangleIn:   return wxEXPOSED_FULL;
    case RectanglePart: return wxEXPOSED_PART;
    }
    return wxEXPOSED_NOT;
}

//-----------------------------------------------------------------------------
// necessary, if drawing needs server grabbing (e.g. rubber-bands)
//-----------------------------------------------------------------------------

void wxWindowDC::BeginDrawing(void)
{
    // only grab server, if drawing is possible
    if (!DRAWABLE)
	return;
    XGrabServer(DPY);
}

void wxWindowDC::EndDrawing(void)
{
    // only ungrab server if drawing is possible
    if (!DRAWABLE)
	return;
    XUngrabServer(DPY);
}

//-----------------------------------------------------------------------------
// get size of device context
//-----------------------------------------------------------------------------

void wxWindowDC::GetSize(float *w, float *h)
{
//     if (!WIDTH || !HEIGHT) {
	if (DPY && DRAWABLE) {
	    int sdummy; unsigned int udummy; Window wdummy;
	    XGetGeometry(DPY, DRAWABLE, &wdummy, &sdummy, &sdummy,
			 &WIDTH, &HEIGHT, &udummy, &DEPTH);
	} else {
	    wxDC::GetSize(w, h);
	    return;
	}
//     }

    if (w) *w = XDEV2LOGREL(WIDTH);
    if (h) *h = YDEV2LOGREL(HEIGHT);
}

//-----------------------------------------------------------------------------
// methods unique to wxWindowDC
//-----------------------------------------------------------------------------

void wxWindowDC::Initialize(Display* dpy, Screen* scn, Drawable drawable)
{
    Drawable GC_drawable; // necessary create a GC (needed for depth,...)

    DPY = dpy; SCN = scn;

    if (drawable) {
	Window wdummy; int sdummy; unsigned int udummy;
	 // I have a specified drawable -> get width, height, and depth
	GC_drawable = DRAWABLE = drawable;
	XGetGeometry(DPY, DRAWABLE, &wdummy, &sdummy, &sdummy,
		     &WIDTH, &HEIGHT, &udummy, &DEPTH);
    } else {
	GC_drawable = wxAPP_ROOT; // defaults to root
	DEPTH = wxDisplayDepth(); // depth is display depth
    }
    Colour = (DEPTH != 1); // accept everything else than depth one as colour display

    XGCValues values; unsigned long mask;
    values.foreground = BlackPixelOfScreen(SCN);
    values.background = WhitePixelOfScreen(SCN);
    values.graphics_exposures = TRUE;
    values.line_width = 1;
    mask = GCForeground | GCBackground | GCGraphicsExposures | GCLineWidth;
    PEN_GC   = XCreateGC(DPY, GC_drawable, mask, &values);
    TEXT_GC  = XCreateGC(DPY, GC_drawable, mask, &values);
    values.foreground = WhitePixelOfScreen(SCN);
    values.background = BlackPixelOfScreen(SCN);
    BG_GC    = XCreateGC(DPY, GC_drawable, mask, &values);
    BRUSH_GC = XCreateGC(DPY, GC_drawable, mask, &values);
    // set drawing tools
    SetTextForeground(wxBLACK);
    SetTextBackground(wxWHITE);
    SetBackground(wxWHITE_BRUSH);
    SetBrush(wxWHITE_BRUSH);
    SetPen(wxBLACK_PEN);
    // set display scaling
    int width  = WidthOfScreen(SCN);
    int height = HeightOfScreen(SCN);
    mm_to_pix_x = float(width)  / float(WidthMMOfScreen(SCN));
    mm_to_pix_y = float(height) / float(HeightMMOfScreen(SCN));
}

void wxWindowDC::Destroy(void)
{
    if (PEN_GC)    XFreeGC(DPY, PEN_GC);
    if (BRUSH_GC)  XFreeGC(DPY, BRUSH_GC);
    if (TEXT_GC)   XFreeGC(DPY, TEXT_GC);
    if (BG_GC)     XFreeGC(DPY, BG_GC);
    PEN_GC = BRUSH_GC = TEXT_GC = BG_GC = NULL;
}

void wxWindowDC::SetCanvasClipping(void)
{
    if (CURRENT_REG)
	XDestroyRegion(CURRENT_REG);
    if (USER_REG || EXPOSE_REG) {
	CURRENT_REG = XCreateRegion();
	XIntersectRegion(EXPOSE_REG ? EXPOSE_REG : USER_REG,
			 USER_REG ? USER_REG : EXPOSE_REG,
			 CURRENT_REG);
	XSetRegion(DPY, PEN_GC, CURRENT_REG);
	XSetRegion(DPY, BRUSH_GC, CURRENT_REG);
	XSetRegion(DPY, BG_GC, CURRENT_REG);
	XSetRegion(DPY, TEXT_GC, CURRENT_REG);
    } else {
	CURRENT_REG = NULL;
	XSetClipMask(DPY, PEN_GC, None);
	XSetClipMask(DPY, BRUSH_GC, None);
	XSetClipMask(DPY, BG_GC, None);
	XSetClipMask(DPY, TEXT_GC, None);
    }
}

void wxWindowDC::GetClippingBox(float *x, float *y, float *w, float *h)
{
    if (CURRENT_REG) {
	XRectangle r;
	XClipBox(CURRENT_REG, &r);
	*x = XDEV2LOG(r.x);
	*y = YDEV2LOG(r.y);
	*w = XDEV2LOGREL(r.width);
	*h = YDEV2LOGREL(r.height);
    } else {
	*x = *y = *w = *h = 0.0;
    }
}

//-----------------------------------------------------------------------------
// Set scale and origin -> adjust origin of GC stipples and tiles
//-----------------------------------------------------------------------------

void wxWindowDC::ComputeScaleAndOrigin(void)
{
    wxDC::ComputeScaleAndOrigin();

    int ts_x_origin = XLOG2DEV(0);
    int ts_y_origin = YLOG2DEV(0);

    if (DPY && X && BRUSH_GC && BG_GC) {
	XSetTSOrigin(DPY, BRUSH_GC, ts_x_origin, ts_y_origin);
	XSetTSOrigin(DPY, BG_GC,    ts_x_origin, ts_y_origin);
	// no need for PEN_GC and TEXT_GC
    }
}

//-----------------------------------------------------------------------------
// access to X implementation
//-----------------------------------------------------------------------------

GC&		wxWindowDC::PenGC(void)		{ return PEN_GC; }
GC&		wxWindowDC::BrushGC(void)	{ return BRUSH_GC; }
GC&		wxWindowDC::TextGC(void)	{ return TEXT_GC; }
GC&		wxWindowDC::BgGC(void)		{ return BG_GC; }
Region&		wxWindowDC::UserReg(void)	{ return USER_REG; }
Region&		wxWindowDC::ExposeReg(void)	{ return EXPOSE_REG; }
Region&		wxWindowDC::CurrentReg(void)	{ return CURRENT_REG; }
Display*&	wxWindowDC::Dpy(void)		{ return DPY; }
Screen*&	wxWindowDC::Scn(void)		{ return SCN; }
Drawable&	wxWindowDC::DrawArea(void)	{ return DRAWABLE; }
Window&		wxWindowDC::DrawWindow(void)	{ return DRAW_WINDOW; }
unsigned int&	wxWindowDC::Width(void)		{ return WIDTH; }
unsigned int&	wxWindowDC::Height(void)	{ return HEIGHT; }
unsigned int&	wxWindowDC::Depth(void)		{ return DEPTH; }
Colormap	wxWindowDC::CMap(void)		{ return CMAP; }
