# Toggle widget
# Bert Bos <bert@let.rug.nl>
# Version 2.1 for FWF V4.0
# 
# $Id: Toggle.w,v 1.1 1996-09-25 09:24:19+02 mho Exp $

@class XfwfToggle(XfwfButton)  @file=Toggle

@ The |XfwfToggle| is a button that switches states with every activation
(which is by default with every mouse click). The states are named `on' and
`off'. The states can be indicated with a $\surd$ before the label Two
callbacks report the changed state to the application: |onCallback| is called
when the button switches to `on', |offCallback| is called when the button
switches back to `off'.


@EXPORTS

@ The symbolic constants are used to distinguish different
indicator types.

	@def XfwfSquareIndicator = 0

	@def XfwfDiamondIndicator = 1

	@def XfwfButtonIndicator = 2

@public

@ The |onCallback| is called by the |toggle| action or by the |switch_on|
action, if the previous state was `off'. The |call_data| argument will contain
the |XEvent| pointer that trigerred the action function.

	@var <Callback> XtCallbackList onCallback = NULL

@ The |offCallback| is called from the |toggle| or |switch_off| action, if the
previous state was `on'. The |call_data| argument will be a pointer to the
|XEvent| that caused the action.

	@var <Callback> XtCallbackList offCallback = NULL

@ The variable |on| records the state of the widget: |True| means `on' and
|False| means `off'.

	@var Boolean on = False

@ The state of the |XfwfToggle| is shown by an indicator of
|indicatorSize| and |indicatorType|. The suggested size for the
indicator is about $10\times10$. The default indicator is an
empty or a filled square.

	@var Dimension indicatorSize = 0

	@var int indicatorType = XfwfSquareIndicator

@ The default |frameType| is set to no frame. State is shown by an indicator.

        @var FrameType frameType = XfwfNoFrame

@ The left and right margin around the label are set by default to 0

	@var leftMargin = 0
	@var rightMargin = 0

@exports

@ |XfwfSetToggle| is a shortcut for a call to XtVaSetValues. It allows to
redraw the indicator only and not the complete toggle to reduce flickering.

@proc XfwfSetToggle($, Boolean state)
{
    if (! XtIsSubclass($, xfwfToggleWidgetClass))
	XtError("XfwfSetToggle called with incorrect widget type");

    if ($indicatorType == XfwfButtonIndicator) {
	XtVaSetValues($, "on", state, NULL);
    } else {
	$on = state;
	draw_indicator($);
    }
}

@translations

@ The |toggle| action toggles the widget between `on' and `off'. By
default it is bound to a click of the left mouse button as well as to
the Return key.

	@trans <Btn1Down>,<Btn1Up>: toggle()
	@trans Button1<Leave>: leave()
	@trans <Key>space: toggle()
	@trans <EnterNotify>: enter()
	@trans <LeaveNotify>: leave()

@actions

@ The |toggle| action switches the state. Depending on the resources
it might change the tickmark. The |onCallback| or |offCallback|
functions are called, with the event as |call_data| argument.

@proc toggle
{
    XfwfSetToggle($, !$on);
    XtCallCallbackList($, $on ? $onCallback : $offCallback, event);
}

@ The |switch_on| action switches the button to `on' if it is `off',
otherwise it does nothing. By default it isn't bound to any event. If
the widget is changed, the |onCallback| is called with the event as
|call_data|.

@proc switch_on
{
    if (!$on) {
	XfwfSetToggle($, True);
        XtCallCallbackList($, $onCallback, event);
    }
}

@ The |switch_off| action switches the widget to `off' if the state is
`on', otherwise it does nothing. When the widget changes states, the
|offCallback| is called, with a pointer to the |XEvent| structure as
|call_data| argument.

@proc switch_off
{
    if ($on) {
	XfwfSetToggle($, False);
        XtCallCallbackList($, $offCallback, event);
    }
}

@methods

@ The GC's are created for the first time. Leave additional space for
the indicator - if any.

@proc initialize
{
    if (!$indicatorSize || $indicatorSize > $font->ascent)
	$indicatorSize = $font->ascent;
    if ($indicatorType == XfwfButtonIndicator) {
	$frameType = ($on ? XfwfSunken : XfwfRaised);
    } else if ($shrinkToFit) {
	/* copied from label::initialize, redone to leave place for the indicator */
	Position x, y; Dimension w, h, wd, ht;

	$compute_inside($, &x, &y, &w, &h);
	wd = $label_width + $width - w;
	ht = $label_height + $height - h;
	$set_abs_location($, CWWidth | CWHeight, 0, 0, wd, ht);
    }
}

@ The |compute_inside| method leaves enough space for the indicator.

@proc compute_inside
{
    #compute_inside($, x, y, w, h);
    if ($indicatorType != XfwfButtonIndicator) {
	/* shifted by highlight border and indicator sizes */
	*x += 2 * $highlightThickness + $indicatorSize;
	*w -= $indicatorSize;
    }
}

@ |set_values| changes the frametype, if this toggle has no indicator.

@proc set_values
{
    Boolean redraw = False;

    if ($on != $old$on) {
	if ($indicatorType == XfwfButtonIndicator)
	    $frameType = ($on ? XfwfSunken : XfwfRaised);
        redraw = True;
    }
    if ($indicatorType != $old$indicatorType) {
	if ($indicatorType == XfwfButtonIndicator)
	    $frameType = ($on ? XfwfSunken : XfwfRaised);
	else
	    $frameType = XfwfNoFrame;
	redraw = True;
    }
    if ($indicatorSize != $old$indicatorSize)
	redraw = True;

    return redraw;
}

@ The |expose| method uses the |expose| method of the superclass to draw the
button and then possibly adds a tick mark.

@proc _expose
{
    #_expose($, event, region);
    /* automatically drawn, if highlighted */
    if ( ! $traversal_focus)
	draw_indicator($);
}

@ The highlight and unhighlight methods have to be overriden to skip the
indicator.

@proc highlight_border
{
    if ($highlightThickness == 0 || !XtIsRealized($))
	return;

    if ($indicatorType == XfwfButtonIndicator) {
	#highlight_border($);
    } else {
	Position x = $highlightThickness;
	Position y = ($height - $indicatorSize) / 2;
	switch ($indicatorType) {
	default:
	    XtWarning("XfwfToggle has wrong indicatorType, using square!");
	case XfwfSquareIndicator:
	    XfwfDrawSquare($, XtWindow($), $lightgc, $darkgc, $sunkengc, $backgroundgc, $bordergc,
			   x, y, $indicatorSize, 2, $highlightThickness, $on);
	    break;
	case XfwfDiamondIndicator:
	    XfwfDrawDiamond($, XtWindow($), $lightgc, $darkgc, $sunkengc, $backgroundgc, $bordergc,
			    x, y, $indicatorSize, 2, $highlightThickness, $on);
	    break;
	}
    }
}

@proc unhighlight_border
{
    if ($highlightThickness == 0 || !XtIsRealized($))
	return;

    if ($indicatorType == XfwfButtonIndicator) {
	#unhighlight_border($);
    } else {
	Position x = $highlightThickness;
	Position y = ($height - $indicatorSize) / 2;
	switch ($indicatorType) {
	default:
	    XtWarning("XfwfToggle has wrong indicatorType, using square!");
	case XfwfSquareIndicator:
	    XfwfDrawSquare($, XtWindow($), $lightgc, $darkgc, $sunkengc,
			   $backgroundgc, $backgroundgc,
			   x, y, $indicatorSize, $frameWidth,
			   $highlightThickness, $on);
	    break;
	case XfwfDiamondIndicator:
	    XfwfDrawDiamond($, XtWindow($), $lightgc, $darkgc, $sunkengc,
			    $backgroundgc, $backgroundgc,
			    x, y, $indicatorSize, $frameWidth,
			    $highlightThickness, $on);
	    break;
	}
    }
}

@utilities

@ |draw_indicator| draws the square or the diamond to show the state of the
toggle.

@proc draw_indicator($)
{
    if ( XtIsRealized($) && $indicatorType != XfwfButtonIndicator ) {
	/* draw indicator centred in height */
	Position x = $highlightThickness;
	Position y = ($height - $indicatorSize) / 2;
	switch ($indicatorType) {
	default:
	    XtWarning("XfwfToggle has wrong indicatorType, using square!");
	case XfwfSquareIndicator:
	    XfwfDrawSquare($, XtWindow($), $lightgc, $darkgc, $sunkengc,
			   $backgroundgc, $bordergc,
			   x, y, $indicatorSize, $frameWidth,
			   ($traversal_focus ? $highlightThickness : 0), $on);
	    break;
	case XfwfDiamondIndicator:
	    XfwfDrawDiamond($, XtWindow($), $lightgc, $darkgc, $sunkengc,
			    $backgroundgc, $bordergc,
			    x, y, $indicatorSize, $frameWidth,
			    ($traversal_focus ? $highlightThickness : 0), $on);
	    break;
	}
    }
}
