/* $Id$ */

/* Copyright (C) 1995, 1996, 1997 Free Software Foundation, Inc.
   This file is part of the Gnome Top Library.
   Contributed by Martin Baulig <martin@home-of-linux.org>, April 1998.

   The Gnome Top Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The Gnome Top Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <glibtop.h>
#include <glibtop/xmalloc.h>
#include <glibtop/proclist.h>

#include <glibtop_suid.h>

static const unsigned long _glibtop_sysdeps_proclist =
(1 << GLIBTOP_PROCLIST_TOTAL) + (1 << GLIBTOP_PROCLIST_NUMBER) +
(1 << GLIBTOP_PROCLIST_SIZE);

/* Fetch list of currently running processes.
 *
 * IMPORTANT NOTE:
 *   On error, this function MUST return NULL and set buf->flags to zero !
 *   On success, it returnes a pointer to a list of buf->number elements
 *   each buf->size big. The total size is stored in buf->total. */

unsigned *
glibtop_get_proclist_p (glibtop *server, glibtop_proclist *buf,
			int64_t which, int64_t arg)
{
	register struct proc *pp;
	register int i, nproc = 0;
	unsigned *proc_list = NULL;
	size_t proc_size;

	glibtop_init_p (server, (1 << GLIBTOP_SYSDEPS_PROCLIST), 0);

	memset (buf, 0, sizeof (glibtop_proclist));

	/* Read process table from kernel. */	

	_glibtop_read_proc_table (server);

	/* Count number of processes. */

	for (pp = server->machine.proc_table, i = 0;
	     i < server->machine.nproc; pp++, i++) {
		if (pp->p_stat == 0)
			continue;
		else
			nproc++;
	}

	if (nproc == 0) /* Should never happen. */
		return NULL;

	/* Allocate space for process list. */

	proc_size = nproc * sizeof (unsigned);

	proc_list = glibtop_malloc_r (server, proc_size);

	/* Write process list. */

	for (pp = server->machine.proc_table, i = 0, nproc = 0;
	     i < server->machine.nproc; pp++, i++) {
		if (pp->p_stat == 0)
			continue;
		proc_list [nproc++] = pp->p_pid;
	}

	/* Since everything is ok now, we can set buf->flags, fill in the remaining fields
	   and return proc_list. */

	buf->flags = _glibtop_sysdeps_proclist;

	buf->size = sizeof (unsigned);
	buf->number = nproc;

	buf->total = nproc * sizeof (unsigned);

	return proc_list;
}
