/**
 *
 * $Id: CascadeB.c,v 1.96 1998/10/17 23:09:57 rwscott Exp $
 *
 * Copyright (C) 1995 Free Software Foundation, Inc.
 *
 * This file is part of the GNU LessTif Library.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **/

static const char rcsid[] = "$Id: CascadeB.c,v 1.96 1998/10/17 23:09:57 rwscott Exp $";

#define	DO_PIXMAP
#define	DO_SANITY

#include <LTconfig.h>
#include <XmI/XmI.h>
#include <XmI/MacrosI.h>

#include <Xm/XmP.h>
#include <Xm/CascadeBP.h>
#include <Xm/CascadeBGP.h>
#include <Xm/RowColumnP.h>
#include <Xm/TearOffP.h>
#include <Xm/TransltnsP.h>
#include <Xm/MenuShellP.h>
#include <Xm/MenuUtilP.h>
#include <Xm/ScreenP.h>

#ifndef __STDC__
#include <varargs.h>
#else
#include <stdarg.h>
#endif

#include <XmI/DebugUtil.h>

#include <stdio.h>

#ifndef XmUNSPECIFIED
#define XmUNSPECIFIED (~0)
#endif

/* Forward Declarations */

static void class_initialize();

static void class_part_initialize(WidgetClass w_class);

static void initialize(Widget request, Widget new_w,
		       ArgList args, Cardinal *num_args);

static void initialize_prehook(Widget request, Widget new_w,
			       ArgList args, Cardinal *num_args);

static void initialize_posthook(Widget request, Widget new_w,
				ArgList args, Cardinal *num_args);

static void resize(Widget w);

static void destroy(Widget w);

static void expose(Widget w, XEvent *event, Region region);

static Boolean set_values(Widget current, Widget request, Widget new_w,
			  ArgList args, Cardinal *num_args);

static void CleanupMenuBar(Widget w, XEvent *event,
			   String *params, Cardinal *num_params);

static void StartDrag(Widget, XEvent *, String *, Cardinal *);

static void KeySelect(Widget, XEvent *, String *, Cardinal *);

static void ArmAndActivate(Widget w, XEvent *event,
			   String *params, Cardinal *num_params);

static void _XmCBMenuBarSelect(Widget, XEvent *, String *, Cardinal *);

static void _XmCBMenuBarDoSelect(Widget, XEvent *, String *, Cardinal *);

static void MenuBarEnter(Widget, XEvent *, String *, Cardinal *);

static void MenuBarLeave(Widget, XEvent *, String *, Cardinal *);

static void DelayedArm(Widget, XEvent *, String *, Cardinal *);

static void CheckDisarm(Widget, XEvent *, String *, Cardinal *);

static void ArmAndPost(Widget, XEvent *, String *, Cardinal *);

void _XmCBCalcDimensions(Widget w);

static void border_unhighlight(Widget w);

static void border_highlight(Widget w);

static void MenuProcEntry(int proc, Widget rc,...);

/*
 * Resources for the cascade button class
 */
#define Offset(field) XtOffsetOf(XmCascadeButtonRec, cascade_button.field)
#define POffset(field) XtOffsetOf(XmCascadeButtonRec, primitive.field)
static XtResource resources[] =
{
    {
	XmNactivateCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(activate_callback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNcascadingCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(cascade_callback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNsubMenuId, XmCMenuWidget, XmRMenuWidget,
	sizeof(Widget), Offset(submenu),
	XmRMenuWidget, (XtPointer)NULL
    },
    {
	XmNcascadePixmap, XmCPixmap, XmRPrimForegroundPixmap,
	sizeof(Pixmap), Offset(cascade_pixmap),
	XmRImmediate, (XtPointer)XmUNSPECIFIED_PIXMAP
    },
    {
	XmNmappingDelay, XmCMappingDelay, XmRInt,
	sizeof(int), Offset(map_delay),
	XmRImmediate, (XtPointer)180
    },
    /* resources we override from XmLabel/XmPrimitive */
    {
	XmNshadowThickness, XmCShadowThickness, XmRHorizontalDimension,
	sizeof(Dimension), POffset(shadow_thickness),
	XmRCallProc, (XtPointer)_XmPrimitiveShadowThicknessDefault
    },
    {
	XmNtraversalOn, XmCTraversalOn, XmRBoolean,
	sizeof(Boolean), POffset(traversal_on),
	XmRImmediate, (XtPointer)True
    },
    {
	XmNhighlightThickness, XmCHighlightThickness, XmRHorizontalDimension,
	sizeof(Dimension), POffset(highlight_thickness),
	XmRCallProc, (XtPointer)_XmPrimitiveHighlightThicknessDefault
    },
    {
	XmNmarginWidth, XmCMarginWidth, XmRHorizontalDimension,
	sizeof(Dimension), XtOffsetOf(XmCascadeButtonRec, label.margin_width),
	XmRImmediate, (XtPointer)XmINVALID_DIMENSION
    }
};

static XtActionsRec actions[] =
{
    {"DelayedArm", DelayedArm},
    {"CheckDisarm", CheckDisarm},
    {"StartDrag", StartDrag},
    {"DoSelect", _XmCBMenuBarDoSelect},
    {"KeySelect", KeySelect},
    {"MenuBarSelect", _XmCBMenuBarSelect},
    {"MenuBarEnter", MenuBarEnter},
    {"MenuBarLeave", MenuBarLeave},
    {"CleanupMenuBar", CleanupMenuBar},
    {"Help", _XmCBHelp},
};

/* *INDENT-OFF* */
static XmBaseClassExtRec _XmCascadeBCoreClassExtRec = {
    /* next_extension            */ NULL,
    /* record_type               */ NULLQUARK,                             
    /* version                   */ XmBaseClassExtVersion,
    /* size                      */ sizeof(XmBaseClassExtRec),
    /* initialize_prehook        */ initialize_prehook,
    /* set_values_prehook        */ XmInheritSetValuesPrehook,
    /* initialize_posthook       */ initialize_posthook,
    /* set_values_posthook       */ XmInheritSetValuesPosthook,
    /* secondary_object_class    */ XmInheritClass,
    /* secondary_object_create   */ XmInheritSecObjectCreate,
    /* get_secondary_resources   */ XmInheritGetSecResData,
    /* fast_subclass             */ { 0 },
    /* get_values_prehook        */ XmInheritGetValuesPrehook,
    /* get_values_posthook       */ XmInheritGetValuesPosthook,
    /* class_part_init_prehook   */ NULL,
    /* class_part_init_posthook  */ NULL,
    /* ext_resources             */ NULL,
    /* compiled_ext_resources    */ NULL,
    /* num_ext_resources         */ 0,
    /* use_sub_resources         */ False,
    /* widget_navigable          */ XmInheritWidgetNavigable,
    /* focus_change              */ XmInheritFocusChange,
    /* wrapper_data              */ NULL
};

XmPrimitiveClassExtRec _XmCascadeBPrimClassExtRec = {
    /* next_extension      */ NULL,
    /* record_type         */ NULLQUARK,
    /* version             */ XmPrimitiveClassExtVersion,
    /* record_size         */ sizeof(XmPrimitiveClassExtRec),
    /* widget_baseline     */ XmInheritBaselineProc,
    /* widget_display_rect */ XmInheritDisplayRectProc,
    /* widget_margins      */ NULL
};

XmCascadeButtonClassRec xmCascadeButtonClassRec = {
    /* Core class part */
    {
	/* superclass            */ (WidgetClass) &xmLabelClassRec,
        /* class_name            */ "XmCascadeButton",
	/* widget_size           */ sizeof(XmCascadeButtonRec),
	/* class_initialize      */ class_initialize,
	/* class_part_initialize */ class_part_initialize,
	/* class_inited          */ False,
	/* initialize            */ initialize,
	/* initialize_hook       */ NULL,
	/* realize               */ XtInheritRealize,
	/* actions               */ actions,
	/* num_actions           */ XtNumber(actions),
	/* resources             */ resources,
	/* num_resources         */ XtNumber(resources),
	/* xrm_class             */ NULLQUARK,
	/* compress_motion       */ True,
	/* compress_exposure     */ XtExposeCompressMaximal,
	/* compress_enterleave   */ True,
	/* visible_interest      */ False,
	/* destroy               */ destroy,
	/* resize                */ resize,
	/* expose                */ expose,
	/* set_values            */ set_values,
	/* set_values_hook       */ NULL,
	/* set_values_almost     */ XtInheritSetValuesAlmost,
	/* get_values_hook       */ NULL,
	/* accept_focus          */ NULL,
	/* version               */ XtVersion,
	/* callback offsets      */ NULL,
	/* tm_table              */ NULL,
	/* query_geometry        */ XtInheritQueryGeometry,
	/* display_accelerator   */ NULL,
	/* extension             */ (XtPointer)&_XmCascadeBCoreClassExtRec
    },
    /* Primitive Class part */
    {
	/* border_highlight      */ border_highlight,
        /* border_unhighlight    */ border_unhighlight,
        /* translations          */ XtInheritTranslations,
        /* arm_and_activate_proc */ ArmAndActivate,
        /* synthetic resources   */ NULL,
	/* num syn res           */ 0,
        /* extension             */ (XtPointer)&_XmCascadeBPrimClassExtRec,
    },
    /* Label Class part */
    {
        /* setOverrideCallback */ XmInheritSetOverrideCallback,
        /* menuProcs           */ XmInheritMenuProc,
        /* translations        */ XtInheritTranslations,
	/* extension           */ NULL
    },
    /* CascadeButton Class part */
    {
	/* extension */ NULL
    }
};
/* *INDENT-ON* */


WidgetClass xmCascadeButtonWidgetClass = (WidgetClass)&xmCascadeButtonClassRec;

static XtTranslations menub_trans;
static XtTranslations popup_trans;

static void
class_initialize()
{
    _XmCascadeBCoreClassExtRec.record_type = XmQmotif;

    menub_trans = XtParseTranslationTable(_XmCascadeB_menubar_events);

    popup_trans = XtParseTranslationTable(_XmCascadeB_p_events);
}

static void
class_part_initialize(WidgetClass widget_class)
{
    _XmFastSubclassInit(widget_class, XmCASCADE_BUTTON_BIT);
}

static void
initialize_prehook(Widget request, Widget new_w,
		   ArgList args, Cardinal *num_args)
{
    DEBUGOUT(XdbDebug(__FILE__, new_w, "%s:initialize_prehook()\n", __FILE__));

    _XmSaveCoreClassTranslations(new_w);

    if (XmIsRowColumn(XtParent(new_w)) &&
	RC_Type(XtParent(new_w)) == XmMENU_BAR)
    {
	DEBUGOUT(XdbDebug(__FILE__, new_w, "    MenuBar translations\n"));

	CoreClassTranslations(new_w) = (String)menub_trans;
    }
    else if (XmIsRowColumn(XtParent(new_w)) &&
	     (RC_Type(XtParent(new_w)) == XmMENU_POPUP ||
	      RC_Type(XtParent(new_w)) == XmMENU_PULLDOWN))
    {
	DEBUGOUT(XdbDebug(__FILE__, new_w, "    Popup translations\n"));

	CoreClassTranslations(new_w) = (String)popup_trans;
    }
    else
    {
	DEBUGOUT(XdbDebug(__FILE__, new_w, "    Default translations\n"));
    }

}

static void
initialize_posthook(Widget request, Widget new_w,
		    ArgList args, Cardinal *num_args)
{
    _XmRestoreCoreClassTranslations(new_w);
}

static void
initialize(Widget request, Widget new_w, ArgList args, Cardinal *num_args)
{
    DEBUGOUT(XdbDebug(__FILE__, new_w, "%s:initialize()\n", __FILE__));

    CB_SetArmed(new_w, False);
    CB_ArmedPixmap(new_w) = (Pixmap)NULL;
    CB_CascadePixmap(new_w) = (Pixmap)NULL;

    if (!XtIsSubclass(XtParent(new_w), xmRowColumnWidgetClass))
    {
	_XmError(new_w, "Cascade parent must be a RowColumn.");
    }

    CB_Cascade_width(new_w) = 0;
    CB_Cascade_height(new_w) = 0;
    CB_Cascade_x(new_w) = 0;
    CB_Cascade_y(new_w) = 0;

    if (Lab_MenuType(new_w) == XmMENU_BAR ||
	Lab_MenuType(new_w) == XmMENU_POPUP ||
	Lab_MenuType(new_w) == XmMENU_PULLDOWN)
    {
	Lab_Highlight(new_w) = 0;
    }
    else if (Lab_MenuType(new_w) != XmMENU_OPTION)
    {
	_XmError(new_w, "Cascade parent is incorrect type.");
    }

    if (Lab_MarginWidth(new_w) == (Dimension)XmINVALID_DIMENSION)
    {
	/*
	 * do something funky for the pixmap, but for now do nothing
	 * This is a little weird.  Label's man page suggests that
	 * marginRight gets modified, not marginWidth.
	 * -- FIX ME!
	 * MLM: If the button doesn't have a shadow, MarginWidth of 0
	 * is ok.  Otherwise, need marginWidth of 2.
	 */
	if (Lab_MenuType(new_w) == XmMENU_BAR)
	{
	    Lab_MarginWidth(new_w) = Prim_ShadowThickness(new_w) +
		Xm3D_ENHANCE_PIXEL * 2;
	}
	else
	{
	    Lab_MarginWidth(new_w) = Prim_ShadowThickness(new_w);
	}
    }
    if (Lab_MenuType(new_w) == XmMENU_OPTION)
    {
	Lab_MarginHeight(new_w) = Prim_ShadowThickness(new_w);
	Lab_MarginTop(new_w) = Prim_HighlightThickness(new_w);
	Lab_MarginBottom(new_w) = Prim_HighlightThickness(new_w);
    }

    if ((RC_Type(XtParent(new_w)) == XmMENU_PULLDOWN) ||
	(RC_Type(XtParent(new_w)) == XmMENU_POPUP) ||
	(RC_Type(XtParent(new_w)) == XmMENU_OPTION))	/* rws 23 Mar 1997 */
    {
	_XmCreateArrowPixmaps(new_w);
    }

    if (CB_Submenu(new_w))
    {
	/* Make sure the RC also knows how to locate us */
	RC_MenuSubmenu(new_w);
    }

    _XmCBCalcDimensions(new_w);
    if (XtWidth(request) != 0)
    {
    	XtWidth(new_w) = XtWidth(request);
    }
    if (XtHeight(request) != 0)
    {
    	XtHeight(new_w) = XtHeight(request);
    }
    CB_Timer(new_w) = 0;
    LabClass_MenuProcs(XtClass(new_w)) = MenuProcEntry;
}

static Dimension
ComputeMaximumWidth(Widget cb)
{
    Dimension wd = 0;
    Widget sm = CB_Submenu(cb);
    int i;
 
    if (!sm)
    {
	return 0;
    }

    for (i = 0; i < MGR_NumChildren(sm); i++)
    {
        if (XtWidth(MGR_Children(sm)[i]) > wd)
        {
            wd = XtWidth(MGR_Children(sm)[i]);
        }
    }

    return wd;
}
 
void
_XmCBCalcDimensions(Widget w)
{
    Dimension cw, ch, width, height;

    if (RC_Type(XtParent(w)) == XmMENU_OPTION ||
	RC_Type(XtParent(w)) == XmMENU_PULLDOWN ||
	RC_Type(XtParent(w)) == XmMENU_POPUP)
    {
	Lab_MarginRight(w) = CB_Cascade_width(w) +
	    2 * Prim_ShadowThickness(w) +
	    Prim_ShadowThickness(w);
    }

    XtWidth(w) = XtHeight(w) = 0;

    if (Lab_IsText(w))
    {
        XmString string;
 
        if (_XmStringIsXmString((XmString)Lab_Label(w)))
        {
            DEBUGOUT(XdbDebug(__FILE__, w,
                   "_XmCBGCalcDimensions: convert Lab_Label to _XmString\n"));
            Lab_Label(w) = _XmStringCreate((XmString)Lab_Label(w));
        }

	_XmStringExtent(Lab_Font(w), Lab_Label(w), &cw, &ch);

        /* rws 16 Aug 1997
         * Do not let a cascade button be shorter than the height of one
         * character.  This is to make sure that the pixmap can be drawn
         * correctly. This shows in Mosaic with option buttons in a form.
         */
        string = XmStringCreateSimple("M");
        if (ch < XmStringHeight(Lab_Font(w), string))
        {
            ch = XmStringHeight(Lab_Font(w), string);
        }
 
        XmStringFree(string);
    }
    else			/* pixmap */
    {
	_XmLabelGetPixmapSize(w, Lab_Pixmap(w), &cw, &ch);
    }

    if (RC_Type(XtParent(w)) == XmMENU_OPTION && CB_Submenu(w) &&
	ComputeMaximumWidth(w) > cw)
    {
	cw = ComputeMaximumWidth(w);
	/* rws */cw -= 2 * LabG_MarginWidth(w);
	/* rws */cw -= 2 * LabG_MarginWidth(w);
	Lab_TextRect_width(w) = cw;
    }

    width = (Lab_Highlight(w)
	     + Lab_Shadow(w)
	     + Lab_MarginLeft(w)
	     + Lab_MarginWidth(w)
	     + cw
	     + Lab_MarginWidth(w)
	     + Lab_MarginRight(w)
	     + Lab_Shadow(w)
	     + Lab_Highlight(w));

    height = (Lab_Highlight(w)
	      + Lab_Shadow(w)
	      + Lab_MarginTop(w)
	      + Lab_MarginHeight(w)
	      + ch
	      + Lab_MarginHeight(w)
	      + Lab_MarginBottom(w)
	      + Lab_Shadow(w)
	      + Lab_Highlight(w));

    XtWidth(w) = width;
    XtHeight(w) = height;

    CB_Cascade_x(w) = XtWidth(w) -
			CB_Cascade_width(w) -
			2 * Prim_ShadowThickness(w) -
			Prim_HighlightThickness(w);

    CB_Cascade_y(w) = Lab_TextRect_y(w);
}

static void
resize(Widget w)
{
    DEBUGOUT(XdbDebug(__FILE__, w, "%s:resize(%d) - %dx%d\n",
    			__FILE__, __LINE__,
    			XtWidth(w), XtHeight(w)));

#define superclass (&xmLabelClassRec)
    (*superclass->core_class.resize) (w);
#undef superclass

    CB_Cascade_x(w) = XtWidth(w) -
			CB_Cascade_width(w) -
			2 * Prim_ShadowThickness(w) -
			Prim_HighlightThickness(w);

    CB_Cascade_y(w) = Lab_TextRect_y(w);
}

static void
destroy(Widget w)
{
    DEBUGOUT(XdbDebug(__FILE__, w, "%s:destroy()\n", __FILE__));

    if (CB_ArmedPixmap(w))
    {
	_XmFreeScratchPixmap((XmScreen)XmGetXmScreen(XtScreen(w)),
			 CB_ArmedPixmap(w));
    }

    if (CB_CascadePixmap(w))
    {
	_XmFreeScratchPixmap((XmScreen)XmGetXmScreen(XtScreen(w)),
			 CB_CascadePixmap(w));
    }
}

static Boolean
set_values(Widget old, Widget request, Widget new_w,
	   ArgList args, Cardinal *num_args)
{
    Boolean refresh_needed = False;

    DEBUGOUT(XdbDebug(__FILE__, new_w,
		      "%s:set_values() - %i args\n"
		      "\t    old X %5i Y %5i W %5i H %5i\n"
		      "\trequest X %5i Y %5i W %5i H %5i\n"
		      "\t  new_w X %5i Y %5i W %5i H %5i\n",
		      __FILE__,
		      *num_args,
		      XtX(old), XtY(old),
		      XtWidth(old), XtHeight(old),
		      XtX(request), XtY(request),
		      XtWidth(request), XtHeight(request),
		      XtX(new_w), XtY(new_w),
		      XtWidth(new_w), XtHeight(new_w)));
    DEBUGOUT(XdbPrintArgList(__FILE__, new_w, args, *num_args, False));

    if (CB_CascadePixmap(old) != CB_CascadePixmap(new_w))
    {
#if 0
	XFreePixmap(XtDisplay(old), CB_CascadePixmap(old));
	XFreePixmap(XtDisplay(new_w), CB_ArmedPixmap(new_w));
#endif
	_XmCreateArrowPixmaps(new_w);

	if (Lab_RecomputeSize(new_w))
	{
	    _XmCBCalcDimensions(new_w);
	}

	refresh_needed = True;
    }

    if (Lab_Label(new_w) != Lab_Label(old))
    {
	if (Lab_RecomputeSize(new_w))
	{
	    _XmCBCalcDimensions(new_w);
	}

	refresh_needed = True;
    }

    if (XtSensitive(new_w) != XtSensitive(old))
    {
	DEBUGOUT(XdbDebug(__FILE__, new_w,
			  "SetValues: sensitive changes to %d\n",
			  XtSensitive(new_w)));

	refresh_needed = True;
    }

    if (CB_Submenu(old) != CB_Submenu(new_w))
    {
	/* Make sure the RC also knows how to locate us */
	RC_MenuSubmenu(new_w);
	if (Lab_RecomputeSize(new_w))
	{
	    _XmCBCalcDimensions(new_w);
	}
	refresh_needed = True;
    }

    return refresh_needed;
}

static void
expose(Widget w, XEvent *event, Region region)
{
    DEBUGOUT(XdbDebug(__FILE__, w, "%s:expose() Armed(%d) type %s\n",
    		      __FILE__,
		      CB_IsArmed(w), XdbMenuType2String(Lab_MenuType(w))));

#if 0
    /* rws 8 May 1998
       This messes up Mozilla's Bookmark menu buttons which are sub-classed
       from PushB and CascadeB.
     */
    resize(w);
#else
    (XtClass(w)->core_class.resize)(w);
#endif

    /* The label's expose routine should be called after the cascade button's
     * expose routine. */

    if (CB_IsArmed(w) || Lab_MenuType(w) == XmMENU_OPTION)
    {
	XmCascadeButtonHighlight(w, True);
    }
    else
    {
	XmCascadeButtonHighlight(w, False);
    }

#define superclass (&xmLabelClassRec)
    (*superclass->core_class.expose) (w, event, region);
#undef superclass
}

static void
border_unhighlight(Widget w)
{
    DEBUGOUT(XdbDebug(__FILE__, w, "CBBorderUnhighlight\n"));

    /* with zero width, we don't need this... */
    if (Prim_HighlightThickness(w) == 0)
    {
	return;
    }

    if (XmIsManager(XtParent(w)))
    {
	_XmDrawHighlight(XtDisplay(w), XtWindow(w), XmParentBackgroundGC(w),
			 0, 0, XtWidth(w), XtHeight(w),
			 Prim_HighlightThickness(w), LineSolid);
    }
    else
    {
	_XmClearBorder(XtDisplay(w),
		       XtWindow(w),
		       0, 0,
		       XtWidth(w), XtHeight(w),
		       Prim_HighlightThickness(w));
    }

    Prim_Highlighted(w) = False;
    Prim_HighlightDrawn(w) = False;
}

static void
border_highlight(Widget w)
{
    DEBUGOUT(XdbDebug(__FILE__, w, "CBBorderHighlight\n"));

    /* with zero width, we don't need this... */
    if (Prim_HighlightThickness(w) == 0)
    {
	return;
    }

    _XmDrawHighlight(XtDisplay(w), XtWindow(w), Prim_HighlightGC(w),
		     0, 0, XtWidth(w), XtHeight(w),
		     Prim_HighlightThickness(w), LineSolid);

    Prim_Highlighted(w) = True;
    Prim_HighlightDrawn(w) = True;
}

/* ACTION PROCS */
/*
 * CleanupMenuBar
 *    w       is a cascadeButton
 */
static void
CleanupMenuBar(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    DEBUGOUT(XdbDebug(__FILE__, w, "CleanupMenuBar()\n"));

    if (w == NULL)
    {
	return;			/* Try not to crash */
    }

    RCClass_MenuProcs(XtClass(XtParent(w)))(XmMENU_BAR_CLEANUP,
					    XtParent(w), NULL);
}

static void
KeySelect(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    DEBUGOUT(XdbDebug(__FILE__, w, "KeySelect\n"));

    /* If we're in keyboard traversal mode, and if we have a submenu 
       attached, call the cascading callback, post the submenu and 
       allow keyboard traversal of it. */

    /* else call the activate callback, and unpost all active menus
       in the cascade */
}

void
_XmCascadingPopup(Widget cb, XEvent *event, Boolean doCascade)
{
    /* FIX ME: doCascade? */
    XmAnyCallbackStruct cbs;

    RC_MenuCascading(cb, event);

    cbs.reason = XmCR_CASCADING;
    cbs.event = event;
    XtCallCallbackList(cb,
		       (XmIsGadget(cb)
			? CBG_CascadeCall(cb)
			: CB_CascadeCall(cb)),
		       &cbs);
}

/*
 *    DoSelect()
 *
 *      Calls the callbacks in XmNcascadingCallback, posts the submenu attached
 *      to the CascadeButton and enables keyboard traversal within the menu.
 *      If the CascadeButton does not have a submenu attached, this action
 *      calls the callbacks in XmNactivateCallback, activates the
 *      CascadeButton, and unposts all posted menus in the cascade.
 */
void
_XmCBMenuBarDoSelect(Widget w, XEvent *event,
		     String *params, Cardinal *num_params)
{
    Widget submenu;

    submenu = (XmIsGadget(w) ? CBG_Submenu(w) : CB_Submenu(w));

    DEBUGOUT(XdbDebug(__FILE__, w, "_XmCBMenuBarDoSelect()\n"));

    /* queue events until the next button event. */
    XAllowEvents(XtDisplay(w), SyncPointer, CurrentTime);

    _XmRecordEvent(event);

    /* if we have a submenu attached, call the cascading callback, 
       post the submenu and allow keyboard traversal of it. */

    if (submenu)
    {
	/* Turn on keyboard traversal */
	_XmSetInDragMode(w, False);

	DEBUGOUT(XdbDebug0(__FILE__, submenu, "    RC_CascadeBtn(%s)-> %s\n",
			   XtName(submenu), XtName(w)));

	RC_CascadeBtn(submenu) = w;
	RC_PopupPosted(XtParent(w)) = submenu;

	DEBUGOUT(XdbDebug0(__FILE__, w, "    RC_PopupPosted(%s) set to %s\n",
			   XtName(XtParent(w)),
			   submenu ? XtName(submenu) : "(null)"));
    }
    else
    {				/* We have no menu */
	XtCallbackList cbl;

	/* rws 8 Jul 1998
	   We have no menu but, we may have been armed, filesb/test5. So
	   let's clean up anything that needs to be first.
	 */
	RCClass_MenuProcs(XtClass(XtParent(w)))(XmMENU_BAR_CLEANUP,
                                                    XtParent(w), NULL);
	/* Actually all this no-menu thing is totally wrong.
	 *
	 * This code does call the callback, which makes apps work,
	 * but it doesn't do arm/disarm, and also it still activates
	 * if the user moves away from the menubar, and then releases
	 * the mouse button.
	 *
	 * Hence something is really really wrong here.
	 * Danny 13/3/1997
	 * For the record, I also wrote this :-)
	 *
	 * FIX ME
	 */
	/* Call ActivateCallback */

	cbl = XmIsGadget(w) ? CBG_ActivateCall(w) : CB_ActivateCall(w);

	/* this should take care of activation when the cursor is not
	   in the window any more. Find a way to test it and then
	   un-comment it. (taken from PushB)
	 */
	/*
	if (ev && (ev->type == KeyPress || ev->type == KeyRelease
		   || ((ev->x >= 0 && ev->x < XtWidth(w))
		       && (ev->y >= 0 && ev->y < XtHeight(w)))))
	*/
	{
	    if (cbl)
	    {
		XmAnyCallbackStruct cbs;

		cbs.reason = XmCR_ACTIVATE;
		cbs.event = event;

		XtCallCallbackList(w, cbl, &cbs);
	    }
	}
    }

    CB_SetArmed(w, False);
}

void
_XmCBMenuBarSelect(Widget w, XEvent *event,
		   String *params, Cardinal *num_params)
{
    Widget menu, mb;
    Boolean poppedUp;

    DEBUGOUT(XdbDebug(__FILE__, w, "_XmCBMenuBarSelect()\n"));
    if (Lab_MenuType(w) == XmMENU_OPTION)
    {
	ArmAndPost(w, event, params, num_params);

	return;
    }

    /* first thing we do is free up the next button event. */
    XAllowEvents(XtDisplay(w), SyncPointer, CurrentTime);

    _XmRecordEvent(event);

    mb = XtParent(w);
    menu = XmIsGadget(w) ? CBG_Submenu(w) : CB_Submenu(w);

    if (!menu)
    {
	DEBUGOUT(XdbDebug(__FILE__, w, "_XmCBMenuBarSelect: no menu!\n"));
	/* rws 24 Jun 1998
	   Oh well, Motif still highlights the button and gives up the menu
	   cursor. So lets go for it anyway!
	 */
	/*
	return;
	*/
    }

    RC_MenuShellPopdown(w, event, &poppedUp);
    /* Set drag mode */
    _XmSetInDragMode(w, True);

    if (XmIsGadget(w))
    {
	CBG_SetArmed(w, True);
    }
    else
    {
	CB_SetArmed(w, True);
    }

    /* highlight the cascade button. */
    XmCascadeButtonHighlight(w, True);

    if (!RC_IsArmed(mb))
    {
	DEBUGOUT(XdbDebug(__FILE__, w, "MENU_ARM\n"));

	RCClass_MenuProcs(XtClass(mb))(XmMENU_ARM, mb, NULL);
    }

    _XmCascadingPopup(w, event, True);
    MGR_ActiveChild(XtParent(w)) = w;
}

static void
ArmAndPost(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    DEBUGOUT(XdbDebug(__FILE__, w, "ArmAndPost()\n"));

    XmProcessTraversal(w, XmTRAVERSE_CURRENT);

    MGR_ActiveChild(XtParent(w)) = w;
    if (!RC_IsArmed(CBG_Submenu(w)))
    {
	DEBUGOUT(XdbDebug(__FILE__, w, "MENU_ARM\n"));

	RCClass_MenuProcs(XtClass(XtParent(w)))(XmMENU_ARM, XtParent(w), NULL);
    }

    _XmCascadingPopup(w, event, True);
}

static void
MenuBarEnter(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    DEBUGOUT(XdbDebug(__FILE__, w, "MenuBarEnter\n"));

    /* if we're not dragging, do nothing. */
    if (!_XmGetInDragMode(w))
    {
	DEBUGOUT(XdbDebug(__FILE__, w, "Not dragging\n"));
	return;
    }

    /* if we're already armed, do nothing */
    if (CB_IsArmed(w))
    {
	DEBUGOUT(XdbDebug(__FILE__, w, "Already armed\n"));
	return;
    }

    /* treat this like a menu bar select event */
    _XmCBMenuBarSelect(w, event, params, num_params);
}

static void
MenuBarLeave(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    DEBUGOUT(XdbDebug(__FILE__, w, "MenuBarLeave\n"));
    if (CB_IsArmed(w) && RC_PopupPosted(XtParent(w)) != CB_Submenu(w))
    {
	XmCascadeButtonHighlight(w, False);
    }
}

static void
StartDrag(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    Boolean validButton;
    Boolean poppedUp;

    DEBUGOUT(XdbDebug(__FILE__, w, "StartDrag()\n"));

    /* queue events up until the next button event. */
    XAllowEvents(XtDisplay(w), SyncPointer, CurrentTime);

    _XmRecordEvent(event);

    /* Is it even the right kind of event? */
    if (!event || event->type != ButtonPress)
    {
	return;
    }

    /* Was it the right button? */
    RC_MenuButton(w, event, &validButton);

    if (!validButton)
    {
	return;
    }

    RC_MenuShellPopdown(w, event, &poppedUp);
    _XmCascadingPopup(w, event, True);

    _XmSetInDragMode(w, True);
}

static void
CascadePopupHandler(XtPointer clientData, XtIntervalId *id)
{
    Widget w = (Widget)clientData;
    Boolean poppedUp;

    DEBUGOUT(XdbDebug(__FILE__, w, "CascadePopupHandler\n"));

    CB_Timer(w) = 0;

    RC_MenuShellPopdown(w, NULL, &poppedUp);
    _XmCascadingPopup(w, NULL, True);	/* FIX ME: NULL? */
}

static void
DelayedArm(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    DEBUGOUT(XdbDebug(__FILE__, w, "DelayedArm()\n"));

    /* Signal server to release next pointer event */
    XAllowEvents(XtDisplay(w), SyncPointer, CurrentTime);

    if (_XmGetInDragMode(w))
    {
	CB_Timer(w) = XtAppAddTimeOut(XtWidgetToApplicationContext(w),
				      CB_MapDelay(w),
				      CascadePopupHandler,
				      (XtPointer)w);

	MGR_ActiveChild(XtParent(w)) = w;
	CB_SetArmed(w, True);

	XmCascadeButtonHighlight(w, True);
    }
}

static void
CheckDisarm(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    int x = ((XLeaveWindowEvent *)event)->x;
    int y = ((XLeaveWindowEvent *)event)->y;

    DEBUGOUT(XdbDebug(__FILE__, w, "CheckDisarm()\n"));

    XAllowEvents(XtDisplay(w), SyncPointer, CurrentTime); /* NEW */

    if (_XmGetInDragMode(w))
    {
	if (CB_Timer(w))
	{
	    XtRemoveTimeOut(CB_Timer(w));
	    CB_Timer(w) = 0;
	    if (!RC_PopupPosted(XtParent(w)))
	    {
		    CB_SetArmed(w, False);
		    XmCascadeButtonHighlight(w, False);
	    }
	}

	/*
	 * If a submenu is already popped up, pop it down unless
	 * we've left on the right edge.
	 * The test is a bit elaborate because the right edge is 
	 * not a straight line (at least, in M*tif: FIX ME).
	 */
	 /* rws 7 Dec 1997
	  * Not only that, if the cascade button was close enough to the
	  * right edge of the screen the submenu may have been posted to
	  * the left of the cascade button. In this case the x of the
	  * leave event could be less than 0.  So if the submenu was posted
	  * to the left of the cascade button use minus the width of the
	  * submenu to compare with the x value to indicate that we left
	  * by the right side.
	  */
	/*
	printf("%d %d %d %d %d\n",
	    x,
	    XtWidth(RC_PopupPosted(XtParent(w))),
	    XtX(XtParent(RC_PopupPosted(XtParent(w)))),
	    XtX(XtParent(XtParent(w))),
	    XtWidth(XtParent(XtParent(w)))
	);
	*/
	if ( _XmGetRC_PopupPosted(XtParent(w)) && 
			XtX(_XmGetRC_PopupPosted(XtParent(w))) <
			XtX(XtParent(XtParent(w))))
	{
		/* submenu was posted to the left of the cascade button */
		x += XtWidth(_XmGetRC_PopupPosted(XtParent(w)));
	}
	if ((x <= 0 || y <= 0 || y >= XtHeight(w)) /*&&
	    RC_PopupPosted(XtParent(w)) != NULL*/)
	{
	    Boolean poppedUp;

	    RC_MenuShellPopdown(w, event, &poppedUp);
	    CB_SetArmed(w, False);
	    XmCascadeButtonHighlight(w, False);

	    /* FIX ME: more (disarm, RC_CascadeBtn) */
	}
    }
}

void
_XmCBHelp(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    DEBUGOUT(XdbDebug(__FILE__, w, "_XmCBHelp\n"));

    /* clean up things */

    /* invoke help callbacks */
    XtCallActionProc(w, "PrimitiveHelp", event, params, *num_params);
}

static void
ArmAndActivate(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    DEBUGOUT(XdbDebug(__FILE__, w, "%s:ArmAndActivate()\n", 
    		__FILE__));
    MGR_ActiveChild(XtParent(w)) = w;
    if (Lab_MenuType(w) == XmMENU_BAR)
    {
	_XmMenuFocus(w, XmMENU_FOCUS_SET, CurrentTime);
	_XmCBMenuBarSelect(w, event, params, num_params);
	_XmCBMenuBarDoSelect(w, event, params, num_params);
	_XmMenuArmItem(MGR_Children(CB_Submenu(w))[0]);
	/* I would like to use this but currently only the Vendor focus data
	 * is used not the MenuShell focus data
	XmProcessTraversal(MGR_Children(CB_Submenu(w))[0], XmTRAVERSE_HOME);
	 */
    }
    else if (Lab_MenuType(w) == XmMENU_OPTION)
    {
	XmProcessTraversal(w, XmTRAVERSE_CURRENT);

	if (!RC_IsArmed(CB_Submenu(w)))
	{
	    DEBUGOUT(XdbDebug(__FILE__, w, "MENU_ARM\n"));

	    RCClass_MenuProcs(XtClass(XtParent(w)))(XmMENU_ARM,
						    XtParent(w), NULL);
	}

	CascadePopupHandler(w, NULL);
    }
    else /* pullright */
    {
	CascadePopupHandler(w, NULL);
    }
}


#ifdef __STDC__
static void
MenuProcEntry(int proc, Widget w,...)
{
    va_list arg_list;
    /*
    XEvent *event;
    */
    extern void _XmTearOffArm();
 
    va_start(arg_list, w);
#else
MenuProcEntry(proc, w, va_alist)
     int proc;
     Widget w;
     va_dcl
{
    va_list arglist;
    /*
    XEvent *event;
    */
    extern void _XmTearOffArm();
 
    va_start(arglist);
#endif

    switch (proc)
    {
    case XmMENU_ARM:
    	{
	  /* XtExposeProc exp = XtClass(w)->core_class.expose; */

	    CB_SetArmed(w, True);
	    XmCascadeButtonHighlight(w, True);
	    /*
	    (exp) (w, event, (Region)NULL);
	    */
    	}
    	break;
    case XmMENU_DISARM:
    	{
	  /* XtExposeProc exp = XtClass(w)->core_class.expose; */

	    CB_SetArmed(w, False);
	    XmCascadeButtonHighlight(w, False);
	    /*
	    (exp) (w, event, (Region)NULL);
	    */
    	}
    	break;
    default:
	_XmWarning(w, "%s(%d) - Invalid menuProc function", __FILE__, __LINE__);
	break;
    }

    va_end(arg_list);
}

void
XmCascadeButtonHighlight(Widget cb, Boolean highlight)
{
    DEBUGOUT(XdbDebug(__FILE__, cb,
	     "XmCascadeButtonHighlight(hl %d, armed %d, apm 0x%X, cpm 0x%X)\n",
		      highlight,
		      XmIsPrimitive(cb) ? CB_IsArmed(cb) : CBG_IsArmed(cb),
		  XmIsPrimitive(cb) ? CB_ArmedPixmap(cb) : CBG_ArmedPixmap(cb),
	    XmIsPrimitive(cb) ? CB_CascadePixmap(cb) : CBG_CascadePixmap(cb)));

    if (!XtIsRealized(cb))
    {
	return;
    }

    if (XmIsGadget(cb))
    {
	XmCascadeButtonGadgetHighlight(cb, highlight);

	return;
    }
    else if (!XmIsPrimitive(cb))
    {
	_XmError(cb,
		 "XmCascadeButtonHighlight called with non-cascade "
		 "button widget");

	return;
    }

#ifdef	DO_SANITY
    if (CB_CascadePixmap(cb) == 0)
    {
	DEBUGOUT(XdbDebug(__FILE__, cb, "CascadePixmap has NULL value\n"));
    }

    if (CB_ArmedPixmap(cb) == 0)
    {
	DEBUGOUT(XdbDebug(__FILE__, cb, "ArmedPixmap has NULL value\n"));
    }
#endif

    if (Lab_MenuType(cb) == XmMENU_OPTION)
    {
	_XmDrawShadows(XtDisplayOfObject(cb),
		       XtWindowOfObject(cb),
		       Prim_TopShadowGC(cb),
		       Prim_BottomShadowGC(cb),
		       Lab_Highlight(cb),
		       Lab_Highlight(cb),
		       XtWidth(cb) - 2 * Lab_Highlight(cb),
		       XtHeight(cb) - 2 * Lab_Highlight(cb),
		       Lab_Shadow(cb),
		       highlight ? (int)XmSHADOW_OUT : (int)XmNO_LINE);
    }
    else
    {
	_XmDrawShadows(XtDisplayOfObject(cb),
		       XtWindowOfObject(cb),
		       Prim_TopShadowGC(cb),
		       Prim_BottomShadowGC(cb),
		       Lab_Highlight(cb),
		       Lab_Highlight(cb),
		       XtWidth(cb) - 2 * Lab_Highlight(cb),
		       XtHeight(cb) - 2 * Lab_Highlight(cb),
		       Lab_Shadow(cb),
		       highlight ? (int)XmSHADOW_OUT : (int)XmNO_LINE);
    }

    /* now draw the pixmap 
     * -- this pixmap stuff is (at least until we get the other pixmap)
     * only for cascade buttons inside pulldowns.  Don't do it for
     * option menus. */
    switch (Lab_MenuType(cb))
    {
    case XmMENU_POPUP:
    case XmMENU_PULLDOWN:
    case XmMENU_OPTION:	/* rws 23 Mar 1997 */

	if (CB_IsArmed(cb))
	{
	    if (CB_ArmedPixmap(cb) != XmUNSPECIFIED_PIXMAP &&
		CB_ArmedPixmap(cb) != 0)
	    {
		DEBUGOUT(XdbDebug(__FILE__, cb,
				  "XCopyArea 0x%X -> 0x%X %d %d %dx%d\n",
				  CB_ArmedPixmap(cb), XtWindow(cb),
				  CB_Cascade_x(cb), CB_Cascade_y(cb),
				  CB_Cascade_width(cb), CB_Cascade_height(cb)));

		XCopyArea(XtDisplay(cb), CB_ArmedPixmap(cb), XtWindow(cb),
			  Lab_NormalGC(cb),
			  0, 0,
			  CB_Cascade_width(cb), CB_Cascade_height(cb),
			  CB_Cascade_x(cb), CB_Cascade_y(cb));
	    }
	}
	else if (CB_CascadePixmap(cb) != XmUNSPECIFIED_PIXMAP &&
		 CB_CascadePixmap(cb) != 0)
	{
	    DEBUGOUT(XdbDebug(__FILE__, cb,
			      "XCopyArea 0x%X -> 0x%X geo %d %d %dx%d\n",
			      CB_CascadePixmap(cb), XtWindow(cb),
			      CB_Cascade_x(cb), CB_Cascade_y(cb),
			      CB_Cascade_width(cb), CB_Cascade_height(cb)));

	    XCopyArea(XtDisplay(cb), CB_CascadePixmap(cb), XtWindow(cb),
		      Lab_NormalGC(cb),
		      0, 0,
		      CB_Cascade_width(cb), CB_Cascade_height(cb),
		      CB_Cascade_x(cb), CB_Cascade_y(cb));
	}
    }
}

Widget
XmCreateCascadeButton(Widget parent, char *name,
		      Arg *arglist, Cardinal argcount)
{
    return XtCreateWidget(name,
			  xmCascadeButtonWidgetClass,
			  parent,
			  arglist, argcount);
}
