/* $Id: mode.c,v 1.29 1998/10/18 15:53:12 becka Exp $
***************************************************************************

   Mode management for XF86DGA

   Copyright (C) 1997,1998 Steve Cheng   [steve@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "Xvisual.h"

static int GGI_xf86dga_setdisplayframe(ggi_visual *vis, int num)
{
	struct Xhooks *xhook=LIBGGI_PRIVATE(vis);
	ggi_directbuffer *db = _ggi_db_find_frame(vis, num);

	if (db == NULL) {
		return -1;
	}

	vis->d_frame_num = num;

	XF86DGASetViewPort(xhook->display, xhook->screen,
			   vis->origin_x, LIBGGI_MODE(vis)->virt.y 
			   * vis->d_frame_num + vis->origin_y);

	return 0;
}


static int GGI_xf86dga_setorigin(ggi_visual *vis,int x,int y)
{
	struct Xhooks *xhook=LIBGGI_PRIVATE(vis);
	ggi_mode *mode=LIBGGI_MODE(vis);

	if (x<0 || x > mode->virt.x - mode->visible.x ||
	    y<0 || y > mode->virt.y - mode->visible.y)
	     return -1;

	XF86DGASetViewPort(xhook->display, xhook->screen,
			   x, mode->virt.y*vis->d_frame_num + y);

	vis->origin_x=x;
	vis->origin_y=y;
	
	return 0;
}

/*
** Finds the appropriate XF86VidMode modeline for x/y.
*/
static int _GGI_xf86dga_findmode(ggi_visual *vis, int visible_x, int visible_y)
{
	struct Xhooks *xhook=LIBGGI_PRIVATE(vis);
	int i;

	/* Find a suitable XF86VidMode and return it */
	for(i = 0; i<xhook->num_modes; i++)
		if(	visible_x == xhook->modes[i].x &&
			visible_y == xhook->modes[i].y)
			{ return i; }

	/* mode not found */
	return -1;
}


int GGI_xf86dga_getapi(ggi_visual *vis,int num, char *apiname ,char *arguments)
{
	switch(num) {
		case 0: strcpy(apiname, "display-dga");
			strcpy(arguments, "");
			return 0;
		case 1: strcpy(apiname, "generic-stubs");
			strcpy(arguments, "");
			return 0;
		case 2: strcpy(apiname, "generic-color");
			strcpy(arguments, "");
			return 0;
		case 3:
			sprintf(apiname, "generic-linear-%d", GT_SIZE(LIBGGI_MODE(vis)->graphtype));
			strcpy(arguments,"");
			return 0;
	}
	return -1;
}


static int _GGI_xf86dga_maskgetshift(int mask)
{
	int shift;
	for (shift = 16; mask != 0; mask >>= 1) {
		shift--;
	}
	return shift;
}


int GGI_xf86dga_setmode(ggi_visual *vis,ggi_mode *tm)
{ 
	struct Xhooks *xhook;
	XVisualInfo    vinfo;
	int flags, i, err, id;
	char sugname[256],args[256];


	/* First, check if the mode can be set */
	if(GGI_xf86dga_checkmode(vis, tm))
		return -1;

	xhook=LIBGGI_PRIVATE(vis);

	_GGI_xf86dga_freedbs(vis);

	/* Set XF86VidMode */
	XF86VidModeSwitchToMode(xhook->display, xhook->screen,
		xhook->dgamodes[
			_GGI_xf86dga_findmode(vis, tm->visible.x, tm->visible.y)
		]);

/* 
 * Not needed. We already have the information.
 *
 * However, for the paranoid, the fields in xhook can be checked 
 * against the information here, and panic if they don't match.
 */
#if 0
	XF86DGAGetViewPortSize(xhook->display, xhook->screen, 
		(int*) &tm->visible.x, (int*) &tm->visible.y);
	DPRINT("viewport: %dx%d\n", tm->visible.x, tm->visible.y);
#endif	
	
	/* Get all the events */
	XGrabKeyboard(xhook->display, DefaultRootWindow(xhook->display), 
		True, GrabModeAsync, GrabModeAsync, CurrentTime);

	XGrabPointer(xhook->display, DefaultRootWindow(xhook->display),
		True, PointerMotionMask | ButtonPressMask | ButtonReleaseMask,
		GrabModeAsync, GrabModeAsync, None, None, CurrentTime);

	if (xhook->cmap)
		XFreeColormap(xhook->display,xhook->cmap);
	if (xhook->cmap2)
		XFreeColormap(xhook->display,xhook->cmap2);

	XMatchVisualInfo(xhook->display, xhook->screen,
			 GT_DEPTH(tm->graphtype),
			 GT_SCHEME(tm->graphtype) == GT_PALETTE ? PseudoColor
			 : TrueColor, &vinfo);

	if (GT_SCHEME(tm->graphtype) == GT_PALETTE) {
		xhook->cmap = XCreateColormap(xhook->display,
				DefaultRootWindow(xhook->display),
				vinfo.visual, AllocAll);
		xhook->cmap2 = XCreateColormap(xhook->display,
				DefaultRootWindow(xhook->display),
				vinfo.visual, AllocAll);

		DPRINT("%d-bit visual: X-lib colormap allocated %x.\n",
		       GT_DEPTH(tm->graphtype), xhook->cmap);
	} else {
		xhook->cmap=0;
		xhook->cmap2=0;
	}

	DPRINT("Ready, now taking crash course!\n"); 

	flags = XF86DGADirectGraphics | XF86DGADirectMouse | XF86DGADirectKeyb;
	XF86DGADirectVideo(xhook->display, xhook->screen, flags);

	_ggiZapMode(vis,~GGI_DL_OPDISPLAY);

	if (GT_SCHEME(tm->graphtype) == GT_PALETTE) {
		XLIB_PRIV(vis)->nocols
			= 1<< GT_DEPTH(tm->graphtype);
		XLIB_PRIV(vis)->activecmap=0;
		
		if(vis->palette)
			free(vis->palette);
		vis->palette = _ggi_malloc(sizeof(ggi_color) * XLIB_PRIV(vis)->nocols);
	}

	xhook->pixperframe = ((GT_SIZE(tm->graphtype)*xhook->stride + 7)/8)
		* tm->virt.y;
	vis->d_frame_num = 0;
	
	/* Set up directbuffers */
	for(i = 0; i < tm->frames; i++) {
		LIBGGI_APPLIST(vis)->last_targetbuf = _ggi_db_add_buffer(LIBGGI_APPLIST(vis), _ggi_db_get_new());
		LIBGGI_APPBUFS(vis)[i]->frame = i;
		LIBGGI_APPBUFS(vis)[i]->type
			= GGI_DB_NORMAL | GGI_DB_SIMPLE_PLB;
		LIBGGI_APPBUFS(vis)[i]->read = LIBGGI_APPBUFS(vis)[i]->write
			= (unsigned char*)(xhook->fb) + xhook->pixperframe * i;
		LIBGGI_APPBUFS(vis)[i]->layout = blPixelLinearBuffer;
		LIBGGI_APPBUFS(vis)[i]->buffer.plb.stride
			= (GT_SIZE(tm->graphtype) * xhook->stride + 7)/8;
		LIBGGI_APPBUFS(vis)[i]->buffer.plb.pixelformat
			= LIBGGI_PIXFMT(vis);
		DPRINT_MODE("DB: %d, addr: %p, stride: %d\n", i, 
			    LIBGGI_APPBUFS(vis)[i]->read, 
			    LIBGGI_APPBUFS(vis)[i]->buffer.plb.stride);
	}
	LIBGGI_APPLIST(vis)->first_targetbuf = LIBGGI_APPLIST(vis)->last_targetbuf - (tm->frames-1);
	
	vis->origin_x=0;
	vis->origin_y=0;

	/* Fill in ggi_pixelformat */
	memset(LIBGGI_PIXFMT(vis), 0, sizeof(ggi_pixelformat));

	LIBGGI_PIXFMT(vis)->red_shift  =
		16 + _GGI_xf86dga_maskgetshift(
			LIBGGI_PIXFMT(vis)->red_mask=vinfo.red_mask);
	LIBGGI_PIXFMT(vis)->green_shift  =
		16 + _GGI_xf86dga_maskgetshift(
			LIBGGI_PIXFMT(vis)->green_mask=vinfo.green_mask);
	LIBGGI_PIXFMT(vis)->blue_shift  =
		16 + _GGI_xf86dga_maskgetshift(
			LIBGGI_PIXFMT(vis)->blue_mask=vinfo.blue_mask);
	LIBGGI_PIXFMT(vis)->depth = GT_DEPTH(tm->graphtype);
	LIBGGI_PIXFMT(vis)->size = GT_SIZE(tm->graphtype);

	if (GT_SCHEME(tm->graphtype) == GT_PALETTE)
		LIBGGI_PIXFMT(vis)->clut_mask = (1 << GT_DEPTH(tm->graphtype)) - 1;
	else
		LIBGGI_PIXFMT(vis)->clut_mask = 0;

	DPRINT_MODE("X: Red,   shift: %ld, mask: 0x%x\n",
		    LIBGGI_PIXFMT(vis)->red_shift,
		    LIBGGI_PIXFMT(vis)->red_mask);
	DPRINT_MODE("X: Green, shift: %ld, mask: 0x%x\n",
		    LIBGGI_PIXFMT(vis)->green_shift,
		    LIBGGI_PIXFMT(vis)->green_mask);
	DPRINT_MODE("X: Blue,  shift: %ld, mask: 0x%x\n",
		    LIBGGI_PIXFMT(vis)->blue_shift,
		    LIBGGI_PIXFMT(vis)->blue_mask);

	memcpy(LIBGGI_MODE(vis),tm,sizeof(ggi_mode));

	for(id=1;0==GGI_xf86dga_getapi(vis,id,sugname,args);id++) {
		err=(_ggiOpenDL(vis, sugname, args, NULL)==NULL);
		if (err) {
			fprintf(stderr,"display-dga: Can't open the %s (%s) library.\n",
				sugname,args);
			/* In our special case, fail is always fatal. */
			return err;
		} else {
			DPRINT_MODE("Success in loading %s (%s)\n",sugname,args);
		}
	}

	vis->opdraw->setorigin=GGI_xf86dga_setorigin;
	vis->opdraw->setdisplayframe=GGI_xf86dga_setdisplayframe;

	/* color */
	if (GT_SCHEME(tm->graphtype) == GT_PALETTE)
		vis->opcolor->setpalvec=GGI_xf86dga_setpalvec;
#if 0	/* generic-color does this for us */
	vis->opcolor->mapcolor=GGI_xf86dga_mapcolor;
	vis->opcolor->unmappixel=GGI_xf86dga_unmappixel;
#endif

	ggiIndicateChange(vis, GGI_CHG_APILIST);

	return 0;
}

#define WANT_CHECKONEBPP
#define _GGIcheckonebpp _GGI_xf86dga_checkonebpp
#include "../common/modelist.inc"

/**********************************/
/* check any mode (text/graphics) */
/**********************************/
int GGI_xf86dga_checkmode(ggi_visual *vis,ggi_mode *tm)
{
	struct Xhooks *xhook;
	int err = 0;	/* Flag: if the mode being checked is not found */

	if (vis==NULL) {
		DPRINT("Visual==NULL\n");
		return -1;
	}

	xhook=LIBGGI_PRIVATE(vis);

	if (tm->visible.x == GGI_AUTO) tm->visible.x = xhook->width;
	if (tm->visible.y == GGI_AUTO) tm->visible.y = tm->virt.y;

	if ((tm->dpp.x!=1 && tm->dpp.x!=GGI_AUTO) ||
	    (tm->dpp.y!=1 && tm->dpp.y!=GGI_AUTO)) {
		err = -1;
	}
	tm->dpp.x = 1;
	tm->dpp.y = 1;

	if (GT_DEPTH(tm->graphtype) != xhook->depth 
	    || GT_SIZE(tm->graphtype) != xhook->size) {
		if (tm->graphtype != GT_AUTO)
			err = -1;
		tm->graphtype = GT_CONSTRUCT(xhook->depth,
		      (xhook->depth <= 8) ? GT_PALETTE : GT_TRUECOLOR,
		      xhook->size);
	}
	
	/* Try to find a suitable XF86DGA mode. */
#if 1
	if (_GGI_xf86dga_checkonebpp(vis, tm, XLIB_PRIV(vis)->modes))
		err = -1;
#else
	if(_GGI_xf86dga_findmode(vis, tm->visible.x, tm->visible.y)==-1) {
		int sug_x, sug_y, new_x, new_y, i;

		/* If that cannot be done, flag it as an error */
		err = -1;

		/* Then try to find another mode with
		   the next higher resolution */

		sug_x = sug_y = 0x7FFFFFFF;	/* Just the maximum */

		for(i = 0; i<xhook->num_modes; i++) {
			new_x = xhook->modes[i].x;
			new_y = xhook->modes[i].y;
			if(	new_x >= tm->visible.x &&
				new_y >= tm->visible.y &&
				new_x <= sug_x &&
				new_y <= sug_y)
			{
				sug_x = new_x;
				sug_y = new_y;
			}
		}
		
		/* If the above failed, then use
		   the highest resolution possible */
		if(	sug_x == 0x7FFFFFFF &&
			sug_y == 0x7FFFFFFF)
		{
			sug_x = sug_y = 0;

			for(i = 0; i<xhook->num_modes; i++) {
				new_x = xhook->modes[i].x;
				new_y = xhook->modes[i].y;
				if(new_x >= sug_x && new_y >= sug_y) {
					sug_x = new_x;
					sug_y = new_y;
				}
			}
		}
		
		tm->visible.x = sug_x;
		tm->visible.y = sug_y;
	}			
#endif
	if (tm->virt.x==GGI_AUTO) tm->virt.x = xhook->width;
	if (tm->virt.y==GGI_AUTO) tm->virt.y = tm->visible.y;

	/* Virtual x-resolution is unchangeable */
	if(tm->virt.x != xhook->width) {
		tm->virt.x = xhook->width;
		err = -1;
	}
	if (tm->virt.y > xhook->height) {
		tm->virt.y = xhook->height;
		err = -1;
	} else if (tm->virt.y < tm->visible.y) {
		tm->virt.y = tm->visible.y;
		err = -1;
	}	
	
	if (tm->frames * ((GT_SIZE(tm->graphtype)*xhook->stride + 7)/8) 
	    * tm->virt.y > xhook->mem_size*1024) {
		tm->frames = xhook->mem_size*1024 /
			(((GT_SIZE(tm->graphtype)*xhook->stride + 7)/8)
			 * tm->virt.y);
		err = -1;
	}
	if (tm->frames < 1) {
		if (tm->frames != GGI_AUTO) {
			err = -1;
		}
		tm->frames = 1;
	}

	return err;	
}

/************************/
/* get the current mode */
/************************/
int GGI_xf86dga_getmode(ggi_visual *vis,ggi_mode *tm)
{
	DPRINT("In GGI_xf86dga_getmode(%p,%p)\n",vis,tm);
	if (vis==NULL)
		return -1;
	
	if (LIBGGI_MODE(vis)==NULL)
		return -1;

	memcpy(tm,LIBGGI_MODE(vis),sizeof(ggi_mode));

	return 0;
}

/*************************/
/* set the current flags */
/*************************/
int GGI_xf86dga_setflags(ggi_visual *vis,ggi_flags flags)
{
	LIBGGI_FLAGS(vis)=flags;
	return 0;
}
