/* $Id: visual.c,v 1.31 1998/10/24 04:03:49 marcus Exp $
***************************************************************************

   Initializing tiles

   Copyright (C) 1998 Steve Cheng    [steve@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "tilevisual.h"

void _GGI_tile_freedbs(ggi_visual *vis) {
	int i;
        
	for (i=LIBGGI_APPLIST(vis)->num-1; i >= 0; i--) {
		free(LIBGGI_APPBUFS(vis)[i]->write);
		_ggi_db_free(LIBGGI_APPBUFS(vis)[i]);
		_ggi_db_del_buffer(LIBGGI_APPLIST(vis), i);
	}
}


#define ARGUMENT_FORMAT  "display-tile:\n\
    The argument format is `offx,offy,sizex,sizey,(subdisplay):...',\n\
    where offx and offy are the tile's offset from the main display,\n\
    sizex and sizey are the size of the tile,\n\
    subdisplay is the display string to open for the tile,\n\
    and ... is more tiles following the same format as above...\n"

int GGIdlinit(ggi_visual *vis,const char *args,void *argptr)
{
	struct TileHooks *tilehook;
	char target[1024];
	int sx, sy, vx, vy, n, i=0;

	if (! args) {
		fprintf(stderr, "display-tile needs the real targets as arguments.\n");
		fprintf(stderr, ARGUMENT_FORMAT);
		return -1;
	}

	LIBGGI_GC(vis) = _ggi_malloc(sizeof(ggi_gc));

	LIBGGI_PRIVATE(vis) = tilehook = _ggi_malloc(sizeof(struct TileHooks));
	tilehook->opmansync = _ggi_malloc(sizeof(_ggi_opmansync));

	tilehook->buf=NULL;
	tilehook->use_db=1;

	/* parse each visual */
	for (;;) {

		sx = sy = vx = vy = 0;

		while (*args && isspace((int)*args)) args++;

		if (! *args) break;

		if (strncmp(args, "-usedb:", 7) == 0) {
			DPRINT_MISC("display-tile: Enabling DB\n");
			tilehook->use_db = 1;
			args += 7; continue;
		}
		if (strncmp(args, "-nodb:", 6) == 0) {
			DPRINT_MISC("display-tile: Disabling DB\n");
			tilehook->use_db = 0;
			args += 6; continue;
		}

		/* Avoid overflowing tables. If someone ever needs more than 256
		   we'll do dynamic allocation instead. */
		if (i==MAX_VISUALS)
			ggiPanic("display-tile: FIXME: visual limit reached!\n");

		if ((sscanf(args, "%d , %d , %d , %d %n", &sx, &sy,
			    &vx, &vy, &n) != 4) || 
		    (args += n, *args++ != ',')) {

			fprintf(stderr, ARGUMENT_FORMAT);
			free(tilehook);
			return GGI_DL_ERROR;
		}
		
		/* Empty tile! */
		if(vx<=0 || vy<=0 || sx<0 || sy<0) {
			fprintf(stderr, "display-tile: erroneous coordinates for tile #%d!\n", i);
			free(tilehook);
			return GGI_DL_ERROR;
		}

		tilehook->vis_origins[i].x = sx;
		tilehook->vis_origins[i].y = sy;
		tilehook->vis_sizes[i].x = vx;
		tilehook->vis_sizes[i].y = vy;

		args = ggParseTarget((char *)args, target, 1024);

		if (! args) {
			fprintf(stderr,"display-tile: parsetarget error.\n");
			free(tilehook);
			return GGI_DL_ERROR;
		}

		DPRINT_MISC("display-tile: visual #%d is %s (%d,%d)[%dx%d]\n",
			i, target, sx, sy, vx, vy);

		if (! (tilehook->vislist[i]=ggiOpen(target,NULL)) ) {
			fprintf(stderr,"display-tile: Opening of target %s failed.\n", target);
			free(tilehook);
			return GGI_DL_ERROR;
		}

		/* Add giiInputs, if we have them. */
		if (tilehook->vislist[i]->input)
		{
			vis->input=giiJoinInputs(vis->input,tilehook->vislist[i]->input);
			tilehook->vislist[i]->input=NULL;	/* Destroy old reference */
		}

		if(tilehook->use_db) {
			/* Don't need SYNC mode, we do it ourselves */
			ggiSetFlags(tilehook->vislist[i], GGIFLAG_ASYNC);
		}

		/* check for ':' separator */
		
		while (*args && isspace((int)*args)) args++;
		
		if (*args && (*args != ':')) {
			fprintf(stderr, "display-tile: expecting ':' between targets.\n");
			free(tilehook);
			return GGI_DL_ERROR;
		}

		if (*args == ':') args++;
		i++;
	}

	tilehook->numvis=i;

	if (tilehook->numvis == 0) {
		fprintf(stderr, "display-tile needs the real targets as arguments.\n");
		free(tilehook);
		return -1;
	}

	LIBGGI_SELECT_FD(vis) = LIBGGI_SELECT_FD(tilehook->vislist[0]);

	if(tilehook->use_db) {
		if(!_ggiAddDL(vis, "helper-mansync", (char*) tilehook->opmansync, NULL, 0)) {
			fprintf(stderr, "display-tile: Cannot load required helper-mansync! (for DB mode)\n");
			return GGI_DL_ERROR;
		}

		MANSYNC_init(vis);
	}

	/* Has mode management */
	vis->opdisplay->getmode=GGI_tile_getmode;
	vis->opdisplay->setmode=GGI_tile_setmode;
	vis->opdisplay->checkmode=GGI_tile_checkmode;
	vis->opdisplay->getapi=GGI_tile_getapi;
	vis->opdisplay->setflags=GGI_tile_setflags;
	
	if(tilehook->use_db)
		vis->opdisplay->flush=GGI_tile_flush_db;
	else
		vis->opdisplay->flush=GGI_tile_flush;

	/* Has event management */
	vis->opdisplay->eventpoll=GGI_tile_eventpoll;
	vis->opdisplay->eventread=GGI_tile_eventread;

	return GGI_DL_OPDISPLAY;
}

int GGIdlcleanup(ggi_visual *vis)
{
	struct TileHooks *tilehook = LIBGGI_PRIVATE(vis);
	int i;

	if(tilehook->use_db) {
		MANSYNC_deinit(vis);

		_GGI_tile_freedbs(vis);
	}

	if(tilehook->buf)
		free(tilehook->buf);

	for(i = 0; i<tilehook->numvis; i++)
		ggiClose(tilehook->vislist[i]);

	free(tilehook->opmansync);
	free(tilehook);
	free(LIBGGI_GC(vis));

	return 0;
}


