/* $Id: vtswitch.c,v 1.7 1998/09/20 21:24:46 marcus Exp $
***************************************************************************

   Display-SUID

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <dlfcn.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>

#include <sys/ioctl.h>
#include <sys/mman.h>

#include <linux/vt.h>
#include <linux/kd.h>

#include <ggi/ggi.h>

#include <ggi/internal/ggi-dl.h>
#include <ggi/internal/internal.h>
#include "debug.h"


static int our_vt  = -1;
static int last_vt = -1;

static int vt_fd = -1;   /* same as kbd_fd used elsewhere */


/* Variables for the signal handlers of the horrible VT_PROCESS thingy
 */

static volatile int __vt_back;
static ggi_visual * __vt_vis;


extern void handle_switched_away(ggi_visual *vis);
extern void handle_switched_back(ggi_visual *vis);


#define SETSIG(sig, fun) \
{ \
	struct sigaction sa; \
\
	sa.sa_handler = fun; \
	sa.sa_flags = 0; \
	sa.sa_mask = 0; \
	sigaction(sig, &sa, NULL); \
}


static void release_vt_handler(int n)
{
	SETSIG(SIGUSR1, release_vt_handler);

	__vt_back=0;

	DPRINT("release_vt_handler START\n");

	/* Lock the console while we reset the graphics hardware */

/* !!!	ioctl(vt_fd, VT_LOCKSWITCH);  */
	{
		handle_switched_away(__vt_vis);
	}
/* !!!	ioctl(vt_fd, VT_UNLOCKSWITCH);  */

	/* now release the VT */

	ioctl(vt_fd, VT_RELDISP, 1);

	/* suspend program until we get switched back */
	
	DPRINT("release_vt_handler SUSPEND\n");

	while (! __vt_back) {    

		/* Note: we rely on the acquire signal interrupting us
		 */
		pause();
	}

	DPRINT("release_vt_handler DONE\n");
}


static void acquire_vt_handler(int n)
{
	SETSIG(SIGUSR2, acquire_vt_handler);

	DPRINT("acquire_vt_handler START\n");

	/* acknowledge the VT */

	ioctl(vt_fd, VT_RELDISP, VT_ACKACQ);

	/* Lock the console while we restore the graphic hardware to
	 * what we had before the console switch.
	 */

/* !!!	ioctl(vt_fd, VT_LOCKSWITCH);   */
	{
		handle_switched_back(__vt_vis);
	}
/* !!!	ioctl(vt_fd, VT_UNLOCKSWITCH);   */

	__vt_back=1;

	DPRINT("acquire_vt_handler DONE\n");
}


int vtswitch_open(ggi_visual *vis)
{
	/* do the VT switching magic here */

	char filename[80];

	int qry_fd;
	struct vt_stat qry_stat;
	struct vt_mode qry_mode;


	/* query free VT and current VT */

	qry_fd = open("/dev/tty0", O_RDWR);

	if (qry_fd < 0) {
		perror("display-fbdev:/dev/tty0");
		return -1;
	}

	if (ioctl(qry_fd, VT_OPENQRY, &our_vt) != 0) {
		DPRINT("display-fbdev: VT_OPENQRY failed (no free "
			"consoles).\n");
		close(qry_fd);
		return -1;
	}
	
	if (ioctl(qry_fd, VT_GETSTATE, &qry_stat) != 0) {
		DPRINT("display-fbdev: VT_GETSTATE failed.\n");
		close(qry_fd);
		return -1;
	}
	
	last_vt = qry_stat.v_active;

	close(qry_fd);

	DPRINT("display-fbdev: free VT=%d, current VT=%d\n", our_vt, last_vt);
	

	/* detach from current tty */

	if (1)
	{
		qry_fd = open("/dev/tty", O_WRONLY);

		if (qry_fd >= 0) {
			ioctl(qry_fd, TIOCNOTTY);
			close(qry_fd);
		}

		DPRINT("display-fbdev: Detached from current tty.\n");
	}


	/* open new VT */

	sprintf(filename, "/dev/tty%d", our_vt);

	vt_fd = open(filename, O_RDWR | O_NDELAY); if (vt_fd < 0) {
		fprintf(stderr, "display-fbdev: couldn't open '%s'.\n",
			filename);
		return -1;
	}

	fprintf(stderr, "display-fbdev: Using VT %d.\n", our_vt);
	
	
	if ((ioctl(vt_fd, VT_ACTIVATE,   our_vt) < 0) ||
	    (ioctl(vt_fd, VT_WAITACTIVE, our_vt) < 0)) {

		fprintf(stderr, "display-fbdev: Activating VT failed.\n");
		close(vt_fd);
		return -1;
	}

	/* Disable normal text on the console */

	ioctl(vt_fd, KDSETMODE, KD_GRAPHICS);
	

	/* Unfortunately, we need to take control over VT switching like
	 * Xfree and SVGAlib does.  Sigh.
	 */

	ioctl(vt_fd, VT_GETMODE, &qry_mode);

	qry_mode.mode = VT_PROCESS;
	qry_mode.relsig = SIGUSR1;
	qry_mode.acqsig = SIGUSR2;
	
	__vt_back = 1;
	__vt_vis  = vis;

	SETSIG(SIGUSR1, release_vt_handler);
	SETSIG(SIGUSR2, acquire_vt_handler);

	if (ioctl(vt_fd, VT_SETMODE, &qry_mode) < 0) {
		fprintf(stderr, "display-fbdev: Setting VT mode failed.\n");
		close(vt_fd);
		return -1;
	}


	LIBGGI_SELECT_FD(vis) = vt_fd;

	DPRINT("display-fbdev: vtswitch_open Success.\n");

	return vt_fd;
}


int vtswitch_close(ggi_visual *vis)
{
	struct vt_mode qry_mode;
	

	/* handle VT */

	ioctl(vt_fd, KDSETMODE, KD_TEXT);

	if (ioctl(vt_fd, VT_GETMODE, &qry_mode) == 0) {
		qry_mode.mode = VT_AUTO;
		ioctl(vt_fd, VT_SETMODE, &qry_mode);
	}
	
	signal(SIGUSR1, SIG_DFL);
	signal(SIGUSR2, SIG_DFL);

	ioctl(vt_fd, VT_ACTIVATE,   last_vt);
	ioctl(vt_fd, VT_WAITACTIVE, last_vt);

	close(vt_fd);
		

	LIBGGI_SELECT_FD(vis) = -1;
	LIBGGI_FB_LINEAR(vis) = NULL;

	DPRINT("display-fbdev: vtswitch_close Success.\n");

	return 0;
}
