/* $Id: crossblit.c,v 1.2 1998/10/06 21:19:07 becka Exp $
***************************************************************************

   Graphics library for GGI.

   Copyright (C) 1998 Andreas Beck [becka@ggi-project.org]
   Copyright (C) 1997 Jason McMullan [jmcc@ggi-project.org]
   Copyright (C) 1998 Andrew Apted [andrew.apted@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <ggi/internal/ggi-dl.h>

int (*fallback_crossblit) (ggi_visual *src,int sx,int sy,int w, int h,
			   ggi_visual *dst,int dx,int dy);

/* Help function for 8 bit to 16 bit crossblitting.
 */
static void crossblit_8_to_16(ggi_visual *src, int sx, int sy, int w, int h,
			      ggi_visual *dst, int dx, int dy)
{
	char *srcp, *dstp;
	uint16 conv_tab[256];
	int i;

	DPRINT_LIBS("linear-16: crossblit_8_to_16.\n");

	/* Creates the conversion table. A bit simplistic approach, perhaps?
	 */
	for (i = 0; i < 256; i++) {
		ggi_pixel pixel;
		ggi_color col;

		ggiUnmapPixel(src, i, &col);
		pixel = ggiMapColor(dst, &col);
		conv_tab[i] = (uint16) pixel;
	}

	srcp = (char *) LIBGGI_CURREAD (src);
	dstp = (char *) LIBGGI_CURWRITE(dst);
	
	srcp += LIBGGI_FB_R_STRIDE(src)*sy + sx;
	dstp += LIBGGI_FB_W_STRIDE(dst)*dy + dx + dx;

	for (; h > 0; h--) {
		uint16 *dstpw = (uint16 *) dstp;
		uint8 *srcpb  = (uint8 *)  srcp;
		i = w >> 3;

		/* We don't believe in the optimizing capabilities of the
		 * compiler hence unroll manually.
		 */

		switch (w & 0x7) {
			for (; i > 0; i--) {
			case 0x0: *dstpw++ = conv_tab[ (uint32) *srcpb++];
			case 0x7: *dstpw++ = conv_tab[ (uint32) *srcpb++];
			case 0x6: *dstpw++ = conv_tab[ (uint32) *srcpb++];
			case 0x5: *dstpw++ = conv_tab[ (uint32) *srcpb++];
			case 0x4: *dstpw++ = conv_tab[ (uint32) *srcpb++];
			case 0x3: *dstpw++ = conv_tab[ (uint32) *srcpb++];
			case 0x2: *dstpw++ = conv_tab[ (uint32) *srcpb++];
			case 0x1: *dstpw++ = conv_tab[ (uint32) *srcpb++];
			}
		}

		srcp += LIBGGI_FB_R_STRIDE(src);
		dstp += LIBGGI_FB_W_STRIDE(dst);
	}
}

/* Help function for crossblitting when destination and source have same stdformat.
 */
static void crossblit_same(ggi_visual *src, int sx, int sy, int w, int h,
			   ggi_visual *dst, int dx, int dy)
{
	char *srcp, *dstp;
	DPRINT_LIBS("linear-16: DB-accelerating crossblit.\n");
	
	srcp = (char *) LIBGGI_CURREAD (src);
	dstp = (char *) LIBGGI_CURWRITE(dst);

	srcp += LIBGGI_FB_R_STRIDE(src)*sy + sx + sx;
	dstp += LIBGGI_FB_W_STRIDE(dst)*dy + dx + dx;

	for (; h > 0; h--) {
		memcpy(dstp, srcp, w << 1);
		srcp += LIBGGI_FB_R_STRIDE(src);
		dstp += LIBGGI_FB_W_STRIDE(dst);
	}
}

/* Cross blitting of a rectangular area.
 */
int GGIcrossblit(ggi_visual *src, int sx, int sy, int w, int h, 
		 ggi_visual *dst, int dx, int dy)
{
	/* clipping */

	if (dx < LIBGGI_GC(dst)->cliptl.x) {
		int diff = LIBGGI_GC(dst)->cliptl.x - dx;
		sx += diff;
		dx += diff;
		w  -= diff;
	}
	if (dy < LIBGGI_GC(dst)->cliptl.y) {
		int diff = LIBGGI_GC(dst)->cliptl.y - dy;
		sy += diff;
		dy += diff;
		h  -= diff;
	}
	if (dx+w > LIBGGI_GC(dst)->clipbr.x) {
		w = LIBGGI_GC(dst)->clipbr.x - dx;
	}
	if (dy+h > LIBGGI_GC(dst)->clipbr.y) {
		h = LIBGGI_GC(dst)->clipbr.y - dy;
	}
	
	if ((w <= 0) || (h <= 0)) {
		return 0;
	}
	
	if (sx < 0 || sy < 0 ||
	    sx+w > LIBGGI_VIRTX(src) || 
	    sy+h > LIBGGI_VIRTY(src)) {

		return fallback_crossblit(src, sx, sy, w, h, dst, dx, dy);
	}

	/* Check if the src-read and dst-write buffers are in the same layout and
	 * that the destination pixelformat is sane.
	 */

	if (src->r_frame->layout == dst->w_frame->layout &&
	    dst->w_frame->buffer.plb.pixelformat->stdformat != 0)
	{
		uint32 srcformat = src->w_frame->buffer.plb.pixelformat->stdformat;
		uint32 dstformat = dst->w_frame->buffer.plb.pixelformat->stdformat;

		/* These are the accelerated cases. If neither suits then
		 * fall back to the default.
		 */

		if (srcformat == GGI_DB_STD_8a8i8 && w * h > 256 ) {
			crossblit_8_to_16(src, sx, sy, w, h, dst, dx, dy);
			return 0;
		}
		else if (srcformat == dstformat) {
			crossblit_same(src, sx, sy, w, h, dst, dx, dy);
			return 0;
		}
	}

	return fallback_crossblit(src, sx, sy, w, h, dst, dx, dy);
}

#if 0
/* This is the old stretching code.  I haven't deleted it -- it could
 * come it handy when implementing ggiStretchBlit() in libggi2d.
 */

	{		/* STETCH */

		ggi_pixel pixel;
		int errx,erry,ddx,ssx,ty,tx,dtx,dty;
	
		ddx=dx;ssx=sx;erry=(dh-sh)>>1;
		ty =sy+sh;tx =sx+sw;
		dty=dy+dh;dtx=dx+dw;
	
		/* The basic stretching core. Please speed me up. Teunis ? */
		for(;sy<ty;sy++,srcp+=LIBGGI_FB_WIDTH(src)*sizeof(uint32)) {
			erry+=dh;
			if (erry<=0) continue;

			while(erry>0) {
				dx=ddx;sx=ssx;errx=0;
				for(;sx<tx;sx++) {
					errx+=dw;
					if (errx<=0) continue;
					pixel=*((uint32 *)srcp+sx);

					while(errx>0 && dx<dtx) {
						*((uint32 *)dstp+dx++)=pixel;
						errx-=sw;
					}
				}
				dy++;erry-=sh;dstp+=plb->stride;
				if (dy>=dty) break;
			}
		}

	}		/* STETCH */
#endif
