// Fl_Tile.C - Group of 2 or 4 "tiles" that can be resized by dragging border

// The size of the first child determines where the resize border is.
// The resizebox is used to limit where the border can be dragged to.

#include <FL/Fl.H>
#include <FL/Fl_Tile.H>
#include <stdlib.h>

// Drag the edges that were initially at oldx,oldy to newx,newy:
// pass zero as oldx or oldy to disable drag in that direction:

void Fl_Tile::position(int oix, int oiy, int newx, int newy) {
  Fl_Widget*const* a = array();
  for (int i=children(); i--;) {
    Fl_Widget* o = *a++;
    if (o == resizable()) continue;
    int f = 0;
    int X = o->x();
    int R = X+o->w();
    if (oix) {
      int t = o->ix();
      if (t == oix || t>oix && X<newx || t<oix && X>newx) {X = newx; f = 1;}
      t = o->ix()+o->iw();
      if (t == oix || t>oix && R<newx || t<oix && R>newx) {R = newx; f = 1;}
    }
    int Y = o->y();
    int B = Y+o->h();
    if (oiy) {
      int t = o->iy();
      if (t == oiy || t>oiy && Y<newy || t<oiy && Y>newy) {Y = newy; f = 1;}
      t = o->iy()+o->ih();
      if (t == oiy || t>oiy && B<newy || t<oiy && B>newy) {B = newy; f = 1;}
    }
    if (f) {o->resize(X,Y,R-X,B-Y); o->redraw();}
  }
}

// move the lower-right corner (sort of):
void Fl_Tile::resize(int X,int Y,int W,int H) {
  // remember how much to move the child widgets:
  int dx = X-x();
  int dy = Y-y();
  int dw = W-w();
  int dh = H-h();
  // resize this (skip the Fl_Group resize):
  Fl_Widget::resize(X,Y,W,H);
  Fl_Widget* r = resizable(); if (!r) r = this;
  // find bottom-right of resiable:
  int OR = r->ix()+r->iw();
  int NR = X+W-(ix()+iw()-OR);
  int OB = r->iy()+r->ih();
  int NB = Y+H-(iy()+ih()-OB);
  // move everything to be on correct side of new resizable:":
  Fl_Widget*const* a = array();
  for (int i=children(); i--;) {
    Fl_Widget* o = *a++;
    int X = o->x()+dx;
    int R = X+o->w();
    if (o->ix() >= OR) X += dw; else if (X > NR) X = NR;
    if (o->ix()+o->iw() >= OR) R += dw; else if (R > NR) R = NR;
    int Y = o->y()+dy;
    int B = Y+o->h();
    if (o->iy() >= OB) Y += dh; else if (Y > NB) Y = NB;
    if (o->iy()+o->ih() >= OB) B += dh; else if (B > NB) B = NB;
    o->resize(X,Y,R-X,B-Y); o->redraw();
  }
}

int Fl_Tile::handle(int event) {
  static int sdrag;
  static int sdx, sdy;
  static int sx, sy;
#define DRAGH 1
#define DRAGV 2
#define GRABAREA 3
#define BIGGRABAREA 20

  int mx = Fl::event_x();
  int my = Fl::event_y();

  switch (event) {

  case FL_PUSH: {
    int mindx = 100;
    int mindy = 100;
    int oldx = 0;
    int oldy = 0;
    Fl_Widget*const* a = array();
    for (int i=children(); i--;) {
      Fl_Widget* o = *a++;
      if (o == resizable()) continue;
      if (o->ix()+o->iw() < ix()+iw() && o->y()<=my && o->y()+o->h()>=my) {
	int t = mx - (o->x()+o->w());
	if (abs(t) < mindx) {
	  mindx = abs(t);
	  oldx = o->ix()+o->iw();
	}
      }
      if (o->iy()+o->ih() < iy()+ih() && o->x()<=mx && o->x()+o->w()>=mx) {
	int t = my - (o->y()+o->h());
	if (abs(t) < mindy) {
	  mindy = abs(t);
	  oldy = o->iy()+o->ih();
	}
      }
    }
    if (mindx > GRABAREA && mindy > GRABAREA)
      if (Fl_Group::handle(event)) return 1;
    sdrag = 0; sx = sy = 0;
    if (mindx < BIGGRABAREA) {sdrag = DRAGH; sx = oldx;}
    if (mindy < BIGGRABAREA) {sdrag |= DRAGV; sy = oldy;}
    return sdrag != 0;
  }

 case FL_DRAG: {
    if (!sdrag) return 0; // should not happen
    Fl_Widget* r = resizable(); if (!r) r = this;
    int newx;
    if (sdrag&DRAGH) {
      newx = Fl::event_x()-sdx;
      if (newx < r->x()) newx = r->x();
      else if (newx > r->x()+r->w()) newx = r->x()+r->w();
    } else
      newx = sx;
    int newy;
    if (sdrag&DRAGV) {
      newy = Fl::event_y()-sdy;
      if (newy < r->y()) newy = r->y();
      else if (newy > r->y()+r->h()) newy = r->y()+r->h();
    } else
      newy = sy;
    position(sx,sy,newx,newy);
    return 1;}

  case FL_RELEASE:
    sdrag = 0;
    return 1;
 }

  return Fl_Group::handle(event);
}

// end of Fl_Tile.C
