package Net::Finger;

BEGIN {
	use strict;
	use IO::Socket;
	use Exporter ();

	use vars qw(@ISA @EXPORT $VERSION);
	@ISA = qw(Exporter);
	@EXPORT = qw(&finger);
}

$VERSION = '1.0';

sub _call_finger {
	my ($host, $port) = @_;
	my $sd = \*S;

	$host     ||= 'localhost';
	if ($port !~ /\d+/) {
		$port = (getservbyname('finger', 'tcp'))[2] || 79;
	}

	$sd = IO::Socket::INET->new(PeerAddr => $host,
	                            PeerPort => $port,
	                            Proto    => 'tcp');
	return $sd;
}

sub finger {
	my %params = @_;
	my (@response, $username, $host, $port, $sd);

	$username = $params{'Username'};
	$host     = $params{'Host'};
	if ($host =~ /:/) {
		($host, $port) = split(/:/, $host);
	} else {
		$port = $params{'Port'};
	}

	$sd = Net::Finger::_call_finger($host, $port) or return undef;
	$sd->send("$username\r\n");
	while (<$sd>) {
		push (@response, $_);
	}
	return wantarray ? @response : \@response;
}

1;

__END__

=head1 NAME

Net::Finger - Perl finger client interface

=head1 SYNOPSIS

  use Net::Finger;

  $response = finger('Username' => 'foobar',
                     'Host' => 'some.domain.com');


=head1 DESCRIPTION

Net::Finger connects to a local or remote finger daemon and makes a finger
request.

=over 4

=item Getting a finger response

   $response = finger('Username' => 'foobar',
                      'Host' => 'some.domain.com');
                      'Port' => '79');

   or 

   @response = finger('Username' => 'foobar',
                      'Host' => 'some.domain.com',
                      'Port => '79');

Get finger information on I<Host> using the protocol defined in RFC1288.
If no I<Host> is specified,  'localhost' is used.  If no I<Userid> is
specified, a null request is sent.  The default port of 79 is used if a
non-numeric port (or no port) is specified, and a port cannot be obtained
using B<getservbyname>.

You may use I<domain.somewhere.com:79> notation for the host and port.  If
you do, the module will use that port even if you specify a different port
afterwards.

The finger function returns a reference to an array containing the response
if it is called in scalar context.  If called in array context, it returns
an array.  Calling the function in scalar context is preferable, as it
can save needless coyping of data.

=item Parsing the output:

Parsing the finger output is pretty much left as an exercise for the
reader/coder.  Finger output varies too much from machine to machine.
RFC1288 only requires a finger daemon to provide the user's full name.
Actually, it also requires vending machines to never eat your money, but
how often do either of these really happen?

=back

=head1 AUTHOR

  Frank Sheiness <archon@unix.bigots.org>

=head1 COPYRIGHT

Copyright (c) 1997 Frank Sheiness. All rights reserved.  This program is
free software; you can redistribute it and/or modify it under the same
terms as Perl itself.

=head1 CREDITS

  Graham Barr <gbarr@ti.com>
  Randal Schwartz <merlyn@stonehenge.com>

=cut

1;

__END__
