/*****************************************************************************
 *                                                                           *
 * Programm:  paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Verwendet: GTK, Imlib                                                     *
 * Modul:     gcutarea.c                                                     *
 *            Callback function to cut area                                  *
 * Autor:     Andreas Tille                                                  *
 * Datum:     07.10.1998                                                     *
 *                                                                           *
 *****************************************************************************/

#include <unistd.h>
#include <assert.h>
#include <gdk/gdkkeysyms.h>
#include "paul.h"
#include "callback.h"

/**********************************************************************************
 * Global variables                                                               *
 **********************************************************************************/

typedef enum {
   NOWHERE,
   OUT,
   LEFT, 
   TOP,
   RIGHT,
   BOTTOM,
   LEFTTOP,
   RIGHTTOP,
   RIGHTBOTTOM,
   LEFTBOTTOM,
   INNER
} CURSOR_LOCATION;
#define CORNER  (INNER + 1)

typedef struct 
{
   GtkWidget       *cut,    /* drawable to show pixmap         */
                   *pos,    /* Label to show current position  */
                   *size;   /* Label to show current size      */
   GdkPixmap       *pm;     /* Backing pixmap for drawing area */
   GdkGC           *XorGc;  /* GC for drawing XOR-lines        */
   BOX             *rubber,
                   *old;
   ITEM            *para;
   GList           *pl;
   CURSOR_LOCATION  where;
   int              xi, yi; /* Cursor coordinates when moving whole box */
} CUT;

static CUT *Cut = NULL;

static void FreeWidget(GtkWidget *widget)
/* ensure that widget is NULL after destroying 
 */
{
   if ( widget ) {
      gtk_widget_destroy(widget);
      widget = NULL;
   }
}


static void FreeCut(CUT *cut)
/* free all necessary parts of CUT structure
 */
{
   if ( !Cut ) return;
   if ( Cut->pm ) gdk_imlib_free_pixmap(Cut->pm);
   Cut->pm = NULL;
   gdk_gc_destroy(Cut->XorGc);
   Cut->XorGc = NULL;
   FreeWidget(Cut->cut);
   FreeWidget(Cut->pos);
   FreeWidget(Cut->size);
   FREE(Cut->rubber);
   FREE(Cut->old);
   FREE(Cut->para);
   Cut = NULL;
}

static void DrawRubber(CUT *cut)
/* Draw the rubber box */
{
   char buf[64];
   
   if ( !cut || !cut->rubber ) return;
   
   if ( !cut->old ) assert ( (cut->old = malloc(sizeof(BOX))) );
   else {
      if ( cut->old->x1 == cut->rubber->x1 &&
           cut->old->y1 == cut->rubber->y1 &&
           cut->old->x2 == cut->rubber->x2 &&
           cut->old->y2 == cut->rubber->y2 ) return;
      gdk_draw_rectangle(cut->cut->window, cut->XorGc, FALSE, cut->old->x1, cut->old->y1, 
                         cut->old->x2 - cut->old->x1, cut->old->y2 - cut->old->y1);
      gdk_draw_rectangle(cut->pm, cut->XorGc, FALSE, cut->old->x1, cut->old->y1, 
                         cut->old->x2 - cut->old->x1, cut->old->y2 - cut->old->y1);
   }

   cut->old->x1 = cut->rubber->x1;
   cut->old->y1 = cut->rubber->y1;
   cut->old->x2 = cut->rubber->x2;
   cut->old->y2 = cut->rubber->y2;
   gdk_draw_rectangle(cut->cut->window, cut->XorGc, FALSE, cut->old->x1, cut->old->y1, 
                      cut->old->x2 - cut->old->x1, cut->old->y2 - cut->old->y1);
   gdk_draw_rectangle(cut->pm, cut->XorGc, FALSE, cut->old->x1, cut->old->y1, 
                      cut->old->x2 - cut->old->x1, cut->old->y2 - cut->old->y1);

   sprintf(buf, "%4ix%4i", cut->rubber->x2 - cut->rubber->x1 + 1, cut->rubber->y2 - cut->rubber->y1 + 1);
   gtk_label_set(GTK_LABEL(Cut->size), buf);
   gtk_widget_show(Cut->size);
}

static void SetRubberCorner(int x, int *x1, int *x2, GtkAdjustment *adj1, GtkAdjustment *adj2)
/* set the nearest corner to click (behaviour for right mouse button)
 */
{
   if ( x <= *x1 ) { /* left or top */
      gtk_adjustment_set_value(adj1, (gfloat)(*x1 = x)); 
      return;
   }
   if ( x >= *x2 ) { /* right or bottom */
      gtk_adjustment_set_value(adj2, (gfloat)(*x2 = x)); 
      return;
   }
   if ( x - *x1 < *x2 - x ) { /* inner, but near to left or top */
      gtk_adjustment_set_value(adj1, (gfloat)(*x1 = x)); 
      return;
   }
   gtk_adjustment_set_value(adj2, (gfloat)(*x2 = x)); /* inner, but near to right or bottom */
}

static void SetCursor(GdkWindow *window, int type)
{
   GdkCursor *cursor;

   cursor = gdk_cursor_new(type);
   gdk_window_set_cursor(window, cursor);
   gdk_cursor_destroy(cursor);
}

static void CutParameterChangedOK(GtkButton *button, PAUL *p)
/* called when parameters were changed and OK button was pressed
 * --- Parameter: ---
 * GtkButton *button: OK button widget (can be ignored)
 * PAUL      *p     : 
 */
{
   char buf[256];
   
   if ( ( Cut->rubber->x1 == 0                    && Cut->rubber->y1 == 0 &&
          Cut->rubber->x2 == BILD(Cut->pl)->W - 1 && Cut->rubber->y2 == BILD(Cut->pl)->H - 1 ) ||
        Cut->rubber->x1 == Cut->rubber->x2        || Cut->rubber->y1 == Cut->rubber->y2 )
      return;

   if ( !(p->opt->cut) ) assert ( ( p->opt->cut = malloc(sizeof(BOX)) ) );
   p->opt->cut->x1 = Cut->rubber->x1;
   p->opt->cut->y1 = Cut->rubber->y1;
   p->opt->cut->x2 = Cut->rubber->x2 + 1;
   p->opt->cut->y2 = Cut->rubber->y2 + 1;
   
   sprintf(buf, "Section from %s", BILD(Cut->pl)->file);
   Cut->pl->data = CutArea(BILD(Cut->pl), p->opt->cut, p->opt->f, buf, APPCUT, 1);
   FREE(p->opt->cut);
   RefreshImage(BILD(Cut->pl));
   ApplyPicture(p->show->window, BILD(p->activ = Cut->pl));
   if ( Info(p->opt->f) ) FileInfo(NULL, p);
}

static int SetParameterIfValid(int *val2set, int val, int cond, GtkAdjustment *adj)
/* Set value in address if condition is true
 */
{
   if ( cond ) {
      *val2set = val;
      return TRUE;
   }
   if ( *val2set == val ) return FALSE;  /* it's not necessary to set the value */
   gdk_beep();
   gtk_adjustment_set_value(adj, *val2set);
   return FALSE;
}


void CutParameterEdited(GtkAdjustment *adj, GtkWidget *spinner)
/* called while editing box parameters
 * --- Parameter: ---
 * GtkButton *button: OK button widget (can be ignored)
 * PAUL      *p     : 
 */
{
   int  *chg = gtk_object_get_user_data(GTK_OBJECT(adj)),
         neu = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spinner));

   if        ( chg == &(Cut->rubber->x1) ) {
      if ( !SetParameterIfValid(&(Cut->rubber->x1), neu, neu < Cut->rubber->x2, adj) ) return;
   } else if ( chg == &(Cut->rubber->y1) ) {
      if ( !SetParameterIfValid(&(Cut->rubber->y1), neu, neu < Cut->rubber->y2, adj) ) return;
   } else if ( chg == &(Cut->rubber->x2) ) {
      if ( !SetParameterIfValid(&(Cut->rubber->x2), neu, neu > Cut->rubber->x1, adj) ) return;
   } else if ( chg == &(Cut->rubber->y2) ) {
      if ( !SetParameterIfValid(&(Cut->rubber->y2), neu, neu > Cut->rubber->y1, adj) ) return;
   } else return;
   
   DrawRubber(Cut);
}

static CURSOR_LOCATION WhereIsIt(BOX *rubber, int x, int y) 
/* obtain cursor position relative to the selected rubber box
 */
{
#define FRAME  3

   if ( !rubber ) return OUT;
   if ( rubber->x1 + FRAME <= x && x <= rubber->x2 - FRAME &&
        rubber->y1 + FRAME <= y && y <= rubber->y2 - FRAME ) return INNER;
   if ( rubber->x1 - FRAME <= x && x <= rubber->x1 + FRAME ) {
      if ( rubber->y1 - FRAME <= y && y <= rubber->y1 + FRAME ) return LEFTTOP;
      if ( rubber->y2 - FRAME <= y && y <= rubber->y2 + FRAME ) return LEFTBOTTOM;
      if ( rubber->y1 + FRAME <  y && y <  rubber->y2 + FRAME ) return LEFT;
      return OUT;
   }
   if ( rubber->x2 - FRAME <= x && x <= rubber->x2 + FRAME ) {
      if ( rubber->y1 - FRAME <= y && y <= rubber->y1 + FRAME ) return RIGHTTOP;
      if ( rubber->y2 - FRAME <= y && y <= rubber->y2 + FRAME ) return RIGHTBOTTOM;
      if ( rubber->y1 + FRAME <  y && y <  rubber->y2 + FRAME ) return RIGHT;
      return OUT;
   }
   if ( rubber->y1 - FRAME <= y && y <= rubber->y1 + FRAME &&
        rubber->x1 + FRAME <  x && x <  rubber->x2 + FRAME ) return TOP;
   if ( rubber->y2 - FRAME <= y && y <= rubber->y2 + FRAME &&
        rubber->x1 + FRAME <  x && x <  rubber->x2 + FRAME ) return BOTTOM;
   if ( rubber->x1 + 2 > x || x > rubber->x2 - 2 ||
        rubber->y1 + 2 > y || y > rubber->y2 - 2 ) return OUT;
   return NOWHERE;
}

static void SetCursorAccording2Position(GtkWidget *cut, int pos)
{
   switch ( pos ) {
      case INNER:       SetCursor(cut->window, GDK_FLEUR);
	                break;
      case OUT:         SetCursor(cut->window, GDK_CROSSHAIR);
	                break;
      case TOP:         SetCursor(cut->window, GDK_SB_V_DOUBLE_ARROW);
	                break;
      case BOTTOM:      SetCursor(cut->window, GDK_SB_V_DOUBLE_ARROW);
	                break;
      case LEFT:        SetCursor(cut->window, GDK_SB_H_DOUBLE_ARROW);
	                break;
      case RIGHT:       SetCursor(cut->window, GDK_SB_H_DOUBLE_ARROW);
	                break;
      case LEFTTOP:     SetCursor(cut->window, GDK_TOP_LEFT_CORNER);
	                break;
      case RIGHTBOTTOM: SetCursor(cut->window, GDK_BOTTOM_RIGHT_CORNER);
                        break;
      case RIGHTTOP:    SetCursor(cut->window, GDK_TOP_RIGHT_CORNER);
                        break;
      case LEFTBOTTOM:  SetCursor(cut->window, GDK_BOTTOM_LEFT_CORNER);
                        break;
      case NOWHERE:     SetCursor(cut->window, GDK_DOT);
   }
}

static void HandleRubber(int x, int y, GdkModifierType state)
{
   int          i, w, h;
   register int xi, yi;
   
   /************************************************************************************
    * Set correct adjustment values                                                    *
    ************************************************************************************/
   switch ( i = Cut->where ) {
      case INNER:       if ( !(state & GDK_BUTTON1_MASK) ) break;
                        if ( (xi = x - Cut->xi) < 0 ) xi = 0;
                        if ( xi + (w = Cut->rubber->x2 - Cut->rubber->x1) > BILD(Cut->pl)->W - 1 ) 
                           xi = BILD(Cut->pl)->W - 1 - w;
                        gtk_adjustment_set_value(Cut->para->adj, xi);
                        if ( (yi = y - Cut->yi) < 0 ) yi = 0;
                        if ( yi + (h = Cut->rubber->y2 - Cut->rubber->y1) > BILD(Cut->pl)->H - 1 ) 
                           yi = BILD(Cut->pl)->H - 1 - h;
                        gtk_adjustment_set_value(((Cut->para)+1)->adj, yi);
                        gtk_adjustment_set_value(((Cut->para)+2)->adj, xi + w);
                        gtk_adjustment_set_value(((Cut->para)+3)->adj, yi + h);
	                break;
      case OUT:         break;
      case TOP:         if ( !(state & GDK_BUTTON1_MASK) || y < 0 ) break;
                        gtk_adjustment_set_value(((Cut->para)+1)->adj, y);
	                break;
      case BOTTOM:      if ( !(state & GDK_BUTTON1_MASK) || y >= BILD(Cut->pl)->H ) break;
                        gtk_adjustment_set_value(((Cut->para)+3)->adj, y);
	                break;
      case LEFT:        if ( !(state & GDK_BUTTON1_MASK) || x < 0 ) break;
                        gtk_adjustment_set_value(Cut->para->adj, x);
	                break;
      case RIGHT:       if ( !(state & GDK_BUTTON1_MASK) || x >= BILD(Cut->pl)->W ) break;
                        gtk_adjustment_set_value(((Cut->para)+2)->adj, x);
	                break;
      case LEFTTOP:     i = CORNER; /* cause it is the same behaviour */
	                break;
      case RIGHTBOTTOM: i = CORNER; /* cause it is the same behaviour */
                        break;
      case RIGHTTOP:    i = CORNER; /* cause it is the same behaviour */
                        break;
      case LEFTBOTTOM:  i = CORNER; /* cause it is the same behaviour */
                        break;
      case NOWHERE:     break;
   }

   if ( ((state & GDK_BUTTON1_MASK) && (i == OUT || i == CORNER)) || (state & GDK_BUTTON3_MASK) ) {
      SetRubberCorner((gfloat)x, &(Cut->rubber->x1), &(Cut->rubber->x2), Cut->para->adj, 
                      ((Cut->para)+2)->adj);
      SetRubberCorner((gfloat)y, &(Cut->rubber->y1), &(Cut->rubber->y2), ((Cut->para)+1)->adj, 
                      ((Cut->para)+3)->adj);
   }

   if ( (state & (GDK_BUTTON1_MASK | GDK_BUTTON3_MASK)) ) 
      DrawRubber(Cut);
}

static void CutMotionEvent(GtkWidget *cut, GdkEvent *event, PAUL *p)
{
   char                buf[64];
   int                 x, y;
   GdkModifierType     state;

   if ( !event ) return;
   if ( !GTK_IS_WIDGET(cut) ) return;

   if ( event->type == GDK_ENTER_NOTIFY )
      gdk_window_get_pointer(cut->window, &x, &y, &state);
   else if ( event->type == GDK_MOTION_NOTIFY ) {
      /* if (event->motion.is_hint) gdk_window_get_pointer(cut->window, &x, &y, &state);
       * else { */
      x     = event->motion.x;
      y     = event->motion.y;
      state = event->motion.state;
      /*  } */
   } else return;  /* this shouldn't occure */

   sprintf(buf, "%4i, %4i", x, y);
   gtk_label_set(GTK_LABEL(Cut->pos), buf);
   gtk_widget_show(Cut->pos);

   if ( !state ) {
      SetCursorAccording2Position(cut, WhereIsIt(Cut->old, x, y));
      return ;
   }
   HandleRubber(x, y, state);
}

void CutLeaveEvent(GtkWidget *cut, GdkEvent *event, PAUL *p)
{
   int              x, y;
   GdkModifierType  state;
   PICTURE         *bild;

   gtk_label_set(GTK_LABEL(Cut->pos), "");
   if ( Cut->where == NOWHERE ) return;
   /* if rubber box is resized check for clean leaving of the window */
   bild = BILD(Cut->pl);
   gdk_window_get_pointer(cut->window, &x, &y, &state);
   if ( x < 0 ) x = 0;
   else if ( x >= bild->W ) x = bild->W - 1;
   if ( y < 0 ) y = 0;
   else if ( y >= bild->H ) y = bild->H - 1;
   HandleRubber(x, y, state);
}

static void CutImageEnd(GtkWidget *widget, void *data)
/* free the pixmap and destroy XorGc
 * --- Parameter: ---
 * ___ not used ___
 */
{
   FreeCut(Cut);
}

static gint CutConfigureEvent(GtkWidget *cut, GdkEventConfigure *event, PICTURE *bild)
/* Create a new backing pixmap of the appropriate size 
 * --- Parameter: ---
 * GtkWidget         *cut   : drawing area with image to cut and rubber box
 * GdkEventConfigure *event : configure_event
 * PICTURE           *p     : picture to cut (originally p->activ, but p->activ may
 *                            change while cutting the picture)
 */
{
   if ( !IS_PICTURE(bild) ) return FALSE;
   if ( Cut->pm ) return TRUE;
   if ( !gdk_imlib_render(bild->im, bild->W, bild->H) ) return FALSE;
   Cut->pm = gdk_imlib_copy_image(bild->im);
   /* It was considered to be to hard to get the current pointer position right *
    * ... leave it out                                                          */
   return TRUE;
}

static void CutExposeEvent(GtkWidget *cut, GdkEventExpose *event)
/* Redraw the screen from the backing pixmap 
 * --- Parameter: ---
 * GtkWidget         *cut   : drawing area with image to cut and rubber box
 * GdkEventConfigure *event : expose_event
 */
{
   GdkColor   color;

   gdk_draw_pixmap(cut->window, cut->style->fg_gc[GTK_WIDGET_STATE(cut)], Cut->pm,
		  event->area.x, event->area.y,
		  event->area.x, event->area.y,
		  event->area.width, event->area.height);

   SetCursor(cut->window, GDK_CROSSHAIR);
   
   if ( Cut->XorGc ) gdk_gc_destroy(Cut->XorGc);
   Cut->XorGc= gdk_gc_new(cut->window);
   gdk_color_white(gdk_window_get_colormap(cut->window), &color);
   gdk_gc_set_foreground(Cut->XorGc, &color);
   gdk_gc_set_function(Cut->XorGc, GDK_XOR);
}

static void CutButtonPressEvent(GtkWidget *cut, GdkEventButton *event)
{
   register int x = (int)event->x, y = (int)event->y;
   
   Cut->where = WhereIsIt(Cut->old, x, y);
   if ( Cut->where == INNER && event->button == 1 ) {
      Cut->xi = x - Cut->old->x1;
      Cut->yi = y - Cut->old->y1;
      if ( Cut->xi < 0 || Cut->old->x2 - Cut->old->x1 < Cut->xi || 
           Cut->yi < 0 || Cut->old->y2 - Cut->old->y1 < Cut->yi ) {
	 g_warning("Invalid position while trying to move box");
         Cut->xi = Cut->yi = -1;
      }
   }
   
   /**************************************************************************************
    * delete rubberbox if mouse button 2 (middle) was pressed                            *
    **************************************************************************************/
   if ( event->button == 2 ) {
      register PICTURE *bild = BILD(Cut->pl);;
      
      if ( !Cut->old ) return;
      gtk_adjustment_set_value(Cut->para->adj, Cut->rubber->x1 = 0);
      gtk_adjustment_set_value(((Cut->para)+1)->adj, Cut->rubber->y1 = 0);
      gtk_adjustment_set_value(((Cut->para)+2)->adj, Cut->rubber->x2 = bild->W - 1);
      gtk_adjustment_set_value(((Cut->para)+3)->adj, Cut->rubber->y2 = bild->H - 1);
      FREE((Cut->old));

      /* Instead of repainting the whole pixmap it might be good to draw rectangle  *
       * only if it is not equal to the image borders.                              */
      if ( Cut->pm ) gdk_imlib_free_pixmap(Cut->pm);
      if ( !gdk_imlib_render(bild->im, bild->W, bild->H) ) return;
      Cut->pm = gdk_imlib_copy_image(bild->im);
      gdk_draw_pixmap(cut->window, cut->style->fg_gc[GTK_WIDGET_STATE(cut)], Cut->pm,
                      0, 0, 0, 0, bild->W - 1, bild->H - 1);
      return;
   }

   /**************************************************************************************
    * when initial state (that means also after pressing button 2) draw a new box with   *
    * the left mouse button                                                              *
    **************************************************************************************/
   if ( event->button == 1 && !(Cut->old) ) {
      Cut->rubber->x1 = x;
      gtk_adjustment_set_value(Cut->para->adj, x);
      Cut->rubber->y1 = y;
      gtk_adjustment_set_value(((Cut->para)+1)->adj, y);
      Cut->rubber->x2 = x;
      gtk_adjustment_set_value(((Cut->para)+2)->adj, x);
      Cut->rubber->y2 = y;
      gtk_adjustment_set_value(((Cut->para)+3)->adj, y);
   } else 
      if ( (event->button == 1 && Cut->where == OUT) ||
           (event->button == 3 ) ) {
         SetRubberCorner(x, &(Cut->rubber->x1), &(Cut->rubber->x2), 
                            Cut->para->adj,     ((Cut->para)+2)->adj);
         SetRubberCorner(y, &(Cut->rubber->y1), &(Cut->rubber->y2),
	                    ((Cut->para)+1)->adj, ((Cut->para)+3)->adj);
   }
   DrawRubber(Cut);
}

static void CutButtonReleaseEvent(GtkWidget *cut, GdkEventButton *event)
{
   SetCursorAccording2Position(cut, WhereIsIt(Cut->old, (int)event->x, (int)event->y));
   Cut->where = NOWHERE;
   Cut->xi    = Cut->yi = -1;
}

void ThisCutCallback(GtkWidget *widget, PAUL *p)
{
   static    GtkWidget *top;
   GtkWidget           *ruler, 
                       *table, /* 2x2 hold rulers in (2,1) and (1,2), pixmap in (2,2) */
                       *cut,   /* gtk_pixmap with image to cut                        */
                       *frame,
                       *pos,
                       *viewport,
                       *hbox, *vbox;
   char                 buf[256];
   PICTURE             *bild = BILD(p->activ);
   ITEM                *topleft, *botright;
   BOX                 *rubber;
   
   if ( top ) {
      gtk_widget_destroy(top);
      top = NULL;
      return;
   }

   top = gtk_dialog_new();
   gtk_signal_connect(GTK_OBJECT(top), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &top);
   gtk_signal_connect(GTK_OBJECT(top), "destroy", GTK_SIGNAL_FUNC(CutImageEnd), p);

   sprintf(buf, "Cut %s", bild->file);
   gtk_window_set_title(GTK_WINDOW(top), buf);

   /**********************************************************************************
    * construct the cut area (table with sliders & image window)                     *
    **********************************************************************************/

   table = gtk_table_new(2, 2, /* homogeneous */ FALSE);
   gtk_table_set_col_spacing(GTK_TABLE(table), 0, 1);
   gtk_table_set_row_spacing(GTK_TABLE(table), 0, 1);
   gtk_container_border_width(GTK_CONTAINER (table), 2);
   gtk_box_pack_start(GTK_BOX(GTK_DIALOG(top)->vbox), table, FALSE, FALSE, 0);
                                    /* dont allow resizing! ^^^^^  ^^^^^  */

   /* the empty box in the top-left corner */
   frame = gtk_frame_new(NULL);
   gtk_frame_set_shadow_type(GTK_FRAME(frame),GTK_SHADOW_OUT);
   gtk_table_attach(GTK_TABLE(table), frame, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);

   /**********************************************************************************
    * the image itself                                                               *
    **********************************************************************************/
   viewport = gtk_frame_new(NULL);
   gtk_frame_set_shadow_type(GTK_FRAME(viewport), GTK_SHADOW_IN);
   gtk_table_attach(GTK_TABLE(table), viewport, 1, 2, 1, 2, 0, 0, 0, 0);
   hbox = gtk_hbox_new(FALSE, 0);  /* this seems to be neccessary to awoid resizing of drawing area */
   gtk_container_add(GTK_CONTAINER(viewport), hbox);   

   cut    = gtk_drawing_area_new();
   gtk_drawing_area_size(GTK_DRAWING_AREA(cut), bild->W, bild->H);
   gtk_box_pack_start(GTK_BOX(hbox), cut, FALSE, FALSE, 0);

   /**********************************************************************************
    * ... with all it's signals and events                                           *
    **********************************************************************************/
   gtk_widget_set_events(cut, GDK_EXPOSURE_MASK | GDK_LEAVE_NOTIFY_MASK | GDK_BUTTON_PRESS_MASK 
			 | GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK
			 /*                         | GDK_POINTER_MOTION_HINT_MASK  */
                         | GDK_ENTER_NOTIFY_MASK );
  /* Signals used to handle backing pixmap */
   gtk_signal_connect(GTK_OBJECT(cut), "expose_event", GTK_SIGNAL_FUNC(CutExposeEvent), NULL);
   gtk_signal_connect(GTK_OBJECT(cut), "configure_event", GTK_SIGNAL_FUNC(CutConfigureEvent), 
                      BILD(p->activ));

  /* Event signals */
   gtk_signal_connect(GTK_OBJECT(cut), "motion_notify_event", GTK_SIGNAL_FUNC(CutMotionEvent), p);
   gtk_signal_connect(GTK_OBJECT(cut), "enter_notify_event", GTK_SIGNAL_FUNC(CutMotionEvent), p);
   gtk_signal_connect(GTK_OBJECT(cut), "leave_notify_event", GTK_SIGNAL_FUNC(CutLeaveEvent), p);
   gtk_signal_connect(GTK_OBJECT(cut), "button_press_event", GTK_SIGNAL_FUNC(CutButtonPressEvent), p);
   gtk_signal_connect(GTK_OBJECT(cut), "button_release_event", GTK_SIGNAL_FUNC(CutButtonReleaseEvent), p);

   if ( Cut ) FreeCut(Cut);
   assert ( (Cut = calloc(1, sizeof(CUT))) );
   Cut->cut    = cut;                         /* set global cut widget here */
   /**********************************************************************************
    * the rulers to know where you are                                               *
    **********************************************************************************/
   ruler = gtk_hruler_new();
   gtk_ruler_set_range(GTK_RULER(ruler), 0, bild->W - 1, 0, bild->W - 1);
   gtk_table_attach(GTK_TABLE(table), ruler, 1, 2, 0, 1, GTK_FILL, 0, 0, 0);
   gtk_signal_connect_object(GTK_OBJECT(cut), "motion_notify_event",
         GTK_SIGNAL_FUNC(GTK_WIDGET_CLASS(GTK_OBJECT(ruler)->klass)->motion_notify_event),
         GTK_OBJECT(ruler));
 
   ruler = gtk_vruler_new();
   gtk_ruler_set_range(GTK_RULER(ruler), 0, bild->H - 1, 0, bild->H - 1);
   gtk_table_attach(GTK_TABLE(table), ruler, 0, 1, 1, 2, 0, GTK_FILL, 0, 0);

   gtk_signal_connect_object(GTK_OBJECT(cut), "motion_notify_event",
         GTK_SIGNAL_FUNC(GTK_WIDGET_CLASS(GTK_OBJECT(ruler)->klass)->motion_notify_event),
         GTK_OBJECT (ruler));

   /**********************************************************************************
    * action area                                                                    *
    **********************************************************************************/

   vbox = gtk_vbox_new(FALSE, 3);
   gtk_container_border_width(GTK_CONTAINER(GTK_DIALOG(top)->action_area), 3);
   gtk_container_add(GTK_CONTAINER(GTK_BOX(GTK_DIALOG(top)->action_area)), vbox);

   /**********************************************************************************
    * Cursor positions in the dialog action area                                     *
    **********************************************************************************/

   hbox = gtk_hbox_new(FALSE, 1);
   gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

   /**********************************************************************************
    * current cursor position                                                        *
    **********************************************************************************/
   table = gtk_table_new(2, 2, /* homogeneous */ FALSE);
   gtk_container_border_width(GTK_CONTAINER(table), 5);
   gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, FALSE, 0);
   pos = gtk_label_new("current:");
   gtk_table_attach(GTK_TABLE(table), pos, 0, 1, 0, 1, 0, 0, 0, 0);

   frame = gtk_frame_new(NULL);
   gtk_widget_set_usize(frame, 60, 0);
   gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
   gtk_table_attach(GTK_TABLE(table), frame, 1, 2, 0, 1, 0, 0, 0, 0);

   Cut->pos = gtk_label_new("");
   gtk_container_add(GTK_CONTAINER(frame), Cut->pos);

   /**********************************************************************************
    * current width and height                                                       *
    **********************************************************************************/

   pos = gtk_label_new("size:");
   gtk_table_attach(GTK_TABLE(table), pos, 0, 1, 1, 2, 0, 0, 0, 0);

   frame = gtk_frame_new(NULL);
   gtk_widget_set_usize(frame, 60, 0);
   gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
   gtk_table_attach(GTK_TABLE(table), frame, 1, 2, 1, 2, 0, 0, 0, 0);

   sprintf(buf, "%4ix%4i", bild->W, bild->H);
   Cut->size = gtk_label_new(buf);
   gtk_container_add(GTK_CONTAINER(frame), Cut->size);

   /**********************************************************************************
    * left top box position                                                          *
    **********************************************************************************/
   
   assert ( (topleft = malloc(4*sizeof(ITEM))) );
   assert ( (rubber  = malloc(sizeof(BOX))) );
   if ( p->opt->cut ) {
      rubber->x1 = p->opt->cut->x1;
      rubber->y1 = p->opt->cut->y1;
      rubber->x2 = p->opt->cut->x2;
      rubber->y2 = p->opt->cut->y2;
   } else {
      rubber->x1 = 0;
      rubber->y1 = 0;
      rubber->x2 = bild->W - 1;
      rubber->y2 = bild->H - 1;
   }
   
   topleft->name       = "left:";
   topleft->typ        = P_INT;
   topleft->val        = &(rubber->x1);
   topleft->upper      = bild->W - 1;
   topleft->lower      = 0;
   topleft->valchg     = GTK_SIGNAL_FUNC(CutParameterEdited);
   topleft->adj        = NULL;
   (topleft+1)->name   = "top:";
   (topleft+1)->typ    = P_INT;
   (topleft+1)->val    = &(rubber->y1);
   (topleft+1)->upper  = bild->H - 1;
   (topleft+1)->lower  = 0;
   (topleft+1)->valchg = GTK_SIGNAL_FUNC(CutParameterEdited);
   (topleft+1)->adj    = NULL;
   table = BuildParameterTable(hbox, topleft, 2);

   /**********************************************************************************
    * right bottom box position                                                      *
    **********************************************************************************/
   
   botright = topleft + 2;
   
   botright->name       = "right:";
   botright->typ        = P_INT;
   botright->val        = &(rubber->x2);
   botright->upper      = bild->W - 1;
   botright->lower      = 0;
   botright->valchg     = GTK_SIGNAL_FUNC(CutParameterEdited);
   botright->adj        = NULL;
   (botright+1)->name   = "bottom:";
   (botright+1)->typ    = P_INT;
   (botright+1)->val    = &(rubber->y2);
   (botright+1)->upper  = bild->H - 1;
   (botright+1)->lower  = 0;
   (botright+1)->valchg = GTK_SIGNAL_FUNC(CutParameterEdited);
   (botright+1)->adj    = NULL;
   table = BuildParameterTable(hbox, botright, 2);

   /**********************************************************************************
    * OK / Cancel button                                                             *
    **********************************************************************************/

   hbox = gtk_hbox_new(FALSE, 1);
   gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
   CreateOkButton(top, GTK_WIDGET(hbox), NULL, GTK_SIGNAL_FUNC(CutParameterChangedOK), p, 
                  NULL, NULL);
   CreateCancelButton(top, GTK_WIDGET(hbox));

   /**********************************************************************************
    * Initialize global CUT structure                                                *
    **********************************************************************************/

   Cut->rubber = rubber;
   Cut->para   = topleft;
   Cut->pl     = p->activ;
   Cut->xi     = Cut->yi = -1;

   gtk_widget_show_all(top);
}



