// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  IVector2.h

  Stephane Rehel, April 23rd, 1996
  from pntvect.h, from mathcls.h, etc, ..., from Aug 1992.
*/

#ifndef __IVector2_h
#define __IVector2_h

#ifndef __Common_h
#include "Common.h"
#endif

#define SCL int

class IPoint2;

/////////////////////////////////////////////////////////////////////////////
// IVector2 class: vector in 2 dimensions
//
class IVector2
{
  friend class IPoint2;
protected:
  union
    {
    struct
      {
      SCL x, y;
      } s;
    SCL vect[2];
    };

public:
  IVector2()
    {}
  IVector2( const SCL _x, const SCL _y )
    {
    s.x= _x;
    s.y= _y;
    }
  IVector2( const IVector2& v )
    {
    s.x= v.s.x;
    s.y= v.s.y;
    }

  const SCL x() const { return s.x; }
  const SCL y() const { return s.y; }
  const SCL geti( int i ) const
    {
    return vect[i];
    }

  SCL* getCoord() { return vect; }
  const SCL* getConstCoord() const { return vect; }

  SCL& operator [] ( int i )
    {
    return vect[i];
    }

  void setx( const SCL _x ) { s.x= _x; }
  void sety( const SCL _y ) { s.y= _y; }

  IVector2& operator= ( const IVector2& v )
    {
    s.x= v.s.x;
    s.y= v.s.y;
    return *this;
    }
  void zero()
    {
    s.x= s.y= SCL(0);
    }

  boolean operator== ( const IVector2& v ) const
    {
    return s.x == v.s.x &&
           s.y == v.s.y;
    }
  boolean operator!= ( const IVector2& v ) const
    {
    return ! operator== ( v );
    }
  IVector2 operator+ ( const IVector2& v ) const
    {
    return IVector2( s.x+v.s.x, s.y+v.s.y );
    }
  IVector2& operator+= ( const IVector2& v )
    {
    s.x+= v.s.x;
    s.y+= v.s.y;
    return *this;
    }
  IVector2 operator- () const
    {
    return IVector2( -s.x, -s.y );
    }
  IVector2 operator- ( const IVector2& v ) const
    {
    return IVector2( s.x-v.s.x, s.y-v.s.y );
    }
  IVector2& operator-= ( const IVector2& v )
    {
    s.x-= v.s.x;
    s.y-= v.s.y;
    return *this;
    }
  IVector2 operator* ( const SCL d ) const
    {
    return IVector2( s.x*d, s.y*d );
    }
  IVector2& operator*= ( const SCL d )
    {
    s.x *= d;
    s.y *= d;
    return *this;
    }
  IVector2 operator/ ( const SCL d ) const
    {
    return IVector2( s.x/d, s.y/d );
    }
  IVector2& operator/= ( const SCL d )
    {
    s.x /= d;
    s.y /= d;
    return *this;
    }
  SCL operator| ( const IVector2& v ) const
    {
    return s.x*v.s.x + s.y*v.s.y;
    }
  SCL operator ^ ( const IVector2& v ) const
    {
    return s.x*v.s.y - v.s.x*s.y;
    }

  boolean isNull() const
    {
    return s.x==SCL(0) && s.y==SCL(0);
    }

  // colinear
  boolean operator / ( const IVector2& v ) const
    {
    return (*this^v) == SCL(0);
    }

  // colinear and same orient.
  boolean operator % ( const IVector2& v ) const
    {
    return (*this/v) && sameOrient(v);
    }

  // *this and v supposed to be colinear...
  boolean sameOrient( const IVector2& v ) const
    {
    return (s.x==SCL(0)) ? sameSign(s.y,v.s.y)
                         : sameSign(s.x,v.s.x);
    }
  SCL absNorm() const
    {
    return ((s.x<0) ? (-s.x) : s.x)
         + ((s.y<0) ? (-s.y) : s.y);
    }

  SCL min() const
    {
    return ::min( s.x, s.y );
    }
  SCL max() const
    {
    return ::max( s.x, s.y );
    }
};

/////////////////////////////////////////////////////////////////////////////

inline IVector2 operator*( const SCL d, const IVector2& v )
{
  return IVector2( v.x()*d, v.y()*d );
}


/////////////////////////////////////////////////////////////////////////////
// IPoint2 class: Integer Point in 2 dimensions
//
class IPoint2
{
protected:
  union
    {
    struct
      {
      SCL x, y;
      } s;
    SCL vect[2];
    };

public:
  IPoint2()
    {}
  IPoint2( const SCL _x, const SCL _y )
    {
    s.x= _x;
    s.y= _y;
    }

  inline operator IVector2 () const
    {
    return IVector2( x(), y() );
    }

  const SCL x() const { return s.x; }
  const SCL y() const { return s.y; }
  const SCL geti( int i ) const
    {
    return vect[i];
    }

  SCL* getCoord() { return vect; }
  const SCL* getConstCoord() const { return vect; }

  SCL& operator [] ( int i )
    {
    return vect[i];
    }

  void setx( const SCL _x ) { s.x= _x; }
  void sety( const SCL _y ) { s.y= _y; }

  IPoint2& operator= ( const IPoint2& p )
    {
    s.x= p.s.x;
    s.y= p.s.y;
    return *this;
    }
  void zero()
    {
    s.x= s.y= SCL(0);
    }

  boolean operator== ( const IPoint2& p ) const
    {
    return s.x == p.s.x &&
           s.y == p.s.y;
    }
  boolean operator!= ( const IPoint2& p ) const
    {
    return ! operator== ( p );
    }
  IPoint2 operator+ ( const IPoint2& p ) const
    {
    return IPoint2( s.x+p.s.x, s.y+p.s.y );
    }
  IPoint2& operator+=( const IPoint2& p )
    {
    s.x+= p.s.x;
    s.y+= p.s.y;
    return *this;
    }
  IVector2 operator- ( const IPoint2& p ) const
    {
    return IVector2( s.x-p.s.x, s.y-p.s.y );
    }
  IPoint2 operator* ( const SCL k ) const
    {
    return IPoint2( s.x*k, s.y*k );
    }
  IPoint2 operator/ ( const SCL k ) const
    {
    return IPoint2( s.x/k, s.y/k );
    }

  IPoint2 operator + ( const IVector2& v ) const
    {
    return IPoint2( s.x+v.s.x, s.y+v.s.y );
    }
  IPoint2& operator+= ( const IVector2& v )
    {
    s.x+= v.s.x;
    s.y+= v.s.y;
    return *this;
    }
  IPoint2 operator - ( const IVector2& v ) const
    {
    return IPoint2( s.x-v.s.x, s.y-v.s.y );
    }
  IPoint2& operator-= ( const IVector2& v )
    {
    s.x-= v.s.x;
    s.y-= v.s.y;
    return *this;
    }
  IPoint2& operator*= ( const SCL d )
    {
    s.x *= d;
    s.y *= d;
    return *this;
    }
  IPoint2& operator/= ( const SCL d )
    {
    s.x /= d;
    s.y /= d;
    return *this;
    }

  IPoint2& addx( const SCL dx )
    {
    s.x+= dx;
    return *this;
    }
  IPoint2& addy( const SCL dy )
    {
    s.y+= dy;
    return *this;
    }

  SCL min() const
    {
    return ::min( s.x, s.y );
    }
  SCL max() const
    {
    return ::max( s.x, s.y );
    }
  IPoint2 min( const IPoint2& p ) const
    {
    return IPoint2( ::min( s.x, p.s.x ),
                    ::min( s.y, p.s.y ) );
    }
  IPoint2 max( const IPoint2& p ) const
    {
    return IPoint2( ::max( s.x, p.s.x ),
                    ::max( s.y, p.s.y ) );
    }
};

/////////////////////////////////////////////////////////////////////////////

IVector2 min( const IVector2& v1, const IVector2& v2 );
IVector2 max( const IVector2& v1, const IVector2& v2 );
IPoint2 min( const IPoint2& v1, const IPoint2& v2 );
IPoint2 max( const IPoint2& v1, const IPoint2& v2 );

/////////////////////////////////////////////////////////////////////////////

#undef SCL

#endif // ifndef __IVector2_h

