// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  MLImageFormat.C

  Stephane Rehel

  September 20 1997
*/

#include "tools/MLFilename.h"

#include "MLImageFormat.h"

#include "MLImageReader.h"
#include "MLImageWriter.h"

/////////////////////////////////////////////////////////////////////////////

// static
int MLImageFormat::getNFormats()
{
  return int(LAST_FMT) - int(NULL_FMT) - 1;
}

/////////////////////////////////////////////////////////////////////////////

// static
// 1 <= i <= nFormats
MLImageFormat::FORMAT MLImageFormat::getFromIndex( int i )
{
  if( i < int(NULL_FMT) || i >= int(LAST_FMT) )
    return NULL_FMT;

  return FORMAT(i);
}

/////////////////////////////////////////////////////////////////////////////

// static
MLString MLImageFormat::getName( MLImageFormat::FORMAT fmt )
{
  switch(fmt)
    {
    case PNM: return "PNM";
    case GIF: return "GIF";
    case TGA: return "TARGA";
    case RGB: return "IRIS RGB";
    case BMP: return "Windows BMP";
    case JPG: return "JPEG";
    case TIF: return "TIFF";
    default:
      break;
    }

  return "";
}

/////////////////////////////////////////////////////////////////////////////

// static
MLImageFormat::FORMAT MLImageFormat::getFromName( const MLString& name )
{
  if( name.length() == 0 )
    return NULL_FMT;

  for( int i= 1; i <= getNFormats(); ++i )
    {
    MLImageFormat::FORMAT fmt= getFromIndex(i);
    if( getName(fmt) == name )
      return fmt;
    }

  return NULL_FMT;
}

/////////////////////////////////////////////////////////////////////////////

// static
MLString MLImageFormat::getExtension( MLImageFormat::FORMAT fmt )
{
  switch(fmt)
    {
    case PNM: return ".pnm";
    case GIF: return ".gif";
    case TGA: return ".tga";
    case RGB: return ".rgb";
    case BMP: return ".bmp";
    case JPG: return ".jpg";
    case TIF: return ".tif";
    default:
      break;
    }

  return "";
}

/////////////////////////////////////////////////////////////////////////////

// static
MLImageFormat::FORMAT MLImageFormat::getFromFilename(
                                                   const MLString& filename )
{
  MLString ext= MLFilename(filename).getExtension().toLowerCase();

  if( ext == ".pbm" || ext == ".pgm" || ext == ".pnm" || ext == ".ppm" )
    return PNM;

  if( ext == ".gif" )
    return GIF;

  if( ext == ".tga" )
    return TGA;

  if( ext == ".rgb" )
    return RGB;

  if( ext == ".bmp" )
    return BMP;;

  if( ext == ".jpg" )
    return JPG;

  if( ext == ".tif" || ext == ".tiff" )
    return TIF;

  return NULL_FMT;
}

/////////////////////////////////////////////////////////////////////////////

// static
boolean MLImageFormat::availableReader( FORMAT fmt )
{
  MLImageReader* r= MLImageReader::getReader(fmt);

  if( r == 0 )
    return false;

  boolean ok= r->available();

  delete r;

  return ok;
}

/////////////////////////////////////////////////////////////////////////////

// static
boolean MLImageFormat::availableWriter( FORMAT fmt )
{
  MLImageWriter* w= MLImageWriter::getWriter(fmt);

  if( w == 0 )
    return false;

  boolean ok= w->available();

  delete w;

  return ok;
}

/////////////////////////////////////////////////////////////////////////////
