/* UI for Load/Save, etc.
 *
 * gsumi version 0.5
 *
 * Copyright 1997 Owen Taylor <owt1@cornell.edu>
*/

#include <stdio.h>
#include <stdlib.h>
#include "gsumi.h"

/* some forward declarations */
static void save_as_dialog();

/* a pointer to a function that gets called after a "save changes"
   query */
typedef void (*SaveChangesAction) (void);

typedef struct FileOpData_ FileOpData;

struct FileOpData_ {
  GtkWidget *file_select;
  SaveChangesAction after_action;
  FileType file_type;
};


static int
type_to_compress(FileType type, char *name)
{
  if (type == FILE_PBM)
    return 0;
  else if (type == FILE_PBM_GZ)
    return 1;
  else if (type == FILE_BY_EXTENSION)
    {
      int len = strlen(name);
      if (len > 2 && !strcmp(&name[len-3],".gz"))
	return 1;
    }
  return 0;
}

static void
file_type_callback(GtkWidget *w, FileType *store)
{
  *store = (FileType)gtk_object_get_data(GTK_OBJECT(w),"file_type");
}

static GtkWidget *
file_type_menu(FileType *store)
{
  GtkWidget *menu;
  GtkWidget *menu_item;

  menu = gtk_menu_new();

  menu_item = gtk_menu_item_new_with_label("By extension");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
		      (GtkSignalFunc)file_type_callback, store);
  gtk_object_set_data(GTK_OBJECT(menu_item),"file_type",
		      (gpointer)FILE_BY_EXTENSION);
  gtk_menu_append(GTK_MENU(menu),menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_menu_item_new_with_label("pbm.gz");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
		      (GtkSignalFunc)file_type_callback, store);
  gtk_object_set_data(GTK_OBJECT(menu_item),"file_type",
		      (gpointer)FILE_PBM_GZ);
  gtk_menu_append(GTK_MENU(menu),menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_menu_item_new_with_label("pbm");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
		      (GtkSignalFunc)file_type_callback, store);
  gtk_object_set_data(GTK_OBJECT(menu_item),"file_type",
		      (gpointer)FILE_PBM);
  gtk_menu_append(GTK_MENU(menu),menu_item);
  gtk_widget_show (menu_item);

  return menu;
}



static gchar *
insert_filename(const char *before, const char *after)
{
  char *new_str;
  char *p;

  if (info.name) {
    p = info.name + strlen(info.name);

    while (p>info.name && (*(p-1) != '/')) p--;
    new_str = g_new(char,strlen(before)+strlen(p)+strlen(after)+1);
    strcpy(new_str,before);
    strcat(new_str,p);
    strcat(new_str,after);
  } else {
    new_str = g_new(char,strlen(before)+strlen("(Untitled)")+strlen(after)+1);
    strcpy(new_str,before);
    strcat(new_str,"(Untitled)");
    strcat(new_str,after);
  }

  return new_str;
}

void
set_filename ()
{
  char *title;

  title = insert_filename("gsumi - ","");
  gdk_window_set_title (info.mainwin->window, title);
  g_free(title);
}
/*
 * Save changes dialog
 *
 */

typedef struct SaveChangesDialog_ SaveChangesDialog;

struct SaveChangesDialog_ {
  GtkWidget *shell;
  SaveChangesAction action;
};

static void
save_changes_yes_callback(GtkWidget *widget,
			  SaveChangesDialog *dialog)
{
  SaveChangesAction action = dialog->action;

  gtk_widget_destroy(dialog->shell);
  g_free(dialog);

  if (info.name) {
    save_bitmap(info.name,type_to_compress(info.file_type,info.name));
    info.is_changed = 0;
    (action)();
  } else {
    save_as_dialog(action);
  }
  
}

static void
save_changes_no_callback(GtkWidget *widget,
			 SaveChangesDialog *dialog)
{
  SaveChangesAction action = dialog->action;

  gtk_widget_destroy(dialog->shell);
  g_free(dialog);

  (action)();
}
			 
static void
save_changes_cancel_callback(GtkWidget *widget,
			     SaveChangesDialog *dialog)
{
  gtk_widget_destroy(dialog->shell);
  g_free(dialog);
}
			 
static void
save_changes_dialog(SaveChangesAction action)
{
  char *text;
  GtkWidget *label;
  GtkWidget *button;

  SaveChangesDialog *dialog = g_new(SaveChangesDialog, 1);
  dialog->action = action;

  dialog->shell = gtk_dialog_new();
  gtk_window_set_title (GTK_WINDOW (dialog->shell), "Save Changes?");

  text = insert_filename("Save changes to ","?");

  label = gtk_label_new(text);
  gtk_misc_set_padding(GTK_MISC(label),10,10);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog->shell)->vbox), label, 
		      TRUE, TRUE, 0);
  gtk_widget_show(label);

  /* button row */

  button = gtk_button_new_with_label("Yes");
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog->shell)->action_area), 
		      button, TRUE, TRUE, 0);
  
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      (GtkSignalFunc) save_changes_yes_callback,
		      dialog);

  gtk_widget_show (button);
  gtk_widget_grab_default (button);

  button = gtk_button_new_with_label("No");
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog->shell)->action_area), 
		      button, TRUE, TRUE, 0);
  
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      (GtkSignalFunc) save_changes_no_callback,
		      dialog);
  gtk_widget_show (button);

  button = gtk_button_new_with_label("Cancel");
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog->shell)->action_area), 
		      button, TRUE, TRUE, 0);
  
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      (GtkSignalFunc) save_changes_cancel_callback,
		      dialog);
  gtk_widget_show (button);

  gtk_widget_show(dialog->shell);
}

/*
 * New image dialog
 */

typedef struct NewImageDialog_ NewImageDialog;

struct NewImageDialog_ {
  GtkWidget *width_entry;
  GtkWidget *height_entry;
  GtkWidget *shell;
};

static void
new_image_ok_callback (GtkWidget *widget,
		       NewImageDialog *dialog)
{ 
  int width, height;
  rect old_rec,update_rec;

  char buffer[16];

  width = atoi(gtk_entry_get_text (GTK_ENTRY (dialog->width_entry)));
  height = atoi(gtk_entry_get_text (GTK_ENTRY (dialog->height_entry)));

  if (width <= 0)
    {
      sprintf(buffer,"%d",bitmap->width);
      gtk_entry_set_text(GTK_ENTRY(dialog->width_entry),buffer);
      return;
    }

  if (height <= 0)
    {
      sprintf(buffer,"%d",bitmap->height);
      gtk_entry_set_text(GTK_ENTRY(dialog->height_entry),buffer);
      return;
    }

  if (info.name) 
    {
      g_free(info.name);
      info.name = NULL;
      set_filename();
    }

  old_rec = bitmap->rec;
  bitmap_init(width,height);
  rect_union(&update_rec,&bitmap->rec,&old_rec);
  bitmap_update(&update_rec);

  undo_reset();
  init_scale();

  gtk_widget_destroy(dialog->shell);
  g_free(dialog);
}

static void
new_image_cancel_callback (GtkWidget *widget,
			   NewImageDialog *dialog)
{
  gtk_widget_destroy(dialog->shell);
  g_free(dialog);
}

static void
new_image_dialog ()
{
  char buffer[16];
  NewImageDialog *dialog;
  GtkWidget *label;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *button;

  dialog = g_new(NewImageDialog,1);

  dialog->shell = gtk_dialog_new();
  gtk_window_set_title (GTK_WINDOW (dialog->shell), "New Image");

  vbox = gtk_vbox_new (FALSE, 5);
  gtk_container_border_width(GTK_CONTAINER (vbox), 10);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog->shell)->vbox), vbox, 
		      TRUE, TRUE, 0);

  /* width entry */

  hbox = gtk_hbox_new(FALSE,2);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  
  label = gtk_label_new("Width:");
  gtk_box_pack_start (GTK_BOX (hbox),label,FALSE,FALSE,2);

  dialog->width_entry = gtk_entry_new();
  sprintf(buffer,"%d",bitmap->width);
  gtk_entry_set_text(GTK_ENTRY(dialog->width_entry),buffer);
  gtk_box_pack_start (GTK_BOX (hbox),dialog->width_entry,FALSE,FALSE,2);
		     
  gtk_widget_show(dialog->width_entry);
  
  gtk_widget_show(label);
  gtk_widget_show(hbox);

  /* height entry */
  hbox = gtk_hbox_new(FALSE,2);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  
  label = gtk_label_new("Height:");
  gtk_box_pack_start (GTK_BOX (hbox),label,FALSE,FALSE,2);

  dialog->height_entry = gtk_entry_new();
  sprintf(buffer,"%d",bitmap->height);
  gtk_entry_set_text(GTK_ENTRY(dialog->height_entry),buffer);
  gtk_box_pack_start (GTK_BOX (hbox),dialog->height_entry,FALSE,FALSE,2);
  gtk_widget_show(dialog->height_entry);
  
  gtk_widget_show(label);
  gtk_widget_show(hbox);
  
  /* buttons */

  button = gtk_button_new_with_label("OK");
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog->shell)->action_area), 
		      button, TRUE, TRUE, 0);
  
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      (GtkSignalFunc) new_image_ok_callback,
		      dialog);

  gtk_widget_show (button);
  gtk_widget_grab_default (button);

  button = gtk_button_new_with_label("Cancel");
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog->shell)->action_area), 
		      button, TRUE, TRUE, 0);
  
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      (GtkSignalFunc) new_image_cancel_callback,
		      dialog);

  gtk_widget_show (button);

  gtk_widget_show(vbox);
  gtk_widget_show(dialog->shell);
}

void
do_new (GtkWidget *widget, gpointer data)
{
  if (info.is_changed) 
    save_changes_dialog(new_image_dialog);
  else
    new_image_dialog();
}

static void
open_ok_callback (GtkWidget *w,
		  FileOpData *data)
{
  char *fn = gtk_file_selection_get_filename (
		    GTK_FILE_SELECTION (data->file_select));
  int len;

  if (!fn || !fn[0]) 
    return;			/* error message ? */

  len = strlen(fn);
  
  if (!load_bitmap(fn,type_to_compress(data->file_type,fn)))
    return;			/* error message */

  /* success ! */
  undo_reset();
  init_scale();
  
  if (info.name) 
    {
      g_free(info.name);
    }
  info.name = g_new(char,strlen(fn)+1);
  strcpy(info.name,fn);
  set_filename();
  info.file_type = data->file_type;
  
  gtk_widget_destroy(GTK_WIDGET(data->file_select));
  g_free(data);
}

static void
do_real_open()
{
  GtkWidget *window = NULL;
  GtkWidget *options_frame;
  GtkWidget *hbox;
  GtkWidget *label;
  GtkWidget *menu;
  GtkWidget *option_menu;

  FileOpData *data = g_new(FileOpData,1);

  window = gtk_file_selection_new ("Open File");
  gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);

  data->file_select = window;
  data->after_action = NULL;
  data->file_type = info.file_type;

  /* Options frame */

  options_frame = gtk_frame_new("Open Options");
  gtk_frame_set_shadow_type (GTK_FRAME(options_frame),
			     GTK_SHADOW_ETCHED_IN);
  gtk_container_add (GTK_CONTAINER(GTK_FILE_SELECTION (window)->main_vbox),
		     options_frame);

  hbox = gtk_hbox_new(FALSE,5);
  gtk_container_border_width(GTK_CONTAINER(hbox),5);
  gtk_container_add(GTK_CONTAINER(options_frame),hbox);
  gtk_widget_show(hbox);
  
  label = gtk_label_new("Determine file type:");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);
  gtk_widget_show(label);

  menu = file_type_menu(&data->file_type);
  option_menu = gtk_option_menu_new();

  gtk_box_pack_start(GTK_BOX(hbox), option_menu, TRUE, TRUE, 0);
  gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), menu);
  gtk_option_menu_set_history(GTK_OPTION_MENU(option_menu),data->file_type);
  gtk_widget_show(option_menu);

  gtk_widget_show(options_frame);


  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (window)->ok_button),
		      "clicked", (GtkSignalFunc) open_ok_callback,
		      data);
  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (window)->cancel_button),
			     "clicked", (GtkSignalFunc) gtk_widget_destroy,
			     GTK_OBJECT (window));

  if (info.name)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (data->file_select),
				     info.name);

  gtk_widget_show (window);
}

void
do_open (GtkWidget *w, gpointer data)
{
  if (info.is_changed)
    save_changes_dialog(do_real_open);
  else
    do_real_open();
}

/*
 * Save As.. dialog 
 */

static void
save_as_ok_callback (GtkWidget *w,
		     FileOpData *data)
{
  SaveChangesAction action = data->after_action;

  char *fn = gtk_file_selection_get_filename (
			  GTK_FILE_SELECTION (data->file_select));

  if (fn && fn[0]) 
    {
      /* FIXME, make save_bitmap return information about success */
      save_bitmap(fn,type_to_compress(data->file_type,fn));

      if (info.name) 
	{
	  g_free(info.name);
	}
      info.name = g_new(char,strlen(fn)+1);
      strcpy(info.name,fn);
      set_filename();
      info.file_type = data->file_type;
      info.is_changed = 0;

      gtk_widget_destroy(data->file_select);
      g_free(data);
      if (action)
	(action)();
    }
  /* display an error message? */
}

static void
save_as_cancel_callback (GtkWidget *w,
		     FileOpData *data)
{
  SaveChangesAction action = data->after_action;

  gtk_widget_destroy(data->file_select);
  g_free(data);
  
  if (action)
    (action)();
}

static void
save_as_dialog(SaveChangesAction after_action)
{
  GtkWidget *window = NULL;
  GtkWidget *options_frame;
  GtkWidget *label;
  GtkWidget *option_menu;
  GtkWidget *menu;
  GtkWidget *hbox;

  FileOpData *data = g_new(FileOpData,1);
  
  data->file_select = window = gtk_file_selection_new ("Save File As...");
  data->after_action = after_action;
  data->file_type = info.file_type;

  gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);


  /* Options frame */

  options_frame = gtk_frame_new("Save Options");
  gtk_frame_set_shadow_type (GTK_FRAME(options_frame),
			     GTK_SHADOW_ETCHED_IN);
  gtk_container_add (GTK_CONTAINER(GTK_FILE_SELECTION (window)->main_vbox),
		     options_frame);

  hbox = gtk_hbox_new(FALSE,5);
  gtk_container_border_width(GTK_CONTAINER(hbox),5);
  gtk_container_add(GTK_CONTAINER(options_frame),hbox);
  gtk_widget_show(hbox);
  
  label = gtk_label_new("Determine file type:");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);
  gtk_widget_show(label);

  menu = file_type_menu(&data->file_type);
  option_menu = gtk_option_menu_new();

  gtk_box_pack_start(GTK_BOX(hbox), option_menu, TRUE, TRUE, 0);
  gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), menu);
  gtk_option_menu_set_history(GTK_OPTION_MENU(option_menu),data->file_type);
  gtk_widget_show(option_menu);

  gtk_widget_show(options_frame);

    
  gtk_signal_connect ( GTK_OBJECT (GTK_FILE_SELECTION (window)->ok_button),
		       "clicked", (GtkSignalFunc) save_as_ok_callback,
		       data);
  gtk_signal_connect ( GTK_OBJECT (GTK_FILE_SELECTION (window)->cancel_button),
		       "clicked", 
		       (GtkSignalFunc) save_as_cancel_callback,
		       data);

  if (info.name)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (data->file_select),
				     info.name);
  else if (info.file_type == FILE_PBM)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (data->file_select),
				     "untitled.pbm");
  else
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (data->file_select),
				     "untitled.pbm.gz");

  gtk_widget_show (window);

  return;
}

void
do_save_as (GtkWidget *w, gpointer data)
{
  save_as_dialog(NULL);
}

void
do_save (GtkWidget *w, gpointer data)
{
  if (info.name) {
    if (info.is_changed) {      
      save_bitmap(info.name,type_to_compress(info.file_type,info.name));
      info.is_changed = 0;
    }
  } else {
    save_as_dialog(NULL);
  }
}

static void 
do_real_exit ()
{

  gtk_main_quit ();
}

void
do_exit (GtkWidget *w, gpointer data)
{
  if (info.is_changed)
    save_changes_dialog(do_real_exit);
  else
    do_real_exit();
}

