/* cqcam - Color Quickcam capture programs
 * Copyright (C) 1996-1998 by Patrick Reynolds
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <gtk/gtkadjustment.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkcheckbutton.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkhscale.h>
#include <gtk/gtklabel.h>
#include <gtk/gtksignal.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkwidget.h>
#include <gtk/gtkwindow.h>

#include <stdio.h>

#include <cqcsrv.h>
#include "gcam.h"
#include "dimensions.h"

static void camera_setting(GtkObject *adj, unsigned char *command);
static void program_setting(GtkObject *adj, unsigned char *loc);
static void program_flag(GtkObject *chk, unsigned char *loc);

int options_brightness, options_white, options_black;
int options_red, options_green, options_blue;
int options_auto_b, options_auto_rgb, options_despeckle, options_32bpp;

static GtkAdjustment *adj_brightness, *adj_red, *adj_green, *adj_blue;

static struct {
  char *name;
  int initial;
  unsigned char command;
  gpointer location;
  GtkAdjustment **slider;
  int do_dimensions_feedback;
} controls[] = {
  { "Brightness",       201, CQCSRV_SET_BRIGHTNESS, &options_brightness, &adj_brightness, 0 },
  { "Black level",      130, CQCSRV_SET_BLACK,      &options_white,      0,               0 },
  { "White level",      130, CQCSRV_SET_WHITE,      &options_black,      0,               0 },
  { "Red level",        128, 0,                     &options_red,        &adj_red,        0 },
  { "Green level",      128, 0,                     &options_green,      &adj_green,      0 },
  { "Blue level",       128, 0,                     &options_blue,       &adj_blue,       0 },
  
  { "Auto-brightness",  1,   0,                     &options_auto_b,     0,               0 },
  { "Auto-RGB",         1,   0,                     &options_auto_rgb,   0,               0 },
  { "Despeckling",      1,   0,                     &options_despeckle,  0,               0 },
  { "32-bit transfers", 0,   0,                     &options_32bpp,      0,               1 }
};
static const int nsliders = 6;
static const int ncheckboxes = 4;

static void this_destroy(GtkWidget *, GtkWidget *killme) {
  gtk_widget_destroy(killme);
}

GtkWidget *options_create_window() {
  GtkWidget *w = gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_window_set_title(GTK_WINDOW(w), "Gcam Options");
  gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, TRUE);
  gtk_container_border_width(GTK_CONTAINER(w), 10);

  GtkWidget *vbox = gtk_vbox_new(FALSE, 10);  
  GtkWidget *hbox = gtk_hbox_new(FALSE, 10);
  GtkWidget *left = gtk_vbox_new(TRUE, 15);
  GtkWidget *middle = gtk_vbox_new(TRUE, 15);
  GtkWidget *right = gtk_vbox_new(TRUE, 15);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 10);
  gtk_box_pack_start(GTK_BOX(hbox), left, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), middle, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), right, FALSE, FALSE, 0);
  gtk_container_add(GTK_CONTAINER(w), vbox);
  int i;
  for (i=0; i<nsliders; i++) {
    GtkWidget *label = gtk_label_new(controls[i].name);
    GtkObject *adj = gtk_adjustment_new(*(int*)controls[i].location,
      1, 254, 1, 10, 1);
    GtkWidget *slider = gtk_hscale_new(GTK_ADJUSTMENT(adj));
    gtk_widget_set_usize(slider, 200, 30);
    gtk_box_pack_start(GTK_BOX(left), label, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(middle), slider, TRUE, TRUE, 0);
    if (controls[i].command)
      gtk_signal_connect(GTK_OBJECT(GTK_RANGE(slider)->adjustment),
        "value_changed", GTK_SIGNAL_FUNC(camera_setting), &controls[i].command);
    if (controls[i].location)
      gtk_signal_connect(GTK_OBJECT(GTK_RANGE(slider)->adjustment),
        "value_changed", GTK_SIGNAL_FUNC(program_setting),
        controls[i].location);
    if (controls[i].do_dimensions_feedback)
      gtk_signal_connect(GTK_OBJECT(GTK_RANGE(slider)->adjustment),
        "value_changed", GTK_SIGNAL_FUNC(dimensions_feedback), 0);
    if (controls[i].slider)
      *controls[i].slider = GTK_RANGE(slider)->adjustment;
  }
  for (i=nsliders; i<nsliders+ncheckboxes; i++) {
    GtkWidget *checkbox = gtk_check_button_new_with_label(controls[i].name);
    gtk_box_pack_start(GTK_BOX(right), checkbox, FALSE, FALSE, 0);
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(checkbox),
      *(int*)controls[i].location);
    gtk_signal_connect(GTK_OBJECT(checkbox), "toggled",
      GTK_SIGNAL_FUNC(program_flag), controls[i].location);
    if (controls[i].do_dimensions_feedback)
      gtk_signal_connect(GTK_OBJECT(checkbox), "toggled",
        GTK_SIGNAL_FUNC(dimensions_feedback), 0);
  }

  GtkWidget *close_hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), close_hbox, FALSE, FALSE, 0);
  GtkWidget *close = gtk_button_new_with_label("Close");
  gtk_signal_connect(GTK_OBJECT(close), "clicked",
    GTK_SIGNAL_FUNC(this_destroy), w);
  gtk_box_pack_end(GTK_BOX(close_hbox), close, FALSE, FALSE, 0);
  GTK_WIDGET_SET_FLAGS(close, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(close);
  
  return w;
}

static void camera_setting(GtkObject *adj, unsigned char *command) {
  send_command(out_fd, *command, (int)GTK_ADJUSTMENT(adj)->value);
}

static void program_setting(GtkObject *adj, unsigned char *loc) {
  *(int*)loc = (int)GTK_ADJUSTMENT(adj)->value;
}

static void program_flag(GtkObject *chk, unsigned char *loc) {
  *(int*)loc = GTK_TOGGLE_BUTTON(chk)->active;
}

void options_init() {
  int i;
  for (i=0; i<nsliders+ncheckboxes; i++)
    if (controls[i].location)
      *(int*)controls[i].location = controls[i].initial;
}

void options_feedback() {
  gtk_adjustment_set_value(adj_brightness, options_brightness);
  gtk_adjustment_set_value(adj_red,        options_red);
  gtk_adjustment_set_value(adj_green,      options_green);
  gtk_adjustment_set_value(adj_blue,       options_blue);
}
