/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <sys/types.h>	/* kill */
#include <stdio.h>	/* FILE, sprintf, fopen, fclose, fprintf, ... */
#include <string.h>	/* strchr, strcmp, strlen, strcpy, memchr, memmove */
#include <stdlib.h>	/* strtol */
#include <unistd.h>	/* read, close, fork, pipe, exec, fcntl, _exit */
                        /* getpid, unlink */
#include <errno.h>      /* errno */
#include <fcntl.h>	/* fcntl */
#include <signal.h>	/* kill, signal... */
#include <netinet/in.h>	/* ntohl */

#include <gtk/gtk.h>

#include "xqf.h"
#include "pref.h"
#include "stat.h"
#include "utils.h"
#include "server.h"
#include "dialogs.h"
#include "dns.h"


#define MAX_TOKENS	64


static void stat_next (struct stat_job *job);


static int failed (char *name, char *arg) {
  fprintf (stderr, "%s(%s) failed: %s\n", name, (arg)? arg : "", 
                                                          g_strerror (errno));
  return TRUE;
}


static int tokenize (char *str, char *token[], char dlm) {
  int tokens = 0;

  if (!str || str[0] == '\0')
    return 0;

  while (str && tokens < MAX_TOKENS) {
    token[tokens++] = str;
    str = strchr (str, dlm);
    if (str)
      *str++ = '\0';
  }

  return tokens;
}


static struct player *parse_player (char *token[], int n, 
                                                       enum server_type type) {
  struct player *player = NULL;
  char *ptr;

  switch (type) {

  case QW_SERVER:
    if (n < 8)
      break;

    player = g_malloc (sizeof (struct player) + 
                               strlen (token[1]) + 1 + strlen (token[7]) + 1);
    player->id    = strtol (token[0], NULL, 10);
    player->frags = strtol (token[2], NULL, 10);
    player->time  = strtol (token[3], NULL, 10);
    player->shirt = strtol (token[4], NULL, 10);
    player->pants = strtol (token[5], NULL, 10);
    player->ping  = strtol (token[6], NULL, 10);

    ptr = (char *) player + sizeof (struct player);
    player->name = strcpy (ptr, token[1]);
    player->skin = strcpy (ptr + strlen (token[1]) + 1, token[7]);

    if (player->shirt < 0 || player->shirt > 13)
      player->shirt = 0;

    if (player->pants < 0 || player->pants > 13)
      player->pants = 0;

    break;

  case Q2_SERVER:
    if (n < 3)
      break;

    player = g_malloc (sizeof (struct player) + strlen (token[0]) + 1);
    player->frags = strtol (token[1], NULL, 10);
    player->ping  = strtol (token[2], NULL, 10);

    ptr = (char *) player + sizeof (struct player);
    player->name = strcpy (ptr, token[0]);

    player->id = 0;
    player->time  = 0;
    player->shirt = 0;
    player->pants = 0;
    player->skin = NULL;

    break;

  case ANY_SERVER:
    break;

  }

  return player;
}


int qsort_players (const void *a, const void *b) {
  struct player *p1 = * (struct player **) a;
  struct player *p2 = * (struct player **) b;
  int sortmode = player_clist_columns[psort_column].sortmode;
  int invert = 0;
  int res;

  if (sortmode < 0) {
    sortmode = -sortmode;
    invert = 1;
  }

  switch (sortmode) {
  case SORT_PLAYER_FRAGS:
    res = p1->frags - p2->frags;
    break;

  case SORT_PLAYER_ID:
    res = p1->id - p2->id;
    break;

  case SORT_PLAYER_NAME:
    if (p1->name)
      res = (p2->name)? strcmp (p1->name, p2->name) : 1;
    else
      res = (p2->name)? -1 : 0;
    break;

  case SORT_PLAYER_TIME:
    res = p1->time - p2->time;
    break;

  case SORT_PLAYER_COLOR:
    res = p1->pants - p2->pants;
    if (!res) 
      res = p1->shirt - p2->shirt;
    break;

  case SORT_PLAYER_PING:
    res = p1->ping - p2->ping;
    break;

  case SORT_PLAYER_SKIN:
    if (p1->skin)
      res = (p2->skin)? strcmp (p1->skin, p2->skin) : 1;
    else
      res = (p2->skin)? -1 : 0;
    break;

  case SORT_PLAYER_NONE:
  default:
    res = 0;
    break;
  }

  return (invert) ? -res : res;
}


int qsort_servers (const void *a, const void *b) {
  struct server *s1 = * (struct server **) a;
  struct server *s2 = * (struct server **) b;
  int sortmode = server_clist_columns[ssort_column].sortmode;
  int invert = 0;
  int res;

  if (sortmode < 0) {
    sortmode = -sortmode;
    invert = 1;
  }

  switch (sortmode) {
  case SORT_SERVER_NAME:
    if (s1->name)
      res = (s2->name)? strcmp (s1->name, s2->name) : 1;
    else
      res = (s2->name)? -1 : 0;
    break;

  case SORT_SERVER_ADDRESS:
    if (s1->host->ip.s_addr == s2->host->ip.s_addr)
      res = s1->port - s2->port;
    else
      res = (ntohl (s1->host->ip.s_addr) > ntohl (s2->host->ip.s_addr))? 1:-1;
    break;

  case SORT_SERVER_MAP:
    if (s1->map)
      res = (s2->map)? strcmp (s1->map, s2->map) : 1;
    else
      res = (s2->map)? -1 : 0;
    break;

  case SORT_SERVER_PLAYERS:
    res = s1->curplayers - s2->curplayers;
    if (!res) 
      res = s1->maxplayers - s2->maxplayers;
    break;

  case SORT_SERVER_PING:
    if (s1->ping >= 0) {
      if (s2->ping >= 0) {
	res = s1->ping - s2->ping;
	if (!res) 
	  res = s1->retries - s2->retries;
      }	
      else {
	res = -1;
      }
    }
    else
      res = (s2->ping >= 0)? 1 : 0;
    break;

  case SORT_SERVER_TO:
    if (s1->retries >= 0) {
      if (s2->retries >= 0) {
	res = s1->retries - s2->retries;
	if (!res) 
	  res = s1->ping - s2->ping;
      }
      else {
	res = -1;
      }
    }
    else
      res = (s2->retries >= 0)? 1 : 0;
    break;

  case SORT_SERVER_GAME:
    if (s1->game) {
      if (s2->game) {
	res = g_strcasecmp (s1->game, s2->game);
	if (!res)
	  res = strcmp (s1->game, s2->game);
      }
      else {
	res = 1;
      }
    }
    else
      res = (s2->game)? -1 : 0;
    break;

  case SORT_SERVER_NONE:
  default:
    res = 0;
    break;
  }

  return (invert) ? -res : res;
}


int qsort_serverinfo (const void *a, const void *b) {
  char *rule1 =  ((char **) a) [0];
  char *value1 = ((char **) a) [1];
  char *rule2 =  ((char **) b) [0];
  char *value2 = ((char **) b) [1];
  int sortmode = serverinfo_clist_columns[isort_column].sortmode;
  int invert = 0;
  int res;

  if (sortmode < 0) {
    sortmode = -sortmode;
    invert = 1;
  }

#if 0
  if (*rule1 == '*')
    rule1++;

  if (*rule2 == '*')
    rule2++;
#endif

  switch (sortmode) {

  case SORT_INFO_RULE:
    res = strcmp (rule1, rule2);
    if (!res)
      res = strcmp (value1, value2);
    break;

  case SORT_INFO_VALUE:
    res = strcmp (value1, value2);
    if (!res)
      res = strcmp (rule1, rule2);
    break;

  case SORT_INFO_NONE:
  default:
    res = 0;
    break;
  }

  return (invert) ? -res : res;
}


static void free_players (struct player **p, int n) {
  struct player **ptr = p;

  if (p) {
    while (n-- > 0)
      g_free (*ptr++);
    g_free (p);
  }
}


static void stat_close_conn (struct stat_job *job) {
  struct master *m;

  if (!job || !job->conn)
    return;

  job->progress.tasks_done++;
  job->progress.subtasks_done = 0;
  job->progress.subtasks_total = 0;

  if (job->conn->pid > 0) {
    kill (job->conn->pid, SIGTERM);
    job->conn->pid = 0;
  }

  if (job->conn->fd >= 0) {
    gdk_input_remove (job->conn->tag);
    close (job->conn->fd);
    job->conn->fd = -1;
  }

  if (job->conn->tmpfile) {
    unlink (job->conn->tmpfile);
    g_free (job->conn->tmpfile);
    job->conn->tmpfile = NULL;
  }

  if (job->conn->master) {
    if (job->conn->servers) {
      m = job->conn->master;
      free_servers (m->servers);
      m->servers = job->conn->servers;
      job->conn->master = NULL;
    }
  }
  else {
    free_servers (job->conn->servers);
    free_servers (job->servers);
    job->servers = NULL;
  }
  job->conn->servers = NULL;
}


static void stat_job_free_resolver_data (struct stat_job *job) {
  if (job->hosts) {
    g_slist_foreach (job->hosts, (GFunc) host_unref, NULL);
    g_slist_free (job->hosts);
    job->hosts = NULL;
  }

  if (job->names) {
    g_slist_foreach (job->names, (GFunc) g_free, NULL);
    g_slist_free (job->names);
    job->names = NULL;
  }
}



void stat_job_free (struct stat_job *job) {
  if (job->masters)
    g_slist_free (job->masters);

  if (job->servers)
    free_servers (job->servers);

  stat_job_free_resolver_data (job);

  g_slist_free (job->master_handlers);
  g_slist_free (job->server_handlers);
  g_slist_free (job->host_handlers);
  g_slist_free (job->state_handlers);
  g_slist_free (job->close_handlers);
  g_free (job);
}


static char **parse_serverinfo (char *token[], int n) {
  char **info;
  int size = 0;
  int i, j;
  char *ptr, *info_ptr;

  if (n == 0)
    return NULL;

  for (size = 0, i = 0; i < n; i++) {
    size += strlen (token[i]) + 1;
  }

  info = g_malloc (sizeof (char *) * (n + 1) * 2 + size);
  info_ptr = (char *) info + sizeof (char *) * (n + 1) * 2;

  for (i = 0, j = 0; i < n; i++) {
    ptr = strchr (token[i], '=');

    if (ptr)
      *ptr++ = '\0';

    info [j++] = strcpy (info_ptr, token[i]);
    info_ptr += strlen (token[i]) + 1;

    if (ptr) {
      info [j++] = strcpy (info_ptr, ptr);
      info_ptr += strlen (ptr) + 1;
    }
    else {
      info [j++] = NULL;
    }
  }

  info [j + 1] = info [j] = NULL;

  return info;
}


static struct master *parse_master (struct stat_job *job, 
                                                       char *token[], int n) {
  struct host *h;
  struct master *m;
  char *addr;
  int port;
  GSList *list;
  int i;
  char *endptr;

  if (n < 3)
    return NULL;

  if (strcmp (token[0], "QWM") && strcmp (token[0], "Q2M"))
    return NULL;

  if (!parse_address (token[1], &addr, &port))
    return NULL;

  h = host_add (addr);
  g_free (addr);
  if (!h)
    return NULL;

  h->ref_count++;

  m = (struct master *) job->conn->master;
  if (m->host == h && m->port == port) {
    i = strtol (token[2], &endptr, 10);
    if (endptr == token[2])	/* timeout */
      i = -1;

    job->progress.subtasks_done = 0;
    job->progress.subtasks_total = (i > 0)? i : 0;

    for (list = job->master_handlers; list; list = list->next)
      (* (master_func) list->data) (job, m, i);

    host_unref (h);
    return m;
  }

  host_unref (h);
  return NULL;
}


static struct server *parse_server (char *token[], int n, struct master *m) {
  struct host *h;
  struct server *s;
  enum server_type type;
  char *addr;
  int port;

  if (n < 3)
    return NULL;

  if (strcmp (token[0], "QW") == 0) {
    type = QW_SERVER;
  }
  else if (strcmp (token[0], "Q2") == 0) {
    type = Q2_SERVER;
  }
  else {
    return NULL;
  }

  if (!parse_address (token[1], &addr, &port))
    return NULL;

  h = host_add (addr);
  g_free (addr);
  if (!h)
    return NULL;

  s = server_add (h, port, type);

  if (s->ref_count > 0) {
    if (s->map) {
      g_free (s->map);
      s->map  = NULL;
    }
    if (s->players) {
      free_players (s->players, s->curplayers);
      s->players = NULL;
    }
    if (s->info) {
      g_free (s->info);
      s->info = NULL;
      s->game = NULL;
    }
    s->flags = 0;
  }

  if (n >= 8) {
    s->type = type;    /* QW <--> Q2 */

    if (s->name) 
      g_free (s->name);

    if (*(token[2]))		/* if name is not empty */
      s->name = g_strdup (token[2]);

    if (*(token[3]))            /* if map is not empty */
      s->map  = g_strdup (token[3]);

    s->maxplayers = strtol (token[4], NULL, 10);
    s->curplayers = strtol (token[5], NULL, 10);

    s->ping = strtol (token[6], NULL, 10);
    if (s->ping > MAX_PING)
      s->ping = MAX_PING;

    s->retries = strtol (token[7], NULL, 10);
    if (s->retries > maxretries)
      s->retries = maxretries;
  }
  else {
    s->maxplayers = 0;
    s->curplayers = 0;
    s->retries = MAX_RETRIES;

    if (m)
      s->type = m->type;

    if (n == 3 && strcmp (token[2], "DOWN") == 0)
      s->ping = MAX_PING + 1;
    else 
      s->ping = MAX_PING;	/* TIMEOUT */
  }

  return s;
}


static void parse_qstat_record_part2 (GSList *strings, struct stat_job *job,
                                                           struct server *s) {
  int n;
  char *token[MAX_TOKENS];
  char **info_ptr;
  struct player *p;
  GSList *plist = NULL;
  int pnum = 0;
  GSList *tmp;
  int i;

  n = tokenize ((char *) strings->data, token, QSTAT_DELIM);
  if (n == 0)
    return;

  s->info = parse_serverinfo (token, n);
  if (s->info) {
    for (info_ptr = s->info; info_ptr && *info_ptr; info_ptr += 2) {
      if ((s->type == QW_SERVER && strcmp (*info_ptr, "*gamedir") == 0) ||
	  (s->type == Q2_SERVER && strcmp (*info_ptr, "gamedir") == 0)) {
	s->game = info_ptr[1];
	continue;
      }
      else if ((s->type == QW_SERVER && strcmp (*info_ptr, "*cheats") == 0) ||
	       (s->type == Q2_SERVER && strcmp (*info_ptr, "cheats") == 0 && 
		                                     info_ptr[1][0] != '0')) {
	s->flags |= SERVER_CHEATS;
      }
      else if (strcmp (*info_ptr, "needpass") == 0) {
	i = strtol (info_ptr[1], NULL, 10);
	if (i & 1)
	  s->flags |= SERVER_PASSWORD;
	if (i & 2)
	  s->flags |= SERVER_SP_PASSWORD;
      }
    }

    strings = strings->next;
    while (strings) {
      n = tokenize ((char *) strings->data, token, QSTAT_DELIM);
      p = parse_player (token, n, s->type);
      if (!p)		/* error, try to recover */
	return;

      plist = g_slist_prepend (plist, p);
      pnum++;

      strings = strings->next;
    }

    s->curplayers = pnum;
    if (pnum == 0)
      return;
  
    s->players = g_malloc (sizeof (struct player *) * pnum);
    for (tmp = plist, i = 0; tmp; i++) {
      s->players[i] = (struct player *) tmp->data;
      tmp = tmp->next;
    }
    g_slist_free (plist);
  }
}


static void parse_qstat_record (GSList *strings, struct stat_job *job) {
  struct master *m;
  struct server *s;
  char *token[MAX_TOKENS];
  int n;
  GSList *list;

  if (!strings)
    return;	/* error, try to recover */

  n = tokenize ((char *) strings->data, token, QSTAT_DELIM);
  if (n < 3)
    return;     /* error, try to recover */

  m = parse_master (job, token, n);
  if (m) {
    free_servers (job->conn->servers);
    job->conn->servers = NULL;

    job->conn->master = m;
    return;
  }

  s = parse_server (token, n, job->conn->master);
  if (s) {
    job->conn->servers = g_slist_prepend (job->conn->servers, s);
    s->ref_count++; 

    strings = strings->next;
    if (strings) {
      parse_qstat_record_part2 (strings, job, s);
    }

    job->progress.subtasks_done++;
    
    for (list = job->server_handlers; list; list = list->next)
      (* (server_func) list->data) (job, s);
  }
}


static void adjust_pointers (GSList *list, gpointer new, gpointer old) {
  while (list) {
    list->data = new + (list->data - old);
    list = list->next;
  }
}


static void stat_input_callback (struct stat_job *job, int fd, 
                                                GdkInputCondition condition) {
  struct statconn *c = job->conn;
  int first_used = 0;
  int blocked = FALSE;
  char *tmp;
  int res;

  if (c->bufsize - c->pos < BUFFER_TRESHOLD) {
    if (c->bufsize >= BUFFER_MAXSIZE) {
      fprintf (stderr, "server record is too large\n");
      stat_close_conn (job);
      stat_next (job);
      return;
    }
    c->bufsize += c->bufsize;
    tmp = c->buf;
    c->buf = g_realloc (c->buf, c->bufsize);
    adjust_pointers (c->strings, c->buf, tmp);
  }

  res = read (fd, c->buf + c->pos, c->bufsize - c->pos);
  if (res < 0) {
    if (errno == EAGAIN || errno == EWOULDBLOCK)
      return;
    failed ("read", NULL);
    stat_close_conn (job);
    stat_next (job);
    return;
  }
  if (res == 0) {	/* EOF */
    job->conn->pid = 0;
    stat_close_conn (job);
    stat_next (job);
    return;
  }

  tmp = c->buf + c->pos;
  c->pos += res;

  while (res && (tmp = memchr (tmp, '\n', res)) != NULL) {
    *tmp++ = '\0';

    if (c->buf[c->lastnl] == '\0') {
      blocked = TRUE;
      gdk_input_remove (job->conn->tag);

      parse_qstat_record (c->strings, job);

      g_slist_free (c->strings);
      c->strings = NULL;

      first_used = c->lastnl + 1;
    }
    else {
      c->strings = g_slist_append (c->strings, c->buf + c->lastnl);
    }

    c->lastnl = tmp - c->buf;
    res = c->buf + c->pos - tmp;
  }

  if (first_used) {
    if (first_used == c->pos) {
      c->pos = 0;
      c->lastnl = 0;
    }
    else {
      tmp = c->buf + first_used;
      memmove (c->buf, c->buf + first_used, c->pos - first_used);
      c->pos -= first_used;
      c->lastnl -= first_used;
      adjust_pointers (c->strings, c->buf, tmp);
    }
  }

  if (blocked) {
    job->conn->tag = gdk_input_add (job->conn->fd, GDK_INPUT_READ, 
                                 (GdkInputFunction) stat_input_callback, job);
  }
}


static void parse_resolved (char *str, struct stat_job *job) {
  struct host *h;
  GSList *list;
  char *token[MAX_TOKENS];
  int n;
  char *idstr = NULL;

  if (!str || !str[0])
    return;

  n = tokenize (str, token, RESOLVE_DELIM);
  if (n != 3)
    return;

  if (*token[0]) {
    h = host_add (token[0]);
    if (!h) {			/* token[0] is not an IP address */
      idstr = token[0];
      h = host_add (token[1]);
    }

    job->progress.subtasks_done++;

    if (h) {
      if (*token[2]) {
	if (h->name)
	  g_free (h->name);
	h->name = g_strdup (token[2]);
      }
      h->ref_count++;
    }

    for (list = job->host_handlers; list; list = list->next)
      (* (host_func) list->data) (job, h, idstr);

    if (h) {
      host_unref (h);
    }
  }
}


static void resolver_input_callback (struct stat_job *job, int fd, 
                                                GdkInputCondition condition) {
  struct statconn *c = job->conn;
  int first_used = 0;
  int blocked = FALSE;
  char *tmp;
  int res;

  if (c->pos >= BUFFER_MINSIZE - BUFFER_TRESHOLD) {
    fprintf (stderr, "resolved host name is too large\n");
    stat_close_conn (job);
    stat_next (job);
    return;
  }

  res = read (fd, c->buf + c->pos, c->bufsize - c->pos);
  if (res < 0) {
    if (errno == EAGAIN || errno == EWOULDBLOCK)
      return;
    failed ("read", NULL);
    stat_close_conn (job);
    stat_next (job);
    return;
  }
  if (res == 0) {	/* EOF */
    job->conn->pid = 0;
    stat_close_conn (job);
    stat_next (job);
    return;
  }

  tmp = c->buf + c->pos;
  c->pos += res;

  while (res && (tmp = memchr (tmp, '\n', res)) != NULL) {
    *tmp++ = '\0';

    blocked = TRUE;
    gdk_input_remove (job->conn->tag);

    parse_resolved (c->buf + first_used, job);

    first_used = tmp - c->buf;
    res = c->buf + c->pos - tmp;
  }

  if (first_used > 0) {
    if (first_used != c->pos) {
      memmove (c->buf, c->buf + first_used, c->pos - first_used);
    }
    c->pos = c->pos - first_used;
  }

  if (blocked) {
    job->conn->tag = gdk_input_add (job->conn->fd, GDK_INPUT_READ, 
                             (GdkInputFunction) resolver_input_callback, job);
  }
}


static void set_nonblock (int fd) {
  int flags;

  flags = fcntl (fd, F_GETFL, 0);
  if (flags < 0 || fcntl (fd, F_SETFL, flags | O_NONBLOCK) < 0) {
    failed ("fcntl", NULL);
    return;
  }
}


static void start_qstat (struct stat_job *job, char *argv[]) {
  pid_t pid;
  int pipefds[2];

  if (pipe (pipefds) < 0) {
    failed ("pipe", NULL);
    return;
  }

  pid = fork ();
  if (pid < (pid_t) 0) {
    failed ("fork", NULL);
    return;
  }

  if (pid) {	/* parent */
    close (pipefds[1]);

    set_nonblock (pipefds[0]);

    if (!job->conn) {
      job->conn = g_malloc (sizeof (struct statconn));
      job->conn->buf = g_malloc (BUFFER_MINSIZE);
      job->conn->bufsize = BUFFER_MINSIZE;
      job->conn->tmpfile = NULL;
    }

    job->conn->pid = pid;
    job->conn->fd = pipefds[0];
    job->conn->pos = 0;
    job->conn->lastnl = 0;

    job->conn->strings = NULL;
    job->conn->servers = NULL;
    job->conn->master  = NULL;

    job->conn->tag = gdk_input_add (job->conn->fd, GDK_INPUT_READ, 
                                 (GdkInputFunction) stat_input_callback, job);
  }
  else {	/* child */
    close (pipefds[0]);
    dup2 (pipefds[1], 1);
    close (pipefds[1]);

    execvp (argv[0], argv);

    failed ("execvp", argv[0]);
    _exit (1);
  }
}


static void stat_close (struct stat_job *job, int killed) {
  GSList *list;
  GSList *tmp;

  if (job->conn) {
    if (job->conn->pid > 0)
      stat_close_conn (job);

    if (job->conn->buf)
      g_free (job->conn->buf);

    g_free (job->conn);
    job->conn = NULL;
  }

  if (job->masters) {
    g_slist_free (job->masters);
    job->masters = NULL;
  }

  if (job->servers) {
    free_servers (job->servers);
    job->servers = NULL;
  }

  stat_job_free_resolver_data (job);

  list = slist_replicate (job->close_handlers);
  for (tmp = list; tmp; tmp = tmp->next) {
    (* (close_func) tmp->data) (job, killed);
  }
  g_slist_free (list);
}


#define MAX_SERVERS_IN_CMDLINE	8

static void stat_open_conn_qstat (struct stat_job *job) {
  struct master *m = NULL;
  struct server *s = NULL;
  GSList *tmp;
  char *argv[16 + MAX_SERVERS_IN_CMDLINE*2];
  int argi = 0;
  char buf[64 + MAX_SERVERS_IN_CMDLINE*64];
  int bufi = 0;
  FILE *f = NULL;
  char *fn = NULL;

  if (!job || (!job->masters && !job->servers))
    return;

  argv[argi++] = "qstat";

  argv[argi++] = "-maxsimultaneous";
  argv[argi++] = &buf[bufi];
  bufi += 1 + sprintf (&buf[bufi], "%d", maxsimultaneous);

  argv[argi++] = "-retry";
  argv[argi++] = &buf[bufi];
  bufi += 1 + sprintf (&buf[bufi], "%d", maxretries);
  
  argv[argi++] = "-raw";
  argv[argi++] = &buf[bufi];
  bufi += 1 + sprintf (&buf[bufi], "%c", QSTAT_DELIM);

  argv[argi++] = "-R";

  argv[argi++] = "-P";

  job->progress.subtasks_done = 0;

  if (job->masters) {
    job->progress.subtasks_total = 0;

    m = (struct master *) job->masters->data;
    job->masters = g_slist_remove (job->masters, m);

    argv[argi++] = (m->protocol == QW_SERVER)? "-qw" : "-q2m";
    argv[argi++] = &buf[bufi];
    bufi += 1 + sprintf (&buf[bufi], "%s:%d", m->host->address, m->port);
  }
  else {
    job->progress.subtasks_total = g_slist_length (job->servers);

    job->servers = g_slist_reverse (job->servers);

    tmp = job->servers;
    
    if (g_slist_length (tmp) <= MAX_SERVERS_IN_CMDLINE) {
      while (tmp) {
	s = (struct server *) tmp->data;

	argv[argi++] = "-qws";
	argv[argi++] = &buf[bufi];
	bufi += 1 + sprintf (&buf[bufi], "%s:%d", s->host->address, s->port);

	tmp = tmp->next;
      }
    }
    else {
      argv[argi++] = "-f";
      fn = &buf[bufi];
      bufi += 1 + sprintf (fn, "/tmp/xqf-qstat.%d", getpid ());
      argv[argi++] = fn;

      f = fopen (fn, "w");
      if (!f) {
	dialog_ok (NULL, "/tmp is not writable");
	stat_close (job, TRUE);	/* stop everything */
	return;
      }

      while (tmp) {
	s = (struct server *) tmp->data;
	fprintf (f, "%s %s:%d\n", (s->type == QW_SERVER)? "QW" : "Q2",
                                                   s->host->address, s->port);
	tmp = tmp->next;
      }

      fclose (f);
    }
  }
  
  argv[argi] = NULL;

#ifdef DEBUG
  for (argi = 0; argv[argi]; argi++)
    fprintf (stderr, "%s ", argv[argi]);
  fprintf (stderr, "\n");
#endif

  start_qstat (job, argv);
  job->conn->tmpfile = g_strdup (fn);
  job->conn->master = m;
}


static void stat_open_conn_resolver (struct stat_job *job) {
  pid_t pid;
  int pipefds[2];
  GSList *strings = NULL;
  GSList *tmp;
  struct host *h;

  if (!job || (!job->hosts && !job->names))
    return;

  for (tmp = job->hosts; tmp; tmp = tmp->next) {
    h = (struct host *) tmp->data;
    strings = g_slist_prepend (strings, h->address);
  }

  for (tmp = job->names; tmp; tmp = tmp->next) {
    strings = g_slist_prepend (strings, tmp->data);
  }

  strings = g_slist_reverse (strings);

  if ((pipe (pipefds) < 0 && failed ("pipe", NULL)) ||
      ((pid = fork ()) < (pid_t) 0 && failed ("fork", NULL))) {
    g_slist_free (strings);
    return;
  }

  if (pid) {	/* parent */
    close (pipefds[1]);
    set_nonblock (pipefds[0]);

    if (!job->conn) {
      job->conn = g_malloc (sizeof (struct statconn));
      job->conn->buf = g_malloc (BUFFER_MINSIZE);
      job->conn->bufsize = BUFFER_MINSIZE;
      job->conn->tmpfile = NULL;
    }

    job->conn->pid = pid;
    job->conn->fd = pipefds[0];
    job->conn->pos = 0;
    job->conn->lastnl = 0;

    job->conn->strings = NULL;
    job->conn->servers = NULL;
    job->conn->master  = NULL;

    job->conn->tag = gdk_input_add (job->conn->fd, GDK_INPUT_READ, 
                             (GdkInputFunction) resolver_input_callback, job);

    job->progress.subtasks_total = g_slist_length (strings);
    job->progress.subtasks_done = 0;

    g_slist_free (strings);
  }
  else {	/* child */
    on_sig (SIGHUP,  _exit);
    on_sig (SIGINT,  _exit);
    on_sig (SIGQUIT, _exit);
    on_sig (SIGBUS,  _exit);
    on_sig (SIGSEGV, _exit);
    on_sig (SIGPIPE, _exit);
    on_sig (SIGTERM, _exit);
  
    close (pipefds[0]);
    dup2 (pipefds[1], 1);
    close (pipefds[1]);

    resolve (strings, (maxsimultaneous <= 4)? 4 : 8);

    _exit (0);
  }
}


static void change_state (struct stat_job *job, enum stat_state state) {
  GSList *tmp;
  for (tmp = job->state_handlers; tmp; tmp = tmp->next) {
    (* (state_func) tmp->data) (job, state);
  }
}


static void stat_next (struct stat_job *job) {
  if (job->masters) {
    stat_open_conn_qstat (job);
    change_state (job, STAT_SOURCE);
  }
  else if (job->servers) {
    stat_open_conn_qstat (job);
    change_state (job, STAT_SERVER);
  }
  else if (job->hosts || job->names) {
    stat_open_conn_resolver (job);
    change_state (job, STAT_RESOLVE);
    stat_job_free_resolver_data (job);
  }
  else
    stat_close (job, FALSE);
}


void stat_start (struct stat_job *job) {
  int nmasters = g_slist_length (job->masters);

  if (nmasters <= 1)
    job->progress.tasks_total = 1;
  else
    job->progress.tasks_total = nmasters + 1;

  if (default_always_resolve)
    job->progress.tasks_total++;

  stat_next (job);
}


void stat_stop (struct stat_job *job) {
  stat_close (job, TRUE);
}


struct stat_job *stat_job_create (GSList *masters, GSList *servers, 
                                               GSList *hosts, GSList *names) {
  struct stat_job *job;

  job = g_malloc (sizeof (struct stat_job));
  job->masters = masters;
  job->servers = servers;
  job->hosts   = hosts;
  job->names   = names;
  job->conn    = NULL;
  job->master_handlers = NULL;
  job->server_handlers = NULL;
  job->host_handlers  = NULL;
  job->state_handlers  = NULL;
  job->close_handlers  = NULL;

  job->single	= FALSE;
  job->redraw	= FALSE;

  job->progress.tasks_total = 0;
  job->progress.tasks_done = 0;
  job->progress.subtasks_total = 0;
  job->progress.subtasks_done = 0;

  job->data    = NULL;

  return job;
}


float stat_job_progress (struct stat_job *job) {
  float onetask, onesubtask;
  int done, subdone;

  if (!job)
    return 0;

  onetask = (job->progress.tasks_total)? 1.0 / job->progress.tasks_total : 0;
  onesubtask = (job->progress.subtasks_total)? 
                                   onetask / job->progress.subtasks_total : 0;

  done = (job->progress.tasks_done > job->progress.tasks_total)?
                         job->progress.tasks_total : job->progress.tasks_done;

  subdone = (job->progress.subtasks_done > job->progress.subtasks_total)?
                   job->progress.subtasks_total : job->progress.subtasks_done;

  return  onetask * done + onesubtask * subdone;
}


