/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>	/* fprintf, sprintf */
#include <string.h>	/* strchr, strncpy */
#include <stdlib.h>	/* strtol */

#include <gtk/gtk.h>

#include "xqf.h"
#include "dns.h"
#include "server.h"


struct server_hash {
  int num;
  GSList **nodes;
};

static struct server_hash servers = { 251, NULL };


static int server_hash_func (struct host *h, int port) {
  unsigned char *ptr;

  if (!h)
    return 0;

  ptr = (char *) &h->ip.s_addr;
  return (ptr[0] + (ptr[1] << 2) + (ptr[2] << 4) + (ptr[3] << 6) + port) % 
                                                                  servers.num;
}


struct server *server_new (struct host *h, int port, enum server_type type) {
  struct server *s;

  if (!h || port <= 0 || (type != QW_SERVER && type != Q2_SERVER))
    return NULL;

  s = g_malloc (sizeof (struct server));

  s->host = h;
  h->ref_count++;

  s->port = port;
  s->type = type;
  s->name = NULL;
  s->players = NULL;
  s->info = NULL;
  s->game = NULL;
  s->map  = NULL;
  s->maxplayers = 0;
  s->curplayers = 0;
  s->ping = -1;
  s->retries = -1;
  s->flags = 0;
  s->ref_count = 0;
  return s;
}


struct server *server_add (struct host *h, int port, enum server_type type) {
  struct server *s;
  GSList *ptr;
  int node;
  int i;

  if (!h)
    return NULL;

  if (port < 0) {
    switch (type) {

    case QW_SERVER:
      port = QW_DEFAULT_PORT;
      break;

    case Q2_SERVER:
      port = Q2_DEFAULT_PORT;
      break;

    default:
      return NULL;
    }
  }

  node = server_hash_func (h, port);

  if (!servers.nodes) {
    servers.nodes = g_malloc (sizeof (GSList *) * servers.num);
    for (i = 0; i < servers.num; i++)
      servers.nodes[i] = NULL;
  }
  else {
    for (ptr = servers.nodes[node]; ptr; ptr = ptr->next) {
      s = (struct server *) ptr->data;
      if (s->host == h && s->port == port)
	return s;
    }
  }

  s = server_new (h, port, type);
  if (s)
    servers.nodes[node] = g_slist_prepend (servers.nodes[node], s);
  return s;
}


void server_unref (struct server *s) {
  int node;

  if (!s || !servers.nodes)
    return;

  s->ref_count--;

  if (s->ref_count <= 0) {
    node = server_hash_func (s->host, s->port);
    servers.nodes[node] = g_slist_remove (servers.nodes[node], s);

    host_unref (s->host);
    if (s->name) g_free (s->name);
    if (s->map) g_free (s->map);
    if (s->players) g_free (s->players);
    if (s->info) g_free (s->info);
    g_free (s);
  }
}


int servers_total (void) {
  int i;
  int size = 0;

  if (!servers.nodes)
    return 0;

  for (i = 0; i < servers.num; i++) {
    size += g_slist_length (servers.nodes[i]);
  }
  return size;
}


GSList *all_servers (void) {
  GSList *list = NULL;
  GSList *tmp;
  int i;

  if (!servers.nodes)
    return NULL;

  for (i = 0; i < servers.num; i++) {
    for (tmp = servers.nodes[i]; tmp; tmp = tmp->next) {
      list = g_slist_prepend (list, tmp->data);
    }
  }

  for (tmp = list; tmp; tmp = tmp->next) {
    ((struct server *) tmp->data) -> ref_count++;
  }

  return list;
}


int parse_address (char *str, char **addr, int *port) {
  char *ptr;

  if(!str || !addr || !port)
    return FALSE;

  ptr = strchr (str, ':');
  if (!ptr) {
    *port = -1;
    *addr = g_strdup (str);
    return TRUE;
  }

  *port = strtol (ptr + 1, NULL, 10);
  if ((ptr == str) || (*port <= 0 || *port >= 64*1024)) {
    *port = -1;
    *addr = NULL;
    return FALSE;
  }

  *addr = g_malloc (ptr - str + 1);
  strncpy (*addr, str, ptr - str);
  (*addr) [ptr - str] = '\0';
  return TRUE;
}


void free_servers (GSList *list) {
  if (list) {
    g_slist_foreach (list, (GFunc) server_unref, NULL);
    g_slist_free (list);
  }
}


