/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>	/* sprintf */
#include <stdlib.h>	/* strtol */
#include <string.h>	/* strcmp */
#include <sys/time.h>	/* FD_SETSIZE */
#include <sys/types.h>	/* FD_SETSIZE */

#include <gtk/gtk.h>

#include "xqf.h"
#include "pref.h"
#include "dialogs.h"
#include "skin.h"
#include "utils.h"
#include "qrun.h"
#include "srv-prop.h"
#include "pixmaps.h"
#include "xutils.h"


struct user_info user;

int 	default_rate = 2500;
int 	default_cl_nodelta = FALSE;
int 	default_cl_predict = TRUE;
int 	default_noaim = FALSE;
int 	default_pushlatency = TRUE;
int 	default_windowed_mouse = FALSE;
int 	default_nosound = FALSE;
int 	default_nocdaudio = FALSE;
char 	*default_name = NULL;
char 	*default_team = NULL;
char 	*default_qw_skin = NULL;
char 	*default_q2_skin = NULL;
char 	*default_qw_cfg = NULL;
char 	*default_q2_cfg = NULL;
int 	default_top_color = 0;
int 	default_bottom_color = 0;
int 	default_noskins = 0;
char 	*default_quake_dir = NULL;
char	*default_qw_cmd = NULL;
char 	*default_q2_dir = NULL;
char	*default_q2_cmd = NULL;
int	default_terminate = FALSE;
int	default_iconify = FALSE;
int 	default_b_switch = 0;
int 	default_w_switch = 0;
int	default_save_lists = TRUE;
int	default_auto_favorites = FALSE;
int	default_always_resolve = TRUE;

int	filter_retries = 3;
int	filter_ping = 2000;
int 	filter_not_full = 0;
int 	filter_not_empty = 0;
int	filter_no_cheats = 0;
int	filter_no_password = 0;

int     maxretries = 3;
int     maxsimultaneous = 20;

int	filter_prefs_changed = FALSE;

char 	*real_quake_dir = NULL;
char 	*real_q2_dir = NULL;


static	int pref_top_color;
static	int pref_bottom_color;
static	int pref_b_switch;
static	int pref_w_switch;
static	int pref_noskins;
static	char *pref_qw_skin;
static	char *pref_q2_skin;
static  char *pref_quake_dir = NULL;
static  char *pref_q2_dir = NULL;

static  char *real_pref_quake_dir = NULL;
static  char *real_pref_q2_dir = NULL;


static  GtkWidget *rate_spinner;
static  GtkWidget *cl_nodelta_check_button;
static  GtkWidget *cl_predict_check_button;
static  GtkWidget *noaim_check_button;
static  GtkWidget *pushlat_check_button;
static  GtkWidget *windowed_mouse_check_button;
static  GtkWidget *nosound_check_button;
static  GtkWidget *nocdaudio_check_button;
static  GtkWidget *name_entry;
static  GtkWidget *team_entry;
static  GtkWidget *qw_skin_preview = NULL;
static  GtkWidget *q2_skin_preview = NULL;
static  GtkWidget *qw_skin_combo;
static  GtkWidget *q2_skin_combo;
static  GtkWidget *qw_cfg_combo;
static  GtkWidget *q2_cfg_combo;
static  GtkWidget *top_color_button;
static  GtkWidget *bottom_color_button;
static  GtkWidget *quake_dir_entry;
static 	GtkWidget *qw_cmd_entry;
static  GtkWidget *q2_dir_entry;
static 	GtkWidget *q2_cmd_entry;
static  GtkWidget *terminate_check_button;
static  GtkWidget *iconify_check_button;
static  GtkWidget *save_lists_check_button;
static  GtkWidget *auto_favorites_check_button;
static  GtkWidget *always_resolve_check_button;

static  GtkWidget *filter_retries_spinner;
static  GtkWidget *filter_ping_spinner;
static  GtkWidget *filter_not_full_check_button;
static  GtkWidget *filter_not_empty_check_button;
static  GtkWidget *filter_no_cheats_check_button;
static  GtkWidget *filter_no_password_check_button;

static  GtkWidget *maxretries_spinner;
static  GtkWidget *maxsimultaneous_spinner;

static 	guchar *qw_skin_data = NULL;
static  int qw_skin_is_valid = TRUE;

static 	guchar *q2_skin_data = NULL;
static  int q2_skin_is_valid = TRUE;

static	GtkWidget *color_menu = NULL;


static void get_new_defaults (void) {
  int i;

  if (default_name) g_free (default_name);
  default_name = strdup_strip (gtk_entry_get_text (GTK_ENTRY (name_entry)));

  if (default_team) g_free (default_team);
  default_team = strdup_strip (gtk_entry_get_text (GTK_ENTRY (team_entry)));

  if (default_qw_cfg) g_free (default_qw_cfg);
  default_qw_cfg = strdup_strip (gtk_entry_get_text (
                                GTK_ENTRY (GTK_COMBO (qw_cfg_combo)->entry)));

  if (default_q2_cfg) g_free (default_q2_cfg);
  default_q2_cfg = strdup_strip (gtk_entry_get_text (
                                GTK_ENTRY (GTK_COMBO (q2_cfg_combo)->entry)));

  default_rate = gtk_spin_button_get_value_as_int (
                                              GTK_SPIN_BUTTON (rate_spinner));

  default_cl_nodelta = GTK_TOGGLE_BUTTON (cl_nodelta_check_button)->active;
  default_cl_predict = GTK_TOGGLE_BUTTON (cl_predict_check_button)->active;
  default_noaim = 1 - GTK_TOGGLE_BUTTON (noaim_check_button)->active;
  default_pushlatency = GTK_TOGGLE_BUTTON (pushlat_check_button)->active;

  if (default_qw_cmd) g_free (default_qw_cmd);
  default_qw_cmd = 
                 strdup_strip (gtk_entry_get_text (GTK_ENTRY (qw_cmd_entry)));

  if (default_q2_cmd) g_free (default_q2_cmd);
  default_q2_cmd = 
                 strdup_strip (gtk_entry_get_text (GTK_ENTRY (q2_cmd_entry)));

  default_windowed_mouse = 
                      GTK_TOGGLE_BUTTON (windowed_mouse_check_button)->active;
  default_nosound = TRUE - GTK_TOGGLE_BUTTON (nosound_check_button)->active;
  default_nocdaudio = 
                    TRUE - GTK_TOGGLE_BUTTON (nocdaudio_check_button)->active;
  default_terminate = GTK_TOGGLE_BUTTON (terminate_check_button)->active;
  default_iconify = GTK_TOGGLE_BUTTON (iconify_check_button)->active;
  default_save_lists = GTK_TOGGLE_BUTTON (save_lists_check_button)->active;
  default_auto_favorites = 
                      GTK_TOGGLE_BUTTON (auto_favorites_check_button)->active;
  default_always_resolve = 
                      GTK_TOGGLE_BUTTON (always_resolve_check_button)->active;

  i = gtk_spin_button_get_value_as_int (
                                    GTK_SPIN_BUTTON (filter_retries_spinner));
  if (filter_retries != i) {
    filter_retries = i;
    filter_prefs_changed = TRUE;
  }

  i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (filter_ping_spinner));
  if (filter_ping != i) {
    filter_ping = i;
    filter_prefs_changed = TRUE;
  }

  i = GTK_TOGGLE_BUTTON (filter_not_full_check_button)->active;
  if (filter_not_full != i) {
    filter_not_full = i;
    filter_prefs_changed = TRUE;
  }

  i = GTK_TOGGLE_BUTTON (filter_not_empty_check_button)->active;
  if (filter_not_empty != i) {
    filter_not_empty = i;
    filter_prefs_changed = TRUE;
  }

  i = GTK_TOGGLE_BUTTON (filter_no_cheats_check_button)->active;
  if (filter_no_cheats != i) {
    filter_no_cheats = i;
    filter_prefs_changed = TRUE;
  }

  i = GTK_TOGGLE_BUTTON (filter_no_password_check_button)->active;
  if (filter_no_password != i) {
    filter_no_password  = i;
    filter_prefs_changed = TRUE;
  }

  maxretries = gtk_spin_button_get_value_as_int (
                                         GTK_SPIN_BUTTON (maxretries_spinner));

  maxsimultaneous = gtk_spin_button_get_value_as_int (
                                    GTK_SPIN_BUTTON (maxsimultaneous_spinner));

  /* things that are already set by callbacks */

  default_top_color    = pref_top_color;
  default_bottom_color = pref_bottom_color;
  default_b_switch     = pref_b_switch;
  default_w_switch     = pref_w_switch;
  default_noskins      = pref_noskins;

  if (default_qw_skin) g_free (default_qw_skin);
  default_qw_skin = pref_qw_skin;
  pref_qw_skin = NULL;

  if (default_q2_skin) g_free (default_q2_skin);
  default_q2_skin = pref_q2_skin;
  pref_q2_skin = NULL;

  if (default_quake_dir) g_free (default_quake_dir);
  default_quake_dir = strdup_strip (gtk_entry_get_text (
                                               GTK_ENTRY (quake_dir_entry)));
  if (real_quake_dir) g_free (real_quake_dir);
  real_quake_dir = expand_tilde (default_quake_dir, user.home);

  if (pref_quake_dir) g_free (pref_quake_dir);
  if (real_pref_quake_dir) g_free (real_pref_quake_dir);
  pref_quake_dir = NULL;
  real_pref_quake_dir = NULL;

  if (default_q2_dir) g_free (default_q2_dir);
  default_q2_dir = strdup_strip (gtk_entry_get_text (GTK_ENTRY (q2_dir_entry)));
  if (real_q2_dir) g_free (real_q2_dir);
  real_q2_dir = expand_tilde (default_q2_dir, user.home);

  if (pref_q2_dir) g_free (pref_q2_dir);
  if (real_pref_q2_dir) g_free (real_pref_q2_dir);
  pref_q2_dir = NULL;
  real_pref_q2_dir = NULL;
}


static void set_pref_defaults (void) {
  pref_top_color    = default_top_color;
  pref_bottom_color = default_bottom_color;
  pref_b_switch     = default_b_switch;
  pref_w_switch     = default_w_switch;
  pref_noskins      = default_noskins;

  pref_qw_skin      = g_strdup (default_qw_skin);
  pref_q2_skin      = g_strdup (default_q2_skin);

  pref_quake_dir    = g_strdup (default_quake_dir);
  real_pref_quake_dir = g_strdup (real_quake_dir);

  pref_q2_dir       = g_strdup (default_q2_dir);
  real_pref_q2_dir  = g_strdup (real_q2_dir);
}


static void update_qw_skins (char *initstr) {
  GList *list;
  char *str = NULL;

  if (qw_skin_preview == NULL)
    return;			/* no configuration window */

  if (qw_skin_data) {
    g_free (qw_skin_data);
    qw_skin_data = NULL;
  }

  list = get_qw_skin_list (real_pref_quake_dir);

  if (initstr) {
    combo_set_vals (qw_skin_combo, list, initstr);
  }
  else {
    str = g_strdup (gtk_entry_get_text (
                               GTK_ENTRY (GTK_COMBO (qw_skin_combo)->entry)));
    combo_set_vals (qw_skin_combo, list, str);
  }

  if (list) {
    qw_skin_data = get_skin (pref_qw_skin, real_pref_quake_dir, FALSE);
    g_list_foreach (list, (GFunc) g_free, NULL);
    g_list_free (list);
  }

  if (str) 
    g_free (str);

  if (qw_skin_data || qw_skin_is_valid) {
    draw_qw_skin (qw_skin_preview, qw_skin_data, 
                                           pref_top_color, pref_bottom_color);
    qw_skin_is_valid = (qw_skin_data)? TRUE : FALSE;
  }
}


static void update_qw_cfgs (char *initstr) {
  GList *cfgs;
  char *str = NULL;

  cfgs = find_custom_cfgs (QW_SERVER, real_pref_quake_dir, NULL);

  if (initstr) {
    combo_set_vals (qw_cfg_combo, cfgs, initstr);
  }
  else {
    str = g_strdup (gtk_entry_get_text (
                                GTK_ENTRY (GTK_COMBO (qw_cfg_combo)->entry)));
    combo_set_vals (qw_cfg_combo, cfgs, str);
  }

  if (cfgs) {
    g_list_foreach (cfgs, (GFunc) g_free, NULL);
    g_list_free (cfgs);
  }

  if (str)
    g_free (str);
}


static void qw_dir_entry_activate_callback (GtkWidget *widget, gpointer data) {

  if (pref_quake_dir) g_free (pref_quake_dir);
  pref_quake_dir = strdup_strip (gtk_entry_get_text (
                                                 GTK_ENTRY (quake_dir_entry)));
  if (real_pref_quake_dir) g_free (real_pref_quake_dir);
  real_pref_quake_dir = expand_tilde (pref_quake_dir, user.home);

  update_qw_skins (NULL);
  update_qw_cfgs (NULL);
}


static void qw_skin_combo_changed_callback (GtkWidget *widget, gpointer data) {
  char *new_skin;

  new_skin = strdup_strip (
           gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (qw_skin_combo)->entry)));

  if (!pref_qw_skin && !new_skin)
    return;

  if (pref_qw_skin && new_skin) {
    if (strcmp (pref_qw_skin, new_skin) == 0) {
      g_free (new_skin);
      return;
    }
  }

  if (pref_qw_skin) g_free (pref_qw_skin);
  pref_qw_skin = new_skin;

  if (qw_skin_data) g_free (qw_skin_data);
  qw_skin_data = get_skin (pref_qw_skin, real_pref_quake_dir, FALSE);

  if (qw_skin_data || qw_skin_is_valid) {
    draw_qw_skin (qw_skin_preview, qw_skin_data, 
                                           pref_top_color, pref_bottom_color);
    qw_skin_is_valid = (qw_skin_data)? TRUE : FALSE;
  }
}


static GtkWidget *color_button_event_widget = NULL;


static void set_player_color (GtkWidget *widget, int i) {

  if (color_button_event_widget == top_color_button) {
    if (pref_top_color != i) {
      pref_top_color = i;
      set_bg_color (top_color_button, pref_top_color);
      if (qw_skin_is_valid) {
	draw_qw_skin (qw_skin_preview, qw_skin_data, 
                                           pref_top_color, pref_bottom_color);
      }
    }
    return;
  }

  if (color_button_event_widget == bottom_color_button) {
    if (pref_bottom_color != i) {
      pref_bottom_color = i;
      set_bg_color (bottom_color_button, pref_bottom_color);
      if (qw_skin_is_valid) {
	draw_qw_skin (qw_skin_preview, qw_skin_data,
                                           pref_top_color, pref_bottom_color);
      }
    }
    return;
  }
}


static int color_button_event_callback (GtkWidget *widget, GdkEvent *event) {
  GdkEventButton *bevent; 

  if (event->type == GDK_BUTTON_PRESS) {
    bevent = (GdkEventButton *) event; 
    color_button_event_widget = widget;

    if (color_menu == NULL)
      color_menu = create_color_menu (set_player_color);

    gtk_menu_popup (GTK_MENU (color_menu), NULL, NULL, NULL, NULL,
	                                         bevent->button, bevent->time);
    return TRUE;
  }
  return FALSE;
}


static GtkWidget *qw_skin_box_create (void) {
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *alignment;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;

  vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_border_width (GTK_CONTAINER (vbox), 6);

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  /* QW Skin ComboBox */

  alignment = gtk_alignment_new (0, 0, 0, 0);
  gtk_box_pack_start (GTK_BOX (hbox), alignment, FALSE, FALSE, 0);

  qw_skin_combo = gtk_combo_new ();
  gtk_entry_set_max_length (GTK_ENTRY (GTK_COMBO (qw_skin_combo)->entry), 256);
  gtk_widget_set_usize (GTK_COMBO (qw_skin_combo)->entry, 112, -1);
  gtk_combo_set_use_arrows_always (GTK_COMBO (qw_skin_combo), TRUE);
  gtk_combo_set_case_sensitive (GTK_COMBO (qw_skin_combo), TRUE);
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO (qw_skin_combo)->entry), 
           "changed", GTK_SIGNAL_FUNC (qw_skin_combo_changed_callback), NULL);
  gtk_container_add (GTK_CONTAINER (alignment), qw_skin_combo);
  gtk_widget_show (qw_skin_combo);

  gtk_widget_show (alignment);

  /* Top and Bottom Colors */

  table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_box_pack_end (GTK_BOX (hbox), table, FALSE, FALSE, 2);

  /* Top (Shirt) Color */

  label = gtk_label_new ("Top");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 0, 1);
  gtk_widget_show (label);

  top_color_button = gtk_button_new_with_label (" ");
  gtk_widget_set_usize (top_color_button, 40, -1);
  gtk_signal_connect (GTK_OBJECT(top_color_button), "event",
                          GTK_SIGNAL_FUNC (color_button_event_callback), NULL);
  gtk_table_attach_defaults (GTK_TABLE (table), top_color_button, 1, 2, 0, 1);
  set_bg_color (top_color_button, fix_qw_player_color (pref_top_color));
  gtk_widget_show (top_color_button);

  /* Bottom (Pants) Color */

  label = gtk_label_new ("Bottom");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 1, 2);
  gtk_widget_show (label);

  bottom_color_button = gtk_button_new_with_label (" ");
  gtk_widget_set_usize (bottom_color_button, 40, -1);
  gtk_signal_connect (GTK_OBJECT(bottom_color_button), "event",
		          GTK_SIGNAL_FUNC (color_button_event_callback), NULL);
  gtk_table_attach_defaults (GTK_TABLE (table), bottom_color_button, 1, 2, 1, 2);
  set_bg_color (bottom_color_button, fix_qw_player_color (pref_bottom_color));
  gtk_widget_show (bottom_color_button);

  gtk_widget_show (table);

  gtk_widget_show (hbox);

  /* Skin Preview  */

  alignment = gtk_alignment_new (0.5, 0.5, 0, 0);
  gtk_box_pack_start (GTK_BOX (vbox), alignment, FALSE, FALSE, 0);

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_container_add (GTK_CONTAINER (alignment), frame);

  qw_skin_preview = gtk_preview_new (GTK_PREVIEW_COLOR);
  gtk_preview_size (GTK_PREVIEW (qw_skin_preview), 320, 200);
  gtk_container_add (GTK_CONTAINER (frame), qw_skin_preview);
  gtk_widget_show (qw_skin_preview);

  gtk_widget_show (frame);

  gtk_widget_show (alignment);

  gtk_widget_show (vbox);

  return vbox;
}


static void update_q2_skins (char *initstr) {
  GList *list;
  char *str = NULL;

  if (q2_skin_preview == NULL)
    return;			/* no configuration window */

  if (q2_skin_data) {
    g_free (q2_skin_data);
    q2_skin_data = NULL;
  }

  list = get_q2_skin_list (real_pref_q2_dir);

  if (initstr) {
    combo_set_vals (q2_skin_combo, list, initstr);
  }
  else {
    str = g_strdup (gtk_entry_get_text (
                               GTK_ENTRY (GTK_COMBO (q2_skin_combo)->entry)));
    combo_set_vals (q2_skin_combo, list, str);
  }

  if (list) {
    q2_skin_data = get_skin (pref_q2_skin, real_pref_q2_dir, TRUE);
    g_list_foreach (list, (GFunc) g_free, NULL);
    g_list_free (list);
  }

  if (str) 
    g_free (str);

  if (q2_skin_data || q2_skin_is_valid) {
    draw_q2_skin (q2_skin_preview, q2_skin_data, Q2_SKIN_SCALE);
    q2_skin_is_valid = (q2_skin_data)? TRUE : FALSE;
  }
}


static void update_q2_cfgs (char *initstr) {
  GList *cfgs;
  char *str = NULL;

  cfgs = find_custom_cfgs (Q2_SERVER, real_pref_q2_dir, NULL);

  if (initstr) {
    combo_set_vals (q2_cfg_combo, cfgs, initstr);
  }
  else {
    str = g_strdup (gtk_entry_get_text (
                                GTK_ENTRY (GTK_COMBO (q2_cfg_combo)->entry)));
    combo_set_vals (q2_cfg_combo, cfgs, str);
  }

  if (cfgs) {
    g_list_foreach (cfgs, (GFunc) g_free, NULL);
    g_list_free (cfgs);
  }

  if (str)
    g_free (str);
}


static void q2_dir_entry_activate_callback (GtkWidget *widget, gpointer data) {

  if (pref_q2_dir) g_free (pref_q2_dir);
  pref_q2_dir = strdup_strip (gtk_entry_get_text (GTK_ENTRY (q2_dir_entry)));
  if (real_pref_q2_dir) g_free (real_pref_q2_dir);
  real_pref_q2_dir = expand_tilde (pref_q2_dir, user.home);

  update_q2_skins (NULL);
  update_q2_cfgs (NULL);
}


static void q2_skin_combo_changed_callback (GtkWidget *widget, gpointer data) {
  char *new_skin;

  new_skin = strdup_strip (
           gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (q2_skin_combo)->entry)));

  if (!pref_q2_skin && !new_skin)
    return;

  if (pref_q2_skin && new_skin) {
    if (strcmp (pref_q2_skin, new_skin) == 0) {
      g_free (new_skin);
      return;
    }
  }

  if (pref_q2_skin) g_free (pref_q2_skin);
  pref_q2_skin = new_skin;

  if (q2_skin_data) g_free (q2_skin_data);
  q2_skin_data = get_skin (pref_q2_skin, real_pref_q2_dir, TRUE);

  if (q2_skin_data || q2_skin_is_valid) {
    draw_q2_skin (q2_skin_preview, q2_skin_data, Q2_SKIN_SCALE);
    q2_skin_is_valid = (q2_skin_data)? TRUE : FALSE;
  }
}


static GtkWidget *q2_skin_box_create (void) {
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *alignment;
  GtkWidget *frame;

  vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_border_width (GTK_CONTAINER (vbox), 6);

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  /* Skin Preview  */

  alignment = gtk_alignment_new (0, 0.5, 0, 0);
  gtk_box_pack_start (GTK_BOX (hbox), alignment, FALSE, FALSE, 0);

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_container_add (GTK_CONTAINER (alignment), frame);

  q2_skin_preview = gtk_preview_new (GTK_PREVIEW_COLOR);
  gtk_preview_size (GTK_PREVIEW (q2_skin_preview), 
                                      32 * Q2_SKIN_SCALE, 32 * Q2_SKIN_SCALE);
  gtk_container_add (GTK_CONTAINER (frame), q2_skin_preview);
  gtk_widget_show (q2_skin_preview);

  gtk_widget_show (frame);
  gtk_widget_show (alignment);

  /* Q2 Skin ComboBox */

  alignment = gtk_alignment_new (1.0, 0, 0, 0);
  gtk_box_pack_end (GTK_BOX (hbox), alignment, FALSE, FALSE, 0);

  q2_skin_combo = gtk_combo_new ();
  gtk_entry_set_max_length (GTK_ENTRY (GTK_COMBO (q2_skin_combo)->entry), 256);
  gtk_widget_set_usize (GTK_COMBO (q2_skin_combo)->entry, 144, -1);
  gtk_combo_set_use_arrows_always (GTK_COMBO (q2_skin_combo), TRUE);
  gtk_combo_set_case_sensitive (GTK_COMBO (q2_skin_combo), TRUE);
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO (q2_skin_combo)->entry),
           "changed", GTK_SIGNAL_FUNC (q2_skin_combo_changed_callback), NULL);
  gtk_container_add (GTK_CONTAINER (alignment), q2_skin_combo);
  gtk_widget_show (q2_skin_combo);

  gtk_widget_show (alignment);

  gtk_widget_show (hbox);

  gtk_widget_show (vbox);

  return vbox;
}


static GtkWidget *player_profile_qw_page (void) {
  GtkWidget *page_vbox;
  GtkWidget *alignment;
  GtkWidget *frame;
  GtkWidget *qw_skin;
  GtkWidget *hbox;
  GtkWidget *label;

  page_vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_border_width (GTK_CONTAINER (page_vbox), 6);

  /* QW Skin */

  alignment = gtk_alignment_new (0.5, 0.5, 0, 0);
  gtk_box_pack_start (GTK_BOX (page_vbox), alignment, FALSE, FALSE, 0);

  frame = gtk_frame_new (" Skin ");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_add (GTK_CONTAINER (alignment), frame);

  qw_skin = qw_skin_box_create ();
  gtk_container_add (GTK_CONTAINER (frame), qw_skin);

  gtk_widget_show (frame);
  gtk_widget_show (alignment);

  /* QW Team */

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_box_pack_start (GTK_BOX (page_vbox), hbox, FALSE, FALSE, 4);

  label = gtk_label_new ("Team");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  team_entry = gtk_entry_new_with_max_length (32);
  gtk_widget_set_usize (team_entry, 96, -1);
  if (default_team) {
    gtk_entry_set_text (GTK_ENTRY (team_entry), default_team);
    gtk_entry_set_position (GTK_ENTRY (team_entry), 0);
  }
  gtk_box_pack_start (GTK_BOX (hbox), team_entry, FALSE, FALSE, 0);
  gtk_widget_show (team_entry);

  /* QW Custom CFG */

  qw_cfg_combo = gtk_combo_new ();
  gtk_entry_set_max_length (GTK_ENTRY (GTK_COMBO (qw_cfg_combo)->entry), 256);
  gtk_widget_set_usize (GTK_COMBO (qw_cfg_combo)->entry, 96, -1);
  gtk_combo_set_case_sensitive (GTK_COMBO (qw_cfg_combo), TRUE);
  gtk_box_pack_end (GTK_BOX (hbox), qw_cfg_combo, FALSE, FALSE, 0);
  gtk_widget_show (qw_cfg_combo);

  label = gtk_label_new ("Custom CFG");
  gtk_box_pack_end (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  gtk_widget_show (hbox);

  gtk_widget_show (page_vbox);

  qw_skin_is_valid = TRUE;
  update_qw_skins (pref_qw_skin);
  update_qw_cfgs (default_qw_cfg);

  return page_vbox;
}


static GtkWidget *player_profile_q2_page (void) {
  GtkWidget *page_vbox;
  GtkWidget *alignment;
  GtkWidget *frame;
  GtkWidget *q2_skin;
  GtkWidget *hbox;
  GtkWidget *label;

  page_vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_border_width (GTK_CONTAINER (page_vbox), 8);

  alignment = gtk_alignment_new (0.5, 0.5, 0, 0);
  gtk_box_pack_start (GTK_BOX (page_vbox), alignment, FALSE, FALSE, 0);

  frame = gtk_frame_new (" Model/Skin ");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_add (GTK_CONTAINER (alignment), frame);

  q2_skin = q2_skin_box_create ();
  gtk_container_add (GTK_CONTAINER (frame), q2_skin);

  gtk_widget_show (frame);
  gtk_widget_show (alignment);

  /* Q2 Custom CFG */

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_box_pack_start (GTK_BOX (page_vbox), hbox, FALSE, FALSE, 4);

  q2_cfg_combo = gtk_combo_new ();
  gtk_entry_set_max_length (GTK_ENTRY (GTK_COMBO (q2_cfg_combo)->entry), 256);
  gtk_widget_set_usize (GTK_COMBO (q2_cfg_combo)->entry, 96, -1);
  gtk_combo_set_case_sensitive (GTK_COMBO (q2_cfg_combo), TRUE);
  gtk_box_pack_end (GTK_BOX (hbox), q2_cfg_combo, FALSE, FALSE, 0);
  gtk_widget_show (q2_cfg_combo);

  label = gtk_label_new ("Custom CFG");
  gtk_box_pack_end (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  gtk_widget_show (hbox);

  gtk_widget_show (page_vbox);

  q2_skin_is_valid = TRUE;
  update_q2_skins (pref_q2_skin);
  update_q2_cfgs (default_q2_cfg);

  return page_vbox;
}


static GtkWidget *player_profile_page (void) {
  GtkWidget *page_vbox;
  GtkWidget *notebook;
  GtkWidget *page;
  GtkWidget *label;
  GtkWidget *pixmap;
  GtkWidget *hbox;

  page_vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_border_width (GTK_CONTAINER (page_vbox), 8);

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_box_pack_start (GTK_BOX (page_vbox), hbox, FALSE, FALSE, 8);

  /* Player Name */

  label = gtk_label_new ("Name");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  name_entry = gtk_entry_new_with_max_length (32);
  gtk_widget_set_usize (name_entry, 96, -1);
  if (default_name) {
    gtk_entry_set_text (GTK_ENTRY (name_entry), default_name);
    gtk_entry_set_position (GTK_ENTRY (name_entry), 0);
  }
  gtk_box_pack_start (GTK_BOX (hbox), name_entry, FALSE, FALSE, 0);
  gtk_widget_show (name_entry);

  gtk_widget_show (hbox);

  notebook = gtk_notebook_new ();
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
  gtk_box_pack_start (GTK_BOX (page_vbox), notebook, FALSE, FALSE, 0);

  hbox = gtk_hbox_new (FALSE, 0);

  pixmap = gtk_pixmap_new (q_pix, q_pix_mask);
  gtk_box_pack_start (GTK_BOX (hbox), pixmap, FALSE, FALSE, 2);
  gtk_widget_show (pixmap);

  label = gtk_label_new ("QuakeWorld");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 2);
  gtk_widget_show (label);

  gtk_widget_show (hbox);

  page = player_profile_qw_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, hbox);

  hbox = gtk_hbox_new (FALSE, 0);

  pixmap = gtk_pixmap_new (q2_pix, q2_pix_mask);
  gtk_box_pack_start (GTK_BOX (hbox), pixmap, FALSE, FALSE, 2);
  gtk_widget_show (pixmap);

  label = gtk_label_new ("Quake2");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 2);
  gtk_widget_show (label);

  gtk_widget_show (hbox);

  page = player_profile_q2_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, hbox);

  gtk_widget_show (notebook);

  gtk_widget_show (page_vbox);

  return page_vbox;
}


static char *wb_switch_labels[9] = {
  "--- Anything ---",
  "Axe",
  "Shotgun",
  "Super Shotgun",
  "Nailgun",
  "Super Nailgun",
  "Grenade Launcher",
  "Rocket Launcher",
  "ThunderBolt"
};


static void set_w_switch_callback (GtkWidget *widget, int i) {
  pref_w_switch = i;
}


static void set_b_switch_callback (GtkWidget *widget, int i) {
  pref_b_switch = i;
}


static GtkWidget *create_wb_switch_menu (void (*callback) (GtkWidget *, int)) {
  GtkWidget *menu;
  GtkWidget *menu_item;
  int i;

  menu = gtk_menu_new ();

  for (i = 0; i < 9; i++) {
    menu_item = gtk_menu_item_new_with_label (wb_switch_labels[i]);
    gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
			GTK_SIGNAL_FUNC (callback), (gpointer) i);
    gtk_menu_append (GTK_MENU (menu), menu_item);
    gtk_widget_show (menu_item);
  }

  return menu;
}


static void noskins_option_menu_callback (GtkWidget *widget, int i) {
  pref_noskins = i;
}


static GtkWidget *create_noskins_menu (void) {
  GtkWidget *menu;
  GtkWidget *menu_item;

  menu = gtk_menu_new ();

  menu_item = gtk_menu_item_new_with_label ("Use skins");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                 GTK_SIGNAL_FUNC (noskins_option_menu_callback), (gpointer) 0);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_menu_item_new_with_label ("Don't use skins");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                 GTK_SIGNAL_FUNC (noskins_option_menu_callback), (gpointer) 1);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_menu_item_new_with_label ("Don't download new skins");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                 GTK_SIGNAL_FUNC (noskins_option_menu_callback), (gpointer) 2);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  return menu;
}


static GtkWidget *gameplay_options_page (void) {
  GtkWidget *page_vbox;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *frame;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *option_menu;
  GtkObject *adj;

  page_vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_border_width (GTK_CONTAINER (page_vbox), 8);

  /* QuakeWorld frame */

  frame = gtk_frame_new (" QuakeWorld ");
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_container_border_width (GTK_CONTAINER (table), 6);
  gtk_container_add (GTK_CONTAINER (frame), table);

  /* QW command line */

  label = gtk_label_new ("Command Line");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, 
                                                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (label);

  qw_cmd_entry = gtk_entry_new ();
  if (default_qw_cmd) {
    gtk_entry_set_text (GTK_ENTRY (qw_cmd_entry), default_qw_cmd);
    gtk_entry_set_position (GTK_ENTRY (qw_cmd_entry), 0);
  }
  gtk_table_attach_defaults (GTK_TABLE (table), qw_cmd_entry, 1, 2, 0, 1);
  gtk_widget_show (qw_cmd_entry);

  /* QW working directory */

  label = gtk_label_new ("Working Directory");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, 
                                                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (label);

  quake_dir_entry = gtk_entry_new ();
  if (pref_quake_dir) {
    gtk_entry_set_text (GTK_ENTRY (quake_dir_entry), pref_quake_dir);
    gtk_entry_set_position (GTK_ENTRY (quake_dir_entry), 0);
  }
  gtk_table_attach_defaults (GTK_TABLE (table), quake_dir_entry, 1, 2, 1, 2);
  gtk_signal_connect (GTK_OBJECT (quake_dir_entry), "activate",
		      GTK_SIGNAL_FUNC (qw_dir_entry_activate_callback), NULL);
  gtk_signal_connect (GTK_OBJECT (quake_dir_entry), "focus_out_event",
		      GTK_SIGNAL_FUNC (qw_dir_entry_activate_callback), NULL);
  gtk_widget_show (quake_dir_entry);

  gtk_widget_show (table);
  gtk_widget_show (frame);

  /* Quake II frame */

  frame = gtk_frame_new (" Quake II ");
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_container_border_width (GTK_CONTAINER (table), 6);
  gtk_container_add (GTK_CONTAINER (frame), table);

  /* Q2 command line */

  label = gtk_label_new ("Command Line");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, 
                                                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (label);

  q2_cmd_entry = gtk_entry_new ();
  if (default_q2_cmd) {
    gtk_entry_set_text (GTK_ENTRY (q2_cmd_entry), default_q2_cmd);
    gtk_entry_set_position (GTK_ENTRY (q2_cmd_entry), 0);
  }
  gtk_table_attach_defaults (GTK_TABLE (table), q2_cmd_entry, 1, 2, 0, 1);
  gtk_widget_show (q2_cmd_entry);

  /* Q2 working directory */

  label = gtk_label_new ("Working Directory");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, 
                                                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (label);

  q2_dir_entry = gtk_entry_new ();
  if (pref_q2_dir) {
    gtk_entry_set_text (GTK_ENTRY (q2_dir_entry), pref_q2_dir);
    gtk_entry_set_position (GTK_ENTRY (q2_dir_entry), 0);
  }
  gtk_table_attach_defaults (GTK_TABLE (table), q2_dir_entry, 1, 2, 1, 2);
  gtk_signal_connect (GTK_OBJECT (q2_dir_entry), "activate",
		      GTK_SIGNAL_FUNC (q2_dir_entry_activate_callback), NULL);
  gtk_signal_connect (GTK_OBJECT (q2_dir_entry), "focus_out_event",
		      GTK_SIGNAL_FUNC (q2_dir_entry_activate_callback), NULL);
  gtk_widget_show (q2_dir_entry);

  gtk_widget_show (table);
  gtk_widget_show (frame);

  /* _windowed_mouse & nosound & nocdaudio */

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_box_pack_start (GTK_BOX (page_vbox), hbox, FALSE, FALSE, 0);

  /* _windowed_mouse */

  windowed_mouse_check_button = 
                  gtk_check_button_new_with_label ("Client should grab mouse");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (windowed_mouse_check_button), 
                                                       default_windowed_mouse);
  gtk_box_pack_start (GTK_BOX (hbox), windowed_mouse_check_button, FALSE, FALSE, 0);
  gtk_widget_show (windowed_mouse_check_button);

  /* nocdaudio */

  nocdaudio_check_button = gtk_check_button_new_with_label ("CD Audio");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (nocdaudio_check_button), 
                                                    TRUE - default_nocdaudio);
  gtk_box_pack_end (GTK_BOX (hbox), nocdaudio_check_button, FALSE, FALSE, 0);
  gtk_widget_show (nocdaudio_check_button);

  /* nosound */

  nosound_check_button = gtk_check_button_new_with_label ("Sound");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (nosound_check_button), 
                                                      TRUE - default_nosound);
  gtk_box_pack_end (GTK_BOX (hbox), nosound_check_button, FALSE, FALSE, 0);
  gtk_widget_show (nosound_check_button);

  gtk_widget_show (hbox);

  /* Network Options */

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  table = gtk_table_new (2, 3, FALSE);
  gtk_container_border_width (GTK_CONTAINER (table), 6);
  gtk_table_set_row_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacing (GTK_TABLE (table), 0, 8);
  gtk_container_add (GTK_CONTAINER (frame), table);

  /* Skins */

  option_menu = gtk_option_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), 
                                                       create_noskins_menu ());
  gtk_option_menu_set_history (GTK_OPTION_MENU (option_menu), pref_noskins);
  gtk_table_attach_defaults (GTK_TABLE (table), option_menu, 0, 1, 0, 1);
  gtk_widget_show (option_menu);

  /* rate */

  label = gtk_label_new ("rate");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 0, 1);
  gtk_widget_show (label);

  adj = gtk_adjustment_new (default_rate, 0.0, 25000.0, 500.0, 1000.0, 0.0);

  rate_spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 0, 0);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (rate_spinner), 
                                                            GTK_UPDATE_ALWAYS);
  gtk_widget_set_usize (rate_spinner, 64, -1);
  gtk_table_attach_defaults (GTK_TABLE (table), rate_spinner, 2, 3, 0, 1);
  gtk_widget_show (rate_spinner);

  /* cl_nodelta */

  cl_nodelta_check_button = gtk_check_button_new_with_label ("cl_nodelta");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (cl_nodelta_check_button), 
                                                           default_cl_nodelta);
  gtk_table_attach_defaults (GTK_TABLE (table), cl_nodelta_check_button, 1, 3, 1, 2);
  gtk_widget_show (cl_nodelta_check_button);

  /* cl_predict_players (cl_predict in Q2) */

  cl_predict_check_button = gtk_check_button_new_with_label ("cl_predict_players");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (cl_predict_check_button), 
                                                           default_cl_predict);
  gtk_table_attach_defaults (GTK_TABLE (table), cl_predict_check_button, 0, 1, 1, 2);
  gtk_widget_show (cl_predict_check_button);

  gtk_widget_show (table);
  gtk_widget_show (frame);

  /* QW specific features */

  frame = gtk_frame_new (" QuakeWorld only ");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_border_width (GTK_CONTAINER (vbox), 6);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  /* w_switch & b_switch */

  label = gtk_label_new ("The highest weapon that Quake should switch to");
  gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  /* w_switch */

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  label = gtk_label_new ("upon a weapon pickup");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  option_menu = gtk_option_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), 
                               create_wb_switch_menu (set_w_switch_callback));
  gtk_option_menu_set_history (GTK_OPTION_MENU (option_menu), pref_w_switch);
  gtk_box_pack_end (GTK_BOX (hbox), option_menu, FALSE, FALSE, 0);
  gtk_widget_show (option_menu);

  gtk_widget_show (hbox);

  /* b_switch */

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  label = gtk_label_new ("upon a backpack pickup");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  option_menu = gtk_option_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), 
                                create_wb_switch_menu (set_b_switch_callback));
  gtk_option_menu_set_history (GTK_OPTION_MENU (option_menu), pref_b_switch);
  gtk_box_pack_end (GTK_BOX (hbox), option_menu, FALSE, FALSE, 0);
  gtk_widget_show (option_menu);

  gtk_widget_show (hbox);

  /* noaim */

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  pushlat_check_button = gtk_check_button_new_with_label (
                                                     "Automatic Pushlatency");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (pushlat_check_button), 
                                                         default_pushlatency);
  gtk_box_pack_end (GTK_BOX (hbox), pushlat_check_button, FALSE, FALSE, 0);
  gtk_widget_show (pushlat_check_button);

  noaim_check_button = gtk_check_button_new_with_label ("Autoaiming");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (noaim_check_button), 
                                                           1 - default_noaim);
  gtk_box_pack_start (GTK_BOX (hbox), noaim_check_button, FALSE, FALSE, 0);
  gtk_widget_show (noaim_check_button);

  gtk_widget_show (hbox);

  gtk_widget_show (vbox);
  gtk_widget_show (frame);

  gtk_widget_show (page_vbox);

  return page_vbox;
}


static GtkWidget *filter_options_page (void) {
  GtkWidget *page_vbox;
  GtkWidget *alignment;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GtkObject *adj;

  page_vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_border_width (GTK_CONTAINER (page_vbox), 8);

  frame = gtk_frame_new (" Server would pass filter if ");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  alignment = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);
  gtk_container_add (GTK_CONTAINER (frame), alignment);

  table = gtk_table_new (6, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_container_border_width (GTK_CONTAINER (table), 6);
  gtk_container_add (GTK_CONTAINER (alignment), table);

  /* max ping */

  label = gtk_label_new ("ping is less than");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL,
                                                                         0, 0);
  gtk_widget_show (label);

  adj = gtk_adjustment_new (filter_ping, 0.0, MAX_PING, 100.0, 1000.0, 0.0);

  filter_ping_spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 0, 0);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (filter_ping_spinner), 
                                                            GTK_UPDATE_ALWAYS);
  gtk_widget_set_usize (filter_ping_spinner, 64, -1);
  gtk_table_attach_defaults (GTK_TABLE (table), filter_ping_spinner, 1, 2, 0, 1);
  gtk_widget_show (filter_ping_spinner);

  /* max timeouts */

  label = gtk_label_new ("the number of retires is fewer than");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 
                                                                         0, 0);
  gtk_widget_show (label);

  adj = gtk_adjustment_new (filter_retries, 0.0, MAX_RETRIES, 1.0, 1.0, 0.0);

  filter_retries_spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 0, 0);
  gtk_widget_set_usize (filter_retries_spinner, 64, -1);
  gtk_table_attach_defaults (GTK_TABLE (table), filter_retries_spinner, 
                                                                   1, 2, 1, 2);
  gtk_widget_show (filter_retries_spinner);

  /* not full */

  filter_not_full_check_button = 
                            gtk_check_button_new_with_label ("it is not full");
  gtk_toggle_button_set_state (
            GTK_TOGGLE_BUTTON (filter_not_full_check_button), filter_not_full);
  gtk_table_attach_defaults (GTK_TABLE (table), filter_not_full_check_button, 
                                                                   0, 2, 2, 3);
  gtk_widget_show (filter_not_full_check_button);

  /* not empty */

  filter_not_empty_check_button = 
                           gtk_check_button_new_with_label ("it is not empty");
  gtk_toggle_button_set_state (
          GTK_TOGGLE_BUTTON (filter_not_empty_check_button), filter_not_empty);
  gtk_table_attach_defaults (GTK_TABLE (table), filter_not_empty_check_button, 
                                                                   0, 2, 3, 4);
  gtk_widget_show (filter_not_empty_check_button);

  /* no cheats */

  filter_no_cheats_check_button = 
                    gtk_check_button_new_with_label ("cheats are not allowed");
  gtk_toggle_button_set_state (
          GTK_TOGGLE_BUTTON (filter_no_cheats_check_button), filter_no_cheats);
  gtk_table_attach_defaults (GTK_TABLE (table), filter_no_cheats_check_button, 
                                                                   0, 2, 4, 5);
  gtk_widget_show (filter_no_cheats_check_button);

  /* no password */

  filter_no_password_check_button = 
                      gtk_check_button_new_with_label ("no password required");
  gtk_toggle_button_set_state (
      GTK_TOGGLE_BUTTON (filter_no_password_check_button), filter_no_password);
  gtk_table_attach_defaults (GTK_TABLE (table), filter_no_password_check_button, 
                                                                   0, 2, 5, 6);
  gtk_widget_show (filter_no_password_check_button);

  gtk_widget_show (table);
  gtk_widget_show (alignment);
  gtk_widget_show (frame);
  gtk_widget_show (page_vbox);

  return page_vbox;
}


static void terminate_toggled_callback (GtkWidget *widget, gpointer data) {
  int terminate;

  terminate = GTK_TOGGLE_BUTTON (terminate_check_button)->active;
  gtk_widget_set_sensitive (iconify_check_button, TRUE - terminate);
}


static GtkWidget *general_options_page (void) {
  GtkWidget *page_vbox;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *hbox;
  GtkObject *adj;

  page_vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_border_width (GTK_CONTAINER (page_vbox), 8);

  /* On Startup */

  frame = gtk_frame_new (" On Startup ");
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  /* Refresh Favorites */

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_container_border_width (GTK_CONTAINER (hbox), 6);
  gtk_container_add (GTK_CONTAINER (frame), hbox);

  auto_favorites_check_button = 
                        gtk_check_button_new_with_label ("Refresh Favorites");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (auto_favorites_check_button),
                                                       default_auto_favorites);
  gtk_box_pack_start (GTK_BOX (hbox), auto_favorites_check_button, 
                                                             FALSE, FALSE, 0);
  gtk_widget_show (auto_favorites_check_button);

  gtk_widget_show (hbox);

  gtk_widget_show (frame);

  /* On Exit */

  frame = gtk_frame_new (" On Exit ");
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  /* Save master lists */

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_container_border_width (GTK_CONTAINER (hbox), 6);
  gtk_container_add (GTK_CONTAINER (frame), hbox);

  save_lists_check_button = 
                        gtk_check_button_new_with_label ("Save server lists");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (save_lists_check_button), 
                                                           default_save_lists);
  gtk_box_pack_start (GTK_BOX (hbox), save_lists_check_button, FALSE, FALSE, 0);
  gtk_widget_show (save_lists_check_button);

  gtk_widget_show (hbox);

  gtk_widget_show (frame);

  /* On Quake Launch */

  frame = gtk_frame_new (" On Quake Launch ");
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  /* Terminate */

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_container_border_width (GTK_CONTAINER (hbox), 6);
  gtk_container_add (GTK_CONTAINER (frame), hbox);

  terminate_check_button = gtk_check_button_new_with_label ("Terminate XQF");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (terminate_check_button),
                                                            default_terminate);
  gtk_signal_connect (GTK_OBJECT (terminate_check_button), "toggled",
                           GTK_SIGNAL_FUNC (terminate_toggled_callback), NULL);
  gtk_box_pack_start (GTK_BOX (hbox), terminate_check_button, FALSE, FALSE, 0);
  gtk_widget_show (terminate_check_button);

  /* Iconify */

  iconify_check_button = 
                       gtk_check_button_new_with_label ("Iconify XQF window");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (iconify_check_button), 
                                                             default_iconify);
  gtk_box_pack_end (GTK_BOX (hbox), iconify_check_button, FALSE, FALSE, 0);
  if (default_terminate)
    gtk_widget_set_sensitive (iconify_check_button, FALSE);
  gtk_widget_show (iconify_check_button);

  gtk_widget_show (hbox);
  gtk_widget_show (frame);

  /* Lookup Host Names */

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_box_pack_start (GTK_BOX (page_vbox), hbox, FALSE, FALSE, 0);

  always_resolve_check_button = 
                        gtk_check_button_new_with_label ("Lookup Host Names");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (always_resolve_check_button),
                                                      default_always_resolve);
  gtk_box_pack_start (GTK_BOX (hbox), always_resolve_check_button, 
                                                             FALSE, FALSE, 0);
  gtk_widget_show (always_resolve_check_button);

  gtk_widget_show (hbox);

  /* QStat preferences -- maxsimultaneous & maxretries */

  frame = gtk_frame_new (" QStat ");
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_container_border_width (GTK_CONTAINER (table), 6);
  gtk_container_add (GTK_CONTAINER (frame), table);

  /* maxsimultaneous */

  label = gtk_label_new ("Number of simultaneous servers to query");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 0, 1);
  gtk_widget_show (label);

  adj = gtk_adjustment_new (maxsimultaneous, 1.0, FD_SETSIZE, 1.0, 5.0, 0.0);

  maxsimultaneous_spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 0, 0);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (maxsimultaneous_spinner),
                                                           GTK_UPDATE_ALWAYS);
  gtk_widget_set_usize (maxsimultaneous_spinner, 48, -1);
  gtk_table_attach (GTK_TABLE (table), maxsimultaneous_spinner, 1, 2, 0, 1,
                                                                  0, 0, 0, 0);
  gtk_widget_show (maxsimultaneous_spinner);

  /* maxretries */

  label = gtk_label_new ("Number of retries");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 1, 2);
  gtk_widget_show (label);

  adj = gtk_adjustment_new (maxretries, 1.0, MAX_RETRIES, 1.0, 1.0, 0.0);

  maxretries_spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 0, 0);
  gtk_widget_set_usize (maxretries_spinner, 48, -1);
  gtk_table_attach (GTK_TABLE (table), maxretries_spinner, 1, 2, 1, 2, 
                                                                  0, 0, 0, 0);
  gtk_widget_show (maxretries_spinner);

  gtk_widget_show (table);
  gtk_widget_show (frame);

  gtk_widget_show (page_vbox);

  return page_vbox;
}


void preferences_dialog (int page_num) {
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *label;
  GtkWidget *notebook;
  GtkWidget *page;
  GtkWidget *button;
  GtkWidget *window;

  set_pref_defaults ();

  window = dialog_create_modal_transient_window (
                                             "XQF: Preferences", TRUE, FALSE);
  if (!GTK_WIDGET_REALIZED (window))
    gtk_widget_realize (window);

  allocate_quake_player_colors (window->window);

  vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_border_width (GTK_CONTAINER (vbox), 8);
  gtk_container_add (GTK_CONTAINER (window), vbox);
         
  /*
   *  Notebook
   */

  notebook = gtk_notebook_new ();
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
  gtk_box_pack_start (GTK_BOX (vbox), notebook, FALSE, FALSE, 0);

  page = player_profile_page ();
  label = gtk_label_new (" Player Profile ");
  gtk_widget_show (label);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

  page = gameplay_options_page ();
  label = gtk_label_new (" Gameplay ");
  gtk_widget_show (label);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

  page = filter_options_page ();
  label = gtk_label_new (" Filtering ");
  gtk_widget_show (label);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

  page = general_options_page ();
  label = gtk_label_new (" General ");
  gtk_widget_show (label);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

  gtk_notebook_set_page (GTK_NOTEBOOK (notebook), page_num);

  gtk_widget_show (notebook);

  /* 
   *  Buttons at the bottom
   */

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  button = gtk_button_new_with_label ("Cancel");
  gtk_widget_set_usize (button, 80, -1);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                    GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (window));
  gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_show (button);

  button = gtk_button_new_with_label ("Apply");
  gtk_widget_set_usize (button, 80, -1);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		                     GTK_SIGNAL_FUNC (get_new_defaults), NULL);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                    GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (window));
  gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  gtk_widget_show (hbox);

  gtk_widget_show (vbox);

  gtk_widget_show (window);

  filter_prefs_changed = FALSE;

  gtk_main ();

  unregister_window (window);

  /* clean up */

  if (color_menu) {
    gtk_widget_destroy (color_menu);
    color_menu = NULL;
  }

  qw_skin_preview = NULL;
  q2_skin_preview = NULL;

  if (qw_skin_data) { 
    g_free (qw_skin_data);
    qw_skin_data = NULL; 
  }

  if (q2_skin_data) { 
    g_free (q2_skin_data);
    q2_skin_data = NULL; 
  }
}


void user_fix_defaults (void) {
  if (!default_qw_cmd)
    default_qw_cmd = g_strdup ("qwcl.x11");
  if (!default_q2_cmd)
    default_q2_cmd = g_strdup ("quake2");
  if (!default_name)
    default_name = g_strdup (user.name);
}


int fix_qw_player_color (int color) {
  color = color % 16;
  if (color > 13)
    color = 13;
  return color;
}

