/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <gtk/gtk.h>

#include "xqf.h"
#include "pref.h"
#include "filter.h"
#include "psearch.h"
#include "server.h"


static inline int builtin_filter (struct server *s);
static inline int builtin_filter_inv (struct server *s);
static inline int find_player_filter (struct server *s);


struct filter filters[] = {
  { "All Servers",   NULL, FALSE, NULL,			NULL },
  { "Filtered",	     NULL, FALSE, builtin_filter,	NULL },
  { "Filtered Out",  NULL, FALSE, builtin_filter_inv,	NULL },
  { "Find Player",   NULL, FALSE, find_player_filter, 	NULL },
  { NULL, 	     NULL, FALSE, NULL,			NULL }
};


struct filter 	 *cur_filter = &filters[0];


static inline int builtin_filter (struct server *s) {
  if (s->ping == -1)	/* no information */
    return TRUE;

  if (s->retries < filter_retries && 
      s->ping < filter_ping && 
      (!filter_not_full || s->curplayers != s->maxplayers) && 
      (!filter_not_empty || s->curplayers != 0) &&
      (!filter_no_cheats || (s->flags & SERVER_CHEATS) == 0) &&
      (!filter_no_password || (s->flags & SERVER_PASSWORD) == 0))
    return TRUE;

  return FALSE;
}


static inline int builtin_filter_inv (struct server *s) {
  return TRUE - builtin_filter (s);
}


static inline int find_player_filter (struct server *s) {
  int i;

  for (i = 0; i < s->curplayers; i++) {
    if (test_player (s->players[i]))
      return TRUE;
  }
  return FALSE;
}


void build_filtered_list (struct filter *filter, struct master *master) {
  GSList *l = NULL;
  GSList *nbl = NULL;
  struct filter *nb = NULL;	/* "neighbour" filter */
  GSList *ptr;
  struct server *s;

  if (!filter || !master)
    return;

  if (!filter->func) {
    for (ptr = master->servers; ptr; ptr = ptr->next) {
      s = (struct server *) ptr->data;
      filter->servers = g_slist_prepend (filter->servers, s);
      s->ref_count++;
    }
    filter->servers = g_slist_reverse (filter->servers);
    filter->built = TRUE;
  }	
  else {
    if (filter == &filters[1] || filter == &filters[2]) {
      nb = &filters[&filters[2] - filter + 1];
    }

    for (ptr = master->servers; ptr; ptr = ptr->next) {
      s = (struct server *) ptr->data;
      if ((*filter->func) ((struct server *)ptr->data)) {
	l = g_slist_prepend (l, s);
	s->ref_count++;
      }
      else {
	if (nb) {
	  nbl = g_slist_prepend (nbl, s);
	  s->ref_count++;
	}
      }
    }

    l = g_slist_reverse (l);
    filter->servers = l;
    filter->built = TRUE;

    if (nb) {
      nbl = g_slist_reverse (nbl);
      nb->servers = nbl;
      nb->built = TRUE;
    }
  }
}


int filter_add_server (struct filter *f, struct server *s) {
  struct filter *nb = NULL;	/* "neighbour" filter */

  if (!f || !s)
    return FALSE;

  if (!f->built) {	/* should not happen */
#ifdef DEBUG
    fprintf (stderr, "filter_add_server(): BUG! filter %s is not built\n", 
                                                                     f->name);
#endif
    return FALSE;
  }

  if (g_slist_find (f->servers, s))
    return TRUE;

  if (!f->func) {
    f->servers = g_slist_append (f->servers, s);
    s->ref_count++;
    return TRUE;
  }
  else {
    if (f == &filters[1] || f == &filters[2]) {
      nb = &filters[&filters[2] - f + 1];
    }

    if ((*f->func) (s)) {
      f->servers = g_slist_append (f->servers, s);
      s->ref_count++;
      return TRUE;
    }
    else {
      if (nb) {
	if (!g_slist_find (nb->servers, s)) {
	  nb->servers = g_slist_append (nb->servers, s);
	  s->ref_count++;
	}
      }
    }
  }

  return FALSE;
}


void free_filter_data (struct filter *f) {
  if (f->built) {
    free_servers (f->servers);
    f->servers = NULL;
    f->built = FALSE;
  }
}


void free_filters_data (void) {
  struct filter *f;

  for (f = filters; f->name; f++)
    free_filter_data (f);
}

