/*
    SABRE Fighter Plane Simulator 
    Copyright (c) 1997 Dan Hammer
    Portions Donated By Antti Barck

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*************************************************
 *           SABRE Fighter Plane Simulator              *
 * Version: 0.1                                  *
 * File   : txtrmap.C                            *
 * Date   : March, 1997                          *
 * Author : Dan Hammer                           *
 *          08/20/97                             *
 *          Antti Barck                          *
 *          Use unsigned int when read/writing   *
 *          compressed files to make compatible  *
 *          with 64bit machines                  *
 *************************************************/
#include <stdio.h>
#include <iostream.h>
#include <fstream.h>
#include <math.h>
#include <limits.h>
#include <values.h>
#include <string.h>
#include "defs.h"
#include "grafix.h"
#include "vga_13.h"
#include "vmath.h"
#include "simerr.h"
#include "simfile.h"
#include "colormap.h"
#include "txtrmap.h"
#include "comp.h"
#include "unc.h"

TextrMap nullmap;
TextrMap_Manager *map_man = NULL;
int TextrMap_Manager::reserved = 64;

void TextrMap::copy(TextrMap &tm)
{
  bytes = tm.bytes;
  size = tm.size;
  cbytes = tm.cbytes;
  csize = tm.csize;
  map_w = tm.map_w;
  map_h = tm.map_h;
  trans_colr = tm.trans_colr;
  delf = 1;
  tm.delf = 0;
  flags = tm.flags;
  memcpy(id,tm.id,sizeof(id));
  if (tm.cmaps != NULL)
    {
      cmaps = new ColorMap[N_COLORS];
      MYCHECK(cmaps != NULL);
      for (int i=0;i<N_COLORS;i++)
	cmaps[i] = tm.cmaps[i];
    }
  if (tm.mc != NULL && tm.nm > 0)
    {
      nm = tm.nm;
      mc = new int[nm];
      MYCHECK(mc != NULL);
      for (int i=0;i<nm;i++)
	mc[i] = tm.mc[i];
    }
}

void TextrMap::read(istream &is)
{
  char *cptr;
  int ii;
  char c = ' ';
  memset(id,' ',sizeof(id));
  READ_TOK('{',is,c)
    is >> id;
  cptr = id + strlen(id) - 1;
  c = *cptr;
  if (c == '*' || c == '@')
    {
      is >> trans_colr;
      *cptr = '\0';
      if (c == '@')
	read_compressed(0);
      else
	read_compressed(1);
    }
  else
    {
      is >>  map_w >> map_h >> trans_colr;
      id[31] = 0;
      int nbytes = map_w * map_h;
      if (bytes != NULL)
	delete [] bytes;
      bytes = new unsigned char[nbytes];
      MYCHECK(bytes != NULL);
      delf = 1;
      for (int i=0;i<nbytes;i++)
	{
	  is >> ii;
	  bytes[i] = (unsigned char) ii;
	}
    }

  is >> c;
  if (c == '*')
    {
      is >> nm;
      cmaps = new ColorMap[N_COLORS];
      MYCHECK(cmaps != NULL);
      mc = new int[nm];
      MYCHECK(mc != NULL);
      ColorMap tmp;
      for (int i=0;i<nm;i++)
	{
	  is >> tmp;
	  mc[i] = tmp.id_color;
	  cmaps[tmp.id_color] = tmp;
	}
    }
  READ_TOK('}',is,c);

}

void TextrMap::write(ostream &os)
{
  os << "{\n" << id << "*\n" << trans_colr << "\n";
  compress();
  os << "}\n";
}

istream & operator >>(istream &is, TextrMap &tm)
{
  tm.read(is);
  return (is);
}

ostream & operator <<(ostream &os, TextrMap &tm)
{
  tm.write(os);
  return (os);
}

void TextrMap::compress()
{
  unsigned int  n;
  unsigned long tgtsize;
  unsigned long srcsize;
  char     *tgt;
  char     *fname;
  ofstream os;

  if (bytes != NULL)
    {
      tgtsize = (unsigned long) map_w * map_h;
      srcsize = tgtsize;
      tgt = new char[tgtsize];
      if ((n = memcompress(tgt,tgtsize,(char *)bytes,srcsize)))
	{
	  fname = new char[strlen(id) + 10];
#ifdef SABREWIN
	  sprintf(fname,"tzp\\%s.tzp",id);
#else
	  sprintf(fname,"tzp/%s.tzp",id);
#endif
	  if (open_libos(os,fname))
	    {
	      os.write(&map_w,sizeof(map_w));
	      os.write(&map_h,sizeof(map_h));
	      os.write(&n,sizeof(n));
	      os.write(tgt,n);
	    }
	  delete [] fname;
	}
      else
	{
	  printf("compression error: %d\n",n);
	}
      delete [] tgt;
    }
}

#ifdef SABREWIN
void TextrMap::read_compressed(int cflg)
{
  unsigned int      n;
  char              *fname;
  FILE              *f;
  int               nread;
  char              *path;
  fname = new char[strlen(id) + 10];
  sprintf(fname,"tzp\\%s.tzp",id);
  path = build_libpath(fname);
   if ((f = fopen(path,"rb")) != NULL)
    {
      fread(&map_w,sizeof(map_w),1,f);
      fread(&map_h,sizeof(map_h),1,f);
      fread(&n,sizeof(n),1,f);
      size = map_w * map_h;
      csize = n;
      cbytes = new unsigned char[csize];
      MYCHECK(cbytes != NULL);
      nread = fread(cbytes,1,csize,f);
      if (ferror(f))
   	{
	     error_jump("TextrMap: error reading from file %s %d bytes\n",
		    fname,nread);
	   delete [] cbytes;
	    cbytes = NULL;
	   return;
	   }
      fclose(f);
      if (bytes != NULL)
	{
	  delete [] bytes;
	  bytes = NULL;
	}
      if (!cflg)
	getBytes(0);
    }
}
#else
void TextrMap::read_compressed(int cflg)
{
  unsigned int      n;
  char              *fname;
  ifstream          is;


  fname = new char[strlen(id) + 10];
  sprintf(fname,"tzp/%s.tzp",id);
  if (open_is(is,fname))
    {
      is.read(&map_w,sizeof(map_w));
      is.read(&map_h,sizeof(map_h));
      is.read(&n,sizeof(n));
      size = map_w * map_h;
      csize = n;
      cbytes = new unsigned char[csize];
      MYCHECK(cbytes != NULL);
      is.read(cbytes,csize);
      if (!is.good())
	{
	  printf("TextrMap: error reading from file %s\n",
		 fname);
	  delete [] cbytes;
	  cbytes = NULL;
	  return;
	}
      is.close();
      if (bytes != NULL)
	{
	  delete [] bytes;
	  bytes = NULL;
	}
      if (!cflg)
	getBytes(0);
    }
}
#endif

unsigned char *TextrMap::getBytes(int cflg)
{
  long r;

  if (bytes != NULL)
    return (bytes);
  if (cbytes == NULL)
    {
      printf("TextrMap:: cbytes are NULL\n");
      return (NULL);
    }
  bytes = new unsigned char[size];
  MYCHECK(bytes != NULL);
  r = unc_mem_to_mem(bytes,size,cbytes,csize);
  if (r <= 0)
    {
      printf("TextrMap %s: decompression error\n",
      	     id);
      delete [] bytes;
      bytes = NULL;
    }
  else if (!cflg)
    {
      delete [] cbytes;
      cbytes = NULL;
    }
  return (bytes);
}

void TextrMap::clearBytes()
{
  if (bytes != NULL)
    {
      delete [] bytes;
      bytes = NULL;
    }
}

void TextrMap::setColorMaps(float r)
{
  if (cmaps != NULL)
    for (int i=0;i<nm;i++)
      cmaps[mc[i]].calcMappedColor(r);
}

void TextrMap::setColorMaps(int base, int range, int col)
{
  if (cmaps != NULL)
    for (int i=0;i<nm;i++)
      cmaps[mc[i]].calcMappedColor(base,range,col);
}

TextrMap_Manager::TextrMap_Manager(int n)
{
  n_maps = n;
  tmaps = new TextrMap[n_maps];
  MYCHECK(tmaps != NULL);
  nxt = 0;
}

int TextrMap_Manager::add_map(TextrMap &tm)
{
  if (nxt < n_maps - 1)
    {
      tmaps[nxt++] = tm;
      return 1;
    }
  else
    return 0;
}


void TextrMap_Manager::read(istream &is)
{
  int n;
  if (tmaps == NULL)
    {
      is >> n_maps;
      if (n_maps <= 0)
	return;
      tmaps = new TextrMap[n_maps];
      MYCHECK(tmaps != NULL);
      for (int i=0;i<n_maps;i++)
	is >> tmaps[i];
      nxt = n_maps - 1;
    }
  else
    {
      is >> n;
      for (int i=0;i<n;i++)
	{
	  TextrMap tmp;
	  if (nxt < n_maps - 1)
	    {
	      is >> tmaps[nxt];
	      nxt++;
	    }
	  else
	    is >> tmp;
	}
    }
}

istream &operator >>(istream &is, TextrMap_Manager &tm)
{
  tm.read(is);
  return (is);
}

void TextrMap_Manager::read_file(char *path)
{
  ifstream is;
  if (open_is(is,path))
    is >> *this;
}

void TextrMap_Manager::write(ostream &os)
{
  os << nxt << '\n';
  for (int i=0;i<nxt;i++)
    os << tmaps[i] << '\n';
}


ostream &operator <<(ostream &os, TextrMap_Manager &tm)
{
  tm.write(os);
  return (os);
}

void TextrMap_Manager::write_file(char *path)
{
  ofstream os;
  if (open_libos(os,path))
    write(os);
}


TextrMap *TextrMap_Manager::get_map_ptr(char *id)
{
  TextrMap *result = NULL;
  for (int i=0;i<n_maps;i++)
    {
      if (!strcmp(id,tmaps[i].id))
	{
	  result = &tmaps[i];
	  break;
	}
    }
  return (result);
}




