/*-------------------------------------------------------------------------*/
/* xcircuit.c --- An X-windows program for drawing circuit diagrams	   */
/* Copyright (c) 1998  Tim Edwards, Johns Hopkins University        	   */
/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*      Uses the Xw widget set 						   */
/*      written by Tim Edwards, 8/13/93    				   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <limits.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#include "Xw/Xw.h"
#include "Xw/Form.h"
#include "Xw/WorkSpace.h"
#include "Xw/PButton.h"
#include "Xw/SText.h"
#include "Xw/Cascade.h"
#include "Xw/PopupMgr.h"
#include "Xw/MenuBtn.h"
#include "Xw/BBoard.h"
#include "Xw/TextEdit.h"

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "cursors.h"
#include "colordefs.h"
#include "xcircuit.h"
#include "menudefs.h"
#include "menudep.h"

/*-------------------------------------------------------------------------*/
/* Local defines							   */
/*-------------------------------------------------------------------------*/

#define STIPPLES     8

#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "xcircuit.xpm"
#endif

/*-------------------------------------------------------------------------*/
/* Global Variable definitions						   */
/*-------------------------------------------------------------------------*/

char	 _STR2[250];  /* Specifically for text returned from the popup prompt */
char	 _STR[150];          /* Generic multipurpose string */
Widget	 message1, message2, message3, top;
Display  *dpy;	             /* Works well to make this globally accessible */
Window   win;
Colormap cmap;
short	 popups;	             /* total number of popup widgets on the screen */
Pixmap   STIPPLE[STIPPLES];  /* Polygon fill-style stipple patterns */


static char STIPDATA[STIPPLES][4] = {
   "\000\004\000\001",
   "\000\005\000\012",
   "\001\012\005\010",
   "\005\012\005\012",
   "\016\005\012\007",
   "\017\012\017\005",
   "\017\012\017\016",
   "\000\000\000\000"
};

Cursor	appcursors[NUM_CURSORS];
ApplicationData appdata;
Clientdata areastruct;
object	areaobject;
int *appcolors;
int number_colors;
colorindex *colorlist;

XtIntervalId printtime_id;
short beeper;

/* if position of Symbol-Oblique changes, revise files.c #define */

short fontcount;
char **fonts;
char *prefonts[FONTS] = {
 "Times-Roman", "Times-Italic",      "Times-Bold",     "Times-BoldItalic",
 "Helvetica",   "Helvetica-Oblique", "Helvetica-Bold", "Helvetica-BoldOblique",
 "Courier",     "Courier-Oblique",   "Courier-Bold",   "Courier-BoldOblique",
 "Symbol",      "Symbol-Oblique"
};

extern short eventmode, textpos, pushes, help_up;
extern float version;
extern menustruct TopButtons[];
extern short maxbuttons;

#ifdef DOUBLEBUFFER
extern Pixmap dbuf;
#endif

/* Bad hack for problems with the DECstation. . . don't know why */
#ifdef UniqueContextProblem
#undef XUniqueContext
XContext XUniqueContext()
{
   return XrmUniqueQuark();
}
#endif
/* End of bad hack. . . */

/*------------------------------------------------------------------------*/
/* Declarations of functions defined externally to xcircuit.c		  */
/*------------------------------------------------------------------------*/

extern void selectbutton(), releasebutton(), keyhandler(),
	loadlibrary(), Wprintf(), setfloat(), setscalex(), setscaley(),
	calcbbox(), setint(), setfilename(), setlabel(), undrawtext(),
	setpagelabel(), genfilelist(), newfilelist();
extern void redrawtext(), setpmode(), setorient(), getgeneric(), setdscale(),
	W2printf(), initsplines(), setpagesize();
extern void startloadfile(), importfile(), loadblib(), loadulib(), loadlgf();
extern float getpsscale();
extern int xc_alloccolor(), lookdirectory();
extern XtCallbackProc resizearea(), drawarea();
extern XtCallbackProc setfile(), drawhbar(), drawvbar();

void renamepage(), W1printf();
void newfilelist(), makecursors();

XtEventHandler panhbar(), panvbar();
XtCallbackProc setcolor(), updatename(), updatetext();

/*-------------------------------------------------------------------------*/
/* Initial Resource Management						   */
/*-------------------------------------------------------------------------*/

static XtResource resources[] = {

  /* Color scheme 1 */

  { XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, fg), XtRString, "Black"},
  { XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, bg), XtRString, "White"},
  { "gridcolor", "GridColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, gridpix), XtRString, "Gray95"},
  { "snapcolor", "SnapColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, snappix), XtRString, "Red"},
  { "selectcolor", "SelectColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, selectpix), XtRString, "Gold3"},
  { "querycolor", "QueryColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, querypix), XtRString, "Turquoise"},
  { "axescolor", "AxesColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, axespix), XtRString, "Antique White"},
  { "offbuttoncolor", "OffButtonColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, buttonpix), XtRString, "Gray70"},
  { "auxiliarycolor", "AuxiliaryColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, auxpix), XtRString, "Green3"},
  { "barcolor", "BarColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, barpix), XtRString, "Tan"},

  /* Color scheme 2 */

  { "foreground2", XtCForeground, XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, fg2), XtRString, "White"},
  { "background2", XtCBackground, XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, bg2), XtRString, "DarkSlateGray"},
  { "gridcolor2", "GridColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, gridpix2), XtRString, "Gray40"},
  { "snapcolor2", "SnapColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, snappix2), XtRString, "Red"},
  { "selectcolor2", "SelectColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, selectpix2), XtRString, "Gold"},
  { "querycolor2", "QueryColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, querypix2), XtRString, "Turquoise"},
  { "axescolor2", "AxesColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, axespix2), XtRString, "NavajoWhite4"},
  { "offbuttoncolor2", "OffButtonColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, buttonpix2), XtRString, "Gray70"},
  { "auxiliarycolor2", "AuxiliaryColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, auxpix2), XtRString, "Green"},
  { "barcolor2", "BarColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, barpix2), XtRString, "Tan"},

  /* Other XDefaults-set properties */

  { XtNfont, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
      XtOffset(ApplicationDataPtr, xcfont), XtRString, "Fixed"},
  { XtNwidth, XtCWidth, XtRInt, sizeof(int),
      XtOffset(ApplicationDataPtr, width), XtRString, "800"},
  { XtNheight, XtCHeight, XtRInt, sizeof(int),
      XtOffset(ApplicationDataPtr, height), XtRString, "600"}
};

/*-------------------------------------------------------------------------*/
/* Update the list of colors in the colormap				   */
/*-------------------------------------------------------------------------*/

addtocolorlist(button, cvalue)
  Widget button;
  int    cvalue;
{
   number_colors++;
   colorlist = (colorindex *)realloc(colorlist, number_colors *
		sizeof(colorindex));
   colorlist[number_colors - 1].cbutton = button;
   colorlist[number_colors - 1].color.pixel = cvalue; 

   /* Get and store the RGB values of the new color */
   XQueryColor(dpy, cmap, &(colorlist[number_colors - 1].color));
}

/*-------------------------------------------------------------------------*/
/* Add a new color button to the color menu				   */
/*-------------------------------------------------------------------------*/

void addnewcolorentry(ccolor)	/* called if new color button needs to be made */
   int ccolor;
{
   Widget colormenu, newbutton;
   Arg wargs[2];
   int i, n = 0;

   /* check to see if entry is already in the color list */

   for (i = 0; i < number_colors; i++)
      if (colorlist[i].color.pixel == ccolor) break;

   /* make new entry in the menu */

   if (i == number_colors) {
      colormenu = XtNameToWidget(top, ColorsPath);
      XtnSetArg(XtNlabelType, XwRECT);
      XtnSetArg(XtNrectColor, ccolor);

      newbutton = XtCreateWidget("NewColor", XwmenubuttonWidgetClass,
         colormenu, wargs, n);
      XtAddCallback (newbutton, XtNselect, (XtCallbackProc)setcolor, NULL);
      XtManageChild(newbutton);

      addtocolorlist(newbutton, ccolor);
   }
}

/*-------------------------------------------------------------------------*/
/* This recursive function looks down the button menu hierarchy and        */
/*   creates the necessary buttons and submenus.			   */
/*   Menu entries are marked if the corresponding "size" entry in the	   */
/*   menu structure is > 0.						   */
/*-------------------------------------------------------------------------*/

void makesubmenu(menuname, attachname, buttonmenu, arraysize, manager)
char *menuname, *attachname;
menuptr buttonmenu;
int arraysize;
Widget manager;
{
   short i, n;
   int cval;
   WidgetList newbuttons;
   Widget popupshell, cascade;
   Arg	wargs[6];
   menuptr p;
   char popupname[30];

   newbuttons = (WidgetList) XtMalloc (arraysize * sizeof (Widget));
   sprintf(popupname, "popup%s", menuname);
   popupshell = XtCreatePopupShell (popupname, transientShellWidgetClass,
	manager,  NULL, 0);

   XtSetArg (wargs[0], XtNattachTo, attachname);
   cascade = XtCreateManagedWidget (menuname, XwcascadeWidgetClass,
	popupshell, wargs, 1);
   
   for (p = buttonmenu, i = 0; p < buttonmenu + arraysize; p++, i++) {
      n = 0;
      if (p->size > 0 && p->submenu == NULL) { /* This denotes a marked entry */
	 XtnSetArg(XtNsetMark, True);
      }
      XtnSetArg(XtNfont, areastruct.textstruct);
      if (p->name[0] == '_') {  /* Color button */
	 cval = xc_alloccolor(p->name + 1);
	 XtnSetArg(XtNlabelType, XwRECT);
	 XtnSetArg(XtNrectColor, cval);
      }
      else if (p->name[0] == ':') {  /* Stipple button */
	 XtnSetArg(XtNlabelType, XwRECT);
         if (((int)(p->passeddata) == (FILLED | FILLSOLID))) {
	    XtnSetArg(XtNrectColor, BlackPixel(dpy,DefaultScreen(dpy)));
	 }
	 else {
	    XtnSetArg(XtNrectStipple, STIPPLE[((int)(p->passeddata) &
		  FILLSOLID) >> 5]);
	 }
      }
	 
      newbuttons[i] = XtCreateWidget(p->name, XwmenubuttonWidgetClass,
	   cascade, wargs, n);
      if (p->submenu != NULL)
	 makesubmenu(p->name, p->name, p->submenu, p->size, manager);
      else 
	 XtAddCallback (newbuttons[i], XtNselect, (XtCallbackProc)p->func,
		p->passeddata);

      /* For color buttons, maintain a list of Widgets and color values */

      if (p->name[0] == '_') addtocolorlist(newbuttons[i], cval);
   }
   XtManageChildren (newbuttons, arraysize);
}

/*-------------------------------------------------------------------------*/
/* Hierarchical Menu Creator						   */
/*   This function creates the top level of buttons which are arranged     */
/*   across the top starting at the left edge.  For each button 	   */
/*   that has a submenu, a Popup manager is created, and then menu	   */
/*   panes are attached to the manager in a hierarchical fashion.	   */
/*   Note: Returns widget for last button on top level			   */
/*-------------------------------------------------------------------------*/

Widget createmenus (Widget form)
{
   int i, maxmgrs = 0, n = 0, j = 0;
   WidgetList buttonw, mgr_shell, menu_mgr;
   Arg	wargs[5];

   buttonw = (WidgetList) XtMalloc(maxbuttons * sizeof(Widget));

   for (i = 0; i < maxbuttons; i++) 
      if (TopButtons[i].submenu != NULL) maxmgrs++;

   mgr_shell = (WidgetList) XtMalloc(maxmgrs * sizeof(Widget));
   menu_mgr = (WidgetList) XtMalloc(maxmgrs * sizeof(Widget));

   for (i = 0; i < maxbuttons; i++) {

#ifdef SCHEMA
      if (!strcmp(TopButtons[i].name, "Netlist")
	    	&& !areastruct.schemon) {
	  maxbuttons--;
	  if (i == maxbuttons) break;
	  else continue;
      }
#endif
      XtnSetArg(XtNheight, ROWHEIGHT);
      XtnSetArg(XtNlabel, TopButtons[i].name);
      XtnSetArg(XtNfont, areastruct.textstruct);
      if (i > 0) {
	 XtnSetArg(XtNxRefWidget, buttonw[i - 1]);
	 XtnSetArg(XtNxAddWidth, True);
      }
      buttonw[i] = XtCreateManagedWidget(TopButtons[i].name, 
	 XwmenuButtonWidgetClass, form, wargs, n); n = 0;

      if(TopButtons[i].submenu == NULL) 
	 XtAddCallback(buttonw[i], XtNselect,
		(XtCallbackProc)TopButtons[i].func, NULL);
      else {
         mgr_shell[j] = XtCreatePopupShell("mgr_shell", shellWidgetClass,
	    buttonw[i], NULL, 0);
         menu_mgr[j] = XtCreateManagedWidget("menu_mgr", XwpopupmgrWidgetClass,
	    mgr_shell[j], NULL, 0);
	 makesubmenu(TopButtons[i].name, "menu_mgr", TopButtons[i].submenu, 
	    TopButtons[i].size, menu_mgr[j]);
	 j++;
      }
   }
   return buttonw[i - 1];
}

/*---------------*/
/* Quit xcircuit */
/*---------------*/

XtCallbackProc quit(w, clientdata, calldata)
  Widget w;
  caddr_t clientdata, calldata;
{
   /* free up CTM Matrix Stack */
   free(areastruct.MatStack);

   /* free the colormap if a new one has been installed */

   if (cmap != DefaultColormap(dpy, DefaultScreen(dpy))) {
      XFreeColormap(dpy, cmap);
   }

   exit(0);
}

/*--------------*/
/* Null routine */
/*--------------*/

XtCallbackProc DoNothing(w, clientdata, calldata)
  Widget w;
  caddr_t clientdata, calldata;
{
   /* Nothing here! */
}

/*-------------------------------------------------------------------------*/
/* Popup dialog box routines						   */
/*-------------------------------------------------------------------------*/
/* Propogate any key event from the dialog box into the textedit widget    */
/*-------------------------------------------------------------------------*/

XtEventHandler propevent(w, editwidget, event)
  Widget w, editwidget;
  XEvent *event;
{
   Window ewin = XtWindow(editwidget);

   event->xany.window = ewin;
   XSendEvent(dpy, ewin, False, KeyPressMask, event);
}

/*-------------------------------------------------------------------------*/
/* Destroy an interactive text-editing popup box 			   */
/*-------------------------------------------------------------------------*/

XtCallbackProc destroypopup(button, callstruct, calldata)
  Widget button;
  popupstruct *callstruct;
  caddr_t calldata;
{
   Arg	wargs[1];

   if (eventmode == NORMAL_MODE) eventmode = POPUP_MODE;

   if(XtNameToWidget(callstruct->popup, "help2") != NULL) help_up = False;

   if (callstruct->buttonptr->button != NULL) {  

      /* return the button to its normal state */

      XtSetArg(wargs[0], XtNforeground, callstruct->buttonptr->foreground);
      XtSetValues(callstruct->buttonptr->button, wargs, 1);
   
      XtAddCallback(callstruct->buttonptr->button, XtNselect, 
	  (XtCallbackProc)callstruct->buttonptr->buttoncall,
	  callstruct->buttonptr->dataptr);
   }

   XtDestroyWidget(callstruct->popup);
   popups--;

   /* free the allocated structure space */

   free(callstruct->buttonptr);
   free(callstruct);
}

/*-------------------------------------------------------------------------*/
/* Pull text from the popup prompt buffer into a global string variable    */
/*-------------------------------------------------------------------------*/

XtCallbackProc gettext(button, callstruct, calldata)
  Widget 	button;
  popupstruct	*callstruct;
  caddr_t 	calldata;
{
   void *function = callstruct->setvalue;

   sprintf(_STR2, "%.249s", XwTextCopyBuffer(callstruct->textw)); 

   /* functions which use the file selector should look for directory name */
   /* in string rather than a file name, and take appropriate action.	   */

   if (callstruct->fileprompt == True) {
      if (lookdirectory(_STR2)) {
         newfilelist(XtNameToWidget(XtParent(button), "Filelist"),
               callstruct->textw);
	 return;
      }
   }

   /* call the function which sets the variable according to type */
   /* This is in format (function)(calling-widget, ptr-to-data)   */

   (*(callstruct->setvalue))(callstruct->buttonptr->button,
	 callstruct->buttonptr->dataptr);

   if (callstruct->fileprompt == True)
      newfilelist(XtNameToWidget(XtParent(button), "Filelist"), callstruct->textw);
   destroypopup(button, callstruct, calldata);
}

/*-------------------------------------------------------------------------*/
/* Grab text from the "output properties" window			   */
/*-------------------------------------------------------------------------*/

XtCallbackProc getproptext(button, callstruct, calldata)
  Widget 	button;
  propstruct	*callstruct;
  caddr_t 	calldata;
{
   /* areastruct.filename[areastruct.page] can be realloc'd by the call */
   /* to *(callstruct->setvalue), so callstruct->dataptr may no longer  */
   /* be pointing to the data.						*/

   Arg wargs[1];
   short fileyes = (callstruct->setvalue == setfilename);

   sprintf(_STR2, "%.249s", XwTextCopyBuffer(callstruct->textw)); 
   (*(callstruct->setvalue))(button, callstruct->dataptr);

   /* special stuff for filename changes */

   if (fileyes) {
      char blabel[50];
      Widget wrbutton;
      struct stat statbuf;

      /* get updated file information */

      if (strstr(areastruct.filename[areastruct.page], ".") == NULL)
         sprintf(blabel, "%s.ps", areastruct.filename[areastruct.page]);
      else sprintf(blabel, "%s", areastruct.filename[areastruct.page]);
      if (stat(blabel, &statbuf) == 0) {
         sprintf(blabel, " Overwrite File ");
         if (beeper) XBell(dpy, 100);
         Wprintf("    Warning:  File exists");
      }
      else {
         sprintf(blabel, " Write File ");
         if (errno == ENOTDIR)
            Wprintf("Error:  Incorrect pathname");
         else if (errno == EACCES)
            Wprintf("Error:  Path not readable");
	 Wprintf("  ");
      }

      wrbutton = XtNameToWidget(XtParent(button), "Write File");
      XtSetArg(wargs[0], XtNlabel, blabel);
      XtSetValues(wrbutton, wargs, 1);
   }

   /* objectdata->name is not malloc'd, so is not changed by call to */
   /* *(callstruct->setvalue).					   */

   else if (callstruct->dataptr == objectdata->name) {
      printname(objectdata);
      renamepage(areastruct.page);
   }

   /* Button title changes from "Apply" to "Okay" */

   XtSetArg(wargs[0], XtNlabel, "Okay");
   XtSetValues(callstruct->buttonw, wargs, 1);
}

/*-------------------------------------------------------------------------*/
/* Update scale, width, and height in response to change of one of them	   */
/*-------------------------------------------------------------------------*/

XtCallbackProc updatetext(button, callstruct, calldata)
  Widget	button;
  WidgetList	callstruct;
  caddr_t	calldata;
{
   float oscale, psscale;
   char edit[3][50];
   short i, n, posit;
   char  *pdptr;
   Arg	 wargs[2];

   oscale = areastruct.outscale[areastruct.page];
   psscale = getpsscale(oscale, areastruct.page);

   sprintf(edit[0], "%6.5f", oscale);
   sprintf(edit[1], "%6.5f", (objectdata->width * psscale) / 72);
   sprintf(edit[2], "%6.5f", (objectdata->height * psscale) / 72);

   for (i = 0; i < 3; i++) {
      n = 0;
      XtnSetArg(XtNstring, edit[i]);
      pdptr = strchr(edit[i], '.');
      posit = (pdptr != NULL) ? (short)(pdptr - edit[i]) : strlen(edit[i]);
      XtnSetArg(XtNinsertPosition, posit);
      XtSetValues(callstruct[i + 2], wargs, n);
   }
}

/*-------------------------------------------------------------------------*/
/* Update the object name in response to a change in filename		   */
/*-------------------------------------------------------------------------*/

XtCallbackProc updatename(button, callstruct, calldata)
  Widget        button;
  WidgetList    callstruct;
  caddr_t       calldata;
{
   short i, n, posit;
   char  *rootptr;
   Arg   wargs[2]; 
      
   if (strstr(objectdata->name, "Page ") != NULL || strstr(objectdata->name,
	"Page_") != NULL || objectdata->name[0] == '\0') {

      rootptr = strrchr(areastruct.filename[areastruct.page], '/');
      if (rootptr == NULL) rootptr = areastruct.filename[areastruct.page];
      else rootptr++;

      sprintf(objectdata->name, "%.79s", rootptr);
  
      n = 0;
      posit = strlen(objectdata->name);
      XtnSetArg(XtNstring, objectdata->name);
      XtnSetArg(XtNinsertPosition, posit);
      XtSetValues(callstruct[1], wargs, n);
      printname(objectdata);
      renamepage(areastruct.page);
   }
}

/*-------------------------------------------------------------------------*/
/* Create a popup window with "OK" and "Cancel" buttons,		   */
/* and text and label fields.						   */
/*-------------------------------------------------------------------------*/

void popupprompt(button, request, current, function, datastruct, fileprompt)
Widget  button;
char *request, *current;
void (*function)();
buttonsave *datastruct;
Boolean fileprompt;
{
    Arg         wargs[8];
    Widget      popup, dialog, okbutton, cancelbutton, entertext, tmpwidget;
    Widget	staticarea;
    XWMHints	*wmhints;	/* for proper input focus */
    Position    xpos, ypos;
    short	n = 0;
    Dimension	height, width, areawidth, areaheight, bwidth, owidth;
    static char defaultTranslations[] = "<Key>Return:	execute()";
    XGCValues	promptvalues;
    popupstruct	*okaystruct;

    height = (current == NULL) ? ROWHEIGHT * 4 : ROWHEIGHT * 5;
    if (fileprompt) height += LISTHEIGHT;

    width = XTextWidth(areastruct.textstruct, request, strlen(request)) + 20;
    bwidth = XTextWidth(areastruct.textstruct, "Cancel", strlen("Cancel")) + 50;
    owidth = XTextWidth(areastruct.textstruct, "Okay", strlen("Okay")) + 50;
    if (width < 400) width = 400;

    XtnSetArg(XtNwidth, &areawidth);
    XtnSetArg(XtNheight, &areaheight);
    XtGetValues(areastruct.area, wargs, n); n = 0;
    XtTranslateCoords(areastruct.area, (Position) (areawidth / 2 - width 
	/ 2 + popups * 20), (Position) (areaheight / 2 - height / 2 +
	popups * 20), &xpos, &ypos);
    XtnSetArg(XtNx, xpos);
    XtnSetArg(XtNy, ypos);
    popup = XtCreatePopupShell("prompt", transientShellWidgetClass,
        button, wargs, n); n = 0;
    popups++;

    XtnSetArg(XtNlayout, XwIGNORE);
    XtnSetArg(XtNwidth, width);
    XtnSetArg(XtNheight, height);
    dialog = XtCreateManagedWidget("dialog", XwbulletinWidgetClass,
        popup, wargs, n); n = 0;

    XtnSetArg(XtNx, 20);
    XtnSetArg(XtNy, ROWHEIGHT - 10 + (fileprompt ? LISTHEIGHT : 0));
    XtnSetArg(XtNstring, request);
    XtnSetArg(XtNborderWidth, 0);
    XtnSetArg(XtNgravity, WestGravity);
    staticarea = XtCreateManagedWidget("static", XwstaticTextWidgetClass,
	dialog, wargs, n); n = 0;

    XtnSetArg(XtNx, 20);
    XtnSetArg(XtNy, height - ROWHEIGHT - 10);
    XtnSetArg(XtNwidth, owidth); 
    XtnSetArg(XtNfont, areastruct.textstruct);
    okbutton = XtCreateManagedWidget("Okay", XwmenuButtonWidgetClass, 
	dialog, wargs, n); n = 0;

    okaystruct = (popupstruct *) malloc(sizeof(popupstruct));
    okaystruct->buttonptr = datastruct;
    okaystruct->popup = popup;
    okaystruct->fileprompt = fileprompt;

    if (current != NULL) {   /* A Text Edit widget is required */
       char		*pdptr;
       short		posit;

       XtnSetArg(XtNx, width - bwidth - 20);
       XtnSetArg(XtNy, height - ROWHEIGHT - 10);
       XtnSetArg(XtNwidth, bwidth);
       XtnSetArg(XtNfont, areastruct.textstruct);
       cancelbutton = XtCreateManagedWidget("Cancel", XwmenuButtonWidgetClass, 
	   dialog, wargs, n); n = 0; 

       XtnSetArg(XtNx, 20);
       XtnSetArg(XtNy, ROWHEIGHT + 10 + (fileprompt ? LISTHEIGHT : 0));
       XtnSetArg(XtNheight, ROWHEIGHT + 5);
       XtnSetArg(XtNstring, current);
       pdptr = strchr(current, '.');
       posit = (pdptr != NULL) ? (short)(pdptr - current) : strlen(current);
       XtnSetArg(XtNinsertPosition, posit);
       XtnSetArg(XtNgrow, XwGrowHorizontal);
       entertext = XtCreateManagedWidget("Edit", XwtextEditWidgetClass,
	   dialog, wargs, n); n = 0; 

       okaystruct->textw = entertext;
       okaystruct->setvalue = function;

       XtAddCallback(okbutton, XtNselect, (XtCallbackProc)gettext, okaystruct);
       XtAddEventHandler(dialog, KeyPressMask, False,
	  (XtEventHandler)propevent, entertext);
       XtAddEventHandler(staticarea, KeyPressMask, False,
	  (XtEventHandler)propevent, entertext);
       XtOverrideTranslations(entertext, XtParseTranslationTable
	  (defaultTranslations));
       XtAddCallback(entertext, XtNexecute, (XtCallbackProc)gettext, okaystruct);
       XtAddCallback(cancelbutton, XtNselect, (XtCallbackProc)destroypopup, okaystruct);

       /* Generate file prompting widget */

       if (fileprompt) genfilelist(dialog, entertext, width);
    }
    else XtAddCallback(okbutton, XtNselect, (XtCallbackProc)destroypopup, okaystruct);

    XtPopup(popup, XtGrabNone);

    /* set the input focus for the window */

    wmhints = XAllocWMHints();
    wmhints->flags |= InputHint;
    wmhints->input = True;
    XSetWMHints(dpy, XtWindow(popup), wmhints);
    XFree(wmhints);

    if (current != NULL) XDefineCursor(dpy, XtWindow(entertext), 
	TEXTPTR);
}

/*-------------------------------------------------------------------------*/
/* Create a popup window for property changes				   */
/*-------------------------------------------------------------------------*/

#define MAXPROPS 7
propstruct okstruct[MAXPROPS], fpokstruct;

XtCallbackProc dooutput(button, clientdata, calldata)
  Widget        button;
  caddr_t       clientdata, calldata;
{
   buttonsave  *savebutton;
   Arg         wargs[8];
   Widget      popup, dialog, okbutton, tmpwidget, titlearea, wrbutton;
   Widget      fpentertext, fpokay;
   WidgetList  staticarea, entertext, okays;
   XWMHints    *wmhints;	/* for proper input focus */
   Position    xpos, ypos;
   short       n = 0;
   Dimension   height, width, areawidth, areaheight, bwidth, owidth, wwidth;
   XGCValues   promptvalues;
   char	       *pdptr;
   short       posit, i;
   popupstruct *donestruct;
   void		  (*function[MAXPROPS])();
   XtCallbackProc (*update[MAXPROPS])();
   char	statics[MAXPROPS][50], edit[MAXPROPS][75], request[150];
   char fpedit[75], outname[75];
   void	*data[MAXPROPS], *objpointer;
   float oscale = areastruct.outscale[areastruct.page];
   float psscale = getpsscale(oscale, areastruct.page);
   struct stat statbuf;
   static char defaultTranslations[] = "<Key>Return:	execute()";

   if (pushes != 0) {
      Wprintf("Can only save a top-level page!");
      return;
   }
   savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   getgeneric(savebutton, button, dooutput, NULL); 

   sprintf(request, "PostScript output properties (Page %d):", 
	areastruct.page + 1);
   sprintf(statics[0], "Filename:");
   sprintf(statics[1], "Page label:");
   sprintf(statics[2], "Scale:");
   if (areastruct.coordstyle[areastruct.page] == CM) {
      sprintf(statics[3], "X Size (cm):");
      sprintf(statics[4], "Y Size (cm):");
   }
   else {
      sprintf(statics[3], "X Size (in):");
      sprintf(statics[4], "Y Size (in):");
   }
   sprintf(statics[5], "Orientation:");
   sprintf(statics[6], "Mode:");
   calcbbox(objectdata);
   sprintf(edit[0], "%s", areastruct.filename[areastruct.page]);
   sprintf(edit[1], "%s", objectdata->name);
   sprintf(edit[2], "%6.5f", oscale);
   if (areastruct.coordstyle[areastruct.page] == CM) {
      sprintf(edit[3], "%6.5f", (objectdata->width * psscale) / IN_CM_CONVERT);
      sprintf(edit[4], "%6.5f", (objectdata->height * psscale) / IN_CM_CONVERT);
   }
   else {
      sprintf(edit[3], "%6.5f", (objectdata->width * psscale) / 72.0);
      sprintf(edit[4], "%6.5f", (objectdata->height * psscale) / 72.0);
   }
   sprintf(edit[5], "%s", (areastruct.orient[areastruct.page] == 0)
	? "Portrait" : "Landscape");
   sprintf(edit[6], "%s", (areastruct.pmode[areastruct.page] == 0)
	? "Embedded (EPS)" : "Full page (PS)");
   function[0] = setfilename;
   function[1] = setpagelabel;
   function[2] = setfloat;
   function[3] = setscalex;
   function[4] = setscaley;
   function[5] = setorient;
   function[6] = setpmode;
   update[0] = updatename;
   update[1] = update[5] = update[6] = NULL;
   update[2] = updatetext;
   update[3] = updatetext;
   update[4] = updatetext;
   data[0] = &(areastruct.filename[areastruct.page]);
   data[1] = objectdata->name;
   data[2] = data[3] = data[4] = &(areastruct.outscale[areastruct.page]);
   data[5] = &(areastruct.orient[areastruct.page]);
   data[6] = &(areastruct.pmode[areastruct.page]);

   entertext = (WidgetList) XtMalloc (7 * sizeof (Widget));
   staticarea = (WidgetList) XtMalloc (7 * sizeof (Widget));
   okays = (WidgetList) XtMalloc (6 * sizeof (Widget));

   /* get file information */

   if (strstr(edit[0], ".") == NULL)
      sprintf(outname, "%s.ps", edit[0]);  
   else sprintf(outname, "%s", edit[0]);
   if (stat(outname, &statbuf) == 0) {
      sprintf(outname, "Overwrite File");
      Wprintf("  Warning:  File exists");
   }
   else {
      sprintf(outname, "Write File");
      if (errno == ENOTDIR)
	 Wprintf("Error:  Incorrect pathname");
      else if (errno == EACCES)
	 Wprintf("Error:  Path not readable");
      else
         Wprintf("  ");
   }

   height = ROWHEIGHT * 17;  /* 3 + (2 * MAXPROPS) */
   width = XTextWidth(areastruct.textstruct, request, strlen(request)) + 20;
   bwidth = XTextWidth(areastruct.textstruct, "Close", strlen("Close")) + 50;
   owidth = XTextWidth(areastruct.textstruct, "Apply", strlen("Apply")) + 50;
   wwidth = XTextWidth(areastruct.textstruct, outname, strlen(outname)) + 80;
   if (width < 500) width = 500;

   XtnSetArg(XtNwidth, &areawidth);
   XtnSetArg(XtNheight, &areaheight);
   XtGetValues(areastruct.area, wargs, n); n = 0;
   XtTranslateCoords(areastruct.area, (Position) (areawidth / 2 - width 
	/ 2 + popups * 20), (Position) (areaheight / 2 - height / 2 +
	popups * 20), &xpos, &ypos);
   XtnSetArg(XtNx, xpos);
   XtnSetArg(XtNy, ypos);
   popup = XtCreatePopupShell("prompt", transientShellWidgetClass,
        areastruct.area, wargs, n); n = 0;
   popups++;

   XtnSetArg(XtNlayout, XwIGNORE);
   XtnSetArg(XtNwidth, width);
   XtnSetArg(XtNheight, height);
   dialog = XtCreateManagedWidget("dialog", XwbulletinWidgetClass,
        popup, wargs, n); n = 0;

   XtnSetArg(XtNx, 20);
   XtnSetArg(XtNy, ROWHEIGHT - 10);
   XtnSetArg(XtNstring, request);
   XtnSetArg(XtNborderWidth, 0);
   XtnSetArg(XtNgravity, WestGravity);
   XtnSetArg(XtNbackground, BARCOLOR);
   titlearea = XtCreateManagedWidget("title", XwstaticTextWidgetClass,
	dialog, wargs, n); n = 0;

   XtnSetArg(XtNx, 20);
   XtnSetArg(XtNy, height - ROWHEIGHT - 10);
   XtnSetArg(XtNwidth, owidth); 
   XtnSetArg(XtNfont, areastruct.textstruct);
   okbutton = XtCreateManagedWidget("Close", XwmenuButtonWidgetClass, 
	dialog, wargs, n); n = 0;

   XtnSetArg(XtNx, width - wwidth - 20);
   XtnSetArg(XtNy, height - ROWHEIGHT - 10);
   XtnSetArg(XtNwidth, wwidth);
   XtnSetArg(XtNfont, areastruct.textstruct);
   XtnSetArg(XtNlabel, outname);
   wrbutton = XtCreateManagedWidget("Write File", XwmenuButtonWidgetClass,
	dialog, wargs, n); n = 0;

   for (i = 0; i < MAXPROPS; i++) {
      XtnSetArg(XtNx, 20);
      XtnSetArg(XtNy, ROWHEIGHT + 15 + (i * 2 * ROWHEIGHT));
      XtnSetArg(XtNstring, statics[i]);
      XtnSetArg(XtNborderWidth, 0);
      XtnSetArg(XtNgravity, WestGravity);
      staticarea[i] = XtCreateManagedWidget("static", XwstaticTextWidgetClass,
	   dialog, wargs, n); n = 0;

      XtnSetArg(XtNx, 150);
      XtnSetArg(XtNy, ROWHEIGHT + 10 + (i * 2 * ROWHEIGHT));
      if (i < 5) {
         XtnSetArg(XtNheight, ROWHEIGHT + 5);
         XtnSetArg(XtNstring, edit[i]);
         XtnSetArg(XtNwidth, width - owidth - 190);
         pdptr = strchr(edit[i], '.');
         posit = (pdptr != NULL) ? (short)(pdptr - edit[i]) : strlen(edit[i]);
         XtnSetArg(XtNinsertPosition, posit);
         XtnSetArg(XtNgrow, XwGrowHorizontal);
         entertext[i] = XtCreateManagedWidget("Edit", XwtextEditWidgetClass,
	     dialog, wargs, n); n = 0; 

         XtnSetArg(XtNx, width - owidth - 20);
         XtnSetArg(XtNy, ROWHEIGHT + 10 + (i * 2 * ROWHEIGHT));
         XtnSetArg(XtNwidth, owidth); 
         XtnSetArg(XtNfont, areastruct.textstruct);
         okays[i] = XtCreateManagedWidget("Apply", XwmenuButtonWidgetClass, 
	    dialog, wargs, n); n = 0;

         okstruct[i].textw = entertext[i];
	 okstruct[i].buttonw = okays[i];
         okstruct[i].setvalue = function[i];
         okstruct[i].dataptr = data[i];

         XtAddCallback(okays[i], XtNselect, (XtCallbackProc)getproptext, &okstruct[i]);
	 if (update[i] != NULL)
            XtAddCallback(okays[i], XtNselect, (XtCallbackProc)update[i], entertext);
         XtOverrideTranslations(entertext[i], XtParseTranslationTable
              	(defaultTranslations));
         XtAddCallback(entertext[i], XtNexecute, (XtCallbackProc)getproptext,
		&okstruct[i]);
         if (update[i] != NULL) XtAddCallback(entertext[i], XtNexecute,
		(XtCallbackProc)update[i], entertext);

      }
      else {
	 XtnSetArg(XtNlabel, edit[i]);
         XtnSetArg(XtNfont, areastruct.textstruct);
         entertext[i] = XtCreateManagedWidget("Toggle", XwpushButtonWidgetClass,
	    dialog, wargs, n); n = 0;
	 XtAddCallback(entertext[i], XtNselect, (XtCallbackProc)function[i], data[i]);
      }
   }

   /* If full-page pmode is chosen, there is an additional text structure.
      Make this text structure always but allow it to be managed and
      unmanaged as necessary. */

   if (areastruct.coordstyle[areastruct.page] == CM) {
      sprintf(fpedit, "%3.2f x %3.2f cm",
	 (float)areastruct.pagesize[areastruct.page].x / IN_CM_CONVERT,
	 (float)areastruct.pagesize[areastruct.page].y / IN_CM_CONVERT);
   }
   else {
      sprintf(fpedit, "%3.2f x %3.2f in",
	 (float)areastruct.pagesize[areastruct.page].x / 72.0,
	 (float)areastruct.pagesize[areastruct.page].y / 72.0);
   }
   XtnSetArg(XtNx, 260);
   XtnSetArg(XtNy, ROWHEIGHT + 10 + (12 * ROWHEIGHT));
   XtnSetArg(XtNheight, ROWHEIGHT + 5);
   XtnSetArg(XtNstring, fpedit);
   XtnSetArg(XtNwidth, width - owidth - 300);
   pdptr = strchr(fpedit, '.');
   posit = (pdptr != NULL) ? (short)(pdptr - fpedit) : strlen(fpedit);
   XtnSetArg(XtNinsertPosition, posit);
   XtnSetArg(XtNgrow, XwGrowHorizontal);
   fpentertext = XtCreateWidget("fpedit", XwtextEditWidgetClass,
       dialog, wargs, n); n = 0;

   XtnSetArg(XtNx, width - owidth - 20);
   XtnSetArg(XtNy, ROWHEIGHT + 10 + (12 * ROWHEIGHT));
   XtnSetArg(XtNwidth, owidth);
   XtnSetArg(XtNfont, areastruct.textstruct);
   XtnSetArg(XtNlabel, "Apply");
   fpokay = XtCreateWidget("fpokay", XwmenuButtonWidgetClass,
      dialog, wargs, n); n = 0;

   fpokstruct.textw = fpentertext;
   fpokstruct.buttonw = fpokay;
   fpokstruct.setvalue = setpagesize;
   fpokstruct.dataptr = &(areastruct.pagesize[areastruct.page]);

   XtAddCallback(fpokay, XtNselect, (XtCallbackProc)getproptext, &fpokstruct);
   XtOverrideTranslations(fpentertext, XtParseTranslationTable
        (defaultTranslations));
   XtAddCallback(fpentertext, XtNexecute, (XtCallbackProc)getproptext, &fpokstruct);

   if (areastruct.pmode[areastruct.page] == 1) {
      XtManageChild(fpentertext);
      XtManageChild(fpokay);
   }

   /* end of pagesize extra Widget definitions */

   donestruct = (popupstruct *) malloc(sizeof(popupstruct));
   donestruct->popup = popup;
   donestruct->buttonptr = savebutton;
   XtAddCallback(okbutton, XtNselect, (XtCallbackProc)destroypopup, donestruct);

   /* Send setfile() the widget entertext[0] in case because user sometimes
      forgets to type "okay" but buffer contains the expected filename */

   XtAddCallback(wrbutton, XtNselect, (XtCallbackProc)setfile, entertext[0]);

   /* Begin Popup */

   XtPopup(popup, XtGrabNone);

   /* set the input focus for the window */

   wmhints = XAllocWMHints();
   wmhints->flags |= InputHint;
   wmhints->input = True;
   XSetWMHints(dpy, XtWindow(popup), wmhints);
   XFree(wmhints);
   for (i = 0; i < 5; i++)
      XDefineCursor(dpy, XtWindow(entertext[i]), TEXTPTR);
}

/*-------------------------------------------------*/
/* Print a string to the message widget. 	   */ 
/* Note: Widget message must be a global variable. */
/* For formatted strings, format first into _STR   */
/*-------------------------------------------------*/

XtTimerCallbackProc clrmessage(id, clientdata)
  XtIntervalId	id;
  caddr_t	clientdata;
{
   Wprintf(" ");
}

void W1printf(string)
char *string;
{
   Arg	wargs[1];

   XtSetArg(wargs[0], XtNstring, string);
   XtSetValues(message1, wargs, 1);
   XtSetArg(wargs[0], XtNheight, ROWHEIGHT);
   XtSetValues(message1, wargs, 1);
}   

void W2printf(string)
char *string;
{
   Arg	wargs[1];

   XtSetArg(wargs[0], XtNstring, string);
   XtSetValues(message2, wargs, 1);
   XtSetArg(wargs[0], XtNheight, ROWHEIGHT);
   XtSetValues(message2, wargs, 1);
}   

void Wprintf(string)
char *string;
{
   Arg	wargs[1];

   if (printtime_id != 0) {
      XtRemoveTimeOut(printtime_id);
      printtime_id = 0;
   }
   XtSetArg(wargs[0], XtNstring, string);
   XtSetValues(message3, wargs, 1);
   XtSetArg(wargs[0], XtNheight, ROWHEIGHT);
   XtSetValues(message3, wargs, 1);

   /* 10 second timeout */
   printtime_id = XtAddTimeOut(10000, (XtTimerCallbackProc)clrmessage, NULL);
}   

/*------------------------------------------------------------------------------*/
/* When all else fails, install your own colormap			 	*/
/*------------------------------------------------------------------------------*/

int installowncmap()
{
   int i;
   Colormap newcmap;

   /* allocate a new colormap */

   newcmap = XCopyColormapAndFree(dpy, cmap);

   if (newcmap == (Colormap)NULL) return (-1);

   cmap = newcmap;

   if (areastruct.area != (Widget)NULL) {
      if (XtIsRealized(areastruct.area)) {
         Widget colormenu = XtNameToWidget(top, ColorsPath);

         XSetWindowColormap(dpy, XtWindow(top), cmap);
         XSetWindowColormap(dpy, win, cmap);
         if (colormenu != (Widget)NULL)
            XSetWindowColormap(dpy, XtWindow(colormenu), cmap);
      }
   }

   return(1);
}

/*------------------------------------------------------------------------------*/
/* Find the nearest color in the colormap to cvexact and return its pixel value */
/*------------------------------------------------------------------------------*/

int findnearcolor(cvexact)
   XColor *cvexact;
{
   /* find the nearest-matching color in the colormap */
 
   int i, ncolors = DisplayCells(dpy, DefaultScreen(dpy));
   XColor *cmcolors;
   long rdist, bdist, gdist;
   unsigned long dist, mindist;
   int minidx;

   cmcolors = (XColor *)malloc(ncolors * sizeof(XColor));

   for (i = 0; i < ncolors; i++) {
      cmcolors[i].pixel = i;
      cmcolors[i].flags = DoRed | DoGreen | DoBlue;
   }
   XQueryColors(dpy, cmap, cmcolors, ncolors); 

   mindist = ULONG_MAX;
   for (i = 0; i < ncolors; i++) {
      rdist = (cmcolors[i].red - cvexact->red);
      bdist = (cmcolors[i].blue - cvexact->blue);
      gdist = (cmcolors[i].green - cvexact->green);
      dist = rdist * rdist + bdist * bdist + gdist * gdist;
      if (dist < mindist) {
	 mindist = dist;
	 minidx = i;
      }
   }
   free(cmcolors);

   /* if we can't get the right color or something reasonably close, */
   /* then allocate our own colormap.  If we've allocated so many    */
   /* colors that the new colormap is full, then we give up and use  */
   /* whatever color was closest, no matter how far away it was.     */

   if (dist > 30000) {
      if (installowncmap() > 0) {
         if (XAllocColor(dpy, cmap, cvexact) != 0)
	    minidx = cvexact->pixel;
      }
   }

   return minidx;
}

/*-------------------------------------------------------------------------*/
/* Hack on the resource database Color allocation			   */
/*									   */
/* This overrides the built-in routine.  The difference is that if a	   */
/* color cell cannot be allocated (colormap is full), then the color	   */
/* map is searched for the closest RGB value to the named color.	   */
/*									   */
/* This code depends on Display *dpy being set:  Do not install the	   */
/* converter until after XtInitialize().				   */
/*-------------------------------------------------------------------------*/

XtConverter CvtStringToPixel(args, nargs, fromVal, toVal)
   XrmValuePtr args, fromVal, toVal;
   int	*nargs;
{
   static XColor cvcolor;
   XColor cvexact;

   if (dpy == NULL) return;

   if (*nargs != 0)
      XtWarning("String to Pixel conversion needs no arguments");

   /* Color defaults to black if name is not found */

   if (XAllocNamedColor(dpy, cmap, (char *)fromVal->addr, &cvcolor, &cvexact)
	 == 0) {
      if (XLookupColor(dpy, cmap, (char *)fromVal->addr, &cvexact, &cvcolor) == 0)
	 cvcolor.pixel = BlackPixel(dpy, DefaultScreen(dpy));
      else
         cvcolor.pixel = findnearcolor(&cvexact);
   }

   toVal->size = sizeof(unsigned long);
   toVal->addr = (caddr_t) &(cvcolor.pixel);
}

/*-------------------------------------------------------------------------*/
/* Allocate new color using the CvtStringToPixel routine		   */
/*-------------------------------------------------------------------------*/

int xc_alloccolor(name)
  char *name;
{
   XrmValue fromC, toC;
   int zval = 0, i, pixval;

   fromC.size = strlen(name);
   fromC.addr = name;

   CvtStringToPixel(NULL, &zval, &fromC, &toC);
   pixval = (int)(*((unsigned long *)toC.addr));

   return pixval;
}

/*-------------------------------------------------------------------------*/
/* Allocate new color from RGB values (e.g., from PS file "scb" command)   */
/*-------------------------------------------------------------------------*/

int rgb_alloccolor(red, green, blue)
  int red, green, blue;
{
   XColor newcolor;
   int i, pixval = -1;

   /* first check if color within RGB roundoff error exists in the list */

   for (i = 0; i < number_colors; i++) {
      if (abs(colorlist[i].color.red - red) < 200 &&
	     abs(colorlist[i].color.green - green) < 200 &&
	     abs(colorlist[i].color.blue - blue) < 200) {
	 pixval = colorlist[i].color.pixel;
	 break;
      }
   }

   /* if color is not already in list, try to allocate it; if allocation */
   /* fails, grab the closest match in the colormap.			 */

   if (pixval < 0) {
      newcolor.red = red;
      newcolor.green = green;
      newcolor.blue = blue;
      newcolor.flags = DoRed | DoGreen | DoBlue;
      if (XAllocColor(dpy, cmap, &newcolor) == 0)
         pixval = findnearcolor(&newcolor);
      else
	 pixval = newcolor.pixel;
   }
   return pixval;
}

/*-------------------------------------------------------------------------*/
/* Make the cursors from the cursor bit data				   */
/*-------------------------------------------------------------------------*/

void makecursors()
{
   XColor fgcolor, bgcolor;

   bgcolor.pixel = BACKGROUND;
   fgcolor.pixel = FOREGROUND;
   XQueryColor(dpy, cmap, &fgcolor);
   XQueryColor(dpy, cmap, &bgcolor);

   ARROW = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, arrow_bits,
	arrow_width, arrow_height), XCreateBitmapFromData(dpy, win, arrowmask_bits,
	arrow_width, arrow_height), &fgcolor, &bgcolor, arrow_x_hot, arrow_y_hot);
   CROSS = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, cross_bits,
	cross_width, cross_height), XCreateBitmapFromData(dpy, win, crossmask_bits,
	cross_width, cross_height), &fgcolor, &bgcolor, cross_x_hot, cross_y_hot);
   SCISSORS = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, scissors_bits,
	scissors_width, scissors_height), XCreateBitmapFromData(dpy, win,
	scissorsmask_bits, scissors_width, scissors_height), &fgcolor,
	&bgcolor, scissors_x_hot, scissors_y_hot);
   EDCURSOR = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, exx_bits,
	exx_width, exx_height), XCreateBitmapFromData(dpy, win, exxmask_bits,
	exx_width, exx_height), &fgcolor, &bgcolor, exx_x_hot, exx_y_hot);
   COPYCURSOR = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, copy_bits,
	copy_width, copy_height), XCreateBitmapFromData(dpy, win, copymask_bits,
	copy_width, copy_height), &fgcolor, &bgcolor, copy_x_hot, copy_y_hot);
   ROTATECURSOR = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, rot_bits,
	rot_width, rot_height), XCreateBitmapFromData(dpy, win, rotmask_bits,
	rot_width, rot_height), &fgcolor, &bgcolor, circle_x_hot, circle_y_hot);
   QUESTION = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, question_bits,
	question_width, question_height), XCreateBitmapFromData(dpy, win,
	questionmask_bits, question_width, question_height),
	&fgcolor, &bgcolor, question_x_hot, question_y_hot);
   CIRCLE = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, circle_bits,
	circle_width, circle_height), XCreateBitmapFromData(dpy, win, circlemask_bits,
	circle_width, circle_height), &fgcolor, &bgcolor, circle_x_hot, circle_y_hot);
   TEXTPTR = XCreateFontCursor(dpy, XC_xterm);

   XRecolorCursor(dpy, TEXTPTR, &fgcolor, &bgcolor);
}

/*-------------------------------------------------------------------------*/
/* Main program 							   */
/*-------------------------------------------------------------------------*/

main(argc, argv) 
  int argc; 
  char *argv[]; 
{ 
   Widget form, lastbutton, corner;
   XColor color, ignore;
   XGCValues    values;
   XWMHints	*wmhints;	/* for proper input focus */
   Arg		wargs[12];
   short i, j, k = 0, n = 0, maxcolors;
   short initargc = argc;	/* because XtInitialize() absorbs the     */
   objectptr	*page;		/* -schem flag and renumbers argc! (bug?) */
   Pixmap	icon, mask;
   char		*argv0;		/* find root of argv[0] */

   /*-----------------------------------------------------------*/
   /* Find the root of the command called from the command line */
   /*-----------------------------------------------------------*/

   argv0 = strrchr(argv[0], '/');
   if (argv0 == NULL)
      argv0 = argv[0];
   else
      argv0++;

   /*---------------------------*/
   /* Check for schematic flag  */
   /*---------------------------*/

#ifdef SCHEMA
   areastruct.schemon = False;
   if (!strcmp(argv0, SCHEM_PROG)) areastruct.schemon = True;
   else {
      for (k = argc - 1; k > 0; k--) {
	 if (!strncmp(argv[k], SCHEM_FLAG, strlen(SCHEM_FLAG))) {
	    areastruct.schemon = True;
	    break;
	 }
      }
   }
#endif
   
   /*---------------------------*/
   /* initialize user variables */
   /*---------------------------*/

   areastruct.area = (Widget)NULL;

   areastruct.psfont = 4;
   areastruct.justify = FLIPINV;
   areastruct.page = 0;
   areastruct.pages = PAGES;

   areastruct.filename = (char **)malloc(PAGES * sizeof(char *));
   areastruct.wirewidth = (float *)malloc(PAGES * sizeof(float));
   areastruct.outscale = (float *)malloc(PAGES * sizeof(float));
   areastruct.viewscale = (float *)malloc(PAGES * sizeof(float));
   areastruct.orient = (short *)malloc(PAGES * sizeof(short));
   areastruct.pmode = (short *)malloc(PAGES * sizeof(short));
   areastruct.drawingscale = (XPoint *)malloc(PAGES * sizeof(XPoint));
   areastruct.coordstyle = (short *)malloc(PAGES * sizeof(short));
   areastruct.pcorner = (XPoint *)malloc(PAGES * sizeof(XPoint));
   areastruct.pagesize = (XPoint *)malloc(PAGES * sizeof(XPoint));

   areastruct.wirewidth[0] = 2.0;
   areastruct.outscale[0] = 1.0;
   areastruct.viewscale[0] = 0.5;
   areastruct.pmode[0] = 0;
   areastruct.orient[0] = 0;
   areastruct.drawingscale[0].x = areastruct.drawingscale[0].y = 1;
   areastruct.coordstyle[0] = FRAC_INCH;
   areastruct.pagesize[0].x = 612;
   areastruct.pagesize[0].y = 792;

   areastruct.vscale = &areastruct.viewscale[0];
   areastruct.lowerleft = &areastruct.pcorner[0];  /* values set below */
   areastruct.textscale = 1.0;
   areastruct.style = UNCLOSED;
   areastruct.invert = False;
   areastruct.axeson = True;
   areastruct.snapspace = 16;
   areastruct.snapto = True;
   areastruct.gridon = True;
   areastruct.gridspace = 32;
   areastruct.selects = 0;
   areastruct.selectlist = NULL;
   areastruct.manhatn = False;
   areastruct.boxedit = MANHATTAN;
   areastruct.builtins = 0;
   areastruct.userobjs = 0;
   areastruct.library = (objectptr *) malloc(sizeof(objectptr));
   areastruct.userlib = (objectptr *) malloc(sizeof(objectptr));
   areastruct.editstack = (objectptr) malloc(sizeof(object));
   areastruct.tempvscale = 0.5;
   areastruct.hlevels = 0;
   areastruct.hierarchy = (objinstptr *) malloc(sizeof(objinstptr));
   areastruct.deletes = 0;
   areastruct.delbuffer = (objectptr *) malloc(DELBUFSIZE * sizeof(objectptr));
   areastruct.pagelist = (objectptr *) malloc(PAGES * sizeof(objectptr));
   for (page = areastruct.pagelist; page < areastruct.pagelist + PAGES; page++)
      *page = NULL;

   /* Set up the predefined fonts */

   fonts = (char **) malloc(FONTS * sizeof(char *));
   for (i = 0; i < FONTS; i++)
      fonts[i] = prefonts[i];
   fontcount = FONTS;

   initmem(areastruct.editstack);
   for (i = 0; i < LIBS; i++) {
      areastruct.libtop[i] = (objectptr) malloc(sizeof(object));
      initmem(areastruct.libtop[i]);
   }
   strcpy(areastruct.libtop[1]->name, "Built-in Library");
   strcpy(areastruct.libtop[0]->name, "User Library");

   areastruct.topobject = (objinstptr) malloc(sizeof(objinst));
   *(areastruct.pagelist) = &areaobject;
   areastruct.topobject->thisobject = &areaobject;
   areastruct.topobject->scale = 1.0;
   areastruct.topobject->rotation = 1;
   areastruct.topobject->position.x = 0;
   areastruct.topobject->position.y = 0;
   *areastruct.hierarchy = areastruct.topobject;

   /* note: objectdata is defined as areastruct.topobject->thisobject */
   sprintf(objectdata->name, "Page 1");
   areastruct.filename[0] = (char *)malloc((strlen(objectdata->name) + 1)
	* sizeof(char));
   strcpy(areastruct.filename[0], objectdata->name);
   objectdata->width = 0;
   objectdata->height = 0;
   objectdata->lowerleft.x = 0;
   objectdata->lowerleft.y = 0;
   initmem(objectdata);

   popups = 0;        /* no popup windows yet */
   pushes = 0;        /* at the top of the hierarchy */
   beeper = 1;        /* Ring bell on certain warnings or errors */

   sscanf(VERSION, "%f", &version);  /* version of xcircuit */

   initsplines();	/* create lookup table of spline parameters */

   /*-----------------------------*/
   /* Create the widget hierarchy */
   /*-----------------------------*/

#ifdef SCHEMA
   if (!strcmp(argv0, SCHEM_PROG))
      top = XtInitialize(argv0, "XSchema", NULL, 0, &argc, argv);
   else
      top = XtInitialize(argv0, "XCircuit", NULL, 0, &argc, argv);
#else
   top = XtInitialize(argv0, "XCircuit", NULL, 0, &argc, argv);
#endif

   dpy = XtDisplay(top);
   win = DefaultRootWindow(dpy);
   cmap = DefaultColormap(dpy, DefaultScreen(dpy));

   /*-------------------------*/
   /* Create stipple patterns */
   /*-------------------------*/

   for (i = 0; i < STIPPLES; i++)
      STIPPLE[i] = XCreateBitmapFromData(dpy, win, STIPDATA[i], 4, 4);

   /*----------------------------------------*/
   /* Allocate space for the basic color map */
   /*----------------------------------------*/

   number_colors = 0;
   colorlist = (colorindex *)malloc(sizeof(colorindex));
   appcolors = (int *) malloc(NUMBER_OF_COLORS * sizeof(int));

   /*-------------------------------*/
   /* Get the application resources */
   /*-------------------------------*/

   XtAddConverter(XtRString, XtRPixel, (XtConverter)CvtStringToPixel, NULL, 0);
   XtGetApplicationResources(top, &appdata, resources, XtNumber(resources), 
	NULL, 0);
   areastruct.textstruct = appdata.xcfont;

   XtnSetArg(XtNwidth, appdata.width);
   XtnSetArg(XtNheight, appdata.height);
   XtnSetArg(XtNforeground, appdata.fg);
   XtnSetArg(XtNbackground, appdata.bg);
   XtnSetArg(XtNcolormap, cmap);

   XtSetValues(top, wargs, n); n = 0;
   form = XtCreateManagedWidget("Form", XwformWidgetClass, top, NULL, 0);

   /* Set up the buttons and Graphics drawing area */

   lastbutton = createmenus(form);

   XtnSetArg(XtNxRefWidget, lastbutton);
   XtnSetArg(XtNyRefWidget, form);
   XtnSetArg(XtNxAddWidth, True);
   XtnSetArg(XtNxAttachRight, True);
   XtnSetArg(XtNheight, ROWHEIGHT);
   sprintf(_STR, "   Welcome to Xcircuit Version %s", VERSION);
   XtnSetArg(XtNstring, _STR);
   XtnSetArg(XtNxResizable, True);
   XtnSetArg(XtNgravity, WestGravity);
   XtnSetArg(XtNfont, appdata.xcfont);
   XtnSetArg(XtNwrap, False);
   XtnSetArg(XtNstrip, False);
   message1 = XtCreateManagedWidget("Message1", XwstaticTextWidgetClass,
	form, wargs, n); n = 0;

   /*------------------------*/
   /* add scrollbar widget   */
   /*------------------------*/

   XtnSetArg(XtNyRefWidget, lastbutton);
   XtnSetArg(XtNyAddHeight, True);
   XtnSetArg(XtNxResizable, False);
   XtnSetArg(XtNwidth, SBARSIZE);
   XtnSetArg(XtNyResizable, True);
   XtnSetArg(XtNborderWidth, 1);
   areastruct.scrollbarv = XtCreateManagedWidget("SBV", XwworkSpaceWidgetClass, 
	form, wargs, n); n = 0;
   
   /*-----------------------------*/
   /* For the sake of beauty. . . */
   /*-----------------------------*/

   XtnSetArg(XtNyRefWidget, areastruct.scrollbarv);
   XtnSetArg(XtNyAddHeight, True);
   XtnSetArg(XtNyAttachBottom, True);
   XtnSetArg(XtNyAttachOffset, ROWHEIGHT + 4);
   XtnSetArg(XtNheight, SBARSIZE);
   XtnSetArg(XtNwidth, SBARSIZE);
   XtnSetArg(XtNborderWidth, 1);
   corner = XtCreateManagedWidget("corner", widgetClass,
        form, wargs, n); n = 0;

   /*-------------------------*/
   /* The main drawing window */
   /*-------------------------*/

   XtnSetArg(XtNyRefWidget, lastbutton);
   XtnSetArg(XtNyAddHeight, True);
   XtnSetArg(XtNyAttachBottom, True);
   XtnSetArg(XtNyAttachOffset, ROWHEIGHT + SBARSIZE + 3);
   XtnSetArg(XtNxOffset, SBARSIZE + 1);
   XtnSetArg(XtNxAttachRight, True);
   XtnSetArg(XtNxResizable, True);
   XtnSetArg(XtNyResizable, True);
   areastruct.area = XtCreateManagedWidget("Area", XwworkSpaceWidgetClass, 
	form, wargs, n); n = 0;

   /*-------------------------*/
   /* And the other scrollbar */
   /*-------------------------*/

   XtnSetArg(XtNyRefWidget, areastruct.area);
   XtnSetArg(XtNyAddHeight, True);
   XtnSetArg(XtNxOffset, SBARSIZE + 1);
   XtnSetArg(XtNxAttachRight, True);
   XtnSetArg(XtNyResizable, False);
   XtnSetArg(XtNheight, SBARSIZE);
   XtnSetArg(XtNxResizable, True);
   XtnSetArg(XtNborderWidth, 1);
   areastruct.scrollbarh = XtCreateManagedWidget("SBH", XwworkSpaceWidgetClass, 
	form, wargs, n); n = 0;

   /*------------------------------------------------*/
   /* Supplementary message widgets go at the bottom */
   /*------------------------------------------------*/

   XtnSetArg(XtNyRefWidget, areastruct.scrollbarh);
   XtnSetArg(XtNyAddHeight, True);
   XtnSetArg(XtNyOffset, -2);
   XtnSetArg(XtNheight, ROWHEIGHT);
   XtnSetArg(XtNstring, "Editing: Page 1");
   XtnSetArg(XtNxResizable, False);
   XtnSetArg(XtNgravity, WestGravity);
   XtnSetArg(XtNfont, appdata.xcfont);
   XtnSetArg(XtNwrap, False);
   message2 = XtCreateManagedWidget("Message2", XwstaticTextWidgetClass,
	form, wargs, n); n = 0;

   XtnSetArg(XtNyRefWidget, areastruct.scrollbarh);
   XtnSetArg(XtNyAddHeight, True);
   XtnSetArg(XtNyOffset, -2);
   XtnSetArg(XtNxAttachRight, True);
   XtnSetArg(XtNxRefWidget, message2);
   XtnSetArg(XtNxAddWidth, True);
   XtnSetArg(XtNheight, ROWHEIGHT);
   XtnSetArg(XtNstring, "Don't Panic");
   XtnSetArg(XtNxResizable, True);
   XtnSetArg(XtNgravity, WestGravity);
   XtnSetArg(XtNfont, appdata.xcfont);
   XtnSetArg(XtNwrap, False);
   message3 = XtCreateManagedWidget("Message3", XwstaticTextWidgetClass,
	form, wargs, n); n = 0;

   /* Setup callback routines for the area widget */
   /* Use Button1Press event to add the callback which tracks motion;  this */
   /*   will reduce the number of calls serviced during normal operation */ 

   XtAddEventHandler(areastruct.area, ButtonPressMask, False,
	(XtEventHandler)selectbutton, NULL);
   XtAddEventHandler(areastruct.area, ButtonReleaseMask, False,
	(XtEventHandler)releasebutton, NULL);
   XtAddEventHandler(areastruct.area, KeyPressMask, False,
	(XtEventHandler)keyhandler, NULL);

   XtAddCallback(areastruct.area, XtNexpose, (XtCallbackProc)drawarea, NULL);
   XtAddCallback(areastruct.area, XtNresize, (XtCallbackProc)resizearea, NULL);

   /* Setup callback routines for the scrollbar widgets */

   XtAddEventHandler(areastruct.scrollbarh, ButtonPressMask, False,
	(XtEventHandler)panhbar, NULL);
   XtAddEventHandler(areastruct.scrollbarv, ButtonPressMask, False,
	(XtEventHandler)panvbar, NULL);

   XtAddCallback(areastruct.scrollbarh, XtNexpose, (XtCallbackProc)drawhbar, NULL);
   XtAddCallback(areastruct.scrollbarv, XtNexpose, (XtCallbackProc)drawvbar, NULL);
   XtAddCallback(areastruct.scrollbarh, XtNresize, (XtCallbackProc)drawhbar, NULL);
   XtAddCallback(areastruct.scrollbarv, XtNresize, (XtCallbackProc)drawvbar, NULL);

   /*--------------------*/
   /* Realize the Widget */
   /*--------------------*/

   XtRealizeWidget(top);

   /*----------------------------------------------------*/
   /* Let the window manager set the input focus for the */
   /* window and inform the window manager of the icon   */
   /* pixmap (which may or may not be useful, depending  */
   /* on the particular window manager).		 */
   /*----------------------------------------------------*/

   wmhints = XAllocWMHints();

#ifdef HAVE_XPM

   /* Create the xcircuit icon pixmap */

   XpmCreatePixmapFromData(dpy, win, xcircuit_xpm, &icon, NULL, NULL);

   wmhints->flags |= InputHint | IconPixmapHint;
   wmhints->icon_pixmap = icon;
#else
   wmhints->flags |= InputHint;
#endif

   wmhints->input = True;
   XSetWMHints(dpy, XtWindow(top), wmhints);
   XFree(wmhints);

   /*---------------------------------------------------*/
   /* Define basic display variables 			*/
   /* Redefine win to be just the drawing area window   */
   /*---------------------------------------------------*/

   win = XtWindow(areastruct.area);

   /*--------------------------------------------------*/
   /* Setup the (simple) colormap and make the cursors */
   /*--------------------------------------------------*/

   setcolorscheme(True);
   areastruct.color = DEFAULTCOLOR;

   /*-----------------------------*/
   /* Create the Graphics Context */
   /*-----------------------------*/

   values.foreground = BlackPixel(dpy, DefaultScreen(dpy));
   values.background = WhitePixel(dpy, DefaultScreen(dpy));
   values.font = areastruct.textstruct->fid;

   areastruct.gc = XCreateGC(dpy, win, GCForeground | GCBackground | GCFont, &values);

   /* Setup a passive grab on the area widget for mouse buttons 1 & 2 */

   XGrabButton(dpy, AnyButton, AnyModifier, win, True, ButtonPressMask |
	ButtonMotionMask | ButtonReleaseMask, GrabModeAsync, GrabModeAsync,
	win, None);

   /* set the area widget width and height, center userspace (0,0) on screen */

   XtSetArg(wargs[0], XtNwidth, &areastruct.width);
   XtSetArg(wargs[1], XtNheight, &areastruct.height);
   XtGetValues(areastruct.area, wargs, 2);
   areastruct.lowerleft->x = -areastruct.width;
   areastruct.lowerleft->y = -areastruct.height;

   areastruct.MatStackDepth = 1; /* Top transformation matrix */
   areastruct.MatStack = (Matrix *)malloc(sizeof(Matrix)); /* original allocation */
   UResetCTM(DCTM);
   UMakeWCTM(DCTM);  		 /* Find TM for window */

#ifdef DOUBLEBUFFER
   if (dbuf == (Pixmap)NULL)
      dbuf = XCreatePixmap(dpy, win, areastruct.width, areastruct.height,
	      DefaultDepthOfScreen(XtScreen(areastruct.area)));
#endif

   /*----------------------------------------------------------*/
   /* Check home directory for initial settings	& other loads; */
   /* Load the (default) built-in set of objects 	       */
   /*----------------------------------------------------------*/

   loadrcfile();

   /*-----------------------------------------------------*/
   /* Set the cursor as a crosshair for the area widget.  */
   /*-----------------------------------------------------*/

   XDefineCursor (dpy, win, CROSS);

   /*-------------------------------------------------*/
   /* Parse the command line for initial file to load */
   /*-------------------------------------------------*/

   if (argc == 2 + (k != 0) || initargc == 2 + (k != 0)) {
      strcpy(_STR2, argv[(k == 1) ? 2 : 1]);
      startloadfile();
   }

   /*----------------------*/
   /* Start the event loop */
   /*----------------------*/

   XtMainLoop();
}

/*-------------------------------------------------------------------------*/
