
/* reader.c - 
 * 	Multi-format file readers for analog waveform data.
 * In this file is the generic read routine that calls format-specific
 * readers, and also some common support routines.
 *
 * $Log: reader.c,v $
 * Revision 1.3  1998/09/17 18:23:29  tell
 * added backpointer from DataSet to DataFile
 *
 * Revision 1.2  1998/09/10 21:04:44  tell
 * added routines to free DataFiles and DataSets
 * gathered HAVE_REGEXP stuff into one place so code is more readable
 * misc other fixes
 *
 * Revision 1.1  1998/08/31 20:58:17  tell
 * Initial revision
 *
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <glib.h>
#include "reader.h"

#ifdef HAVE_POSIX_REGEXP
#include <regex.h>
#define REGEXP_T regex_t
#define regexp_test(c,s) (regexec((c), (s), 0, NULL, 0) == 0)
static regex_t *
regexp_compile(char *str)
{
	int err;
	char ebuf[128];
	regex_t *creg;

	creg = g_new(regex_t, 1);
	err = regcomp(creg, str, REG_NOSUB|REG_EXTENDED);
	if(err) {
		regerror(err, creg, ebuf, sizeof(ebuf));
		fprintf(stderr, "internal error (in regexp %s):\n", str);
		fprintf(stderr, "  %s\n", ebuf);
		exit(1);
	}
	return creg;
}

#else		
#include "regexp.h"	/* Henry Spencer's V8 regexp */
#define REGEXP_T regexp
#define regexp_test(c,s) regexec((c), (s))
#define regexp_compile(s) regcomp(s)
#endif

int analog_read_debug = 0;

typedef DataFile* (*PFD)(char *name, FILE *fp);

typedef struct {
	char *name;
	PFD rdfunc;
	char *fnrexp;
	REGEXP_T *creg;/* compiled form of regexp */
} DFormat;

/* caution: filename patterns are full egrep-style 
 * regular expressions, not shell-style globs
 */
static DFormat format_tab[] = {
	{"hsbin", hs_read_file_bin, "\\.(tr|sw|ac)[0-9]$" },
	{"hspice", hs_read_file_ascii, "\\.(tr|sw|ac)[0-9]$" },
	{"cazm", cz_read_file, "\\.[BNW]$" },
};
static const int NFormats = sizeof(format_tab)/sizeof(DFormat);

/*
 * Read an analog data file.
 *  If the format name is non-NULL, only tries reading in specified format.
 *  If format not specified, tries to guess based on filename, and if
 *  that fails, tries all of the readers until one sucedes.
 *  Returns NULL on failure after printing an error message.
 * 
 * TODO: use some kind of callback or exception so that client
 * can put the error messages in a GUI or somthing.
 */
extern DataFile *analog_read_file(char *name, char *format)
{
	FILE *fp;
	DataFile *df;
	int i;

	unsigned int tried = 0; /* bitmask of formats. */

	g_assert(NFormats <= 8*sizeof(tried));
	fp = fopen(name, "r");
	if(fp == NULL) {
		perror(name);
		return NULL;
	}

	if(format == NULL) {
		for(i = 0; i < NFormats; i++) {
			if(!format_tab[i].creg) {
				format_tab[i].creg = regexp_compile(format_tab[i].fnrexp);
			}
			if(regexp_test(format_tab[i].creg, name))
			{
				tried |= 1<<i;
				printf("%s: trying format \"%s\" based on filename\n", name, format_tab[i].name);
				df = (format_tab[i].rdfunc)(name, fp);
				if(df) {
					fclose(fp);
					return df;
				}
				if(fseek(fp, 0L, SEEK_SET) < 0) {
					perror(name);
					return NULL;
				}
					
			}
		}
		if(tried == 0)
			printf("%s: couldn't guess a format from filename suffix.\n", name);
		/* no success with formats whose regexp matched filename,
		* try the others.
		*/
		for(i = 0; i < NFormats; i++) {
			if((tried & (1<<i)) == 0) {
				df = (format_tab[i].rdfunc)(name, fp);
				if(df) {
					printf("read succeded with format %s\n", format_tab[i].name);
					fclose(fp);
					return df;
				}
				tried |= 1<<i;
				if(fseek(fp, 0L, SEEK_SET) < 0) {
					perror(name);
					return NULL;
				}
			}
		}
		return NULL;
	} else { /* look for specified format only */
		for(i = 0; i < NFormats; i++) {
			if(0==strcmp(format, format_tab[i].name)) {
				df = (format_tab[i].rdfunc)(name, fp);
				if(df) {
					fclose(fp);
					return df;
				} else {
					fprintf(stderr, "failed to read %s using format %s\n", name, format_tab[i].name);
					return NULL;
				}
			}
		}
		fprintf(stderr, "File format \"%s\" is not supported; try one of these:\n", format);
		for(i = 0; i < NFormats; i++) {
			fprintf(stderr, "\t%s\n", format_tab[i].name);
		}
		
		return NULL;
	}
}


/***************************************************************************
 * routines common to all analog file formats, and generic support routines
 */

static char *vartype_names[] = {
	"Unknown", "Time", "Voltage", "Current", "Frequency"
};
const int nvartype_names = sizeof(vartype_names)/sizeof(char *);

char *vartype_name_str(VarType type)
{
	static char buf[32];
	if(type < nvartype_names)
		return vartype_names[type];
	else {
		sprintf(buf, "type-%d", type);
		return buf;
	}
}

/*
 * read whole line into buffer, expanding buffer if needed.
 * line buffer must be allocated with g_malloc/g_new
 * returns 0 or EOF.
 */
int
fread_line(FILE *fp, char **bufp, int *bufsize)
{
	int c;
	int n = 0;
	while(((c = getc(fp)) != EOF) && c != '\n') {
		(*bufp)[n++] = c;
		if(n >= *bufsize) {
			*bufsize *= 2;
			*bufp = g_realloc(*bufp, *bufsize);
		}
	}
	(*bufp)[n] = 0;
	if(c == EOF)
		return EOF;
	else
		return 0;
}

/*
 * free up the memory used by a DataFile 
 */
void
an_free_datafile(DataFile *df)
{
	int i;
	if(df) {
		if(df->iv) {
			an_free_dataset(&df->iv->d);
		}
		for(i = 0; i < df->ndv; i++) {
			if(df->dv[i])
				an_free_dataset(&df->dv[i]->d);
		}
		if(df->dv)
			g_free(df->dv);
		g_free(df->filename);
		g_free(df);
	}
}

/*
 * initialize common elements of DataSet structure 
 */ 
void
an_init_dataset(DataSet *ds, DataFile *df, char *name, VarType type)
{
	ds->name = g_strdup(name);
	ds->type = type;
	ds->dfile = df;
	ds->min = G_MAXDOUBLE;
	ds->max = G_MINDOUBLE;
	ds->nvalues = 0;
	
	ds->bpsize = DS_INBLKS;
	ds->bptr = g_new0(double *, ds->bpsize);
	ds->bptr[0] = g_new(double, DS_DBLKSIZE);
	ds->bpused = 1;
}

/*
 * free up memory pointed to by a DataSet, but not the dataset itself.
 */
void
an_free_dataset(DataSet *ds)
{
	int i;
	if(ds->name)
		g_free(ds->name);
	for(i = 0; i < ds->bpused; i++)
		if(ds->bptr[i])
			g_free(ds->bptr[i]);
	g_free(ds->bptr);
}

/*
 * expand dataset's storage to add one more block.
 */
void
an_expand_dset(DataSet *ds)
{
	if(ds->bpused >= ds->bpsize) {
		ds->bpsize *= 2;
		ds->bptr = g_realloc(ds->bptr, ds->bpsize * sizeof(double*));
	}
	ds->bptr[ds->bpused++] = g_new(double, DS_DBLKSIZE);
}

/*
 * set single value in dataset.   Probably can be inlined.
 */
inline void
an_set_point(DataSet *ds, int n, double val)
{
	int blk, off;
	blk = ds_blockno(n);
	off = ds_offset(n);
	g_assert(blk <= ds->bpused);
	g_assert(off < DS_DBLKSIZE);

	ds->bptr[blk][off] = val;
	if(val < ds->min)
		ds->min = val;
	if(val > ds->max)
		ds->max = val;
}

/*
 * get single point from dataset.   Probably can be inlined.
 */
inline double
an_get_point(DataSet *ds, int n)
{
	int blk, off;
	blk = ds_blockno(n);
	off = ds_offset(n);
	g_assert(blk <= ds->bpused);
	g_assert(off < DS_DBLKSIZE);

	return ds->bptr[blk][off];
}

/*
 * Use a binary search to return the index of the point 
 * whose value is the largest not greater than ival.  
 * Only works on independent-variables, which we require to
 * be nondecreasing)
 * 
 * Further, if there are duplicate values, returns the highest index
 * that has the same value.
 */
int
an_find_point(IVar *iv, double ival)
{
	DataSet *ds = &iv->d;
	double cval;
	int a, b;
	int n = 0;
	a = 0;
	b = ds->nvalues - 1;

	while(a+1 < b) {
		cval = an_get_point(ds, (a+b)/2);
/*		printf("%3d %3d %g\n", a, b, cval); */
		if(ival < cval)
			b = (a+b)/2;
		else
			a = (a+b)/2;


		g_assert(n++ < 32);  /* > 2 ** 32 points?  must be a bug! */
	}
/*
	while(a < ds->nvalues && 
	      an_get_point(ds, a) == an_get_point(ds, a+1))
		a++;
*/
	return a;
}

/*
 * return the value of the specified dependent variable at the point where
 * its corresponding dependent variable has the specified value.
 *
 */
double
an_interp_value(DVar *dv, double ival)
{
	int li, ri;   /* index of points to left and right of desired value */
	double lx, rx;  /* independent variable at li and ri */
	double ly, ry;  /* dependent variable at li and ri */

	li = an_find_point(dv->iv, ival);
	ri = li + 1;
	if(ri >= dv->d.nvalues)
		return an_get_point((DataSet *)dv, ri);

	lx = an_get_point((DataSet *)dv->iv, li);
	rx = an_get_point((DataSet *)dv->iv, ri);
	g_assert(lx <= ival);
	ly = an_get_point((DataSet *)dv, li);
	ry = an_get_point((DataSet *)dv, ri);

	if(ival > rx) { /* no extrapolation allowed! */
		return ry;
	}

	return ly + (ry - ly) * ((ival - lx)/(rx - lx));
}

