
(define-class <inet-addr> (<object>) :bvec)

(define-class <inet-socket-addr> (<object>) :bvec)

(define-method write-object ((self <inet-addr>) port)
 (format port "#[<inet-addr> ~a]" (inet-addr->string self)))

(define-method write-object ((self <inet-socket-addr>) p)
  (bind ((host port (inet-socket-addr-parts self)))
    (format p "#[<inet-socket-addr> ~a:~d]" 
	    (inet-addr->string host)
	    port)))
  
(define-syscall-glue (inet-addr->string (self <inet-addr>))
{
  REG0 = make_string( inet_ntoa( *self ) );
  RETURN1();
})

(define-syscall-glue (make-inet-addr (hostspec <raw-string>))
  literals: ((& <inet-addr>))
{
  struct in_addr a;
  struct hostent *h;
  char **alias;

  if (isdigit(hostspec[0]))
    {
      a.s_addr = inet_addr( hostspec );
    }
  else
    {
      h = gethostbyname( (char *)hostspec );
  
      if (h)
	{
	  a = *(struct in_addr *)*h->h_addr_list;
	}
      else
	{
	  scheme_error( "hostname ~s lookup failed", 1, raw_hostspec );
	}
    }

  REG0 = bvec_alloc( sizeof a, TLREF(0) );
  memcpy( PTR_TO_DATAPTR(REG0), &a, sizeof a );
  RETURN1();
})

(define-syscall-glue (make-inet-socket-addr (host <inet-addr>)
					    (port <raw-int>))
  literals: ((& <inet-socket-addr>))
{
  struct sockaddr_in a;

  memset( &a, 0, sizeof a );

  a.sin_addr = *host;
  a.sin_port = htons((short)port);

  REG0 = bvec_alloc( sizeof a, TLREF(0) );
  memcpy( PTR_TO_DATAPTR(REG0), &a, sizeof a );
  RETURN1();
})

(define-syscall-glue (inet-socket-addr-parts (addr <inet-socket-addr>))
 literals: ((& <inet-addr>))
{
  obj a;

  a = bvec_alloc( sizeof( struct in_addr ), TLREF(0) );
  memcpy( PTR_TO_DATAPTR(a), &addr->sin_addr, sizeof( struct in_addr ) );
  REG0 = a;
  REG1 = int2fx( ntohs(addr->sin_port) );
  RETURN(2);
})

(define-syscall-glue (socket-address-family->integer family)
    literals: ('address-family/unix 'address-family/internet)
{
    if (EQ(family,LITERAL(0)))
	REG0 = int2fx( PF_UNIX );
    else if (EQ(family,LITERAL(1)))
	REG0 = int2fx( PF_INET );
    else
	REG0 = FALSE_OBJ;
    RETURN1();
})

(define-syscall-glue (socket-type->integer type)
  literals: ('socket-type/stream
	     'socket-type/datagram
	     'socket-type/raw)
{
    if (EQ(type,LITERAL(0)))
	REG0 = int2fx( SOCK_STREAM );
    else if (EQ(type,LITERAL(1)))
	REG0 = int2fx( SOCK_DGRAM );
    else if (EQ(type,LITERAL(2)))
	REG0 = int2fx( SOCK_RAW );
    else
	REG0 = FALSE_OBJ;
    RETURN1();
})

(define-syscall-glue (socket-create (proto_family <raw-int>)
				 (sock_type <raw-int>)
				 (protocol <raw-int>))
{
int fd;

    fd = socket( proto_family, sock_type, protocol );
    REG0 = (fd < 0) ? FALSE_OBJ : int2fx(fd);
    RETURN1();
})

(define-syscall-glue (socket-listen (socket_fd <raw-int>)
				 (queue_depth <raw-int>))
{
int rc;

    rc = listen( socket_fd, queue_depth );
    REG0 = (rc < 0) ? FALSE_OBJ : TRUE_OBJ;
    RETURN1();
})

(define-syscall-glue (socket-bind/unix (socket_fd <raw-int>)
				    (path <string>))
{
int rc;

    rc = bind( socket_fd, 
	       (struct sockaddr *)string_text(path), 
	       string_length(path) );
    REG0 = (rc < 0) ? FALSE_OBJ : TRUE_OBJ;
    RETURN1();
})

(define-syscall-glue (socket-bind/inet (socket_fd <raw-int>)
				    (port <raw-int>))
{
struct sockaddr_in addr;
int rc;

    memset( &addr, 0, sizeof addr );
    addr.sin_family = PF_INET;
    addr.sin_port = htons((short)port);
    addr.sin_addr.s_addr = INADDR_ANY;
    
    rc = bind( socket_fd, (struct sockaddr *)&addr, sizeof addr );
    REG0 = (rc < 0) ? FALSE_OBJ : TRUE_OBJ;
    RETURN1();
})

(define-syscall-glue (socket-accept (socket_fd <raw-int>))
{
struct sockaddr_in addr;
int n, fd;

    memset( &addr, 0, sizeof addr );
    n = sizeof addr;
    fd = accept( socket_fd, (struct sockaddr *)&addr, &n );
    if (fd < 0)
    {
	REG0 = FALSE_OBJ;
	RETURN1();
    }
    else
    {
	REG0 = int2fx(fd);
	REG1 = make_string( inet_ntoa( addr.sin_addr ) );
	RETURN(2);
    }
})

(define-syscall-glue (socket-connect/inet (socket_fd <raw-int>)
				       (port <raw-int>)
				       (host_id <raw-string>))
{
struct sockaddr_in addr;
int rc;

    memset( &addr, 0, sizeof addr );
    addr.sin_family = PF_INET;
    addr.sin_port = htons((short)port);
    addr.sin_addr.s_addr = inet_addr( host_id );
    
    rc = connect( socket_fd, (struct sockaddr *)&addr, sizeof addr );
    REG0 = (rc < 0) ? FALSE_OBJ : TRUE_OBJ;
    RETURN1();
})

(define-syscall-glue (host-name->address (hostname <raw-string>))
{
struct hostent *h;
obj names, prev_name;
char **alias;

    h = gethostbyname( (char *)hostname );

    if (h)
    {
	names = cons( make_string( h->h_name ), NIL_OBJ );
	prev_name = names;
	for (alias=h->h_aliases; *alias; alias++)
	{
	obj n;
	
	    n = cons( make_string( *alias ), NIL_OBJ );
	    gvec_write_fresh_ptr( prev_name, SLOT(1), n );
	    prev_name = n;
	}
	REG0 = make_string( inet_ntoa( *(struct in_addr *)*h->h_addr_list ));
	REG1 = prev_name;
	RETURN(2);
    }
    else
    {
	REG0 = FALSE_OBJ;
	RETURN1()
    }
})

(define-syscall-glue (host-address->name (address <raw-string>))
{
struct in_addr addr;
struct hostent *h;

    addr.s_addr = inet_addr( address );
    h = gethostbyaddr( (char *)&addr, 4, AF_INET );
    if (h)
    {
      obj names, prev_name;
      char **alias;

	names = cons( make_string( h->h_name ), NIL_OBJ );
	prev_name = names;
	for (alias=h->h_aliases; *alias; alias++)
	{
	obj n;
	
	    n = cons( make_string( *alias ), NIL_OBJ );
	    gvec_write_fresh_ptr( prev_name, SLOT(1), n );
	    prev_name = n;
	}
	REG0 = make_string( inet_ntoa( *(struct in_addr *)*h->h_addr_list ));
	REG1 = names;
	RETURN(2);
    }
    else
    {
	REG0 = FALSE_OBJ;
	RETURN1()
    }
})

(define-class <fd-select-set> (<object>) :bvec)

(define-glue (make-fd-set read_list write_list exception_list)
  literals: ((& <fd-select-set>))
{
  obj set, list;
  int i;
  fd_set *p;

  set = alloc( 3*sizeof(fd_set), TLREF(0) );
  memset( PTR_TO_DATAPTR(set), 0, 3*sizeof(fd_set) );
  
  p = (fd_set *)PTR_TO_DATAPTR(set);
  for (i=0; i<3; i++, p++)
    {
      list = reg_ref(i);
      while (PAIR_P(list))
	{
	  obj fd;

	  fd = pair_car(list);
	  list = pair_cdr(list);
	  if (!OBJ_ISA_FIXNUM(fd))
	    {
	      scheme_error( "make-fd-set: ~s in fd-list[~d] is invalid",
			    2, fd, int2fx(i) );
	    }
	  FD_SET( fx2int(fd), p );
	}
      if (!EQ(list,NIL_OBJ))
	{
	  scheme_error( "make-fd-set: fd-list[~d] is invalid", 1, int2fx(i));
	}
    }
  REG0 = set;
  RETURN1();
})

(define-syscall-glue (fd-select delay_ms (set <fd-select-set>))
{
  fd_set temp[3];
  struct timeval *tp, t;
  int n;

  memcpy( temp, (fd_set *)PTR_TO_DATAPTR(set), 3*sizeof(fd_set) );

  if (OBJ_ISA_FIXNUM(delay_ms))
    {
      t.tv_sec = fx2int(delay_ms)/1000;
      t.tv_usec = (fx2int(delay_ms) % 1000)*1000;
      tp = &t;
    }
  else
    {
      tp = NULL;
    }
  
  n = select( FD_SETSIZE, &temp[0], &temp[1], &temp[2], tp );
  if (n < 0)
    {
      REG0 = FALSE_OBJ;
      RETURN1();
    }
  else if (n == 0)
    {
      REG0 = NIL_OBJ;
      REG1 = NIL_OBJ;
      REG2 = NIL_OBJ;
    }
  else
    {
      obj result;
      int i, fd;

      for (i=0; i<3; i++)
	{
	  result = NIL_OBJ;
	  for (fd=0; fd<FD_SETSIZE; fd++)
	    {
	      if (FD_ISSET(fd,&temp[i]))
		{
		  result = cons( int2fx(fd), result );
		}
	    }
	  reg_set(i,result);
	}
    }
  RETURN(3);
})

(define-syscall-glue (recv-from (fd <raw-int>)
				(buf <raw-string>)
				(offset <raw-int>)
				(len <raw-int>)
				(peekq <raw-bool>)
		                (oobq <raw-bool>)
				from_class)
{
  int n, from_len;
  char from[128];

  from_len = 128;

  n = recvfrom( fd, 
	        buf + offset,
	        len,
	        (peekq ? MSG_PEEK : 0) | (oobq ? MSG_OOB : 0),
	        (struct sockaddr *)from,
	        &from_len );

  if (truish(from_class))
    {
      REG1 = bvec_alloc( from_len, from_class );
      memcpy( PTR_TO_DATAPTR(REG1), from, from_len );
    }
  else
    REG1 = FALSE_OBJ;

  if (n < 0)
    REG0 = FALSE_OBJ;
  else
    REG0 = int2fx(n);
  RETURN(2);
})

(define-syscall-glue (send-to (fd <raw-int>)
			      (buf <raw-string>)
			      (offset <raw-int>)
			      (len <raw-int>)
			      (oobq <raw-bool>)
			      to)
{
 int n;

  n = sendto( fd, 
	      buf + offset,
	      len,
	      (oobq ? MSG_OOB : 0),
	      PTR_TO_DATAPTR(to),
	      SIZEOF_PTR(to) );
  if (n < 0)
    REG0 = FALSE_OBJ;
  else
    REG0 = int2fx(n);
  RETURN1();
})
