#|------------------------------------------------------------*-Scheme-*--|
 | File:    modules/repl/objforms.scm
 |
 |          Copyright (C)1997 Donovan Kolbly <d.kolbly@rscheme.org>
 |          as part of the RScheme project, licensed for free use.
 |          See <http://www.rscheme.org/> for the latest information.
 |
 | File version:     1.31
 | File mod date:    1997.11.29 23:10:32
 | System build:     v0.7.2, 97.12.21
 | Owned by module:  repl
 |
 | Purpose:          Object system forms
 `------------------------------------------------------------------------|#

(define (class-name->hash (name <symbol>))
  (string->hash (number->string (symbol->hash name))))

;; nb:  going through a lambda in these special forms is simply
;; to help the ability to reload files and still work

(define (make-objsys-forms)
  (list (make <definer>
	      name: 'define-class
	      compiler-description: 'define-class
	      syntax-checker: check-define-class-syntax)
	(make <definer>
	      name: 'define-generic-function
	      compiler-description: 'define-generic-function
	      syntax-checker: check-define-generic-function-syntax)
	(make <definer>
	      name: 'define-method
	      compiler-description: 'define-method
	      syntax-checker: check-define-method-syntax)
	(make <special-form>
	      name: 'make
	      compiler-description: 'make)))

(define (objsys-compiler->proc description)
  (case description
    ((make) compile-make)
    ((define-class) compile-tl-define-class)
    ((define-method) compile-tl-define-method)
    ((define-generic-function) compile-tl-define-generic-function)
    (else #f)))

(%early-once-only (add-special-form-compiler! objsys-compiler->proc))

(define (compile-tl-define-class tl-form tl-envt dyn-envt)
  (let ((name (cadr tl-form))
	(supers (parse-supers (caddr tl-form) tl-envt tl-envt))
	(new-slots '())
	(heap-type 0)
	(image-mode #f)
	(category #f)
	(metaclass <<standard-class>>)
	(prim-type #f))
    (if *compile-verbose*
	(format #t "compiling top-level class: ~s\n" name))
    (let loop ((i (cdddr tl-form)))
      (if (pair? i)
	  (if (symbol? (car i))
	      (case (car i)
		((:gvec) (set! heap-type 0)
			 (loop (cdr i)))
		((:bvec) (set! heap-type 1)
			 (loop (cdr i)))
		((:immob) (set! heap-type 2)
			  (loop (cdr i)))
		((:abstract) (set! heap-type 3)
			     (loop (cdr i)))
		((:weak1) (set! heap-type 4)
			  (loop (cdr i)))
		((heap-type:) (set! heap-type (cadr i))
			      (assert (fixnum? heap-type))
			      (loop (cddr i)))
		((image-mode:) (set! image-mode (cadr i))
			       (assert (fixnum? (cadr i)))
			       (loop (cddr i)))
		((prim-type:) (set! prim-type (cadr i))
			      (assert (symbol? (cadr i)))
			      (loop (cddr i)))
		((metaclass:) 
		 (set! metaclass (parse-type-expr (cadr i)
						  tl-envt
						  dyn-envt))
		 (loop (cddr i)))
		((class-category:)
		 (set! category (cadr i))
		 (assert (fixnum? category))
		 (loop (cddr i)))
		(else
		 (set! new-slots 
		       (cons (parse-slot-descriptor supers 
						  (car i) 
						  '() 
						  tl-envt
						  dyn-envt)
			     new-slots))
		 (loop (cdr i))))
	      (if (pair? (car i))
		  (begin
		    (assert (symbol? (caar i)))
		    (set! new-slots
			  (cons (parse-slot-descriptor supers 
						       (caar i) 
						       (cdar i) 
						       tl-envt
						       dyn-envt)
				new-slots))
		    (loop (cdr i)))
		  (error/syntax "Illegal slot-descriptor: ~s" (car i))))))
    ;; make sure we only specify a category if there isn't one for
    ;; the parent; if not specified, inherit the parent's
    (if category
	(if (pair? supers)
	    (if (not (eq? (class-category (actual-value (car supers))) 0))
		(error/syntax "Parent class already has a category")))
	(if (null? supers)
	    (set! category 0)
	    (set! category (class-category (actual-value (car supers))))))
    ;; make image-mode: 1 be implicit for bvec's 
    (if (not image-mode)
	(if (eq? heap-type 1)
	    (set! image-mode 1)))
    (let* ((num-new-slots (length new-slots))
	   (base-size (if (null? supers)
			  0
			  (instance-size (actual-value (car supers)))))
	   (c (make metaclass
		    class-name: name
		    heap-type: heap-type
		    image-mode: image-mode
		    superclasses: supers
		    direct-slots: (reverse new-slots)
		    instance-size: (+ num-new-slots base-size)
		    corresponding-primtype: prim-type
		    class-precedence-list: '()
		    class-category: category
		    class-hash: (class-name->hash name)
		   ;; here is the sort of thing
		   ;; you can't do at compile time,
		   ;; because we can't have just 
		   ;; arbitrary pointers into
		   ;; imported modules' images
		    all-slots: #f)))
      ;; fix up the slots' indexes
      (for-each (lambda (slot i)
		  (set-index! slot (+ base-size i)))
		(reverse new-slots)
		(range num-new-slots))
      ;; create the setters 'n' getters
      (for-each (lambda (slot)
		  (if (setter slot)
		      (set-setter! 
		       slot
		       (create-setter-method slot c tl-envt dyn-envt)))
		  (if (getter slot)
		      (set-getter! 
		       slot
		       (create-getter-method slot c tl-envt dyn-envt))))
		new-slots)
      ;
      (finalize-class c)
      (install-tl-def name 
		      tl-envt
		      dyn-envt 
		      (make <top-level-var>
			    name: name
			    value: c
			    write-prot: #t))
      name)))

;; at this point,
;; method-form is just the argument decls and the body
;; e.g.   (((self <pair>) port) 
;;         (format port "(~s . ~s)" (car self) (cdr self)))
;;


(define (compile-the-method gf-name method-form envt dyn-envt)
  (let ((name gf-name)
	(formals (trust-me-for-dispatched-args (car method-form)))
	(dispatch-on (cadr (caar method-form)))
	(body (cdr method-form)))
    (if *compile-verbose*
	(format #t "compiling top-level method for ~s (~s)\n" 
		name dispatch-on))
    (let* ((cc (make-code-ctx (list (list 'function-scope
					  name
					  dispatch-on))))
	   (ic (compile/procedure name formals body envt dyn-envt))
	   (asm (procedure->aml ic '() cc)))
      (if *show-aml*
	  (show-aml asm))
      (make <method>
	    template: (aml->template asm cc)
	    function-specializers: (compute-specializers formals envt dyn-envt)
	    environment: #f))))

;; currently, only the first argument is dispatched, so 
;; that's the only one that's trusted

(define (trust-me-for-dispatched-args formals)
  (let ((dispatched (car formals)))
    (cons (list (car dispatched)
		(cadr dispatched)
		':trust-me)
	  (cdr formals))))

(define (create-getter-method (s <slot-descriptor>) 
			      (for <<class>>)
			      envt
			      dyn-envt)
  (let ((tm (make <getter>
		  template: #f
		  environment: (make-gvec <binding-envt>
					  #f
					  (index s))
		  slot-descriptor: s
		  type-restriction: (type-restriction s)
		  index: (index s)
		  function-specializers: (list for))))
    (register-implicit-method tm 'getter-template)
    (provide-method (getter s) envt dyn-envt tm)))

;; these functions return the <method> created for them

(define (create-setter-method (s <slot-descriptor>) 
			      (for <<class>>)
			      envt
			      dyn-envt)
  (let ((n (setter s))
	(t (type-restriction s))
	(check? (not (eq? (actual-value (type-restriction s))
			  (actual-value (xform (well-known '<object>)
					       'value))))))
    (if *compile-verbose*
	(if (not check?)
	    (format #t "setter method (~s) is unrestricted\n" n)
	    (format #t "setter method (~s) is restricted: ~s\n" n t)))
    (let ((tm (make <setter>
		    template: #f
		    environment: (if check?
				     (make-gvec <binding-envt>
						#f
						(index s)
						(type-restriction s))
				     (make-gvec <binding-envt>
						#f
						(index s)))
		    slot-descriptor: s
		    type-restriction: t
		    index: (index s)
		    function-specializers: (list for t))))
      (if check?
	  (register-implicit-method tm 'restricted-setter-template)
	  (register-implicit-method tm 'setter-template))
      (provide-method n envt dyn-envt tm))))

;;
;; this function arranges for a method to have it's 
;; `fn-template' pointer set later.  This is necessary
;; only because modules BEFORE objsys may want
;; setters & getters to be defined
;;
;; this should be optimized so that it set's the fn-template
;; to point directly to the value of the appropriate well-known
;; binding IFF such a well-known binding is available.

(define (register-implicit-method (m <method>) type)
  (set-template! m (value (well-known type))))

(define (provide-method gf-name envt dyn-envt this-method)
  (let ((bdg (lookup-aliased gf-name envt dyn-envt)))
    ;;
    (if (not bdg)
	(set! bdg (create-default-gf gf-name envt dyn-envt))
	;; must be a module variable, not imported...
	(if (and (instance? bdg <top-level-var>)
		 (eq? (value bdg) '#unbound))
	    (set-value! bdg (make-default-gf gf-name))))
    ;;
    ;; make sure it's a GF
    ;;
    (if (or (not (instance? (actual-bdg bdg) 
			    <top-level-var>))
	    (not (instance? (value (actual-bdg bdg)) 
			    <generic-function>)))
	(error/semantic "binding for ~s is not a generic function"
			gf-name))
    (target-add-method (xform bdg 'value) this-method)
    this-method))
    
(define (compile-tl-define-method tl-form tl-envt dyn-envt)
  (let* ((tl-form (install-next-method-syntax tl-form))
	 (name (cadr tl-form)))
    (if *compile-verbose*
	(format #t "compiling top-level method on: ~s\n" name))
    (provide-method name
		    tl-envt
		    dyn-envt
		    (compile-the-method name
					(cddr tl-form)
					tl-envt
					dyn-envt))
    name))

;;
;; rewrite a `define-method' expr to supply some syntax for `next-method'
;; to the body of the method
;;

(define (install-next-method-syntax form)
  ;;
  ;; syntactically convert a formals list
  ;; into a call to an expr.  If the formals
  ;; contains a #rest, then the combo will involve 'apply*'
  ;;
  (define (recombo head formals)
    (let* ((apply? #f)
	   (f (process-formals$ formals
				(lambda () '())
				(lambda (last)
				  (set! apply? #t)
				  (list last))
				(lambda (item rest)
				  (cons (if (pair? item)
					    (car item)
					    item)
					rest)))))
      (if apply?
	  (append '(apply*) f (list head))
	  (cons head f))))
  ;;
  (let ((gf (cadr form))
	(args (caddr form))
	(body (cdddr form)))
    (let ((fnext (list '(exported-value objsys find-next-method-1)
		       gf 
		       (cadr (car args)))))
      `(define-method ,gf ,args
	 (let-syntax ((next-method (syntax-form () 
				     ,(recombo fnext args))
				   (syntax-form stuff
				     (,fnext . stuff))
				   (else
				    (lambda args
				      (if (null? args)
					  ,(recombo fnext args)
					  (apply* args ,fnext))))))
	   ,@body)))))

(define (make-default-gf name)
  (let ((gf (make <generic-function>
		  ;;
		  ;; the fn-template will get filled in later
		  ;;
		  template: #f
		  generic-function-name: name
		  function-specializers: (xform (well-known '<object>) 'value)
		  generic-function-methods: '())))
    ;;
    ;; set up the envt of the GF to bind two things, the
    ;; first of which is a pointer back to the GF itself.
    ;; (the other can be used for caching things by the GF
    ;;  dispatcher)
    ;;
    (finalize-generic-function gf)
    gf))

(define (create-default-gf name envt dyn-envt)
  (let ((gf (make-default-gf name)))
    (install-tl-def name
		    envt
		    dyn-envt
		    (make <top-level-var>
			  name: name 
			  value: gf
			  write-prot: #t))))

(define (compile-tl-define-generic-function tl-form tl-envt dyn-envt)
  (let ((name (cadr tl-form)))
    (if *compile-verbose*
	(format #t "compiling top-level generic function: ~s\n" name))
    (create-default-gf name tl-envt dyn-envt)
    name))


(define (parse-supers super-names lex-envt dyn-envt)
  (map (lambda (n)
	 (parse-type-expr n lex-envt dyn-envt))
       super-names))


;; split a keywords list into two parts,
;; the first part is the list of slot initializing keywords,
;; in order of appearance in the class.
;; the second part is all the rest, in the order of initial occurrence
;;
;; the kw-list is in <icode> form, as it has been returned from
;; compile-keyword-list

(define (split-keywords-list the-class kw-list)
    (let ((inits (make-vector (instance-size the-class) #f))
    	  (all-slots (slot-descriptors the-class))
	  (others '()))
	(let loop ((kw kw-list))
	    (if (null? kw)
	    	;; make sure required slots are supplied
		;; and check slot types where possible
		(begin
		    (finalize-initial-values inits all-slots)
		    (cons (vector->list inits)
			  (reverse others)))
		;; we know that (car kw) is an <ic-const>
		;; and (cadr kw) exists and is some expr
		(let ((nm (value (car kw))))
		    (let ((sd (find-slot-with-init-kwd nm all-slots)))
			(if sd
			    (if (vector-ref inits (index sd))
				(error/semantic
				    "slot ~s multiply initialized"
				    nm)
				(vector-set! inits (index sd) (cadr kw)))
			    (set! others (cons (cadr kw) 
					       (cons (car kw) others))))
			(loop (cddr kw))))))))
				
(define (find-slot-with-init-kwd kwd all-slots)
    (let loop ((i all-slots))
	(if (null? i)
	    #f
	    (if (eq? (init-keyword (car i)) kwd)
		(car i)
		(loop (cdr i))))))

(define (finalize-initial-values inits all-slots)
  (for-each
   (lambda ((sd <slot-descriptor>))
     (let ((initer (vector-ref inits (index sd)))
	   (mode (initialization-mode sd)))
       (if initer
	   (if (eq? mode 'prohibited)
	       (error/semantic
		"prohibited slot specified: ~s"
		(name sd))
	       (vector-set! inits
			    (index sd)
			    (coerced-expr initer
					  (type-restriction sd))))
	   (if (eq? mode 'required)
	       (error/semantic
		"required slot not specified: ~s"
		(name sd))
	       (vector-set!
		inits
		(index sd)
		(init-value sd))))))
   all-slots))


;; this function is responsible for locating bindings for the special
;; names that are known to the compiler, which are usually primops
;; or special cooperation between the compiler and the runtime system,
;; like make-instance

(define (tl-ref-well-known name)
  (let ((b (well-known name)))
    (compile-ref (actual-bdg b) b *basis* *basis* 'value)))


;;

(define (target-expr-value expr lex-envt dyn-envt)
  (eval-in-envt expr lex-envt))
