/*
 * ULCNTL.C - control structure functions for Ultra
 *
 * Source Version: 4.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "ultra.h"
#ifdef HAVE_JPEGLIB
#include "gsrast.h"
#endif

int
 UL_plot_date = TRUE,
 UL_simple_append = FALSE,
 UL_N_Curves;

static void
 SC_DECLARE(UL_draw_plot, (PG_device *dev)),
 SC_DECLARE(_UL_draw_plot, (PG_device *dev)),
 SC_DECLARE(_UL_print_curve_labels, 
            (PG_device *dev, int *mark)),
 SC_DECLARE(_UL_fix_text_buf, 
            (PG_text_box *fileBox, PG_text_box *screenBox));


/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_INIT_HASH - set up the Ultra hash table */

void UL_init_hash()
   {

    UL_OBJECT_S  = SC_strsavef("object", "char*:UL_INIT_HASH:object");

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_INSTALL_GLOBAL_VARS - install the global variables */

void UL_install_global_vars()
   {

    SX_install_global_vars();

    SS_install_cf("ascii-output-format",
                  "Variable: Controls format for ASCII output of floating point numbers\n     Usage: ascii-output-format <format>",
                  SS_acc_ptr,
                  &UL_ascii_output_format);

    SS_install_cf("data-directory",
                  "Variable: Sets the default data directory\n     Usage: data-directory <path-name>",
                  SS_acc_string,
                  UL_data_directory);

    SS_install_cf("default-npts",
                  "Variable: The default number of points used to make new curves.\n     Used by SPAN, LINE, etc.",
                  SS_acc_int,
                  &UL_default_npts);

    SS_install_cf("derivative-tolerance",
                  "Variable: Threshold for first derivative change.\n     Usage: derivative-tolerance <real>",
                  SS_acc_REAL,
                  &UL_derivative_tolerance);

    SS_install_cf("display-name",
                  "Variable: Display-name - <host>:<display>.<screen>\n     Usage: display-name <string>",
                  SS_acc_ptr,
                  &UL_display_name);

    SS_install_cf("display-type",
                  "Variable: Controls display mode\n     Usage: display-type <string>",
                  SS_acc_ptr,
                  &UL_display_type);

    SS_install_cf("display-title",
                  "Variable: Controls display title\n     Usage: display-title <string>",
                  SS_acc_ptr,
                  &UL_display_title);

/* KLMN */

    SS_install_cf("n-curves",
                  "Variable: Total number of curves in the system\n     Usage: n-curves",
                  SS_acc_int,
                  &UL_N_Curves);

    SS_install_cf("n-curves-read",
                  "Variable: Total number of curves read from files\n     Usage: n-curves-read",
                  SS_acc_int,
                  &UL_n_curves_read);

/* OPQR */

    SS_install_cf("plot-date",
                  "Variable: Print date and time on hardcopy if TRUE\n     Usage: plot-date on | off",
                  SS_acc_int,
                  &UL_plot_date);


/* STUV */

    SS_install_cf("simple-append",
                  "Variable: Do simple concatenations with append-curves\n     Usage: simple-append [ on | off]",
                  SS_acc_int,
                  &UL_simple_append);

    SS_install_cf("view-aspect",
                  "Variable: Viewport aspect ratio\n     Usage: view-aspect <real>",
                  SS_acc_REAL,
                  &UL_view_aspect);

    SS_install_cf("view-height",
                  "Variable: Viewport height in fraction of screen height\n     Usage: view-height <real>",
                  SS_acc_REAL,
                  &UL_view_height);

    SS_install_cf("view-origin-x",
                  "Variable: X comp of viewport origin (frac of window width)\n     Usage: view-origin-x <real>",
                  SS_acc_REAL,
                  &UL_view_x);

    SS_install_cf("view-origin-y",
                  "Variable: Y comp of viewport origin (frac of window width)\n     Usage: view-origin-y <real>",
                  SS_acc_REAL,
                  &UL_view_y);

    SS_install_cf("view-width",
                  "Variable: Viewport width in fraction of screen width\n     Usage: view-width <real>",
                  SS_acc_REAL,
                  &UL_view_width);

/* WXYZ */

    SS_install_cf("window-height",
                  "Variable: Window height in fraction of screen width\n     Usage: window-height <real>",
                  SS_acc_REAL,
                  &UL_window_height);

    SS_install_cf("window-origin-x",
                  "Variable: X comp of window origin (frac of screen width)\n     Usage: window-origin-x <real>",
                  SS_acc_REAL,
                  &UL_window_x);

    SS_install_cf("window-origin-y",
                  "Variable: Y comp of window origin (frac of screen width)\n     Usage: window-origin-y <real>",
                  SS_acc_REAL,
                  &UL_window_y);

    SS_install_cf("window-width",
                  "Variable: Window width in fraction of screen width\n     Usage: window-width <real>",
                  SS_acc_REAL,
                  &UL_window_width);

    SS_install_cf("window-height-PS",
                  "Variable: PostScript window height in fraction of screen width\n     Usage: window-height-PS <real>",
                  SS_acc_REAL,
                  &UL_window_height_PS);

    SS_install_cf("window-origin-x-PS",
                  "Variable: PostScript X comp of window origin (frac of screen width)\n     Usage: window-origin-x-PS <real>",
                  SS_acc_REAL,
                  &UL_window_x_PS);

    SS_install_cf("window-origin-y-PS",
                  "Variable: PostScript Y comp of window origin (frac of screen width)\n     Usage: window-origin-y-PS <real>",
                  SS_acc_REAL,
                  &UL_window_y_PS);

    SS_install_cf("window-width-PS",
                  "Variable: PostScript window width in fraction of screen width\n     Usage: window-width-PS <real>",
                  SS_acc_REAL,
                  &UL_window_width_PS);

    SS_install_cf("window-height-CGM",
                  "Variable: CGM window height in fraction of screen width\n     Usage: window-height-CGM <real>",
                  SS_acc_REAL,
                  &UL_window_height_CGM);

    SS_install_cf("window-origin-x-CGM",
                  "Variable: CGM X comp of window origin (frac of screen width)\n     Usage: window-origin-x-CGM <real>",
                  SS_acc_REAL,
                  &UL_window_x_CGM);

    SS_install_cf("window-origin-y-CGM",
                  "Variable: CGM Y comp of window origin (frac of screen width)\n     Usage: window-origin-y-CGM <real>",
                  SS_acc_REAL,
                  &UL_window_y_CGM);

    SS_install_cf("window-width-CGM",
                  "Variable: CGM window width in fraction of screen width\n     Usage: window-width-CGM <real>",
                  SS_acc_REAL,
                  &UL_window_width_CGM);

#ifdef HAVE_JPEGLIB
    SS_install_cf("window-height-JPEG",
                  "Variable: JPEG window height in fraction of screen width\n     Usage: window-height-JPEG <real>",
                  SS_acc_REAL,
                  &UL_window_height_JPEG);

    SS_install_cf("window-origin-x-JPEG",
                  "Variable: JPEG X comp of window origin (frac of screen width)\n     Usage: window-origin-x-JPEG <real>",
                  SS_acc_REAL,
                  &UL_window_x_JPEG);

    SS_install_cf("window-origin-y-JPEG",
                  "Variable: JPEG Y comp of window origin (frac of screen width)\n     Usage: window-origin-y-JPEG <real>",
                  SS_acc_REAL,
                  &UL_window_y_JPEG);

    SS_install_cf("window-width-JPEG",
                  "Variable: JPEG window width in fraction of screen width\n     Usage: window-width-JPEG <real>",
                  SS_acc_REAL,
                  &UL_window_width_JPEG);
#endif

    return;}

/*--------------------------------------------------------------------------*/

/*                          DOCUMENTATION FUNCTIONS                         */

/*--------------------------------------------------------------------------*/

/* UL_DESCRIBE - the Ultra documentation function */

object *UL_describe(argl)
   object *argl;
   {object *obj;

    UL_prep_arg(argl);

    for ( ; !SS_nullobjp(argl); argl = SS_cdr(argl))
        {obj = SS_car(argl);
         if (obj != NULL)
            {if (!SS_prim_des(SS_outdev, obj))
                PRINT(stdout, " Unknown function\n");};};

    UL_pause(TRUE);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_APROPOS - finds all commands which contain the substring s in their 
 *            - name or description
 */

object *UL_apropos(obj)
   object *obj;
   {int flag;
    char *s;

    s = UL_get_string(obj);
    SC_banner("APROPOS");
    PRINT(stdout, "Apropos: %s\n\n", s);

/* search SCHEME hash table */
    flag = SS_prim_apr(stdout, s, SS_symtab);
    if (!flag)
       PRINT(stdout, "No documentation on %s\n", s);

    UL_pause(TRUE);

    return(SS_f);}

/*--------------------------------------------------------------------------*/

/*                         CURVE PLOTTING ROUTINES                          */

/*--------------------------------------------------------------------------*/

/* UL_PLOT - main plotting routine
 *         - check the current mode against mode and reinit if different
 *         - before plotting
 */

object *UL_plot()
   {static REAL last_state = 0.0;

    if (UL_graphics_device != NULL)
       {if (last_state != SX_label_space)
           {last_state = SX_label_space;
            UL_mode_text();
            UL_mode_graphics();};

        UL_set_graphics_state(UL_graphics_device);
        UL_draw_plot(UL_graphics_device);};

    PG_make_device_current(PG_console_device);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_PRINTSCR - dump the screen to the printer */

object *UL_printscr()
   {double width, height;

    if (SX_ps_flag == TRUE)
       {if (UL_PS_device == NULL)
           {if ((SX_ps_name == NULL) || (strlen(SX_ps_name) == 0))
               SS_error("PS-NAME IS NULL - UL_PRINTSCR", SS_null);
            if ((SX_ps_type == NULL) || (strlen(SX_ps_type) == 0))
               SS_error("PS-TYPE IS NULL - UL_PRINTSCR", SS_null);
            UL_PS_device = PG_make_device("PS", SX_ps_type, SX_ps_name);
            UL_set_graphics_state(UL_PS_device);
            UL_PS_device = PG_open_device(UL_PS_device,
					  UL_window_x_PS,
					  UL_window_y_PS,
					  UL_window_width_PS,
					  UL_window_height_PS);

            if (UL_PS_device == NULL)
               SS_error("CAN'T OPEN POSTSCRIPT FILE - UL_PRINTSCR", SS_null);};

/* the set graphics state operation is necessary after PG_open_device
 * because a set window is done then which overrides gymin, gymax, etc
 */
        width  = UL_PS_device->window_width;
        height = UL_PS_device->window_height;
        UL_set_graphics_state(UL_PS_device);
        UL_PS_device->window_width  = width;
        UL_PS_device->window_height = height;

        UL_draw_plot(UL_PS_device);};

    if (SX_cgm_flag == TRUE)
       {if (UL_CGM_device == NULL)
           {if ((SX_cgm_name == NULL) || (strlen(SX_cgm_name) == 0))
               SS_error("CGM-NAME IS NULL - UL_PRINTSCR", SS_null);
            if ((SX_cgm_type == NULL) || (strlen(SX_cgm_type) == 0))
               SS_error("CGM-TYPE IS NULL - UL_PRINTSCR", SS_null);
            UL_CGM_device = PG_make_device("CGM", SX_cgm_type, SX_cgm_name);
            UL_set_graphics_state(UL_CGM_device);
            UL_CGM_device = PG_open_device(UL_CGM_device,
					   UL_window_x_CGM,
					   UL_window_y_CGM,
					   UL_window_width_CGM,
					   UL_window_height_CGM);

            if (UL_CGM_device == NULL)
               SS_error("CAN'T OPEN CGM FILE - UL_PRINTSCR", SS_null);};

/* the set graphics state operation is necessary after PG_open_device
 * because a set window is done then which overrides gymin, gymax, etc
 */
        width  = UL_CGM_device->window_width;
        height = UL_CGM_device->window_height;
        UL_set_graphics_state(UL_CGM_device);
        UL_CGM_device->window_width  = width;
        UL_CGM_device->window_height = height;

        UL_draw_plot(UL_CGM_device);};

#ifdef HAVE_JPEGLIB
    if (SX_jpeg_flag == TRUE)
       {if (UL_JPEG_device == NULL)
           {if ((SX_jpeg_name == NULL) || (strlen(SX_jpeg_name) == 0))
               SS_error("JPEG-NAME IS NULL - UL_PRINTSCR", SS_null);
            if ((SX_jpeg_type == NULL) || (strlen(SX_jpeg_type) == 0))
               SS_error("JPEG-TYPE IS NULL - UL_PRINTSCR", SS_null);
            UL_JPEG_device = PG_make_device("JPEG", SX_jpeg_type, SX_jpeg_name);
            UL_set_graphics_state(UL_JPEG_device);
            UL_JPEG_device = PG_open_device(UL_JPEG_device,
					    UL_window_x_JPEG,
					    UL_window_y_JPEG,
					    UL_window_width_JPEG,
					    UL_window_height_JPEG);

            if (UL_JPEG_device == NULL)
               SS_error("CAN'T OPEN POSTSCRIPT FILE - UL_PRINTSCR", SS_null);};

/* the set graphics state operation is necessary after PG_open_device
 * because a set window is done then which overrides gymin, gymax, etc
 */
        width  = UL_JPEG_device->window_width;
        height = UL_JPEG_device->window_height;
        UL_set_graphics_state(UL_JPEG_device);
        UL_JPEG_device->window_width  = width;
        UL_JPEG_device->window_height = height;

        UL_draw_plot(UL_JPEG_device);};
#endif

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_DRAW_PLOT - coordinate all of the plot drawing */

static void UL_draw_plot(dev)
   PG_device *dev;

   {

    PG_set_finish_state(dev, FALSE);

/* draw the curves */
    _UL_draw_plot(dev);

/* update the interface for a hardcopy device
 * the tricky part is mapping the coordinate systems of the different
 * devices - I got lazy and let the read/write interface routines
 * do it at the cost of a temporary file
 */
    if (dev != UL_graphics_device)
       {int i, niobs;
	PG_interface_object **piobs, **iobs, *piob, *iob;
	PG_text_box *b, *pb;

/* remove any existing interface objects */
	niobs = dev->n_interface_objects;
        iobs  = dev->interface_objects;
        for (i = 0; i < niobs; i++)
            _PG_rl_interface_object(iobs[i], TRUE);

	dev->n_interface_objects = 0;
        dev->interface_objects   = NULL;
        
/* GOTCHA: this has to be reworked for annot to work in -s mode */
        if (UL_graphics_device != NULL)
	   {PG_write_interface(UL_graphics_device, "tmp.pui");
	    PG_read_interface(dev, "tmp.pui");
	    SC_remove("tmp.pui");

	    piobs = UL_graphics_device->interface_objects;
	    iobs  = dev->interface_objects;
	    niobs = dev->n_interface_objects;
	    for (i = 0; i < niobs; i++)
	        {iob  = iobs[i];
                 piob = piobs[i];
                 if (strcmp(iob->type, PG_TEXT_OBJECT_S) == 0)
		    {b  = (PG_text_box *) iob->obj;
                     pb = (PG_text_box *) piob->obj;
                     b->border   = pb->border;
                     b->align    = pb->align;
                     b->angle    = pb->angle;
                     b->standoff = pb->standoff;
                     if ((b->n_lines  != pb->n_lines) || (b->n_chars_line != pb->n_chars_line))
                        _UL_fix_text_buf(b, pb);};};

	    PG_draw_interface_objects(dev);};}

    else
       PG_draw_interface_objects(dev);

#if 0
/* look up the value of the annotation list */
    b = SS_lk_var_val(UL_ann_lst, SS_Env);
    if (!SS_nullobjp(b))
       {for (; SS_consp(b); b = SS_cdr(b))
            {argl = SS_car(b);

             x     = 0.0;
             y     = 0.0;
             text  = NULL;
             color = dev->WHITE;
             SS_args(argl,
                     SC_DOUBLE_I, &x,
                     SC_DOUBLE_I, &y,
                     SC_STRING_I, &text,
                     SC_INTEGER_I, &color,
                     0);

             StoW(dev, x, y);

/* if log axis options are on exponentiate here for log there */
             if (dev->ifxlog)
                x = POW(10.0, x);
 
             if (dev->ifylog)
                y = POW(10.0, y);

             PG_set_text_color(dev, color);
             PG_write_WC(dev, x, y, text);};};
#endif
                 
    PG_finish_plot(dev);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _UL_FIX_TEXT_BUF - fix discrepancy in text buffer size between
 *                  - screen device and output file
 */

static void _UL_fix_text_buf(fileBox, screenBox)
   PG_text_box *fileBox, *screenBox;
   {int n_lines, n_chars, i;
    char **bf;

    n_lines = (fileBox->n_lines > screenBox->n_lines) ?
               fileBox->n_lines : screenBox->n_lines;
    n_chars = (fileBox->n_chars_line > screenBox->n_chars_line) ?
               fileBox->n_chars_line : screenBox->n_chars_line;

/* allocate the space */
    bf = FMAKE_N(char *, n_lines, "_UL_FIX_TEXT_BUF:bf");
    for (i = 0; i < n_lines; i++)
        bf[i] = FMAKE_N(char, n_chars,
                        "_UL_FIX_TEXT_BUF:bf[]");

/* copy the contents
 * don't copy more strings than we've got; free up old string space
 */
    for (i = 0; i < fileBox->n_lines; i++)
        {strcpy(bf[i], fileBox->text_buffer[i]);
	 SFREE(fileBox->text_buffer[i]);}

    fileBox->text_buffer  = bf;
    fileBox->n_lines      = n_lines;
    fileBox->n_chars_line = n_chars;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _UL_DRAW_PLOT - main plot control routine */

static void _UL_draw_plot(dev)
   PG_device *dev;
   {int i, color, first, *plc, *ppt;
    int ptyp;
    curve *data;
    pcons *info;
    REAL xmx, xmn, ymx, ymn;

    first = TRUE;

    PG_make_device_current(dev);
    PG_clear_window(dev);

    UL_plot_limits(dev, TRUE, &xmn, &xmx, &ymn, &ymx);
    PG_set_window(dev, xmn, xmx, ymn, ymx);

/* plot all of the current curves */
    data = UL_dataset;
    for (i = 0; i < UL_N_Curves; i++)
        {if (data[i].id != ' ')
            {info = data[i].info;
             SC_assoc_info(info,
			   "PLOT-TYPE", &ppt,
			   "LINE-COLOR", &plc,
			   NULL);
	     color = (plc != NULL) ? *plc : dev->BLUE;
	     ptyp  = (ppt != NULL) ? *ppt : CARTESIAN;
             if (color >= 0)
                {if ((_PG_plot_type != CARTESIAN) ||
		     ((ptyp != HISTOGRAM) && (ptyp != ERROR_BAR)))
		    {info = PG_set_plot_type(info, _PG_plot_type, _PG_axis_type);};

                 if (first)
                    {first = FALSE;
                     PG_axis(dev, _PG_axis_type);};

                 PG_plot_curve(dev, data[i].xp, data[i].yp, data[i].n,
                               info, 1);

                 if ((dev->data_id == TRUE) && (_PG_plot_type != INSEL))
                    {PG_set_text_color(dev, color);
                     PG_draw_data_ids(dev, data[i].xp, data[i].yp,
                                      data[i].n, data[i].id, info);
                     PG_set_text_color(dev, dev->WHITE);};};};};


    if (first)
       {first = FALSE;
	PG_axis(dev, _PG_axis_type);};

    if (PG_plot_labels)
       _UL_print_curve_labels(dev, UL_data_index);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* UL_PLOT_LIMITS - return the current plot limits */

void UL_plot_limits(dev, pflg, pxmn, pxmx, pymn, pymx)
   PG_device *dev;
   int pflg;
   REAL *pxmn, *pxmx, *pymn, *pymx;
   {int i, j, color, *plc;
    curve *data;
    REAL xmax, xmin, ymax, ymin;
    REAL xmx, xmn, ymx, ymn;
    REAL *rc, *tc, rco, tco, xco, yco;
    REAL HC;

    HC = HUGE_REAL*HUGE_REAL*HUGE_REAL;

    xmin = HUGE_REAL;
    ymin = HUGE_REAL;
    xmax = -HUGE_REAL;
    ymax = -HUGE_REAL;

    data = UL_dataset;
    for (i = 0; i < UL_N_Curves; i++)
        {if (data[i].id != ' ')
            {SC_assoc_info((pcons *) data[i].info,
			   "LINE-COLOR", &plc,
			   NULL);
	     color = (plc != NULL) ? *plc : 1;
             if ((color >= 0) || (PG_hide_rescale != TRUE))
                {xmn = data[i].xmin;
                 xmx = data[i].xmax;
                 ymn = data[i].ymin;
                 ymx = data[i].ymax;

                 if ((xmn > HC) || (xmx > HC) || (ymn > HC) || (ymx > HC) ||
                     (xmn < -HC) || (xmx < -HC) || (ymn < -HC) || (ymx < -HC))
                    continue;

                 xmin = min(xmin, xmn);
                 xmax = max(xmax, xmx);
                 ymin = min(ymin, ymn);
                 ymax = max(ymax, ymx);};};};

    if (xmin >= xmax)
       {xmin = 0.0;
        xmax = 1.0;};

    if ((ymin == ymax) && (ymin != 0.0))
       {double dy;

        dy = ABS(0.01*(ymin + ymax));
        ymin -= dy;
        ymax += dy;}

    else if (ymin >= ymax)
       {ymin = 0.0;
        ymax = 1.0;}

    if (dev == NULL)
       {if (SX_autorange == TRUE)
	   {ymx = ymax;
	    ymn = ymin;}
	else
	   {ymx = UL_gymax;
	    ymn = UL_gymin;};

	if (SX_autodomain == TRUE)
	   {xmx = xmax;
	    xmn = xmin;}
	else
	   {xmx = UL_gxmax;
	    xmn = UL_gxmin;};}

    else
       {PG_get_viewport_WC(dev, &xmn, &xmx, &ymn, &ymx);

	if (SX_autorange == TRUE)
	   {ymx = ymax;
	    ymn = ymin;};

	if (SX_autodomain == TRUE)
	   {xmx = xmax;
	    xmn = xmin;};};

    if (pflg && (_PG_plot_type == POLAR))
       {xmin = HUGE_REAL;
        ymin = HUGE_REAL;
        xmax = -HUGE_REAL;
        ymax = -HUGE_REAL;
        for (i = 0; i < UL_N_Curves; i++)
            {if (data[i].id != ' ')
                {rc = data[i].yp;
                 tc = data[i].xp;
                 for (j = 0; j < data[i].n; j++)
                     {rco = rc[j];
                      tco = tc[j];
                      if (((tco < xmn) || (xmx < tco)) ||
                          ((rco < ymn) || (ymx < rco)))
                         continue;
                      xco = rco*cos(tco);
                      yco = rco*sin(tco);
                      xmin = min(xmin, xco);
                      xmax = max(xmax, xco);
                      ymin = min(ymin, yco);
                      ymax = max(ymax, yco);};};};
        xmn = xmin - 0.005;
        xmx = xmax + 0.005;
        ymn = ymin - 0.005;
        ymx = ymax + 0.005;};

    *pxmn = xmn;
    *pxmx = xmx;
    *pymn = ymn;
    *pymx = ymx;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _UL_PRINT_CURVE_LABELS - print the curve labels at the bottom
 *                        - of the screen/page
 */

static void _UL_print_curve_labels(dev, mark)
   PG_device *dev;
   int *mark;
   {int i, j, color, size, id, md, clp;
    curve *data;
    REAL xmn, xmx, ymn, ymx, dxt, dyt, dxtfile, dsxf;
    REAL yo, xolbl, xoxmn, xoxmx, xoymn, xoymx, xofil;
    REAL xsn, xsx, ysn, ysx, dsx, dsy, xlog, ylog;
    REAL xb, yb;
    char label[MAXLINE], *face, *style, *s;

    if (!HARDCOPY_DEVICE(dev) && (SX_label_space <= 0.0))
       return;

    PG_get_clipping(dev, &clp);
    PG_get_font(dev, &face, &style, &size);
    PG_get_text_color(dev, &color);
    PG_set_text_color(dev, dev->WHITE);

    xlog = dev->ifxlog;
    ylog = dev->ifylog;

    PG_set_axis_log_scale(dev, FALSE, FALSE);
    PG_set_clipping(dev, FALSE);

    PG_get_bound(dev, &xmn, &xmx, &ymn, &ymx);

    if (HARDCOPY_DEVICE(dev))
       {PtoS(dev, 0, 0, xb, yb);
        StoW(dev, xb, yb);
        if (UL_plot_date)
           {PG_set_font(dev, face, style, 6);
            memset(label, '\0', MAXLINE);
	    s = SC_date();
            strcpy(label, s);
	    SFREE(s);
#ifdef UNIX
# ifndef SUNMOS
            label[strlen(label)] = ' ';
            gethostname(&label[strlen(label)], MAXLINE - strlen(label));
# endif
#endif
            PG_get_text_ext(dev, label, &dxt, &dyt);
            PG_write_WC(dev, xmx - dxt, ymx + 3.0*dyt, label);};}

    PG_get_viewport_WC(dev, &xmn, &xmx, &ymn, &ymx);
    PG_set_font(dev, face, style, SX_label_type_size);

    xsn = 0.0;
    xsx = 0.992;
    if (HARDCOPY_DEVICE(dev))
       {xsn = 0.03;
        xsx = 0.96;}
    ysn = 0.0;
    ysx = 0.0;

    data = UL_dataset;
    for (i = -1, j = 0, dxt = 0, dxtfile = 0; j < NDISPLAY; j++)
        {if (mark[j] >= 0)
            {i  = mark[j];
             if (data[i].file != NULL)
                {PG_get_text_ext_NDC(dev, data[i].file, &dxt, &dyt);
                 dxtfile = max(dxtfile, dxt);};};};
    if (i == -1)
       {PG_set_clipping(dev, clp);
        PG_set_text_color(dev, color);
        PG_set_axis_log_scale(dev, xlog, ylog);
        PG_set_font(dev, face, style, size);
        return;};

    PG_get_text_ext_NDC(dev, "FILE", &dxt, &dyt);
    dsxf  = dxt/2;
    dxt   = max(dxtfile, dxt);
    xofil = xsx;
    if (dxtfile > 0)
      xofil -= dxt;
    StoW(dev, xsn, ysn);
    StoW(dev, xofil, ysx);
    dxtfile /= dev->bxw_s;
    dsxf    /= dev->bxw_s;

    PG_get_text_ext(dev, " -0.00e+00", &dxt, &dyt);
    dsx   = dxt/2 - dsxf;
    xoymx = xofil - dxt;
    if (dxtfile > 0)
       xoymx -= dsxf/2;
    xoymn = xoymx - dxt;
    xoxmx = xoymn - dxt;
    xoxmn = xoxmx - dxt;

    PG_get_text_ext(dev, "CURVE", &dxt, &dyt);
    xolbl = xsn + dxt + dsxf/2;

    dsy = (ymx - ymn)*(1.0 + dev->botspace + dev->topspace);
    yo  = ymn - dev->botspace*(ymx - ymn) - 0.16*dsy;
    if (HARDCOPY_DEVICE(dev))
      yo  = ymn - dev->botspace*(ymx - ymn) - 0.11*dsy;

/* print the header */
    PG_write_WC(dev, xsn + dsxf/2, yo, "CURVE");
    PG_write_WC(dev, xolbl+dsx, yo, " LABEL");

    if ((SX_label_length < 40) || SX_squeeze_labels)
       {PG_write_WC(dev, xoxmn+dsx, yo, "XMIN");
        PG_write_WC(dev, xoxmx+dsx, yo, "XMAX");
        PG_write_WC(dev, xoymn+dsx, yo, "YMIN");
        PG_write_WC(dev, xoymx+dsx, yo, "YMAX");
        if (dxtfile > 0)
           PG_write_WC(dev, xofil+max((0.5*dxtfile)-dsxf, 0.), yo, "FILE");};

    PG_get_text_ext(dev, "   M  ", &dxt, &dyt);
    xolbl = xsn + dxt + dsxf/2;
    yo -= 2*dyt;

    for (j = 0; j < NDISPLAY; j++)
        {if (HARDCOPY_DEVICE(dev) && (yo <= yb))
#ifdef HAVE_JPEGLIB
	    if (!JPG_DEVICE(dev))
#endif
	       break;

         if (mark[j] >= 0)
            {int tc, *ptc;

	     i  = mark[j];
             id = data[i].id;
             md = data[i].modified;
             s  = data[i].text;

	     if (SX_label_color_flag == TRUE)
	        {SC_assoc_info(data[i].info,
			       "LINE-COLOR", &ptc,
			       NULL);
		 tc = (ptc != NULL) ? *ptc : dev->BLUE;
		 PG_set_text_color(dev, tc);};

/* prep the label text */
             strcpy(label, s);
             if (SX_squeeze_labels)
                 SC_squeeze_blanks(label);
             if (strlen(label) > SX_label_length)
                label[SX_label_length] = '\0';

             PG_write_WC(dev, xsn, yo, "    %c", id);
             if (md)
                 PG_write_WC(dev, xsn + dxt, yo, "*", id);
             PG_write_WC(dev, xolbl, yo, " %s", label);

             if ((SX_label_length < 40) || SX_squeeze_labels)
                {PG_write_WC(dev, xoxmn, yo, "%10.2e", data[i].xmin);
                 PG_write_WC(dev, xoxmx, yo, "%10.2e", data[i].xmax);
                 PG_write_WC(dev, xoymn, yo, "%10.2e", data[i].ymin);
                 PG_write_WC(dev, xoymx, yo, "%10.2e", data[i].ymax);

                 if (data[i].file != NULL)
                    PG_write_WC(dev, xofil, yo, "%s", data[i].file);};

#ifdef HAVE_JPEGLIB
	     if (JPG_DEVICE(dev))
	        yo -= 0.5*dyt;
#endif
	     yo -= dyt;};};

    PG_set_clipping(dev, clp);
    PG_set_text_color(dev, color);
    PG_set_axis_log_scale(dev, xlog, ylog);
    PG_set_font(dev, face, style, size);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_PAUSE - wait if apropriate before going on */

void UL_pause(pf)
   int pf;
   {if (pf)
       SX_plot_flag = FALSE;

    if (SS_interactive == ON)
       PRINT(stdout, "\n");

    return;}
    
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
