/*
 * SXPGS.C - PGS extensions in SX
 *
 * Source Version: 3.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "sx.h"

#define LABEL_TEXT_SIZE 30

REAL
 SX_window_x,
 SX_window_x_P,
 SX_window_y,
 SX_window_y_P,
 SX_window_width,
 SX_window_width_P,
 SX_window_height,
 SX_window_height_P;

extern void
 SC_DECLARE(SX_install_pgs_iob, (byte)),
 SC_DECLARE(SX_install_pgs_primitives, (byte));

static int
 *rendering_mode[4][6] =
    {{&SX_render_def, &SX_render_def, &SX_render_def,
      &SX_render_def, &SX_render_def, &SX_render_def},
     {&SX_render_def, &SX_render_1d_1d, &SX_render_def,
      &SX_render_def, &SX_render_def, &SX_render_def},
     {&SX_render_def, &SX_render_2d_1d, &SX_render_2d_2d,
      &SX_render_def, &SX_render_def, &SX_render_def},
     {&SX_render_def, &SX_render_def, &SX_render_def,
      &SX_render_def, &SX_render_def, &SX_render_def}};

#if 0

static char
 *SX_BW_PALETTE_S       = "bw",
 *SX_SPECTRUM_PALETTE_S = "spectrum";

static object
 *SC_DECLARE(_SX_get_mapping_info,
          (object *argl, PDBfile **file, char **pname));

#endif

static object
 *SC_DECLARE(SX_def_file_graph, (object *obj)),
 *SC_DECLARE(SX_devicep, (object *obj)),
 *SC_DECLARE(SX_graphp, (object *obj)),
 *SC_DECLARE(SX_imagep, (object *obj)),
 *SC_DECLARE(SX_dev_attributesp, (object *obj)),
 *SC_DECLARE(SX_make_device, (object *argl)),
 *SC_DECLARE(SX_open_device, (object *argl)),
 *SC_DECLARE(SX_close_device, (object *argl)),
 *SC_DECLARE(SX_draw_plot, (object *argl)),
 *SC_DECLARE(SX_make_pgs_graph, (object *argl)),
 *SC_DECLARE(SX_make_image, (object *argl)),
 *SC_DECLARE(SX_build_image, (object *argl)),
 *SC_DECLARE(SX_draw_image, (object *argl)),
 *SC_DECLARE(SX_device_props, (object *argl)),
 *SC_DECLARE(SX_menu_item_type, (object *argl)),
 *SC_DECLARE(SX_get_ascii_image_name, (object *arg)),
 *SC_DECLARE(SX_get_ascii_mapping_name, (object *arg)),
 *SC_DECLARE(SX_pdbdata_graph, (object *argl)),
 *SC_DECLARE(SX_pdbdata_image, (object *argl)),
 *SC_DECLARE(SX_graph_pdbdata, (object *argl)),
 *SC_DECLARE(SX_image_pdbdata, (object *argl)),
 *SC_DECLARE(SX_graph_pdbcurve, (object *argl)),
 *SC_DECLARE(_SX_pdbdata_graph, 
             (PDBfile *file, char *name, syment *ep)),
 *SC_DECLARE(_SX_pdbcurve_graph, 
             (PDBfile *file, char *name, syment *ep)),
 *SC_DECLARE(SX_set_view_angle, (object *argl)),
 *SC_DECLARE(SX_draw_domain, (object *argl)),
 *SC_DECLARE(SX_set_dom_limits, (object *argl)),
 *SC_DECLARE(SX_set_ran_limits, (object *argl)),
 *SC_DECLARE(SX_dom_extrema, (object *argl)),
 *SC_DECLARE(SX_ran_extrema, (object *argl)),
 *SC_DECLARE(SX_dom_limits, (object *argl)),
 *SC_DECLARE(SX_ran_limits, (object *argl)),
 *SC_DECLARE(SX_set_attr_graph, (object *argl)),
 *SC_DECLARE(SX_set_label, (object *argl)),
 *SC_DECLARE(SX_get_label, (object *obj));

static void
 SC_DECLARE(_SX_wr_ggraph, (object *obj, object *strm)),
 SC_DECLARE(_SX_rl_ggraph, (object *obj)),
 SC_DECLARE(_SX_wr_gimage, (object *obj, object *strm)),
 SC_DECLARE(_SX_rl_gimage, (object *obj)),
 SC_DECLARE(_SX_wr_gdevice, (object *obj, object *strm)),
 SC_DECLARE(_SX_rl_gdevice, (object *obj)),
 SC_DECLARE(_SX_wr_gdev_attr, (object *obj, object *strm)),
 SC_DECLARE(_SX_rl_gdev_attr, (object *obj)),
 SC_DECLARE(_SX_attach_rendering_1d,
         (PG_graph *data, int rendering, int axis_type, int start)),
 SC_DECLARE(_SX_set_limits, (char *t, PM_set *s, object *argl));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_INSTALL_PGS_FUNCS - install the PGS extensions to Scheme */
 
void SX_install_pgs_funcs()
   {
    SS_install("pdbdata->pg-graph",
               "Read a PGS graph object from a PDB file\nFORM (pdbdata->pg-graph <file> <name>)",
               SS_nargs,
               SX_pdbdata_graph, SS_PR_PROC);

    SS_install("pdbdata->pg-image",
               "Read a PGS image object from a PDB file\nFORM (pdbdata->pg-image <file> <name>)",
               SS_nargs,
               SX_pdbdata_image, SS_PR_PROC);

    SS_install("pg-build-image",
               "Build a PGS image object\nFORM (pg-build-image <data> <kmax> <lmax> [<name> <xmin> <xmax> <ymin> <ymax> <zmin> zmax>])",
               SS_nargs,
               SX_build_image, SS_PR_PROC);

    SS_install("pg-close-device",
               "Closes a PGS device",
               SS_nargs,
               SX_close_device, SS_PR_PROC);

    SS_install("pg-def-graph-file",
               "Set up a PDB file to recieve PGS graph objects",
               SS_sargs,
               SX_def_file_graph, SS_PR_PROC);

    SS_install("pg-device?",
               "Returns #t if the object is a PGS device, and #f otherwise",
               SS_sargs,
               SX_devicep, SS_PR_PROC);

    SS_install("pg-device-attributes?",
               "Returns #t if the object is a set of PGS device attributes, and #f otherwise",
               SS_sargs,
               SX_dev_attributesp, SS_PR_PROC);

    SS_install("pg-device-properties",
               "Return a list with NAME, TYPE, and TITLE of the given device",
               SS_nargs,
               SX_device_props, SS_PR_PROC);

    SS_install("pg-domain-extrema",
               "Return the domain extrema\nFORM (pg-domain-extrema <graph>)",
               SS_sargs,
               SX_dom_extrema, SS_PR_PROC);

    SS_install("pg-domain-limits",
               "Return the domain plotting limits\nFORM (pg-domain-limits <graph>)",
               SS_sargs,
               SX_dom_limits, SS_PR_PROC);

    SS_install("pg-draw-domain",
               "Draws the mesh specified by a PML set object\nFORM (pg-draw-domain <device> <set>)",
               SS_nargs,
               SX_draw_domain, SS_PR_PROC);

    SS_install("pg-draw-graph",
               "Draws a PGS graph object\nFORM (pg-draw-graph <device> <graph> <rendering> [...])",
               SS_nargs,
               SX_draw_plot, SS_PR_PROC);

    SS_install("pg-draw-image",
               "Draws a PGS image object\nFORM (pg-draw-image <device> <image>)",
               SS_nargs,
               SX_draw_image, SS_PR_PROC);

    SS_install("pg-graph?",
               "Returns #t if the object is a PGS graph, and #f otherwise",
               SS_sargs,
               SX_graphp, SS_PR_PROC);

    SS_install("pg-get-label",
               "Return the label of the specified mapping/graph/image/set",
               SS_sargs,
               SX_get_label, SS_PR_PROC);

    SS_install("pg-graph->pdbcurve",
               "Write an ULTRA curve object to a PDB file\nFORM (pg-graph->pdbcurve <curve> <file>)",
               SS_nargs,
               SX_graph_pdbcurve, SS_PR_PROC);

    SS_install("pg-graph->pdbdata",
               "Write a PGS graph object to a PDB file\nFORM (pg-graph->pdbdata <graph> <file>)",
               SS_nargs,
               SX_graph_pdbdata, SS_PR_PROC);

    SS_install("pg-grotrian-graph?",
               "Returns #t if the object is a PGS grotrian graph, and #f otherwise",
               SS_sargs,
               SX_grotrian_graphp, SS_PR_PROC);

    SS_install("pg-image?",
               "Returns #t if the object is a PGS image, and #f otherwise",
               SS_sargs,
               SX_imagep, SS_PR_PROC);

    SS_install("pg-image-name",
               "Return the image referenced by name or menu number",
               SS_nargs,
               SX_get_ascii_image_name, SS_PR_PROC);

    SS_install("pg-image->pdbdata",
               "Write a PGS image object to a PDB file\nFORM (pg-image->pdbdata <image> <file>)",
               SS_nargs,
               SX_image_pdbdata, SS_PR_PROC);

    SS_install("pg-make-device",
               "Returns a PGS device with NAME, TYPE, and TITLE",
               SS_nargs,
               SX_make_device, SS_PR_PROC);

    SS_install("pg-make-graph",
               "Return a PGS graph object\nFORM (pg-make-graph <domain> <range> [<color> <width> <style> <id>])",
               SS_nargs,
               SX_make_pgs_graph, SS_PR_PROC);

    SS_install("pg-make-image",
               "Make a PGS image object\nFORM (pg-make-image <data> <kmax> <lmax> [<name> <xmin> <xmax> <ymin> <ymax> <zmin> zmax>])",
               SS_nargs,
               SX_make_image, SS_PR_PROC);

    SS_install("pg-mapping-name",
               "Return the mapping referenced by name or menu number",
               SS_nargs,
               SX_get_ascii_mapping_name, SS_PR_PROC);

    SS_install("pg-menu-item-type",
               "Return the name of the type of the referenced menu item",
               SS_nargs,
               SX_menu_item_type, SS_PR_PROC);

    SS_install("pg-open-device",
               "Opens a PGS device DEV at (X, Y) with (DX, DY)",
               SS_nargs,
               SX_open_device, SS_PR_PROC);

    SS_install("pg-range-extrema",
               "Return the range extrema\nFORM (pg-range-extrema <graph>)",
               SS_sargs,
               SX_ran_extrema, SS_PR_PROC);

    SS_install("pg-range-limits",
               "Return the range plotting limits\nFORM (pg-range-limits <graph>)",
               SS_sargs,
               SX_ran_limits, SS_PR_PROC);

    SS_install("pg-set-domain-limits!",
               "Set the domain plotting limits for the given graph",
               SS_nargs,
               SX_set_dom_limits, SS_PR_PROC);

    SS_install("pg-set-graph-attribute!",
               "Set an attribute of the given graph",
               SS_nargs,
               SX_set_attr_graph, SS_PR_PROC);

    SS_install("pg-set-label!",
               "Set the graph/mapping/image/set label to the given string",
               SS_nargs,
               SX_set_label, SS_PR_PROC);

    SS_install("pg-set-range-limits!",
               "Set the range plotting limits for the given graph",
               SS_nargs,
               SX_set_ran_limits, SS_PR_PROC);

    SS_install("pg-set-view-angle!",
               "Set the viewers Euler angles theta, phi, and chi",
               SS_nargs,
               SX_set_view_angle, SS_PR_PROC);

/* low level PGS functions */
    SX_install_pgs_primitives();

/* interface object functions */
    SX_install_pgs_iob();

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DEF_FILE_GRAPH - define PM_set and PM_mapping to 
 *                   - a PDB file thereby preparing it for mappings
 */

static object *SX_def_file_graph(obj)
   object *obj;
   {PDBfile *file;
    g_file *po;

    po = NULL;
    SS_args(obj,
            G_FILE, &po,
            0);

    if (strcmp(po->type, SX_PDBFILE_S) != 0)
       SS_error("REQUIRE PDB FILE - SX_DEF_FILE_GRAPH", obj);

    file = FILE_FILE(PDBfile, po);

    if (!PD_inquire_type(file, "PM_mapping"))
       if (!PD_def_mapping(file))
	  SS_error("CAN`T DEFINE MAPPINGS - SX_DEF_FILE_GRAPH", obj);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GRAPH_PDBDATA - given a graph object
 *                  - allocate and fill a PDB data object with the
 *                  - PM_mapping and return it
 */

static object *SX_graph_pdbdata(argl)
   object *argl;
   {PG_graph *g;
    PM_mapping *f;
    char *name;
    PDBfile *file;
    g_file *po;

    g    = NULL;
    po   = NULL;
    name = NULL;
    SS_args(argl,
            G_GRAPH, &g,
            G_FILE, &po,
            SC_STRING_I, &name,
            0);

    if (strcmp(po->type, SX_PDBFILE_S) != 0)
       SS_error("REQUIRE PDB FILE - SX_GRAPH_PDBDATA", argl);
    file = FILE_FILE(PDBfile, po);

    if (g == NULL)
       SS_error("BAD GRAPH - SX_GRAPH_PDBDATA", argl);

/* disconnect any function pointers or undefined structs/members */
    for (f = g->f; f != NULL; f = f->next)
        {f->domain->opers = NULL;
         f->range->opers = NULL;};

    return(SX_pdbdata_handler(file, name, "PM_mapping *", &(g->f) , TRUE));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PDBDATA_GRAPH - read a PG_graph out of a PDB file and
 *                  - return a PGS graph object
 *                  - FORM:
 *                  -    (pdbdata->pgs_graph <file> <name>)
 */

static object *SX_pdbdata_graph(argl)
   object *argl;
   {object *obj;
    int i;
    char *name;
    PDBfile *file;
    SX_menu_item *mitems;
    g_file *po;
    syment *ep;

    po  = NULL;
    obj = SS_null;
    SS_args(argl,
            G_FILE, &po,
            SS_OBJECT_I, &obj,
            0);

    if (strcmp(po->type, SX_PDBFILE_S) != 0)
       SS_error("REQUIRE PDB FILE - SX_PDBDATA_GRAPH", argl);

    file = FILE_FILE(PDBfile, po);

    if (SS_integerp(obj))
       {mitems = po->menu;
        if (mitems == NULL)
           {_SX_get_menu(po);
            mitems = po->menu;};

        i = SS_INTEGER_VALUE(obj);
        if ((i < 1) || (i > po->n_menu_items))
          return(SS_null);
        name = mitems[i-1].vname;}
    else
       {argl = SS_cdr(argl);
        name = SC_strsavef(SS_get_string(obj),
               "char*:SX_PDBDATA_GRAPH:name");};

/* check to see whether or not the variable is in the file */
    ep = PD_inquire_entry(file, name, TRUE, NULL);
    if (ep == NULL)
       return(SS_null);

    if (strcmp(PD_entry_type(ep), "PM_mapping *") == 0)
       return(_SX_pdbdata_graph(file, name, ep));

    else if (SC_strstr(name, "curve") != NULL)
       return(_SX_pdbcurve_graph(file, name, ep));

    else
       SS_error("VARIABLE NOT MAPPING - SX_PDBDATA_GRAPH",
                SS_mk_string(name));

    return(SS_null);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_PDBDATA_GRAPH - worker for mapping side of SX_pdbdata_graph */

static object *_SX_pdbdata_graph(file, name, ep)
   PDBfile *file;
   char *name;
   syment *ep;
   {int ndd, ndr, clr, ret;
    char *info_type, *tail;
    byte *info;
    char dname[MAXLINE];
    SC_address data;
    PG_graph *g;
    PM_mapping *f;
    PM_set *domain, *range;
    static int id = 'A';

/* read the mapping */
    if (file == SX_vif)
       {data.diskaddr = PD_entry_address(ep);
        f = *(PM_mapping **) data.memaddr;}
    else
       {if (!PD_read(file, name, &f))
           SS_error(PD_err, SS_null);};

/* reconnect any function pointers or undefined structs/members */
    domain = f->domain;
    range  = f->range;
    if (domain == NULL)
       {if (PD_has_directories(file))
           {strcpy(dname, name);
            tail = strrchr(dname, '/');
            tail = (tail == NULL) ? dname : tail + 1;
            strcpy(tail, f->name);
            PD_process_set_name(tail);}
        else
           {strcpy(dname, f->name);
            PD_process_set_name(dname);};

        if (!PD_read(file, dname, &data.memaddr))
           SS_error(PD_err, SS_null);

        domain = f->domain = (PM_set *) data.memaddr;};

    ndd = (domain == NULL) ? 0 : domain->dimension_elem;
    ndr = (range == NULL) ? 0 : range->dimension_elem;
    clr = id % 14 + 1;

    info_type = SC_PCONS_P_S;
    if ((ndd == 1) && (ndr == 1))
       info = (byte *) PG_set_line_info(NULL, CARTESIAN, CARTESIAN,
					SOLID, FALSE, 0, clr, 0, 0.0);

    else if ((ndd == 2) && (ndr == 1))
       info = (byte *) PG_set_tds_info(NULL, PLOT_CONTOUR, CARTESIAN,
				       SOLID, clr, _PG_contour_n_levels,
				       1.0, 0.0, 0.0, 0.0, 0.0, HUGE);

    else if ((ndd == 2) && (ndr == 2))
       info = (byte *) PG_set_tdv_info(NULL, PLOT_VECTOR, CARTESIAN,
				       SOLID, clr, 0.0);
    else
       info = NULL;

    g = PG_make_graph_from_mapping(f, info_type, info, id++, NULL);

    ret = TRUE;
    for (f = g->f; f != NULL; f = f->next)
        {domain = f->domain;
         range  = f->range;
         if (domain != NULL)
            {ret &= PM_set_opers(domain);
	     if (domain->info_type == NULL)
                domain->info_type = SC_PCONS_P_S;};

         if (range != NULL)
            {ret &= PM_set_opers(range);
	     if (range->info_type == NULL)
                range->info_type = SC_PCONS_P_S;};

/* GOTCHA: it is not necessary to die here if you only want to view
 *         the data
 *         doing algebra would be a problem later on however
         if (ret == FALSE)
            SS_error("NO FIELD FOR TYPE - SX_PDBDATA_GRAPH", SS_null);
 */
       };

    if (id > 'Z')
       id = 'A';

    return(SX_mk_graph(g));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_PDBCURVE_GRAPH - worker for curve side of SX_pdbdata_graph */

static object *_SX_pdbcurve_graph(file, name, ep)
   PDBfile *file;
   char *name;
   syment *ep;
   {int n, clr;
    static int id = 'A';
    char label[MAXLINE];
    REAL *x, *y;
    REAL xmn, xmx, ymn, ymx;
    pcons *info;
    PG_graph *g;
    PM_mapping *f;
    PM_set *domain, *range;

/* read the mapping */

    if (!PD_read_pdb_curve(file, name, &x, &y, &n, label,
                           &xmn, &xmx, &ymn, &ymx, TRUE))
       SS_error(PD_err, SS_null);

    domain = PM_make_set("X values", SC_REAL_S, FALSE, 1, n, 1, x);
    range  = PM_make_set("Y values", SC_REAL_S, FALSE, 1, n, 1, y);

    f    = PM_make_mapping(label, PM_LR_S, domain, range, N_CENT, NULL);
    clr  = id % 14 + 1;
    info = PG_set_line_info(NULL, CARTESIAN, CARTESIAN,
			    SOLID, FALSE, 0, clr, 0, 0.0);
    g    = PG_make_graph_from_mapping(f, SC_PCONS_P_S, info, id++, NULL);

    if (id > 'Z')
       id = 'A';

    return(SX_mk_graph(g));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MK_GRAPH - encapsulate a PG_graph as an object */

object *SX_mk_graph(g)
   PG_graph *g;
   {object *op;

    op = SS_mk_object(g, G_GRAPH, SELF_EV, g->f->name);
    op->print   = _SX_wr_ggraph;
    op->release = _SX_rl_ggraph;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_WR_GGRAPH - print a g_graph */

static void _SX_wr_ggraph(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "<GRAPH|%c>", GRAPH_IDENTIFIER(obj));

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_RL_GGRAPH - gc a graph */

static void _SX_rl_ggraph(obj)
   object *obj;
   {PG_graph *g;

    g = SS_GET(PG_graph, obj);
    if (g->info != NULL)
       SFREE(g->info);
    SFREE(g);

/* GOTCHA: don't know if it is safe to GC the mapping or its sets */

    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MK_DEV_ATTRIBUTES - encapsulate a PG_dev_attributes as an object */

object *SX_mk_dev_attributes(da)
   PG_dev_attributes *da;
   {object *op;

    op = SS_mk_object(da, G_DEV_ATTRIBUTES, SELF_EV, NULL);
    op->print   = _SX_wr_gdev_attr;
    op->release = _SX_rl_gdev_attr;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_WR_GDEV_ATTR - print a g_dev_attributes */

static void _SX_wr_gdev_attr(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "<DEV_ATTRIBUTES>");

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_RL_GDEV_ATTR - gc a dev_attributes */

static void _SX_rl_gdev_attr(obj)
   object *obj;
   {PG_dev_attributes *da;

    da = SS_GET(PG_dev_attributes, obj);
    SFREE(da);

    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_GET_REF_MAP - get the particular mapping referred to 
 *                 - return SS_null if the index is too large
 *                 - for this file
 */

object *_SX_get_ref_map(po, indx, dtype)
   g_file *po;
   int indx;
   char *dtype;
   {int n;
    char type;
    object *argl, *ret;
    SX_menu_item *mitems;

    mitems = po->menu;
    n      = po->n_menu_items;
    if (indx > n)
       return(SS_null);

    ret  = SS_null;
    type = mitems[indx-1].type[3];
    switch (type)
       {case 'i' :
             argl = SS_make_list(G_FILE, po,
                                 SC_INTEGER_I, &indx,
                                 0);
             ret = SX_pdbdata_image(argl);
             SS_GC(argl);
             break;

        case 'm' :
        default  :
             argl = SS_make_list(G_FILE, po,
				 SC_INTEGER_I, &indx,
				 0);
	     ret = SX_pdbdata_graph(argl);
	     SS_GC(argl);
	     break;};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MENU_ITEM_TYPE - return the type of a menu item */

static object *SX_menu_item_type(argl)
   object *argl;
   {int n, indx;
    char bf[MAXLINE], *s;
    g_file *po;
    SX_menu_item *mitems;

    po   = NULL;
    indx = -1;
    SS_args(argl,
            G_FILE, &po,
            SC_INTEGER_I, &indx,
            0);

    _SX_get_menu(po);

    indx--;
    mitems = po->menu;
    n      = po->n_menu_items;
    if ((indx < 0) || (indx >= n))
       return(SS_null);

    strcpy(bf, mitems[indx].type + 3);

    return(SS_mk_string(SC_strtok(bf, " *", s)));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GET_ASCII_IMAGE_NAME - given a file and reference to an image by
 *                         - name or menu number,
 *                         - return the image name as an object
 */

static object *SX_get_ascii_image_name(argl)
   object *argl;
   {long i;
    char *name;
    SX_menu_item *mitems;
    g_file *po;

    argl = SX_get_file(argl, &po);
    argl = SS_car(argl);
    if (SS_integerp(argl))
       {mitems = po->menu;
        if (mitems == NULL)
           {_SX_get_menu(po);
            mitems = po->menu;};

        i = SS_INTEGER_VALUE(argl);
        if ((i < 1) || (i > po->n_menu_items))
           return(SS_null);

        name = mitems[i-1].vname;}
    else
       name = SC_strsavef(SS_get_string(argl),
              "char*:SX_GET_ASCII_IMAGE_NAME:name");

    return(SS_mk_string(name));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GET_ASCII_MAPPING_NAME - given a file and reference to a mapping by
 *                           - name or menu number,
 *                           - return the mapping name as an object
 */

static object *SX_get_ascii_mapping_name(argl)
   object *argl;
   {long i;
    char *name;
    SX_menu_item *mitems;
    g_file *po;

    argl = SX_get_file(argl, &po);
    argl = SS_car(argl);
    if (SS_integerp(argl))
       {mitems = po->menu;
        if (mitems == NULL)
           {_SX_get_menu(po);
            mitems = po->menu;};

        i = SS_INTEGER_VALUE(argl);
        if ((i < 1) || (i > po->n_menu_items))
           return(SS_null);

        name = mitems[i-1].vname;}
    else
       name = SC_strsavef(SS_get_string(argl),
              "char*:SX_GET_ASCII_MAPPING_NAME:name");

    return(SS_mk_string(name));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PDBDATA_IMAGE - read a PG_image out of a PDB file and
 *                  - return a PGS image object
 *                  - FORM:
 *                  -    (pdbdata->pgs_image <file> <name>)
 */

static object *SX_pdbdata_image(argl)
   object *argl;
   {object *obj;
    int i;
    char *name;
    SX_menu_item *mitems;
    PDBfile *file;
    g_file *po;
    syment *ep;
    SC_address data;
    PG_image *im;

    if (!SS_consp(argl))
       SS_error("BAD ARGUMENT LIST - SX_PDBDATA_IMAGE", argl);

/* if the first object is a pdbfile, use it, otherwise, use default file */
    argl = SX_get_file(argl, &po);
    file = (PDBfile *) po->file;

    obj = SS_car(argl);
    if (SS_integerp(obj))
       {mitems = po->menu;
        if (mitems == NULL)
           {_SX_get_menu(po);
            mitems = po->menu;};

        i    = SS_INTEGER_VALUE(obj);
        if ((i < 1) || (i > po->n_menu_items))
           return(SS_null);
        name = mitems[i-1].vname;}
    else
       {argl = SS_cdr(argl);
        name = SC_strsavef(SS_get_string(obj),
               "char*:SX_PDBDATA_IMAGE:name");};

/* check to see whether or not the variable is in the file */
    ep = PD_inquire_entry(file, name, TRUE, NULL);
    if (ep == NULL)
       SS_error("VARIABLE NOT FOUND - SX_PDBDATA_IMAGE", obj);

    if (strcmp(PD_entry_type(ep), "PG_image *") != 0)
       SS_error("VARIABLE NOT IMAGE - SX_PDBDATA_IMAGE", obj);

/* read the mapping */
    if (file == SX_vif)
       {data.diskaddr = PD_entry_address(ep);
	im = *(PG_image **) data.memaddr;}
    else
       {if (!PD_read(file, name, &im))
           SS_error(PD_err, obj);};

    return(SX_mk_image(im));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MK_IMAGE - encapsulate a PG_image as an object */

object *SX_mk_image(im)
   PG_image *im;
   {object *op;

    op = SS_mk_object(im, G_IMAGE, SELF_EV, im->label);
    op->print   = _SX_wr_gimage;
    op->release = _SX_rl_gimage;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_WR_GIMAGE - print a g_image */

static void _SX_wr_gimage(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "<IMAGE|%s>",
                              IMAGE_NAME(obj));

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_RL_GIMAGE - gc a image */

static void _SX_rl_gimage(obj)
   object *obj;
   {

/* GOTCHA - don't know right thing to do here. See _SX_rl_ggraph.
   If image released, ci 1; ci 1 crashes pdbview. */
/*
    PG_image *im;

    im = SS_GET(PG_image, obj);
    SFREE(im->label);
    SFREE(im->element_type);
    SFREE(im->buffer);
    SFREE(im);
*/

    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GRAPH_PDBCURVE - given a PM_mapping or PG_graph object for a curve,
 *                     write the curve to the specified file.
 */

static object *SX_graph_pdbcurve(argl)
   object *argl;
   {PM_mapping *f;
    g_file *po;
    PDBfile *file;
    char curve_name[MAXLINE];
    int i;

    f  = NULL;
    po = NULL;
    SS_args(argl,
            G_MAPPING, &f,
            G_FILE, &po,
            0);

    if ((po == NULL) || (po == SX_gvif))
       file = SX_vif;
    else if (strcmp(po->type, SX_PDBFILE_S) == 0)
       file = FILE_FILE(PDBfile, po);
    else
       SS_error("BAD FILE - SX_GRAPH_PDBCURVE", argl);

    if (f == NULL)
       SS_error("BAD ARGUMENT - SX_GRAPH_PDBCURVE", argl);

    _SX_get_menu(po);
    for (i = 0; TRUE; i++)
        {sprintf(curve_name, "curve%04d", i);
         if (PD_inquire_entry(file, curve_name, TRUE, NULL) == NULL)
            break;};

    PD_wrt_pdb_curve(file, f->name, (int) f->domain->n_elements,
                     *(REAL **) f->domain->elements,
                     *(REAL **) f->range->elements, i);

/* add to menu */
    _SX_push_menu_item(po, curve_name, "PG_curve");

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_IMAGE_PDBDATA - given a PG_image object,
 *                  - allocate and fill a PDB data object with the PG_image.
 *                  - Write the image to the file and return the data object.
 */

static object *SX_image_pdbdata(argl)
   object *argl;
   {PG_image *f;
    g_file *file;
    PDBfile *strm;
    char image_name[MAXLINE];
    long i;
    object *ret;

    f    = NULL;
    file = NULL;
    SS_args(argl,
            G_IMAGE, &f,
            G_FILE, &file,
            0);

    if ((file == NULL) || (file == SX_gvif))
       {file = SX_gvif;
	strm = SX_vif;}
    else if (strcmp(file->type, SX_PDBFILE_S) == 0)
       strm = (PDBfile *) file->file;
    else
       SS_error("BAD FILE - SX_IMAGE_PDBDATA", argl);

    if (f == NULL)
       SS_error("BAD ARGUMENT - SX_IMAGE_PDBDATA", argl);

    _SX_get_menu(file);
    for (i = 0; TRUE; i++)
        {sprintf(image_name, "Image%ld", i);
         if (PD_inquire_entry(strm, image_name, TRUE, NULL) == NULL)
            break;};

    ret = SX_pdbdata_handler(strm, image_name, "PG_image *", &f , TRUE);

/* add to menu */
    _SX_push_menu_item(file, image_name, "PG_image *");

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DEVICEP - function version of SX_DEVICEP macro */

static object *SX_devicep(obj)
   object *obj;
   {return(SX_DEVICEP(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GRAPHP - function version of SX_GRAPHP macro */

static object *SX_graphp(obj)
   object *obj;
   {return(SX_GRAPHP(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_IMAGEP - function version of SX_IMAGEP macro */

static object *SX_imagep(obj)
   object *obj;
   {return(SX_IMAGEP(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DEV_ATTRIBUTESP - function version of SX_DEV_ATTRIBUTESP macro */

static object *SX_dev_attributesp(obj)
   object *obj;
   {return(SX_DEV_ATTRIBUTESP(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAKE_DEVICE - SX level interface to PG_make_device */

static object *SX_make_device(argl)
   object *argl;
   {PG_device *dev;
    char *name, *type, *title;

    name  = NULL;
    type  = NULL;
    title = NULL;
    SS_args(argl,
            SC_STRING_I, &name,
            SC_STRING_I, &type,
            SC_STRING_I, &title,
            0);

    if (title == NULL)
       {if (strcmp(name, "PS") == 0)
           {title = SX_ps_name;
	    type  = SX_ps_type;}
        else if (strcmp(name, "CGM") == 0)
           {title = SX_cgm_name;
	    type  = SX_cgm_type;}
#ifdef HAVE_JPEGLIB
        else if (strcmp(name, "JPEG") == 0)
           {title = SX_jpeg_name;
	    type  = SX_jpeg_type;}
#endif
        else
           SS_error("No title specified in SX_make_device call", SS_null);};
 
    dev = PG_make_device(name, type, title);

    return(SX_mk_graphics_device(dev));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DEVICE_PROPS - return a list of device properties: NAME, TYPE, and
 *                 - TITLE
 */

static object *SX_device_props(argl)
   object *argl;
   {PG_device *dev;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_DEVICE_PROPS", SS_null);

    return(SS_make_list(SC_STRING_I, dev->name,
                        SC_STRING_I, dev->type,
                        SC_STRING_I, dev->title,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MK_GRAPHICS_DEVICE - encapsulate a PG_device as an object */

object *SX_mk_graphics_device(dev)
   PG_device *dev;
   {object *op;

    if (dev == NULL)
       op = SS_null;
    else
       {op = SS_mk_object(dev, G_DEVICE, SELF_EV, dev->title);
	op->print   = _SX_wr_gdevice;
	op->release = _SX_rl_gdevice;};

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_WR_GDEVICE - print a g_device */

static void _SX_wr_gdevice(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "<GRAPHICS-DEVICE|%s,%s,%s>",
                              DEVICE_NAME(obj),
                              DEVICE_TYPE(obj),
                              DEVICE_TITLE(obj));

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_RL_GDEVICE - gc a device */

static void _SX_rl_gdevice(obj)
   object *obj;
   {PG_device *dev;

    dev = SS_GET(PG_device, obj);
    _PG_rl_device(dev);

    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_EXPOSE_EVENT_HANDLER - handle expose events  */

void SX_expose_event_handler(dev, ev)
   PG_device *dev;
   PG_event *ev;
   {SX_motion_event_handler(dev, ev);

/*
    PRINT(stdout, ".");
    SX_plot();
*/
    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MOTION_EVENT_HANDLER - handle motion events  */

void SX_motion_event_handler(dev, ev)
   PG_device *dev;
   PG_event *ev;
   {if (SX_show_mouse_location)
       PG_print_pointer_location(dev,
                                 SX_show_mouse_location_x,
                                 SX_show_mouse_location_y,
                                 TRUE);
    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_UPDATE_EVENT_HANDLER - handle update events  */

void SX_update_event_handler(dev, ev)
   PG_device *dev;
   PG_event *ev;
   {int width, height, ncol;

    PG_make_device_current(dev);

    SX_window_height_P = dev->window_height;
    SX_window_width_P  = dev->window_width;
    SX_window_x_P      = dev->window_x;
    SX_window_y_P      = dev->window_y;

    PG_query_screen(dev, &width, &height, &ncol);
    SX_window_width  = SX_window_width_P/width;
    SX_window_height = SX_window_height_P/width; /* I really mean the width */

    SX_window_x = SX_window_x_P/width;
    SX_window_y = SX_window_y_P/width;           /* I really mean the width */

/*  SX_plot(); */

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DEFAULT_EVENT_HANDLER - handle events that get through to here */

void SX_default_event_handler(dev, ev)
   PG_device *dev;
   PG_event *ev;
   {SX_motion_event_handler(dev, ev);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_OPEN_DEVICE - SX level interface to PG_open_device */

static object *SX_open_device(argl)
   object *argl;
   {PG_device *dev;
    double xf, yf, dxf, dyf;

    dev = NULL;
    xf  = yf  = 0.0;
    dxf = dyf = 0.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &xf,
            SC_DOUBLE_I, &yf,
            SC_DOUBLE_I, &dxf,
            SC_DOUBLE_I, &dyf,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_OPEN_DEVICE", SS_null);

/* PG_open_device set dev->hard_copy_device so we can't query it here */
    if (!SX_gr_mode &&
	(dev->type_index != PS_DEVICE) &&
	(dev->type_index != CGMF_DEVICE))
       return(SS_f);

    dev->autodomain = TRUE;
    dev->autorange  = TRUE;
    dev->data_id    = TRUE;

    dev = PG_open_device(dev, xf, yf, dxf, dyf);
    if (dev == NULL)
       return(SS_f);

    putln = (PFfprintf) SX_fprintf;
    getln = (PFfgets) PG_wind_fgets;

    PG_make_device_current(dev);
    PG_set_window(dev, 0.0, 1.0, 0.0, 1.0);
    PG_release_current_device(dev);

    PG_set_default_event_handler(dev, SX_default_event_handler);
    PG_set_motion_event_handler(dev, SX_motion_event_handler);
    PG_set_expose_event_handler(dev, SX_expose_event_handler);
    PG_set_update_event_handler(dev, SX_update_event_handler);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_CLOSE_DEVICE - SX level interface to PG_close_device */

static object *SX_close_device(argl)
   object *argl;
   {PG_device *dev;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    PG_close_device(dev);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_NEXT_COLOR - return the next available color */

int _SX_next_color(dev)
   PG_device *dev;
   {static int color = 3;

    if (SX_default_color != -1)
       {if (dev == NULL)
           return(_PG_trans_color(PG_console_device, SX_default_color));
        else
           return(_PG_trans_color(dev, SX_default_color));}

    else
       {color = max(color + 1, 1);
        if (dev == NULL)
           return(_PG_trans_color(PG_console_device, color));
        else
           return(_PG_trans_color(dev, color));};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAKE_PGS_GRAPH - build a PGS graph object out of a domain, range
 *                   - and attribute set
 *                   -
 *                   - FORM:
 *                   - (pg-make-graph <domain> <range>
 *                   -                 [<centering> <color> <width> <style>
 *                   -                  <emap> <name>])
 *                   -
 *                   - Centering defaults to zone
 *                   - Color defaults to BLUE
 *                   - Width defaults to thinnest possible
 *                   - Style defaults to SOLID
 *                   - Existence map thru which to plot defaults to all 1's
 *                   - mapping name
 */

static object *SX_make_pgs_graph(argl)
   object *argl;
   {PG_graph *g;
    pcons *info;
    PM_set *domain, *range;
    C_array *arr;
    pcons *inf;
    REAL width;
    char label[MAXLINE], *name, *emap;
    int color, style, centering, clr;
    static int id = '@';

    color     = 1;
    style     = SOLID;
    centering = N_CENT;
    width     = 0.0;
    arr       = NULL;
    name      = NULL;
    if (++id > 'Z')
       id = 'A';

    SS_args(argl,
            G_SET, &domain,
            G_SET, &range,
            SC_INTEGER_I, &centering,
            SC_INTEGER_I, &color,
            SC_REAL_I, &width,
            SC_INTEGER_I, &style,
	    G_NUM_ARRAY, &arr,
            SC_STRING_I, &name,
            0);

    if (name == NULL)
       sprintf(label, "%s->%s", domain->name, range->name);
    else
       strcpy(label, name);

/* build the graph
 * NOTE: since the addition of PG_graph info member there will have to
 * be work here to reconnect the rendering attributes
 */
    clr  = id % 14 + 1;
    info = PG_set_line_info(NULL, CARTESIAN, CARTESIAN,
			    SOLID, FALSE, 0, clr, 0, 0.0);
    g = PG_make_graph_from_sets(label, domain, range, centering,
                                SC_PCONS_P_S, info, id, NULL);

    if (arr == NULL)
       {emap = FMAKE_N(char, domain->n_elements,
                       "SX_MAKE_PGS_GRAPH:emap");
        memset(emap, 1, domain->n_elements);}
    else
       {emap = NULL;
        CONVERT(SC_CHAR_S, &emap, arr->type, arr->data, arr->length, FALSE);};

    inf = (pcons *) g->f->map;
    inf = SC_add_alist(inf, "EXISTENCE", SC_STRING_S, (byte *) emap);
    g->f->map = (byte *) inf;

    return(SX_mk_graph(g));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_VIEW_ANGLE - set the current view angles */

static object *SX_set_view_angle(argl)
   object *argl;
   {SS_args(argl,
	    SC_DOUBLE_I, &SX_theta,
	    SC_DOUBLE_I, &SX_phi,
	    SC_DOUBLE_I, &SX_chi,
	    0);

    return(argl);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DRAW_DOMAIN - main Scheme level domain plot control routine */

static object *SX_draw_domain(argl)
   object *argl;
   {PG_device *dev;
    PM_set *data, *nxt, *p;
    int type;
    pcons *info;
    object *obj, *extr;

    if (SS_consp(argl))
       SX_GET_OBJECT_FROM_LIST(SX_DEVICEP(obj), dev,
                               SS_GET(PG_device, obj),
                               argl, "BAD DEVICE - SX_DRAW_DOMAIN");

/* get the list of sets */
    data = nxt = NULL;
    while (SS_consp(argl))
       {obj = SS_car(argl);
        if (SX_SETP(obj))
           {if (data == NULL)
               data = nxt = SS_GET(PM_set, obj);
            else
               {nxt->next = SS_GET(PM_set, obj);
                nxt = nxt->next;};}
        else
           break;
        argl = SS_cdr(argl);};

    type  = PLOT_WIRE_MESH;
    extr  = SS_null;
    SS_args(argl,
            SC_INTEGER_I, &type,
	    SS_OBJECT_I, &extr,
	    0);

    if (!SS_nullobjp(extr))
       _SX_set_limits("LIMITS", data, extr);

    if (data == NULL)
       SS_error("BAD DOMAIN - SX_DRAW_DOMAIN", obj);

    if (data->info_type == NULL)
       data->info_type = SC_PCONS_P_S;

    info = (pcons *) data->info;

    info = PG_set_plot_type(info, type, CARTESIAN);
    SC_CHANGE_VALUE_ALIST(info, REAL, "REAL *", "THETA", DEG_RAD*SX_theta);
    SC_CHANGE_VALUE_ALIST(info, REAL, "REAL *", "PHI",   DEG_RAD*SX_phi);
    SC_CHANGE_VALUE_ALIST(info, REAL, "REAL *", "CHI",   DEG_RAD*SX_chi);

    data->info = (byte *) info;

    PG_domain_plot(dev, data, NULL);

/* disconnect the meshes */
    for (p = data; p != NULL; p = nxt)
        {nxt = p->next;
         p->next = NULL;};

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DRAW_PLOT - main Scheme level plot control routine */

static object *SX_draw_plot(argl)
   object *argl;
   {PG_device *dev;
    PG_graph *data, *nxt, *g;
    PM_mapping *f;
    PM_set *domain, *range;
    pcons *info;
    object *obj;
    int domain_dim, range_dim, rendering;
	
    if (SS_consp(argl))
       SX_GET_OBJECT_FROM_LIST(SX_DEVICEP(obj), dev,
                               SS_GET(PG_device, obj),
                               argl, "BAD DEVICE - SX_DRAW_PLOT");

    if (!SX_OK_TO_DRAW(dev))
       return(SS_f);

/* get the list of graphs */
    data = nxt = NULL;
    for ( ; SS_consp(argl); argl = SS_cdr(argl))
        {obj = SS_car(argl);
	 if (SX_GRAPHP(obj))
            {if (data == NULL)
                data = nxt = SS_GET(PG_graph, obj);
             else
                {nxt->next = SS_GET(PG_graph, obj);
		 nxt = nxt->next;};}

	 else if (SX_MAPPINGP(obj))
            {f = SS_GET(PM_mapping, obj);

/* here's a GC nightmare! */
	     g = PG_make_graph_from_mapping(f, NULL, NULL,
					    SX_map_count(), NULL);
	     if (data == NULL)
                data = nxt = g;
	     else
                {nxt->next = g;
		 nxt = nxt->next;};}
	 else
            break;};

    if (data == NULL)
       SS_error("BAD GRAPH - SX_DRAW_PLOT", obj);

    if (_SX_grotrian_graphp(data))
       {PG_grotrian_plot(dev, data);
        return(SS_t);};

    f          = data->f;
    domain     = f->domain;
    range      = f->range;
    domain_dim = domain->dimension_elem;
    range_dim  = range->dimension_elem;
    rendering  = data->rendering;
    info       = (pcons *) data->info;

    if (SS_consp(argl))
       {SX_GET_OBJECT_FROM_LIST(SS_integerp(obj), rendering,
                                SS_INTEGER_VALUE(obj),
                                argl, "BAD PLOT TYPE - SX_DRAW_PLOT");

	info = PG_set_plot_type(info, rendering, CARTESIAN);

        switch (rendering)
           {case PLOT_IMAGE :
                 data->render = PG_image_plot;
                 break;

            case PLOT_FILL_POLY :
                 data->render = PG_poly_fill_plot;
                 break;

            case PLOT_SURFACE   :
            case PLOT_WIRE_MESH :
                 data->render = PG_surface_plot;

                 break;

            case PLOT_MESH :
                 data->render = PG_mesh_plot;
                 break;

            case PLOT_CONTOUR :
                 if (SS_consp(argl))
                    SX_GET_OBJECT_FROM_LIST(SS_integerp(obj),
                                            _PG_contour_n_levels,
                                            SS_INTEGER_VALUE(obj),
                                            argl,
                                            "BAD NUMBER OF LEVELS - SX_DRAW_PLOT");

                 data->render = PG_contour_plot;
                 break;

            case PLOT_VECTOR :
                 data->render = PG_vector_plot;
                 break;

            case CARTESIAN :
            case LOGICAL   :
                 data->render = PG_curve_plot;
                 break;

            case HISTOGRAM :
                 if (SS_consp(argl))
                    {SX_GET_INTEGER_FROM_LIST(_PG_hist_start, argl,
                                            "BAD HISTOGRAM OPTION - SX_DRAW_PLOT");};
                 _SX_attach_rendering_1d(data, HISTOGRAM, CARTESIAN, _PG_hist_start);
                 break;

            case POLAR :
                 _SX_attach_rendering_1d(data, POLAR, POLAR, 0);
                 break;

            case INSEL :
                 _SX_attach_rendering_1d(data, INSEL, INSEL, 0);
                 break;

            case SCATTER :
                 _SX_attach_rendering_1d(data, SCATTER, CARTESIAN, 0);
                 break;

            default :
                 break;};

	*rendering_mode[domain_dim][range_dim] = rendering;};

    rendering = *rendering_mode[domain_dim][range_dim];

    if ((rendering == PLOT_SURFACE) ||
        (rendering == PLOT_WIRE_MESH) ||
        (rendering == PLOT_MESH))
       {SC_ADD_VALUE_ALIST(info, REAL, "REAL *", "THETA", DEG_RAD*SX_theta);
	SC_ADD_VALUE_ALIST(info, REAL, "REAL *", "PHI", DEG_RAD*SX_phi);
	SC_ADD_VALUE_ALIST(info, REAL, "REAL *", "CHI", DEG_RAD*SX_chi);};

    data->rendering = rendering;
    data->info      = (byte *) info;
/*    dev->data_id    = TRUE; */

    putln = (PFfprintf) SX_fprintf;

    if (domain->info_type != NULL)
       {if (strcmp(domain->info_type, SC_PCONS_P_S) == 0)
	  dev->autodomain = (SC_assoc_entry((pcons *) domain->info,
					    "LIMITS") == NULL);};

    if (range->info_type != NULL)
       {if (strcmp(range->info_type, SC_PCONS_P_S) == 0)
	  dev->autorange  = (SC_assoc_entry((pcons *) range->info,
					    "LIMITS") == NULL);};

    if (data->f->next != NULL)
       PG_draw_picture(dev, data->f, rendering,
		       TRUE, dev->WHITE, SOLID, 1.2,
		       FALSE, dev->WHITE, SOLID, 0.0);

    else if (data->render != NULL)
       data->render(dev, data);

    else
       PG_draw_graph(dev, data);

    PG_draw_interface_objects(dev);

    putln = (PFfprintf) SS_printf;

/* unchain the list of graphs */
    while (data->next != NULL)
       {nxt = data->next;
        data->next = NULL;
        data = nxt;};

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_ATTACH_RENDERING_1D - set up the rendering and axis type for all
 *                         - of the graphs in the list
 */

static void _SX_attach_rendering_1d(data, rendering, axis_type, start)
   PG_graph *data;
   int rendering, axis_type, start;
   {PG_graph *g;
    pcons *line_info;

    for (g = data; g != NULL; g = g->next)
        {line_info = (pcons *) g->info;
	 line_info = PG_set_plot_type(line_info, rendering, axis_type);

         if (rendering == SCATTER)
	    {SC_CHANGE_VALUE_ALIST(line_info,
				   int, SC_INTEGER_P_S,
				   "SCATTER", 1);}

         else if (rendering == HISTOGRAM)
	    {SC_CHANGE_VALUE_ALIST(line_info,
				   int, SC_INTEGER_P_S,
				   "HIST-START", start);};

         g->render = PG_curve_plot;
         g->info   = (byte *) line_info;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAKE_IMAGE - crack the arg list, make, and return an image object
 *                - Usage: (pg-make-image <data> <kmax> <lmax>
 *                -                       [<name> <xmin> <xmax>
 *                -                        <ymin> <ymax>
 *                -                        <zmin> <zmax>])
 */

static object *SX_make_image(argl)
   object *argl;
   {static char *default_name = "Image";
    char *name;
    int kmax, lmax;
    double xmin, xmax, ymin, ymax, zmin, zmax;
    C_array *arr;
    PG_image *im;

    name = default_name;
    kmax = lmax = 0;
    xmin = xmax = 0.0;
    ymin = ymax = 0.0;
    zmin = zmax = 0.0;

    arr = NULL;
    SS_args(argl,
            G_NUM_ARRAY, &arr,
            SC_INTEGER_I, &kmax,
            SC_INTEGER_I, &lmax,
            SC_STRING_I, &name,
            SC_DOUBLE_I, &xmin,
            SC_DOUBLE_I, &xmax,
            SC_DOUBLE_I, &ymin,
            SC_DOUBLE_I, &ymax,
            SC_DOUBLE_I, &zmin,
            SC_DOUBLE_I, &zmax,
            0);

    if (arr == NULL)
       SS_error("BAD DATA - SX_MAKE_IMAGE", argl);

    if (name == NULL)
       name = default_name;

    im = PG_make_image(name, arr->type, arr->data,
		       xmin, xmax, ymin, ymax, zmin, zmax,
		       kmax, lmax, 8, NULL);

    return(SX_mk_image(im));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_BUILD_IMAGE - crack the arg list, build, and return an image object
 *                - Usage: (pg-build-image <dev> <data> <kmax> <lmax>
 *                -                         [<name> <xmin> <xmax>
 *                -                          <ymin> <ymax>
 *                -                          <zmin> <zmax>])
 */

static object *SX_build_image(argl)
   object *argl;
   {static char *default_name = "Image";
    char *name;
    int kmax, lmax;
    double xmin, xmax, ymin, ymax, zmin, zmax;
    C_array *arr;
    PG_device *dev;
    PG_image *im;

    name = default_name;
    kmax = lmax = 0;
    xmin = xmax = 0.0;
    ymin = ymax = 0.0;
    zmin = zmax = 0.0;

    dev = NULL;
    arr = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            G_NUM_ARRAY, &arr,
            SC_INTEGER_I, &kmax,
            SC_INTEGER_I, &lmax,
            SC_STRING_I, &name,
            SC_DOUBLE_I, &xmin,
            SC_DOUBLE_I, &xmax,
            SC_DOUBLE_I, &ymin,
            SC_DOUBLE_I, &ymax,
            SC_DOUBLE_I, &zmin,
            SC_DOUBLE_I, &zmax,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_BUILD_IMAGE", argl);

    if (arr == NULL)
       SS_error("BAD DATA - SX_BUILD_IMAGE", argl);

    if (name == NULL)
       name = default_name;

    im = PG_build_image(dev, name, arr->type, arr->data,
                        kmax, lmax,
                        xmin, xmax, ymin, ymax, zmin, zmax);

    return(SX_mk_image(im));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DRAW_IMAGE - Scheme level image plot control routine */

static object *SX_draw_image(argl)
   object *argl;
   {PG_device *dev;
    PG_image *image;
    REAL xmn, xmx, ymn, ymx, *pv;
    pcons *alist;
    int viewport_box_on;

    dev             = NULL;
    image           = NULL;
    xmn             = 0.0;
    xmx             = 0.0;
    ymn             = 0.0;
    ymx             = 0.0;
    alist           = NULL;
    viewport_box_on = FALSE;

    SS_args(argl,
            G_DEVICE, &dev,
            G_IMAGE, &image,
            SC_REAL_I, &xmn,
            SC_REAL_I, &xmx,
            SC_REAL_I, &ymn,
            SC_REAL_I, &ymx,
            0);

    if (!SX_OK_TO_DRAW(dev))
       return(SS_f);

/* If viewport box has been specified, create an assoc list and pass it in. */
    if ((xmn < xmx) && (ymn < ymx))
       {viewport_box_on = TRUE;
        pv = FMAKE_N(REAL, 4, "SX_draw_image:pv");
        pv[0] = xmn;
        pv[1] = xmx;
        pv[2] = ymn;
        pv[3] = ymx;
        alist = SC_change_alist(alist, "VIEW-PORT", SC_REAL_P_S, pv);}
            
    PG_draw_image(dev, image, image->label, alist);

    if (viewport_box_on)
       {alist = SC_rem_alist(alist, "VIEW-PORT");
        SFREE_N(pv, 4);}    

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_ATTR_GRAPH - set an attribute of a PG_graph object
 *                   - usage: (pg-set-graph-attribute! <graph>
 *                   -                         <name> <type> <value>)
 */

static object *SX_set_attr_graph(argl)
   object *argl;
   {PG_graph *g;
    char *name, *type;
    object *val;
    pcons *inf;
    byte *v;
    C_array *arr;

    g    = NULL;
    name = NULL;
    type = NULL;
    val  = SS_null;
    SS_args(argl,
            G_GRAPH, &g,
            SC_STRING_I, &name,
            SC_STRING_I, &type,
            SS_OBJECT_I, &val,
            0);

    if ((g == NULL) || (name == NULL) || (type == NULL))
       SS_error("INSUFFICIENT ARGUMENTS - SX_SET_ATTR_GRAPH", argl);

/* get the current list */
    if (g->info_type != NULL)
       {if (strcmp(g->info_type, SC_PCONS_P_S) == 0)
	   inf = (pcons *) g->info;
        else
	   inf = NULL;}
    else
       inf = NULL;

    if (strcmp(type, "nil") == 0)
       {if (inf != NULL)
	   g->info = (byte *) SC_rem_alist(inf, name);
	return(SS_t);};

    if (SS_consp(val))
       {object *obj;
	obj = SS_null;
	SS_Assign(obj, SX_list_array(val));
	SS_args(obj,
		G_NUM_ARRAY, &arr,
		0);
	v = arr->data;
	SC_mark(v, 1);
	SS_GC(obj);
	SC_mark(v, -1);}
    else if ((strcmp(type, "int *") == 0) ||
	(strcmp(type, "integer *") == 0))
       {v = SC_alloc(1L, sizeof(int), NULL);
	SS_args(val,
		SC_INTEGER_I, v,
		0);}
    else if (strcmp(type, SC_DOUBLE_P_S) == 0)
       {v = SC_alloc(1L, sizeof(double), NULL);
	SS_args(val,
		SC_DOUBLE_I, v,
		0);}
    else if (strcmp(type, SC_STRING_S) == 0)
       {SS_args(val,
		SC_STRING_I, &v,
		0);}
    else
       SS_error("CAN'T HANDLE TYPE - SX_SET_ATTR_GRAPH", argl);

    inf = SC_change_alist(inf, name, type, v);

    g->info_type = SC_PCONS_P_S;
    g->info      = (byte *) inf;

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_SET_LIMITS - add/modify the specified plotting limits for the
 *                - given graph
 */

static void _SX_set_limits(t, s, argl)
   char *t;
   PM_set *s;
   object *argl;
   {int i, nd, nset;
    REAL *extr, *pe, xmn, xmx;

    if (SS_nullobjp(argl))
       extr = NULL;

    else
       {nd   = s->dimension_elem;
        extr = FMAKE_N(REAL, 2*nd, "_SX_SET_LIMITS:extr");
        pe   = extr;
        for (i = 0; i < nd; i++, argl = SS_cddr(argl))
            {if (SS_nullobjp(argl))
                {SFREE(extr);
		 break;};

	     nset = SS_args(argl,
			    SC_REAL_I, &xmn,
			    SC_REAL_I, &xmx,
			    0);

	     if (nset < 2)
	        SS_error("INSUFFICIENT LIMITS SPECS - _SX_SET_LIMITS",
			 argl);

	     *pe++ = xmn;
             *pe++ = xmx;};};

    PM_set_limits(s, extr);
    
    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_GET_LIMITS - return the plotting LIMITS of the given set if any
 *                - in a form suitable for _SX_SET_LIMITS
 */

static object *_SX_get_limits(s)
   PM_set *s;
   {int i, nd;
    REAL *limits;
    object *obj, *lst;

    lst    = SS_null;
    nd     = 2*(s->dimension_elem);
    limits = PM_get_limits(s);
    
    if (limits != NULL)
       {limits = PM_array_real(s->es_type, limits, nd, NULL);

	for (i = 0; i < nd; i++)
	    {obj = SS_mk_float(*limits++);
	     lst = SS_mk_cons(obj, lst);};
    
	SFREE(limits);

	if (lst != SS_null)
	   lst = SS_reverse(lst);};

    return(lst);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_GET_EXTREMA - return the extrema of the given set in a form suitable
 *                 - for _SX_SET_LIMITS
 */

static object *_SX_get_extrema(s)
   PM_set *s;
   {int i, nd;
    REAL *extr;
    object *obj, *lst;

    nd = 2*(s->dimension_elem);
    
    extr = PM_array_real(s->element_type, s->extrema, nd, NULL);

    lst = SS_null;
    for (i = 0; i < nd; i++)
        {obj = SS_mk_float(*extr++);
	 lst = SS_mk_cons(obj, lst);};
    
    SFREE(extr);

    if (lst != SS_null)
       lst = SS_reverse(lst);

    return(lst);}
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_MAP_RAN_EXTREMA - return the overall extrema of the given, possibly linked,
 *                     -  mapping in a form suitable for _SX_SET_LIMITS
 */

static object *_SX_map_ran_extrema(m)
   PM_mapping *m;
   {int i, nd;
    REAL extr[20], rextr[20];
    PM_set *s;
    PM_mapping *f;
    object *obj, *lst;

/* NOTE: allow up to 10 dimensional ranges */
    for (i = 0; i < 10; i++)
        {rextr[2*i]   =  HUGE_REAL;
         rextr[2*i+1] = -HUGE_REAL;};

    for (f = m; f != NULL; f = f->next)
        {s  = f->range;
         nd = 2*(s->dimension_elem);
    
         PM_array_real(s->element_type, s->extrema, nd, extr);

         for (i = 0; i < (nd/2); i++)
             {rextr[2*i]   = min(rextr[2*i], extr[2*i]);
              rextr[2*i+1] = max(rextr[2*i+1], extr[2*i+1]);};}

    lst = SS_null;
    for (i = 0; i < nd; i++)
        {obj = SS_mk_float(rextr[i]);
	 lst = SS_mk_cons(obj, lst);};
    
    if (lst != SS_null)
       lst = SS_reverse(lst);

    return(lst);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_MAP_DOM_EXTREMA - return the overall extrema of the given, possibly linked,
 *                     -  mapping in a form suitable for _SX_SET_LIMITS
 */

static object *_SX_map_dom_extrema(m)
   PM_mapping *m;
   {int i;
    REAL extr[4], rextr[4];
    PM_set *s;
    PM_mapping *f;
    object *obj, *lst;

    rextr[0] =  HUGE_REAL;
    rextr[1] = -HUGE_REAL;
    rextr[2] =  HUGE_REAL;
    rextr[3] = -HUGE_REAL;

    for (f = m; f != NULL; f = f->next)
        {s  = f->domain;
    
         PM_array_real(s->element_type, s->extrema, 4, extr);

         rextr[0] = min(rextr[0], extr[0]);
         rextr[1] = max(rextr[1], extr[1]);
         rextr[2] = min(rextr[2], extr[2]);
         rextr[3] = max(rextr[3], extr[3]);};

    lst = SS_null;
    for (i = 0; i < 4; i++)
        {obj = SS_mk_float(rextr[i]);
	 lst = SS_mk_cons(obj, lst);};
    
    if (lst != SS_null)
       lst = SS_reverse(lst);

    return(lst);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_IM_DOM_EXTREMA - return the extrema of the given image domain in a form
 *                    - suitable for _SX_SET_LIMITS
 */

static object *_SX_im_dom_extrema(im)
   PG_image *im;
   {object *obj, *lst;

    lst = SS_null;
    obj = SS_mk_float(im->ymax);
    lst = SS_mk_cons(obj, lst);
    obj = SS_mk_float(im->ymin);
    lst = SS_mk_cons(obj, lst);
    obj = SS_mk_float(im->xmax);
    lst = SS_mk_cons(obj, lst);
    obj = SS_mk_float(im->xmin);
    lst = SS_mk_cons(obj, lst);

    return(lst);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_IM_RAN_EXTREMA - return the extrema of the given image range in a form
 *                    - suitable for _SX_SET_LIMITS
 */

static object *_SX_im_ran_extrema(im)
   PG_image *im;
   {object *obj, *lst;

    lst = SS_null;
    obj = SS_mk_float(im->zmax);
    lst = SS_mk_cons(obj, lst);
    obj = SS_mk_float(im->zmin);
    lst = SS_mk_cons(obj, lst);

    return(lst);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_DOM_LIMITS - set the domain plotting limits for the given graph
 *               - form:
 *               -   (pg-set-domain-limits! <graph> (<x1_min> <x1_max> .... ))
 */

static object *SX_set_dom_limits(argl)
   object *argl;
   {PM_set *s;
    object *obj;

    s = NULL;
    obj = SS_car(argl);
    if (SX_SETP(obj))
       s = SS_GET(PM_set, obj);
    else if (SX_MAPPINGP(obj))
       s = SS_GET(PM_mapping, obj)->domain;
    else if (SX_GRAPHP(obj))
       s = SS_GET(PG_graph, obj)->f->domain;
    else
       SS_error("NO SET IMPLIED - SX_SET_DOM_LIMITS", obj);

    _SX_set_limits("LIMITS", s, SS_cadr(argl));

    return(SS_car(argl));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_RAN_LIMITS - set the range plotting limits for the given graph
 *               - form:
 *               -   (pg-set-range-limits! <graph> (<x1_min> <x1_max> .... ))
 */

static object *SX_set_ran_limits(argl)
   object *argl;
   {PM_set *s;
    object *obj;

    s = NULL;
    obj = SS_car(argl);
    if (SX_SETP(obj))
       s = SS_GET(PM_set, obj);
    else if (SX_MAPPINGP(obj))
       s = SS_GET(PM_mapping, obj)->range;
    else if (SX_GRAPHP(obj))
       s = SS_GET(PG_graph, obj)->f->range;
    else
       SS_error("NO SET IMPLIED - SX_SET_RAN_LIMITS", obj);

    _SX_set_limits("LIMITS", s, SS_cadr(argl));

    return(SS_car(argl));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DOM_EXTREMA - return the domain extrema for the given drawable form:
 *                -   (pg-domain-extrema <drawable>)
 */

static object *SX_dom_extrema(argl)
   object *argl;
   {PM_set *s;
    PM_mapping *m;
    object *obj;

    s = NULL;
    obj = SS_car(argl);
    if (SX_SETP(obj))
       s = SS_GET(PM_set, obj);
    else if (SX_MAPPINGP(obj))
       {m = SS_GET(PM_mapping, obj);
        return(_SX_map_dom_extrema(m));}
    else if (SX_GRAPHP(obj))
       {m = SS_GET(PG_graph, obj)->f;
        return(_SX_map_dom_extrema(m));}
    else if (SX_IMAGEP(obj))
       return(_SX_im_dom_extrema(SS_GET(PG_image, obj)));
    else
       SS_error("NO SET IMPLIED - SX_DOM_EXTREMA", obj);

    return(_SX_get_extrema(s));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DOM_LIMITS - return the domain plotting LIMITS if any
 *               - for the given drawable form:
 *               -   (pg-domain-limits <drawable>)
 */

static object *SX_dom_limits(argl)
   object *argl;
   {PM_set *s;
    object *obj;

    s = NULL;
    obj = SS_car(argl);
    if (SX_SETP(obj))
       s = SS_GET(PM_set, obj);
    else if (SX_MAPPINGP(obj))
       s = SS_GET(PM_mapping, obj)->domain;
    else if (SX_GRAPHP(obj))
       s = SS_GET(PG_graph, obj)->f->domain;
    else if (SX_IMAGEP(obj))
       return(SS_null);
/*       return(_SX_im_dom_extrema(SS_GET(PG_image, obj))); */
    else
       SS_error("NO SET IMPLIED - SX_DOM_LIMITS", obj);

    return(_SX_get_limits(s));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_RAN_EXTREMA - return the range extrema for the given drawable form:
 *               -   (pg-range-extrema <drawable>)
 */

static object *SX_ran_extrema(argl)
   object *argl;
   {PM_set *s;
    PM_mapping *m;
    object *obj;

    s = NULL;
    obj = SS_car(argl);
    if (SX_SETP(obj))
       s = SS_GET(PM_set, obj);
    else if (SX_MAPPINGP(obj))
       {m = SS_GET(PM_mapping, obj);
        return(_SX_map_ran_extrema(m));}
    else if (SX_GRAPHP(obj))
       {m = SS_GET(PG_graph, obj)->f;
        return(_SX_map_ran_extrema(m));}
    else if (SX_IMAGEP(obj))
       return(_SX_im_ran_extrema(SS_GET(PG_image, obj)));
    else
       SS_error("NO SET IMPLIED - SX_RAN_EXTREMA", obj);

    return(_SX_get_extrema(s));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_RAN_LIMITS - return the range plotting LIMITS if any
 *               - for the given drawable form:
 *               -   (pg-range-extrema <drawable>)
 */

static object *SX_ran_limits(argl)
   object *argl;
   {PM_set *s;
    object *obj;

    s = NULL;
    obj = SS_car(argl);
    if (SX_SETP(obj))
       s = SS_GET(PM_set, obj);
    else if (SX_MAPPINGP(obj))
       s = SS_GET(PM_mapping, obj)->range;
    else if (SX_GRAPHP(obj))
       s = SS_GET(PG_graph, obj)->f->range;
    else if (SX_IMAGEP(obj))
       return(SS_null);
/*       return(_SX_im_ran_extrema(SS_GET(PG_image, obj))); */
    else
       SS_error("NO SET IMPLIED - SX_RAN_LIMITS", obj);

    return(_SX_get_limits(s));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_LABEL - set the label of the drawable's mapping 
 *              - usage: (pg-set-label! <drawable> <label>)
 */

static object *SX_set_label(argl)
   object *argl;
   {object *obj;
    char *label;

    obj   = NULL;
    label = NULL;
    SS_args(argl,
            SS_OBJECT_I, &obj,
            SC_STRING_I, &label,
            0);

    if ((obj == NULL) || (label == NULL))
       SS_error("INSUFFICIENT ARGUMENTS - SX_SET_LABEL", argl);

    label = SC_strsavef(label, "char*:SX_SET_LABEL:label");

    if (SX_SETP(obj))
       {PM_set *s;

        s = SS_GET(PM_set, obj);
	SFREE(s->name);
	s->name = label;}

    else if (SX_MAPPINGP(obj))
       {PM_mapping *s;

	s = SS_GET(PM_mapping, obj);
	SFREE(s->name);
	s->name = label;}

    else if (SX_GRAPHP(obj))
       {PM_mapping *s;

	s = SS_GET(PG_graph, obj)->f;
	SFREE(s->name);
	s->name = label;}

    else if (SX_IMAGEP(obj))
       {PG_image *s;

	s = SS_GET(PG_image, obj);
	SFREE(s->label);
	s->label = label;}

    else
       SS_error("BAD DRAWABLE - SX_SET_LABEL", obj);

    return(SS_car(argl));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GET_LABEL - return the label of the drawable's mapping 
 *              - usage: (pg-get-label <drawable>)
 */

static object *SX_get_label(obj)
   object *obj;
   {char *label;

    if (SX_SETP(obj))
       {PM_set *s;

        s = SS_GET(PM_set, obj);
	label = s->name;}

    else if (SX_MAPPINGP(obj))
       {PM_mapping *s;

	s = SS_GET(PM_mapping, obj);
	label = s->name;}

    else if (SX_GRAPHP(obj))
       {PM_mapping *s;

	s = SS_GET(PG_graph, obj)->f;
	label = s->name;}

    else if (SX_IMAGEP(obj))
       {PG_image *s;

	s = SS_GET(PG_image, obj);
	label = s->label;}

    else
       SS_error("BAD DRAWABLE - SX_GET_LABEL", obj);

    return(SS_mk_string(label));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
