/*
 * PCFIO.C - I/O hooks suitable for use with PDBLib
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "ppc.h"

#define MAX_FILES 100

#define OPEN_MSG     "PC_FOPEN"
#define SETVBUF_MSG  "PC_FSETVBUF"
#define CLOSE_MSG    "PC_FCLOSE"
#define FLUSH_MSG    "PC_FFLUSH"
#define TELL_MSG     "PC_FTELL"
#define SEEK_MSG     "PC_FSEEK"
#define READ_MSG     "PC_FREAD"
#define WRITE_MSG    "PC_FWRITE"
#define PUTS_MSG     "PC_PUTS"
#define GETS_MSG     "PC_FGETS"
#define GETC_MSG     "PC_FGETC"
#define UNGETC_MSG   "PC_FUNGETC"
#define EXIT_MSG     "PC_FEXIT"

#define REPLY(msg, val)                                                      \
   {printf("%s:%ld\n", msg, (long) val);                                     \
    fflush(stdout);                                                          \
    if (_PC_debug)                                                           \
       {fprintf(_PC_diag, "%s:%ld\n", msg, (long) val);                      \
        fflush(_PC_diag);};}

struct s_REMOTE_FILE
   {PROCESS *pp;
    int type;
    int fid;
    long file_size;
    char *buffer;
    long sb_addr;
    long size;
    long cf_addr;};

typedef struct s_REMOTE_FILE REMOTE_FILE;

static HASHTAB
 *hosts;

static int
 SC_DECLARE(PC_rexit, (char *host));

extern long
 SC_DECLARE(_PC_transfer_bytes, 
            (int cfd, long nbe, FILE *fp)),
 SC_DECLARE(_PC_get_cmd_resp, (PROCESS *pp, char *msg));

/*--------------------------------------------------------------------------*/

/*                           SERVER SIDE ROUTINES                           */

/*--------------------------------------------------------------------------*/

/* PC_FILE_ACCESS - carry out file access commands on the named host
 *                - this is the server end
 *                - the command codes are:
 *                -   PC_FOPEN     'o'
 *                -   PC_FCLOSE    'c'
 *                -   PC_FFLUSH    'f'
 *                -   PC_FTELL     't'
 *                -   PC_FSEEK     's'
 *                -   PC_FREAD     'r'
 *                -   PC_FWRITE    'w'
 *                -   PC_FPUTS     'p'
 *                -   PC_FGETS     'g'
 *                -   PC_FGETC     'h'
 *                -   PC_FUNGETC   'u'
 *                -   PC_FEXIT     'x'
 */

int PC_file_access(log)
   int log;
   {int code, cfd, indx, ret, n_files;
    FILE *file[MAX_FILES], *fp;
    char s[MAXLINE], *bf, *t;
    static int first = TRUE;

    _PC_debug = log;

    if (_PC_debug)
       _PC_diag = fopen("PC_fs.log", "w");

    cfd = -1;
    while (TRUE)
       {if (_PC_debug)
           {fprintf(_PC_diag, "\nWaiting for command ... ");
            fflush(_PC_diag);};

        if (fgets(s, MAXLINE, stdin) == NULL)
           continue;

        if (_PC_debug)
           {fprintf(_PC_diag, "received: %s", s);
            fflush(_PC_diag);};

        code = s[0];
        indx = s[1];
        if (indx < 'A')
           continue;

        indx -= 'A';
        fp = file[indx];

        if (_PC_debug)
           {fprintf(_PC_diag, "Doing: %c on file %d(%p)\n", code, indx, fp);
            fflush(_PC_diag);};

        switch (code)
           {case PC_FOPEN :
                 {char *name, *mode;
                  int i, status;
                  long here, len;

                  if (first)
                     {int port;

                      port = PC_init_server(GET_PORT, TRUE);
                      if (port == -1)
                         return(-1);

                      REPLY("USE_PORT", port);

                      cfd = PC_init_server(GET_CONNECTION, TRUE);

                      first = FALSE;};

                  name = SC_strtok(s+2, ",\n", t);
                  mode = SC_strtok(NULL, ",\n", t);

                  if (_PC_debug)
                     {fprintf(_PC_diag, "   Open: %s, %s ... ", name, mode);
                      fflush(_PC_diag);};

                  fp = fopen(name, mode);

                  status = -1;
                  if (fp != NULL)
                     {for (i = 0; i < MAX_FILES; i++)
                          {if (file[i] == NULL)
                              {file[i] = fp;
                               break;};};

                      status = (i < MAX_FILES) ? i : -1;};

                  if (_PC_debug)
                     {fprintf(_PC_diag, "status(%d)\n", status);
                      fflush(_PC_diag);};

                  REPLY(OPEN_MSG, status);

/* report the file length */
		  here = ftell(fp);
                  fseek(fp, 0, SEEK_END);
                  len  = ftell(fp);
                  fseek(fp, here, SEEK_SET);

                  REPLY(TELL_MSG, len);}

                 break;

            case PC_FSETVBUF :
                 {int ptype, type, size;
                  char *bf;

                  ptype = SC_stol(SC_strtok(s+2, ",\n", t));
                  size  = SC_stol(SC_strtok(NULL, ",\n", t));

		  switch (ptype)
		     {case 1 :
		 	   type = _IOFBF;
			   bf   = FMAKE_N(char, size,
                                  "PC_FILE_ACCESS:bf");
			   break;
		      case 2 :
			   type = _IOLBF;
			   bf   = FMAKE_N(char, size,
                                  "PC_FILE_ACCESS:bf");
			   break;
		      case 3 :
			   type = _IONBF;
			   bf   = NULL;
			   break;
		      default :
			   
                           break;};

		  REPLY(SETVBUF_MSG, setvbuf(fp, bf, type, size));};

                 break;

            case PC_FCLOSE :
                 file[indx] = NULL;
                 if (_PC_debug)
                    {fprintf(_PC_diag, "   Close: %d(%p) ... ", indx, fp);
                     fflush(_PC_diag);};

                 REPLY(CLOSE_MSG, fclose(fp));
                 break;

            case PC_FFLUSH :
                 REPLY(FLUSH_MSG, fflush(fp));
                 break;

            case PC_FTELL :
                 REPLY(TELL_MSG, ftell(fp));
                 break;

            case PC_FSEEK :
                 {long addr;
                  int offset, whence;

                  addr   = SC_stol(SC_strtok(s+2, ",\n", t));
                  offset = SC_stol(SC_strtok(NULL, ",\n", t));

		  switch (offset)
		     {default :
		      case 0  :
		 	   whence = SEEK_SET;
			   break;
		      case 1 :
		 	   whence = SEEK_CUR;
			   break;
		      case 2 :
		 	   whence = SEEK_END;
                           break;};

                  ret = fseek(fp, addr, whence);
                  REPLY(SEEK_MSG, ret);};

                 break;

            case PC_FREAD :
                 {long nir, nbw;
                  size_t nbi, ni;

                  nbi = SC_stol(SC_strtok(s+2, ",\n", t));
                  ni  = SC_stol(SC_strtok(NULL, ",\n", t));
                  bf  = FMAKE_N(char, nbi*ni,
                        "PC_FILE_ACCESS:bf");

                  nir = fread(bf, nbi, ni, fp);
                  REPLY(READ_MSG, nir);

                  nbw = write(cfd, bf, nbi*nir);
                  REPLY(READ_MSG, nbw);

                  SFREE(bf);};

                 break;

            case PC_FWRITE :
                 {long nbe, nbw;
                  size_t nbi, ni;

                  nbi = SC_stol(SC_strtok(s+2, ",\n", t));
                  ni  = SC_stol(SC_strtok(NULL, ",\n", t));
                  nbe = nbi*ni;
                  nbw = _PC_transfer_bytes(cfd, nbe, fp);

                  REPLY(WRITE_MSG, nbw);};

                 break;

            case PC_FPUTS :
                 {long nce, nct;

                  nce = SC_stol(SC_strtok(s+2, ",\n", t));
                  nct = _PC_transfer_bytes(cfd, nce, fp);

                  REPLY(PUTS_MSG, nct);};

                 break;

            case PC_FGETS :
                 {char *ret;
                  int nc;

                  nc = SC_stol(SC_strtok(s+2, ",\n", t));

                  bf = FMAKE_N(char, nc+1,
                               "PC_FILE_ACCESS:bf");
                  memset(bf, '\n', nc+1);
                  ret = fgets(bf, nc, fp);
                  printf("%s", bf);
                  SFREE(bf);

                  REPLY(GETS_MSG, ((ret == NULL) ? -1 : 1));};

                 break;

            case PC_FGETC :
                 REPLY(GETC_MSG, fgetc(fp));
                 break;

            case PC_FUNGETC :
                 {int c;

                  c = SC_stol(SC_strtok(s+2, ",\n", t));
                  REPLY(UNGETC_MSG, ungetc(c, fp));};

                 break;

            case PC_FEXIT :
                 {int i;

                  ret = TRUE;
                  for (i = 0; i < n_files; i++)
                      ret &= fclose(file[i]);

/* close the data socket */
                  close(cfd);

                  REPLY(EXIT_MSG, ret);};

                 return(ret);

            default :
                 break;};};}

/*--------------------------------------------------------------------------*/

/*                              CLIENT SIDE ROUTINES                        */

/*--------------------------------------------------------------------------*/

/* PC_ROPEN - open the file via server on specified host
 *          - binds to io_open_hook
 *          - corresponds to PC_FOPEN message
 */

FILE *PC_ropen(name, mode)
   char *name, *mode;
   {char s[MAXLINE], host[MAXLINE], fname[MAXLINE], *t;
    int ret, type, get_data_line, log;
    long len;
    PROCESS *pp;
    REMOTE_FILE *fp;
    static int data = -1;

    log = FALSE;

/* if no host table initialize it */
    if (hosts == NULL)
       hosts = SC_make_hash_table(HSZSMALL, FALSE);

/* NOTE: remote file access from the MAC has been temporarily disabled, due
         to a conflict in directory designation syntax. This should be fixed
         someday...
*/

#ifndef MAC
/* find the host name part */
    if (strchr(name, ':') == NULL)
       {strcpy(host, "local");
        strcpy(fname, name);
        type = PC_LOCAL;}
    else
       {strcpy(s, name);
        strcpy(host, SC_strtok(s, ":", t));
        strcpy(fname, SC_strtok(NULL, "\n", t));
        type = PC_REMOTE;};

#else

    strcpy(host, "local");
    strcpy(fname, name);
    type = PC_LOCAL;

#endif

    fp = FMAKE(REMOTE_FILE, "PC_ROPEN:fp");
    fp->type = type;

    if (type == PC_LOCAL)
       {fp->pp = (PROCESS *) fopen(name, mode);
        if (fp->pp == NULL)
           {SFREE(fp);
            return(NULL);};}

    else
       {get_data_line = FALSE;

/* find the relevant process */
        pp = (PROCESS *) SC_def_lookup(host, hosts);
        if (pp == NULL)
           {char *argv[4];
            
            sprintf(s, "%s:pcexec", host);
            argv[0] = s;
            argv[1] = "-f";

/* NOTE: log should be turned on under the debugger to see the
 * transaction file from the server end
 */
	    if (log)
               {argv[2] = "-l";
                argv[3] = NULL;}
	    else
               argv[2] = NULL;

            pp = PC_open(argv, NULL, mode);
            if (pp == NULL)
               {SFREE(fp);
                return(NULL);};

            PC_block(pp);
            SC_install(host, pp, "PROCESS", hosts);

            get_data_line = TRUE;};

/* send the PC_FOPEN command to the server */
        PC_printf(pp, "%c%c%s,%s\n", PC_FOPEN, 'A', fname, mode);

        if (get_data_line)
           {int port;

            port = _PC_get_cmd_resp(pp, "USE_PORT");
            if (port < 5000)
               {PC_close(pp);
                return(NULL);};

            data = PC_init_client(SC_strtok(s, ",:", t), port);
            if (data == -1)
               {PC_close(pp);
                SFREE(fp);
                return(NULL);};

            PC_block_fd(data);
            pp->data = data;};

/* wait for the status */
        ret = _PC_get_cmd_resp(pp, OPEN_MSG);

/* wait for the file length */
        len = _PC_get_cmd_resp(pp, TELL_MSG);

        if ((ret != -1) && (pp->data != -1))
           {fp->pp        = pp;
            fp->fid       = ret + 'A';
            fp->file_size = len;}
        else
	   {PC_close(pp);
	    SFREE(fp);
	    return(NULL);};};

    return((FILE *) fp);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_RSETVBUF - set the file buffer
 *             - binds to io_setvbuf_hook
 *             - corresponds to PC_FSETVBUF message
 */

int PC_rsetvbuf(stream, bf, type, size)
   FILE *stream;
   char *bf;
   int type;
   size_t size;
   {REMOTE_FILE *fp;
    PROCESS *pp;
    int ret, ptype;

    fp = (REMOTE_FILE *) stream;
    if (fp == NULL)
       return(FALSE);

    if ((stream == stdout) || (stream == stdin) || (stream == stderr))

#ifdef MAC

       ret = FALSE;

#else

       ret = setvbuf(stream, bf, type, size);

#endif

    else if (fp->type == PC_LOCAL)
       ret = setvbuf((FILE *) fp->pp, bf, type, size);

    else
       {pp = fp->pp;

        switch (type)
           {case _IOFBF :
	         ptype = 1;
                 fp->buffer = bf;
                 fp->size   = size;
                 break;

            case _IOLBF :
	         ptype = 2;
                 fp->buffer = bf;
                 fp->size   = size;
                 break;

            case _IONBF :
	         ptype = 3;
                 fp->buffer = NULL;
                 fp->size   = 0L;
                 break;

            default :
	         return(-1);};

/* send the PC_FSETVBUF command to the server */
        PC_printf(pp, "%c%c,%d,%d\n", PC_FSETVBUF, fp->fid, ptype, size);

/* wait for the response */
        ret = _PC_get_cmd_resp(pp, SETVBUF_MSG);};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_RCLOSE - close the file wherever it is
 *           - binds to io_close_hook
 *           - corresponds to PC_FCLOSE message
 */

int PC_rclose(stream)
   FILE *stream;
   {REMOTE_FILE *fp;
    PROCESS *pp;
    int ret;

    fp = (REMOTE_FILE *) stream;
    if (fp == NULL)
       return(FALSE);

    if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       ret = fclose(stream);

    else if (fp->type == PC_LOCAL)
       ret = fclose((FILE *) fp->pp);

    else
       {pp = fp->pp;

/* send the PC_FCLOSE command to the server */
        PC_printf(pp, "%c%c\n", PC_FCLOSE, fp->fid);

/* wait for the response */
        ret = _PC_get_cmd_resp(pp, CLOSE_MSG);};

    SFREE(fp);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_RFLUSH - do an fflush on the file wherever it is
 *           - binds to io_flush_hook
 *           - corresponds to PC_FFLUSH message
 */

int PC_rflush(stream)
   FILE *stream;
   {REMOTE_FILE *fp;
    PROCESS *pp;

    fp = (REMOTE_FILE *) stream;
    if (fp == NULL)
       return(FALSE);

    if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       return(fflush(stream));

     else if (fp->type == PC_LOCAL)
       return(fflush((FILE *) fp->pp));

    else
       {pp = fp->pp;

/* send the PC_FFLUSH command to the server */
        PC_printf(pp, "%c%c\n", PC_FFLUSH, fp->fid);

/* wait for the response */
        return(_PC_get_cmd_resp(pp, FLUSH_MSG));};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_RTELL - do an ftell on the file wherever it is
 *          - binds to io_tell_hook
 *          - corresponds to PC_FTELL message
 */

long PC_rtell(stream)
   FILE *stream;
   {REMOTE_FILE *fp;
    PROCESS *pp;
    long addr;

    fp = (REMOTE_FILE *) stream;
    if (fp == NULL)
       return(-1L);

    if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       return(ftell(stream));

    else if (fp->type == PC_LOCAL)
       return(ftell((FILE *) fp->pp));

    else
       {pp = fp->pp;

/* send the PC_FTELL command to the server */
        PC_printf(pp, "%c%c\n", PC_FTELL, fp->fid);

/* wait for the response */
        addr = _PC_get_cmd_resp(pp, TELL_MSG);

/* sync the local and remote addresses */
        if (addr != fp->cf_addr)
           fp->cf_addr = addr;

        return(addr);};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_RSEEK - do an fseek on the file wherever it is
 *          - binds to io_seek_hook
 *          - corresponds to PC_FSEEK message
 */

int PC_rseek(stream, addr, offset)
   FILE *stream;
   long addr;
   int offset;
   {REMOTE_FILE *fp;
    PROCESS *pp;
    int poffs;

    fp = (REMOTE_FILE *) stream;
    if (fp == NULL)
       return(-1);

    if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       return(fseek(stream, addr, offset));

    else if (fp->type == PC_LOCAL)
       return(fseek((FILE *) fp->pp, addr, offset));

    else
       {pp = fp->pp;

        switch (offset)
           {case SEEK_SET :
	         poffs = 0;
		 fp->cf_addr = addr;
                 break;

            case SEEK_CUR :
	         poffs = 1;
		 fp->cf_addr += addr;
                 break;

            case SEEK_END :
	         poffs = 2;
		 fp->cf_addr = fp->file_size + addr;
                 break;

            default :
	         return(-1);};

/* send the PC_FSEEK command to the server */
        PC_printf(pp, "%c%c%ld,%d\n", PC_FSEEK, fp->fid, addr, poffs);

/* wait for the response */
        return(_PC_get_cmd_resp(pp, SEEK_MSG));};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_RREAD - do an fread on the file wherever it is
 *          - binds to io_read_hook
 *          - corresponds to PC_FREAD message
 */

size_t PC_rread(s, nbi, ni, stream)
   char *s;
   size_t nbi, ni;
   FILE *stream;
   {REMOTE_FILE *fp;
    PROCESS *pp;

    fp = (REMOTE_FILE *) stream;
    if (fp == NULL)
       return(0);

    if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       return(fread(s, nbi, ni, stream));

    else if (fp->type == PC_LOCAL)
       return(fread(s, nbi, ni, (FILE *) fp->pp));

    else
       {int nir, nbw, nbr, nbt;
        char *ps;

        pp = fp->pp;

/* send the PC_FREAD command to the server */
        PC_printf(pp, "%c%c%ld,%ld\n", PC_FREAD, fp->fid, nbi, ni);

/* wait for the response */
        nir = _PC_get_cmd_resp(pp, READ_MSG);

/* read the data looping until all expected bytes come in */
        nbt = nbi*nir;
        for (ps = s; TRUE; ps += nbr)
            {nbr  = read(pp->data, ps, nbt);
             nbt -= nbr;
             if (nbt <= 0)
                break;};

/* wait for the response */
        nbw = _PC_get_cmd_resp(pp, READ_MSG);

        if (nbi*nir != nbw)
           nir = 0;

/* adjust the local idea of the current disk address */
        fp->cf_addr += nbw;

/* there is some checking to do here */
        return((nir < 0) ? 0 : (size_t) nir);};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_RWRITE - do an fwrite on the file wherever it is
 *           - binds to io_write_hook
 *           - corresponds to PC_FWRITE message
 */

size_t PC_rwrite(s, nbi, ni, stream)
   char *s;
   size_t nbi, ni;
   FILE *stream;
   {REMOTE_FILE *fp;
    PROCESS *pp;
    long nbw, niw;

    fp = (REMOTE_FILE *) stream;
    if ((fp == NULL) || (nbi*ni == 0))
       return(0);

    if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       return(fwrite(s, nbi, ni, stream));

    else if (fp->type == PC_LOCAL)
       return(fwrite(s, nbi, ni, (FILE *) fp->pp));

    else
       {pp = fp->pp;

/* send the PC_FWRITE command to the server */
        PC_printf(pp, "%c%c%ld,%ld\n", PC_FWRITE, fp->fid, nbi, ni);

/* write the data */
        write(pp->data, s, nbi*ni);

        nbw = _PC_get_cmd_resp(pp, WRITE_MSG);

/* adjust the local idea of the current disk address */
        fp->cf_addr += nbw;

/* must return items even though easier to deal with bytes to this point */
	niw = nbw/nbi;

        return(niw);};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_RPUTS - do an fputs on the file
 *          - binds to io_puts_hook
 *          - corresponds to PC_FPUTS message
 */

int PC_rputs(s, stream)
   char *s;
   FILE *stream;
   {REMOTE_FILE *fp;
    PROCESS *pp;
    int nc;
    long nbw;

    fp = (REMOTE_FILE *) stream;
    if (fp == NULL)
       return(0);

    if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       return(fputs(s, stream));

    else if (fp->type == PC_LOCAL)
       return(fputs(s, (FILE *) fp->pp));

    else
       {pp = fp->pp;

        nc = strlen(s);

/* send the PC_FPUTS command to the server */
        PC_printf(pp, "%c%c%ld\n", PC_FPUTS, fp->fid, nc);

/* write the data */
        write(pp->data, s, nc);

        nbw = _PC_get_cmd_resp(pp, PUTS_MSG);

/* adjust the local idea of the current disk address */
        fp->cf_addr += nbw;

/* wait for the response */
        return((nbw != nc) ? EOF : 0);};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_RGETS - do an fgets on the file wherever it is
 *          - binds to io_gets_hook
 *          - corresponds to PC_FGETS message
 */

char *PC_rgets(s, nc, stream)
   char *s;
   int nc;
   FILE *stream;
   {REMOTE_FILE *fp;
    PROCESS *pp;
    int ret;

    fp = (REMOTE_FILE *) stream;
    if (fp == NULL)
       return(NULL);

    if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       return(fgets(s, nc, stream));

     else if (fp->type == PC_LOCAL)
       return(fgets(s, nc, (FILE *) fp->pp));

    else
       {pp = fp->pp;

/* send the PC_FGETS command to the server */
        PC_printf(pp, "%c%c%ld\n", PC_FGETS, fp->fid, nc);

/* get the string */
        PC_gets(s, nc, pp);

/* adjust the local idea of the current disk address */
        fp->cf_addr += strlen(s);

/* wait for the response */
        ret = _PC_get_cmd_resp(pp, GETS_MSG);

        return((ret == -1) ? NULL : s);};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_RGETC - do an fgetc on the file wherever it is
 *          - binds to io_getc_hook
 *          - corresponds to PC_FGETC message
 */

int PC_rgetc(stream)
   FILE *stream;
   {REMOTE_FILE *fp;
    PROCESS *pp;

    fp = (REMOTE_FILE *) stream;
    if (fp == NULL)
       return(EOF);

    if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       return(fgetc(stream));

    else if (fp->type == PC_LOCAL)
       return(fgetc((FILE *) fp->pp));

    else
       {pp = fp->pp;

/* send the PC_FGETC command to the server */
        PC_printf(pp, "%c%c\n", PC_FGETC, fp->fid);

/* adjust the local idea of the current disk address */
        fp->cf_addr++;

/* wait for the response */
        return(_PC_get_cmd_resp(pp, GETC_MSG));};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_RUNGETC - do an fungetc on the file wherever it is
 *            - binds to io_ungetc_hook
 *            - corresponds to PC_FUNGETC message
 */

int PC_rungetc(c, stream)
   int c;
   FILE *stream;
   {REMOTE_FILE *fp;
    PROCESS *pp;

    fp = (REMOTE_FILE *) stream;
    if (fp == NULL)
       return(0);

    if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       return(ungetc(c, stream));

    else if (fp->type == PC_LOCAL)
       return(ungetc(c, (FILE *) fp->pp));

    else
       {pp = fp->pp;

/* send the PC_FUNGETC command to the server */
        PC_printf(pp, "%c%c%c\n", PC_FUNGETC, fp->fid, c);

/* adjust the local idea of the current disk address */
        fp->cf_addr--;

/* wait for the response */
        return(_PC_get_cmd_resp(pp, UNGETC_MSG));};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_REXIT - close a remote file server
 *          - corresponds to PC_FEXIT message
 */

static int PC_rexit(host)
   char *host;
   {int ret;
    PROCESS *pp;

    ret = FALSE;

    pp = (PROCESS *) SC_def_lookup(host, hosts);
    if (pp != NULL)

/* send the PC_FEXIT command to the server */
       {PC_printf(pp, "%c%c\n", PC_FEXIT, 'A');

/* NOTE: this dies when the rexec'd process goes away first
        ret = _PC_get_cmd_resp(pp, EXIT_MSG);
 */

        PC_close(pp);
        SC_hash_rem(host, hosts);};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_EXIT_ALL - close all remote file servers
 *             - with the PC_FEXIT message
 */

int PC_exit_all()
   {hashel *np, **tb;
    int i, sz, ret;
    char *host;

    if (hosts == NULL)
       return(TRUE);

    ret = TRUE;

/* ream out the host table */
    sz = hosts->size;
    tb = hosts->table;
    for (i = 0; i < sz; i++)
        while ((np = tb[i]) != NULL)
           {host = np->name;
            ret &= PC_rexit(host);};

/* release the host table */
    SC_rl_hash_table(hosts);

    hosts = NULL;

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

