/*
 * GSTXT.C - PGS text sub-window routines
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
  
#include "pgs.h"

#define CTL_A '\001'
#define CTL_B '\002'
#define CTL_D '\004'
#define CTL_E '\005'
#define CTL_F '\006'
#define CTL_H '\010'
#define CTL_J '\012'
#define CTL_K '\013'
#define CTL_M '\015'
#define CTL_N '\016'
#define CTL_P '\020'
#define DEL   '\177'

#define PG_PLACE_TEXT(dev, dx, dy, p)                                        \
   {dx = dev->tcurx;                                                         \
    dy = dev->tcury;                                                         \
    PG_move_tx_abs(dev, dx, dy);                                             \
    PG_get_text_ext(dev, p, &dx, &dy);                                       \
    dev->tcurx += dx;                                                        \
    PG_write_text(dev, stdscr, p);}

static PFByte
 *_PG_std_keymap = NULL;

static char
 *SC_DECLARE(_PG_backup_char, (PG_text_box *b, char *p, int n)),
 *SC_DECLARE(_PG_text_gets, (PG_text_box *b, char *buffer,
			     int maxlen, FILE *stream));

static int
 SC_DECLARE(_PG_backward_chars, (PG_text_box *b, int n));

static void
 SC_DECLARE(_PG_clear_text_box, (PG_text_box *b, int i)),
 SC_DECLARE(_PG_move_to, (PG_text_box *b, int c, int l)),
 SC_DECLARE(_PG_write_text, (PG_text_box *b, char *s)),
 SC_DECLARE(_PG_newline, (PG_text_box *b)),
 SC_DECLARE(_PG_delete_back, (PG_text_box *b, int n)),
 SC_DECLARE(_PG_fixup_eol, (PG_text_box *b)),
 SC_DECLARE(_PG_goto_bol, (PG_text_box *b)),
 SC_DECLARE(_PG_goto_eol, (PG_text_box *b)),
 SC_DECLARE(_PG_kill_eol, (PG_text_box *b)),
 SC_DECLARE(_PG_forward_char, (PG_text_box *b)),
 SC_DECLARE(_PG_backward_char, (PG_text_box *b)),
 SC_DECLARE(_PG_next_line, (PG_text_box *b)),
 SC_DECLARE(_PG_previous_line, (PG_text_box *b)),
 SC_DECLARE(_PG_delete_current, (PG_text_box *b)),
 SC_DECLARE(_PG_delete_previous, (PG_text_box *b));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_OPEN_TEXT_RECT - open up a special region in which to enter and
 *                   - display text
 */
 
PG_text_box *PG_open_text_rect(dev, name, type, keymap, crv)
   PG_device *dev;
   char *name;
   int type;
   PFByte *keymap;
   PG_curve *crv;
   {char **bf;
    REAL x1, x2, y1, y2;
    REAL dx, dy, line_frac;
    int i, ni, n_linep, n_lines, n_chars;
    PG_text_box *b;

    if (_PG_std_keymap == NULL)
       {ni = 256;
	_PG_std_keymap = FMAKE_N(PFByte, ni,
                         "PG_OPEN_TEXT_RECT:std_keymap");

	if ((_SC_zero_space != 1) && (_SC_zero_space != 2))

	   memset(_PG_std_keymap, 0, ni*sizeof(PFByte));

        _PG_std_keymap[CTL_A] = (PFByte) _PG_goto_bol;
        _PG_std_keymap[CTL_B] = (PFByte) _PG_backward_char;
        _PG_std_keymap[CTL_D] = (PFByte) _PG_delete_current;
        _PG_std_keymap[CTL_E] = (PFByte) _PG_goto_eol;
        _PG_std_keymap[CTL_F] = (PFByte) _PG_forward_char;
        _PG_std_keymap[CTL_H] = (PFByte) _PG_delete_previous;
        _PG_std_keymap[CTL_J] = (PFByte) _PG_newline;
        _PG_std_keymap[CTL_K] = (PFByte) _PG_kill_eol;
        _PG_std_keymap[CTL_M] = (PFByte) _PG_newline;
        _PG_std_keymap[CTL_N] = (PFByte) _PG_next_line;
        _PG_std_keymap[CTL_P] = (PFByte) _PG_previous_line;
        _PG_std_keymap[DEL]   = (PFByte) _PG_delete_previous;};

    b = FMAKE(PG_text_box, "PG_OPEN_TEXT_RECT:b");

    PG_get_text_ext_NDC(dev, "s", &dx, &dy);
    PG_get_curve_extent(dev, crv, NORMC, &x1, &x2, &y1, &y2);
    n_linep = ABS(y1 - y2)/dy;
    n_chars = ABS(x1 - x2)/dx;

    n_linep = max(1, n_linep);
    n_chars = max(1, n_chars);

/* keep bottom line well away from the margin */
    line_frac = n_linep*(dev->char_height_s);
    if (line_frac > 0.99)
       n_linep--;
    
/* this might want to be different at some point */
    n_lines = n_linep;

    b->name         = SC_strsavef(name, "char*:PG_OPEN_TEXT_BOX:name");
    b->type         = type;
    b->dev          = dev;
    b->bnd          = crv;
    b->line         = 0;
    b->col          = 0;
    b->align        = LEFT;
    b->mode         = TEXT_INSERT;
    b->foreground   = dev->WHITE;
    b->background   = dev->BLACK;
    b->border       = 0.1;
    b->standoff     = 5;
    b->n_lines      = n_lines;
    b->n_chars_line = n_chars;
    b->n_lines_page = n_linep;
    b->text_buffer  = bf = FMAKE_N(char *, n_lines,
                           "PG_OPEN_TEXT_RECT:text_buffer");

    for (i = 0; i < n_lines; i++)
        bf[i] = FMAKE_N(char, n_chars, "PG_OPEN_TEXT_RECT:bf[]");

    if (keymap == NULL)
       b->keymap = _PG_std_keymap;
    else
       b->keymap = keymap;

    b->backup  = _PG_backup_char;
    b->gets    = _PG_text_gets;
    b->clear   = _PG_clear_text_box;
    b->write   = _PG_write_text;
    b->newline = _PG_newline;

    _PG_clear_text_box(b, 0);

    return(b);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_OPEN_TEXT_BOX - open up a special region in which to enter and
 *                  - display text
 */
 
PG_text_box *PG_open_text_box(dev, name, type, keymap, xf, yf, dxf, dyf)
   PG_device *dev;
   char *name;
   int type;
   PFByte *keymap;
   double xf, yf, dxf, dyf;
   {PG_curve *crv;

    crv = PG_make_box_curve(dev, NORMC, 0.0, 0.0,
			    xf, xf + dxf,
			    yf, yf + dyf);

    return(PG_open_text_rect(dev, name, type, keymap, crv));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_CLOSE_TEXT_BOX - close a text box */
 
void PG_close_text_box(b)
   PG_text_box *b;
   {int i, ln;
    char **bf;

    ln = b->n_lines;
    bf = b->text_buffer;
    for (i = 0; i < ln; i++)
        {SFREE(bf[i]);};

    SFREE(bf);

    PG_release_curve(b->bnd);

    SFREE(b);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_CLEAR_TEXT_BOX - clear the current page
 *                    - and go to the line i on the screen
 */
 
static void _PG_clear_text_box(b, i)
   PG_text_box *b;
   int i;
   {PG_device *dev;
    PG_curve *crv;
    char **bf;
    double bwid;
    int ln, cn;

    dev = b->dev;
    crv = b->bnd;

    PG_set_color_fill(dev, b->background, FALSE);
    PG_fill_curve(dev, crv);

    b->col  = 0;
    b->line = i;

    bf = b->text_buffer;
    ln = b->n_lines;
    cn = b->n_chars_line;
    for (i = 0; i < ln; i++)
        {memset(bf[i], 0, cn);};

    bwid = b->border;
    if (bwid > 0.0)
       {REAL wid;

        PG_get_line_width(dev, &wid);
        PG_set_line_width(dev, bwid);
        PG_draw_curve(dev, crv, FALSE);
        PG_set_line_width(dev, wid);};

    PG_update_vs(dev);

    PG_release_current_device(dev);

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_REFRESH_TEXT_BOX - redraw all lines of text in the box
 *                     - do a clear first
 */

void PG_refresh_text_box(b)
   PG_text_box *b;
   {char **bf;
    int i, ln;
    double ang, bwid;
    REAL x, y;
    PG_device *dev;
    PG_curve *crv;

    dev = b->dev;
    crv = b->bnd;
    ang = b->angle;
    if (ang != 0)
       b->n_lines = 1;
    ln  = b->n_lines;
    bf  = b->text_buffer;

    if (HARDCOPY_DEVICE(dev))
       {PG_set_color_fill(dev, b->background, FALSE);}
    else
       PG_set_fill_color(dev, b->background);

    PG_fill_curve(dev, crv);

    bwid = b->border;
    if (bwid > 0.0)
       {REAL wid;

        PG_get_line_width(dev, &wid);
        PG_set_line_width(dev, bwid);
        (*dev->set_line_color)(dev, b->foreground, FALSE);
        PG_draw_curve(dev, crv, FALSE);
        PG_set_line_width(dev, wid);};

    PG_get_char_path(dev, &x, &y);
    PG_set_char_path(dev, cos(ang), sin(ang));
    (*dev->set_text_color)(dev, b->foreground, FALSE);

    for (i = 0; i < ln; i++)
        {_PG_move_to(b, 0, i);
	 PG_write_text(dev, stdscr, bf[i]);};

    PG_set_char_path(dev, x, y);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_MOVE_TO - move to the specified line and column of a text box */

static void _PG_move_to(b, c, l)
   PG_text_box *b;
   int c, l;
   {PG_device *dev;
    PG_curve *bnd;
    int ix, iy, so, xo, yo, xr, yr, xl, yl, *x, *y;
    REAL fx, fy, dx, dxe, dxp, dy, sign;
    double ang;
    char *bf;

    dev  = b->dev;
    sign = (dev->quadrant == QUAD_ONE) ? 1.0 : -1.0;
    ang  = b->angle;
    bf   = b->text_buffer[l];
    PG_get_text_ext_NDC(dev, bf, &dxe, &dy);
    PG_get_text_ext_NDC(dev, bf+c, &dxp, &dy);
    dx   = dxe*dev->window_width;

    bnd = b->bnd;
    so  = b->standoff;
    x   = bnd->x;
    xl  = x[0];
    xr  = x[1];
    xo  = bnd->x_origin + xl;
    y   = bnd->y;
    yl  = y[0];
    yr  = y[3];
    if (ABS(sin(ang)) > sin(PI/4.0))
       yo = bnd->y_origin + yl;
    else
       yo = bnd->y_origin + yr - 0.8*sign*dy*dev->window_height;

    switch (b->align)
       {default   :
        case LEFT :
             ix = xo + cos(ang)*so;
             iy = yo + sign*sin(ang)*so;
             break;

        case CENTER :
             ix = xo + cos(ang)*0.5*(xr - xl - dx);
             iy = yo + sign*sin(ang)*0.5*(ABS(yr - yl) - dx);
             break;

        case RIGHT :
             ix = xo + xr - xl - cos(ang)*(so + dx);
             iy = yo + sign*sin(ang)*(ABS(yr - yl) - dx - so);
             break;};

    PtoS(dev, ix, iy, fx, fy);

    fx += cos(ang)*(dxe - dxp) + sin(ang)*(l*dy);
    fy += sin(ang)*(dxe - dxp) - cos(ang)*(l*dy);

    StoW(dev, fx, fy);
    PG_move_tx_abs(dev, fx, fy);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_INSERT_TEXT - insert text into the given string
 *                 - return the overflow
 */

static char *_PG_insert_text(bf, ncpl, c, s)
   char *bf;
   int ncpl, c;
   char *s;
   {char *tmp, *p, *q, *overflow;
    int i, n, nb, ns, cl;

    nb = strlen(bf);
    ns = strlen(s);

    tmp = FMAKE_N(char, ncpl+ns, "_PG_INSERT_TEXT:tmp");
    strcpy(tmp, bf);

    p = tmp + nb;
    q = p - ns;
    n = nb - c;
    for (i = 0; i < n; i++, *p-- = *q--);

    memcpy(tmp+c, s, ns);

    cl = ns + nb - ncpl;
    if (cl > 0)
       {strcpy(s, tmp+ncpl);
	memcpy(bf, tmp, ncpl);
        bf[ncpl] = '\0';
        overflow = s;}

    else
       {strcpy(bf, tmp);
        overflow = NULL;};

    SFREE(tmp);

    return(overflow);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_CLEAR_LINE - clear the line to the end */

static void _PG_clear_line(b, l)
   PG_text_box *b;
   int l;
   {PG_device *dev;
    char *bf;

    dev = b->dev;
    bf  = b->text_buffer[l];

    _PG_move_to(b, 0, l);
    (*dev->set_text_color)(dev, b->background, FALSE);
    PG_write_text(dev, stdscr, bf);

    return;}    

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DRAW_CURSOR - draw the cursor in the inverse video or normal */

void _PG_draw_cursor(b, inv)
   PG_text_box *b;
   int inv;
   {PG_device *dev;
    double xo, yo, xmn, xmx, ymn, ymx;
    REAL dx, dy;
    PG_curve *crv;
    char *bf, crsr[2];
    int c, l, fc, bc, eol;

/* it is possible for the PG_text_box to be GC'd to test the device
 * and just leave if its gone
 */
    dev = b->dev;
    if (dev == NULL)
       return;

    c = b->col;
    l = b->line;

    bf  = b->text_buffer[l];

    crsr[0] = bf[c];
    crsr[1] = '\0';
    if (crsr[0] == '\0')
       {eol = TRUE;
        crsr[0] = 's';}
    else
       eol = FALSE;

    _PG_move_to(b, c, l);
    PG_get_text_ext_NDC(dev, crsr, &dx, &dy);

    xo = 0.0;
    yo = 0.0;

    xmn = dev->tcurx;
    ymn = dev->tcury;
    WtoS(dev, xmn, ymn);
    ymn -= 0.3*dy;

    xmx = xmn + dx;
    ymx = ymn + dy;
    crv = PG_make_box_curve(dev, NORMC, xo, yo, xmn, xmx, ymn, ymx);

    if (inv)
       {fc = b->foreground;
        bc = b->background;}
    else
       {fc = b->background;
        bc = b->foreground;};

    PG_set_fill_color(dev, fc);
    PG_fill_curve(dev, crv);
    (*dev->set_text_color)(dev, bc, FALSE);

    if (!eol)
       PG_write_text(dev, stdscr, crsr);

    PG_release_curve(crv);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_REDRAW_TEXT_LINE - draw a proper image of the text line */

static void _PG_redraw_text_line(b, c, l, flag)
   PG_text_box *b;
   int c, l, flag;
   {PG_device *dev;
    char *bf;

    dev = b->dev;
    bf  = b->text_buffer[l];

    _PG_clear_line(b, l);
    _PG_move_to(b, 0, l);
    (*dev->set_text_color)(dev, b->foreground, FALSE);
    PG_write_text(dev, stdscr, bf);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_WRITE_TEXT - write out text to the appropriate device */
 
static void _PG_write_text(b, s)
   PG_text_box *b;
   char *s;
   {char *bf, *overflow;
    int col, ln, lmx, ncpl, cr, cl, ns, fl;

    col  = b->col;
    ln   = b->line;
    lmx  = b->n_lines_page;
    ncpl = b->n_chars_line;
    ns   = strlen(s);

    switch (b->mode)
       {default          :
        case TEXT_INSERT :
	     for (overflow = s;
		  (overflow != NULL) && (ln < lmx);
		  col = 0, ln++)
	         {_PG_clear_line(b, ln);
		  bf       = b->text_buffer[ln];
		  overflow = _PG_insert_text(bf, ncpl, col, overflow);};

             col = b->col + ns;
             ln  = b->line;
             for (fl = FALSE;
		  (col >= ncpl) && (ln < lmx);
		  col -= ncpl, ln++, fl = TRUE)
                 _PG_redraw_text_line(b, 0, ln, fl);

	     b->col  = col;
             if (ln < lmx)
	        {_PG_redraw_text_line(b, col, ln, TRUE);
		 b->line = ln;};
	     break;

        case TEXT_OVERWRITE :
             bf = b->text_buffer[ln];
             cl = ncpl - col;
	     if (ns <= cl)
	        {strcpy(bf+col, s);
		 _PG_redraw_text_line(b, col, ln, FALSE);
		 b->col = col + ns;}
	     else
	        {strncpy(bf+col, s, cl);
		 _PG_redraw_text_line(b, col, ln, FALSE);

		 bf = b->text_buffer[++(b->line)];
		 cr = ns - cl;
		 strncpy(bf, s+cl, cr);
		 _PG_redraw_text_line(b, cr, ++ln, TRUE);

		 b->col = cr;};

	     break;};

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_NEWLINE - do a controlled newline */
 
static void _PG_newline(b)
   PG_text_box *b;
   {int i, line, nl, nc;
    char *p0, *p, **bf;
    PG_device *dev;

    dev  = b->dev;
    line = b->line++;
    bf   = b->text_buffer;
    nl   = b->n_lines_page - 2;
    nc   = b->n_chars_line;

/* this is for a MORE window */
    if (b->type == MORE_WINDOW)
       {if (line < nl)
	   line++;
	else
	   {nl++;
            _PG_move_to(b, 0, nl);
	    PG_write_text(dev, stdscr, "More ...");
            PG_update_vs(dev);
	    PG_get_char(dev);

	    PG_set_fill_color(dev, b->background);
	    PG_fill_curve(dev, b->bnd);

	    for (i = 0; i < nl; i++)
	        memset(bf[i], 0, nc);
	    line = 0;};}

/* this is for a SCROLLING window */
    else if (b->type == SCROLLING_WINDOW)
       {if (line < b->n_lines_page)
	   line++;
        else
	   {p0 = bf[0];
	    memset(p0, 0, nc);

	    PG_set_fill_color(dev, b->background);
	    PG_fill_curve(dev, b->bnd);

	    for (i = 0; i < line; i++)
	        {_PG_move_to(b, 0, i);
		 p = bf[i] = bf[i+1];
		 PG_write_text(dev, stdscr, p);};

	    bf[line] = p0;
	    line = nl+1;};}

    else
       {nl += 2;
        line = min(line, nl);};
    
    b->line = line;
    b->col  = 0;

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_BACKUP_CHAR - move the text cursor back the
 *                 - specified number of characters in the buffer
 *                 - pointed to by p
 */

static char *_PG_backup_char(b, p, n)
   PG_text_box *b;
   char *p;
   int n;
   {REAL dx, dy;
    PG_device *dev;

    dev = b->dev;

    p -= n;

    PG_get_text_ext(dev, p, &dx, &dy);

    PG_move_tx_rel(dev, -dx, 0);
    (*dev->set_text_color)(dev, b->background, FALSE);
    PG_write_text(dev, stdscr, p);

    PG_move_tx_rel(dev, 0, 0);

    return(p);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_TEXT_GETS - do fgets for a text box */
 
static char *_PG_text_gets(b, buffer, maxlen, stream)
   PG_text_box *b;
   char *buffer;
   int maxlen;
   FILE *stream;
   {char *p, eof;
    Register char c;
    static char prompt = '_', cr = 13, bs = 8;
    char **bf;
    REAL dx, dy;
    int line, nb, nc;

    p   = buffer;
    eof = (char) EOF;
 
    if (stream == stdin)
       {PG_device *dev;

        dev  = b->dev;
        line = b->line;
        bf   = b->text_buffer;

        p[0] = prompt;
        p[1] = '\0';

        PG_PLACE_TEXT(dev, dx, dy, p);
        p++;

        while (((c = PG_get_char(dev)) != eof) && (c != cr))
           {if (c != bs)
               {p    = _PG_backup_char(b, p, 1);
                p[0] = (c == '\n') ? ';' : c;
                p[1] = prompt;
                p[2] = '\0';

                PG_PLACE_TEXT(dev, dx, dy, p);

                p += 2;}
            else if (p > buffer+1)
               {p    = _PG_backup_char(b, p, 2);
                p[0] = prompt;
                p[1] = '\0';

                PG_PLACE_TEXT(dev, dx, dy, p);

                p++;};};

        p = _PG_backup_char(b, p, 1);

        if (c == cr)
           {*p++ = '\n';
            *p = '\0';
            p  = buffer;

            nb = strlen(bf[line]);
            nc = b->n_chars_line;
            strncat(bf[line], p, nc-nb-2);

            _PG_newline(b);};
                     
        if (c == eof)
           p = NULL;

        return(p);}
    else
       return(io_gets(p, maxlen, stream));}
 
/*--------------------------------------------------------------------------*/

/*                            EDITING COMMANDS                              */

/*--------------------------------------------------------------------------*/

/* _PG_BACKWARD_CHARS - move backward n characters */

static int _PG_backward_chars(b, n)
   PG_text_box *b;
   int n;
   {int oln, ocol, nln, ncol;

    oln  = b->line;
    ocol = b->col;
    nln  = oln;
    ncol = ocol - n;
    if (ncol < 0)
       {if (oln == 0)
           return(FALSE);

        nln  = oln - 1;
	ncol = strlen(b->text_buffer[nln]) - n + 1;};

    b->line = nln;
    b->col  = ncol;

    _PG_redraw_text_line(b, ncol, nln, TRUE);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DELETE_BACK - delete the character n back from current position */

static void _PG_delete_back(b, n)
   PG_text_box *b;
   int n;
   {char *bf, *p, *s;
    int col, ln;

    if (_PG_backward_chars(b, n))
       {ln  = b->line;
	col = b->col;
	_PG_clear_line(b, ln);

	bf = b->text_buffer[ln] + col;
	for (p = bf, s = bf + 1; (*p++ = *s++););

	_PG_redraw_text_line(b, col, ln, TRUE);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_FIXUP_EOL - move the position back to the end of text if needed */

static void _PG_fixup_eol(b)
   PG_text_box *b;
   {int colx, col, ln;

    ln = b->line;
    if (ln < b->n_lines)
       {col  = b->col;
	colx = strlen(b->text_buffer[ln]);
	if (colx < col)
	   b->col = colx;

	_PG_redraw_text_line(b, col, ln, TRUE);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_GOTO_BOL - goto the beginning of the line */

static void _PG_goto_bol(b)
   PG_text_box *b;
   {int ln;

    ln = b->line;

    b->col = 0;

    _PG_redraw_text_line(b, 0, ln, TRUE);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_GOTO_EOL - goto the end of the line */

static void _PG_goto_eol(b)
   PG_text_box *b;
   {int ln, ncol;

    ln = b->line;

    b->col = ncol = strlen(b->text_buffer[ln]);

    _PG_redraw_text_line(b, ncol, ln, TRUE);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_KILL_EOL - kill to the end of the line */

static void _PG_kill_eol(b)
   PG_text_box *b;
   {int col, ln;
    char *bf;

    col = b->col;
    ln = b->line;
    _PG_clear_line(b, ln);

    bf = b->text_buffer[ln];
    bf[col] = '\0';

    _PG_redraw_text_line(b, col, ln, TRUE);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_FORWARD_CHAR - move forward one character */

static void _PG_forward_char(b)
   PG_text_box *b;
   {int oln, ocol, nln, ncol, colx;

    ocol = b->col;
    oln  = b->line;
    colx = strlen(b->text_buffer[oln]);
    ncol = ocol + 1;
    nln  = oln;
    if (ncol > colx)
       {nln  = oln + 1;
        ncol = 0;
        if (nln >= b->n_lines)
           return;};

    b->line = nln;
    b->col  = ncol;

    _PG_redraw_text_line(b, ncol, nln, TRUE);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_BACKWARD_CHAR - move backward one character */

static void _PG_backward_char(b)
   PG_text_box *b;
   {_PG_backward_chars(b, 1);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_NEXT_LINE - move to the next line */

static void _PG_next_line(b)
   PG_text_box *b;
   {int ln, lnx;

    ln  = ++(b->line);
    lnx = b->n_lines;
    if (ln >= lnx)
       b->line = lnx - 1;

    _PG_fixup_eol(b);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PREVIOUS_LINE - move to the previous line */

static void _PG_previous_line(b)
   PG_text_box *b;
   {int ln;

    ln = --(b->line);
    if (ln < 0)
       b->line = 0;

    _PG_fixup_eol(b);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DELETE_CURRENT - delete the current character */

static void _PG_delete_current(b)
   PG_text_box *b;
   {_PG_delete_back(b, 0);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DELETE_PREVIOUS - delete the previous character */

static void _PG_delete_previous(b)
   PG_text_box *b;
   {_PG_delete_back(b, 1);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
