/*
 * GSSURF.C - surface plotting routines for PGS
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"

#define LABS(x) abs((int) (x))

#ifdef BGI
# define SET_PIXEL(ix, iy, color, cmn, cmx)                                  \
    {if ((cmn <= color) && (color <= cmx))                                   \
        putpixel(ix, iy, color);}
#endif

#ifdef MSC
# define SET_PIXEL(ix, iy, color, cmn, cmx)                                  \
    {if ((cmn <= color) && (color <= cmx))                                   \
        {PG_SET_COLOR(color);                                                \
	 _setpixel(ix, iy);};}
#endif

#ifndef SET_PIXEL
# define SET_PIXEL(ix, iy, color, cmn, cmx)                                  \
    {if ((cmn <= color) && (color <= cmx))                                   \
        {lx = aix*ix - bix;                                                  \
         ly = aiy*iy - biy;                                                  \
         surf_img_bf[ly*nx + lx] = color;};}
#endif

static double
 _PG_3d_axis_standoff = 0.03;

static unsigned char
 *surf_img_bf;

static int
 aix,
 aiy,
 bix, 
 biy,
 nx,
 ny;

char
 PG_err[MAXLINE];

/*--------------------------------------------------------------------------*/

/*                     MISCELLANEOUS 3D ROUTINES                            */

/*--------------------------------------------------------------------------*/

/* PG_AXIS_3D - draw a set of axes for a surface plot */

void PG_axis_3d(dev, px, py, pz, n_pts, theta, phi, chi,
		xmn, xmx, ymn, ymx, zmn, zmx,
		norm)
   PG_device *dev;
   REAL *px, *py, *pz;
   int n_pts;
   double theta, phi, chi;
   double xmn, xmx, ymn, ymx, zmn, zmx;
   int norm;
   {REAL p1[9], p2[9], p3[9];

/* draw the axes */
    PG_set_color_line(dev, dev->WHITE, TRUE);
    PG_set_line_style(dev, 3);

    PG_axis_3d_limits(dev, px, py, pz, n_pts, theta, phi, chi,
		      norm, TRUE, FALSE,
		      xmn, xmx, ymn, ymx, zmn, zmx,
		      p1, p2, p3);

/* x-axis */
    PG_draw_line(dev, p1[0], p2[0], p1[1], p2[1]);
    PG_write_WC(dev, p1[1], p2[1], "X");

/* y-axis */
    PG_draw_line(dev, p1[0], p2[0], p1[2], p2[2]);
    PG_write_WC(dev, p1[2], p2[2], "Y");

/* z-axis */
    PG_draw_line(dev, p1[0], p2[0], p1[4], p2[4]);
    PG_write_WC(dev, p1[4], p2[4], "Z");

    PG_set_line_style(dev, 1);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_AXIS_3D_LIMITS - find the rotated limits of 3d axes */

void PG_axis_3d_limits(dev, px, py, pz, n_pts, theta, phi, chi,
		       norm, offs, wcflag,
		       xmn, xmx, ymn, ymx, zmn, zmx,
		       p1, p2, p3)
   PG_device *dev;
   REAL *px, *py, *pz;
   int n_pts;
   double theta, phi, chi;
   int norm, offs, wcflag;
   double xmn, xmx, ymn, ymx, zmn, zmx;
   REAL *p1, *p2, *p3;
   {int i, j, np;
    double x, y, z;
    double dx, dy, dz;
    REAL rx[18], ry[18], rz[18];

/* find endpoints of the axes if not supplied */
    if (((xmn >= xmx) || (ymn >= ymx) || (zmn >= zmx)) && (n_pts > 0))
       {xmx = -HUGE_REAL;
        ymx = -HUGE_REAL;
        zmx = -HUGE_REAL;
        xmn =  HUGE_REAL;
        ymn =  HUGE_REAL;
        zmn =  HUGE_REAL;
	for (i = 0; i < n_pts; i++)
	    {x = px[i];
	     y = py[i];
	     z = pz[i];
	     xmx = max(xmx, x);
	     ymx = max(ymx, y);
	     zmx = max(zmx, z);
	     xmn = min(xmn, x);
	     ymn = min(ymn, y);
	     zmn = min(zmn, z);};};

    if (offs)
       {dx = (xmx - xmn)*_PG_3d_axis_standoff;
        dy = (ymx - ymn)*_PG_3d_axis_standoff;
        dz = (zmx - zmn)*_PG_3d_axis_standoff;}
    else
       {dx = 0.0;
        dy = 0.0;
        dz = 0.0;};

    np = 9;

/* left down back */
    rx[0] = xmn - dx;
    ry[0] = ymn - dy;
    rz[0] = zmn - dz;

/* right down back */
    rx[1] = xmx + dx;
    ry[1] = ymn - dy;
    rz[1] = zmn - dz;

/* left up back */
    rx[2] = xmn - dx;
    ry[2] = ymx + dy;
    rz[2] = zmn - dz;

/* right up back */
    rx[3] = xmx + dx;
    ry[3] = ymx + dy;
    rz[3] = zmn - dz;

/* left down front */
    rx[4] = xmn - dx;
    ry[4] = ymn - dy;
    rz[4] = zmx + dz;

/* right down front */
    rx[5] = xmx + dx;
    ry[5] = ymn - dy;
    rz[5] = zmx + dz;

/* left up front */
    rx[6] = xmn - dx;
    ry[6] = ymx + dy;
    rz[6] = zmx + dz;

/* right up front */
    rx[7] = xmx + dx;
    ry[7] = ymx + dy;
    rz[7] = zmx + dz;

/* absolute origin */
    rx[8] = xmn;
    ry[8] = ymn;
    rz[8] = zmn;

/* rotate the axes */
    PG_rotate_3d_WC(rx, ry, rz, xmn, xmx, ymn, ymx, zmn, zmx,
                    theta, phi, chi, p1, p2, p3, np, norm);

/* if requested set the WC to contain the axes */
    if (wcflag)
       {REAL lxmx, lxmn, lymx, lymn;

        lxmn =  HUGE;
        lymn =  HUGE;
        lxmx = -HUGE;
        lymx = -HUGE;
        for (j = 0; j < 8; j++)
	    {lxmn = min(lxmn, p1[j]);
	     lxmx = max(lxmx, p1[j]);
	     lymn = min(lymn, p2[j]);
	     lymx = max(lymx, p2[j]);};

	PG_set_window(dev, lxmn, lxmx, lymn, lymx);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_ROTATE_3D_WC - rotate the three vector (px, py, pz)
 *                 - by theta, phi, and chi into the three
 *                 - vector (rx, ry, rz)
 *                 - rotations (clockwise) are:
 *                 -    by phi about z axis from positive y axis
 *                 -    by theta about x axis from positive z' axis 
 *                 -    by chi about z axis from positive y'' axis
 *                 - all vectors are in world coordinates
 */

void PG_rotate_3d_WC(px, py, pz, xmn, xmx, ymn, ymx, zmn, zmx,
                     theta, phi, chi, rx, ry, rz, n, norm)
   REAL *px, *py, *pz;
   double xmn, xmx, ymn, ymx, zmn, zmx, theta, phi, chi;
   REAL *rx, *ry, *rz;
   int n, norm;
   {int i;
    REAL ct, st, cp, sp, cc, sc;
    REAL ctcp, stcp, ctsp, stsp;
    REAL r11, r12, r13, r21, r22, r23, r31, r32, r33;
    REAL x, y, z;
    REAL dx, dy, dz;

    ct = cos(theta);
    st = sin(theta);
    cp = cos(phi);
    sp = sin(phi);
    cc = cos(chi);
    sc = sin(chi);

    if (norm)
       {dx = xmx - xmn;
        dy = ymx - ymn;
        dz = zmx - zmn;
	dx = (dx == 0.0) ? 1.0 : 1.0/dx;
        dy = (dy == 0.0) ? 1.0 : 1.0/dy;
        dz = (dz == 0.0) ? 1.0 : 1.0/dz;}
    else
       {dx = 1.0;
        dy = 1.0;
        dz = 1.0;};

    ctcp = ct*cp;
    ctsp = ct*sp;
    stcp = st*cp;
    stsp = st*sp;

    r11 =  cc*cp - sc*stsp;
    r12 =  cc*sp + sc*ctcp;
    r13 =  sc*st;
    r21 = -sc*cp - cc*ctsp;
    r22 = -sc*sp + cc*ctcp;
    r23 =  cc*st;
    r31 =  stsp;
    r32 = -stcp;
    r33 =  ct;

    for (i = 0; i < n; i++)
        {x = (*px++)*dx;
         y = (*py++)*dy;
         z = (*pz++)*dz;

         *rx++ = r11*x + r12*y + r13*z;
         *ry++ = r21*x + r22*y + r23*z;
         *rz++ = r31*x + r32*y + r33*z;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_TRANSFORM_WC_PC - transform a mesh in world coordinates to one
 *                     - in pixel coordinates
 *                     - return the extrema that fall in the viewport
 */

static void _PG_transform_WC_PC(dev, rx, ry, irx, iry, n,
				pxmn, pxmx, pymn, pymx)
   PG_device *dev;
   REAL *rx, *ry;
   int *irx, *iry;
   int n;
   int *pxmn, *pxmx, *pymn, *pymx;
   {int i, ix, iy, ds;
    int ixmn, ixmx, iymn, iymx;
    REAL sxmn, sxmx, symn, symx;
    REAL x, y;

    sxmn = dev->sxmin;
    sxmx = dev->sxmax;
    symn = dev->symin;
    symx = dev->symax;

    ds = dev->resolution_scale_factor;

    ixmn = INT_MAX;
    ixmx = INT_MIN;
    iymn = INT_MAX;
    iymx = INT_MIN;
    for (i = 0; i < n; i++)
        {x = *rx++;
	 y = *ry++;

	 WtoS(dev, x, y);
	 StoP(dev, x, y, ix, iy);

	 ix /= ds;
	 iy /= ds;

         if (((sxmn <= x) && (x <= sxmx)) &&
	     ((symn <= y) && (y <= symx)))
            {ixmn = min(ixmn, ix);
             ixmx = max(ixmx, ix);
             iymn = min(iymn, iy);
             iymx = max(iymx, iy);};

	 *irx++ = ix;
	 *iry++ = iy;};

    *pxmn = ixmn;
    *pxmx = ixmx;
    *pymn = iymn;
    *pymx = iymx;

    return;}

/*--------------------------------------------------------------------------*/

/*                           WORKER LEVEL ROUTINES                          */

/*--------------------------------------------------------------------------*/

/* _PG_FILL_SCAN_LINE - fill in the scan line between ixca and ixcb
 *                    - ixca <= ixcb is guaranteed
 */

static int _PG_fill_scan_line(dev, sweep, ena, enb,
			      ixca, ixcb, idx, iyca, iycb, idy,
			      zca, zcb, zmn, zmx, fca, fcb, fmn, fmx,
			      z_buffer, type, color, mesh)
   PG_device *dev;
   int sweep, ena, enb;
   long ixca, ixcb, idx;
   long iyca, iycb, idy;
   double zca, zcb, zmn, zmx;
   double fca, fcb, fmn, fmx;
   REAL *z_buffer;
   int type, color, mesh;
   {int ixc, iyc, clr, co, in;
    double dz, zc, df, fc, fs, fa, fb, fd;
    int lx, ly, cmx, backgr;
    PG_palette *pl;

    ixc = ixca;
    iyc = iyca;
    zc  = zca;
    fc  = fca;

    if (idy == 0L)
       {dz = (zcb - zca)/(ixcb - ixca + SMALL);
        df = (fcb - fca)/(ixcb - ixca + SMALL);
        in = ixc;}
    else
       {dz = (zcb - zca)/(iycb - iyca + SMALL);
        df = (fcb - fca)/(iycb - iyca + SMALL);
        in = iyc;};

    cmx = dev->current_palette->n_pal_colors + 1;

    if (POSTSCRIPT_DEVICE(dev) && !COLOR_POSTSCRIPT_DEVICE(dev))
       {co     = 2;
        backgr = cmx;}
    else
       {co     = (type == PLOT_SURFACE) << 1;
        backgr = dev->BLACK;};

    if (ena && mesh && (zc >= z_buffer[in]))
       {z_buffer[in] = zc;
	if ((fmn <= fc) && (fc <= fmx))
	   SET_PIXEL(ixc, iyc, color, color, color);
        in++;
        ixc += idx;
        iyc += idy;
        zc  += dz;
        fc  += df;};

/* fill the z-buffer with the contribution from this line's interior */
    switch (type)
       {case PLOT_SURFACE   :
             pl = dev->current_palette;
	     fs = pl->n_pal_colors;
	     fd = fmx - fmn;
             fa = (fd == 0.0) ? 0.0 : fs/fd;
             fb = fmn*fa;
	     for (; (ixc < ixcb) && (iyc < iycb);
		  ixc += idx, iyc += idy, zc += dz, fc += df, in++)
	         if (zc >= z_buffer[in])
		    {z_buffer[in] = zc;
		     clr = fa*fc - fb + co;
		     SET_PIXEL(ixc, iyc, clr, co, cmx);};
             break;

        case PLOT_WIRE_MESH : 
	     for (; (ixc < ixcb) && (iyc < iycb);
		  ixc += idx, iyc += idy, zc += dz, in++)
	         if (zc >= z_buffer[in])
		    {z_buffer[in] = zc;
		     SET_PIXEL(ixc, iyc, backgr, co, cmx);};

	     break;};

    if (enb && mesh && (zc >= z_buffer[in]))
       {z_buffer[in] = zc;
	if ((fmn <= fc) && (fc <= fmx))
	   SET_PIXEL(ixc, iyc, color, color, color);};

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_SCAN_TRIANGLE - scan convert a line crossing the triangle 1-2-3
 *                   - into the z-buffer
 *                   - the line connecting 1 and 2 is the only one visible
 *                   - this is intended to be the boundary of the
 *                   - partitioned polygon
 *                   - point 1 is below point 2 with respect to y
 *                   - sweep 0 is vertical sweep of horizontal lines
 *                   - sweep 1 is horizontal sweep of vertical lines
 */

static int _PG_scan_triangle(dev, z_buffer, iyc, nd, sweep,
			     ixmn, ixmx, iymn, iymx,
			     ix1, iy1, ix2, iy2, ix3, iy3,
			     z1, z2, z3, zmn, zmx, f1, f2, f3,
			     extr, color, mesh, type)
   PG_device *dev;
   REAL *z_buffer;
   int iyc, nd, sweep;
   int ixmn, ixmx, iymn, iymx;
   int ix1, iy1, ix2, iy2, ix3, iy3;
   double z1, z2, z3, zmn, zmx;
   REAL *f1, *f2, *f3, *extr;
   int color, mesh, type;
   {double dy12, dy23, dy31, fmn, fmx;
    long dy1c, dy2c, dy3c;
    long ixca, ixcb, it;
    int ena, enb;
    REAL zca, zcb, zt, fca, fcb;
    int crosses_12, crosses_23, crosses_31;
    int intersects_1, intersects_2, intersects_3;

/* NOTE: short term hack which throws away the generality that has
 * been passed down to this level
 */
    f1 += (nd - 1);
    f2 += (nd - 1);
    f3 += (nd - 1);
    fmn = extr[2*(nd - 1)];
    fmx = extr[2*(nd - 1) + 1];

    dy1c = iy1 - iyc;
    dy2c = iy2 - iyc;
    dy3c = iy3 - iyc;

    intersects_1 = (iy1 == iyc);
    intersects_2 = (iy2 == iyc);
    intersects_3 = (iy3 == iyc);

    crosses_12 = ((iy1 < iyc) && (iyc < iy2));
    crosses_31 = (((iy1 < iyc) && (iyc < iy3)) || ((iy3 < iyc) && (iyc < iy1)));
    crosses_23 = (((iy2 < iyc) && (iyc < iy3)) || ((iy3 < iyc) && (iyc < iy2)));

/* check all cases in which the scan line crosses vertex #1 */
    if (intersects_1)
       {if (intersects_2)
           {if (ix1 <= ix2)
	       {ixca = ix1;
                zca  = z1;
                fca  = *f1;
		ena  = TRUE;
                ixcb = ix2;
                zcb  = z2;
                fcb  = *f2;
		enb  = TRUE;}
	    else
	       {ixca = ix2;
                zca  = z2;
                fca  = *f2;
		ena  = TRUE;
                ixcb = ix1;
                zcb  = z1;
                fcb  = *f1;
		enb  = TRUE;};}

        else if ((intersects_3) || (crosses_23))
	   {dy23 = 1.0/((double) (iy2 - iy3) + SMALL);
	    ixcb = (ix3*dy2c - ix2*dy3c)*dy23;
	    zcb  = (z3*dy2c - z2*dy3c)*dy23;
	    fcb  = (*f3*dy2c - *f2*dy3c)*dy23;
	    enb  = FALSE;

	    if (ix1 <= ixcb)
	       {ixca = ix1;
                zca  = z1;
                fca  = *f1;
		ena  = TRUE;}
	    else
	       {ixca = ixcb;
                zca  = zcb;
                fca  = fcb;
		ena  = enb;
                ixcb = ix1;
                zcb  = z1;
                fcb  = *f1;
		enb  = TRUE;};}

        else
	   {ixca = ix1;
	    zca  = z1;
	    fca  = *f1;
	    ena  = TRUE;
	    ixcb = ix1;
	    zcb  = z1;
	    fcb  = *f1;
	    enb  = FALSE;};}

/* check all cases in which the scan line crosses vertex #2 excluding above */
    else if (intersects_2)
       {if ((intersects_3) || (crosses_31))
	   {dy31 = 1.0/((double) (iy1 - iy3) + SMALL);
	    ixcb = (ix3*dy1c - ix1*dy3c)*dy31;
	    zcb  = (z3*dy1c - z1*dy3c)*dy31;
	    fcb  = (*f3*dy1c - *f1*dy3c)*dy31;
	    enb  = FALSE;

	    if (ix2 <= ixcb)
	       {ixca = ix2;
                zca  = z2;
                fca  = *f2;
		ena  = TRUE;}
	    else
	       {ixca = ixcb;
                zca  = zcb;
                fca  = fcb;
		ena  = enb;
                ixcb = ix2;
                zcb  = z2;
                fcb  = *f2;
		enb  = TRUE;};}
        else
	   {ixca = ix2;
	    zca  = z2;
	    fca  = *f2;
	    ena  = TRUE;
	    ixcb = ix2;
	    zcb  = z2;
	    fcb  = *f2;
	    enb  = FALSE;};}

/* check all cases in which the scan line crosses vertex #3 excluding above */
    else if (intersects_3)
       {if (crosses_12)
	   {dy12 = 1.0/((double) (iy1 - iy2) + SMALL);
	    ixcb = (ix2*dy1c - ix1*dy2c)*dy12;
	    zcb  = (z2*dy1c - z1*dy2c)*dy12;
	    fcb  = (*f2*dy1c - *f1*dy2c)*dy12;
	    enb  = TRUE;

	    if (ix3 <= ixcb)
	       {ixca = ix3;
                zca  = z3;
                fca  = *f3;
		ena  = FALSE;}
	    else
	       {ixca = ixcb;
                zca  = zcb;
                fca  = fcb;
		ena  = enb;
                ixcb = ix3;
                zcb  = z3;
                fcb  = *f3;
		enb  = FALSE;};}
        else
	   {ixca = ix3;
	    zca  = z3;
	    fca  = *f3;
	    ena  = FALSE;
	    ixcb = ix3;
	    zcb  = z3;
	    fcb  = *f3;
	    enb  = FALSE;};}

/* check all cases in which the scan line crosses line 1-2 excluding above */
    else if (crosses_12)
       {dy12 = 1.0/((double) (iy1 - iy2) + SMALL);
	ixcb = (ix2*dy1c - ix1*dy2c)*dy12;
	zcb  = (z2*dy1c - z1*dy2c)*dy12;
	fcb  = (*f2*dy1c - *f1*dy2c)*dy12;
	enb  = TRUE;

        if (crosses_23)
	   {dy23 = 1.0/((double) (iy2 - iy3) + SMALL);
	    ixca = (ix3*dy2c - ix2*dy3c)*dy23;
	    zca  = (z3*dy2c - z2*dy3c)*dy23;
	    fca  = (*f3*dy2c - *f2*dy3c)*dy23;
	    ena  = FALSE;
	    if (ixca > ixcb)
	       {it   = ixca;
		ixca = ixcb;
		ixcb = it;
	        zt   = zca;
		zca  = zcb;
		zcb  = zt;
	        zt   = fca;
		fca  = fcb;
		fcb  = zt;
		ena  = TRUE;
		enb  = FALSE;};}

        else if (crosses_31)
	   {dy31 = 1.0/((double) (iy1 - iy3) + SMALL);
	    ixca = (ix3*dy1c - ix1*dy3c)*dy31;
	    zca  = (z3*dy1c - z1*dy3c)*dy31;
	    fca  = (*f3*dy1c - *f1*dy3c)*dy31;
	    ena  = FALSE;
	    if (ixca > ixcb)
	       {it   = ixca;
		ixca = ixcb;
		ixcb = it;
	        zt   = zca;
		zca  = zcb;
		zcb  = zt;
	        zt   = fca;
		fca  = fcb;
		fcb  = zt;
		ena  = TRUE;
		enb  = FALSE;};}

        else
           return(FALSE);}

/* check all cases in which the scan line crosses line 2-3 excluding above */
    else if (crosses_23)
       {dy23 = 1.0/((double) (iy2 - iy3) + SMALL);
	ixcb = (ix3*dy2c - ix2*dy3c)*dy23;
	zcb  = (z3*dy2c - z2*dy3c)*dy23;
	fcb  = (*f3*dy2c - *f2*dy3c)*dy23;
	enb  = FALSE;
        if (crosses_31)
	   {dy31 = 1.0/((double) (iy1 - iy3) + SMALL);
	    ixca = (ix3*dy1c - ix1*dy3c)*dy31;
	    zca  = (z3*dy1c - z1*dy3c)*dy31;
	    fca  = (*f3*dy1c - *f1*dy3c)*dy31;
	    ena  = FALSE;
	    if (ixca > ixcb)
	       {it   = ixca;
		ixca = ixcb;
		ixcb = it;
	        zt   = zca;
		zca  = zcb;
		zcb  = zt;
	        zt   = fca;
		fca  = fcb;
		fcb  = zt;};}
        else
           return(FALSE);}

    else if (crosses_31)
       return(FALSE);

/* clip to the viewport limits */
    if (((ixca < ixmn) && (ixcb < ixmn)) ||
	((ixca > ixmx) && (ixcb > ixmx)))
       return(FALSE);

    if (ixca < ixmn)
       {zca += (zcb - zca)*((double) (ixmn - ixca))/
	                   ((double) (ixcb - ixca));
        ixca = ixmn;};

    if (ixmx < ixcb)
       {zcb += (zca - zcb)*((double) (ixcb - ixmx))/
	                   ((double) (ixcb - ixca));
        ixcb = ixmx;};

/* printf("%d\t%d\t%d\t%d\t%d\n", ena, enb, iyc, ixca, ixcb); */
    if (sweep == 0)
       _PG_fill_scan_line(dev, sweep, ena, enb,
			  ixca, ixcb, 1L,
			  (long)iyc, iyc+1L, 0L,
			  zca, zcb, zmn, zmx,
			  fca, fcb, fmn, fmx,
			  z_buffer, type, color, mesh);
    else
       _PG_fill_scan_line(dev, sweep, ena, enb,
			  (long)iyc, iyc+1L, 0L,
			  ixca, ixcb, 1L,
			  zca, zcb, zmn, zmx,
			  fca, fcb, fmn, fmx,
			  z_buffer, type, color, mesh);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SCAN_TRIANGLE - scan convert a line crossing the triangle 1-2-3
 *                  - into the z-buffer
 *                  - the lines 1-2 and 2-3 are invisible
 *                  - this is intended to be in the interior of the
 *                  - partitioned polygon
 */

static void PG_scan_triangle(dev, z_buffer, indx, nd, sweep,
			     ixmn, ixmx, iymn, iymx,
			     ix1, iy1, a1, f1, ixc, iyc, ac, fc,
			     ix2, iy2, a2, f2,
			     amn, amx, extr, color, mesh, type)
   PG_device *dev;
   REAL *z_buffer;
   int indx, nd, sweep;
   int ixmn, ixmx, iymn, iymx;
   int ix1, iy1;
   double a1;
   REAL *f1;
   int ixc, iyc;
   double ac;
   REAL *fc;
   int ix2, iy2;
   double a2;
   REAL *f2;
   double amn, amx;
   REAL *extr;
   int color, mesh, type;
   {int iya, iyb;

    if (((ix1 < ixmn) && (ixc < ixmn) && (ix2 < ixmn)) ||
        ((ix1 > ixmx) && (ixc > ixmx) && (ix2 > ixmx)))
       return;

    if (iy1 <= iy2)
       {iya = max(iy2, iyc);
	iyb = min(iy1, iyc);
	if ((iyb <= indx) && (indx <= iya))
	   _PG_scan_triangle(dev, z_buffer, indx, nd, sweep,
			     ixmn, ixmx, iymn, iymx,
			     ix1, iy1, ix2, iy2,
			     ixc, iyc, a1, a2, ac, amn, amx,
			     f1, f2, fc, extr, color, mesh, type);}
    else
       {iya = max(iy1, iyc);
	iyb = min(iy2, iyc);
	if ((iyb <= indx) && (indx <= iya))
	   _PG_scan_triangle(dev, z_buffer, indx, nd, sweep,
			     ixmn, ixmx, iymn, iymx,
			     ix2, iy2, ix1, iy1,
			     ixc, iyc, a2, a1, ac, amn, amx,
			     f2, f1, fc, extr, color, mesh, type);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DRAW_SURFACE - draw a surface plot given a known mesh type, a
 *                  - suitable scanning function, and a scalar array
 *                  - rotate to view angle, plot with hidden line
 *                  - algorithm, and shade if requested
 */

static void _PG_draw_surface(dev, nd, f, extr, x, y, nn, xmn, xmx, ymn, ymx,
			     theta, phi, chi,
			     width, color, mesh, style, type, name,
			     cnnct, alist, fnc_scan)
   PG_device *dev;
   int nd;
   REAL **f, *extr, *x, *y;
   int nn;
   double xmn, xmx, ymn, ymx;
   double theta, phi, chi, width;
   int color, mesh, style, type;
   char *name;
   byte *cnnct;
   pcons *alist;
   PFByte fnc_scan;
   {int i, n, iymn, iymx, n_pixel, mpts, nic, npp, ds;
    int *irx, *iry;
    int ix, iy, jx, jy, lx, ly;
    int axmn, axmx, aymn, aymx;
    REAL uxmn, uxmx, uymn, uymx;
    REAL p1[9], p2[9], p3[9];
    REAL *rx, *ry, *rz, *z_buffer, *ppx, *ppy;
    int ixmn, ixmx;
    REAL vz, vmn, vmx;

    ds = dev->resolution_scale_factor;

/* allocate space for the rotated points */
    mpts = max(nn, 12);
    rx = FMAKE_N(REAL, mpts, "_PG_DRAW_SURFACE:rx");
    ry = FMAKE_N(REAL, mpts, "_PG_DRAW_SURFACE:ry");
    rz = FMAKE_N(REAL, mpts, "_PG_DRAW_SURFACE:rz");

    irx = FMAKE_N(int, nn, "_PG_DRAW_SURFACE:irx");
    iry = FMAKE_N(int, nn, "_PG_DRAW_SURFACE:iry");

/* rotate X into rX */
    PG_rotate_3d_WC(x, y, f[0], xmn, xmx, ymn, ymx, extr[0], extr[1],
                    theta, phi, chi, rx, ry, rz, nn, TRUE);

/* determine the WC including the axis limits */
    PG_axis_3d_limits(dev, x, y, f[0], nn, theta, phi, chi,
		      TRUE, FALSE, TRUE,
		      xmn, xmx, ymn, ymx, extr[0], extr[1],
		      p1, p2, p3);

    PM_convex_hull(p1, p2, 8, &ppx, &ppy, &npp);

    PG_set_clipping(dev, FALSE);
  
/* transform to PC for efficiency */
    _PG_transform_WC_PC(dev, rx, ry, irx, iry, nn,
                        &ixmn, &ixmx, &iymn, &iymx);

/* find mins and maxes for the integer mesh */
    vmn =  HUGE_REAL;
    vmx = -HUGE_REAL;
    for (i = 0; i < nn; i++)
        {vz  = rz[i];
         vmn = min(vmn, vz);
         vmx = max(vmx, vz);};

    vmn -= 1.0e-3*(vmx - vmn);

    PG_get_viewport_size(dev, &jx, &jy, &nx, &ny);

    ny += 2;

/* adjust resolution downward if the image would be too big to fit in memory */
    if (!_PG_allocate_image_buffer(dev, NULL, &nx, &ny))
       return;

/* these values will be used to scale the scan lines */
    lx = ixmx - ixmn + 1;
    ly = iymx - iymn + 1;

/* NOTE: devices such as CGM insist on even values for image dimensions */
    nx = max(nx, lx);
    ny = max(ny, ly);
    if (nx & 1)
       {nx++;
        lx++;
        ixmx++;};

    if (ny & 1)
       {ny++;
        ly++;
        iymx++;};

    n = nx*ny;
    surf_img_bf = FMAKE_N(unsigned char, n,
                          "_PG_DRAW_SURFACE:surf_img_bf");
    if (surf_img_bf == NULL)
       return;

    if (POSTSCRIPT_DEVICE(dev) && !COLOR_POSTSCRIPT_DEVICE(dev))
       i = dev->current_palette->n_pal_colors + 1;
    else if (CGM_DEVICE(dev) && (dev->current_palette != dev->palettes))
       i = 1;
    else
       i = dev->BLACK;
    memset(surf_img_bf, i, n);

    n_pixel = (max(dev->window_width, dev->window_height))/ds;

    z_buffer = FMAKE_N(REAL, n_pixel,
                       "_PG_DRAW_SURFACE:z_buffer");

    aix = lx/(ixmx - ixmn);
    bix = ixmn*aix;
    aix = max(aix, 1);

    switch (dev->quadrant)
       {default        :
        case QUAD_FOUR :
	     aiy = 1;
	     biy = iymn;
             break;

	case QUAD_ONE :
	     aiy = -1;
	     biy = -(ny + iymn - 1);
             break;};

/* do a vertical sweep of horizontal scan lines */
    for (i = iymn; i <= iymx; i++)
        {PM_set_value(z_buffer, n_pixel, vmn);

/* compute the imposed domain limits on the scan line */
         iy = i*ds;

         ix = ixmn*ds;
         PtoS(dev, ix, iy, uxmn, uymn);
         StoW(dev, uxmn, uymn);

         ix = ixmx*ds;
         PtoS(dev, ix, iy, uxmx, uymx);
         StoW(dev, uxmx, uymx);
         if (PM_intersect_line_polygon(&uxmn, &uymn, &uxmx, &uymx,
                                       ppx, ppy, npp, &nic))
            {WtoS(dev, uxmn, uymn);
             StoP(dev, uxmn, uymn, ix, iy);
             axmn = ix/ds;
             aymn = iy/ds;

             WtoS(dev, uxmx, uymx);
             StoP(dev, uxmx, uymx, ix, iy);
             axmx = ix/ds;
             aymx = iy/ds;

	     (*fnc_scan)(dev, z_buffer, rz, nd, f, extr, irx, iry,
			 axmn, axmx, aymn, aymx,
			 i, 0, color, mesh, type, cnnct, alist);};};

#if 0
/* do a horizontal sweep of vertical scan lines */
    for (i = ixmn; i <= ixmx; i++)
        {PM_set_value(z_buffer, n_pixel, vmn);

/* compute the imposed domain limits on the scan line */
         PtoS(dev, ixmn, i, uxmn, uymn);
         PtoS(dev, ixmx, i, uxmx, uymx);
         if (PM_intersect_line_polygon(&uxmn, &uymn, &uxmx, &uymx,
                                       ppx, ppy, npp, &nic))
            {StoP(dev, uxmn, uymn, axmn, aymn);
             StoP(dev, uxmx, uymx, axmx, aymx);

	     (*fnc_scan)(dev, z_buffer, rz, nd, f, extr, irx, iry,
			 axmn, axmx, aymn, aymx,
			 i, 1, dev->WHITE, mesh, type, cnnct, alist);};};
#endif

    ixmn *= ds;
    iymn *= ds;

    switch (PG_hl_clear_mode)
       {case CLEAR_SCREEN :
             PG_clear_window(dev);
             break;
        case CLEAR_VIEWPORT :
             PG_clear_viewport(dev);
             break;
        case CLEAR_FRAME :
             PG_clear_frame(dev);
             break;};

    PG_invert_image_data(surf_img_bf, nx, ny, 1);
    PG_put_image(dev, surf_img_bf, ixmn, iymn, nx, ny);

    PG_draw_polyline(dev, ppx, ppy, npp, FALSE);

    SFREE_N(surf_img_bf, nx*ny);

/* release the temporaries */
    SFREE(irx);
    SFREE(iry);
    SFREE(rx);
    SFREE(ry);
    SFREE(rz);
    SFREE(z_buffer);

    SFREE(ppx);
    SFREE(ppy);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SURFACE_HAND - draw a surface plot */

static void PG_surface_hand(dev, g, type, cnnct, fnc_map, fnc_scan)
   PG_device *dev;
   PG_graph *g;
   int type;
   byte *cnnct;
   PFPREAL fnc_map;
   PFByte fnc_scan;
   {int i, n_nodes, npts, nd, same;
    int centering, color, mesh, style, rexfl;
    char bf[MAXLINE], *mtype, *s;
    REAL width, theta, phi, chi;
    REAL *rx, *ry, dx, dy, xmin, ymin;
    REAL **r, *dextr, *rextr, **afd;
    byte **afs;
    PM_mapping *h;
    PM_set *domain, *range;
    pcons *alst;

    PG_get_fill_bound(dev, mesh);

    if ((g->info != NULL) && (strcmp(g->info_type, SC_PCONS_P_S) == 0))
       {int *pc, *ps, *pm;
        REAL *pt, *pp, *px, *pw;

	pt = NULL;
	pp = NULL;
	px = NULL;
	pw = NULL;
	pc = NULL;
	ps = NULL;
	pm = NULL;
        SC_assoc_info((pcons *) g->info,
                      "THETA", &pt,
                      "PHI", &pp,
                      "CHI", &px,
                      "DRAW-MESH", &pm,
                      "LINE-COLOR", &pc,
                      "LINE-WIDTH", &pw,
                      "LINE-STYLE", &ps,
                      NULL);

	theta = (pt == NULL) ? 0.0 : *pt;
	phi   = (pp == NULL) ? 0.0 : *pp;
	chi   = (px == NULL) ? 0.0 : *px;
	mesh  = (pm == NULL) ? TRUE : *pm;
	color = (pc == NULL) ? dev->WHITE : *pc;
	width = (pw == NULL) ? 0.0 : *pw;
	style = (ps == NULL) ? SOLID : *ps;}

    else
       {theta = 0.0;
	phi   = 0.0;
	chi   = 0.0;
        color = dev->WHITE;
        width = 0.0;
        style = SOLID;};

    mesh   = (type == PLOT_WIRE_MESH) ? TRUE : mesh;
    h      = g->f;
    domain = h->domain;

    strcpy(bf, domain->element_type);
    mtype   = SC_strtok(bf, " *", s);
    npts    = domain->n_elements;
    r       = (REAL **) domain->elements;
    n_nodes = npts;

    rx = PM_array_real(domain->element_type, r[0], npts, NULL);
    ry = PM_array_real(domain->element_type, r[1], npts, NULL);

    dextr = PM_array_real(domain->element_type, domain->extrema, 4, NULL);

    PG_get_viewport_WC(dev, &xmin, &dx, &ymin, &dy);
    dx -= xmin;
    dy -= ymin;

    range = h->range;
    strcpy(bf, range->element_type);
    mtype = SC_strtok(bf, " *", s);
    same  = (strcmp(mtype, SC_REAL_S) == 0);
    npts  = range->n_elements;
    nd    = range->dimension_elem;

    afd   = FMAKE_N(REAL *, nd, "PG_SURFACE_HAND:afd");
    afs   = (byte **) range->elements;

/* find the range limits if any */
    rextr = PM_get_limits(range);

/* setup the range limits */
    rexfl = (rextr == NULL);
    if (same)
       {if (rexfl)
           rextr = (REAL *) range->extrema;
        for (i = 0; i < nd; i++)
            afd[i] = (REAL *) afs[i];}
    else
       {for (i = 0; i < nd; i++)
            CONVERT(SC_REAL_S, &afd[i], mtype, afs[i], npts, FALSE);

        if (rexfl)
           {rextr = NULL;
	    CONVERT(SC_REAL_S, &rextr, mtype, range->extrema,
		    2*nd, FALSE);};};

    PG_register_range_extrema(dev, nd, rextr);

/* find the additional mapping information */
    centering = N_CENT;
    alst = PM_mapping_info(h,
			   "CENTERING", &centering,
			   NULL);

/* this is done consistently with PG_draw_vector */
    switch (centering)
       {case Z_CENT :
             {REAL **afdp;

	      afdp  = FMAKE_N(REAL *, nd, "PG_SURFACE_HAND:afdp");
	      for (i = 0; i < nd; i++)
                  afdp[i] = (*fnc_map)(afd[i], cnnct, alst);

	      _PG_draw_surface(dev, nd, afdp, rextr, rx, ry, n_nodes,
			       xmin, xmin+dx, ymin, ymin+dy,
			       theta, phi, chi, width,
			       color, mesh, style, type,
			       h->name, cnnct, alst, fnc_scan);

	      for (i = 0; i < nd; i++)
		  SFREE(afdp[i]);
	      SFREE(afdp);};

	     break;

        case N_CENT :
             _PG_draw_surface(dev, nd, afd, rextr, rx, ry, n_nodes,
			      xmin, xmin+dx, ymin, ymin+dy,
			      theta, phi, chi, width,
			      color, mesh, style, type,
			      h->name, cnnct, alst, fnc_scan);
	     break;

        case F_CENT :
        case U_CENT :
        default     : break;};

    SFREE(rx);
    SFREE(ry);

    if (!same)
       {for (i = 0; i < nd; i++)
            SFREE(afd[i]);

        if (rexfl)
           SFREE(rextr);};

    SFREE(afd);
    SFREE(dextr);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_PICTURE_SURFACE - setup a window for a surface rendering
 *                          - NOTE: no drawing of any kind is to be done here
 */

PG_picture_desc *PG_setup_picture_surface(dev, data, save, clear)
   PG_device *dev;
   PG_graph *data;
   int save, clear;
   {int rendering, nde, nre, change;
    REAL xend;
    REAL *dpex, *ddex, *pdx, *rpex, *rdex, *prx;
    PG_picture_desc *pd;
    PG_par_rend_info *pri;
    PG_device *dd;
    pcons *alst;

    change = !dev->supress_setup;

    pd = PG_get_rendering_properties(dev, data);

    pd->legend_fl = FALSE;

    rendering = pd->rendering;
    alst      = pd->alist;
    pri       = dev->pri;
    if (pri != NULL)
       {dd = pri->dd;
	if (dd != NULL)
	   {dd->pri->alist  = alst;
	    dd->pri->render = rendering;};};

/* setup the viewport */
    if (rendering == PLOT_SURFACE)
       {if (change)
	   {xend = 0.725 - _PG_palette_width;
	    PG_set_viewport(dev, 0.175, xend, 0.175, 0.825);};}
    else
	pd->palette_fl = FALSE;

    if (change)

/* find the extrema for this frame */
       {PG_find_extrema(data, 0.0, &dpex, &rpex, &nde, &ddex, &nre, &rdex);

/* setup the domain limits */
	pdx = ((dev->autodomain == TRUE) || (dpex == NULL)) ? ddex : dpex;
	PG_set_window(dev, pdx[0], pdx[1], pdx[2], pdx[3]);

/* setup the range limits */
	prx = ((dev->autorange == TRUE) || (rpex == NULL)) ? rdex : rpex;
	PG_register_range_extrema(dev, nre, prx);

	SFREE(ddex);
	SFREE(rdex);

/* GOTCHA: this isn't the only time we want 3d axes */
	if ((pd->theta != HUGE) || (nde == 3))
	   pd->ax_type = CARTESIAN_3D;

/* set surface plot attribute values */
	PG_set_clipping(dev, FALSE);};

    return(pd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_SURFACE_CORE - surface plot skeleton routine */

static void _PG_surface_core(dev, data, cnnct, fnc_map, fnc_scan)
   PG_device *dev;
   PG_graph *data;
   byte *cnnct;
   PFPREAL fnc_map;
   PFByte fnc_scan;
   {PG_graph *g;
    PG_picture_desc *pd;
	
    pd = PG_setup_picture(dev, data, TRUE, FALSE, TRUE);

/* plot all of the current functions */
    for (g = data; g != NULL; g = g->next)
        PG_surface_hand(dev, g, g->rendering, cnnct, fnc_map, fnc_scan);

    PG_finish_picture(dev, data, pd);

    return;}

/*--------------------------------------------------------------------------*/

/*                       SCAN CONVERSION ROUTINES                           */

/*--------------------------------------------------------------------------*/

/* PG_SCAN_CONVERT_LR - using a one line z-buffer
 *                    - scan convert each polygon intersecting the line
 *                    - flag is one of PLOT_WIRE_MESH or PLOT_SURFACE
 *                    - sweep 0 is vertical sweep of horizontal lines
 *                    - sweep 1 is horizontal sweep of vertical lines
 */

static void PG_scan_convert_lr(dev, z_buffer, f1, nd, f, extr, rx, ry,
			       ixmn, ixmx, iymn, iymx,
			       indx, sweep, color, mesh, type,
			       cnnct, alist)
   PG_device *dev;
   REAL *z_buffer;
   REAL *f1;
   int nd;
   REAL **f, *extr;
   int *rx, *ry;
   int ixmn, ixmx, iymn, iymx;
   int indx, sweep, color, mesh, type;
   byte *cnnct;
   pcons *alist;
   {int i, i1, j, j0, k, l, n, nmap, km, lm, eflag;
    int ix1, ix2, ix3, ix4, iy1, iy2, iy3, iy4;
    int ixc, iyc;
    int *rx1, *rx2, *rx3, *rx4;
    int *ry1, *ry2, *ry3, *ry4;
    int *maxes, kmax, lmax;
    double amn, amx;
    REAL u1, u2, u3, u4, uc;
    REAL *v1, *v2, *v3, *v4, *vc;
    REAL *ru1, *ru2, *ru3, *ru4, *ft;
    char *emap;

    maxes = (int *) cnnct;
    kmax  = maxes[0];
    lmax  = maxes[1];
    n     = kmax*lmax;
    nmap  = (kmax - 1) * (lmax - 1);

    LR_MAPPING_INFO(alist, nmap);

    PM_CHECK_EMAP(alist, nmap, eflag, emap);

    amn = extr[0];
    amx = extr[1];

    PM_LOGICAL_ZONE(rx, rx1, rx2, rx3, rx4, kmax);
    PM_LOGICAL_ZONE(ry, ry1, ry2, ry3, ry4, kmax);
    PM_LOGICAL_ZONE(f1, ru1, ru2, ru3, ru4, kmax);

    v1 = FMAKE_N(REAL, nd, "PG_SCAN_CONVERT_LR:v1");
    v2 = FMAKE_N(REAL, nd, "PG_SCAN_CONVERT_LR:v2");
    v3 = FMAKE_N(REAL, nd, "PG_SCAN_CONVERT_LR:v3");
    v4 = FMAKE_N(REAL, nd, "PG_SCAN_CONVERT_LR:v4");
    vc = FMAKE_N(REAL, nd, "PG_SCAN_CONVERT_LR:vc");

    j0 = (nd < 2) ? 0 : 1;

    km = kmax - 1;
    lm = lmax - 1;
    for (l = 0; l < lm; l++)
        for (k = 0; k < km; k++)
            {i  = l*kmax + k;
             i1 = l*km + k;
	     if (emap[i1] == 0)
	        continue;

             iy1 = ry1[i];
             iy2 = ry2[i];
             iy3 = ry3[i];
             iy4 = ry4[i];

             ix1 = rx1[i];
             ix2 = rx2[i];
             ix3 = rx3[i];
             ix4 = rx4[i];

             u1  = ru1[i];
             u2  = ru2[i];
             u3  = ru3[i];
             u4  = ru4[i];

             for (j = j0; j < nd; j++)
                 {ft    = f[j];
                  v1[j] = ft[i+1];
                  v2[j] = ft[i+kmax+1];
                  v3[j] = ft[i+kmax];
                  v4[j] = ft[i];
                  vc[j] = 0.25*(v1[j] + v2[j] + v3[j] + v4[j]);};

             ixc = 0.25*(ix1 + ix2 + ix3 + ix4);
             iyc = 0.25*(iy1 + iy2 + iy3 + iy4);
             uc  = 0.25*(u1 + u2 + u3 + u4);

             if (sweep == 0)
	        {PG_scan_triangle(dev, z_buffer, indx, nd, sweep,
				  ixmn, ixmx, iymn, iymx,
				  ix1, iy1, u1, v1,
				  ixc, iyc, uc, vc,
				  ix2, iy2, u2, v2,
				  amn, amx, extr,
				  color, mesh, type);

		 PG_scan_triangle(dev, z_buffer, indx, nd, sweep,
				  ixmn, ixmx, iymn, iymx,
				  ix2, iy2, u2, v2,
				  ixc, iyc, uc, vc,
				  ix3, iy3, u3, v3,
				  amn, amx, extr,
				  color, mesh, type);

		 PG_scan_triangle(dev, z_buffer, indx, nd, sweep,
				  ixmn, ixmx, iymn, iymx,
				  ix3, iy3, u3, v3,
				  ixc, iyc, uc, vc,
				  ix4, iy4, u4, v4,
				  amn, amx, extr,
				  color, mesh, type);

		 PG_scan_triangle(dev, z_buffer, indx, nd, sweep,
				  ixmn, ixmx, iymn, iymx,
				  ix4, iy4, u4, v4,
				  ixc, iyc, uc, vc,
				  ix1, iy1, u1, v1,
				  amn, amx, extr,
				  color, mesh, type);}
	     else
	        {PG_scan_triangle(dev, z_buffer, indx, nd, sweep,
				  ixmn, ixmx, iymn, iymx,
				  iy1, ix1, u1, v1,
				  iyc, ixc, uc, vc,
				  iy2, ix2, u2, v2,
				  amn, amx, extr,
				  color, mesh, type);

		 PG_scan_triangle(dev, z_buffer, indx, nd, sweep,
				  ixmn, ixmx, iymn, iymx,
				  iy2, ix2, u2, v2,
				  iyc, ixc, uc, vc,
				  iy3, ix3, u3, v3,
				  amn, amx, extr,
				  color, mesh, type);

		 PG_scan_triangle(dev, z_buffer, indx, nd, sweep,
				  ixmn, ixmx, iymn, iymx,
				  iy3, ix3, u3, v3,
				  iyc, ixc, uc, vc,
				  iy4, ix4, u4, v4,
				  amn, amx, extr,
				  color, mesh, type);

		 PG_scan_triangle(dev, z_buffer, indx, nd, sweep,
				  ixmn, ixmx, iymn, iymx,
				  iy4, ix4, u4, v4,
				  iyc, ixc, uc, vc,
				  iy1, ix1, u1, v1,
				  amn, amx, extr,
				  color, mesh, type);};};

    SFREE(v1);
    SFREE(v2);
    SFREE(v3);
    SFREE(v4);
    SFREE(vc);

    if (eflag)
        SFREE(emap);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SCAN_CONVERT_AC - using a one line z-buffer
 *                    - scan convert each polygon intersecting the line
 *                    - flag is one of PLOT_WIRE_MESH or PLOT_SURFACE
 *                    - sweep 0 is vertical sweep of horizontal lines
 *                    - sweep 1 is horizontal sweep of vertical lines
 */

static void PG_scan_convert_ac(dev, z_buffer, f1, nd, f, extr, rx, ry,
			       ixmn, ixmx, iymn, iymx,
			       indx, sweep, color, mesh, type,
			       cnnct, alist)
   PG_device *dev;
   REAL *z_buffer;
   REAL *f1;
   int nd;
   REAL **f, *extr;
   int *rx, *ry;
   int ixmn, ixmx, iymn, iymx;
   int indx, sweep, color, mesh, type;
   byte *cnnct;
   pcons *alist;
   {int j, j0, is, os, iz, oz, is1, is2, in1, in2;
    int ixc, iyc, ix1, iy1, ix2, iy2;
    int *nc, nz, *np, nzp, nsp;
    long **cells, *zones, *sides;
    PM_mesh_topology *mt;
    double amn, amx, norm;
    REAL uc, u1, u2, *vc, *v1, *v2, *ft;

    amn = extr[0];
    amx = extr[1];

    mt = (PM_mesh_topology *) cnnct;

    cells = mt->boundaries;
    zones = cells[2];
    sides = cells[1];

    nc = mt->n_cells;
    nz = nc[2];

    np  = mt->n_bound_params;
    nzp = np[2];
    nsp = np[1];

    v1 = FMAKE_N(REAL, nd, "PG_SCAN_CONVERT_AC:v1");
    v2 = FMAKE_N(REAL, nd, "PG_SCAN_CONVERT_AC:v2");
    vc = FMAKE_N(REAL, nd, "PG_SCAN_CONVERT_AC:vc");

    j0 = (nd < 2) ? 0 : 1;

    for (iz = 0; iz < nz; iz++)
        {oz  = iz*nzp;
	 is1 = zones[oz];
	 is2 = zones[oz+1];

         if (nzp >= CENTER_CELL)
            {in1 = zones[oz + CENTER_CELL];
             uc  = f1[in1];
	     for (j = j0; j < nd; j++)
	         {ft    = f[j];
		  vc[j] = ft[in1];};
	     ixc = rx[in1];
	     iyc = ry[in1];}
         else
  	    {uc  = 0.0;
	     for (j = j0; j < nd; j++)
  	         vc[j] = 0.0;

	     ixc = 0;
  	     iyc = 0;
             for (is = is1; is <= is2; is++)
	         {os  = is*nsp;
		  in1 = sides[os];
                  ixc += rx[in1];
                  iyc += ry[in1];
		  uc  += f1[in1];
		  for (j = j0; j < nd; j++)
		      {ft     = f[j];
		       vc[j] += ft[in1];};};

             norm = 1.0/(is2 - is1 + 1);
	     uc  *= norm;
	     for (j = j0; j < nd; j++)
	         {ft     = f[j];
		  vc[j] *= norm;};
	     ixc *= norm;
	     iyc *= norm;};

	 for (is = is1; is <= is2; is++)
	     {os  = is*nsp;
	      in1 = sides[os];
	      in2 = sides[os+1];

	      u1 = f1[in1];
	      u2 = f1[in2];
	      for (j = j0; j < nd; j++)
		  {ft    = f[j];
		   v1[j] = ft[in1];
		   v2[j] = ft[in2];};

	      ix1 = rx[in1];
	      iy1 = ry[in1];
	      ix2 = rx[in2];
	      iy2 = ry[in2];

              if (sweep == 0)
		 PG_scan_triangle(dev, z_buffer, indx, nd, sweep,
				  ixmn, ixmx, iymn, iymx,
				  ix1, iy1, u1, v1,
				  ixc, iyc, uc, vc,
				  ix2, iy2, u2, v2,
				  amn, amx, extr,
				  color, mesh, type);
	      else
		 PG_scan_triangle(dev, z_buffer, indx, nd, sweep,
				  ixmn, ixmx, iymn, iymx,
				  iy1, ix1, u1, v1,
				  iyc, ixc, uc, vc,
				  iy2, ix2, u2, v2,
				  amn, amx, extr,
				  color, mesh, type);};};

    SFREE(v1);
    SFREE(v2);
    SFREE(vc);

    return;}

/*--------------------------------------------------------------------------*/

/*                             API LEVEL ROUTINES                           */

/*--------------------------------------------------------------------------*/
 
/* PG_SURFACE_PLOT - main surface plot control routine */

#ifdef PCC

void PG_surface_plot(dev, data, va_alist)
   PG_device *dev;
   PG_graph *data;
   va_dcl

#endif

#ifdef ANSI

void PG_surface_plot(PG_device *dev, PG_graph *data, ...)

#endif

   {

    if (strcmp(data->f->category, PM_LR_S) == 0)
       _PG_surface_core(dev, data, (byte *)data->f->domain->max_index,
			PM_zone_node_lr_2d, PG_scan_convert_lr);

    else if (strcmp(data->f->category, PM_AC_S) == 0)
       _PG_surface_core(dev, data, data->f->domain->topology,
			PM_zone_node_ac_2d, PG_scan_convert_ac);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_SURFACE - draw a surface plot given a known mesh type, a
 *                 - suitable scanning function, and a scalar array
 *                 - rotate to view angle, plot with hidden line
 *                 - algorithm, and shade if requested
 */

void PG_draw_surface(dev, a1, a2, extr, x, y, nn, xmn, xmx, ymn, ymx,
		     theta, phi, chi, width, color, style, type, name,
		     mesh_type, cnnct, alist)
   PG_device *dev;
   REAL *a1, *a2, *extr, *x, *y;
   int nn;
   double xmn, xmx, ymn, ymx;
   double theta, phi, chi, width;
   int color, style, type;
   char *name, *mesh_type;
   byte *cnnct;
   pcons *alist;
   {int kmax, lmax;
    int *maxes;
    PM_set *domain, *range;
    PG_graph *g;
    pcons *alst;

/* build the domain set */
    if (strcmp(mesh_type, PM_LR_S) == 0)
       {maxes = (int *) cnnct;
	kmax  = maxes[0];
	lmax  = maxes[1];
	domain = PM_make_set("{x,y}", SC_REAL_S, FALSE, 2, kmax, lmax, 2, x, y);}

    else if (strcmp(mesh_type, PM_AC_S) == 0)
       domain = PM_make_ac_set("{x,y}", SC_REAL_S, FALSE,
			       (PM_mesh_topology *) cnnct, 2, x, y);

/* build the range set */
    if (a2 == NULL)
       range = PM_make_set("{a}", SC_REAL_S, FALSE, 1, nn, 1, a1);
    else
       range = PM_make_set("{u,v}", SC_REAL_S, FALSE, 1, nn, 2, a1, a2);

/* add the attributes */
    alst = SC_copy_alist(alist);
    SC_ADD_VALUE_ALIST(alst, REAL, SC_REAL_P_S, "THETA", theta);
    SC_ADD_VALUE_ALIST(alst, REAL, SC_REAL_P_S, "PHI", phi);
    SC_ADD_VALUE_ALIST(alst, REAL, SC_REAL_P_S, "CHI", chi);
    
    SC_ADD_VALUE_ALIST(alst, int, SC_INTEGER_P_S, "LINE-COLOR", color);
    SC_ADD_VALUE_ALIST(alst, int, SC_INTEGER_P_S, "LINE-STYLE", style);
    SC_ADD_VALUE_ALIST(alst, REAL, SC_REAL_P_S, "LINE-WIDTH", width);

    g = PG_make_graph_from_sets(name, domain, range, N_CENT,
				SC_PCONS_P_S, alst, 'A', NULL);

    g->rendering = type;

    PG_surface_plot(dev, g);

    PG_rl_graph(g, FALSE, FALSE);

    return;}

/*--------------------------------------------------------------------------*/

/*                            FORTRAN API ROUTINES                          */

/*--------------------------------------------------------------------------*/

/* PGPLSF - low level surface plot routine */

FIXNUM F77_ID(pgplsf_, pgplsf, PGPLSF)(devid, px, py, pz, pn,
                                     pxn, pxx, pyn, pyx, pzn, pzx,
                                     pkx, plx, pth, pph, pch, ptyp,
                                     pcol, pwid, psty, pnc, label)
   FIXNUM *devid;
   REAL *px, *py, *pz;
   FIXNUM *pn;
   REAL *pxn, *pxx, *pyn, *pyx, *pzn, *pzx;
   FIXNUM *pkx, *plx;
   REAL *pth, *pph, *pch;
   FIXNUM *ptyp, *pcol;
   REAL *pwid;
   FIXNUM *psty, *pnc;
   F77_string label;
   {PG_device *dev;
    char llabel[MAXLINE];
    int maxes[2];
    REAL ext[4];

    maxes[0] = (int) *pkx;
    maxes[1] = (int) *plx;

    ext[0] = *pzn;
    ext[1] = *pzx;
    ext[2] = *pzn;
    ext[3] = *pzx; 

    SC_FORTRAN_STR_C(llabel, label, *pnc);

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_draw_surface(dev, pz, pz, ext, px, py, *pn,
		    *pxn, *pxx, *pyn, *pyx,
		    *pth, *pph, *pch, *pwid, (int) *pcol, (int) *psty,
		    (int) *ptyp, llabel, PM_LR_S, maxes, NULL);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

