/*
 * GSREND.C - high level rendering control for PGS
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "pgs.h"

/* Rendering Procedure:
 *
 *  1) Render a linked list of mappings with a single rendering serially
 *     (uses PG_draw_graph)
 *
 *        PG_make_device                                (allocate the device)
 *        PG_open_device                        (open the device for drawing)
 *        PG_setup_picture                          (setup viewport, WC, etc)
 *        loop over mappings
 *           render individual mapping           (control pixels in viewport)
 *        PG_finish_picture         (dress with axes, labels, etc, and flush)
 *
 *  2) Render a linked list of mappings with a individual renderings serially
 *     (uses PG_draw_picture)
 *
 *        PG_make_device                                (allocate the device)
 *        PG_open_device                        (open the device for drawing)
 *        PG_setup_picture                          (setup viewport, WC, etc)
 *           (set device flag to supress setup and finish in subsequent
 *            calls to PG_draw_graph)
 *        loop over mappings
 *           PG_draw_graph                       (control pixels in viewport)
 *        PG_finish_picture         (dress with axes, labels, etc, and flush)
 *           (reset device flag supressing setup and finish)
 *
 *  3) Render a linked list of mappings with a single rendering in parallel
 *     (uses PG_draw_graph)
 *
 *        PG_make_device                                (allocate the device)
 *           (select a node to open the requested drawing device
 *            all nodes open an IMAGE device and the selected
 *            node opens the requested device)
 *        PG_open_device                        (open the device for drawing)
 *           (all nodes open IMAGE device and selected node
 *            also opens requested device)
 *        PG_setup_picture                          (setup viewport, WC, etc)
 *           (call to PG_parallel_setup to communicate common
 *            viewport, WC, and extrema values in order to limit each
 *            node to settting only those pixels of the final
 *            picture for which it has data)
 *        loop over mappings
 *           render individual mapping to IMAGE device
 *        PG_finish_picture         (dress with axes, labels, etc, and flush)
 *           (after usual finish_picture all IMAGE devices
 *            transmit their raster to the selected node
 *            (_PG_transmit_images)
 *            which assembles them into a single raster image
 *            (_PG_display_image)
 *            and displays the raster image to the requested device
 *            with the rendering decorations of the selected type
 *            (PG_render_parallel)
 *
 *  4) Render a linked list of mappings with individual renderings in parallel
 *     (uses PG_draw_picture)
 *
 *     analogous to the relationship between (2) and (1) with special
 *     attention paid to the setup supression and parallel rendering issues
 *
 */



#define PG_DPMT_SETUP    105

int
 PG_hl_clear_mode     = CLEAR_SCREEN,
 PG_parallel_graphics = FALSE,
 PG_parallel_simulate = FALSE,
 _PG_restore_viewport = TRUE;

void
 SC_DECLARE((*PG_picture_hook), (PG_device *dev, PG_graph *data,
				 PG_picture_desc *pd)) = NULL;

/*--------------------------------------------------------------------------*/

/*                         PARALLEL GRAPHICS SUPPORT                        */

/*--------------------------------------------------------------------------*/

/* PG_GET_NUMBER_PROCESSORS - handle the number of processors in the
 *                          - presence of PG_parallel_graphics
 */

int PG_get_number_processors()
   {int np;

    if (PG_parallel_graphics)
       np = PC_get_number_processors();

    else
       np = 1;

    return(np);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_PROCESSOR_NUMBER - handle the processor number in the
 *                         - presence of PG_parallel_graphics
 */

int PG_get_processor_number()
   {int ip;

    if (PG_parallel_graphics)
       ip = PC_get_processor_number();

    else
       ip = 0;

    return(ip);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_PARALLEL_SETUP - reset the picture for parallel graphics */

static void PG_parallel_setup(dev, pd)
   PG_device *dev;
   PG_picture_desc *pd;
   {int i, j, ia, ib, ip, ne, nf, np, nr;
    int ndf, ndr, render, dp, dfl;
    int *map;
    REAL xmn, xmx, ymn, ymx;
    REAL vxmn, vxmx, vymn, vymx;
    REAL dx, dy, dvx, dvy, rx, ry;
    REAL *re, *dc, *adc, *pdc;
    PG_par_rend_info *pri;
    PROCESS *pp;
    static int sp[] = {PC_MATCH_NODE, 1, 0,
		       PC_MATCH_TAG, PG_DPMT_SETUP,
		       PC_BLOCK_STATE, FALSE,
		       PC_BUFFER_SIZE, 0,
		       0};

    np = PG_get_number_processors();
    if (np > 1)
       {ip = PG_get_processor_number();

	pri    = dev->pri;
	pp     = pri->pp;
	render = pri->render;
	dp     = pri->ip;
	map    = pri->map;

/* get the data to be shared */
	nr  = dev->range_n_extrema;
	dfl = (nr > 0);

	pri->have_data = dfl;

/* GOTCHA: arbitrarily limiting to 3 dimensional range elements */
	ndr = 6;
	ndf = 10;

	ne = ndf + ndr;
	dc = FMAKE_N(REAL, ne, "PG_PARALLEL_SETUP:dc");

	if (dfl)
	   {PG_get_viewport_WC(dev, dc, dc+1, dc+2, dc+3);
	    PG_get_viewport_NDC(dev, dc+4, dc+5, dc+6, dc+7);

	    re  = dev->range_extrema;
	    pdc = dc + ndf;
	    for (i = 0; i < nr; i++)
	        *pdc++ = re[i];}

	else
	   dc[0] = -HUGE;

	sp[6] = FALSE;
	sp[8] = ne*sizeof(REAL);

/* post the receives */
	if (ip == dp)
	   {adc = FMAKE_N(REAL, np*ne, "PG_PARALLEL_SETUP:adc");

	    pdc = adc + dp*ne;
	    for (i = 0; i < ne; i++)
	        pdc[i] = dc[i];

	    for (i = 0; i < np; i++)
	        {if (i != dp)
		    {sp[2] = i;
		     PC_in(adc+i*ne, SC_REAL_S, ne, pp, sp);};};}

/* post the sends */
	else
	   {sp[2] = dp;
	    PC_out(dc, SC_REAL_S, ne, pp, sp);};

/* wait for the data to move */
	PC_wait(pp);

/* find the global extrema */
	if (ip == dp)
	   {for (i = 0; i < np; i++)
	        {pdc    = adc + i*ne;
		 map[i] = (pdc[0] != -HUGE);};

/* if the node doing the final output has no data get data from
 * a node that has some
 */
	    if (map[dp] == FALSE)
	       {for (i = 0; i < np; i++)
		    {if (map[i] == TRUE)
		        {pdc = adc + i*ne;
			 for (i = 0; i < ne; i++)
			     dc[i] = pdc[i];};};};
	       
/* find global limits by checking everybody who has data */
	    for (i = 0; i < np; i++)
	        {if ((map[i] == TRUE) && (i != dp))
		    {pdc = adc + i*ne;

		     nf = ne >> 1;
		     for (j = 0; j < nf; j++)
		         {ia = 2*j;
			  ib = ia + 1;

			  dc[ia] = min(dc[ia], pdc[ia]);
			  dc[ib] = max(dc[ib], pdc[ib]);};};};

	    SFREE(adc);

	    dc[8] = pri->dd->window_width;
	    dc[9] = pri->dd->window_height;

/* broadcast the extrema back out */
	    for (i = 0; i < np; i++)
	        {if (i != dp)
		    {sp[2] = i;
		     PC_out(dc, SC_REAL_S, ne, pp, sp);};};}

	else
	   {sp[2] = dp;
	    PC_in(dc, SC_REAL_S, ne, pp, sp);};
	
	PC_wait(pp);

/* compute what this node's piece of the global picture is */
	if (dfl)
	   {PG_get_viewport_WC(dev, &xmn, &xmx, &ymn, &ymx);
	    PG_get_viewport_NDC(dev, &vxmn, &vxmx, &vymn, &vymx);

/* re-open the device to get the correct pixel width and height */
	    dev->open_screen(dev, 0.0, 0.0, dc[8], dc[9]);

	    dx  = dc[1] - dc[0];
	    dy  = dc[3] - dc[2];
	    dvx = dc[5] - dc[4];
	    dvy = dc[7] - dc[6];

	    rx = dvx/dx;
	    ry = dvy/dy;

/* reset the device */
	    vxmn += (xmn - dc[0])*rx;
	    vxmx -= (dc[1] - xmx)*rx;
	    if ((render == PLOT_CONTOUR) ||
		(render == PLOT_FILL_POLY) ||
		(render == PLOT_IMAGE) ||
		(render == PLOT_WIRE_MESH) ||
		(render == PLOT_SURFACE) ||
		(render == PLOT_VECTOR) ||
		(render == PLOT_MESH))
	       {vymn += (ymn - dc[2])*ry;
		vymx -= (dc[3] - ymx)*ry;

		PG_set_window(dev, xmn, xmx, ymn, ymx);
		PG_set_viewport(dev, vxmn, vxmx, vymn, vymx);}

	    else
	       {PG_set_window(dev, xmn, xmx, dc[2], dc[3]);
		PG_set_viewport(dev, vxmn, vxmx, vymn, vymx);};

	    pdc = dc + ndf;
	    for (i = 0; i < nr; i++)
	        re[i] = *pdc++;};

	SFREE(dc);};

    return;}

/*--------------------------------------------------------------------------*/

/*                       HIGHEST LEVEL RENDERING CONTROL                    */

/*--------------------------------------------------------------------------*/

/* PG_FIND_EXTREMA - find the extrema of the data and the requeste
 *                 - plotting limits and return it all
 *                 -    STDOFF an additional stand off for the data extrema
 *                 -    DPEX   domain plotting extrema
 *                 -    RPEX   range plotting extrema
 *                 -    DDEX   domain data extrema
 *                 -    RDEX   range data extrema
 *                 -    NDE    dimension of domain elements
 *                 -    NRE    dimension of range elements
 */

void PG_find_extrema(data, stdoff, pdpex, prpex, pnde, pddex, pnre, prdex)
   PG_graph *data;
   REAL stdoff, **pdpex, **prpex;
   int *pnde;
   REAL **pddex;
   int *pnre;
   REAL **prdex;
   {int i, ia, ib, n, ne, nde, nre, datafl;
    REAL dmn, dmx, rmn, rmx, rm, sc, mn, mx;
    REAL *ddex, *ddet, *rdex, *rdet, *lmt;
    PM_set *domain, *range;
    PM_mapping *f;
    PG_graph *g;
    pcons *alst;
    static REAL HC = 0.0;

    if (HC == 0.0)
       HC = HUGE_REAL*HUGE_REAL*HUGE_REAL;

    datafl = PG_render_data_type(data);

/* allocate space for the data extrema */
    if (datafl)
       {domain = data->f->domain;
	range  = data->f->range;
	nde    = (domain == NULL) ? 0 : domain->dimension_elem;
	nre    = (range == NULL) ? 0 : range->dimension_elem;}
    else
       {nde = 2;
	nre = 1;};

    ddex = FMAKE_N(REAL, 2*nde, "PG_FIND_EXTREMA:ddex");
    for (i = 0; i < nde; i++)
        {ddex[2*i]   = HUGE_REAL;
	 ddex[2*i+1] = -HUGE_REAL;};

    rdex = FMAKE_N(REAL, 2*nre, "PG_FIND_EXTREMA:rdex");
    for (i = 0; i < nre; i++)
        {rdex[2*i]   = HUGE_REAL;
	 rdex[2*i+1] = -HUGE_REAL;};

/* fill in the data extrema info */
    if (datafl)
       {ddet = FMAKE_N(REAL, 2*nde, "PG_FIND_EXTREMA:ddet");
	rdet = FMAKE_N(REAL, 2*nre, "PG_FIND_EXTREMA:rdet");

	for (g = data; g != NULL; g = g->next)
	    {for (f = g->f; f != NULL; f = f->next)
	         {domain = f->domain;
		  if (domain != NULL)
		     {nde = domain->dimension_elem;
		      PM_array_real(domain->element_type, domain->extrema, 2*nde, ddet);

		      alst = NULL;
		      if (domain->info_type != NULL)
			{if (strcmp(domain->info_type, SC_PCONS_P_S) == 0)
			    alst = (pcons *) domain->info;};

		      lmt = NULL;
		      SC_assoc_info(alst, "LIMITS", &lmt, NULL);
		      if (lmt != NULL)
			 {n = 2*nre;
			  for (i = 0; i < n; i++)
			      ddex[i] = lmt[i];}

		      else
			 {for (i = 0; i < nde; i++)
			      {ia = 2*i;
			       ib = ia + 1;
			       mn = ddex[ia];
			       mx = ddex[ib];
			       if ((mn > HC) || (mx > HC) ||
				   (mn < -HC) || (mx < -HC))
				  break;
			       ddex[ia] = min(mn, ddet[ia]);
			       ddex[ib] = max(mx, ddet[ib]);};};

/* weed out mappings presenting overflow problems */
		      if (i < nde)
			 continue;};

		  range = f->range;
		  if (range != NULL)
		     {nre = range->dimension_elem;
		      PM_array_real(range->element_type, range->extrema, 2*nre, rdet);

		      alst = NULL;
		      if (range->info_type != NULL)
			{if (strcmp(range->info_type, SC_PCONS_P_S) == 0)
			    alst = (pcons *) range->info;};

		      lmt = NULL;
		      SC_assoc_info(alst, "LIMITS", &lmt, NULL);
		      if (lmt != NULL)
			 {n = 2*nre;
			  for (i = 0; i < n; i++)
			      rdex[i] = lmt[i];}

		      else
			 {for (i = 0; i < nre; i++)
			      {ia = 2*i;
			       ib = ia + 1;
			       mn = rdex[ia];
			       mx = rdex[ib];
			       if ((mn > HC) || (mx > HC) ||
				   (mn < -HC) || (mx < -HC))
				  break;

			       rdex[ia] = min(mn, rdet[ia]);
			       rdex[ib] = max(mx, rdet[ib]);};};

/* weed out mappings presenting overflow problems */
		      if (i < nre)
			 continue;};};};

	SFREE(ddet);
	SFREE(rdet);

	if (pdpex != NULL)
	   *pdpex = PM_get_limits(domain);
	if (prpex != NULL)
	   *prpex = PM_get_limits(range);}

    else
       {PG_image_picture_info(data, NULL,
			      &ddex[0], &ddex[1], &ddex[2], &ddex[3],
			      &rdex[0], &rdex[1]);
	if (pdpex != NULL)
	   *pdpex = NULL;
	if (prpex != NULL)
	   *prpex = NULL;};

/* regularize the domain data extrema */
    for (i = 0; i < nde; i++)
        {ia = 2*i;
	 ib = ia + 1;
	 mn = ddex[ia];
	 mx = ddex[ib];
	 if (mn > mx)
	    {ddex[ia] = 0.0;
	     ddex[ib] = 1.0;};};

/* regularize the range data extrema */
    for (i = 0; i < nre; i++)
        {ia = 2*i;
	 ib = ia + 1;
	 mn = rdex[ia];
	 mx = rdex[ib];
	 if (mn > mx)
	    {rdex[ia] = 0.0;
	     rdex[ib] = 1.0;};};

/* add the standoff (vector plots use this) */
    if (stdoff != 0.0)
       {ne = min(nde, nre);
	for (i = 0; i < ne; i++)
	    {ia = 2*i;
	     ib = ia + 1;
	     dmn = ddex[ia];
	     dmx = ddex[ib];
	     rmn = rdex[ia];
	     rmx = rdex[ib];

	     rm = max(ABS(rmx), ABS(rmn));
	     sc = (rm == 0.0) ? 0.0 : stdoff*(dmx - dmn)/rm;

	     ddex[ia] += (min(0.0, sc*rmn));
	     ddex[ib] += (max(0.0, sc*rmx));};};

/* return the data extrema info */
    if (pnde != NULL)
       *pnde  = nde;
    if (pnre != NULL)
       *pnre  = nre;

    if (pddex != NULL)
       *pddex = ddex;
    else
       SFREE(ddex);

    if (prdex != NULL)
       *prdex = rdex;
    else
       SFREE(rdex);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_EST_RENDERING - estimate a rendering mode based on the
 *                  - given graph
 */

int PG_est_rendering(data)
   PG_graph *data;
   {int domain_dim, range_dim;
    int rend;

    domain_dim = data->f->domain->dimension_elem;
    range_dim  = data->f->range->dimension_elem;

    rend = PLOT_DEFAULT;

    switch (domain_dim)
       {case 1 :
	     rend = PLOT_CURVE;
	     break;

        case 2 :
	     switch (range_dim)
	        {default :
		 case 1  :
		      rend = PLOT_CONTOUR;
                      break;
		 case 2  :
		      rend = PLOT_VECTOR;
		      break;};
	     break;

        case 3 :
	     rend = PLOT_MESH;
	     break;};

    if ((data->info != NULL) &&
	(strcmp(data->info_type, SC_PCONS_P_S) == 0))
       {int *pr;

	SC_assoc_info((pcons *) data->info,
		      "RENDERING-TYPE", &pr,
		      NULL);
	rend = (pr != NULL) ? *pr : rend;};

    return(rend);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_RENDERING_PROPERTIES - return the rendering properties from the
 *                             - specified graph
 */

PG_picture_desc *PG_get_rendering_properties(dev, data)
   PG_device *dev;
   PG_graph *data;
   {int datafl;
    int *pax, *paxf, *paxt, *plb, *plg;
    int *pmsh, *ppty, *ppl;
    REAL *pt, *pp, *pc, *vwprt;
    PG_picture_desc *pd;
    PG_par_rend_info *pri;
    pcons *alst;

    datafl = PG_render_data_type(data);

    pd = FMAKE(PG_picture_desc, "PG_GET_RENDERING_PROPERTIES:pd");

    alst  = NULL;
    if (data->info_type != NULL)
       {if (strcmp(data->info_type, SC_PCONS_P_S) == 0)
	   alst = (pcons *) data->info;};

    vwprt = NULL;
    pax   = NULL;
    paxf  = NULL;
    paxt  = NULL;
    plb   = NULL;
    plg   = NULL;
    pmsh  = NULL;
    ppty  = NULL;
    ppl   = NULL;
    pt    = NULL;
    pp    = NULL;
    pc    = NULL;

    SC_assoc_info(alst,
		  "AXIS-TYPE",    &paxt,
		  "DRAW-AXIS",    &paxf,
		  "DRAW-LABEL",   &plb,
		  "DRAW-LEGEND",  &plg,
		  "DRAW-MESH",    &pmsh,
		  "DRAW-PALETTE", &ppl,
		  "PLOT-TYPE",    &ppty,
		  "THETA",        &pt,
		  "PHI",          &pp,
		  "CHI",          &pc,
		  "VIEW-PORT",    &vwprt,
		  NULL);

    pd->data       = data;
    pd->alist      = alst;
    pd->viewport   = vwprt;
    pd->rendering  = data->rendering;
    pd->pl_type    = (ppty == NULL) ? _PG_plot_type : *ppty;
    pd->mesh_fl    = (pmsh == NULL) ? data->mesh : *pmsh;
    pd->label_fl   = (plb == NULL) ? TRUE : *plb;
    pd->label      = (datafl) ? data->f->name : ((PG_image *) data->f)->label;
    pd->axis_fl    = (pax == NULL) ? TRUE : *pax;
    pd->ax_type    = (paxt == NULL) ? CARTESIAN : *paxt;
    pd->curve_fl   = FALSE;
    pd->palette_fl = (ppl == NULL) ? TRUE : *ppl;
    pd->palette    = dev->current_palette;
    pd->legend_fl  = (plg == NULL)  ? TRUE : *plg;
    pd->n_levels   = 0;
    pd->levels     = NULL;
    pd->theta      = (pt == NULL) ? HUGE : *pt;
    pd->phi        = (pp == NULL) ? HUGE : *pp;
    pd->chi        = (pc == NULL) ? HUGE : *pc;

    pd->vxmin = 0.0;
    pd->vxmax = 0.0;
    pd->vymin = 0.0;
    pd->vymax = 0.0;

    pd->xmin = 0.0;
    pd->xmax = 0.0;
    pd->ymin = 0.0;
    pd->ymax = 0.0;

    pri = dev->pri;
    if (pri != NULL)
       {pri->render = data->rendering;
	if (pri->label == NULL)
	   pri->label = SC_strsavef(pd->label,
				    "PG_GET_RENDERING_PROPERTIES:label");};

    return(pd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_PICTURE - setup a window for the rendering implies by
 *                  - the specified graph DATA
 *                  - NOTE: no drawing of any kind is to be done here
 */

PG_picture_desc *PG_setup_picture(dev, data, save, clear, par)
   PG_device *dev;
   PG_graph *data;
   int save, clear, par;
   {int change, rend;
    REAL xmn, xmx, ymn, ymx;
    REAL vxmn, vxmx, vymn, vymx;
    PG_picture_desc *pd;

    PG_make_device_current(dev);

    change = !dev->supress_setup;

    if (par && change)
       dev->autorange = TRUE;

    if (save && change)
       {PG_get_viewport(dev, &vxmn, &vxmx, &vymn, &vymx);
	PG_get_viewport_WC(dev, &xmn, &xmx, &ymn, &ymx);};

    if (clear && change)
       {switch (PG_hl_clear_mode)
	   {case CLEAR_SCREEN :
	         PG_clear_window(dev);
		 break;
	    case CLEAR_VIEWPORT :
	         PG_clear_viewport(dev);
		 break;
	    case CLEAR_FRAME :
	         PG_clear_frame(dev);
		 break;};};

    pd = NULL;

    rend = data->rendering;
    if ((rend < PLOT_CURVE) || (PLOT_DEFAULT <= rend))
       rend = PG_est_rendering(data);

    switch (rend)
       {case PLOT_CONTOUR :
	     pd = PG_setup_picture_contour(dev, data, save, clear);
	     break;

        case PLOT_CURVE:
	     pd = PG_setup_picture_curve(dev, data, save, clear);
	     break;

        case PLOT_FILL_POLY:
	     pd = PG_setup_picture_fill_poly(dev, data, save, clear);
	     break;

        case PLOT_IMAGE:
	     pd = PG_setup_picture_image(dev, data, save, clear);
	     break;

        case PLOT_MESH:
	     pd = PG_setup_picture_mesh(dev, data, save, clear);
	     break;

        case PLOT_WIRE_MESH:
        case PLOT_SURFACE:
	     pd = PG_setup_picture_surface(dev, data, save, clear);
	     break;

        case PLOT_VECTOR :
	     pd = PG_setup_picture_vector(dev, data, save, clear);
	     break;};

    if (save && change && (pd != NULL))
       {pd->vxmin = vxmn;
	pd->vxmax = vxmx;
	pd->vymin = vymn;
	pd->vymax = vymx;
	pd->xmin = xmn;
	pd->xmax = xmx;
	pd->ymin = ymn;
	pd->ymax = ymx;};

    if (par && change)
       PG_parallel_setup(dev, pd);

    return(pd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_PICTURE - render a list of mappings all in one picture
 *                 - Argument:
 *                 -   PTYP   rendering type
 *                 -   BNDP   on/off flag to show patch boundaries
 *                 -   CBND   line color for patch boundaries
 *                 -   SBND   line style for patch boundaries
 *                 -   WBND   line width for patch boundaries
 *                 -   MSHP   on/off flag to show mesh lines
 *                 -   CMSH   line color for mesh lines
 *                 -   SMSH   line style for mesh lines
 *                 -   WMSH   line width for mesh lines
 */

void PG_draw_picture(dev, f, ptyp, bndp, cbnd, sbnd, wbnd,
		     mshp, cmsh, smsh, wmsh)
   PG_device *dev;
   PM_mapping *f;
   int ptyp, bndp, cbnd, sbnd;
   double wbnd;
   int mshp, cmsh, smsh;
   double wmsh;
   {int fls, axs;
    pcons *inf, *dinf;
    PM_mapping *pf, *nxt;
    PG_graph *g, data;
    PG_picture_desc *pd;
    PG_par_rend_info *pri;
    PG_device *dd;
    static int id = 'A';

/* save device attributes */
    PG_get_fill_bound(dev, fls);
    axs = _PG_axis_on;

/* set device attributes */
    inf = SC_copy_alist((pcons *) f->map);
    SC_CHANGE_VALUE_ALIST(inf, int, "int *", "DRAW-MESH", mshp);

    g = &data;
    memset(g, 0, sizeof(PG_graph));
    data.info_type  = SC_PCONS_P_S;
    data.info       = (byte *) inf;
    data.f          = f;
    data.rendering  = ptyp;
    data.mesh       = (mshp != 0);
    data.identifier = id;

    pd = PG_setup_picture(dev, g, TRUE, TRUE, TRUE);
    dev->supress_setup = TRUE;

/* loop over all mappings */
    for (pf = f; pf != NULL; pf = nxt)
        {nxt = pf->next;

	 pf->next = NULL;

	 g = PG_make_graph_from_mapping(pf, SC_PCONS_P_S, NULL, id, NULL);

	 g->mesh      = (mshp != 0);
	 g->rendering = ptyp;

/* setup the graph's info list */
	 inf = SC_copy_alist((pcons *) pf->map);
	 inf = SC_append_alist((pcons *) g->info, inf);

/* handle the patch boundary attribute */
	 if (bndp)
	    {dinf = (pcons *) pf->domain->info;

	     SC_CHANGE_VALUE_ALIST(dinf, REAL, "REAL *",
				   "DOMAIN-BORDER-WIDTH", wbnd);
	     SC_CHANGE_VALUE_ALIST(dinf, int, "int *",
				   "DOMAIN-BORDER-COLOR", cbnd);
	     SC_CHANGE_VALUE_ALIST(dinf, int, "int *",
				   "DOMAIN-BORDER-STYLE", sbnd);

	     pf->domain->info = (byte *) dinf;};

/* handle the mesh plotting attribute */
	 SC_CHANGE_VALUE_ALIST(inf, int, "int *", "DRAW-MESH", mshp);

/* attach the info list to the graph */
	 g->info = (byte *) inf;

	 if (pf->range == NULL)
	    PG_mesh_plot(dev, g);
	 else
	    PG_draw_graph(dev, g);

/* carefully release the graph - but not the mapping */
	 SC_free_alist(inf, 3);

	 SFREE(g);

	 if (id > 'Z')
	    id = 'A';

	 _PG_axis_on = FALSE;

	 pf->next = nxt;};
	     
    pri = dev->pri;
    if (pri != NULL)
       {pri->render = ptyp;
	dd = pri->dd;
	if (dd != NULL)
	   dd->pri->render = ptyp;};

    if (PG_picture_hook != NULL)
       (*PG_picture_hook)(dev, &data, pd);

#if 1
/* now call the frame done */
    if (dev->finished)
       {PG_finish_plot(dev);}
    else
       {PG_update_vs(dev);};
#else
    PG_finish_picture(dev, &data, pd);
#endif

    dev->supress_setup = FALSE;

/* restore device attributes */
    PG_set_fill_bound(dev, fls);
    _PG_axis_on  = axs;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_FINISH_PICTURE - draw the requested adornments and
 *                   - finish the plot frame if requested
 */

void PG_finish_picture(dev, data, pd)
   PG_device *dev;
   PG_graph *data;
   PG_picture_desc *pd;
   {int i, id, nd, nde, nl, nm, datafl;
    int hvf, rendering,  axty, axfl;
    char *label, t[MAXLINE];
    REAL xa, xb, ya, yb, za, zb;
    REAL xmn, xmx, ymn, ymx;
    REAL vxmn, vxmx, vymn, vymx;
    REAL dxt, xw, yw, dx, dy;
    REAL theta, phi, chi;
    REAL ext[6], *extr, *dextr, *lv;
    PM_set *domain;
    PG_palette *cpl;

    PG_set_clipping(dev, FALSE);

    extr  = dev->range_extrema;
    axfl  = pd->axis_fl;
    axty  = pd->ax_type;
    theta = pd->theta;
    phi   = pd->phi;
    chi   = pd->chi;
    cpl   = pd->palette;

    datafl = PG_render_data_type(data);

/* draw the palette if needed */
    if (pd->palette_fl)
       {if (axty == CARTESIAN_3D)
	   {rendering = data->rendering;
	    nd        = (datafl) ? data->f->range->dimension_elem : 1;
	    if (rendering == PLOT_SURFACE)
	       {if (nd == 1)
		   PG_draw_palette(dev, 0.808, 0.175, 0.808, 0.825,
				   extr[0], extr[1], _PG_palette_width);
	        else
		   PG_draw_palette(dev, 0.808, 0.175, 0.808, 0.825,
				   extr[2], extr[3], _PG_palette_width);};}
	else
	   {hvf = _PG_palette_orientation;
	    if (hvf == VERTICAL)
	       PG_draw_palette(dev, 0.808, 0.175, 0.808, 0.825,
			       extr[0], extr[1], _PG_palette_width);

	    else if (hvf == HORIZONTAL)
	       {nde = dev->range_n_extrema >> 1;
		if ((cpl->pal_dims == NULL) || (nde < 2))
		   PG_draw_palette(dev, 0.175, 0.10, 0.85, 0.10,
				   extr[0], extr[1], _PG_palette_width);
		else
		   PG_draw_2dpalette(dev, 0.325, 0.06, 0.70, 0.175,
				     extr[0], extr[1], extr[2], extr[3],
				     _PG_palette_width);};};};

    PG_set_palette(dev, "standard");

    PG_set_axis_attributes(dev, AXIS_LINECOLOR, dev->line_color,
                                AXIS_LABELCOLOR, dev->text_color, 0);

/* draw the axes if requested */
    if (axfl)
       {if (axty == CARTESIAN_3D)
	   {domain = (datafl) ? data->f->domain : NULL;
	    switch (domain->dimension)
	       {case 2 :
		     if (datafl)
		        {dextr  = PM_array_real(domain->element_type,
						domain->extrema, 4, ext);
			 xa = dextr[0];
			 xb = dextr[1];
			 ya = dextr[2];
			 yb = dextr[3];
			 za = extr[0];
			 zb = extr[1];}
		     else
		        PG_image_picture_info(data, NULL,
					      &xa, &xb, &ya, &yb,
					      &za, &zb);
		     nm = TRUE;
		     break;

		case 3 :
		     if (datafl)
		        {dextr  = PM_array_real(domain->element_type,
						domain->extrema, 6, ext);
			 xa = dextr[0];
			 xb = dextr[1];
			 ya = dextr[2];
			 yb = dextr[3];
			 za = dextr[4];
			 zb = dextr[5];}
		     else
		        PG_image_picture_info(data, NULL,
					      &xa, &xb, &ya, &yb,
					      &za, &zb);
		     nm = FALSE;
		     break;};

	    PG_axis_3d(dev, NULL, NULL, NULL, 0,
		       theta, phi, chi, 
		       xa, xb, ya, yb, za, zb,
		       nm);}
	else
	   PG_axis(dev, axty);};

/* print a label for the picture if requested */
    if (pd->label_fl)
       {label = pd->label;
	if (axty == CARTESIAN_3D)
	   {sprintf(t, "View Angle (%.2f, %.2f, %.2f)",
		    RAD_DEG*theta, RAD_DEG*phi, RAD_DEG*chi);
	    PG_center_label(dev, 0.1, t);};

        PG_get_viewport(dev, &xmn, &xmx, &ymn, &ymx);
        PG_center_label(dev, 0.5*(1.0 + ymx), label);};

/* draw a reference mesh if requested */
    if (pd->mesh_fl)
       {PG_get_viewport(dev, &vxmn, &vxmx, &vymn, &vymx);
	PG_ref_mesh(dev, data, 2, vxmn, vxmx, vymn, vymx, pd->viewport);};
       
/* print the contour level values (legend) */
    if (pd->legend_fl)
       {id = (datafl) ? pd->data->identifier : 'A';
	lv = pd->levels;
	nl = pd->n_levels;

	PG_get_viewport_WC(dev, &xmn, &dx, &ymn, &dy);
	dx -= xmn;
	dy -= ymn;

	PG_get_text_ext(dev, "XX - ", &dxt, &yw);
	xw = 1.05*dx + xmn;
	yw = dy + ymn;
	PG_write_WC(dev, xw, yw, "%s", "Levels: ");
  
	PG_set_char_line(dev, 100);
	for (i = 0; i < nl; i++)
	    {yw  -= 0.05*dy;
	     PG_write_WC(dev, xw, yw, "%c", id+i);
	     PG_write_WC(dev, xw+dxt, yw, "%.2e ", lv[i]);};
             
	PG_set_char_line(dev, 80);

	SFREE(pd->levels);};
  
/* print the curve labels if requested */
    if (pd->curve_fl)
       _PG_print_graph_labels(dev, data);

/* do the final frame update */
    if (dev->finished)
       {PG_finish_plot(dev);}
    else
       {PG_update_vs(dev);};

/* restore saved state */
    if (cpl != NULL)
       dev->current_palette = cpl;

    if (_PG_restore_viewport)
       {vxmn = pd->vxmin;
	vxmx = pd->vxmax;
	vymn = pd->vymin;
	vymx = pd->vymax;

	if ((vxmn != vxmx) && (vymn != vymx))
	   PG_set_viewport(dev, vxmn, vxmx, vymn, vymx);

	xmn = pd->xmin;
	xmx = pd->xmax;
	ymn = pd->ymin;
	ymx = pd->ymax;

	if ((xmn != xmx) && (ymn != ymx))
	   PG_set_window(dev, xmn, xmx, ymn, ymx);};

    SFREE(pd);

/* release the device */
    PG_release_current_device(dev);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DISPATCH_2D - dispatch 2d graphs to the requested rendering
 *                 - routine
 *                 - user supplied hooks should be inserted into this
 *                 - process at some point
 */

static void _PG_dispatch_2d(dev, data, range_dim)
   PG_device *dev;
   PG_graph *data;
   int range_dim;
   {int rendering;

/* get the rendering method for this graph */
    rendering = data->rendering;

    switch (range_dim)
       {case 1 :
	     switch (rendering)
	        {default           :
		 case PLOT_DEFAULT :
		 case PLOT_CONTOUR :
		      PG_contour_plot(dev, data, 2);
		      break;
	         case PLOT_IMAGE :
		      PG_image_plot(dev, data, 2);
		      break;
	         case PLOT_FILL_POLY :
		      PG_poly_fill_plot(dev, data, 2);
		      break;
	         case PLOT_WIRE_MESH :
		 case PLOT_SURFACE   :
		      PG_surface_plot(dev, data, 2);
		      break;
		 case PLOT_MESH :
		      PG_mesh_plot(dev, data, 2);
		      break;};
	     break;

        case 2  :
	     switch (rendering)
	        {case PLOT_DEFAULT :
		 case PLOT_VECTOR  :
                      PG_vector_plot(dev, data, 2);
                      break;
	         case PLOT_FILL_POLY :
		      PG_poly_fill_plot(dev, data, 2);
		      break;
		 case PLOT_MESH :
		      PG_mesh_plot(dev, data, 2);
		      break;};
	     break;

        default :
             break;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DISPATCH_3D - dispatch 3d graphs to the requested rendering
 *                 - routine
 *                 - user supplied hooks should be inserted into this
 *                 - process at some point
 */

static void _PG_dispatch_3d(dev, data, range_dim)
   PG_device *dev;
   PG_graph *data;
   int range_dim;
   {

    PG_mesh_plot(dev, data, 3);

#if 0
/* get the rendering method for this graph */
    rendering = data->rendering;

    switch (range_dim)
       {case 1 :
	     switch (rendering)
	        {default           :
		 case PLOT_DEFAULT :
		 case PLOT_CONTOUR :
		      PG_contour_plot(dev, data, 3);
		      break;
	         case PLOT_FILL_POLY :
		      PG_poly_fill_plot(dev, data, 3);
		      break;
	         case PLOT_WIRE_MESH :
		 case PLOT_SURFACE   :
		      PG_surface_plot(dev, data, 3);
		      break;
		 case PLOT_MESH :
		      PG_mesh_plot(dev, data, 3);
		      break;};
	     break;

        case 2  :
	     switch (rendering)
	        {case PLOT_DEFAULT :
		 case PLOT_VECTOR  :
                      PG_vector_plot(dev, data, 3);
                      break;
		 case PLOT_MESH :
		      PG_mesh_plot(dev, data, 3);
		      break;};
	     break;

        default :
             break;};
#endif

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_DRAW_GRAPH - main graph plotting control routine */

void PG_draw_graph(dev, data)
   PG_device *dev;
   PG_graph *data;
   {int domain_dim, range_dim;

    if (_PG_text_format == NULL)
       _PG_text_format = SC_strsavef("%.0f",
                          "char*:PG_DRAW_GRAPH :format");

    if ((data->info != NULL) &&
	(strcmp(data->info_type, SC_PCONS_P_S) == 0))
       {int *pr;

	SC_assoc_info((pcons *) data->info,
		      "RENDERING-TYPE", &pr,
		      NULL);
	data->rendering = (pr != NULL) ? *pr : data->rendering;};

    domain_dim = data->f->domain->dimension_elem;
    range_dim  = data->f->range->dimension_elem;

    switch (domain_dim)
       {case 1 :
	     switch (range_dim)
	        {case 1 :
		      PG_curve_plot(dev, data);
		      break;
                 default :
		      break;};
	     break;

        case 2 :
	     _PG_dispatch_2d(dev, data, range_dim);
	     break;

        case 3 :
	     _PG_dispatch_3d(dev, data, range_dim);
	     break;

        default :
	     break;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
