/*
 * PDRDWR.C - new read/write routines for PDBLib
 *
 * Source Version: 9.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pdb.h"

#define DONE         1
#define LEAF         2
#define LEAF_ITEM    3
#define LEAF_RET     4
#define LEAF_INDIR   5
#define INDIRECT     6
#define INDIR_ITEM   7
#define INDIR_RET    8
#define BLOCK        9
#define BLOCK_ITEM  10
#define BLOCK_RET   11
#define SKIP_TO     12
#define SKIP_RET    13

#define SAVE_S(s, t)                                                         \
    {rw_str_stack[rw_str_ptr++] = s;                                         \
     s = SC_strsavef(t, "char*:SAVE_S:t");}

#define RESTORE_S(s)                                                         \
    {SFREE(s);                                                               \
     s = rw_str_stack[--rw_str_ptr];}

#define SAVE_I(val)                                                          \
    (rw_lval_stack[rw_lval_ptr++].diskaddr = (long) val)

#define RESTORE_I(val)                                                       \
    (val = rw_lval_stack[--rw_lval_ptr].diskaddr)

#define SAVE_P(val)                                                          \
    (rw_lval_stack[rw_lval_ptr++].memaddr = (char *) val)

#define RESTORE_P(type, val)                                                 \
    (val = (type *) rw_lval_stack[--rw_lval_ptr].memaddr)

#define SET_CONT(ret)                                                        \
   {rw_call_stack[rw_call_ptr++] = ret;                                      \
    dst = _PD_indirection(litype) ? INDIRECT : LEAF;                         \
    continue;}

#define SET_CONT_RD(ret, branch)                                             \
   {rw_call_stack[rw_call_ptr++] = ret;                                      \
    dst = branch;                                                            \
    continue;}

#define GO_CONT                                                              \
   {dst = rw_call_stack[--rw_call_ptr];                                      \
    continue;}

#define GO(lbl)                                                              \
    {dst = lbl;                                                              \
     continue;}

#define INIT_STACKS(_t, _d)                                                  \
    {long _a, _f;                                                            \
     if (rw_list_t == 0)                                                     \
        {rw_list_t = _t;                                                     \
	 rw_list_d = _d;                                                     \
         SC_mem_stats(&_a, &_f, NULL, NULL);                                 \
	 DYN_STK(long, rw_call_stack, rw_call_ptr, rw_call_ptr_x);           \
	 DYN_STK(char *, rw_str_stack, rw_str_ptr, rw_str_ptr_x);            \
	 DYN_STK(SC_address, rw_lval_stack, rw_lval_ptr, rw_lval_ptr_x);     \
         SC_mem_stats_set(_a, _f);};                                         \
     rw_call_ptr = 0L;                                                       \
     rw_lval_ptr = 0L;                                                       \
     rw_str_ptr  = 0L;                                                       \
     litype      = NULL;                                                     \
     lotype      = NULL;                                                     \
     rw_call_stack[rw_call_ptr++] = DONE;}

#define DYN_STK(_t, _s, _p, _px)                                             \
    {if (_p > _px - rw_list_t)                                               \
        {_px += rw_list_d;                                                   \
	 if (_s == NULL)                                                     \
	    {_s = FMAKE_N(_t, _px, "DYN_STK:s");}                            \
	 else                                                                \
	    REMAKE_N(_s, _t, _px);};}

#define START                                                                \
    while (TRUE)                                                             \
       {DYN_STK(long, rw_call_stack, rw_call_ptr, rw_call_ptr_x);            \
	DYN_STK(char *, rw_str_stack, rw_str_ptr, rw_str_ptr_x);             \
        DYN_STK(SC_address, rw_lval_stack, rw_lval_ptr, rw_lval_ptr_x);      \
	switch (dst) {

#define FINISH(f, tag)                                                       \
    default  :                                                               \
         sprintf(bf, "UNDECIDABLE CASE - %s", f);                            \
         PD_error(bf, tag);};}

static long
 *rw_call_stack = NULL,
 rw_call_ptr = 0L,
 rw_call_ptr_x = 0L,
 rw_lval_ptr = 0L,
 rw_lval_ptr_x = 0L,
 rw_str_ptr = 0L,
 rw_str_ptr_x = 0L,
 rw_list_t = 0L,
 rw_list_d = 0L;

static SC_address
 *rw_lval_stack = NULL;

static char
 **rw_str_stack = NULL;

/*--------------------------------------------------------------------------*/

/*                          AUXILLIARY ROUTINES                             */

/*--------------------------------------------------------------------------*/

/* _PD_EFFECTIVE_ADDR - THREADSAFE
 *                    - given a disk address computed on the assumption
 *                    - of one contiguous block and a list of symblocks
 *                    - compute and return the actual disk address
 *                    - also return the number of items remaining in the
 *                    - block after the effective address
 */

static void _PD_effective_addr(paddr, pnitems, bpi, sp)
   long *paddr, *pnitems;
   int bpi;
   symblock *sp;
   {int i;
    long nb, nt, ad, addr;

    addr = *paddr;
    i    = 0;
    ad   = sp[i].diskaddr;
    nt   = addr - ad;
    while (TRUE)
        {nb  = sp[i].number*bpi;
         nt -= nb;

         if ((nb <= 0L) || (nt < 0L))
            break;

         i++;
         ad   = sp[i].diskaddr;
         addr = ad + nt;};

    *paddr   = addr;
    *pnitems = (ad + nb - addr)/bpi;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_NUMBER_REFD - THREADSAFE
 *                 - compute the number of items pointed to by VR
 *                 - return the number of items if successful
 *                 - return -1 if SCORE did NOT allocate the block
 *                 - return -2 if the type is unknown
 */

long _PD_number_refd(vr, type, tab)
   byte *vr;
   char *type;
   HASHTAB *tab;
   {long bytepitem, nitems;

    if (vr == NULL)
       return(0L);

    if ((nitems = SC_arrlen(vr)) <= 0)
       return(-1L);

    if ((bytepitem = _PD_lookup_size(type, tab)) == -1)
       return(-2L);

    nitems /= bytepitem;

    return(nitems);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_INDIRECTION - THREADSAFE
 *                 - return TRUE if the TYPE is an indirect type
 *                 - i.e. has a "*" as the last non-blank character
 */

int _PD_indirection(s)
   char *s;
   {char *t;

    for (t = s + strlen(s); t > s; t--)
        if (*t == '*')
           return(TRUE);

    return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PD_DEREFERENCE - THREADSAFE
 *                - starting at the end of the string work backwards to
 *                - the first non-blank character and if it is a '*'
 *                - insert '\0' in its place
 *                - return a pointer to the beginning of the string
 */

char *PD_dereference(s)
   char *s;
   {char *t;

    for (t = s + strlen(s); t > s; t--)
        if (*t == '*')
           break;

/* check for whitespace to remove - eg. "char *" -> "char" */
    for (t-- ; t > s; t--)
        if (strchr(" \t", *t) == NULL)
           {*(++t) = '\0';
            break;};

    return(s);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PD_RESET_PTR_LIST - THREADSAFE
 *                   - reset the pointer lists for the given file
 *                   - so that indirect connectivity bookkeeping
 *                   - is re-initialized this will give the application
 *                   - some control over this process
 */

int PD_reset_ptr_list(file)
   PDBfile *file;
   {char **ptr_rd_list, **ptr_wr_list;
    long *addr_rd_list, *addr_wr_list, *reta_rd_list;

    ptr_rd_list  = file->ptr_rd_list;
    ptr_wr_list  = file->ptr_wr_list;
    addr_rd_list = file->addr_rd_list;
    addr_wr_list = file->addr_wr_list;
    reta_rd_list = file->reta_rd_list;

    if (addr_rd_list != NULL)
       {SFREE(addr_rd_list);};
    if (addr_wr_list != NULL)
       {SFREE(addr_wr_list);};

    if (ptr_rd_list != NULL)
       {SFREE(ptr_rd_list);};
    if (ptr_wr_list != NULL)
       {SFREE(ptr_wr_list);};

    if (reta_rd_list != NULL)
       {SFREE(reta_rd_list);};

    file->ptr_rd_list  = NULL;
    file->ptr_wr_list  = NULL;
    file->addr_rd_list = NULL;
    file->addr_wr_list = NULL;
    file->reta_rd_list = NULL;

    file->max_rd_indx  = 0L;
    file->max_wr_indx  = 0L;
    file->rd_indx      = 0L;
    file->wr_indx      = 0L;

    return(TRUE);}

/*--------------------------------------------------------------------------*/

/*                           HYPER INDEX SUPPORT                            */

/*--------------------------------------------------------------------------*/

/* _PD_INIT_DIMIND - THREADSAFE
 *                 - fill a dimind struct given the stride and an
 *                 - ASCII index expression
 */

static void _PD_init_dimind(pi, offset, stride, expr)
   dimind *pi;
   long offset;
   long stride;
   char *expr;
   {char s[MAXLINE], *token, *t;
    long start, stop, step;

    if (expr != NULL)
       strcpy(s, expr);
    else
       s[0] = '\0';

    token = SC_strtok(s, " \t:", t);
    if (token == NULL)
       start = 0L;
    else
       start = atol(token);

    token = SC_strtok(NULL, " \t:", t);
    if (token == NULL)
       stop = start;
    else
       stop = atol(token);

    token = SC_strtok(NULL, " \t:", t);
    if (token == NULL)
       step = 1L;
    else
       step = atol(token);

    pi->stride = stride;
    pi->start  = start - offset;
    pi->stop   = stop - offset;
    pi->step   = step;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_COMPUTE_HYPER_STRIDES - THREADSAFE
 *                           - initialize and return an array of dimension
 *                           - indexes representing the hyper strides
 *                           - from the given hyper index expression
 */

static dimind  *_PD_compute_hyper_strides(file, ind, dims, pnd)
   PDBfile *file;
   char *ind;
   dimdes *dims;
   int *pnd;
   {int i, nd;
    long maxs;
    dimdes *pd;
    dimind *pi;

    if (dims == NULL)
       {pi = FMAKE(dimind, "_PD_COMPUTE_HYPER_STRIDES:pi");

        _PD_init_dimind(pi, (long) file->default_offset, 0L,
                        SC_firsttok(ind, ",()[]\n\r"));

        *pnd = 1;
        return(pi);};

/* count the number of dimensions and allocate some temporaries */
    for (nd = 0, pd = dims; pd != NULL; pd = pd->next, nd++);
    pi = FMAKE_N(dimind, nd, "_PD_COMPUTE_HYPER_STRIDES:pi");

/* pre-compute the strides, offsets, and so on for the hyper-space walk */
    if (file->major_order == COLUMN_MAJOR_ORDER)
       {maxs = 1L;
        for (i = nd - 1, pd = dims; i >= 0; i--)
            {_PD_init_dimind(&pi[i], pd->index_min, maxs,
                             SC_firsttok(ind, ",()[]\n\r"));
             if (pd != NULL)
                {maxs *= pd->number;
                 pd    = pd->next;};};}

    else if (file->major_order == ROW_MAJOR_ORDER)
       {for (maxs = 1L, pd = dims->next; pd != NULL; pd = pd->next)
            maxs *= pd->number;

        for (i = 0, pd = dims; i < nd; i++)
            {_PD_init_dimind(&pi[i], pd->index_min, maxs,
                             SC_firsttok(ind, ",()[]\n\r"));
             if (pd->next != NULL)
                {pd    = pd->next;
                 maxs /= pd->number;};};};

    *pnd = nd;

    return(pi);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_HYPER_NUMBER - THREADSAFE
 *                  - return the number of elements implied by a hyper
 *                  - index expression
 *                  -
 *                  -    <expr>  := <spec> | <expr>, <spec>
 *                  -    <spec>  := <start> |
 *                  -               <start>:<stop> |
 *                  -               <start>:<stop>:<step>
 *                  -    <start> := starting integer index value
 *                  -    <stop>  := ending integer index value
 *                  -    <step>  := integer index step value
 *                  -
 */

long _PD_hyper_number(file, indxpr, numb, dims, poff)
   PDBfile *file;
   char *indxpr;
   long numb;
   dimdes *dims;
   long *poff;
   {int nd;
    long i, maxs, sum, offs;
    char s[MAXLINE];
    dimind *pi;

    strcpy(s, indxpr);
    pi = _PD_compute_hyper_strides(file, s, dims, &nd);

    offs = 0L;
    sum  = 1L;
    for (i = 0; i < nd; i++)
        {maxs  = (pi[i].stop - pi[i].start + pi[i].step)/pi[i].step;
         offs += pi[i].start*pi[i].stride;
         sum  *= maxs;};

    SFREE(pi);

    if (poff != NULL)
       *poff = offs;

    return(sum);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PD_HYPER_NUMBER - THREADSAFE
 *                 - return the number of elements implied by a hyper
 *                 - index expression
 */

long PD_hyper_number(file, name, ep)
   PDBfile *file;
   char *name;
   syment *ep;
   {char s[MAXLINE];
    int c;

/* if name is of the form a[...] strip off the name part
 * by design _PD_hyper_number can't handle anything but the index part
 */
    strcpy(s, name);
    c = s[0];
    if (strchr("0123456789-.", c) == NULL)
       SC_firsttok(s, "([");

    return(_PD_hyper_number(file, s,
			    PD_entry_number(ep),
			    PD_entry_dimensions(ep), NULL));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_HYPER_DIMS - return a reduced dimension descriptor for a hyper slab
 *                - preserve size and shape, eliminate degenerate dimensions
 */

dimdes *_PD_hyper_dims(file, name, dims)
   PDBfile *file;
   char *name;
   dimdes *dims;
   {dimdes *ndims, *prev, *next, *dp;
    char s[MAXLINE], *t;
    long n, start, stop, step;

    ndims = NULL;

    strcpy(s, name);
    t = SC_lasttok(s, "[]()");
    strcpy(s, t);

    for (dp = dims; dp != NULL; dp = dims->next)
       {t = SC_firsttok(s, " ,()[]\n\r");
	if (t == NULL)
	   break;

	n = _PD_parse_index_expr(t, dp, &start, &stop, &step);
	if (n <= 1)
	   continue;

	next = _PD_mk_dimensions(file->default_offset, n);

        if (ndims == NULL)
           ndims = next;
        else
	   {prev->next = next;
	    SC_mark(next, 1);};

	prev = next;};

    return(ndims);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_EXPAND_HYPER_NAME - Return hyper-indexed name with any partial index
 *                       - expressions fully expanded
 */

char *_PD_expand_hyper_name(file, name)
   PDBfile *file;
   char *name;
   {int err;
    syment *ep;
    dimdes *dims, *pd;
    long start, stop, step;
    char s[MAXLINE], lname[MAXLINE], index[MAXLINE], expr[MAXLINE], *t;

    strcpy(s, name);
    if (s[strlen(s) - 1] != ']')
       return(SC_strsavef(s, "char*:_PD_EXPAND_HYPER_NAME:s"));
    if (s[strlen(s) - 2] == '[')
       return(NULL);

    strcpy(lname, s);
    t    = strrchr(lname, '[');
    t[0] = '\0';
    ep   = _PD_effective_ep(file, lname, TRUE, NULL);
    dims = PD_entry_dimensions(ep);
    if (dims == NULL)
       {_PD_rl_syment_d(ep);
	return(SC_strsavef(s, "char*:_PD_EXPAND_HYPER_NAME:s"));};

    t = SC_lasttok(s, "[]()");
    strcpy(s, t);

    err = FALSE;
    strcpy(index, "[");
    for (pd = dims; pd != NULL; pd = pd->next)
        {t = SC_firsttok(s, " ,()[]\n\r");
	 if (t == NULL)
	    {err = TRUE;
	     break;};

	 if (_PD_parse_index_expr(t, pd, &start, &stop, &step) == 0)
	    {err = TRUE;
	     break;};

         if (start == stop)
            sprintf(expr, "%ld,", start);

         else if (step <= 1L)
            sprintf(expr, "%ld:%ld,", start, stop);

         else
            sprintf(expr, "%ld:%ld:%ld,", start, stop, step);
            
         strcat(index, expr);};
    
    _PD_rl_syment_d(ep);

    if (err)
       return(NULL);
    else
       {index[strlen(index) - 1] = ']';
        sprintf(s, "%s%s", lname, index);
        return(SC_strsavef(s, "char*:_PD_EXPAND_HYPER_NAME:s"));};}

/*--------------------------------------------------------------------------*/

/*                            WRITE ROUTINES                                */

/*--------------------------------------------------------------------------*/

/* _PD_WR_LEAF_MEMBERS - write the direct leaf data */

static void _PD_wr_leaf_members(file, intype, outtype, nitems, vr)
   PDBfile *file;
   char *intype, *outtype;
   long nitems;
   byte *vr;
   {long bytepitem, in_offs, out_offs;
    int ret;
    size_t nb;
    char *svr, *vbuf, *buf;
    FILE *fp;
    defstr *dpf;

    fp = file->stream;

    dpf       = _PD_lookup_type(outtype, file->chart);
    bytepitem = dpf->size;
    if (bytepitem == -1)
       PD_error("CAN'T GET NUMBER OF BYTES - _PD_WR_LEAF_MEMBERS", PD_WRITE);

/* dispatch all other writes */
    if ((dpf->convert > 0) || (strcmp(intype, outtype) != 0))
       {buf = (char *) SC_alloc(nitems, bytepitem, "_PD_WR_LEAF_MEMBERS:buffer");
        if (buf == NULL)
           PD_error("CAN'T ALLOCATE MEMORY - _PD_WR_LEAF_MEMBERS", PD_WRITE);

	if ((_SC_zero_space == 1) || (_SC_zero_space == 2))
	   memset(buf, 0, nitems*bytepitem);

        vbuf     = buf;
        svr      = vr;
        in_offs  = 0L;
        out_offs = 0L;
        PD_convert(&vbuf, &svr, intype, outtype, nitems,
                   file->host_std, file->std, file->host_std,
                   &in_offs, &out_offs,
                   file->host_chart, file->chart, 0, PD_WRITE);
        nb  = io_write(buf, (size_t) bytepitem, (size_t) nitems, fp);
        ret = (nb == nitems) ? TRUE : FALSE;
        SFREE(buf);}
    else
       {nb  = io_write(vr, (size_t) bytepitem, (size_t) nitems, fp);
        ret = (nb == nitems) ? TRUE : FALSE;};

    if (!ret)
       PD_error("BYTE WRITE FAILED - _PD_WR_LEAF_MEMBERS", PD_WRITE);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_WR_IND_ITAGS - handle the memory of pointers and write the itags
 *                  - correctly
 */

static int _PD_wr_ind_itags(file, vr, nitems, type)
   PDBfile *file;
   byte *vr;
   long nitems;
   char *type;
   {FILE *fp;
    long *pl, *addr_rd_list, rd_indx;
    long i, addr, *addr_wr_list, wr_indx, max_wr_indx;
    char **ppl, **ptr_wr_list, **ptr_rd_list;

    fp = file->stream;

    ptr_wr_list  = file->ptr_wr_list;
    addr_wr_list = file->addr_wr_list;
    max_wr_indx  = file->max_wr_indx;
    wr_indx      = file->wr_indx;

    ptr_rd_list  = file->ptr_rd_list;
    addr_rd_list = file->addr_rd_list;
    rd_indx      = file->rd_indx;

/* the first time out allocate the lists of pointers and disk addresses */
    if (ptr_wr_list == NULL)
       {file->wr_indx      = wr_indx      = 0L;
        file->max_wr_indx  = max_wr_indx  = 50L;
        file->ptr_wr_list  = ptr_wr_list  = FMAKE_N(char *, max_wr_indx,
                                                    "_PD_WR_IND_ITAGS:ptr-list");
        file->addr_wr_list = addr_wr_list = FMAKE_N(long, max_wr_indx,
                                                    "_PD_WR_IND_ITAGS:addr-list");};

/* remove from lists of pointers and disk addresses previously read */
    if (ptr_rd_list != NULL)
       {addr = io_tell(fp);
        pl   = addr_rd_list;
        for (i = 0L; (i < rd_indx) && (addr != *pl); i++, pl++);
        if (i < rd_indx)
           {rd_indx--;
            file->rd_indx   = rd_indx;
            addr_rd_list[i] = addr_rd_list[rd_indx];
            ptr_rd_list[i]  = ptr_rd_list[rd_indx];};};

/* search the pointer list to see if this pointer
 * has been previously written
 */
    ppl = ptr_wr_list;
    for (i = 0L; (i < wr_indx) && (vr != *ppl); i++, ppl++);

/* if the loop did not max out, this has been written before */
    if (i != wr_indx)
       {_PD_wr_itag(file, nitems, type, addr_wr_list[i], FALSE);
        return(FALSE);};

/* otherwise the pointer has not been seen before */
    *ppl = vr;

/* save the address of the header because
 * we don't know the data address yet
 */
    addr = io_tell(fp);
    if (addr == -1L)
       PD_error("FAILED TO FIND ADDRESS - _PD_WR_IND_ITAGS", PD_WRITE);

/* write some info for the read
 * TRUE if this is the first time for the pointer
 */
    _PD_wr_itag(file, nitems, type, addr, TRUE);

/* in recursive situations the way in which the call stack unwinds can cause
 * the incrementing of wr_indx to happen too late and result in overwriting
 * addr_wr_list.  The fix is to assume that the write will succeed, do the
 * increment, check the allocations, and do the write
 */
    addr_wr_list[wr_indx++] = addr;
    (file->wr_indx)++;

/* if the lists are full expand them */
    if (max_wr_indx <= wr_indx)
       {max_wr_indx += 50L;
        REMAKE_N(ptr_wr_list, char *, max_wr_indx);
        REMAKE_N(addr_wr_list, long, max_wr_indx);
        file->ptr_wr_list  = ptr_wr_list;
        file->addr_wr_list = addr_wr_list;
        file->max_wr_indx  = max_wr_indx;};

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_WRITE_HYPER_VIF - write a part of a hyper-surface to memory
 *                     - ADDR is the starting address
 *                     - STOP is the upper bound on the address
 *                     - STEP is the increment of the address for each entry
 *                     - FBYT is the number of bytes in the file for each
 *                     - item to be written
 *                     - HBYT is the number of bytes in memory for each
 *                     - item to be written
 */

static char *_PD_write_hyper_vif(file, in, intype, outtype, sp, hbyt, fbyt,
				 addr, stop, step)
   PDBfile *file;
   char *in, *intype, *outtype;
   symblock *sp;
   int hbyt, fbyt;
   long addr, stop, step;
   {long nitems;

/* items logically contiguous */
    if (step == fbyt)
       {nitems = (stop - addr)/step + 1L;

	if (io_seek(file->stream, addr, SEEK_SET))
	   PD_error("FSEEK FAILED - _PD_WRITE_HYPER_VIF",
		    PD_WRITE);

	_PD_wr_syment(file, in, nitems, intype, outtype);}

/* items logically discontiguous */
    else
       {for (; addr <= stop; addr += step, in += hbyt)
	    {if (io_seek(file->stream, addr, SEEK_SET))
                PD_error("FSEEK FAILED - _PD_WRITE_HYPER_VIF",
                         PD_WRITE);
             _PD_wr_syment(file, in, 1L, intype, outtype);};};

    return(in);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_WRITE_HYPER_SPACE - write a part of a hyper-surface to disk
 *                       - ADDR is the starting address
 *                       - STOP is the upper bound on the address
 *                       - STEP is the increment of the address for each entry
 *                       - FBYT is the number of bytes in the file for each
 *                       - item to be written
 *                       - HBYT is the number of bytes in memory for each
 *                       - item to be written
 */

static char *_PD_write_hyper_space(file, in, intype, outtype, sp, hbyt, fbyt,
                                   addr, stop, step)
   PDBfile *file;
   char *in, *intype, *outtype;
   symblock *sp;
   int hbyt, fbyt;
   long addr, stop, step;
   {long nb, eaddr;

/* items logically contiguous */
    if (step == fbyt)
       {long nitems, niw;

        nitems = (stop - addr)/step + 1L;

/* get writes across blocks correct */
        while (nitems > 0)

/* adjust the address for the correct block */
           {eaddr = addr;
            _PD_effective_addr(&eaddr, &nb, fbyt, sp);
            if (io_seek(file->stream, eaddr, SEEK_SET))
               PD_error("FSEEK FAILED TO FIND ADDRESS - _PD_WRITE_HYPER_SPACE",
                        PD_WRITE);

/* NOTE: this subverts _PD_effective_addr in part, but because _PD_effective_ep
 * cannot be modified to build an effective syment for indirectly referenced data
 * which cannot be distinguished from an explicitly dimensioned array, this
 * is the best that can be done
 */
            if ((eaddr == 0) || (nb == 0))
               {eaddr = addr;
                nb    = nitems;};

            niw = min(nb, nitems);

            _PD_wr_syment(file, in, niw, intype, outtype);

            nitems -= niw;
            addr   += fbyt*niw;
            in     += hbyt*niw;};}

/* items logically discontiguous */
    else
       {for (; addr <= stop; addr += step, in += hbyt)
            {eaddr = addr;
             _PD_effective_addr(&eaddr, &nb, fbyt, sp);
             if (io_seek(file->stream, eaddr, SEEK_SET))
                PD_error("FSEEK FAILED - _PD_WRITE_HYPER_SPACE",
                         PD_WRITE);
             _PD_wr_syment(file, in, 1L, intype, outtype);};};

    return(in);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_WR_HYPER_INDEX - do the real work of parsing an index expression into
 *                    - compontents and recursively determine the disk
 *                    - locations to read into the buffer OUT
 *                    - OUT is to be filled in order from smallest index to
 *                    - largest
 *                    - the offset is specified by the starting address
 *                    - which is ADDR
 *                    - FBYT is the number of bytes in the file for each item
 *                    - to be read
 *                    - HBYT is the number of bytes in memory for each item
 *                    - to be read
 */

static char *_PD_wr_hyper_index(file, out, pi, intype, outtype, addr, 
				sp, hbyt, fbyt)
   PDBfile *file;
   char *out;
   dimind *pi;
   char *intype, *outtype;
   long addr;
   symblock *sp;
   int hbyt, fbyt;
   {long offset, stride, start, stop, step;

/* for each index specification compute the range and recurse */
    stride = fbyt*pi->stride;
    start  = stride*pi->start;
    stop   = stride*pi->stop;
    step   = stride*pi->step;

    stop  = addr + stop - start;
    start = addr;

/* at the bottom of the recursion do the actual operations */
    if (stride <= (long) fbyt)
       {if (file->virtual_internal == TRUE)
	   out = _PD_write_hyper_vif(file, out, intype, outtype, sp,
				     hbyt, fbyt,
				     start, stop, step);
	else
	   out = _PD_write_hyper_space(file, out, intype, outtype, sp,
				       hbyt, fbyt,
				       start, stop, step);}
    else
       for (offset = start; offset <= stop; offset += step)
           out = _PD_wr_hyper_index(file, out, pi + 1, intype, outtype, 
				    offset, sp, hbyt, fbyt);

    return(out);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_WR_SYMENT - write the NUMBER of elements of type, TYPE,
 *               - from memory pointed to by VAR
 *               - returns the number of items successfully written
 *               - this new version is written in a continuation passing
 *               - style so that PDB has control over the stack and
 *               - isn't blowing out the execution stack for long linked
 *               - lists
 */

long _PD_wr_syment(file, vr, nitems, intype, outtype)
   PDBfile *file;
   char *vr;
   long nitems;
   char *intype, *outtype;
   {int dst, size, indir;
    long i;
    defstr *dp;
    memdes *desc, *mem_lst;
    char bf[MAXLINE], *litype, *lotype, *svr, *ttype;

    file->flushed = FALSE;

    INIT_STACKS(100L, 1000L);

    dst = _PD_indirection(intype) ? INDIRECT : LEAF;

    if (dst == LEAF)
       {indir = FALSE;
	litype = intype;
	lotype = outtype;}
    else
       {indir = TRUE;
	SAVE_S(litype, intype);
	SAVE_S(lotype, outtype);};

/* some AIX compilers will erroneously take the default case if
 * this is terminated with a semicolon
 */
    START

    case LEAF :
         _PD_wr_leaf_members(file, litype, lotype, nitems, vr);

         dp = PD_inquire_host_type(file, litype);
         if (dp == NULL)
            PD_error("BAD TYPE - _PD_WR_SYMENT", PD_WRITE);

         mem_lst = dp->members;
         if (!dp->n_indirects || (mem_lst == NULL))
            GO_CONT;

         if (pdb_wr_hook != NULL)
            mem_lst = (*pdb_wr_hook)(file, vr, dp);

/* if type is a struct with pointered members write them out now 
 * for an array of structs write the indirects for each array element
 */
         size = dp->size;
         svr  = vr;
         i    = 0L;

    case LEAF_ITEM :
         if (i >= nitems)
            GO_CONT;

         desc = mem_lst;

    case LEAF_INDIR :
         if (desc == NULL)
            {i++;
             svr += size;
             GO(LEAF_ITEM);};

         PD_CAST_TYPE(ttype, desc, svr+desc->member_offs, svr,
		      PD_error, "BAD CAST - _PD_WR_SYMENT", PD_WRITE);

         SAVE_S(litype, ttype);

         if (!_PD_indirection(litype))
            {RESTORE_S(litype);
             desc = desc->next;
             GO(LEAF_INDIR);};

         SAVE_I(nitems);
         nitems = desc->number;

         SAVE_I(i);
         SAVE_I(size);
         SAVE_P(mem_lst);
         SAVE_P(desc);
         SAVE_P(svr);
         SAVE_P(vr);
         vr = svr + desc->member_offs;
         SET_CONT(LEAF_RET);

    case LEAF_RET :
         RESTORE_P(char, vr);
         RESTORE_P(char, svr);
         RESTORE_P(memdes, desc);
         RESTORE_P(memdes, mem_lst);
         RESTORE_I(size);
         RESTORE_I(i);
         RESTORE_I(nitems);
         RESTORE_S(litype);

         desc = desc->next;
         GO(LEAF_INDIR);

    case INDIRECT :
    
         if (vr == NULL)
            {_PD_wr_itag(file, 0L, litype, -1L, TRUE);
             GO_CONT;};

/* dereference a local copy of the type */
         SAVE_S(litype, litype);
         PD_dereference(litype);

/* write the data */
         i = 0L;

    case INDIR_ITEM :
         if (i >= nitems)
            {RESTORE_S(litype);
             GO_CONT;};

         SAVE_P(vr);
         vr = DEREF(vr);
         if (vr == NULL)
            {_PD_wr_itag(file, 0L, litype, -1L, FALSE);
             RESTORE_P(char, vr);
             i++;
             vr += sizeof(char *);
             GO(INDIR_ITEM);};

         SAVE_I(nitems);
         nitems = _PD_number_refd(vr, litype, file->host_chart);
         if (nitems == -1L)
            {sprintf(bf,
                     "CAN'T GET POINTER LENGTH ON %s - _PD_WR_SYMENT",
                     litype);
             PD_error(bf, PD_WRITE);};

         if (nitems == -2L)
            {sprintf(bf,
                     "UNKNOWN TYPE %s - _PD_WR_SYMENT",
                     litype);
             PD_error(bf, PD_WRITE);};

         if (!_PD_wr_ind_itags(file, vr, nitems, litype))
            {RESTORE_I(nitems);
             RESTORE_P(char, vr);
             i++;
             vr += sizeof(char *);
             GO(INDIR_ITEM);};

         SAVE_I(i);
         SAVE_S(lotype, litype);
         SET_CONT(INDIR_RET);

    case INDIR_RET :
         RESTORE_S(lotype);
         RESTORE_I(i);
         RESTORE_I(nitems);
         RESTORE_P(char, vr);

         i++;
         vr += sizeof(char *);

         GO(INDIR_ITEM);

    case DONE :
         if (indir)
	    {RESTORE_S(lotype);
	     RESTORE_S(litype);};

/* update the end of data mark */
         _PD_eod(file);

         return(nitems);

    FINISH("_PD_WR_SYMENT", PD_WRITE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_HYPER_WRITE - parse an index expression of the form
 *                 -
 *                 -    <expr>  := <spec> | <expr>, <spec>
 *                 -    <spec>  := <start> |
 *                 -               <start>:<stop> |
 *                 -               <start>:<stop>:<step>
 *                 -    <start> := starting integer index value
 *                 -    <stop>  := ending integer index value
 *                 -    <step>  := integer index step value
 *                 -
 *                 - and write the specified elements to the PDBfile
 *                 - from the array provided
 */

int _PD_hyper_write(file, name, ep, vr, intype)
   PDBfile *file;
   char *name;
   syment *ep;
   byte *vr;
   char *intype;
   {int nd, c, hbyt, fbyt;
    char s[MAXLINE], *expr;
    dimdes *dims;
    dimind *pi;

    dims = PD_entry_dimensions(ep);
    strcpy(s, name);
    c = s[strlen(s)-1];
    if (((c != ')') && (c != ']')) || (dims == NULL))
       {if (file->virtual_internal == TRUE)

/* GOTCHA: assuming that if we get here then we are rewriting an
 *         entire entry in a VIF (as opposed to something like an
 *         array element) and that it doesn't really need to be done
 *         We aren't fully set to handle this yet
 */
	   return(TRUE);

	else
	   return(_PD_wr_syment(file, vr, PD_entry_number(ep),
				intype, PD_entry_type(ep)));};

    if (_PD_indirection(PD_entry_type(ep)))
       PD_error("CAN'T HYPER INDEX INDIRECT TYPE - _PD_HYPER_WRITE", PD_WRITE);

    expr = SC_lasttok(s, "[]()");
    strcpy(s, expr);

    pi = _PD_compute_hyper_strides(file, s, dims, &nd);
    if (pi == NULL)
       PD_error("CAN'T FIND HYPER INDICES - _PD_HYPER_WRITE", PD_WRITE);

    fbyt = _PD_lookup_size(PD_entry_type(ep), file->chart);
    if (fbyt == -1)
       PD_error("CAN'T FIND NUMBER OF FILE BYTES - _PD_HYPER_WRITE",
                PD_WRITE);

    hbyt = _PD_lookup_size(intype, file->host_chart);
    if (hbyt == -1)
       PD_error("CAN'T FIND NUMBER OF HOST BYTES - _PD_HYPER_WRITE",
                PD_WRITE);

    _PD_wr_hyper_index(file, vr, pi, intype,
                       PD_entry_type(ep), PD_entry_address(ep),
                       PD_entry_blocks(ep), hbyt, fbyt);

    SFREE(pi);

    return(TRUE);}

/*--------------------------------------------------------------------------*/

/*                            READ ROUTINES                                 */

/*--------------------------------------------------------------------------*/

/* _PD_READ_HYPER_SPACE - read a part of a hyper-surface from a data array
 *                      - ADDR is the starting address
 *                      - STOP is the upper bound on the address
 *                      - STEP is the increment of the address for each entry
 *                      - FBYT is the number of bytes in the file for each
 *                      - item to be read
 *                      - HBYT is the number of bytes in memory for each
 *                      - item to be read
 *                      - EP is a scratch syment for temporary use
 *                      - return the number of item successfully read
 */

static int _PD_read_hyper_space(file, ep, out, intype, outtype, 
				sp, hbyt, fbyt, addr, stop, step)
   PDBfile *file;
   syment *ep;
   char *out, *intype, *outtype;
   symblock *sp;
   int hbyt, fbyt;
   long addr, stop, step;
   {long eaddr, nb, nitems;
    int nrd;

    PD_entry_type(ep)       = intype;
    PD_entry_dimensions(ep) = NULL;

    nrd = 0;
    if (addr >= 0)

/* items logically contiguous */
       {if (step == fbyt)
	   {long niw;

/* read across blocks */
	    nitems = (stop - addr)/step + 1L;
	    while (nitems > 0)
	       {eaddr = addr;

		_PD_effective_addr(&eaddr, &nb, fbyt, sp);

/* NOTE: this subverts _PD_effective_addr in part, but because _PD_effective_ep
 * cannot be modified to build an effective syment for indirectly referenced data
 * which cannot be distinguished from an explicitly dimensioned array, this
 * is the best that can be done
 */
		if ((eaddr == 0) || (nb == 0))
		   {eaddr = addr;
		    nb    = nitems;};

		niw = min(nitems, nb);

		PD_entry_address(ep) = eaddr;
		PD_entry_number(ep)  = niw;
		nrd += _PD_rd_syment(file, ep, outtype, out);

		nitems -= niw;
		addr   += fbyt*niw;
		out    += hbyt*niw;};}

/* items not logically contiguous */
        else
	   {PD_entry_number(ep) = 1L;
	    for (; addr <= stop; addr += step, out += hbyt)
	        {eaddr = addr;
		 _PD_effective_addr(&eaddr, &nb, fbyt, sp);
		 PD_entry_address(ep) = eaddr;
		 nrd += _PD_rd_syment(file, ep, outtype, out);};};}

/* we have a bitstream */
    else
       {defstr* dpf;

	dpf = _PD_lookup_type(intype, file->chart);
/* items logically contiguous */
	if (step == -dpf->size_bits)
	   {nitems = (stop - addr)/step + 1L;

/* NOTE: multi-block bitstreams are not supported */
	    PD_entry_number(ep)  = nitems;
	    PD_entry_address(ep) = addr;
	    nrd += _PD_rd_syment(file, ep, outtype, out);

	    out += hbyt*nitems;}

/* items not logically contiguous */
	else
	   {PD_entry_number(ep) = 1L;
	    for (; addr >= stop; addr += step, out += hbyt)
	        {PD_entry_address(ep) = addr;
		 nrd += _PD_rd_syment(file, ep, outtype, out);};};};

    PD_entry_type(ep) = NULL;

    return(nrd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_RD_HYPER_INDEX - do the real work of parsing an index expression into
 *                    - compontents and recursively determine the disk
 *                    - locations to read into the buffer OUT
 *                    - OUT is to be filled in order from smallest index to
 *                    - largest
 *                    - the offset is specified by the starting address
 *                    - which is ADDR
 *                    - FBYT is the number of bytes in the file for each item
 *                    - to be read
 *                    - HBYT is the number of bytes in memory for each item
 *                    - to be read
 *                    - return the number of item successfully read
 */

static int _PD_rd_hyper_index(file, ep, out, pi, intype, outtype, 
			      addr, sp, hbyt, fbyt)
   PDBfile *file;
   syment *ep;
   char *out;
   dimind *pi;
   char *intype, *outtype;
   long addr;
   symblock *sp;
   int hbyt, fbyt;
   {long offset, stride, start, stop, step;
    int nrd, nir;

/* for each index specification compute the range and recurse */
    stride = fbyt*pi->stride;
    start  = stride*pi->start;
    stop   = stride*pi->stop;
    step   = stride*pi->step;

    if (addr < 0)
       {defstr* dpf;
	dpf  = _PD_lookup_type(intype, file->chart);
	stop = addr - dpf->size_bits*((stop - start)/fbyt);
	step = -dpf->size_bits*(step/fbyt);}
    else
       stop = addr + (stop - start);

/* at the bottom of the recursion do the actual reads */
    nrd = 0;
    if (stride <= (long) fbyt)
       nrd += _PD_read_hyper_space(file, ep, out, intype, outtype, 
				   sp, hbyt, fbyt, addr, stop, step);

    else if (addr < 0)
       {for (offset = -addr; offset <= -stop; offset -= step)
            {nir = _PD_rd_hyper_index(file, ep, out, pi + 1, intype, outtype,
                                     -offset, sp, hbyt, fbyt);
	     nrd += nir;
	     out += nir*hbyt;};}

    else
       {for (offset = addr; offset <= stop; offset += step)
	    {nir = _PD_rd_hyper_index(file, ep, out, pi + 1, intype, outtype,
				      offset, sp, hbyt, fbyt);
	     nrd += nir;
	     out += nir*hbyt;};};

    return(nrd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_RD_LEAF_MEMBERS - read the leaves only for NITEMS of INTYPE
 *                     - from the PDBfile FILE
 *                     - into the location pointed to by VR as type OUTTYPE
 *                     - at this level it guaranteed that the type
 *                     - will not be a pointer
 *                     - return the number of items successfully read
 */

static void _PD_rd_leaf_members(file, vr, nitems, intype, outtype, boffs)
   PDBfile *file;
   char *vr;
   long nitems;
   char *intype, *outtype;
   int boffs;
   {FILE *fp;
    long bytepitemin, in_offs, out_offs, nir, nia;
    char *buf, *vbuf, *svr;
    defstr *dpf;

    fp = file->stream;

    dpf         = _PD_lookup_type(intype, file->chart);
    bytepitemin = dpf->size;
    if (bytepitemin == -1)
       PD_error("CAN'T FIND NUMBER OF BYTES - _PD_RD_LEAF_MEMBERS",
                PD_READ);

    if ((dpf->convert > 0) || (strcmp(intype, outtype) != 0))
       {if (dpf->size_bits)
	   nia = (((nitems*dpf->size_bits + boffs + SC_BITS_BYTE - 1)
	           /SC_BITS_BYTE) + bytepitemin - 1)/bytepitemin;
	else
	   nia = nitems;

	buf = (char *) SC_alloc(nia, bytepitemin,
                                "_PD_RD_LEAF_MEMBERS:buffer");
        if (buf == NULL)
           PD_error("CAN'T ALLOCATE MEMORY - _PD_RD_LEAF_MEMBERS", PD_READ);

	if ((_SC_zero_space == 1) || (_SC_zero_space == 2))
	   memset(buf, 0, nia*bytepitemin);

        nir = io_read(buf, (size_t) bytepitemin, (size_t) nia, fp);
        if (nir == nia)
           {vbuf     = buf;
            svr      = vr;
            in_offs  = 0L;
            out_offs = 0L;
            PD_convert(&svr, &vbuf, intype, outtype, nitems,
                       file->std, file->host_std, file->host_std,
                       &in_offs, &out_offs,
                       file->chart, file->host_chart, boffs, PD_READ);
            SFREE(buf);}
        else
           {SFREE(buf);
            PD_error("FILE READ FAILED - _PD_RD_LEAF_MEMBERS", PD_READ);};}
    else
       {nir = io_read(vr, (size_t) bytepitemin, (size_t) nitems, fp);
        if (nir != nitems)
           PD_error("DATA READ FAILED - _PD_RD_LEAF_MEMBERS", PD_READ);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_RD_IND_TAGS - read itags and setup the read of an indirection
 *                 - return TRUE iff there is indirect data to be read
 */

static int _PD_rd_ind_tags(file, vr, pi)
   PDBfile *file;
   char **vr;
   PD_itag *pi;
   {long i, bytepitem, nitems, *pad, addr, oaddr;
    long *addr_rd_list, *reta_rd_list, rd_indx, max_rd_indx;
    int flag;
    char *type, *pv, **ptr_rd_list, bf2[MAXLINE];
    FILE *fp;

    fp = file->stream;

    ptr_rd_list  = file->ptr_rd_list;
    addr_rd_list = file->addr_rd_list;
    reta_rd_list = file->reta_rd_list;
    max_rd_indx  = file->max_rd_indx;
    rd_indx      = file->rd_indx;

/* the first time out allocate the lists of pointers and disk addresses */
    if (ptr_rd_list == NULL)
       {file->rd_indx      = rd_indx      = 0L;
        file->max_rd_indx  = max_rd_indx  = 50L;
        file->ptr_rd_list  = ptr_rd_list  = FMAKE_N(char *, max_rd_indx,
                                                    "_PD_RD_IND_TAGS:ptr");
        file->addr_rd_list = addr_rd_list = FMAKE_N(long, max_rd_indx,
                                                    "_PD_RD_IND_TAGS:addr");
        file->reta_rd_list = reta_rd_list = FMAKE_N(long, max_rd_indx,
                                                    "_PD_RD_IND_TAGS:reta");};

    if (!_PD_rd_itag(file, pi))
       PD_error("BAD ITAG - _PD_RD_IND_TAGS", PD_READ);

    nitems = pi->nitems;
    type   = pi->type;
    addr   = pi->addr;
    flag   = pi->flag;

/* if it was a NULL pointer stop here */
    if ((addr == -1L) || (nitems == 0L))
       {*vr = NULL;
        return(-1);};

/* search the disk address list to see if this pointer
 * has been previously read
 */
    for (i = 0L, pad = addr_rd_list;
         (i < rd_indx) && (addr != *pad);
         i++, pad++);

/* if the loop didn't max out this has been read before
 * and by the way, get the reference count correct
 */
    if (i != rd_indx)
       {SC_mark(DEREF(vr) = ptr_rd_list[i], 1);
        if (flag)
           {if (io_seek(fp, reta_rd_list[i], SEEK_SET))
               PD_error("FAILED TO FIND RETURN - _PD_RD_IND_TAGS",
                         PD_READ);};
        return(-1);};

/* otherwise the pointer has not been seen before */
    bytepitem = _PD_lookup_size(type, file->host_chart);
    if (bytepitem == -1)
       PD_error("CAN'T FIND NUMBER OF BYTES - _PD_RD_IND_TAGS",
                PD_READ);

    DEREF(vr) = pv = (char *) SC_alloc(nitems, bytepitem,
                                       "_PD_RD_IND_TAGS:pv");

/* add one to the reference count */
    SC_mark(pv, 1);

    *pad = addr;
    ptr_rd_list[rd_indx++] = pv;
    file->rd_indx++;

/* if the lists are full expand them */
    if (max_rd_indx <= rd_indx)
       {max_rd_indx += 50L;
        REMAKE_N(ptr_rd_list, char *, max_rd_indx);
        REMAKE_N(addr_rd_list, long, max_rd_indx);
        REMAKE_N(reta_rd_list, long, max_rd_indx);
        file->ptr_rd_list  = ptr_rd_list;
        file->addr_rd_list = addr_rd_list;
        file->reta_rd_list = reta_rd_list;
        file->max_rd_indx  = max_rd_indx;};

/* if flag != 1 it was written somewhere else
 * GOTCHA: watch for new case of flag == 2 which means a discontiguous block
 *         Deal with this, if and when it arises.
 */
    if (flag != 1)
       {oaddr = io_tell(fp);
        if (oaddr == -1L)
           PD_error("CAN'T FIND CURRENT ADDRESS - _PD_RD_IND_TAGS",
                    PD_READ);

/* jump to the place where the original is described */
        if (io_seek(fp, addr, SEEK_SET))
           PD_error("FAILED TO FIND ADDRESS - _PD_RD_IND_TAGS",
                    PD_READ);

/* read the descriptor so that the file pointer is left on the actual data
 * NOTE: read into separate buffer so not to clobber "type" which is strtok'd
 *       into bf1
 */
        _PD_rfgets(bf2, MAXLINE, fp);

	pi->addr = oaddr;};

    return(i);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_INDEXED_READ_AS - read part of an entry from the PDB file
 *                     - into the location pointed to by VR
 *                     - IND contains one triplet of long ints per variable
 *                     - dimension specifying start, stop, and step for the
 *                     - index
 *                     - return the number of item successfully read
 */

int _PD_indexed_read_as(file, fullpath, type, vr, nd, ind, ep)
   PDBfile *file;
   char *fullpath, *type;
   byte *vr;
   int nd;
   long *ind;
   syment *ep;
   {int i, err;
    long start, stop, step;
    char expr[MAXLINE], index[MAXLINE], hname[MAXLINE];

    switch (setjmp(_PD_read_err))
       {case ABORT    : return(FALSE);
        case ERR_FREE : return(TRUE);
        default       : memset(PD_err, 0, MAXLINE);
                        break;};

    strcpy(index, "(");
    for (i = 0; i < nd; i++)
        {start = ind[0];
         stop  = ind[1];
         step  = ind[2];
         ind  += 3;
         if (start == stop)
            sprintf(expr, "%ld,", start);

         else if (step <= 1L)
            sprintf(expr, "%ld:%ld,", start, stop);
            
         else
            sprintf(expr, "%ld:%ld:%ld,", start, stop, step);
            
         strcat(index, expr);};

    if (strlen(index) > 1)
       {index[strlen(index)-1] = ')';
        sprintf(hname, "%s%s", fullpath, index);}

    else
       strcpy(hname, fullpath);

    _PD_rl_syment_d(ep);
    ep = _PD_effective_ep(file, hname, TRUE, fullpath);
    if (ep == NULL)
       PD_error("CAN'T FIND ENTRY - _PD_INDEXED_READ_AS", PD_READ);

    PD_entry_number(ep) = PD_hyper_number(file, hname, ep);
    if (type == NULL)
       type = PD_entry_type(ep);

    err = _PD_hyper_read(file, hname, type, ep, vr);
    _PD_rl_syment_d(ep);

    return(err);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_HYPER_READ - parse an index expression of the form
 *                -
 *                -    <expr>  := <spec> | <expr>, <spec>
 *                -    <spec>  := <start> |
 *                -               <start>:<stop> |
 *                -               <start>:<stop>:<step>
 *                -    <start> := starting integer index value
 *                -    <stop>  := ending integer index value
 *                -    <step>  := integer index step value
 *                -
 *                - and read the specified elements from the PDBfile
 *                - into the array provided
 *                - return the number of item successfully read
 */

int _PD_hyper_read(file, name, outtype, ep, vr)
   PDBfile *file;
   char *name, *outtype;
   syment *ep;
   byte *vr;
   {int nd, c, nrd, hbyt, fbyt;
    char s[MAXLINE], *expr;
    dimdes *dims;
    dimind *pi;
    syment *dep, *tep;

/* to accomodate certain bad users do one quick test
 * see if the variable name is literally in the file
 * this lets things such as foo(2,10) be variable names
 */
    dep = PD_inquire_entry(file, name, FALSE, NULL);
    if (dep != NULL)
       return(_PD_rd_syment(file, ep, outtype, vr));

    dims = PD_entry_dimensions(ep);
    strcpy(s, name);
    c = s[strlen(s)-1];
    if (((c != ')') && (c != ']')) || (dims == NULL))
       return(_PD_rd_syment(file, ep, outtype, vr));

    if (_PD_indirection(outtype))
       PD_error("CAN'T HYPER INDEX INDIRECT TYPE - _PD_HYPER_READ", PD_READ);

    expr = SC_lasttok(s, "[]()");
    strcpy(s, expr);

    pi = _PD_compute_hyper_strides(file, s, dims, &nd);
    if (pi == NULL)
       PD_error("CAN'T FIND HYPER INDICES - _PD_HYPER_READ", PD_READ);

    fbyt = _PD_lookup_size(PD_entry_type(ep), file->chart);
    if (fbyt == -1)
       PD_error("CAN'T FIND NUMBER OF FILE BYTES - _PD_HYPER_READ", PD_READ);

    hbyt = _PD_lookup_size(outtype, file->host_chart);
    if (hbyt == -1)
       PD_error("CAN'T FIND NUMBER OF HOST BYTES - _PD_HYPER_READ", PD_READ);

/* make a dummy for the hyper read to use as scratch space */
    tep = _PD_mk_syment(NULL, 0L, 0L, &(ep->indirects), NULL);

    nrd = _PD_rd_hyper_index(file, tep, vr, pi,
			     PD_entry_type(ep), outtype,
			     PD_entry_address(ep),
			     PD_entry_blocks(ep), hbyt, fbyt);

    _PD_rl_syment(tep);

    SFREE(pi);

    return(nrd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PD_READ_BITS - read a chunk of data which is a bitstream and unpack it
 *              - arguments are:
 *              -   FILE    the PDBfile to use
 *              -   NAME    the name of the variable in the file
 *              -   TYPE    the target type of the data when unpacked
 *              -   NITEMS  the number of items requested
 *              -   SGNED   TRUE if the data type is signed
 *              -   NBITS   the number of bits per item
 *              -   PADSZ   the number of bits of pad preceding the fields
 *              -   FPP     the number of fields per pad
 *              -   OFFS    offset from the beginning of the input data
 *              -   PAN     the number of items found 
 *              -   PDATA   the data array returned
 */

int PD_read_bits(file, name, type, nitems, sgned, nbits,
		 padsz, fpp, offs, pan, pdata)
   PDBfile *file;
   char *name, *type;
   long nitems;
   int sgned, nbits, padsz, fpp;
   long offs, *pan;
   char **pdata;
   {int ret;

    ret = _PD_rd_bits(file, name, type, nitems, sgned, nbits,
		      padsz, fpp, offs, pan, pdata);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_RD_BITS - read a chunk of data which is a bitstream and unpack it
 *             - arguments are:
 *             -   FILE    the PDBfile to use
 *             -   NAME    the name of the variable in the file
 *             -   TYPE    the target type of the data when unpacked
 *             -   NITEMS  the number of items requested
 *             -   SGNED   TRUE if the data type is signed
 *             -   NBITS   the number of bits per item
 *             -   PADSZ   the number of bits of pad preceding the fields
 *             -   FPP     the number of fields per pad
 *             -   OFFS    offset from the beginning of the input data
 *             -   PAN     the number of items found
 *             -   PDATA   the data array returned
 */

int _PD_rd_bits(file, name, type, nitems, sgned, nbits, padsz, fpp, offs, pan, pdata)
   PDBfile *file;
   char *name, *type;
   long nitems;
   int sgned, nbits, padsz, fpp;
   long offs, *pan;
   char **pdata;
   {int i, ityp, out_flag, onescmp;
    int *intorder;
    long nitemsin, enumb, ebyte, obyte, ni;
    char *etype, *in, *out;
    syment *ep;
    defstr *dp1, *dp2;

    ep = _PD_effective_ep(file, name, TRUE, NULL);
    if (ep == NULL)
       return(FALSE);

/*    nitems = PD_entry_number(ep); */
    nitemsin = PD_entry_number(ep);
    etype  = PD_entry_type(ep);

    dp1 = PD_inquire_type(file, etype);
    if (dp1 == NULL)
       return(FALSE);

    onescmp = dp1->onescmp;

    dp2 = PD_inquire_host_type(file, type);
    if (dp2 == NULL)
       return(FALSE);

    out_flag = dp2->order_flag;

    ebyte  = _PD_lookup_size(etype, file->chart);
    enumb = nitemsin * ebyte;

/*    in    = _PD_alloc_entry(file, etype, nitems); */
    in = MAKE_N(char, enumb);

    SFREE(etype);

    PD_entry_type(ep)   = SC_strsavef(SC_CHAR_S, "char*:_PD_RD_BITS:type");
    PD_entry_number(ep) = enumb;
    ni = _PD_rd_syment(file, ep, SC_CHAR_S, in);
    if (ni != enumb)
       {SFREE(in);
	_PD_rl_syment_d(ep);
	return(FALSE);};

    obyte = _PD_lookup_size(type, file->host_chart);
    out = _PD_alloc_entry(file, type, nitems);

    if (strcmp(type, SC_CHAR_S) == 0)
       ityp = SC_CHAR_I;

    else if (strcmp(type, SC_SHORT_S) == 0)
       ityp = SC_SHORT_I;

    else if (strncmp(type, SC_INTEGER_S, 3) == 0)
       ityp = SC_INTEGER_I;

    else if (strcmp(type, SC_LONG_S) == 0)
       ityp = SC_LONG_I;

    else
       return(FALSE);

    _PD_unp_bits(out, in, ityp, nbits, padsz, fpp, nitems, offs);

/* convert integers */
    if (strcmp(type, SC_CHAR_S) != 0)
       {intorder = (int *) SC_alloc(obyte, sizeof(int), "_PD_convert:intorder");
        if (out_flag == NORMAL_ORDER)
           for (i = 0; i < obyte; intorder[i] = i + 1, i++);
        else                         
           for (i = 0; i < obyte; intorder[i] = obyte - i, i++);

        if (sgned)
           _PD_sign_extend(out, nitems, obyte,
			        nbits, intorder);

        if (onescmp)
           _PD_ones_complement(out, nitems, obyte, intorder);

        SFREE(intorder);};

    *pan   = nitems;
    *pdata = out;

    SFREE(in);
    _PD_rl_syment_d(ep);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PD_RD_SYMENT - from the information in the symbol table entry EP
 *               - read the entry from the PDBfile FILE into the location
 *               - pointed to by VR
 *               - at this point the things have been completely dereferenced
 *               - return the number of items successfully read
 *               - this new version is written in a continuation passing
 *               - style so that PDB has control over the stack and
 *               - isn't blowing out the execution stack for long linked
 *               - lists
 */

long _PD_rd_syment(file, ep, outtype, vr)
   PDBfile *file;
   syment *ep;
   char *outtype;
   byte *vr;
   {FILE *fp;
    int dst, vif, size, boffs;
    long i, n, nitems, bytepitem, addr, eaddr, nrd;
    long flag, rd_indx;
    char bf[MAXLINE], *pv, *litype, *lotype, *svr, **lvr;
    symblock *sp;
    symindir iloc;
    defstr *dp;
    memdes *desc, *mem_lst;
    PD_itag pi;

    fp  = file->stream;
    vif = file->virtual_internal;

    if (!vif && io_flush(fp))
       PD_error("FFLUSH FAILED BEFORE READ - _PD_RD_SYMENT", PD_READ);

    fp   = file->stream;
    iloc = ep->indirects;

    file->flushed = FALSE;

    INIT_STACKS(100L, 1000L);

    SAVE_S(lotype, outtype);

    dst = BLOCK;

/* some AIX compilers will erroneously take the default case if
 * this is terminated with a semicolon
 */
    START

/* NOTE: count on this being right and _PD_effective_ep will handle all
 * issues about partial reads across discontiguous blocks by correctly
 * making an effective syment for which this logic works!!!!!!
 */
    case BLOCK :
         bytepitem = _PD_lookup_size(outtype, file->host_chart);
         if (bytepitem == -1)
            PD_error("CAN'T FIND NUMBER OF BYTES - _PD_RD_SYMENT", PD_READ);

         sp = PD_entry_blocks(ep);
         n  = PD_n_blocks(ep);
         if (n == 1)
            sp[0].number = PD_entry_number(ep);

         pv  = (char *) vr;
         nrd = 0L;
         i   = 0L;

    case BLOCK_ITEM :
         if (i >= n)
            {GO_CONT;};

         addr   = sp[i].diskaddr;
         nitems = sp[i].number;

/* if negative we are staring at a bit address */
         if (addr < 0)
            {eaddr = (-addr) >> 3;
	     boffs = -addr - (eaddr << 3);}
         else
            {eaddr = addr;
	     boffs = 0;};

         if (!vif && io_seek(fp, eaddr, SEEK_SET))
            PD_error("FSEEK FAILED TO FIND ADDRESS - _PD_RD_SYMENT", PD_READ);

         SAVE_I(i);
         SAVE_I(n);
         SAVE_S(litype, PD_entry_type(ep));
         SAVE_P(pv);
         SET_CONT(BLOCK_RET);

    case BLOCK_RET:
         RESTORE_P(char, pv);
         RESTORE_S(litype);
         RESTORE_I(n);
         RESTORE_I(i);

         pv += nitems*bytepitem;
         i++;

         GO(BLOCK_ITEM);

    case LEAF:
         if (vif)
            {SC_address ad;

             ad.diskaddr = addr;
             memcpy(pv, ad.memaddr, nitems*bytepitem);}

         else
            _PD_rd_leaf_members(file, pv, nitems, litype, lotype, boffs);

         nrd += nitems;

/* the host type must be used to get the correct member offsets for the
 * in memory copy - the file ones might be wrong!!
 */
         dp = PD_inquire_host_type(file, lotype);
         if (dp == NULL)
            PD_error("BAD TYPE - _PD_RD_SYMENT", PD_READ);

         mem_lst = dp->members;
         if (!dp->n_indirects || (mem_lst == NULL))
            GO_CONT;

         if (pdb_rd_hook != NULL)
            mem_lst = (*pdb_rd_hook)(dp->members);

/* for an array of structs read the indirects for each array element */
         size = dp->size;
         svr  = pv;
         i    = 0L;

    case LEAF_ITEM :
         if (i >= nitems)
            GO_CONT;

         desc = mem_lst;

    case LEAF_INDIR :
         if (desc == NULL)
            {i++;
             svr += size;
             GO(LEAF_ITEM);};

         if (!_PD_indirection(desc->type))
            {desc = desc->next;
             GO(LEAF_INDIR);};

         SAVE_I(i);
         SAVE_I(size);
         SAVE_P(mem_lst);
         SAVE_P(desc);
         SAVE_P(svr);
         SAVE_P(pv);
         pv  = svr + desc->member_offs;
         lvr = (char **) pv;
         SET_CONT_RD(LEAF_RET, SKIP_TO);

    case LEAF_RET :
         RESTORE_P(char, pv);
         RESTORE_P(char, svr);
         RESTORE_P(memdes, desc);
         RESTORE_P(memdes, mem_lst);
         RESTORE_I(size);
         RESTORE_I(i);

         desc = desc->next;
         GO(LEAF_INDIR);

    case INDIRECT :
         SAVE_P(pv);
         lvr = (char **) pv;
         i   = 0L;

    case INDIR_ITEM :
         if (i >= nitems)
            {RESTORE_P(char, pv);
             nrd += nitems;
             GO_CONT;};

         SAVE_I(i);
         SAVE_I(nrd);
         SAVE_P(lvr);
         lvr = &lvr[i];
         SET_CONT_RD(INDIR_RET, SKIP_TO);

    case INDIR_RET :
         RESTORE_P(char *, lvr);
         RESTORE_I(nrd);
         RESTORE_I(i);
         i++;

         GO(INDIR_ITEM);

    case SKIP_TO :
         if (iloc.addr > 0L)
            {long addr, naitems;

	     addr = iloc.addr;
	     io_seek(fp, iloc.addr, SEEK_SET);
	     iloc.addr = -1L;

	     naitems = iloc.n_ind_type*iloc.arr_offs;
	     addr    = _PD_skip_over(file, naitems, FALSE);};

         if (vif)
            {SC_address ad;

             ad.diskaddr = addr;
             DEREF(pv) = DEREF(ad.memaddr);
             GO_CONT;}

         else if ((rd_indx = _PD_rd_ind_tags(file, lvr, &pi)) == -1)
            {GO_CONT;};

/* now read the data */
         SAVE_I(nrd);
         SAVE_I(nitems);

         nitems = pi.nitems;
         addr   = pi.addr;
         flag   = pi.flag;

         SAVE_S(litype, pi.type);
         SAVE_S(lotype, litype);
         SAVE_I(addr);
         SAVE_I(flag);
         SAVE_I(rd_indx);
         SAVE_P(pv);
         pv = lvr[0];
         SET_CONT(SKIP_RET);

    case SKIP_RET :
         RESTORE_P(char, pv);
         RESTORE_I(rd_indx);
         RESTORE_I(flag);
         RESTORE_S(lotype);
         RESTORE_S(litype);
         RESTORE_I(addr);
         RESTORE_I(nitems);
         RESTORE_I(nrd);

         if (rd_indx >= 0)
	    {long addr;

	     addr = io_tell(fp);
	     if (addr == -1L)
	        PD_error("CAN'T FIND RETURN ADDRESS - _PD_RD_SYMENT",
			 PD_READ);
	     file->reta_rd_list[rd_indx] = addr;};

/* restore the file pointer to its original location if necessary */
         if (flag != 1L)
            {if (io_seek(fp, addr, SEEK_SET))
                PD_error("FAILED TO FIND OLD ADDRESS - _PD_RD_SYMENT",
                         PD_READ);};

         GO_CONT;

    case DONE :
         RESTORE_S(lotype);

         return(nrd);

    FINISH("_PD_RD_SYMENT", PD_READ);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
