/* 
 *  naming-client : a client for the CORBA CosNaming Service
 *
 *  Copyright (C) 1998 Sebastian Wilhelmi; University of Karlsruhe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "name.h"
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <limits.h>

typedef enum {nom_bind, rebind, bind_context, rebind_context, unbind, resolve, 
	      new_context, bind_new_context, destroy, list, not_set} mode_type;

char * mode_names[] =
{
  "bind", "rebind", "bind_context", "rebind_context", "unbind", "resolve", 
  "new_context", "bind_new_context", "destroy", "list"
};

void
Usage( char*name )
{
  fprintf( stderr, "Usage:\n" );
  fprintf( stderr, "  %s [bind|rebind|bind_context|rebind_context]"
	   " name object-id\n", name );
  fprintf( stderr, "  %s [resolve|unbind|bind_new_context|list|destroy]"
	   " name\n", name  );
  fprintf( stderr, "  %s [new_context]\n", name );
  fprintf( stderr, "      - name is delimited by '/'.\n");
  fprintf( stderr, "      - object is a result of CORBA_object_to_string.\n");
  exit( 1 );
}

void Exception( CORBA_Environment* ev )
{
  switch( ev->_major )
    {
    case CORBA_SYSTEM_EXCEPTION:
      fprintf( stderr, "CORBA system exception %s.\n",
	       CORBA_exception_id(ev));
      exit ( 1 );
    case CORBA_USER_EXCEPTION:
      fprintf( stderr, "CORBA user exception: %s.\n",
	       CORBA_exception_id( ev ) );
      exit ( 1 );
    default:
      break;
    }
}

CosNaming_Name*
string_to_name( char* string )
{
  static CosNaming_Name name = { 0, 0, NULL };
  int i, n;
  char *t;

  CORBA_free( name._buffer );

  while( *string=='/' )
    {
      string++;
    }

  for(i = n = 0; string[i]; i++) 
    {
      if(string[i] == '/') n++;
    }

  name._buffer = CORBA_sequence_CosNaming_NameComponent_allocbuf( n + 1 );
  name._length = n + 1;
  for(i = 0; i <= n; i++) 
    {
      t = strchr(string, '/');
      name._buffer[ i ].id = t ? g_strndup(string, t - string): g_strdup(string);
      name._buffer[ i ].kind = name._buffer[i].id;
      string = t + 1;
    }
  CORBA_sequence_set_release(&name, FALSE);
  return( &name );
}

int main(int argc, char* argv[])   {
  CORBA_ORB orb;
  CORBA_Environment ev;

  char objref_str[2048];
  CORBA_Object object = CORBA_OBJECT_NIL, outobject = CORBA_OBJECT_NIL;
  char *object_str;

  int i;
  mode_type mode = not_set;
  
  FILE *infile;

  CosNaming_NamingContext NamingContext_obj;

  CosNaming_Name *name;

  if( argc < 2 )
    {
      Usage(argv[0]);
    }

  CORBA_exception_init(&ev);

  orb = CORBA_ORB_init(&argc, argv, "orbit-local-orb", &ev);

  infile = fopen( "/tmp/namingserviceid", "r" );

  fgets( objref_str, 2048, infile );

  fclose( infile );

  i = strlen(objref_str) - 1;
  while(isspace(objref_str[i]))
    objref_str[i--] = '\0';
  g_assert(!(strlen(objref_str) % 2));
  
  NamingContext_obj = CORBA_ORB_string_to_object(orb, objref_str, &ev);
  Exception( &ev );

  for( i = 0; i < sizeof( mode_names ) / sizeof( mode_names[ 0 ] ); i++ )
    {
      if( strcmp( argv[ 1 ] , mode_names[ i ] ) == 0 )
	{
	  mode = i;
	  break;
	}
    }
  if( mode == not_set )
    {
      fprintf( stderr, "mode %s not supported.\n" , argv[ 1 ] );
      CORBA_Object_release( NamingContext_obj, &ev);
      Exception( &ev );
      Usage(argv[0]);
    }
  switch( mode )
    {
    case nom_bind:
    case rebind: 
    case bind_context: 
    case rebind_context:
    case resolve: 
    case unbind: 
    case bind_new_context: 
    case list: 
    case destroy: 
      if( argc < 3 )
	{
	  CORBA_Object_release( NamingContext_obj, &ev);
	  Exception( &ev );
	  Usage(argv[0]);
	}
      name = string_to_name( argv[ 2 ] );
      break;
    default:
      break;
    }
  switch( mode )
    {
    case nom_bind:
    case rebind: 
    case bind_context: 
    case rebind_context:
      if( argc < 4 )
	{
	  CORBA_Object_release( NamingContext_obj, &ev);
	  Exception( &ev );
	  Usage(argv[0]);
	}
      object = CORBA_ORB_string_to_object(orb, argv[ 3 ], &ev);
      Exception( &ev );
      break;
    default:
      break;
    }
  switch( mode )
    {
    case nom_bind:
      CosNaming_NamingContext_bind( NamingContext_obj, name,
				    object, &ev );
      break;
    case rebind:
      CosNaming_NamingContext_rebind( NamingContext_obj, name, 
				      object, &ev );
      break;
    case bind_context:
      CosNaming_NamingContext_bind_context( NamingContext_obj, name, 
					    object, &ev );
      break;
    case rebind_context:
      CosNaming_NamingContext_rebind_context( NamingContext_obj, name, 
					      object, &ev );
      break;
    case resolve: 
      outobject = CosNaming_NamingContext_resolve( NamingContext_obj, name, &ev );
      break;
    case unbind: 
      CosNaming_NamingContext_unbind( NamingContext_obj, name, &ev );
      break;
    case bind_new_context: 
      outobject = CosNaming_NamingContext_bind_new_context( NamingContext_obj, name, &ev );
      break;
    case new_context:
      outobject = CosNaming_NamingContext_new_context( NamingContext_obj, &ev );
      break;
    case destroy:
      outobject = CosNaming_NamingContext_resolve( NamingContext_obj, name, &ev );
      Exception( &ev );
      CosNaming_NamingContext_destroy( outobject, &ev );
      break;
    case list:
      {
	CosNaming_BindingList* bl;
	int i;

	if( strcmp(argv[2],"/") != 0 )
	  {
	    outobject = CosNaming_NamingContext_resolve( NamingContext_obj, name, &ev );
	    Exception( &ev );
	  }
	else
	  {
	    outobject = NamingContext_obj;
	  }
	CosNaming_NamingContext_list(outobject, 0, &bl, &outobject, &ev);
	for (i = 0; i < bl->_length; i++)
	  {
	    printf( "id: %20s, kind: %20s (%s)\n",
		    bl->_buffer[ i ].binding_name._buffer[0].id,
		    bl->_buffer[ i ].binding_name._buffer[0].kind,
		    bl->_buffer[ i ].binding_type == CosNaming_ncontext ? 
		    "context" : "object" );
	  }
      }
      break;
    default:
      break;
    }
  Exception( &ev );

  if(outobject) {
    object_str = CORBA_ORB_object_to_string(orb, outobject, &ev);
    printf("%s\n", object_str);
    CORBA_free(object_str);
    CORBA_Object_release(outobject, &ev);
  }

  if(object)
    CORBA_Object_release(object, &ev);

  CORBA_Object_release( NamingContext_obj, &ev);
  Exception( &ev );

  return 0;
}

