/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  ORBit: A CORBA v2.2 ORB
 *
 *  Copyright (C) 1998 Richard H. Porter and Red Hat Software
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Phil Dawes <philipd@parallax.co.uk>
 *          Elliot Lee <sopwith@redhat.com>
 *
 */

/*
 *   ORBit specific CORBA_Object functions.
 *
 */

#include <string.h>
#include "config.h"
#include "../IIOP/iiop-endianP.h"
#include "orbit_object_type.h"
#include "corba_object_type.h"
#include "allocators.h"
#include "iop.h"
#include <IIOP/IIOP.h>

static void ORBit_object_try_existing_connections(CORBA_Object obj);
static void CORBA_Object_release_fn(CORBA_Object obj, CORBA_Environment *ev);

ORBit_RootObject_Interface CORBA_Object_epv =
{
	(gpointer)CORBA_Object_release_fn,
};

void ORBit_pseudo_object_init(ORBit_PseudoObject obj,
			      ORBit_PseudoObject_type obj_type,
			      CORBA_Environment *ev)
{
	ORBIT_ROOT_OBJECT(obj)->is_pseudo_object = TRUE;
	ORBIT_ROOT_OBJECT(obj)->refs = 0;
	ORBIT_PSEUDO_OBJECT(obj)->pseudo_object_type = obj_type;
}

static const ORBit_RootObject_Interface CORBA_Policy__epv =
{
	(gpointer)CORBA_Policy_destroy
};

void ORBit_policy_object_init(CORBA_Policy obj,
			      CORBA_PolicyType obj_type,
			      CORBA_Environment *ev)
{
	ORBit_pseudo_object_init(ORBIT_PSEUDO_OBJECT(obj),
				 ORBIT_PSEUDO_POLICY, ev);

	obj->policy_type = obj_type;

	ORBit_RootObject_set_interface(ORBIT_ROOT_OBJECT(obj),
				       (gpointer)&CORBA_Policy__epv,
				       ev);
}

void ORBit_object_reference_init(CORBA_Object obj, CORBA_Environment *ev)
{
	ORBIT_ROOT_OBJECT(obj)->is_pseudo_object = FALSE;
	ORBIT_ROOT_OBJECT(obj)->refs = 0;
	/* set the interface up */
	ORBit_RootObject_set_interface(ORBIT_ROOT_OBJECT(obj),&CORBA_Object_epv,ev);
	/* initialise the reference count */
	ORBIT_ROOT_OBJECT(obj)->refs = 0; 
       
}

CORBA_Object 
ORBit_CORBA_Object_new(CORBA_Environment *ev)
{
	CORBA_Object obj;
	/* Create the object */
	obj = ORBIT_CHUNK_ALLOC(CORBA_Object);
	memset(obj, '\0', sizeof(struct CORBA_Object_struct));

	ORBit_object_reference_init(obj, ev);

	return obj;

}

void
ORBit_set_object_key(CORBA_Object obj,
		     gpointer key,
		     CORBA_unsigned_long key_len)
{
	g_assert(obj->objinfo.iopinfo);

	obj->objinfo.iopinfo->object_key._maximum = 0;
	obj->objinfo.iopinfo->object_key._length = key_len;
	obj->objinfo.iopinfo->object_key._buffer = CORBA_octet_allocbuf(key_len);
	memcpy(obj->objinfo.iopinfo->object_key._buffer, key, key_len);

	obj->objinfo.iopinfo->object_key_data = g_malloc(sizeof(CORBA_unsigned_long) + key_len);
	obj->objinfo.iopinfo->object_key_data->_length = key_len;
	memcpy(obj->objinfo.iopinfo->object_key_data->_buffer, key, key_len);

	obj->objinfo.iopinfo->object_key_vec.iov_base =
		(gpointer)obj->objinfo.iopinfo->object_key_data;
	obj->objinfo.iopinfo->object_key_vec.iov_len = sizeof(CORBA_unsigned_long) + key_len;
}

/* this function is wired up to the RootObject interface */
void 
CORBA_Object_release_fn(CORBA_Object obj, CORBA_Environment *ev)
{

	g_assert(obj!=NULL);

	ORBIT_ROOT_OBJECT_UNREF(obj);

	if(ORBIT_ROOT_OBJECT(obj)->refs <= 0) {
		g_hash_table_remove(obj->orb->objrefs, obj);
		
		if(obj->connection)
			giop_connection_unref(obj->connection);

		g_free(obj->objinfo.obj_id);
		if(obj->objinfo.iopinfo) {
			g_free(obj->objinfo.iopinfo->host);
			g_free(obj->objinfo.iopinfo->object_key_data);
			CORBA_free(obj->objinfo.iopinfo->object_key._buffer);
		}
		
		if(obj->objinfo.orbitinfo) {
			g_free(obj->objinfo.orbitinfo->unix_sock_path);
		}

		ORBIT_CHUNK_FREE(CORBA_Object, obj);
	}
}


/* Sets the interface member in the RootObject to the epv specified*/
void
ORBit_RootObject_set_interface(ORBit_RootObject obj,
			       ORBit_RootObject_Interface* epv,
			       CORBA_Environment *ev)
{
	g_assert(obj!=NULL);
	g_assert(epv!=NULL);

	obj->interface = epv;
}

#define GET_ATOM(x) ({ GIOP_RECV_BUFFER(recv_buffer)->decoder(&x, (GIOP_RECV_BUFFER(recv_buffer)->cur), sizeof(x)); GIOP_RECV_BUFFER(recv_buffer)->cur += sizeof(x); })
#define ALIGNFOR(x) recv_buffer->cur = ALIGN_ADDRESS(recv_buffer->cur, sizeof(x))

CORBA_Object
ORBit_create_object_with_info(GHashTable *obj_info,
			      CORBA_ORB orb,
			      CORBA_Environment *ev)
{
	CORBA_Object new;
	TAG_INTERNET_IOP_info *iopinfo;
	TAG_ORBIT_SPECIFIC_info *orbitinfo;
	CORBA_char *type_id;
	struct CORBA_Object_struct refcheck;

	iopinfo = g_hash_table_lookup(obj_info, "IOP_TAG_INTERNET_IOP");
	orbitinfo = g_hash_table_lookup(obj_info, "IOP_TAG_ORBIT_SPECIFIC");
	type_id = g_hash_table_lookup(obj_info, "type_id");

	if(!type_id || !*type_id) {
		CORBA_exception_set_system(ev, ex_CORBA_MARSHAL,
					   CORBA_COMPLETED_MAYBE);
		return CORBA_OBJECT_NIL;
	}

	if(!iopinfo && (!orbitinfo || !(*orbitinfo->unix_sock_path))) {
		CORBA_exception_set_system(ev, ex_CORBA_MARSHAL,
					   CORBA_COMPLETED_MAYBE);
		return CORBA_OBJECT_NIL;
	}

	/* XXX badhack :) */
	refcheck.objinfo.obj_id = type_id;
	refcheck.objinfo.iopinfo = iopinfo;
	refcheck.objinfo.orbitinfo = orbitinfo;

	new = g_hash_table_lookup(orb->objrefs, &refcheck);
	if(new)
		return CORBA_Object_duplicate(new, ev);

	new = ORBit_CORBA_Object_new(ev);
	new->connection = NULL;
 	new->objinfo.obj_id = g_strdup(type_id);
	new->orb = (CORBA_ORB)CORBA_Object_duplicate((CORBA_Object)orb, ev);
	if(orbitinfo) {
		new->objinfo.orbitinfo = &new->orbitinfo;
		new->orbitinfo.unix_sock_path =
			g_strdup(orbitinfo->unix_sock_path);
	}
	if(iopinfo) {
		new->objinfo.iopinfo = &new->iopinfo;
		new->iopinfo = *iopinfo;
		new->iopinfo.host = g_strdup(iopinfo->host);
		ORBit_set_object_key(new, iopinfo->object_key._buffer, iopinfo->object_key._length);
	}

	if(!orbitinfo && !iopinfo) {
		CORBA_exception_set_system(ev, ex_CORBA_COMM_FAILURE,
					   CORBA_COMPLETED_MAYBE);
		goto failure;
	}

	new->connection = NULL;
	ORBit_object_try_existing_connections(new);

	g_hash_table_insert(orb->objrefs, new, new);
	
	return CORBA_Object_duplicate(new, ev);

 failure:
	CORBA_Object_release_fn(new, ev);
	return CORBA_OBJECT_NIL;
}

CORBA_Object
ORBit_demarshal_object(GIOPRecvBuffer *recv_buffer, CORBA_ORB orb)
{
	GHashTable *objinfo;
	TAG_INTERNET_IOP_info iopinfo;
	TAG_MULTIPLE_COMPONENTS_info mcinfo;
	TAG_ORBIT_SPECIFIC_info orbitinfo;
	CORBA_char *type_id;
	CORBA_unsigned_long len, seq_len, subpart_len;
	CORBA_octet o;
	IOP_ProfileId profile_id;
	int i;
	CORBA_Environment ev;
	CDR_Codec *codec;
	CORBA_Object retval;

	CORBA_exception_init(&ev);

	memset(&iopinfo, '\0', sizeof(iopinfo));
	memset(&mcinfo, '\0', sizeof(mcinfo));
	memset(&orbitinfo, '\0', sizeof(orbitinfo));

	/* Get type_id */
	ALIGNFOR(CORBA_unsigned_long);
	GET_ATOM(len);

	if(len == 0)
		return CORBA_OBJECT_NIL;

	objinfo = g_hash_table_new(g_str_hash, g_str_equal);
	type_id = recv_buffer->cur;
	recv_buffer->cur += len;
	g_hash_table_insert(objinfo, "type_id", type_id);

	/* Decode the sequence<TaggedProfile> */
	ALIGNFOR(CORBA_unsigned_long);
	GET_ATOM(seq_len);
	for(i = 0; i < seq_len; i++) {
		ALIGNFOR(CORBA_unsigned_long);
		GET_ATOM(profile_id);
		switch(profile_id) {
		case IOP_TAG_INTERNET_IOP:
			GET_ATOM(subpart_len); /* The length of the embedded sequence */
			codec = CDR_codec_init();
			codec->buffer = recv_buffer->cur;
			codec->release_buffer = CORBA_FALSE;
			recv_buffer->cur += subpart_len;

			codec->readonly = CORBA_TRUE;
			codec->host_endian = codec->data_endian = FLAG_ENDIANNESS;
			codec->buf_len = subpart_len;

			CDR_get_octet(codec, &o);
			codec->data_endian = o;

			CDR_get_octet(codec, &iopinfo.iiop_major);

			if(iopinfo.iiop_major != 1)
				goto error_exit;
			/* XXX should we check for a specific minor version? */
			CDR_get_octet(codec, &iopinfo.iiop_minor);

			CDR_get_string(codec, &iopinfo.host);

			CDR_get_ushort(codec, &iopinfo.port);

			CDR_get_seq_begin(codec, &iopinfo.object_key._length);
			iopinfo.object_key._buffer =
				g_malloc(iopinfo.object_key._length);
			CDR_buffer_gets(codec, iopinfo.object_key._buffer,
					iopinfo.object_key._length);
			iopinfo.object_key._maximum = iopinfo.object_key._release = 0;
			g_hash_table_insert(objinfo, "IOP_TAG_INTERNET_IOP",
					    &iopinfo);
			CDR_codec_free(codec);
			break;

		case IOP_TAG_MULTIPLE_COMPONENTS:
			GET_ATOM(subpart_len);
			g_warning("IOP_TAG_MULTIPLE_COMPONENTS decoding needs finishing");
			recv_buffer->cur += subpart_len;
			break;

		case IOP_TAG_ORBIT_SPECIFIC:
			GET_ATOM(subpart_len);
			codec = CDR_codec_init();
			codec->buffer = recv_buffer->cur;
			codec->release_buffer = CORBA_FALSE;
			recv_buffer->cur += subpart_len;

			codec->readonly = CORBA_TRUE;
			codec->host_endian = codec->data_endian = FLAG_ENDIANNESS;
			codec->buf_len = subpart_len;

			CDR_get_octet(codec, &o);
			codec->data_endian = o;
			CDR_get_string(codec, &orbitinfo.unix_sock_path);
			CDR_get_ushort(codec, &orbitinfo.ipv6_port);
			g_hash_table_insert(objinfo,
					    "IOP_TAG_ORBIT_SPECIFIC",
					    &orbitinfo);
			CDR_codec_free(codec);
			break;
		}
	}

	retval = ORBit_create_object_with_info(objinfo, orb, &ev);
	g_free(iopinfo.host);
	g_free(iopinfo.object_key._buffer);
	g_free(orbitinfo.unix_sock_path);
	g_hash_table_destroy(objinfo);

	return retval;

 error_exit:
	g_hash_table_destroy(objinfo);
	
	CORBA_exception_set_system(&ev, ex_CORBA_MARSHAL,
				   CORBA_COMPLETED_MAYBE);
	return CORBA_OBJECT_NIL;
}

void
ORBit_marshal_object(GIOPSendBuffer *send_buffer, CORBA_Object obj)
{
	static const CORBA_unsigned_long ioptag = IOP_TAG_INTERNET_IOP,
		orbittag = IOP_TAG_ORBIT_SPECIFIC;
	static const CORBA_octet oc_endian = FLAG_ENDIANNESS;
	static const CORBA_octet iiopversion[] = {1,0};
	CORBA_unsigned_long len;
	CDR_Codec *codec;

	if(!obj) {
		len = 0;
		giop_send_buffer_append_mem_indirect_a(send_buffer, &len,
						       sizeof(len));
		return;
	}

	len = strlen(obj->objinfo.obj_id) + 1;
	giop_send_buffer_append_mem_indirect_a(send_buffer, &len,
					       sizeof(len));
	giop_message_buffer_append_mem(GIOP_MESSAGE_BUFFER(send_buffer),
				       obj->objinfo.obj_id, len);

	len = (obj->objinfo.iopinfo && 1) + (obj->objinfo.orbitinfo && 1);
	giop_send_buffer_append_mem_indirect_a(GIOP_SEND_BUFFER(send_buffer),
					       &len, sizeof(len));
	if(obj->objinfo.iopinfo) {
		giop_message_buffer_append_mem(GIOP_MESSAGE_BUFFER(send_buffer),
					       &ioptag, sizeof(ioptag));

		codec = CDR_codec_init();
		codec->buffer = g_malloc(2048);
		codec->buf_len = 2048;
		codec->readonly = CORBA_FALSE;
		codec->data_endian = codec->host_endian = FLAG_ENDIANNESS;
		CDR_put_octet(codec, oc_endian);
		CDR_put_octets(codec, (gpointer)iiopversion, sizeof(iiopversion));
		CDR_put_string(codec, obj->objinfo.iopinfo->host);
		CDR_put_ushort(codec, obj->objinfo.iopinfo->port);
		CDR_put_ulong(codec, obj->objinfo.iopinfo->object_key._length);
		CDR_put_octets(codec, obj->objinfo.iopinfo->object_key._buffer,
			       obj->objinfo.iopinfo->object_key._length);
		len = codec->wptr;
		giop_send_buffer_append_mem_indirect_a(send_buffer,
						       &len, sizeof(len));
		giop_send_buffer_append_mem_indirect(send_buffer,
						     codec->buffer, codec->wptr);
		CDR_codec_free(codec);
	}
	if(obj->objinfo.orbitinfo) {
		giop_message_buffer_append_mem_a(GIOP_MESSAGE_BUFFER(send_buffer),
						 &orbittag, sizeof(orbittag));
		codec = CDR_codec_init();
		codec->buffer = g_malloc(2048);
		codec->buf_len = 2048;
		codec->readonly = CORBA_FALSE;
		codec->data_endian = codec->host_endian = FLAG_ENDIANNESS;
		CDR_put_octet(codec, oc_endian);
		CDR_put_string(codec, obj->objinfo.orbitinfo->unix_sock_path);
		CDR_put_ushort(codec, obj->objinfo.orbitinfo->ipv6_port);
		len = codec->wptr;
		giop_send_buffer_append_mem_indirect_a(send_buffer,
						       &len, sizeof(len));
		giop_send_buffer_append_mem_indirect(send_buffer,
						     codec->buffer, codec->wptr);
		CDR_codec_free(codec);
	}
}

static void
ORBit_object_try_existing_connections(CORBA_Object obj)
{
	if(obj->objinfo.orbitinfo
	   && !ORBit_parse_unixsock(obj,
				    obj->objinfo.orbitinfo->unix_sock_path, TRUE)) {
		/* Do nothing - success */
	} else if(obj->objinfo.iopinfo
		  && !ORBit_parse_inet(obj, obj->objinfo.iopinfo->host,
				       obj->objinfo.iopinfo->port, TRUE)) {
		/* Do nothing - success */
	}
}

GIOPConnection *
_ORBit_object_get_connection(CORBA_Object obj)
{
	g_return_val_if_fail(obj, NULL);

	if (obj->connection) {
		giop_connection_unref(obj->connection);
		obj->connection = NULL;
	}

	if(obj->objinfo.orbitinfo
	   && !ORBit_parse_unixsock(obj,
				    obj->objinfo.orbitinfo->unix_sock_path, FALSE)) {
		/* Do nothing - success */
	} else if(obj->objinfo.iopinfo
		  && !ORBit_parse_inet(obj, obj->objinfo.iopinfo->host,
				       obj->objinfo.iopinfo->port, FALSE)) {
		/* Do nothing - success */
	} else
		return NULL;

	obj->connection->orb_data = obj->orb;

	return obj->connection;
}
