/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  ORBit: A CORBA v2.2 ORB
 *
 *  Copyright (C) 1998 Richard H. Porter
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Dick Porter <dick@cymru.net>
 *
 */

#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "orbit.h"

static gpointer ORBit_any_into_send_buffer(CORBA_Request req,
					   CORBA_TypeCode tc,
					   gpointer mem,
					   gulong len);
static void ORBit_NVList_into_send_buffer(CORBA_Request req,
					  CORBA_Flags invoke_flags);
static void ORBit_NVList_from_recv_buffer(CORBA_Request req,
					  CORBA_Flags invoke_flags);
static void ORBit_any_from_recv_buffer(CORBA_Request req,
				       CORBA_TypeCode tc,
				       gpointer *intomem,
				       gulong *intomem_size,
				       gboolean allocmem);
				      
/* Section 5.2.1 */
CORBA_Status
CORBA_Object_create_request(CORBA_Object obj,
			    CORBA_Context ctx,
			    CORBA_Identifier operation,
			    CORBA_NVList *arg_list,
			    CORBA_NamedValue *result,
			    CORBA_Request *request,
			    CORBA_Flags req_flags,
			    CORBA_Environment *ev)
{
	CORBA_Request new;

	new=g_new(struct CORBA_Request_type, 1);
	if(new==NULL) {
		CORBA_exception_set_system(ev,
					   ex_CORBA_NO_MEMORY,
					   CORBA_COMPLETED_NO);
		return;
	}

	new->obj=obj;
	new->ctx=ctx;
	new->operation=g_strdup(operation);
	new->result=result;
	new->req_flags=req_flags;
	new->request_id = giop_get_request_id();
	new->arg_list = arg_list;
	new->request_buffer = NULL;

	*request=new;
}

/* Section 5.2, 5.2.2 */
CORBA_Status
CORBA_Request_add_arg(CORBA_Request req,
		      CORBA_Identifier name,
		      CORBA_TypeCode arg_type,
		      void *value,
		      CORBA_long len,
		      CORBA_Flags arg_flags,
		      CORBA_Environment *ev)
{
	gpointer new_value;

	g_assert(req!=NULL);

	if((arg_flags & CORBA_IN_COPY_VALUE) && (arg_flags & CORBA_ARG_IN)) {
		new_value=g_malloc(len);
		if(new_value==NULL) {
			CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY,
						   CORBA_COMPLETED_NO);
			return;
		}
		memcpy(new_value, value, (size_t)len);
	} else {
		new_value=value;
	}

	CORBA_NVList_add_item(req->arg_list, name, arg_type,
			      new_value, len, arg_flags | req->req_flags, ev);
}

/* Section 5.2, 5.2.3 */
CORBA_Status
CORBA_Request_invoke(CORBA_Request req,
		     CORBA_Flags invoke_flags,
		     CORBA_Environment *ev)
{
	CORBA_Request_send(req, invoke_flags, ev);
	if(ev->_major == CORBA_NO_EXCEPTION)
		CORBA_Request_get_response(req, invoke_flags, ev);
}

/* Section 5.2, 5.2.4 */
CORBA_Status CORBA_Request_delete(CORBA_Request req, CORBA_Environment *ev)
{
	g_assert(req!=NULL);

	if(req->operation!=NULL)
		g_free(req->operation);
	if(req->arg_list!=NULL)
		CORBA_NVList_free(req->arg_list, ev);
	if(req->result!=NULL)
		ORBit_NamedValue_free(req->result);
	if(req->request_buffer)
		giop_send_buffer_unuse(req->request_buffer);
	if(req->reply_buffer)
		giop_recv_buffer_unuse(req->reply_buffer);
	g_free(req);
}

/* Section 5.2, 5.3.1 */
CORBA_Status
CORBA_Request_send(CORBA_Request req,
		   CORBA_Flags invoke_flags,
		   CORBA_Environment *ev)
{
	struct { CORBA_unsigned_long opstrlen; char opname[1]; } *opnameinfo;
	struct iovec opvec = { NULL, strlen(req->operation)+1+sizeof(CORBA_unsigned_long) };

	g_error("NYI");

	opnameinfo = g_malloc(strlen(req->operation)+1+sizeof(CORBA_unsigned_long));
	opvec.iov_base = (gpointer)opnameinfo;
	strcpy(opnameinfo->opname, req->operation);

	req->request_buffer =
		giop_send_request_buffer_use(req->obj->connection,
					     NULL, /* XXX context ? */
					     req->request_id,
					     TRUE, /* XXX really? */
					     &req->obj->objinfo.iopinfo->object_key_vec,
					     &opvec,
					     NULL /* XXX principal */
					     );
	g_free(opnameinfo);

	if(!req->request_buffer) {
		CORBA_exception_set_system(ev,
					   ex_CORBA_COMM_FAILURE,
					   CORBA_COMPLETED_NO);
		return;
	}

	ORBit_NVList_into_send_buffer(req, invoke_flags);
				      
	giop_send_buffer_write(req->request_buffer);
}

/* Section 5.3.2 */
CORBA_Status
CORBA_send_multiple_requests(CORBA_Request *reqs,
			     CORBA_Environment *env,
			     CORBA_long count,
			     CORBA_Flags invoke_flags)
{
	int i;
	for(i = 0; i < count; i++)
		CORBA_Request_send(reqs[i], invoke_flags, env);

	
}

/* Section 5.2, 5.3.3
 *
 * Raises: WrongTransaction
 */
CORBA_Status
CORBA_Request_get_response(CORBA_Request req,
			   CORBA_Flags response_flags,
			   CORBA_Environment *ev)
{
	/* XXX we currently ignore response_flags */
	req->reply_buffer = giop_recv_reply_buffer_use(req->request_id, TRUE);

	if(!req->reply_buffer) {
		CORBA_exception_set_system(ev,
					   ex_CORBA_COMM_FAILURE,
					   CORBA_COMPLETED_NO);
		return;
	}

	ORBit_NVList_from_recv_buffer(req, response_flags);

	
}

/* Section 5.3.4
 *
 * Raises: WrongTransaction
 */
CORBA_Status
CORBA_get_next_response(CORBA_Environment *env,
			CORBA_Flags response_flags,
			CORBA_Request *req)
{
	GArray *reqids = g_array_new(FALSE, FALSE,
				     sizeof(CORBA_unsigned_long));
	GArray *connectionlist = g_array_new(FALSE, FALSE,
					     sizeof(GIOPConnection*));
	int i, n;
	GIOPRecvBuffer *rb;

	for(i = 0; req[i]; i++) {
		g_array_append_val(reqids, req[i]->request_id);
		g_array_append_val(connectionlist, req[i]->obj->connection);
	}

	n = i;

	g_assert(!"Not yet implemented\n");
#if 0
	rb = giop_recv_reply_buffer_use_multiple((GIOPConnection **)connectionlist->data,
						 (GIOP_unsigned_long *)reqids->data,
						 n,
						 !(response_flags & CORBA_RESP_NO_WAIT));
#endif

	if(rb) {
		
	}

	g_array_free(connectionlist, TRUE);
	g_array_free(reqids, TRUE);
						 
	
}


/* Section 5.4.1 */
CORBA_Status
CORBA_ORB_create_list(CORBA_ORB orb,
		      CORBA_long count,
		      CORBA_NVList **new_list,
		      CORBA_Environment *ev)
{
	CORBA_NVList *new;

	new=g_new0(CORBA_NVList, 1);
	if(new==NULL) goto new_alloc_failed;

	if(count>0) {
		new->list=g_new0(CORBA_NamedValue, count);

		if(new==NULL) goto newlist_alloc_failed;
	}

	new->count=count;
	*new_list= new;

	return;

 newlist_alloc_failed:
	g_free(new);
 new_alloc_failed:
	CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
}

/* Section 5.4.6 */
CORBA_Status
CORBA_ORB_create_operation_list(CORBA_ORB orb,
				CORBA_OperationDef oper,
				CORBA_NVList **new_list,
				CORBA_Environment *ev)
{
	if(!new_list) {
		CORBA_exception_set_system(ev,
					   ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return;
	}

	g_warning("CORBA_ORB_create_operation_list NYI");

	CORBA_exception_set_system(ev,
				   ex_CORBA_IMP_LIMIT,
				   CORBA_COMPLETED_NO);
}

/* Section 5.4.2 */
CORBA_Status CORBA_NVList_add_item(CORBA_NVList *list,
				   CORBA_Identifier item_name,
				   CORBA_TypeCode item_type,
				   void *value,
				   CORBA_long value_len,
				   CORBA_Flags item_flags,
				   CORBA_Environment *ev)
{
	g_assert(list!=NULL);

	if(list->last_add==list->count) {
		/* grow list */
		ORBit_Trace(TraceMod_ORB, TraceLevel_Warning,
			    "CORBA_NVList_add_item: Extending list\n");
		list->list=(CORBA_NamedValue *)
			g_realloc(list->list,
				  (++list->count)*sizeof(CORBA_NamedValue));

		if(list->list==NULL) {
			CORBA_exception_set_system(ev,
						   ex_CORBA_NO_MEMORY,
						   CORBA_COMPLETED_NO);
			return;
		}
	}

	list->list[list->last_add].name=item_name;
	list->list[list->last_add].argument._type=item_type;
	list->list[list->last_add].argument._value=value;
	list->list[list->last_add].arg_modes=item_flags;

	list->last_add++;

	
}

void ORBit_NamedValue_free(CORBA_NamedValue *nv)
{
	g_assert(nv!=NULL);

	if(nv->name!=NULL) {
		g_free(nv->name);
		if(CORBA_any_get_release(&nv->argument)) {
			g_free(nv->argument._value);
		}
	}
	g_free(nv);
}

/* Section 5.4.3 */
CORBA_Status CORBA_NVList_free(CORBA_NVList *list, CORBA_Environment *ev)
{
	int i;

	g_assert(list!=NULL);

	CORBA_NVList_free_memory(list, ev);

	for(i=0;i<list->count;i++)
		ORBit_NamedValue_free(&list->list[i]);

	g_free(list);

	
}

/* Free any dynamically allocated out-arg memory in the list */
/* Section 5.4.4 */
CORBA_Status CORBA_NVList_free_memory(CORBA_NVList *list, CORBA_Environment *ev)
{
	int i;

	g_assert(list!=NULL);

	if(!(list->flags & CORBA_OUT_LIST_MEMORY))
		return;

	for(i=0;i<list->count;i++) {
		if(list->list[i].arg_modes & CORBA_ARG_OUT &&
		   list->list[i].name != NULL) {
			g_free(list->list[i].name);
			list->list[i].name=NULL;

			if(CORBA_any_get_release(&list->list[i].argument))
				g_free(list->list[i].argument._value);
		}
	}

	
}

/* Section 5.4.5 */
CORBA_Status
CORBA_NVList_get_count(CORBA_NVList *list,
		       CORBA_long *count,
		       CORBA_Environment *ev)
{
	g_assert(list!=NULL);

	*count=list->count;
}

/* Section 5.6.2 */
CORBA_Status CORBA_Context_set_one_value(CORBA_Context ctx, CORBA_Identifier prop_name, char *value, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
}

/* Section 5.6.3 */
CORBA_Status CORBA_Context_set_values(CORBA_Context ctx, CORBA_NVList values, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return;
}

/* Section 5.6.4 */
CORBA_Status CORBA_Context_get_values(CORBA_Context ctx, CORBA_Identifier start_scope, CORBA_Flags op_flags, CORBA_Identifier prop_name, CORBA_NVList **values, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	
}

/* Section 5.6.5 */
CORBA_Status CORBA_Context_delete_values(CORBA_Context ctx, CORBA_Identifier prop_name, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	
}

/* Section 5.6.6 */
CORBA_Status CORBA_Context_create_child(CORBA_Context ctx, CORBA_Identifier ctx_name, CORBA_Context *child_ctx, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	
}

/* Section 5.6.7 */
CORBA_Status CORBA_Context_delete(CORBA_Context ctx, CORBA_Flags del_flags, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
}

/*********** Routines for flattening & unflattening data structures **********/ 

/* This array lets us know whether we can encode something directly or
   we have to recurse into the data structure at runtime
*/
const int sizeofs[] = {
	0, /* null */
	0, /* void */
	sizeof(CORBA_short),
	sizeof(CORBA_long),
	sizeof(CORBA_unsigned_short),
	sizeof(CORBA_unsigned_long),
	sizeof(CORBA_float),
	sizeof(CORBA_double),
	sizeof(CORBA_boolean),
	sizeof(CORBA_char),
	sizeof(CORBA_octet),
	-1, /* any */
	-1, /* TypeCode */
	-1, /* Principal */
	-1, /* objref */
	-1, /* struct */
	-1, /* union */
	sizeof(CORBA_unsigned_long), /* enum */
	-1, /* string */
	-1, /* sequence */
	-1, /* array */
	-1, /* alias */
	-1, /* except */
	sizeof(CORBA_long_long),
	sizeof(CORBA_unsigned_long_long),
	sizeof(CORBA_long_double),
	sizeof(CORBA_wchar),
	-1, /* wstring */
	-1, /* fixed */
	-1, /* recursive */
};

/* If we were sticking one of these in a struct how much space would
   it take up? (-1 means that the struct would not store one of these but
   a pointer to it) */

const int container_sizeofs[] = {
	0, /* null */
	0, /* void */
	sizeof(CORBA_short),
	sizeof(CORBA_long),
	sizeof(CORBA_unsigned_short),
	sizeof(CORBA_unsigned_long),
	sizeof(CORBA_float),
	sizeof(CORBA_double),
	sizeof(CORBA_boolean),
	sizeof(CORBA_char),
	sizeof(CORBA_octet),
	sizeof(CORBA_any), /* any */
	-1, /* TypeCode */
	sizeof(CORBA_Principal), /* Principal */
	-1, /* objref */
	-1, /* struct */
	-1, /* union */
	sizeof(CORBA_unsigned_long), /* enum */
	-1, /* string */

	/* Is this correct, or do we store a pointer to the sequence structure? */
	sizeof(CORBA_sequence_octet), /* sequence - all sequences are the same size */
	-1, /* array */
	-1, /* alias */
	-1, /* except */
	sizeof(CORBA_long_long),
	sizeof(CORBA_unsigned_long_long),
	sizeof(CORBA_long_double),
	sizeof(CORBA_wchar),
	-1, /* wstring */
	-1, /* fixed */
	-1, /* recursive */
};

/* Stub things until we write the Real Deal */
gpointer ORBit_encode_string(CORBA_Request req, CORBA_TypeCode tc, gpointer mem, gulong len)
{
	char *straddr = mem;
	CORBA_unsigned_long str_len = strlen(straddr) + 1;

	giop_send_buffer_append_mem_indirect_a(req->request_buffer,
					       &str_len, sizeof(str_len));

	giop_message_buffer_append_mem(req->request_buffer, straddr, len);
	return mem + len;
}

gpointer ORBit_encode_any(CORBA_Request req, CORBA_TypeCode tc, gpointer mem, gulong len);

gpointer ORBit_encode_TypeCode(CORBA_Request req, CORBA_TypeCode tc, gpointer mem, gulong len)
{
	CORBA_TypeCode etc = mem;
	int i;

	giop_message_buffer_append_mem_a(req->request_buffer,
					 &etc->kind,
					 sizeof(CORBA_unsigned_long));

	switch(etc->kind) {
	case CORBA_tk_fixed:
		giop_message_buffer_append_mem_a(req->request_buffer,
						 &etc->digits,
						 sizeof(etc->digits));
		giop_message_buffer_append_mem_a(req->request_buffer,
						 &etc->scale,
						 sizeof(etc->scale));
		break;
	case CORBA_tk_except:
	case CORBA_tk_struct:
		ORBit_encode_string(req, &TC_string,
				    etc->repo_id, INT_MAX);
		ORBit_encode_string(req, &TC_string,
				    etc->name, INT_MAX);
		giop_message_buffer_append_mem_a(req->request_buffer,
						 &etc->sub_parts,
						 sizeof(etc->sub_parts));
		for(i = 0; i < etc->sub_parts; i++) {
			ORBit_encode_string(req, &TC_string,
					    etc->subnames[i], INT_MAX);
			ORBit_encode_TypeCode(req, &TC_TypeCode,
					      etc->subtypes[i], INT_MAX);
		}
		break;
	case CORBA_tk_union:
		ORBit_encode_string(req, &TC_string,
				    etc->repo_id, INT_MAX);
		ORBit_encode_string(req, &TC_string,
				    etc->name, INT_MAX);
		ORBit_encode_TypeCode(req, &TC_TypeCode,
				      etc->discriminator, INT_MAX);
		giop_message_buffer_append_mem_a(req->request_buffer,
						 &etc->default_index,
						 sizeof(etc->default_index));
		giop_message_buffer_append_mem_a(req->request_buffer,
					     &etc->sub_parts,
					     sizeof(etc->sub_parts));
		for(i = 0; i < etc->sub_parts; i++) {
			ORBit_encode_any(req, &TC_any,
					 &etc->sublabels[i], INT_MAX);
			ORBit_encode_string(req, &TC_string,
					    etc->subnames[i], INT_MAX);
			ORBit_encode_TypeCode(req, &TC_TypeCode,
					      etc->subtypes[i], INT_MAX);
		}
		break;
	case CORBA_tk_enum:
		ORBit_encode_string(req, &TC_string,
				    etc->repo_id, INT_MAX);
		ORBit_encode_string(req, &TC_string,
				    etc->name, INT_MAX);
		giop_message_buffer_append_mem_a(req->request_buffer,
						 &etc->sub_parts,
						 sizeof(etc->sub_parts));
		for(i = 0; i < etc->sub_parts; i++) {
			ORBit_encode_string(req, &TC_string,
					    etc->subnames[i], INT_MAX);
		}
		break;
	case CORBA_tk_sequence:
	case CORBA_tk_array:
		ORBit_encode_TypeCode(req, &TC_TypeCode,
				      etc->subtypes[0], INT_MAX);
		giop_message_buffer_append_mem_a(req->request_buffer,
						 &etc->length,
						 sizeof(etc->length));
		break;
	case CORBA_tk_string:
	case CORBA_tk_wstring:
		giop_message_buffer_append_mem_a(req->request_buffer,
						 &etc->length,
						 sizeof(etc->length));
		break;
	case CORBA_tk_alias:
		ORBit_encode_string(req, &TC_string,
				    etc->repo_id, INT_MAX);
		ORBit_encode_string(req, &TC_string,
				    etc->name, INT_MAX);
		break;
	default:
	};

	return mem + sizeof(struct CORBA_TypeCode_struct);
}

gpointer ORBit_encode_any(CORBA_Request req, CORBA_TypeCode tc, gpointer mem, gulong len)
{
	CORBA_any *inany = (CORBA_any *)mem;
	ORBit_encode_TypeCode(req, &TC_any, inany->_type,
			      sizeof(struct CORBA_TypeCode_struct));
	ORBit_any_into_send_buffer(req, inany->_type, inany->_value, INT_MAX);
	return mem + sizeof(CORBA_any);
}

gpointer ORBit_encode_Principal(CORBA_Request req, CORBA_TypeCode tc, gpointer mem, gulong len)
{
	CORBA_sequence_octet *p = (CORBA_sequence_octet *)mem;

	giop_message_buffer_append_mem_a(req->request_buffer, &p->_length,
					 sizeof(p->_length));
	
	giop_message_buffer_append_mem(req->request_buffer, p->_buffer,
				       p->_length);

	return mem + sizeof(CORBA_sequence_octet);
}

gpointer ORBit_encode_objref(CORBA_Request req, CORBA_TypeCode tc, gpointer mem, gulong len)
{
	CORBA_Object o = (CORBA_Object)mem;
	ORBit_encode_string(req, &TC_string,
			    o->objinfo.obj_id, INT_MAX);
	/* XXX I have no clue what you want for the "name", take this */
	ORBit_encode_string(req, &TC_string,
			    o->objinfo.obj_id, INT_MAX);
	return mem + sizeof(struct CORBA_Object_struct);
}

gpointer ORBit_encode_struct(CORBA_Request req, CORBA_TypeCode tc, gpointer mem, gulong len)
{
	gpointer memnew = mem;
	int i;

	for(i = 0; i < tc->sub_parts; i++) {
		if(sizeofs[tc->subtypes[i]->kind] > 0)
			memnew = ORBit_any_into_send_buffer(req,
							    tc->subtypes[i],
							    mem, len);
		else {
			ORBit_any_into_send_buffer(req,
						   tc->subtypes[i],
						   *((gpointer *)mem),
						   len);
			memnew = mem + sizeof(gpointer);
		}
		len -= memnew - mem;
		mem = memnew;
	}

	return memnew;
}

CORBA_unsigned_long
ORBit_tcval_into_ulong(CORBA_TypeCode tc, gpointer mem)
{
	CORBA_unsigned_long retval;

	switch(tc->kind) {
	case CORBA_tk_short:
		retval = *((CORBA_short *)mem);
		break;
	case CORBA_tk_long:
		retval = *((CORBA_long *)mem);
		break;
	case CORBA_tk_ushort:
		retval = *((CORBA_unsigned_short *)mem);
		break;
	case CORBA_tk_enum:
	case CORBA_tk_ulong:
		retval = *((CORBA_unsigned_long *)mem);
		break;
	case CORBA_tk_boolean:
		retval = *((CORBA_boolean *)mem);
		break;
	case CORBA_tk_octet:
		retval = *((CORBA_octet *)mem);
		break;
	case CORBA_tk_longlong:
		retval = *((CORBA_long_long *)mem);
		break;
	case CORBA_tk_ulonglong:
		retval = *((CORBA_unsigned_long_long *)mem);
		break;
	default:
		retval = 0;
		g_error("Invalid discriminator type %d for union",
			tc->kind);
	}
	return retval;
}

gpointer ORBit_encode_union(CORBA_Request req, CORBA_TypeCode tc, gpointer mem, gulong len)
{
	CORBA_unsigned_long discval;
	gpointer memnew;

	g_assert(sizeofs[tc->discriminator->kind] > 0);

	discval = ORBit_tcval_into_ulong(tc, mem);

	memnew = ORBit_any_into_send_buffer(req, tc->discriminator, mem, len);
	len -= memnew - mem;

	if(sizeofs[tc->subtypes[discval]->kind])
		memnew = ORBit_any_into_send_buffer(req,
						    tc->subtypes[discval],
						    memnew, len);
	else {
		ORBit_any_into_send_buffer(req,
					   tc->subtypes[discval],
					   *((gpointer *)memnew), len);
		memnew = mem + sizeof(gpointer);
	}

	return memnew;
}

gpointer ORBit_encode_sequence(CORBA_Request req, CORBA_TypeCode tc, gpointer mem, gulong len)
{
	/* We just need to get _length ;-) */
	CORBA_sequence_octet *getlen = mem;
	gpointer memnew = mem;
	int i;

	for(i = 0; i < getlen->_length; i++) {
		memnew = ORBit_any_into_send_buffer(req,
						    tc->subtypes[0],
						    mem, len);
		len -= memnew - mem;
		mem = memnew;
	}
	
	return memnew;
}

gpointer ORBit_encode_array(CORBA_Request req, CORBA_TypeCode tc, gpointer mem, gulong len)
{
	return NULL;
}

gpointer ORBit_encode_alias(CORBA_Request req, CORBA_TypeCode tc, gpointer mem, gulong len)
{
	return NULL;
}

gpointer ORBit_encode_except(CORBA_Request req, CORBA_TypeCode tc, gpointer mem, gulong len)
{
	return NULL;
}

#define ORBit_encode_wstring ORBit_encode_string

gpointer ORBit_encode_fixed(CORBA_Request req, CORBA_TypeCode tc, gpointer mem, gulong len)
{
	return NULL;
}

gpointer ORBit_encode_recursive(CORBA_Request req, CORBA_TypeCode tc, gpointer mem, gulong len)
{
	return NULL;
}

/* If we do have to recurse into a data structure at runtime,
   what function should we use to encode it? */
typedef gpointer (*ORBitEncodeFunc)(CORBA_Request req,
				    CORBA_TypeCode tc,
				    gpointer mem,
				    gulong len);

ORBitEncodeFunc encfuncs[] = {
	NULL, /* null */
	NULL, /* void */
	NULL, /* (CORBA_short), */
	NULL, /* (CORBA_long), */
	NULL, /* (CORBA_unsigned_short), */
	NULL, /* (CORBA_unsigned_long), */
	NULL, /* sizeof(CORBA_float), */
	NULL, /* sizeof(CORBA_double), */
	NULL, /* sizeof(CORBA_boolean), */
	NULL, /* sizeof(CORBA_char), */
	NULL, /* sizeof(CORBA_octet), */
	&ORBit_encode_any, /* any */
	&ORBit_encode_TypeCode, /* TypeCode */
	&ORBit_encode_Principal, /* Principal */
	&ORBit_encode_objref, /* objref */
	&ORBit_encode_struct, /* struct */
	&ORBit_encode_union, /* union */
	NULL, /* enum */
	&ORBit_encode_string, /* string */
	&ORBit_encode_sequence, /* sequence */
	&ORBit_encode_array, /* array */
	&ORBit_encode_alias, /* alias */
	&ORBit_encode_except, /* except */
	NULL, /* (CORBA_long_long), */
	NULL, /* (CORBA_unsigned_long_long), */
	NULL, /* (CORBA_long_double), */
	NULL, /* (CORBA_wchar), */
	&ORBit_encode_wstring, /* wstring */
	&ORBit_encode_fixed, /* fixed */
	&ORBit_encode_recursive, /* recursive */
};

static gpointer /* pointer to end of mem that we took care of */
ORBit_any_into_send_buffer(CORBA_Request req,
			   CORBA_TypeCode tc,
			   gpointer mem,
			   gulong len)
{
	gulong used_up = 0;

	g_return_val_if_fail(len == 0, NULL);

	if(sizeofs[tc->kind] > 0) {
		used_up = sizeofs[tc->kind];
		if(used_up > 1)
			giop_message_buffer_append_mem_a(req->request_buffer,
							 mem, len);
		else
			giop_message_buffer_append_mem(req->request_buffer,
						       mem, len);
		mem += used_up;
	} else if(encfuncs[tc->kind]) {
		encfuncs[tc->kind](req, tc, mem, len);
	} else
		g_error("Attempt to encode invalid type %d", tc->kind);
						       
	return mem;
}

static void
ORBit_NamedValue_into_send_buffer(CORBA_Request req,
				  CORBA_NamedValue *nv,
				  CORBA_Flags invoke_flags)
{
	/* Ignore return parameters */
	if(nv->arg_modes & CORBA_ARG_OUT)
		return;

	ORBit_any_into_send_buffer(req, nv->argument._type,
				   nv->argument._value, nv->len);
}

static void
ORBit_NVList_into_send_buffer(CORBA_Request req,
			      CORBA_Flags invoke_flags)
{
	int i;
	for(i = 0; i < req->arg_list->count; i++)
		ORBit_NamedValue_into_send_buffer(req,
						  &req->arg_list->list[i],
						  invoke_flags);
}

static void
ORBit_NamedValue_from_recv_buffer(CORBA_Request req,
				  CORBA_NamedValue *nv,
				  CORBA_Flags invoke_flags)
{
	/* Ignore return parameters */
	if(nv->arg_modes & CORBA_ARG_IN)
		return;

#if 0
	/* XXX Heeeeeeelp - we are supposed to free something here, but what? */
	if(nv->arg_modes & CORBA_IN_COPY_VALUE)
		CORBA_free(&nv->argument._value);
#endif

	ORBit_any_from_recv_buffer(req, nv->argument._type,
				   &nv->argument._value, NULL, TRUE);
}

static void
ORBit_NVList_from_recv_buffer(CORBA_Request req,
			      CORBA_Flags invoke_flags)
{
	int i;
	for(i = 0; i < req->arg_list->count; i++)
		ORBit_NamedValue_from_recv_buffer(req,
						  &req->arg_list->list[i],
						  invoke_flags);
}

/* Requirements for a decoder func interface:
   Each decoder should be able to return a pointer to the memory the type
   was decoded into.

   Each decoder should be able to realloc existing allocated memory to
   meet its size requirements.

   Each decoder should be able to write into existing memory.
*/
typedef void (*ORBitDecodeFunc)(CORBA_Request req,
				CORBA_TypeCode tc,

				/* Pointer to the pointer to the
				   memory to be used for the decoded
				   object. */
				gpointer *intomem,

				/* The size of the existing memory in *intomem,
				   or NULL if *intomem is not a valid ptr
				   and needs to be alloced */
				gulong *intomem_size,

				/* Whether we need to allocate memory
				   for the decoded object or not */
				gboolean allocmem);


#define GET_ULONG(x) GIOP_RECV_BUFFER(req->reply_buffer)->decoder(&x, \
(GIOP_RECV_BUFFER(req->reply_buffer)->cur += sizeof(CORBA_unsigned_long)), \
sizeof(CORBA_unsigned_long))

#define GET_LONG(x) GIOP_RECV_BUFFER(req->reply_buffer)->decoder(&x, \
(GIOP_RECV_BUFFER(req->reply_buffer)->cur += sizeof(CORBA_long)), \
sizeof(CORBA_long))

#define GET_USHORT(x) GIOP_RECV_BUFFER(req->reply_buffer)->decoder(&x, \
(GIOP_RECV_BUFFER(req->reply_buffer)->cur += sizeof(CORBA_unsigned_short)), \
sizeof(CORBA_unsigned_short))

#define GET_SHORT(x) GIOP_RECV_BUFFER(req->reply_buffer)->decoder(&x, \
(GIOP_RECV_BUFFER(req->reply_buffer)->cur += sizeof(CORBA_short)), \
sizeof(CORBA_short))

/* The logic encapsulated in this macro (which is used only in the
   type decoding functions) is:

   If the caller wants us to allocate our own memory then:
	 if we weren't given a pre-allocated block to resize, then
	       allocate the memory for it
	 else
	       reallocate the pre-allocated block to fit our needs, putting
	       our decoded stuff onto the end
   else
	 We use the block of memory that was passed to us by the caller.

*/

#define DECODER_ALLOC_MEMORY(memsize) ({ gpointer __newptr; if(allocmem) { if(!intomem_size) {     __newptr = g_malloc0((memsize)); } else {	__newptr = *intomem + *intomem_size;	*intomem_size += (memsize);	*intomem = g_realloc(*intomem, *intomem_size); } } else { __newptr = *intomem; } __newptr; })

#define DECODER_ALLOC_MEMORY_C(memsize, memchunk) ({ gpointer __newptr; if(allocmem) { if(!intomem_size) {     __newptr = ORBIT_CHUNK_ALLOC(memchunk); } else {	__newptr = *intomem + *intomem_size;	*intomem_size += (memsize);	*intomem = g_realloc(*intomem, *intomem_size); } } else { __newptr = *intomem; } __newptr; })

#define DECLARE_DECODER(typename) void ORBit_decode_##typename##(CORBA_Request req, CORBA_TypeCode tc, gpointer *intomem, gulong *intomem_size, gboolean allocmem)

DECLARE_DECODER(TypeCode);
DECLARE_DECODER(string);
DECLARE_DECODER(any);
DECLARE_DECODER(TypeCode);
DECLARE_DECODER(TypeCode);


void
ORBit_decode_any(CORBA_Request req,
		 CORBA_TypeCode tc,
		 gpointer *intomem,
		 gulong *intomem_size,
		 gboolean allocmem)
{
	CORBA_any *setme;

	setme = DECODER_ALLOC_MEMORY(sizeof(CORBA_any));
	
	ORBit_decode_TypeCode(req, tc, (gpointer *)&setme->_type, NULL, TRUE);
	
	ORBit_any_from_recv_buffer(req, setme->_type, &setme->_value, NULL, TRUE);
}

void
ORBit_decode_TypeCode(CORBA_Request req,
		      CORBA_TypeCode tc,
		      gpointer *intomem,
		      gulong *intomem_size,
		      gboolean allocmem)
{
	CORBA_TypeCode setme;
	int i;

	setme = DECODER_ALLOC_MEMORY_C(sizeof(struct CORBA_TypeCode_struct), CORBA_TypeCode);

	GET_ULONG(setme->kind);

	switch(setme->kind) {
	case CORBA_tk_fixed:
		GET_USHORT(setme->digits);
		GET_SHORT(setme->scale);
		break;
	case CORBA_tk_objref:
		ORBit_decode_string(req, tc, (gpointer *)&setme->repo_id, NULL, TRUE);
		ORBit_decode_string(req, tc, (gpointer *)&setme->name, NULL, TRUE);
		break;
	case CORBA_tk_except:
	case CORBA_tk_struct:
		ORBit_decode_string(req, tc, (gpointer *)&setme->repo_id, NULL, TRUE);
		ORBit_decode_string(req, tc, (gpointer *)&setme->name, NULL, TRUE);
		GET_ULONG(setme->sub_parts);
		setme->subnames = g_new(char *, setme->sub_parts);
		setme->subtypes = g_new(CORBA_TypeCode, setme->sub_parts);
		for(i = 0; i < setme->sub_parts; i++) {
			ORBit_decode_string(req, tc, (gpointer *)&setme->subnames[i],
					    NULL, TRUE);
			ORBit_decode_TypeCode(req, tc, (gpointer *)&setme->subtypes[i],
					      NULL, TRUE);
		}
		break;
	case CORBA_tk_union:
		ORBit_decode_string(req, tc, (gpointer *)&setme->repo_id, NULL, TRUE);
		ORBit_decode_string(req, tc, (gpointer *)&setme->name, NULL, TRUE);
		ORBit_decode_TypeCode(req, tc, (gpointer *)&setme->discriminator,
				      NULL, TRUE);
		GET_LONG(setme->default_index);
		GET_ULONG(setme->sub_parts);
		setme->subnames = g_new(char *, setme->sub_parts);
		setme->subtypes = g_new(CORBA_TypeCode, setme->sub_parts);
		setme->sublabels = g_new0(CORBA_any, setme->sub_parts);
		for(i = 0; i < setme->sub_parts; i++) {
			setme->sublabels[i]._type = setme->discriminator;
			ORBit_any_from_recv_buffer(req, setme->discriminator,
						   (gpointer *)&setme->sublabels[i]._value, NULL, TRUE);
			ORBit_decode_string(req, tc, (gpointer *)&setme->subnames[i],
					    NULL, TRUE);
			ORBit_decode_TypeCode(req, tc, (gpointer *)&setme->subtypes[i],
					      NULL, TRUE);
		}
		break;
	case CORBA_tk_enum:
		ORBit_decode_string(req, tc, (gpointer *)&setme->repo_id, NULL, TRUE);
		ORBit_decode_string(req, tc, (gpointer *)&setme->name, NULL, TRUE);
		GET_ULONG(setme->sub_parts);
		setme->subnames = g_new(char *, setme->sub_parts);
		for(i = 0; i < setme->sub_parts; i++) {
			ORBit_decode_string(req, tc, (gpointer *)&setme->subnames[i],
					    NULL, TRUE);
		}
		break;
	case CORBA_tk_array:
	case CORBA_tk_sequence:
		setme->subtypes = g_new(CORBA_TypeCode, 1);
		ORBit_decode_TypeCode(req, tc, (gpointer *)&setme->subtypes[0],
				      NULL, TRUE);
		GET_ULONG(setme->length);
		break;
	case CORBA_tk_alias:
		ORBit_decode_string(req, tc, (gpointer *)&setme->repo_id, NULL, TRUE);
		ORBit_decode_string(req, tc, (gpointer *)&setme->name, NULL, TRUE);
		break;
	default:
		g_error("Unknown TCKind %d while decoding incoming message.",
			setme->kind);
	}
}

void
ORBit_decode_Principal(CORBA_Request req,
		       CORBA_TypeCode tc,
		       gpointer *intomem,
		       gulong *intomem_size,
		       gboolean allocmem)
{
	CORBA_sequence_octet *val;

	val = DECODER_ALLOC_MEMORY(sizeof(CORBA_sequence_octet));

	GET_ULONG(val->_length);

	val->_buffer = g_malloc(val->_length);

	memcpy(val->_buffer, GIOP_RECV_BUFFER(req->reply_buffer)->cur, val->_length);
	GIOP_RECV_BUFFER(req->reply_buffer)->cur += val->_length;
}

void
ORBit_decode_objref(CORBA_Request req,
		    CORBA_TypeCode tc,
		    gpointer *intomem,
		    gulong *intomem_size,
		    gboolean allocmem)
{
	g_assert(!"Not yet implemented.");
}

void
ORBit_decode_struct(CORBA_Request req,
		    CORBA_TypeCode tc,
		    gpointer *intomem,
		    gulong *intomem_size,
		    gboolean allocmem)
{
	g_assert(!"Not yet implemented.");
}

void
ORBit_decode_union(CORBA_Request req,
		   CORBA_TypeCode tc,
		   gpointer *intomem,
		   gulong *intomem_size,
		   gboolean allocmem)
{
	g_assert(!"Not yet implemented.");
}

void
ORBit_decode_string(CORBA_Request req,
		    CORBA_TypeCode tc,
		    gpointer *intomem,
		    gulong *intomem_size,
		    gboolean allocmem)
{
	CORBA_unsigned_long len;
	gpointer retval;

	GET_ULONG(len);

	retval = DECODER_ALLOC_MEMORY(len);

	memcpy(retval, GIOP_RECV_BUFFER(req->reply_buffer)->cur, len);
	GIOP_RECV_BUFFER(req->reply_buffer)->cur += len;
}

void
ORBit_decode_sequence(CORBA_Request req,
		      CORBA_TypeCode tc,
		      gpointer *intomem,
		      gulong *intomem_size,
		      gboolean allocmem)
{
	CORBA_unsigned_long i, spu;
	CORBA_sequence_octet *into;

	into = DECODER_ALLOC_MEMORY(sizeof(CORBA_sequence_octet));
	GET_ULONG(into->_length);

	spu = sizeofs[tc->subtypes[0]->kind];

	if(spu > 0) {
		gpointer curspot = into->_buffer = g_malloc(spu * into->_length);
		for(i = 0; i < into->_length; i++) {
			GIOP_RECV_BUFFER(req->reply_buffer)->decoder(curspot,
								     GIOP_RECV_BUFFER(req->reply_buffer)->cur, spu);

			GIOP_RECV_BUFFER(req->reply_buffer)->cur += spu;
			curspot += spu;
		}
	} else {
		/* XXX fixme. Stuff like CORBA_any isn't a pointer, needs
		   to go into the array directly */
		gpointer curspot;
		gboolean is_ptr;

		spu = container_sizeofs[tc->subtypes[0]->kind];

		if(spu > 0)
			is_ptr = FALSE;
		else {
			spu = sizeof(gpointer);
			is_ptr = TRUE;
		}

		curspot = into->_buffer =
			g_malloc(spu * into->_length);

		for(i = 0; i < into->_length; i++) {
			ORBit_any_from_recv_buffer(req, tc->subtypes[0],
						   &curspot, NULL, is_ptr);
			curspot += spu;
		}
	}
}

void
ORBit_decode_array(CORBA_Request req,
		   CORBA_TypeCode tc,
		   gpointer *intomem,
		   gulong *intomem_size,
		   gboolean allocmem)
{
	g_assert(!"Not yet implemented.");
}

void
ORBit_decode_alias(CORBA_Request req,
		   CORBA_TypeCode tc,
		   gpointer *intomem,
		   gulong *intomem_size,
		   gboolean allocmem)
{
	g_assert(!"Not yet implemented.");
}

#if 0
void
ORBit_decode_except(CORBA_Request req,
		    CORBA_TypeCode tc,
		    gpointer *intomem,
		    gulong *intomem_size,
		    gboolean allocmem)
{
}
#else
#define ORBit_decode_except ORBit_decode_struct
#endif

#if 0
void
ORBit_decode_wstring(CORBA_Request req,
		     CORBA_TypeCode tc,
		     gpointer *intomem,
		     gulong *intomem_size,
		     gboolean allocmem)
{
}
#else
#define ORBit_decode_wstring ORBit_decode_string
#endif

void
ORBit_decode_fixed(CORBA_Request req,
		   CORBA_TypeCode tc,
		   gpointer *intomem,
		   gulong *intomem_size,
		   gboolean allocmem)
{
	g_assert(!"Not yet implemented.");
}

void
ORBit_decode_recursive(CORBA_Request req,
		       CORBA_TypeCode tc,
		       gpointer *intomem,
		       gulong *intomem_size,
		       gboolean allocmem)
{
	g_assert(!"Not yet implemented.");
}

ORBitDecodeFunc decfuncs[] = {
	NULL, /* null */
	NULL, /* void */
	NULL, /* (CORBA_short), */
	NULL, /* (CORBA_long), */
	NULL, /* (CORBA_unsigned_short), */
	NULL, /* (CORBA_unsigned_long), */
	NULL, /* sizeof(CORBA_float), */
	NULL, /* sizeof(CORBA_double), */
	NULL, /* sizeof(CORBA_boolean), */
	NULL, /* sizeof(CORBA_char), */
	NULL, /* sizeof(CORBA_octet), */
	&ORBit_decode_any, /* any */
	&ORBit_decode_TypeCode, /* TypeCode */
	&ORBit_decode_Principal, /* Principal */
	&ORBit_decode_objref, /* objref */
	&ORBit_decode_struct, /* struct */
	&ORBit_decode_union, /* union */
	NULL, /* enum */
	&ORBit_decode_string, /* string */
	&ORBit_decode_sequence, /* sequence */
	&ORBit_decode_array, /* array */
	&ORBit_decode_alias, /* alias */
	&ORBit_decode_except, /* except */
	NULL, /* (CORBA_long_long), */
	NULL, /* (CORBA_unsigned_long_long), */
	NULL, /* (CORBA_long_double), */
	NULL, /* (CORBA_wchar), */
	&ORBit_decode_wstring, /* wstring */
	&ORBit_decode_fixed, /* fixed */
	&ORBit_decode_recursive, /* recursive */
};

static void
ORBit_any_from_recv_buffer(CORBA_Request req,
			   CORBA_TypeCode tc,
			   gpointer *intomem,
			   gulong *intomem_size,
			   gboolean allocmem)
{
	gulong used_up = sizeofs[tc->kind];

	if(used_up > 0) {
		gpointer decmem;
		decmem = DECODER_ALLOC_MEMORY(used_up);
		GIOP_RECV_BUFFER(req->reply_buffer)->decoder(decmem,
							     GIOP_RECV_BUFFER(req->reply_buffer)->cur,
							     used_up);
		GIOP_RECV_BUFFER(req->reply_buffer)->cur += used_up;
	} else if(encfuncs[tc->kind]) {
		decfuncs[tc->kind](req, tc, intomem, intomem_size, allocmem);
	} else
		g_error("Attempt to decode invalid type %d", tc->kind);
}

