/* 
   XRDrawingEngine.m

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Pascal Forget <pascal@wsc.com>
   Date: March 1996
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: May 1998
   
   This file is part of the GNUstep GUI X/RAW Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>
#include <sys/timeb.h>
#include <gnustep/xraw/XR.h>


//*****************************************************************************
//
// 	XRGState 
//
//		A gState objects stores the state of the XRAW drawing engine and acts 
//		as the the client side representation of it's X GC.
//
//*****************************************************************************

@interface XRGState : NSObject
{
	XRFont *font;
	NSColor *color;
	NSPoint originPoint;						
	NSRect canvas;							// two element g state stack
	XRectangle rectangles[2];
	XRectangle prectangles[2];
	XRContext* context;
	XRView *focusView;
	XRWindow *focusWindow;
	Display *xDisplay;
	Window xWindow;
	Drawable xDrawable;
	XFontStruct *xfont;
	int fontAscender;
	BOOL fontNeedsUpdate;					// unequivocally allow changes 
											// to GC related values			
	BOOL isFlipped;
	BOOL priorFlipped;
	GC xGC;
}

@end
@implementation XRGState

+ (void)initialize
{
	if (self == [XRGState class])
		NSDebugLog(@"Initialize XRGState class\n");
}														

- (id)init
{
XGCValues values;
unsigned long valuemask;
Window xDisplayRootWindow;
XSetWindowAttributes winattrs;

	[super init];	

	context = (XRContext *)[XRContext currentContext];
	xDisplayRootWindow = [context xDisplayRootWindow];
	xDisplay = [context xDisplay];

														// Create an X GC for 
	values.foreground = winattrs.background_pixel;		// the content view  
	values.background = winattrs.background_pixel;		// set it's colors per
	values.function = GXcopy;							// chosen NSColor
	valuemask = (GCForeground | GCBackground | GCFunction);
	xGC = XCreateGC(xDisplay, xDisplayRootWindow, valuemask,&values);

	return self;
}

- (void)dealloc
{
	XFreeGC (xDisplay, xGC);							// Free X window's GC

	[super dealloc];									
}

@end  /* XRGState */

//*****************************************************************************
//
// 		XRDrawingEngine 
//
//*****************************************************************************

static XRContext* context = nil;
static XRView *focusView = nil;
static XRWindow *focusWindow = nil;
static Display *xDisplay = None;
static Window xWindow = None;
static Drawable xDrawable = None;
static XFontStruct *xfont = None;
static int fontAscender;
												// drawing engine globals
static XRectangle rectangles[2];
static XRectangle prectangles[2];

static BOOL fontNeedsUpdate = YES;				// unequivocally allow changes 
												// to GC related values			
static BOOL isFlipped = NO;
static BOOL priorFlipped;

static NSPoint originPoint;						
static NSRect drawX;							// two element g state stack
static NSRect priorDrawX;						// two element g state stack
static XRFont *currentFont = nil;
static NSColor *currentColor = nil;
static NSColor *focusWindowGCColor = nil;
static NSColor *XRLightGrayColor = nil;
static NSColor *XRBlackColor = nil;
static NSColor *XRWhiteColor = nil;
static NSColor *XRDarkGrayColor = nil;

static XColor highlightColor;					// highlight XOR X color
static XColor fColor;							// focused window GC's X color

static GC focusWindowGC;
static GC currentGC;		// Global drawing contexts for the most commonly 
static GC blackGC;      	// used colors: Black, White, Light Gray, and      
static GC whiteGC;      	// Dark Gray.  The drawing engine is smart enough        
static GC darkGrayGC;   	// to use these contexts when passed NSColors           
static GC lightGrayGC;		// that either have grayscale or RGB backings that  
                   	 		// correspond to any of these predefined contexts.

//*****************************************************************************
//
//		initialize_gnustep_backend()  	initialize the AppKit back end
//
//*****************************************************************************

BOOL 
initialize_gnustep_backend(void)
{							
	NSDebugLog(@"Using the GNUstep GUI X/RAW backend.\n");
															// set the concrete
	[GSContext setConcreteClass: [XRContext class]];		// context class

	[XRScreen poseAsClass:[NSScreen class]];				// Have the backend
	[XRFontManager poseAsClass:[NSFontManager class]];		// classes poseAs:
	[XRFont poseAsClass:[NSFont class]];					// their frontend 
	[XRApplication poseAsClass:[NSApplication class]];		// super classes
	[XRView poseAsClass:[NSView class]];
	[XRCell poseAsClass:[NSCell class]];
	[XRMenuCell poseAsClass:[NSMenuItem class]];
	[XRMenuMatrix poseAsClass:[NSMenuMatrix class]];
	[XRMenu poseAsClass:[NSMenu class]];
	[XRColor poseAsClass:[NSColor class]];
	[XRButtonCell poseAsClass:[NSButtonCell class]];
	[XRPopUpButton poseAsClass:[NSPopUpButton class]];
	[XRTextFieldCell poseAsClass:[NSTextFieldCell class]];
	[XRClipView poseAsClass:[NSClipView class]];
	[XRImage poseAsClass: [NSImage class]];
	[XRImageRep poseAsClass: [NSImageRep class]];
	[XRBitmapImageRep poseAsClass: [NSBitmapImageRep class]];
	[XRCachedImageRep poseAsClass: [NSCachedImageRep class]];
	[XRCursor poseAsClass: [NSCursor class]];
	[XRWindow poseAsClass:[NSWindow class]];
														// Initialize the XRAW 
	NSDebugLog(@"Initializing XRAW drawing engine\n");	// drawing engine
	XRInitializeDrawingEngine();
	[XRText poseAsClass:[NSText class]];
												// initialize the font manager											
	[NSFontManager sharedFontManager];			// with the available fonts
														
	[XRDragView _sharedDraggingView];					// create the shared
														// dragging view/window
	return YES;											
}
//*****************************************************************************
//
//		XRCacheSystemImages() loads commonly used system images so that
//		they are immediately available to the application.
//
//*****************************************************************************

void 
XRCacheSystemImages(void)
{
	[NSImage imageNamed: @"common_CloseH.tiff"];
	[NSImage imageNamed: @"common_Close.tiff"];
	[NSImage imageNamed: @"common_ArrowDown.tiff"];
	[NSImage imageNamed: @"common_ArrowLeft.tiff"];
	[NSImage imageNamed: @"common_ArrowRight.tiff"];
	[NSImage imageNamed: @"common_ArrowUp.tiff"];
	[NSImage imageNamed: @"common_Dimple.tiff"];
	[NSImage imageNamed: @"common_RadioOff.tiff"];
	[NSImage imageNamed: @"common_RadioOn.tiff"];
	[NSImage imageNamed: @"common_SwitchOff.tiff"];
	[NSImage imageNamed: @"common_SwitchOn.tiff"];
	[NSImage imageNamed: @"common_ret.tiff"];
}
//*****************************************************************************
//
// 		XRInitializeDrawingEngine()   initialize the XRAW drawing engine 
//
//*****************************************************************************

void 
XRInitializeDrawingEngine(void)
{
XGCValues values;
unsigned long valuemask = (GCForeground | GCBackground | GCFunction);
Window win;
XColor bColor;							
XColor wColor;							
XColor dgColor;							
XColor lgColor;							
														// create an abstract
	context = [XRContext contextWithInfo: nil];			// representation of 
	[XRContext setCurrentContext: context];				// the default drawing
	xDisplay = [(XRContext *)context xDisplay];			// destination
	win = [(XRContext *)context xDisplayRootWindow];				  
														// Allocate, retain and
  	XRWhiteColor = [[NSColor whiteColor] retain];		// create an X GC for		
   	XRBlackColor = [[NSColor blackColor] retain];		// each	of the most			
  	XRDarkGrayColor = [[NSColor darkGrayColor] retain];	// commonly used colors		
  	XRLightGrayColor = [[NSColor lightGrayColor] retain];		
														
    wColor = [(XRColor *)XRWhiteColor xColor];
    bColor = [(XRColor *)XRBlackColor xColor];						

	values.function = GXcopy;							
    values.foreground = wColor.pixel;
    values.background = bColor.pixel;								// create a
    whiteGC = XCreateGC(xDisplay, win, valuemask, &values);			// white GC

    values.foreground = bColor.pixel;								// create a
	blackGC = XCreateGC(xDisplay, win, valuemask, &values);			// black GC

    lgColor = [(XRColor *)XRLightGrayColor xColor];
    values.foreground = lgColor.pixel;
    lightGrayGC = XCreateGC(xDisplay, win, valuemask, &values);	  // light gray

    dgColor = [(XRColor *)XRDarkGrayColor xColor];
    values.foreground = dgColor.pixel;
    darkGrayGC = XCreateGC(xDisplay, win, valuemask, &values);	  // dark gray

//  selectionColor = [[NSColor selectedTextColor] retain];		  // tmp FIX ME
//	highlightColor = [(XRColor *)selectionColor xColor];
	highlightColor = lgColor;

//	XRCacheSystemImages(); 						// cache default system images
}														
//*****************************************************************************
//
// 		return a unique integer to identify each of the app's windows 
//
//*****************************************************************************

int 
XRUniqueWindowTag(void)
{
static int uniqueTag = 0;
					
    return uniqueTag++;
}
//*****************************************************************************
//
// 		Swaps a rectangles black for white and vice-versa (not per OS spec)
//
//*****************************************************************************

void NSHighlightRect(NSRect aRect)       
{														// fill rect via XOR
NSPoint org;										
XGCValues values;
unsigned long valuemask;

	org = aRect.origin;									// translate to X coord
	if(isFlipped)								
		org.y = org.y + drawX.origin.y;
	else
		org.y = drawX.size.height - aRect.origin.y - 
				drawX.origin.y - aRect.size.height;
	org.x = aRect.origin.x + drawX.origin.x;

	if(currentColor == XRDarkGrayColor)					// hack for splitview
		values.foreground = highlightColor.pixel;		  
	else	
		values.foreground = 0x1f1f00;					// default highlight  
														// color
	values.function = GXxor;							
	values.background = 0L;								
	valuemask = (GCFunction | GCForeground | GCBackground);
	XChangeGC(xDisplay, focusWindowGC, valuemask, &values);
														// XOR rect with the 
	XFillRectangle(xDisplay, xWindow, focusWindowGC,	// current draw color
					org.x, org.y, aRect.size.width, aRect.size.height);

	if((Drawable)xWindow != (Drawable)xDrawable)				// XOR backing
		XFillRectangle(xDisplay, xDrawable, focusWindowGC,		// if it exists
						org.x, org.y, aRect.size.width, aRect.size.height);

	values.function = GXcopy;							// return GC to it's
	values.foreground = fColor.pixel;					// original state  
	valuemask = (GCFunction | GCForeground);
	XChangeGC(xDisplay, focusWindowGC, valuemask, &values);
}
//*****************************************************************************
//
// 		Fills rect using current color 
//
//*****************************************************************************

void NSRectFill(NSRect aRect)
{
NSPoint org;

	org = aRect.origin;							

	if(isFlipped)
		org.y = org.y + drawX.origin.y;
	else
		org.y = drawX.size.height - aRect.origin.y - 
				drawX.origin.y - aRect.size.height;
	org.x = aRect.origin.x + drawX.origin.x;

	[focusWindow _windowNeedsFlushInRect:NSMakeRect(org.x, org.y,
				 						aRect.size.width, aRect.size.height)];
														// fill area with the  
	XFillRectangle(xDisplay, xDrawable, currentGC,		// current draw color
					(int)org.x, (int)org.y,
					(int)aRect.size.width, (int)aRect.size.height);
}
//*****************************************************************************
//
// 		Fills an array of count rectangles with the current color. 
//
//*****************************************************************************

void NSRectFillList(const NSRect *rects, int count)
{
int i;
    
    for (i=0; i<count; i++)
		NSRectFill(rects[i]);
}
//*****************************************************************************
//
// 		Fills each rectangle in the array 'rects' with the gray whose
//		is stored at the corresponding location in the array 'grays'.
//		Both arrays must be count elements long.  
//
//*****************************************************************************

void NSRectFillListWithGrays(const NSRect *rects,const float *grays,int count) 
{
}
//*****************************************************************************
//
// 		draw a bordered light gray rectangle with appearance of a button
//
//*****************************************************************************

#define PXBW 1.0										// bezel border width

void NSDrawButton(NSRect aRect, NSRect clipRect)
{
XPoint pnts[6];
XPoint org;

	org.y = drawX.size.height - aRect.origin.y - 			// translate to X
				drawX.origin.y - aRect.size.height;			// coordinates
	org.x = aRect.origin.x + drawX.origin.x;

	[focusWindow _windowNeedsFlushInRect:NSMakeRect(org.x, org.y,
										aRect.size.width, aRect.size.height)];	

	aRect.size.height--;									// -1 so that we 
	aRect.size.width--;										// stay in bounds

	pnts[0].x = org.x;										// draw the top and
	pnts[0].y = (org.y + aRect.size.height);				// left light edges
	pnts[1].x = 0;											// one pixel wide
	pnts[1].y = -(aRect.size.height);	
	pnts[2].x = (aRect.size.width);
	pnts[2].y = 0;
	XDrawLines(xDisplay, xDrawable, whiteGC, pnts , 3, CoordModePrevious);

//	pnts[0].x = org.x;										// draw bottom and 
//	pnts[0].y = (org.y + aRect.size.height);				// right dark edges
	pnts[1].x = (aRect.size.width);
	pnts[1].y = 0;
	pnts[2].x = 0;
	pnts[2].y = -(aRect.size.height);
	pnts[3].x = -PXBW;
	pnts[3].y = PXBW;
	pnts[4].x = 0;
	pnts[4].y = (aRect.size.height - 2*PXBW);				// 2x wide shadow
	pnts[5].x = (PXBW - aRect.size.width);
	pnts[5].y = 0;
	XDrawLines(xDisplay, xDrawable, blackGC, pnts, 6, CoordModePrevious);
}
//*****************************************************************************
//
// 		draw a bordered light gray rectangle with pushed-in button appearance
//
//*****************************************************************************

void NSDrawGrayBezel(NSRect aRect, NSRect clipRect)
{
XPoint pnts[6];
XPoint org;

	org.y = drawX.size.height - aRect.origin.y - 			// translate to X
				drawX.origin.y - aRect.size.height;			// coordinates
	org.x = aRect.origin.x + drawX.origin.x;

	[focusWindow _windowNeedsFlushInRect:NSMakeRect(org.x, org.y,
										aRect.size.width, aRect.size.height)];		

	aRect.size.height--;									// -1 so that we 
	aRect.size.width--;										// stay in bounds

	pnts[0].x = org.x;										// draw the top and
	pnts[0].y = (org.y + aRect.size.height);				// left dark edges
	pnts[1].x = 0;											// two pixels wide
	pnts[1].y = -(aRect.size.height);
	pnts[2].x = (aRect.size.width);
	pnts[2].y = 0;
	pnts[3].x = -PXBW;
	pnts[3].y = PXBW;
	pnts[4].x = -(aRect.size.width - 2*PXBW);				// 2x wide shadow
	pnts[4].y = 0;	
	pnts[5].x = 0;
	pnts[5].y = (aRect.size.height - 2*PXBW);
	XDrawLines(xDisplay, xDrawable, blackGC, pnts , 6, CoordModePrevious);
									
	pnts[0].x = org.x + PXBW;								// draw bottom and 				
//	pnts[0].y = (org.y + aRect.size.height);				// right light	
	pnts[1].x = (aRect.size.width - PXBW);					// edges
	pnts[1].y = 0;
	pnts[2].x = 0;
	pnts[2].y = -(aRect.size.height);						// white outer edge
	XDrawLines(xDisplay, xDrawable, whiteGC, pnts, 3, CoordModePrevious);

	pnts[0].x = org.x + 2*PXBW;						
	pnts[0].y = (org.y + aRect.size.height - PXBW);		
	pnts[1].x = (aRect.size.width - 3*PXBW);
//	pnts[1].y = 0;
//	pnts[2].x = 0;
	pnts[2].y = (2*PXBW - aRect.size.height);				// gray inner edge
	XDrawLines(xDisplay, xDrawable, lightGrayGC, pnts, 3, CoordModePrevious);
}
//*****************************************************************************
//
// 		draw a light gray rectangle whose border is a groove
//
//*****************************************************************************

void NSDrawGroove(NSRect aRect, NSRect clipRect)
{
XPoint pnts[5];
XPoint org;

	org.y = drawX.size.height - aRect.origin.y - 			// translate to X 
				drawX.origin.y - aRect.size.height;			// coordinates
	org.x = floor(aRect.origin.x) + drawX.origin.x;
	
	[focusWindow _windowNeedsFlushInRect:NSMakeRect(org.x, org.y,
										aRect.size.width, aRect.size.height)];		

	aRect.size.height--;									// -1 so that we 
	aRect.size.width--;										// stay in bounds

	pnts[0].x = org.x;										// draw top / left 
	pnts[0].y = (org.y + aRect.size.height);				// edges dark gray
	pnts[1].x = 0;											// one pixel wide
	pnts[1].y = -(aRect.size.height);
	pnts[2].x = (aRect.size.width);
	pnts[2].y = 0;
	XDrawLines(xDisplay, xDrawable, darkGrayGC, pnts, 3, CoordModePrevious);
	
	pnts[0].x = org.x + PXBW;								// draw white rect 
//	pnts[0].y = (org.y + aRect.size.height);				// one pixel wide
//	pnts[1].x = 0;
	pnts[1].y = -(aRect.size.height) + PXBW;
	pnts[2].x = (aRect.size.width) - PXBW;
//	pnts[2].y = 0;
	pnts[3].x = 0;
	pnts[3].y = (aRect.size.height) - PXBW;
	pnts[4].x = (PXBW - aRect.size.width);
	pnts[4].y = 0;
	XDrawLines(xDisplay, xDrawable, whiteGC, pnts, 5, CoordModePrevious);
	
	pnts[0].x = org.x + 2*PXBW;
	pnts[0].y = (org.y + aRect.size.height) - PXBW;
	pnts[1].x = (aRect.size.width) - 3*PXBW;
	pnts[1].y = 0;
	pnts[2].x = 0;
	pnts[2].y = -(aRect.size.height) + 3*PXBW;
	XDrawLines(xDisplay, xDrawable, darkGrayGC, pnts, 3, CoordModePrevious);
}
//*****************************************************************************
//
// 		Draws a frame of width 1.0 using the current color.  
//
//*****************************************************************************

void NSFrameRect(NSRect aRect)  
{
XPoint pnts[5];
XPoint org;

	org.y = drawX.size.height - aRect.origin.y - 			// translate to X
				drawX.origin.y - aRect.size.height;			// coordinates
	org.x = floor(aRect.origin.x) + drawX.origin.x;

	[focusWindow _windowNeedsFlushInRect:NSMakeRect(org.x, org.y,
										aRect.size.width, aRect.size.height)];		

	aRect.size.height--;									// -1 so that we 
	aRect.size.width--;										// stay in bounds

	pnts[0].x = org.x;
	pnts[0].y = (org.y + aRect.size.height);
	pnts[1].x = 0;
	pnts[1].y = -(aRect.size.height);
	pnts[2].x = (aRect.size.width);
	pnts[2].y = 0;
	pnts[3].x = 0;
	pnts[3].y = (aRect.size.height);
	pnts[4].x = -(aRect.size.width);						// draw frame rect
	pnts[4].y = 0;											// with set color
	XDrawLines(xDisplay, xDrawable, currentGC, pnts, 5, CoordModePrevious);
}
//*****************************************************************************
//
// 		Draws a frame of width frameWidth using the current color. 
//
//*****************************************************************************

void NSFrameRectWithWidth(NSRect aRect, float frameWidth) 
{
}
//*****************************************************************************
//
// 		draw a string to focus view's XDrawable 
//
//*****************************************************************************

void XRDrawString(const char *str, NSRect rect)      
{
NSPoint org;

	org = rect.origin;								// translate to X coord's
	if(isFlipped)
		org.y = org.y + drawX.origin.y + fontAscender;
	else
		org.y = drawX.size.height - org.y - drawX.origin.y - 
				rect.size.height;
	org.x += drawX.origin.x;

//		[focusWindow _windowNeedsFlushInRect:NSMakeRect(org.x, org.y,
//										rect.size.width, rect.size.height)];		

														// draw string onto the 
	XDrawString(xDisplay, xDrawable, currentGC,			// drawable using the
				org.x, org.y,							// current color
				str, (int)strlen(str));
}

void PSshow(const char *str)   								// emulate PSshow   
{
NSPoint org;
extern NSPoint originPoint;

	org = originPoint;								// translate to X coord's
	org.y = drawX.size.height - org.y - drawX.origin.y;
	org.x += drawX.origin.x;
														// draw string onto the 
	XDrawString(xDisplay, xDrawable, currentGC,			// drawable using the
				org.x, org.y,							// current color
				str, (int)strlen(str));
}
//*****************************************************************************
//
// 		draw an XImage to focus view's XDrawable 
//
//*****************************************************************************

void XRDrawXImage(XImage *xImage, NSRect rect)      
{
NSPoint org;

	org = rect.origin;								// translate to X coord's
	org.y = drawX.size.height - org.y - drawX.origin.y - rect.size.height;
	org.x += drawX.origin.x;

	[focusWindow _windowNeedsFlushInRect:NSMakeRect(org.x, org.y,
										rect.size.width, rect.size.height)];		

	XPutImage(xDisplay, xDrawable, focusWindowGC, xImage,
				0, 0,
				org.x, org.y,
				rect.size.width, rect.size.height);
}
//*****************************************************************************
//
// 		line drawing functions 
//
//*****************************************************************************

void
XRDrawLine(NSPoint a, NSPoint b, NSColor *color)
{
XPoint o, d;
													// translate to X coord's
	o.y = (int)(drawX.size.height - a.y - drawX.origin.y);	
	d.y = (int)(drawX.size.height - b.y - drawX.origin.y);
	o.x = (int)(a.x + drawX.origin.x);
	d.x = (int)(b.x + drawX.origin.x);				// draw a line in drawable 
													// using current color
	XDrawLine(xDisplay, xDrawable, currentGC, o.x, o.y, d.x, d.y);
}

void 
PSmoveto(float x, float y)   							// move drawing engine    
{														// to origin point
    originPoint.x = x;
    originPoint.y = y;
}

void 
PSlineto(float x, float y)   							// Draw line to point   
{														// using current color
NSPoint bPoint;

    bPoint.x = x;
    bPoint.y = y;
    XRDrawLine(originPoint, bPoint, currentColor);
    originPoint = bPoint;
}

void 
PSrlineto(float x, float y)   							// Draw line relative   
{														// to the current point
NSPoint bPoint;											

    bPoint.x = originPoint.x + x;
    bPoint.y = originPoint.y + y;
    XRDrawLine(originPoint, bPoint, currentColor);
    originPoint = bPoint;
}

void PSstroke(void)							{}			// dummies defines not
void PSnewpath(void)						{}			// implemented in XRAW
void PSclosepath(void)						{}
void PSfill(void)							{}
void PSsetlinewidth(float width)			{}

void PSsetgray(float num)					
{														// set the drawing 
	if(num == 0)										// engine's current 
		XRSetCurrentColor(XRBlackColor);				// color to one of the
	else												// predefined cached
		if(num == 1)									// colors if possible
			XRSetCurrentColor(XRWhiteColor);
		else
			if((num >= .33) && (num < .34))
				XRSetCurrentColor(XRDarkGrayColor);
			else
				if(num == .5)
					XRSetCurrentColor(XRLightGrayColor);
}

//*****************************************************************************
//
// 		set the font to be used by the drawing engine to aFont
//
//*****************************************************************************

void 
XRSetCurrentFont(XRFont *aFont)
{
	if(currentFont != aFont || fontNeedsUpdate)			// if font has changed
		{
		fontNeedsUpdate	= NO;	
		currentFont = aFont;

//		fprintf(stderr, "XRSetCurrentFont: \n");
		if(xDisplay)									// set GC's font and
			{											// store ascender val
			xfont = [currentFont xFontStruct];
			if(focusWindowGC)
				XSetFont(xDisplay, focusWindowGC, xfont->fid);
			XSetFont(xDisplay, lightGrayGC, xfont->fid);
			XSetFont(xDisplay, darkGrayGC, xfont->fid);
			XSetFont(xDisplay, blackGC, xfont->fid);
			XSetFont(xDisplay, whiteGC, xfont->fid);
			fontAscender = [currentFont ascender];
			}
		}
}
//*****************************************************************************
//
// 		set the drawing engine's current color 
//
//*****************************************************************************

void 
XRSetCurrentColor(NSColor *color)
{
	if(currentColor != color)							// if color has changed
		{
		currentColor = color;							// check for default
		if (color == XRLightGrayColor) 					// colors and use the
			currentGC = lightGrayGC;					// respective GC if 
		else											// possible
			{											
			if (color == XRDarkGrayColor) 
				currentGC = darkGrayGC;
			else
				{
				if (color == XRBlackColor)
					currentGC = blackGC;
				else
					{
					if (color == XRWhiteColor)
						currentGC = whiteGC;
					else								// a non default color 
						{								// was selected
						if(xDisplay && focusWindowGC)
							{
							currentGC = focusWindowGC;
							if(focusWindowGCColor != color)
							   {
							   fColor = [(XRColor*)currentColor xColor];
							   XSetForeground(xDisplay,currentGC,fColor.pixel);
							   focusWindowGCColor = color;
							   }
							}
			}	}	}	}	
		XRSetCurrentFont(currentFont);
		}						
}

void 
XRSyncFocusWindowColor(NSColor *color, XRWindow *window)
{														// sync focus window's
	if(focusWindow == window)							// GC color if color is
		focusWindowGCColor = color;						// changed outside of		
}														// the drawing engine

NSColor * XRLightGray(void)			{ return XRLightGrayColor; }		
NSColor * XRBlack(void)				{ return XRBlackColor; }
NSColor * XRWhite(void)				{ return XRWhiteColor; }
NSColor * XRDarkGray(void)			{ return XRDarkGrayColor; }

//*****************************************************************************
//
// 		set and return the drawing engine's current X GC  (X graphics context)
//
//*****************************************************************************

GC XRCurrentGC(void)				{ return currentGC;	}	// return current
															// drawing GC
void XRSetGC(GC aGC)
{													// set the drawing engine's
	currentGC = aGC;								// current GC (X graphics  
													// context)
	if (currentGC && xDisplay) 
		{				
		xfont = [currentFont xFontStruct];
		XSetFont(xDisplay, currentGC, xfont->fid);
		fontAscender = [currentFont ascender];
		}
}
//*****************************************************************************
//	
// 		store and return focus view's origin and size in coordinate system  
//		of it's XWindow 
//
//*****************************************************************************

void XRSetCanvasOrigin(NSPoint aPoint)
{
	drawX.origin.x = aPoint.x;						// store the drawing origin
	drawX.origin.y = aPoint.y;						// of the focus'd XDrawable
	isFlipped = NO;
}			

void XRSetCanvasSize(NSSize aPoint)
{													// store the size of the
	drawX.size.width = aPoint.width;				// focus'd XDrawable
	drawX.size.height = aPoint.height;
}

NSRect XRCanvasRect(void)
{													// return lock focused
	return drawX;									// XDrawable's origin+size					 
}

void XRSetFlipped(void)								// drawing engine coords
{													// are flipped (same as X)
	isFlipped = YES;
}

void PSgsave(void)									// save emulation of a two 
{													// element graphics state
	priorDrawX = drawX;								// stack
	priorFlipped = isFlipped;

	prectangles[0].x = rectangles[0].x;
	prectangles[0].y = rectangles[0].y;
	prectangles[0].width = rectangles[0].width;
	prectangles[0].height = rectangles[0].height;			 
}

void PSgrestore(void)								// restore emulation of a 
{													// two element graphics
	drawX = priorDrawX;								// state stack
	isFlipped = priorFlipped;

	rectangles[0].x = prectangles[0].x;
	rectangles[0].y = prectangles[0].y;
	rectangles[0].width = prectangles[0].width;
	rectangles[0].height = prectangles[0].height;	

	if(focusWindowGC)	
		XRSetPredefinedClipPath();
}

void XRFocusLock(XRView *view, XRWindow *window)			
{															// set draw focus
	focusView = view;
	if(focusWindow != window)								// if focus view is
		{													// in a new window			
		focusWindow = window;								// get more info
		context = (XRContext *)[XRContext currentContext];
		xDisplay = [context xDisplay];
		xWindow = [focusWindow xWindow];
		focusWindowGC = [window xGC];
		xDrawable = [window xDrawable];
		XRSetCanvasSize ([window frame].size);
//		fprintf(stderr, "XRFocusLock: window changed\n");

		focusWindowGCColor = nil;
		fontNeedsUpdate = YES;								// GC's font needs 
		}													// to be set again
	currentGC = focusWindowGC;
	currentColor = nil;
}

void XRWindowGeometryChanged(XRWindow *window, Drawable x_drawable)
{
	if(focusWindow == window)								// if focus window
		{													// geometry changes
		xDrawable = x_drawable;								// update Xdrawable
		XRSetCanvasSize ([window frame].size);				// and canvas size
		}
}															
//*****************************************************************************
//
// 		copy rectangle to new coordinate point 
//
//*****************************************************************************

void XRCopyRectToPoint(NSRect rect, NSPoint point)      
{
NSPoint dest, src;

	dest = rect.origin;								// translate to X coord's
	src = point;									// translate to X coord's

	if(isFlipped)
		{
		src.y += drawX.origin.y;
		dest.y += drawX.origin.y;
		}

	XCopyArea(xDisplay, xDrawable, xDrawable, focusWindowGC,
				src.x, src.y,
				rect.size.width, rect.size.height,
				dest.x, dest.y);
}
//*****************************************************************************
//
// 		flush rect from window's back store 
//
//*****************************************************************************

void XRFlushRect(NSRect rect)      
{
NSPoint org;

	org = rect.origin;									// translate to X coord
	org.y = drawX.size.height - org.y - drawX.origin.y - rect.size.height;
	org.x += drawX.origin.x;
														// if window has back'g
	if ((Drawable)xWindow != (Drawable)xDrawable) 		// copy pixmap backing
		{												// to view's Xwindow 
		XCopyArea(xDisplay, xDrawable, xWindow, currentGC,	
					org.x, org.y,
					rect.size.width, rect.size.height,
					org.x, org.y);
		}
}

void XRFlushWindowRect(NSRect rect)      				// X coords are assumed
{														// copy pixmap backing
	if ((Drawable)xWindow != (Drawable)xDrawable) 		// to view's Xwindow if
		{												// they are not same 
		XCopyArea(xDisplay, xDrawable, xWindow, currentGC,	
					rect.origin.x, rect.origin.y,
					rect.size.width, rect.size.height,
					rect.origin.x, rect.origin.y);
		}
}
//*****************************************************************************
//
// 		clipping routines 
//
//*****************************************************************************

void NSRectClip(NSRect rect)      
{
NSRect xRect;
													// Convert the rectangle to  
	rect = [focusView convertRect:rect toView:nil];	// the window's coordinates
	
	if(rect.origin.y < 0)							// should not happen FIX ME
		{
		fprintf (stderr, "NSRectClip error: negative value in clip rect\n");
		return;
		}										// if view is rotated compute 
												// the bounding box rectangle   
	if ([focusView isRotatedFromBase])			// in the window's coordinates
		rect = [focusView _boundingRectFor:rect];
													// Convert the rectangle to 
	xRect.origin.x = rect.origin.x;					// X coordinates
	xRect.origin.y = drawX.size.height - (rect.origin.y + rect.size.height);
	xRect.size = rect.size;

	XRClipWindowRect(xRect);      					
}

void NSRectClipList(const NSRect *rects, int count)
{
int i;
    
    for (i=0; i<count; i++)
		NSRectClip(rects[i]);
}

void XRClipWindowRect(NSRect rect)      
{														// X coords are assumed
	rectangles[0].x = rect.origin.x;
	rectangles[0].y = rect.origin.y;
	rectangles[0].width = rect.size.width;		// Set the clipping path to the
	rectangles[0].height = rect.size.height;	// rectangles in order to limit 
												// the effects of drawing
	XSetClipRectangles(xDisplay, focusWindowGC, 0, 0, rectangles, 1, Unsorted);	
	XSetClipRectangles(xDisplay, lightGrayGC, 0, 0, rectangles, 1, Unsorted);	
	XSetClipRectangles(xDisplay, darkGrayGC, 0, 0, rectangles, 1, Unsorted);	
	XSetClipRectangles(xDisplay, blackGC, 0, 0, rectangles, 1, Unsorted);	
	XSetClipRectangles(xDisplay, whiteGC, 0, 0, rectangles, 1, Unsorted);	
}

void XRSetPredefinedClipPath(void)
{														// Set a clipping path  
	XSetClipRectangles(xDisplay, focusWindowGC, 0, 0, rectangles, 1, Unsorted);	
	XSetClipRectangles(xDisplay, lightGrayGC, 0, 0, rectangles, 1, Unsorted);	
	XSetClipRectangles(xDisplay, darkGrayGC, 0, 0, rectangles, 1, Unsorted);	
	XSetClipRectangles(xDisplay, blackGC, 0, 0, rectangles, 1, Unsorted);	
	XSetClipRectangles(xDisplay, whiteGC, 0, 0, rectangles, 1, Unsorted);	
}

void XRRemoveClipPath(void)
{
	if(focusWindowGC)
		XSetClipMask(xDisplay, focusWindowGC, None);
	XSetClipMask(xDisplay, lightGrayGC, None);
	XSetClipMask(xDisplay, darkGrayGC, None);
	XSetClipMask(xDisplay, blackGC, None);
	XSetClipMask(xDisplay, whiteGC, None);
}
//*****************************************************************************
//
// 		gState 
//
//*****************************************************************************

int XRAllocateGState(void)
{
XRGState* gs;

	gs = [XRGState new];
	gs->font = currentFont;

	return 0;
}
