------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--     S Y S T E M . T A S K _ P R I M I T I V E S . O P E R A T I O N S    --
--                                                                          --
--                                  B o d y                                 --
--                         (Version for new GNARL)                          --
--                                                                          --
--                             $Revision: 1.4 $                             --
--                                                                          --
--            Copyright (C) 1997, Free Software Foundation, Inc.            --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University. It is --
-- now maintained by Ada Core Technologies Inc. in cooperation with Florida --
-- State University (http://www.gnat.com).                                  --
--                                                                          --
------------------------------------------------------------------------------

--  This is a NT (native) version of this package.

with Interfaces.C;
--  used for int
--           size_t

with Interfaces.C.Strings;
--  used for Null_Ptr

with System.Error_Reporting;
--  used for Shutdown

with System.Interrupt_Management;
--  used for Keep_Unmasked
--           Abort_Task_Interrupt
--           Interrupt_ID

with System.Interrupt_Management.Operations;
--  used for Set_Interrupt_Mask
--           All_Tasks_Mask
pragma Elaborate_All (System.Interrupt_Management.Operations);

with System.OS_Interface;
--  used for various type, constant, and operations

with System.Parameters;
--  used for Size_Type

with System.Tasking;
--  used for Ada_Task_Control_Block
--           Task_ID

with System.Assertions;
pragma Elaborate_All (System.Assertions);

with Unchecked_Conversion;
with Unchecked_Deallocation;

package body System.Task_Primitives.Operations is

   use System.Tasking;
   use Interfaces.C;
   use Interfaces.C.Strings;
   use System.Error_Reporting;
   use System.OS_Interface;
   use System.Parameters;

   ------------------------------------
   -- The thread local storage index --
   ------------------------------------

   TlsIndex          : DWORD;

   --------------------------------------
   -- Some useful conversion functions --
   --------------------------------------

   function To_Task_ID is new Unchecked_Conversion (System.Address, Task_ID);

   function To_Address is new Unchecked_Conversion (Task_ID, System.Address);

   function To_Timespec is new
     Unchecked_Conversion (Long_Long_Integer, timespec);

   -----------------------------------
   --  Condition Variable Functions --
   -----------------------------------

   procedure Initialize_Cond (Cond : access Condition_Variable);

   procedure Finalize_Cond (Cond : access Condition_Variable);

   procedure Cond_Signal (Cond : access Condition_Variable);

   procedure Cond_Wait
     (Cond : access Condition_Variable;
      L    : access RTS_Lock);

   procedure Cond_Timed_Wait
     (Cond      : access Condition_Variable;
      L         : access RTS_Lock;
      Rel_Time  : Duration;
      Timed_Out : out Boolean;
      Status    : out Integer);

   ---------------------
   -- Initialize_Cond --
   ---------------------

   procedure Initialize_Cond (Cond : access Condition_Variable) is
      hEvent            : HANDLE;
   begin
      hEvent := CreateEvent (null, True, False, Null_Ptr);
      pragma Assert (hEvent /= 0
        or else Shutdown ("GNULLI failure---Initialize_Cond (CreateEvent)"));
      Cond.all := Condition_Variable (hEvent);
   end Initialize_Cond;

   -------------------
   -- Finalize_Cond --
   -------------------

   --  No such problem here, DosCloseEventSem has been derived.
   --  What does such refer to in above comment???

   procedure Finalize_Cond (Cond : access Condition_Variable) is
      Result : BOOL;
   begin
      Result := CloseHandle (HANDLE (Cond.all));
      pragma Assert (Result = True
        or else Shutdown ("GNULLI failure---Finalize_Cond (CloseHandle)"));
   end Finalize_Cond;

   -----------------
   -- Cond_Signal --
   -----------------

   procedure Cond_Signal (Cond : access Condition_Variable) is
      Result : BOOL;
   begin
      Result := SetEvent (HANDLE (Cond.all));
      pragma Assert (Result = True
        or else Shutdown ("GNULLI failure---Cond_Signal (SetEvent)"));
   end Cond_Signal;

   ---------------
   -- Cond_Wait --
   ---------------

   --  Pre-assertion: Cond is posted
   --                 L is locked.

   --  Post-assertion: Cond is posted
   --                  L is locked.

   procedure Cond_Wait
     (Cond : access Condition_Variable;
      L    : access RTS_Lock) is

      Result      : DWORD;
      Result_Bool : BOOL;
   begin
      --  Must reset Cond BEFORE L is unlocked.
      Result_Bool := ResetEvent (HANDLE (Cond.all));
      pragma Assert (Result_Bool = True
        or else Shutdown ("GNULLI failure---Cond_Wait (ResetEvent)"));
      Unlock (L);

      --  No problem if we are interrupted here: if the condition is signaled,
      --  WaitForSingleObject will simply not block
      Result := WaitForSingleObject (HANDLE (Cond.all), Wait_Infinite);
      pragma Assert (Result = 0
        or else Shutdown ("GNULLI failure---Cond_Wait (WaitForSingleObject)"));

      Write_Lock (L);
   end Cond_Wait;

   ---------------------
   -- Cond_Timed_Wait --
   ---------------------

   --  Pre-assertion: Cond is posted
   --                 L is locked.

   --  Post-assertion: Cond is posted
   --                  L is locked.

   procedure Cond_Timed_Wait
     (Cond      : access Condition_Variable;
      L         : access RTS_Lock;
      Rel_Time  : Duration;
      Timed_Out : out Boolean;
      Status    : out Integer) is

      Time_Out : DWORD;
      Result   : BOOL;
      --  Rel_Time : Duration;
      Int_Rel_Time : DWORD;
      Wait_Result  : DWORD;

   begin
      --  Must reset Cond BEFORE L is unlocked.

      Result := ResetEvent (HANDLE (Cond.all));
      pragma Assert (Result = True
        or else Shutdown ("GNULLI failure---Cond_Timed_Wait (ResetEvent)"));
      Unlock (L);

      --  No problem if we are interrupted here: if the condition is signaled,
      --  WaitForSingleObject will simply not block

      if Rel_Time <= 0.0 then
         Timed_Out := True;
      else
         --  Time_Out := DWORD (Stimespec_Seconds  (Rel_Time)) * 1000 +
         --        DWORD (Stimespec_NSeconds (Rel_Time) / 1E6);
         Int_Rel_Time := DWORD (Rel_Time);
         Time_Out := Int_Rel_Time * 1000 +
                     DWORD ((Rel_Time - Duration (Int_Rel_Time)) * 1000.0);
         Wait_Result := WaitForSingleObject (HANDLE (Cond.all), Time_Out);
         if Wait_Result = WAIT_TIMEOUT then
            Timed_Out := True;
            Wait_Result := 0;
         else
            Timed_Out := False;
         end if;
      end if;

      Write_Lock (L);

      --  Ensure post-condition

      if Timed_Out then
         Result := SetEvent (HANDLE (Cond.all));
         pragma Assert (Result = True or else
           Shutdown ("GNULLI failure---Cond_Timed_Wait (SetEvent)"));
      end if;
      Status := Integer (Wait_Result);
   end Cond_Timed_Wait;

   ------------------------
   -- public subprograms --
   ------------------------

   ----------
   -- Self --
   ----------

   function Self return Task_ID is
      Self_ID : Task_ID;
   begin
      Self_ID := To_Task_ID (TlsGetValue (TlsIndex));

      --  Since there is no pthread_kill, we test here if we have been
      --  aborted. It's just a simple comparison, so the efficiency
      --  shouldn't be a pbl.
      if Self_ID.LL.Aborted then
         Self_ID.LL.Aborted := False;
         if (Self_ID.Deferral_Level = 0 and then
             Self_ID.Pending_ATC_Level < Self_ID.ATC_Nesting_Level) then

            raise Standard'Abort_Signal;
         end if;
      end if;

      return Self_ID;
   end Self;

   ---------------------
   -- Initialize_Lock --
   ---------------------

   --  Note: mutexes and cond_variables needed per-task basis are
   --        initialized in Intialize_TCB and the Storage_Error is
   --        handled. Other mutexes (such as All_Tasks_Lock, Memory_Lock...)
   --        used in RTS is initialized before any status change of RTS.
   --        Therefore rasing Storage_Error in the following routines
   --        should be able to be handled safely.

   procedure Initialize_Lock
     (Prio : System.Any_Priority;
      L    : access Lock) is

      MutexAttributes   : aliased SECURITY_ATTRIBUTES :=
         (SECURITY_ATTRIBUTES'Size / System.Storage_Unit, Null_Void, False);
   begin
      --  L.Mutex := CreateMutex
      --     (MutexAttributes'Unchecked_Access, False, Null_Ptr);
      L.Mutex := CreateMutex (null, False, Null_Ptr);

      if L.Mutex = 0 then
         raise Storage_Error;
      end if;

      L.Priority := Prio;
   end Initialize_Lock;

   procedure Initialize_Lock (L : access RTS_Lock) is

      MutexAttributes   : aliased SECURITY_ATTRIBUTES :=
         (SECURITY_ATTRIBUTES'Size / System.Storage_Unit, Null_Void, False);
   begin
      L.all := CreateMutex (null, False, Null_Ptr);

      if L.all = 0 then
         raise Storage_Error;
      end if;
   end Initialize_Lock;

   -------------------
   -- Finalize_Lock --
   -------------------

   procedure Finalize_Lock (L : access Lock) is
      Result : BOOL;
   begin
      Result := CloseHandle (L.Mutex);
      pragma Assert (Result = True
        or else Shutdown ("GNULLI failure---Finalize_Lock (CloseHandle)"));
   end Finalize_Lock;

   procedure Finalize_Lock (L : access RTS_Lock) is
      Result : BOOL;
   begin
      Result := CloseHandle (L.all);
      pragma Assert (Result = True
        or else Shutdown ("GNULLI failure---Finalize_Lock (CloseHandle)"));
   end Finalize_Lock;

   ----------------
   -- Write_Lock --
   ----------------

   procedure Write_Lock (L : access Lock; Ceiling_Violation : out Boolean) is
      Result      : DWORD;
   begin

      L.Owner_Priority := Get_Priority (Self);

      if L.Priority < L.Owner_Priority then
         Ceiling_Violation := True;
         return;
      end if;

      Result := WaitForSingleObject (L.Mutex, Wait_Infinite);
      pragma Assert (Result = NO_ERROR or else
        Shutdown ("GNULLI failure---Write_Lock (WaitForSingleObject)"));

      Ceiling_Violation := False;

      --  if L.Priority > L.Owner_Priority then
      --  LabTek : temporary fix to prevent constraint error
      --     declare
      --        Limited_Priority : Priority;
      --     begin
      --        if L.Priority > Standard'Max_Priority then
      --           Limited_Priority := Standard'Max_Priority;
      --        else
      --           Limited_Priority := L.Priority;
      --        end if;
      --        Set_Own_Priority (Limited_Priority);
      --     end;
      --  end if;
   end Write_Lock;

   procedure Write_Lock (L : access RTS_Lock) is
      Result      : DWORD;
   begin
      Result := WaitForSingleObject (HANDLE (L.all), Wait_Infinite);
      pragma Assert (Result = NO_ERROR or else
        Shutdown ("GNULLI failure---Write_Lock (WaitForSingleObject)"));
   end Write_Lock;

   procedure Write_Lock (T : Task_ID) is
      Result      : DWORD;
   begin
      Result := WaitForSingleObject (HANDLE (T.LL.L), Wait_Infinite);
      pragma Assert (Result = NO_ERROR or else
        Shutdown ("GNULLI failure---Write_Lock (WaitForSingleObject)"));
   end Write_Lock;

   ---------------
   -- Read_Lock --
   ---------------

   procedure Read_Lock (L : access Lock; Ceiling_Violation : out Boolean) is
   begin
      Write_Lock (L, Ceiling_Violation);
   end Read_Lock;

   ------------
   -- Unlock --
   ------------

   procedure Unlock (L : access Lock) is
      Result      : BOOL;
   begin
      --  if L.Owner_Priority /= L.Priority then
      --     Set_Own_Priority (L.Owner_Priority);
      --  end if;
      Result := ReleaseMutex (L.Mutex);
      pragma Assert (Result = True or else
        Shutdown ("GNULLI failure---UnLock (ReleaseMutex)"));
   end Unlock;

   procedure Unlock (L : access RTS_Lock) is
      Result      : BOOL;
   begin
      Result := ReleaseMutex (HANDLE (L.all));
      pragma Assert (Result = True or else
        Shutdown ("GNULLI failure---UnLock (ReleaseMutex)"));
   end Unlock;

   procedure Unlock (T : Task_ID) is
      Result      : BOOL;
   begin
      Result := ReleaseMutex (HANDLE (T.LL.L));
      pragma Assert (Result = True or else
        Shutdown ("GNULLI failure---UnLock (ReleaseMutex)"));
   end Unlock;

   -------------
   --  Sleep  --
   -------------

   procedure Sleep (Self_ID : Task_ID) is
   begin
      pragma Assert (Self_ID = Self
        or else Shutdown ("GNULLI failure---Sleep (Self)"));

      Cond_Wait (Self_ID.LL.CV'Access, Self_ID.LL.L'Access);

      if Self_ID.Deferral_Level = 0
        and then Self_ID.Pending_ATC_Level < Self_ID.ATC_Nesting_Level
      then
         Unlock (Self_ID);
         raise Standard'Abort_Signal;
      end if;
   end Sleep;

   -----------------
   -- Sleep_Until --
   -----------------

   procedure Sleep_Until
     (Self_ID  : Task_ID;
      Abs_Time : Duration;
      Timedout : out Boolean) is
   begin
      Sleep_For (Self_ID, Abs_Time - Clock, Timedout);
   end Sleep_Until;

   ---------------
   -- Sleep_For --
   ---------------

   --  For the delay implementation, we need to make sure we achieve
   --  following criterias:
   --  1) We have to delay at least for the amount requested.
   --  2) We have to give up CPU even though the actual delay does not
   --     result in blocking.
   --  3) The implementation has to be efficient so that the delay overhead
   --     is relatively cheap.
   --  1) and 2) are Ada requirements. Even though 2) is an Annex-D
   --     requirement we still want to provide the effect in all cases.
   --     The reason is that users may want to use short delays to implement
   --     their own scheduling effect in the absence of language provided
   --     scheduling policies.

   procedure Sleep_For
     (Self_ID  : Task_ID;
      Rel_Time : Duration;
      Timedout : out Boolean)
   is
      Local_Timedout : Boolean;
      Abs_Time       : Duration;
      Local_Rel      : Duration;
      Result         : Integer;

   begin
      pragma Assert (Self_ID = Self
        or else Shutdown ("GNULLI failure---Self in Sleep_Until"));

      Abs_Time := Clock + Rel_Time;
      if Rel_Time <= 0.0 then
         Timedout := True;
         Yield;
         return;
      end if;

      --  We loop until the requested delay is serviced. For early wakeups,
      --  we check the Clock again and re-request delays until we sleep
      --  at least for the specified amount.

      Local_Rel := Rel_Time;
      loop
         --  Perform delays until one of the following conditions is true:
         --  1) cond_timedwait wakes up due to time expiration.
         --  2) We were interrupted by an abort signal (abortion is pending).
         --  3) We received a wakeup, via cond_signal to our CV.
         --  4) An error has occurred in the OS-provided delay primitive.
         --  Conditions (1), (2), and (3) are normal.
         --  Condition (4) should never happen unless the OS is broken,
         --  or there is an error in our own runtime system code.

         loop
            Cond_Timed_Wait
              (Self_ID.LL.CV'Access, Self_ID.LL.L'Access, Local_Rel,
               Local_Timedout, Result);

            if Result = 0 or else
               (Self_ID.Pending_Action and then
                Self_ID.Pending_ATC_Level < Self_ID.ATC_Nesting_Level)
            then
               Timedout := False;
               return;
            end if;

            if Local_Timedout then
               exit;
            end if;
            Local_Rel := Abs_Time - Clock;
         end loop;

         --  Make sure we delayed long enough. If we did, give up the
         --  CPU. Otherwise, request a delay again with unserviced amount
         --  of time.

         if (Abs_Time <= Clock) then
            Timedout := True;
            Yield;
            exit;
         else
            Local_Rel := Abs_Time - Clock;
         end if;
      end loop;
   end Sleep_For;

   ------------
   -- Wakeup --
   ------------

   procedure Wakeup (T : Task_ID) is
   begin
      Cond_Signal (T.LL.CV'Access);
   end Wakeup;

   -----------
   -- Yield --
   -----------

   procedure Yield is
   begin
      Sleep (0);
   end Yield;

   ------------------
   -- Set_Priority --
   ------------------

   --  Note: Currently, we have only 32 priorities, all in Regular Class.
   --  Priority level 31 is the only value for Interrupt_Priority. (see
   --  package System). A better choice (for Windows NT) would be to have 32
   --  priorities in Regular class for subtype Priority and 32 priorities
   --  in Time-critical class for Interrupt_Priority ???

   procedure Set_Priority (T : Task_ID; Prio : System.Any_Priority) is
      New_Priority : Interfaces.C.int;
      --  Result       : BOOL;
   begin

      case Priority (Prio) is
         when Priority'First .. 4 =>
            New_Priority := Thread_Priority_Idle;
         when 5 .. 9 =>
            New_Priority := Thread_Priority_Lowest;
         when 10 .. Default_Priority - 1 =>
            New_Priority := Thread_Priority_Below_Normal;
         when Default_Priority =>
            New_Priority := Thread_Priority_Normal;
         when Default_Priority + 1 .. 20 =>
            New_Priority := Thread_Priority_Above_Normal;
         when 21 .. 25 =>
            New_Priority := Thread_Priority_Highest;
         when 26 .. Priority'Last =>
            New_Priority := Thread_Priority_Time_Critical;
      end case;

--  FIXME: the priority and priority class should be bumped
      --  Result := SetThreadPriority (T.LL.Thread, New_Priority);

      T.LL.Current_Priority := Interfaces.C.int (Prio);
   end Set_Priority;

   ------------------
   -- Get_Priority --
   ------------------

   function Get_Priority (T : Task_ID) return System.Any_Priority is
   begin
      return System.Any_Priority (T.LL.Current_Priority);
   end Get_Priority;

   ----------------
   -- Enter_Task --
   ----------------

   procedure Enter_Task (Self_ID : Task_ID) is
      procedure Init_Float;
      pragma Import (C, Init_Float, "__gnat_init_float");
      --  Properly initializes the FPU for x86 systems.

      Succeeded : BOOL;
   begin
      Self_ID.LL.Thread := GetCurrentThread;
      Succeeded := TlsSetValue (TlsIndex, To_Address (Self_ID));
      pragma Assert (Succeeded = True
        or else Shutdown ("GNULLI failure---Booster (TlsSetValue)"));
      Init_Float;
   end Enter_Task;

   ----------------------
   --  Initialize_TCB  --
   ----------------------

   procedure Initialize_TCB (Self_ID : Task_ID; Succeeded : out Boolean) is
   begin
      Initialize_Cond (Self_ID.LL.CV'Access);
      Initialize_Lock (Self_ID.LL.L'Access);
      Self_ID.LL.Aborted := False;
      Succeeded := True;
   end Initialize_TCB;

   -----------------
   -- Create_Task --
   -----------------

   procedure Create_Task
     (T          : Task_ID;
      Wrapper    : System.Address;
      Stack_Size : System.Parameters.Size_Type;
      Priority   : System.Any_Priority;
      Succeeded  : out Boolean)
   is
      hTask             : HANDLE;
      TaskId            : aliased DWORD;
      pTaskParameter    : PVOID;
      dwStackSize       : DWORD;
      Result            : DWORD;
      --  Result_Bool       : BOOL;
      Entry_Point       : PTHREAD_START_ROUTINE;

      function To_PTHREAD_START_ROUTINE is new
        Unchecked_Conversion (System.Address, PTHREAD_START_ROUTINE);

   begin

      pTaskParameter := To_Address (T);
      if Stack_Size = System.Parameters.Unspecified_Size then
         dwStackSize := DWORD (Default_Stack_Size);
      else
         dwStackSize    := DWORD (Stack_Size);
      end if;
      Entry_Point    := To_PTHREAD_START_ROUTINE (Wrapper);

      hTask := CreateThread
         (null,
          dwStackSize,
          Entry_Point,
          pTaskParameter,
          DWORD (Create_Suspended),
          TaskId'Unchecked_Access);

      --  Step 1: Create the thread in blocked mode

      if hTask = 0 then
         raise Storage_error;
      end if;

      --  Step 2: set its TCB

      T.LL.Thread := hTask;

      --  Step 3: set its priority (child has inherited priority from parent)

      --  FIXME: the priority and priority class should be bumped
      --  Result_Bool := SetThreadPriority
      --        (hTask, Interfaces.C.int (Priority));

      --  Step 4: Now, start it for good:

      Result := ResumeThread (hTask);
      pragma Assert (Result = 1
        or else Shutdown ("GNULLI failure---Create_Task (ResumeThread)"));

      Succeeded := Result = 1;

   end Create_Task;

   ------------------
   -- Finalize_TCB --
   ------------------

   procedure Finalize_TCB (T : Task_ID) is
      Self_ID  : Task_ID := T;

      procedure Free is new
        Unchecked_Deallocation (Ada_Task_Control_Block, Task_ID);

   begin
      Finalize_Lock (T.LL.L'Access);
      Finalize_Cond (T.LL.CV'Access);
      Free (Self_ID);
   end Finalize_TCB;

   ---------------
   -- Exit_Task --
   ---------------

   procedure Exit_Task is
   begin
      ExitThread (0);
   end Exit_Task;

   ----------------
   -- Abort_Task --
   ----------------

   procedure Abort_Task (T : Task_ID) is
   begin
      T.LL.Aborted := True;
   end Abort_Task;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize (Environment_Task : Task_ID) is
   begin
      Enter_Task (Environment_Task);
   end Initialize;

   -----------
   -- Clock --
   -----------

   function Clock return Duration is
      Systime  : aliased SYSTEMTIME;
      Now      : aliased Long_Long_Integer;

      epoch_1970     : constant := 16#19D_B1DE_D53E_8000#;  -- win32 UTC epoch
      system_time_ns : constant := 100;   -- 100 ns per tick
      Sec_Unit       : constant := 10#1#E9;

   begin
      GetSystemTime (Systime'Access);
      SystemTimeToFileTime (Systime'Address, Now'Address);
      return Duration (Long_Long_Float ((Now - epoch_1970) * system_time_ns) /
                       Long_Long_Float (Sec_Unit));
   exception
      when others =>
         pragma Assert (Shutdown ("exception in Clock"));
         return 0.0;
   end Clock;

begin
   TlsIndex := TlsAlloc;
end System.Task_Primitives.Operations;
