
/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "gladeconfig.h"

#include "gbwidget.h"
#include "glade_menu_editor.h"
#include "editor.h"
#include "palette.h"
#include "property.h"
#include "load.h"
#include "save.h"
#include "utils.h"
#include "project.h"
#include "source.h"
#include "tree.h"

/* Include pixmaps */
#include "graphics/window.xpm"
#include "graphics/dialog.xpm"
#include "graphics/fileseldialog.xpm"
#include "graphics/colorseldialog.xpm"
#include "graphics/fontseldialog.xpm"
#include "graphics/inputdialog.xpm"
#include "graphics/popupmenu.xpm"

enum
  {
    GB_COMPONENT_WINDOW = 0,
    GB_COMPONENT_DIALOG = 1,
    GB_COMPONENT_FILESEL = 2,
    GB_COMPONENT_COLORSEL = 3,
    GB_COMPONENT_FONTSEL = 4,
    GB_COMPONENT_INPUT_DIALOG = 5,
    GB_COMPONENT_POPUPMENU = 6
  };

static GtkWidget *win_main;
static GtkWidget *component_list;
/* This is a list of windows specific to the project, e.g. menu editors.
   These windows are destroyed if another project is opened. */
GList *project_windows = NULL;
static GtkTooltips *tooltips;
static GdkColor tooltips_fgcolor =
{0, 0, 0, 0};
static GdkColor tooltips_bgcolor =
{0, 0xffff, 0xffff, 0};

static GtkWidget *component_pixmaps[7];

static gchar *current_directory = NULL;
static gchar *project_filename = NULL;

/* The position of the main project window */
static gint win_main_x = 0;
static gint win_main_y = 0;

/* These are used to contain the size of the window manager borders around
   our windows, and are used to show/hide windows in the same positions. */
gint windows_x_offset = -1;
gint windows_y_offset = -1;

static void create_main_window (void);
static void on_expose_event (GtkWidget * widget,
			     GdkEvent * event,
			     gpointer data);
static void destroy_window (GtkWidget * widget,
			    gpointer data);

static void on_quit (GtkWidget * widget,
		     gpointer data);

static void on_component_list_select (GtkWidget * clist,
				      gint row,
				      gint column,
				      GdkEventButton * bevent,
				      gpointer data);
static void project_clear (void);
static void new_component (GtkWidget * widget,
			   gpointer data);
static void on_show_button_clicked (GtkWidget * widget,
				    gpointer data);
static gint on_show_button_pressed (GtkWidget * button,
				    GdkEventButton * event,
				    gpointer user_data);
static void show_component (GtkWidget * widget);
static void on_menu_editor_ok (GtkWidget *button,
			       GtkWidget *component);
static void on_menu_editor_apply (GtkWidget *button,
				  GtkWidget *component);
static gint on_menu_editor_close (GtkWidget *menued,
				  gpointer data);
static gint on_key_press_event (GtkWidget * widget,
				GdkEventKey * event,
				gpointer data);
static void on_delete (GtkWidget * widget,
		       gpointer data);

static GtkWidget *new_pixmap (GtkWidget * widget,
			      char **data);

static void on_about (GtkWidget * widget,
		      gpointer data);

static void on_project_new (GtkWidget * widget,
			    gpointer data);
static void on_project_open (GtkWidget * widget,
			     gpointer data);
static void real_open_project (GtkWidget * widget,
			       GtkFileSelection * filesel);
static void on_project_save_as (GtkWidget * widget,
				gpointer data);
static void on_project_save (GtkWidget * widget,
			     gpointer data);
static void on_save_dialog_ok (GtkWidget * widget,
			       GtkFileSelection * filesel);
static void real_save_project (gint warn_before_overwrite);
static void on_project_write_source (GtkWidget * widget,
				     gpointer data);

static void on_source_dialog_ok (GtkWidget * widget,
				 GtkWidget * filesel);

static gchar* get_directory (gchar *filename);


int
main (int argc, char *argv[])
{
#ifdef GLD_HAVE_GTK_1_1
  MSG ("Glade (GTK 1.1)");
#else
  MSG ("Glade (GTK 1.0)");
#endif

  gtk_set_locale ();

  gtk_init (&argc, &argv);

#ifdef ENABLE_NLS
  bindtextdomain (PACKAGE, GLD_LOCALE_DIR);
  textdomain (PACKAGE);
#endif

  gb_widgets_init ();

  create_main_window ();
  gtk_widget_show (win_main);

  editor_init ();

  palette_show (NULL, NULL);
  property_show (NULL, NULL);
  /* Don't show this by default, and also because its not finished. */
  tree_init ();

  gtk_main ();

  return 0;
}


static void
create_main_window ()
{
  GdkColormap *colormap;
  GtkWidget *vbox_main;
  GtkWidget *menubar;
  GtkWidget *menu;
  GtkWidget *menuitem;
  GtkWidget *toolbar;
  GtkWidget *button;
#ifdef GLD_HAVE_GTK_1_1
  GtkAccelGroup *accel_group;
#else
  GtkAcceleratorTable *accelerator_table;
#endif

  win_main = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  /*gtk_widget_set_usize (win_main, 160, 250);*/
  gtk_widget_set_uposition (win_main, win_main_x, win_main_y);

  gtk_signal_connect (GTK_OBJECT (win_main), "destroy",
		      GTK_SIGNAL_FUNC (destroy_window), NULL);
  gtk_signal_connect (GTK_OBJECT (win_main), "delete_event",
		      GTK_SIGNAL_FUNC (on_quit), NULL);
  gtk_signal_connect_after (GTK_OBJECT (win_main), "key_press_event",
			    GTK_SIGNAL_FUNC (on_key_press_event), NULL);
  gtk_signal_connect (GTK_OBJECT (win_main), "expose_event",
		      GTK_SIGNAL_FUNC (on_expose_event), NULL);

  gtk_widget_realize (win_main);

  gtk_window_set_title (GTK_WINDOW (win_main), "Glade");
  gtk_container_border_width (GTK_CONTAINER (win_main), 0);

  tooltips = gtk_tooltips_new ();

  vbox_main = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (win_main), vbox_main);
  gtk_widget_show (vbox_main);

  /* create menu bar */
  menubar = gtk_menu_bar_new ();
  gtk_box_pack_start (GTK_BOX (vbox_main), menubar, FALSE, TRUE, 0);
  gtk_widget_show (menubar);

  /* Create accelerator table */
#ifdef GLD_HAVE_GTK_1_1
  accel_group = gtk_accel_group_get_default ();
#else
  accelerator_table = gtk_accelerator_table_new ();
  gtk_window_add_accelerator_table (GTK_WINDOW (win_main), accelerator_table);
#endif

  /* create File menu */
  menuitem = gtk_menu_item_new_with_label (_("File"));
  gtk_menu_bar_append (GTK_MENU_BAR (menubar), menuitem);
  gtk_widget_show (menuitem);

  menu = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem), menu);

  menuitem = gtk_menu_item_new_with_label (_("New"));
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (on_project_new), NULL);
  gtk_menu_append (GTK_MENU (menu), menuitem);

  gtk_tooltips_set_tip (tooltips, menuitem, _("Create a new project"), NULL);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label (_("Open..."));
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (on_project_open), NULL);
#ifdef GLD_HAVE_GTK_1_1
  gtk_widget_add_accelerator (menuitem, "activate", accel_group,
			      'O', GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
#else
  gtk_widget_install_accelerator (menuitem, accelerator_table,
				  "activate", 'O', GDK_CONTROL_MASK);
#endif
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_tooltips_set_tip (tooltips, menuitem, _("Open an existing project"), NULL);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label (_("Save"));
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (on_project_save), NULL);
#ifdef GLD_HAVE_GTK_1_1
  gtk_widget_add_accelerator (menuitem, "activate", accel_group,
			      'S', GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
#else
  gtk_widget_install_accelerator (menuitem, accelerator_table,
				  "activate", 'S', GDK_CONTROL_MASK);
#endif
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_tooltips_set_tip (tooltips, menuitem, _("Save project"), NULL);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label (_("Save As..."));
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (on_project_save_as), NULL);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_tooltips_set_tip (tooltips, menuitem, _("Save project to a new file"),
			NULL);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new ();
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label (_("Write Source Code..."));
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (on_project_write_source), NULL);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_tooltips_set_tip (tooltips, menuitem, _("Output the project source code"),
			NULL);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new ();
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label (_("Quit"));
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (on_quit), NULL);
#ifdef GLD_HAVE_GTK_1_1
  gtk_widget_add_accelerator (menuitem, "activate", accel_group,
			      'Q', GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
#else
  gtk_widget_install_accelerator (menuitem, accelerator_table,
				  "activate", 'Q', GDK_CONTROL_MASK);
#endif
  gtk_tooltips_set_tip (tooltips, menuitem, _("Quit Glade"), NULL);
  gtk_widget_show (menuitem);


  /* Create Edit menu */
  menuitem = gtk_menu_item_new_with_label (_("Edit"));
  gtk_menu_bar_append (GTK_MENU_BAR (menubar), menuitem);
  gtk_widget_show (menuitem);

  menu = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem), menu);

  menuitem = gtk_menu_item_new_with_label (_("Cut"));
  gtk_menu_append (GTK_MENU (menu), menuitem);
#ifdef GLD_HAVE_GTK_1_1
  gtk_widget_add_accelerator (menuitem, "activate", accel_group,
			      'X', GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
#else
  gtk_widget_install_accelerator (menuitem, accelerator_table,
				  "activate", 'X', GDK_CONTROL_MASK);
#endif
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label (_("Copy"));
  gtk_menu_append (GTK_MENU (menu), menuitem);
#ifdef GLD_HAVE_GTK_1_1
  gtk_widget_add_accelerator (menuitem, "activate", accel_group,
			      'C', GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
#else
  gtk_widget_install_accelerator (menuitem, accelerator_table,
				  "activate", 'C', GDK_CONTROL_MASK);
#endif
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label (_("Paste"));
  gtk_menu_append (GTK_MENU (menu), menuitem);
#ifdef GLD_HAVE_GTK_1_1
  gtk_widget_add_accelerator (menuitem, "activate", accel_group,
			      'V', GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
#else
  gtk_widget_install_accelerator (menuitem, accelerator_table,
				  "activate", 'V', GDK_CONTROL_MASK);
#endif
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label (_("Delete"));
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (on_delete), NULL);
#ifdef GLD_HAVE_GTK_1_1
  gtk_widget_add_accelerator (menuitem, "activate", accel_group,
			      GDK_DELETE, 0, GTK_ACCEL_VISIBLE);
#else
  gtk_widget_install_accelerator (menuitem, accelerator_table,
				  "activate", GDK_DELETE, 0);
#endif
  gtk_widget_show (menuitem);


  /* Create View menu */
  menuitem = gtk_menu_item_new_with_label (_("View"));
  gtk_menu_bar_append (GTK_MENU_BAR (menubar), menuitem);
  gtk_widget_show (menuitem);

  menu = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem), menu);

  menuitem = gtk_menu_item_new_with_label (_("Show Palette"));
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (palette_show),
		      NULL);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_tooltips_set_tip (tooltips, menuitem, _("Show the palette of widgets"),
			NULL);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label (_("Show Properties"));
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (property_show),
		      NULL);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_tooltips_set_tip (tooltips, menuitem,
			_("Show the properties of the selected widget"), NULL);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label (_("Show Widget Tree"));
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (tree_show),
		      NULL);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_tooltips_set_tip (tooltips, menuitem,
			_("Show the project widget tree"), NULL);
  gtk_widget_show (menuitem);

  menuitem = gtk_check_menu_item_new_with_label (_("Show Widget Tooltips"));
  gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM (menuitem),
				 gb_widget_get_show_tooltips ());
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (gb_widget_toggle_show_tooltips),
		      NULL);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_tooltips_set_tip (tooltips, menuitem,
			_("Shows the tooltips of created widgets"), NULL);
  gtk_widget_show (menuitem);


  menuitem = gtk_menu_item_new_with_label (_("Grid"));
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_widget_show (menuitem);

  /* Create Grid submenu */
  menu = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem), menu);

  menuitem = gtk_check_menu_item_new_with_label (_("Show Grid"));
  gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM (menuitem),
				 editor_get_show_grid ());
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (editor_toggle_show_grid),
		      NULL);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_tooltips_set_tip (tooltips, menuitem,
			_("Shows the grid (in fixed containers only)"), NULL);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label (_("Set Grid Options..."));
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (editor_show_grid_settings_dialog),
		      NULL);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_tooltips_set_tip (tooltips, menuitem,
			_("Set the spacing between grid lines"), NULL);
  gtk_widget_show (menuitem);

  menuitem = gtk_check_menu_item_new_with_label (_("Snap to Grid"));
  gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM (menuitem),
				 editor_get_snap_to_grid ());
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (editor_toggle_snap_to_grid),
		      NULL);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_tooltips_set_tip (tooltips, menuitem,
		      _("Snap widgets to the grid (in fixed containers only)"),
			NULL);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label (_("Set Snap Options..."));
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (editor_show_snap_settings_dialog),
		      NULL);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_tooltips_set_tip (tooltips, menuitem,
		      _("Set which parts of a widget snap to the grid"), NULL);
  gtk_widget_show (menuitem);

  /* Create Help menu */
  menuitem = gtk_menu_item_new_with_label (_("Help"));
  gtk_menu_bar_append (GTK_MENU_BAR (menubar), menuitem);
  gtk_widget_show (menuitem);

  menu = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem), menu);

  menuitem = gtk_menu_item_new_with_label (_("Contents"));
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label (_("Index"));
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new ();
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label (_("About..."));
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (on_about),
		      NULL);
  gtk_widget_show (menuitem);


  /* Create component pixmaps */
  component_pixmaps[GB_COMPONENT_WINDOW] = new_pixmap (win_main,
						       window_xpm);
  component_pixmaps[GB_COMPONENT_DIALOG] = new_pixmap (win_main,
						       dialog_xpm);
  component_pixmaps[GB_COMPONENT_FILESEL] = new_pixmap (win_main,
							fileseldialog_xpm);
  component_pixmaps[GB_COMPONENT_COLORSEL] = new_pixmap (win_main,
							 colorseldialog_xpm);
  component_pixmaps[GB_COMPONENT_FONTSEL] = new_pixmap (win_main,
							fontseldialog_xpm);
  component_pixmaps[GB_COMPONENT_INPUT_DIALOG] = new_pixmap (win_main,
							     inputdialog_xpm);
  component_pixmaps[GB_COMPONENT_POPUPMENU] = new_pixmap (win_main,
							  popupmenu_xpm);


  /* Create toolbar */
  toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
  gtk_box_pack_start (GTK_BOX (vbox_main), toolbar, FALSE, TRUE, 0);
  gtk_widget_show (toolbar);

  gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
			   _("New Window"), _("New window"), "",
			   component_pixmaps[GB_COMPONENT_WINDOW],
			   (GtkSignalFunc) new_component,
			   GINT_TO_POINTER (GB_COMPONENT_WINDOW));
  gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
			   _("New Dialog"), _("New dialog"), "",
			   component_pixmaps[GB_COMPONENT_DIALOG],
			   (GtkSignalFunc) new_component,
			   GINT_TO_POINTER (GB_COMPONENT_DIALOG));
  gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
			   _("New File Sel."),
			   _("New file selection dialog"), "",
			   component_pixmaps[GB_COMPONENT_FILESEL],
			   (GtkSignalFunc) new_component,
			   GINT_TO_POINTER (GB_COMPONENT_FILESEL));
  gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
			   _("New Color Sel."),
			   _("New color selection dialog"), "",
			   component_pixmaps[GB_COMPONENT_COLORSEL],
			   (GtkSignalFunc) new_component,
			   GINT_TO_POINTER (GB_COMPONENT_COLORSEL));
#ifdef GLD_HAVE_GTK_1_1
  gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
			   _("New Font Sel."),
			   _("New font selection dialog"), "",
			   component_pixmaps[GB_COMPONENT_FONTSEL],
			   (GtkSignalFunc) new_component,
			   GINT_TO_POINTER (GB_COMPONENT_FONTSEL));
#endif
  gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
			   _("New Input Dialog"),
			   _("New input dialog"), "",
			   component_pixmaps[GB_COMPONENT_INPUT_DIALOG],
			   (GtkSignalFunc) new_component,
			   GINT_TO_POINTER (GB_COMPONENT_INPUT_DIALOG));
  gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
			   _("New Popup Menu"),
			   _("New pop-up menu"), "",
			   component_pixmaps[GB_COMPONENT_POPUPMENU],
			   (GtkSignalFunc) new_component,
			   GINT_TO_POINTER (GB_COMPONENT_POPUPMENU));
  /*gtk_toolbar_append_space (GTK_TOOLBAR (toolbar)); */

  /* Set tooltips colours. */
  colormap = gtk_widget_get_colormap (win_main);
  if (gdk_color_alloc (colormap, &tooltips_fgcolor)
      && gdk_color_alloc (colormap, &tooltips_bgcolor))
    {
      gtk_tooltips_set_colors (tooltips, &tooltips_bgcolor, &tooltips_fgcolor);
      gtk_tooltips_set_colors (GTK_TOOLBAR (toolbar)->tooltips,
			       &tooltips_bgcolor, &tooltips_fgcolor);
    }

  /* Create list of components */
  component_list = gtk_clist_new (1);
  gtk_widget_set_usize (component_list, 160, 120);
  gtk_clist_column_titles_hide (GTK_CLIST (component_list));
  gtk_clist_set_row_height (GTK_CLIST (component_list), 20);
  gtk_clist_set_column_width (GTK_CLIST (component_list), 0, 140);
  gtk_clist_set_policy (GTK_CLIST (component_list), GTK_POLICY_AUTOMATIC,
			GTK_POLICY_AUTOMATIC);
  gtk_signal_connect (GTK_OBJECT (component_list), "select_row",
		      GTK_SIGNAL_FUNC (on_component_list_select), NULL);
  gtk_box_pack_start (GTK_BOX (vbox_main), component_list, TRUE, TRUE, 0);
  gtk_widget_show (component_list);


  /* Create button to view windows */
  button = gtk_button_new_with_label (_("Show"));
  gtk_box_pack_start (GTK_BOX (vbox_main), button, FALSE, FALSE, 5);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_show_button_clicked), NULL);
  gtk_signal_connect (GTK_OBJECT (button), "button_press_event",
		      GTK_SIGNAL_FUNC (on_show_button_pressed), NULL);
}


/* Here we try to figure out the size of the window manager borders around
   our windows, so we can hide & show windows in the same positions easily. */
static void
on_expose_event (GtkWidget * widget, GdkEvent * event, gpointer data)
{
  gint x, y;

  if (windows_x_offset == -1)
    {
      gdk_window_get_origin (widget->window, &x, &y);
      windows_x_offset = x - win_main_x;
      /* Make sure offset seems reasonable. If not, set it to -2 so we don't
         try this again later. */
      if (windows_x_offset < 0 || windows_x_offset > 50)
	windows_x_offset = -2;
      else
	windows_y_offset = y - win_main_y;
    }
}


static void
on_component_list_select (GtkWidget * clist,
			  gint row,
			  gint column,
			  GdkEventButton * bevent,
			  gpointer data)
{
  /* If a component in the clist is double-clicked then show it. */
  if (bevent && bevent->type == GDK_2BUTTON_PRESS)
    on_show_button_clicked (NULL, NULL);
}


static void
on_quit (GtkWidget * widget,
	 gpointer data)
{
  /* Ask to save changes */

  gtk_widget_destroy (win_main);
}


static void
destroy_window (GtkWidget * widget,
		gpointer data)
{
  /* Could the main window be destroyed without the delete event being
     sent? Need to tidy this up a bit. */
  if (widget == win_main)
    gtk_exit (0);
}



/* FIXME: Do we need this? aren't all the pixmaps created by the palette?? */
static GtkWidget *
new_pixmap (GtkWidget * widget,
	    char **data)
{
  GdkPixmap *gdkpixmap;
  GdkBitmap *mask;
  GtkWidget *pixmap;

  gdkpixmap = gdk_pixmap_create_from_xpm_d (widget->window, &mask,
				       &widget->style->bg[GTK_STATE_NORMAL],
					    data);
  pixmap = gtk_pixmap_new (gdkpixmap, mask);
  gdk_pixmap_unref (gdkpixmap);
  gdk_bitmap_unref (mask);
  return pixmap;
}


static void
new_component (GtkWidget * widget,
	       gpointer data)
{
  GtkWidget *component = NULL;
  gchar *name = NULL;
  GdkPixmap *gdkpixmap;
  GdkBitmap *mask;
  gint row;
  gint component_type = GPOINTER_TO_INT (data);

  switch (component_type)
    {
    case GB_COMPONENT_WINDOW:
      component = editor_new_window ();
      break;
    case GB_COMPONENT_DIALOG:
      component = editor_new_dialog ();
      break;
    case GB_COMPONENT_FILESEL:
      component = editor_new_filesel ();
      break;
    case GB_COMPONENT_COLORSEL:
      component = editor_new_colorsel ();
      break;
#ifdef GLD_HAVE_GTK_1_1
    case GB_COMPONENT_FONTSEL:
      component = editor_new_fontsel ();
      break;
#endif
    case GB_COMPONENT_INPUT_DIALOG:
      component = editor_new_input_dialog ();
      break;
    case GB_COMPONENT_POPUPMENU:
      component = editor_new_popupmenu ();
      break;
    default:
      g_warning ("Invalid component\n");
      return;
    }

  name = gtk_widget_get_name (component);
  row = gtk_clist_append (GTK_CLIST (component_list), &name);
  gtk_clist_set_row_data (GTK_CLIST (component_list), row, component);

  gtk_pixmap_get (GTK_PIXMAP (component_pixmaps[component_type]), &gdkpixmap,
		  &mask);
  gtk_clist_set_pixtext (GTK_CLIST (component_list), row, 0, name, 3,
			 gdkpixmap, mask);

  show_component (component);
  gb_widget_show_properties (component);
}


static void
on_show_button_clicked (GtkWidget * widget,
			gpointer data)
{
  GtkWidget *component;
  GList *selection = GTK_CLIST (component_list)->selection;
  if (!selection)
    return;
  component = (GtkWidget*) gtk_clist_get_row_data (GTK_CLIST (component_list),
						   GPOINTER_TO_INT (selection->data));
  g_return_if_fail (component != NULL);
  show_component (component);
}


/* This pops up a popup menu component if the 'Show' button is pressed with the
   right mouse button. */
static gint
on_show_button_pressed (GtkWidget * button, GdkEventButton * event,
			gpointer user_data)
{
  GtkWidget *component;
  GList *selection = GTK_CLIST (component_list)->selection;

  if (event->button != 3)
    return FALSE;

  if (!selection)
    return FALSE;
  component = (GtkWidget*) gtk_clist_get_row_data (GTK_CLIST (component_list),
						   GPOINTER_TO_INT (selection->data));
  if (!GTK_IS_MENU (component))
    return FALSE;

  /*gtk_widget_show (component);*/
  gtk_menu_popup (GTK_MENU (component), NULL, NULL, NULL, NULL,
		  event->button, event->time);
  return TRUE;
}

static void
show_component (GtkWidget * component)
{
  /* Popup menus are shown in the menu editor. */
  if (GTK_IS_MENU (component))
    {
      GladeMenuEditor *menued;

      menued = GLADE_MENU_EDITOR (glade_menu_editor_new ());
      project_add_window (GTK_WIDGET (menued));
      glade_menu_editor_set_menu (menued, GTK_MENU_SHELL (component));
      gtk_signal_connect (GTK_OBJECT (menued->ok_button), "clicked",
			  GTK_SIGNAL_FUNC (on_menu_editor_ok),
			  component);
      gtk_signal_connect (GTK_OBJECT (menued->apply_button), "clicked",
			  GTK_SIGNAL_FUNC (on_menu_editor_apply),
			  component);
      gtk_signal_connect_object (GTK_OBJECT (menued->cancel_button), "clicked",
				 GTK_SIGNAL_FUNC (on_menu_editor_close),
				 GTK_OBJECT (menued));
      gtk_signal_connect_object (GTK_OBJECT (menued), "delete_event",
				 GTK_SIGNAL_FUNC (on_menu_editor_close),
				 GTK_OBJECT (menued));
      gtk_widget_show (GTK_WIDGET (menued));
    }
  else
    {
      gtk_widget_show (component);
      /* This maps the window, which also de-iconifies it according to the
	 ICCCM. */
      gdk_window_show (component->window);
      gdk_window_raise (component->window);
    }
}


static void
on_menu_editor_ok (GtkWidget *button,
		   GtkWidget *component)
{
  GladeMenuEditor *menued;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (button));
  glade_menu_editor_update_menu (menued, GTK_MENU_SHELL (component));
  project_remove_window (GTK_WIDGET (menued));
  gtk_widget_destroy (GTK_WIDGET (menued));
}

static void
on_menu_editor_apply (GtkWidget *button,
		      GtkWidget *component)
{
  GladeMenuEditor *menued;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (button));
  glade_menu_editor_update_menu (menued, GTK_MENU_SHELL (component));
}


static gint
on_menu_editor_close (GtkWidget *menued,
		      gpointer data)
{
  MSG ("In on_menu_editor_close");
  project_remove_window (menued);
  gtk_widget_destroy (menued);
  return TRUE;
}

static gint
on_key_press_event (GtkWidget * widget, GdkEventKey * event, gpointer data)
{
  guint key = event->keyval;
  switch (key)
    {
    case GDK_Delete:
      on_delete (widget, NULL);
      break;
    }
  return TRUE;
}


static void
on_delete (GtkWidget * widget,
	   gpointer data)
{
  GtkWidget *window;
  GList *selection = GTK_CLIST (component_list)->selection;

  if (selection)
    {
      window = gtk_clist_get_row_data (GTK_CLIST (component_list),
				       GPOINTER_TO_INT (selection->data));
      project_delete_component (window);
    }
  else
    {
      editor_on_delete ();
    }
}


static void
project_clear ()
{
  GtkWidget *component;
  GList *tmp_list;
  gint row;

  MSG ("IN project_clear");

  property_set_widget (NULL);
  editor_clear_selection (NULL);
  gb_widget_reset_ids ();

  g_free (project_filename);
  project_filename = NULL;

  /* Close all menu editor windows. */
  tmp_list = project_windows;
  while (tmp_list)
    {
      gtk_widget_destroy (GTK_WIDGET (tmp_list->data));
      tmp_list = tmp_list->next;
    }
  g_list_free (project_windows);
  project_windows = NULL;

  for (row = 0; row < GTK_CLIST (component_list)->rows; row++)
    {
      component = gtk_clist_get_row_data (GTK_CLIST (component_list), row);
      gtk_widget_destroy (component);
      tree_delete_widget (component);
    }
  gtk_clist_clear (GTK_CLIST (component_list));

  /* Delete all GbStyles and reset hash table. Do this after all widgets are
     destroyed. */
  gb_widget_reset_gb_styles ();
}


gboolean
project_add_component (GtkWidget * widget,
		       gchar * name)
{
  gchar *class_name;
  GdkPixmap *gdkpixmap;
  GdkBitmap *mask;
  gint component, row;

  class_name = gtk_type_name (GTK_WIDGET_TYPE (widget));
  if (!strcmp (class_name, "GtkWindow"))
    component = GB_COMPONENT_WINDOW;
  else if (!strcmp (class_name, "GtkDialog"))
    component = GB_COMPONENT_DIALOG;
  else if (!strcmp (class_name, "GtkFileSelection"))
    component = GB_COMPONENT_FILESEL;
  else if (!strcmp (class_name, "GtkColorSelectionDialog"))
    component = GB_COMPONENT_COLORSEL;
  else if (!strcmp (class_name, "GtkFontSelectionDialog"))
    component = GB_COMPONENT_FONTSEL;
  else if (!strcmp (class_name, "GtkInputDialog"))
    component = GB_COMPONENT_INPUT_DIALOG;
  else if (!strcmp (class_name, "GtkMenu"))
    component = GB_COMPONENT_POPUPMENU;
  else
    return FALSE;

  /* Should go in add_signals() ? and only if subtype of a window? */
  gtk_signal_connect (GTK_OBJECT (widget), "delete_event",
		      GTK_SIGNAL_FUNC (editor_close_window), NULL);

  row = gtk_clist_append (GTK_CLIST (component_list), &name);
  gtk_clist_set_row_data (GTK_CLIST (component_list), row, widget);
  gtk_pixmap_get (GTK_PIXMAP (component_pixmaps[component]), &gdkpixmap,
		  &mask);
  gtk_clist_set_pixtext (GTK_CLIST (component_list), row, 0, name, 3,
			 gdkpixmap, mask);

  return TRUE;
}


void
project_rename_component (GtkWidget * widget,
			  gchar * name)
{
  gint row;
  GtkWidget *component;
  gchar *text;
  guint8 spacing;
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  for (row = 0; row < GTK_CLIST (component_list)->rows; row++)
    {
      component = gtk_clist_get_row_data (GTK_CLIST (component_list), row);
      if (component == widget)
	{
	  gtk_clist_get_pixtext (GTK_CLIST (component_list), row, 0, &text,
				 &spacing, &pixmap, &mask);
	  gtk_clist_set_pixtext (GTK_CLIST (component_list), row, 0, name,
				 spacing, pixmap, mask);
	}
    }
}


void
project_delete_component (GtkWidget * widget)
{
  gint row = gtk_clist_find_row_from_data (GTK_CLIST (component_list), widget);
  g_return_if_fail (row != -1);
  gtk_clist_remove (GTK_CLIST (component_list), row);

  /* Set properties widget to NULL, in case the widget or parent is deleted */
  property_set_widget (NULL);

  /* Remove widget from the selection */
  editor_clear_selection (NULL);

  gtk_widget_destroy (widget);
  tree_delete_widget (widget);
}


void
project_foreach_component (GtkCallback callback,
			   gpointer callback_data)
{
  gint row;
  GtkWidget *component;
  for (row = 0; row < GTK_CLIST (component_list)->rows; row++)
    {
      component = gtk_clist_get_row_data (GTK_CLIST (component_list), row);
      if (component)
	(*callback) (component, callback_data);
    }
}


void
project_clear_component_selection ()
{
  GList *selection = GTK_CLIST (component_list)->selection;
  while (selection)
    {
      GList *next = selection->next;
      gtk_clist_unselect_row (GTK_CLIST (component_list),
			      GPOINTER_TO_INT (selection->data), 0);
      selection = next;
    }
}


static void
on_about (GtkWidget * widget,
	  gpointer data)
{
  /* I hope the translations don't overflow the buffer! */
  gchar buf[1024];

  /* VERSION comes from configure.in - the only place it should be defined */
  sprintf (buf,
	   _("G L A D E\n\n"
	     "A GTK+ User Interface Builder\n\n"
	     "Version %s\n\n"
	     "By Damon Chaplin\n\n"
	     "Email: DAChaplin@msn.com\n"
	     "http://www.comp.lancs.ac.uk/~damon/builder/index.html\n"),
	   VERSION);
  show_message_box (buf);
}



/*
 * Loading/Saving projects
 */

static void
on_project_new (GtkWidget * widget,
		gpointer data)
{
  project_clear ();
}


static void
on_project_open (GtkWidget * widget,
		 gpointer data)
{
  GtkWidget *filesel;

  filesel = gtk_file_selection_new (_("Open Project"));
  gtk_window_position (GTK_WINDOW (filesel), GTK_WIN_POS_MOUSE);
  if (current_directory)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel),
				     current_directory);

  gtk_signal_connect_object (GTK_OBJECT (filesel), "delete_event",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (filesel));
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
		   "clicked", GTK_SIGNAL_FUNC (real_open_project), filesel);
  gtk_signal_connect_object (GTK_OBJECT
			     (GTK_FILE_SELECTION (filesel)->cancel_button),
			     "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (filesel));
  gtk_widget_show (filesel);
}


static void
real_open_project (GtkWidget * widget,
		   GtkFileSelection * filesel)
{
  GbStatusCode status;
  gchar *filename;

  /* Close existing project. */
  project_clear ();

  filename = gtk_file_selection_get_filename (filesel);
  g_free (current_directory);
  current_directory = get_directory (filename);

  MSG1 ("Loading: %s", filename);
  status = load_project_file (filename);

  if (status != GB_OK)
    {
      project_clear ();
      MSG2 ("Error %i loading file: %s\n", status, filename);
      show_message_box (_("Error loading file"));
      return;
    }

  project_filename = g_strdup (filename);
  gtk_widget_destroy (GTK_WIDGET (filesel));
}


static void
on_project_save_as (GtkWidget * widget,
		    gpointer data)
{
  GtkWidget *filesel;

  filesel = gtk_file_selection_new (_("Save Project"));
  gtk_window_position (GTK_WINDOW (filesel), GTK_WIN_POS_MOUSE);
  if (current_directory)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel),
				     current_directory);

  gtk_signal_connect_object (GTK_OBJECT (filesel), "delete_event",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (filesel));
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
		   "clicked", GTK_SIGNAL_FUNC (on_save_dialog_ok), filesel);
  gtk_signal_connect_object (GTK_OBJECT
			     (GTK_FILE_SELECTION (filesel)->cancel_button),
			     "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (filesel));
  gtk_widget_show (filesel);
}


static void
on_save_dialog_ok (GtkWidget * widget,
		   GtkFileSelection * filesel)
{
  gchar *filename;

  g_free (project_filename);

  filename = gtk_file_selection_get_filename (filesel);

  g_free (current_directory);
  current_directory = get_directory (filename);

  project_filename = g_strdup (filename);
  real_save_project (TRUE);
  gtk_widget_destroy (GTK_WIDGET (filesel));
}


static void
on_project_save (GtkWidget * widget,
		 gpointer data)
{
  if (project_filename == NULL)
    on_project_save_as (widget, data);
  else
    real_save_project (FALSE);
}


static void
real_save_project (gint warn_before_overwrite)
{
  gint status;

  MSG1 ("Saving: %s\n", project_filename);
  status = save_project_file (project_filename);

  if (status != GB_OK)
    {
      show_message_box (_("Error saving file"));
      MSG1 ("Error saving file:%i", status);
    }
}


static void
on_project_write_source (GtkWidget * widget,
			 gpointer data)
{
  GtkWidget *filesel;

  MSG ("Writing source");

  filesel = gtk_file_selection_new (_("Select source directory"));
  gtk_window_position (GTK_WINDOW (filesel), GTK_WIN_POS_MOUSE);

  gtk_signal_connect_object (GTK_OBJECT (filesel), "delete_event",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (filesel));
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
		      "clicked", GTK_SIGNAL_FUNC (on_source_dialog_ok),
		      filesel);
  gtk_signal_connect_object (GTK_OBJECT
			     (GTK_FILE_SELECTION (filesel)->cancel_button),
			     "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (filesel));
  gtk_widget_show (filesel);
}


static void
on_source_dialog_ok (GtkWidget * widget,
		     GtkWidget * filesel)
{
  gchar *project_directory;
  struct stat filestat;
  GbStatusCode status;

  project_directory = g_strdup (gtk_file_selection_get_filename (GTK_FILE_SELECTION
								 (filesel)));
  gtk_widget_destroy (GTK_WIDGET (filesel));

  if (stat (project_directory, &filestat) != 0)
    {
      if (errno == ENOENT)
	{
	  if (mkdir (project_directory, 0777) != 0)
	    {
	      show_message_box (_("Error creating directory"));
	      return;
	    }
	}
      else
	{
	  show_message_box (_("Error accessing file"));
	  return;
	}
    }

  status = source_write ("project1", project_directory);
  if (status != GB_OK)
    {
      show_message_box (_("Error writing source"));
    }
  g_free (project_directory);
}


static gchar*
get_directory (gchar *filename)
{
  gchar *directory;
  gint i;

  directory = g_strdup (filename);
  for (i = strlen (directory) -1; i >= 0; i--)
    {
      if (directory[i] == '/')
	{
	  directory[i + 1] = '\0';
	  break;
	}
    }
  return directory;
}


void
project_add_window	(GtkWidget     *window)
{
  project_windows = g_list_append (project_windows, window);
}

void
project_remove_window	(GtkWidget     *window)
{
  project_windows = g_list_remove (project_windows, window);
}
