
/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "../gb.h"

/* Include the 21x21 icon pixmap for this widget, to be used in the palette */
#include "../graphics/notebook.xpm"

/*
 * This is the GbWidget struct for this widget (see ../gbwidget.h).
 * It is initialized in the init() function at the end of this file
 */
static GbWidget gbwidget;

/* These are the special children of the widget. */
static gchar *NotebookTab = "Notebook:tab";

static gchar *ShowTabs = "GtkNotebook::show_tabs";
static gchar *ShowBorder = "GtkNotebook::show_border";
static gchar *TabPos = "GtkNotebook::tab_pos";
static gchar *Scrollable = "GtkNotebook::scrollable";
static gchar *TabBorder = "GtkNotebook::tab_border";
static gchar *Popups = "GtkNotebook::popup_enable";

static gchar *GbTabPosChoices[] =
{"Left", "Right", "Top", "Bottom", NULL};
static gint GbTabPosValues[] =
{
  GTK_POS_LEFT,
  GTK_POS_RIGHT,
  GTK_POS_TOP,
  GTK_POS_BOTTOM
};
static gchar *GbTabPosSymbols[] =
{
  "GTK_POS_LEFT",
  "GTK_POS_RIGHT",
  "GTK_POS_TOP",
  "GTK_POS_BOTTOM"
};


static void show_notebook_dialog (GbWidgetNewData * data);
static void on_notebook_dialog_ok (GtkWidget * widget, GbWidgetNewData * data);
static void on_notebook_dialog_destroy (GtkWidget * widget, GbWidgetNewData *
					data);
static GtkWidget *new_label ();
static void gb_notebook_next_page (GtkWidget * menuitem, GtkNotebook * notebook);
static void gb_notebook_prev_page (GtkWidget * menuitem, GtkNotebook * notebook);

/******
 * NOTE: To use these functions you need to uncomment them AND add a pointer
 * to the function in the GbWidget struct at the end of this file.
 ******/

/*
 * Creates a new GtkWidget of class GtkNotebook, performing any specialized
 * initialization needed for the widget to work correctly in this environment.
 * If a dialog box is used to initialize the widget, return NULL from this
 * function, and call data->callback with your new widget when it is done.
 * If the widget needs a special destroy handler, add a signal here.
 */
GtkWidget *
gb_notebook_new (GbWidgetNewData * data)
{
  GtkWidget *new_widget;

  if (data->action == GB_LOADING)
    {
      new_widget = gtk_notebook_new ();
      return new_widget;
    }
  else
    {
      show_notebook_dialog (data);
      return NULL;
    }
}


static void
show_notebook_dialog (GbWidgetNewData * data)
{
  GtkWidget *window, *vbox, *hbox, *separator, *hbbox, *label, *spinbutton;
  GtkWidget *button;
  GtkObject *adjustment;

  window = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);
  gtk_window_set_title (GTK_WINDOW (window), _("New notebook"));

  gtk_signal_connect_object (GTK_OBJECT (window), "delete_event",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (window));
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
		      GTK_SIGNAL_FUNC (on_notebook_dialog_destroy),
		      data);

  vbox = gtk_vbox_new (FALSE, 5);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  /*gtk_container_border_width (GTK_CONTAINER (vbox), 10); */
  gtk_widget_show (vbox);

  hbox = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 5);
  gtk_container_border_width (GTK_CONTAINER (hbox), 10);
  gtk_widget_show (hbox);

  label = gtk_label_new (_("Number of pages:"));
  gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 5);
  gtk_widget_show (label);

  adjustment = gtk_adjustment_new (3, 1, 100, 1, 10, 10);
  spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
  /* save pointer to spinbutton so we can find it easily in the OK handler */
  gtk_object_set_data (GTK_OBJECT (window), "spinbutton", spinbutton);
  gtk_box_pack_start (GTK_BOX (hbox), spinbutton, TRUE, TRUE, 5);
  gtk_widget_set_usize (spinbutton, 50, -1);
  gtk_widget_grab_focus (spinbutton);
  gtk_widget_show (spinbutton);

  separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (vbox), separator, TRUE, TRUE, 5);
  gtk_widget_show (separator);

  hbbox = gtk_hbutton_box_new ();
  gtk_box_pack_start (GTK_BOX (vbox), hbbox, TRUE, TRUE, 5);
  gtk_container_border_width (GTK_CONTAINER (vbox), 10);
  gtk_widget_show (hbbox);

  button = gtk_button_new_with_label (_("OK"));
  gtk_container_add (GTK_CONTAINER (hbbox), button);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_notebook_dialog_ok),
		      data);

  button = gtk_button_new_with_label (_("Cancel"));
  gtk_container_add (GTK_CONTAINER (hbbox), button);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_show (button);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (window));

  gtk_widget_show (window);
  gtk_grab_add (window);
}


static void
on_notebook_dialog_ok (GtkWidget * widget, GbWidgetNewData * data)
{
  GtkWidget *new_widget, *spinbutton, *window;
  gint pages, i;

  window = gtk_widget_get_toplevel (widget);

  /* Only call callback if placeholder/fixed widget is still there */
  if (gb_widget_can_finish_new (data))
    {
      spinbutton = gtk_object_get_data (GTK_OBJECT (window), "spinbutton");
      g_return_if_fail (spinbutton != NULL);
      pages = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spinbutton));

      new_widget = gtk_notebook_new ();
      for (i = 0; i < pages; i++)
	{
	  gtk_notebook_append_page (GTK_NOTEBOOK (new_widget),
				    editor_new_placeholder (),
				    new_label ());
	}
      gb_widget_initialize (new_widget, data);
      (*data->callback) (new_widget, data);
    }
  gtk_widget_destroy (window);
}


static void
on_notebook_dialog_destroy (GtkWidget * widget, GbWidgetNewData * data)
{
  gb_widget_free_new_data (data);
  gtk_grab_remove (widget);
}


GtkWidget *
new_label ()
{
  GtkWidget *label;

  label = gb_widget_new ("GtkLabel", NULL);
  g_return_val_if_fail (label != NULL, NULL);
  gtk_object_set_data (GTK_OBJECT (label), GB_CHILD_NAME_KEY,
		       g_strdup (NotebookTab));
  return label;
}


/*
 * Creates the components needed to edit the extra properties of this widget.
 */
static void
gb_notebook_create_properties (GtkWidget * widget, GbWidgetCreateArgData * data)
{
  property_add_bool (ShowTabs, _("Show Tabs:"), _("If the notebook tabs are shown"));
  property_add_bool (ShowBorder, _("Show Border:"),
		     _("If the notebook border is shown"));
  property_add_choice (TabPos, _("Tab Pos:"),
		       _("The position of the notebook tabs"),
		       GbTabPosChoices);
  property_add_bool (Scrollable, _("Scrollable:"),
		     _("If the notebook tabs are scrollable"));
  property_add_int_range (TabBorder, _("Tab Border:"),
			  _("The size of the notebook tab border"),
			  1, 1000, 1, 10, 1);
  property_add_bool (Popups, _("Show Popup:"), _("If the popup menu is enabled"));
}



/*
 * Gets the properties of the widget. This is used for both displaying the
 * properties in the property editor, and also for saving the properties.
 */
static void
gb_notebook_get_properties (GtkWidget * widget, GbWidgetGetArgData * data)
{
  gint tab_pos, i;

  gb_widget_output_bool (data, ShowTabs, GTK_NOTEBOOK (widget)->show_tabs);
  gb_widget_output_bool (data, ShowBorder, GTK_NOTEBOOK (widget)->show_border);

  tab_pos = GTK_NOTEBOOK (widget)->tab_pos;
  for (i = 0; i < sizeof (GbTabPosValues) / sizeof (GbTabPosValues[0]); i++)
    {
      if (GbTabPosValues[i] == tab_pos)
	gb_widget_output_choice (data, TabPos, i, GbTabPosSymbols[i]);
    }

  gb_widget_output_bool (data, Scrollable, GTK_NOTEBOOK (widget)->scrollable);
  gb_widget_output_int (data, TabBorder, GTK_NOTEBOOK (widget)->tab_border);
  gb_widget_output_bool (data, Popups, (GTK_NOTEBOOK (widget)->menu) ? TRUE :
			 FALSE);
}



/*
 * Sets the properties of the widget. This is used for both applying the
 * properties changed in the property editor, and also for loading.
 */
static void
gb_notebook_set_properties (GtkWidget * widget, GbWidgetSetArgData * data)
{
  gboolean show_tabs, show_border, scrollable, popups;
  gchar *tab_pos;
  gint tab_border, i;

  show_tabs = gb_widget_input_bool (data, ShowTabs);
  if (data->apply)
    gtk_notebook_set_show_tabs (GTK_NOTEBOOK (widget), show_tabs);

  show_border = gb_widget_input_bool (data, ShowBorder);
  if (data->apply)
    gtk_notebook_set_show_border (GTK_NOTEBOOK (widget), show_border);

  tab_pos = gb_widget_input_choice (data, TabPos);
  if (data->apply)
    {
      for (i = 0; i < sizeof (GbTabPosValues) / sizeof (GbTabPosValues[0]); i
	   ++)
	{
	  if (!strcmp (tab_pos, GbTabPosChoices[i])
	      || !strcmp (tab_pos, GbTabPosSymbols[i]))
	    {
	      gtk_notebook_set_tab_pos (GTK_NOTEBOOK (widget), GbTabPosValues
					[i]);
	      break;
	    }
	}
    }

  scrollable = gb_widget_input_bool (data, Scrollable);
  if (data->apply)
    gtk_notebook_set_scrollable (GTK_NOTEBOOK (widget), scrollable);

  tab_border = gb_widget_input_int (data, TabBorder);
  if (data->apply)
    gtk_notebook_set_tab_border (GTK_NOTEBOOK (widget), tab_border);

  popups = gb_widget_input_bool (data, Popups);
  if (data->apply)
    {
      if (popups)
	gtk_notebook_popup_enable (GTK_NOTEBOOK (widget));
      else
	gtk_notebook_popup_disable (GTK_NOTEBOOK (widget));
    }
}



static void
gb_notebook_next_page (GtkWidget * menuitem, GtkNotebook * notebook)
{
  gtk_notebook_next_page (notebook);
}

static void
gb_notebook_prev_page (GtkWidget * menuitem, GtkNotebook * notebook)
{
  gtk_notebook_prev_page (notebook);
}
/*
 * Adds menu items to a context menu which is just about to appear!
 * Add commands to aid in editing a GtkNotebook, with signals pointing to
 * other functions in this file.
 */
static void
gb_notebook_create_popup_menu (GtkWidget * widget, GbWidgetCreateMenuData * data)
{
  GtkWidget *menuitem;
  gint current_page, num_pages;

  current_page = gtk_notebook_current_page (GTK_NOTEBOOK (widget));
  num_pages = g_list_length (GTK_NOTEBOOK (widget)->children);

  menuitem = gtk_menu_item_new_with_label (_("Previous Page"));
  gtk_widget_show (menuitem);
  if (current_page == 0)
    gtk_widget_set_sensitive (menuitem, FALSE);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
	    GTK_SIGNAL_FUNC (gb_notebook_prev_page), GTK_NOTEBOOK (widget));

  menuitem = gtk_menu_item_new_with_label (_("Next Page"));
  gtk_widget_show (menuitem);
  if (current_page == num_pages - 1)
    gtk_widget_set_sensitive (menuitem, FALSE);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
	    GTK_SIGNAL_FUNC (gb_notebook_next_page), GTK_NOTEBOOK (widget));
}



/*
 * Writes the source code needed to create this widget.
 * You have to output everything necessary to create the widget here, though
 * there are some convenience functions to help.
 */
static void
gb_notebook_write_source (GtkWidget * widget, GbWidgetWriteSourceData * data)
{
  gint i;

  if (data->create_widget)
    {
      source_add (data, "  %s = gtk_notebook_new ();\n", data->wname);
    }

  /* We reset the last_child index, so as the tab widgets are written out
     they will start at page 0. */
  gtk_object_set_data (GTK_OBJECT (widget), "last_child", GINT_TO_POINTER (-1));

  gb_widget_write_standard_source (widget, data);

  if (!GTK_NOTEBOOK (widget)->show_tabs)
    source_add (data,
		"  gtk_notebook_set_show_tabs (GTK_NOTEBOOK (%s), FALSE);\n",
		data->wname);
  if (!GTK_NOTEBOOK (widget)->show_border)
    source_add (data,
		"  gtk_notebook_set_show_border (GTK_NOTEBOOK (%s), FALSE);\n",
		data->wname);
  if (GTK_NOTEBOOK (widget)->tab_pos != GTK_POS_TOP)
    {
      for (i = 0; i < sizeof (GbTabPosValues) / sizeof (GbTabPosValues[0]); i
	   ++)
	{
	  if (GbTabPosValues[i] == GTK_NOTEBOOK (widget)->tab_pos)
	    source_add (data,
		    "  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (%s), %s);\n",
			data->wname, GbTabPosSymbols[i]);
	}
    }
  if (GTK_NOTEBOOK (widget)->scrollable)
    source_add (data,
		"  gtk_notebook_set_scrollable (GTK_NOTEBOOK (%s), TRUE);\n",
		data->wname);
  if (GTK_NOTEBOOK (widget)->tab_border != 3)
    source_add (data,
		"  gtk_notebook_set_tab_border (GTK_NOTEBOOK (%s), %i);\n",
		data->wname, GTK_NOTEBOOK (widget)->tab_border);
  if (GTK_NOTEBOOK (widget)->menu)
    source_add (data,
		"  gtk_notebook_popup_enable (GTK_NOTEBOOK (%s));\n",
		data->wname);
}



/*
 * Initializes the GbWidget structure.
 * I've placed this at the end of the file so we don't have to include
 * declarations of all the functions.
 */
GbWidget *
gb_notebook_init ()
{
  /* Initialise the GTK type */
  gtk_notebook_get_type ();

  /* Initialize the GbWidget structure */
  gb_widget_init_struct (&gbwidget);

  /* Fill in the pixmap struct & tooltip */
  gbwidget.pixmap_struct = notebook_xpm;
  gbwidget.tooltip = _("Notebook");

  /* Fill in any functions that this GbWidget has */
  gbwidget.gb_widget_new = gb_notebook_new;
  gbwidget.gb_widget_create_properties = gb_notebook_create_properties;
  gbwidget.gb_widget_get_properties = gb_notebook_get_properties;
  gbwidget.gb_widget_set_properties = gb_notebook_set_properties;
  gbwidget.gb_widget_write_source = gb_notebook_write_source;
  gbwidget.gb_widget_create_popup_menu = gb_notebook_create_popup_menu;

  return &gbwidget;
}
