/*
Copyright (C) 1998-99 Free Software Foundation, Inc.

Author: Frank Heckenbach <frank@pascal.gnu.de>

Support routines for regex.pas

This library is free software; you can redistribute it and/or modify
it under the terms of the GNU Library General Public License as published by
the Free Software Foundation, version 2.

This library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
License for more details.

You should have received a copy of the GNU Library General Public License
along with this library; see the file COPYING.LIB.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
MA 02111-1307, USA.
*/

#include <stdlib.h>
#include <rxposix.h>

typedef unsigned char Boolean;

typedef struct
{
  regex_t    *RegEx;
  regmatch_t *RegMatch;
  int         From, SubExpressions;
  char       *Error;
} RegExType;

void _p_error_regex (RegExType *r, int errcode);
void _p_error_regex (RegExType *r, int errcode)
{
  if (r -> Error)
    {
      free (r -> Error);
      r -> Error = 0;
    }
  if (errcode)
    {
      size_t size = regerror (errcode, r -> RegEx, 0, 0);
      r -> Error = (char *) malloc (size);
      regerror (errcode, r -> RegEx, r -> Error, size);
    }
}

void _p_new_regex (RegExType *r, const char *Expression, int ExpressionLength,
                   Boolean Extended, Boolean CaseInsensitive,
                   Boolean NoSubExpressions, Boolean NewLines);
void _p_new_regex (RegExType *r, const char *Expression, int ExpressionLength,
                   Boolean Extended, Boolean CaseInsensitive,
                   Boolean NoSubExpressions, Boolean NewLines)
{
  int result;
  r -> Error = 0;
  r -> From = 1;
  r -> SubExpressions = 0;
  r -> RegMatch = 0;
  r -> RegEx = (regex_t *) malloc (sizeof (regex_t));
  result = regncomp (r -> RegEx, Expression, ExpressionLength,
            (Extended         ? REG_EXTENDED : 0) |
            (CaseInsensitive  ? REG_ICASE    : 0) |
            (NoSubExpressions ? REG_NOSUB    : 0) |
            (NewLines         ? REG_NEWLINE  : 0));
  if (result)
    {
      _p_error_regex (r, result);
      free (r -> RegEx);
      r -> RegEx = 0;
    }
  else
    {
      r -> SubExpressions = r -> RegEx -> re_nsub - 1;
      r -> RegMatch = (regmatch_t *) malloc ((r -> SubExpressions + 1) * sizeof (regmatch_t));
    }
}

void _p_dispose_regex (RegExType *r);
void _p_dispose_regex (RegExType *r)
{
  if (r -> RegEx)
    {
      regfree (r -> RegEx);
      free (r -> RegEx);
      r -> RegEx = 0;
    }
  if (r -> Error)
    {
      free (r -> Error);
      r -> Error = 0;
    }
  if (r -> RegMatch)
    {
      free (r -> RegMatch);
      r -> RegMatch = 0;
      r -> SubExpressions = -1;
    }
}

Boolean _p_match_regex_from (RegExType *r, const char *Str, int StrLength,
                             Boolean NotBeginningOfLine, Boolean NotEndOfLine, int From);
Boolean _p_match_regex_from (RegExType *r, const char *Str, int StrLength,
                             Boolean NotBeginningOfLine, Boolean NotEndOfLine, int From)
{
  int i = StrLength - From + 1;
  r -> From = From;
  return !regnexec (r -> RegEx, Str + From - 1, (i >= 0) ? i : 0, r -> SubExpressions + 1,
                    &r -> RegMatch, (NotBeginningOfLine ? REG_NOTBOL : 0) |
                    (NotEndOfLine ? REG_NOTEOL : 0));
}

void _p_getmatch_regex (RegExType *r, int n, int *MatchPosition, int *MatchLength);
void _p_getmatch_regex (RegExType *r, int n, int *MatchPosition, int *MatchLength)
{
  if (n < 0 || n > r -> SubExpressions)
    *MatchPosition = *MatchLength = 0;
  else
    {
      *MatchPosition = r -> RegMatch [n].rm_so;
      *MatchLength   = r -> RegMatch [n].rm_eo;
      if (*MatchPosition >= 0 && *MatchLength >= 0)
        {
          *MatchLength -= *MatchPosition;
          *MatchPosition += r -> From;
        }
      else
        *MatchPosition = *MatchLength = 0;
    }
}
