{
Copyright (C) 1998-99 Free Software Foundation, Inc.

Author: Frank Heckenbach <frank@pascal.gnu.de>

Pascal declarations of the GPC RTS that are visible to each program.

This unit contains Pascal declarations of many RTS routines which can
be called from programs. Some other declarations that should not
normally be called from programs, are contained in the unit `Internal'.
Some of these routines correspond to functions built into the compiler,
such as most of the file operations; the declarations here should only
be used if the built-in functions are not applicable for some reason.
Other routines, like many file name and string routines, are not built
in, so the declarations from this unit can be used. Don't copy the
declarations from this unit into your programs, but rather include
this unit with a `uses' statement. The reason is that the internal
declarations, e.g. the `asmnames', may change, and this unit will be
changed accordingly. @@In the future, this unit might be included into
every program automatically, so there will be no need for a `uses'
statement to make the declarations here available.

Note about `protected var' parameters:
Since const parameters in GPC can be passed by value OR by reference
internally, possibly depending on the system, `const foo*' parameters
to C functions *cannot* reliably declared as `const' in Pascal.
However, Extended Pascal's `protected var' can be used since this
guarantees passing by reference.

Note about the `GPC_' prefix:
This is inserted so that some identifiers don't conflict with the
built-in ones. Sometimes, the built-in ones do exactly the same as
the ones declared here, but often enough, they contain some "magic",
so they should be used instead of the plain declarations here.
In general, routines with a `GPC_' prefix should not be called from
programs. They are subject to change or disappering in future GPC
versions.

This file is part of GNU Pascal Library. The GNU Pascal Library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any
later version.

The GNU Pascal Library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with the GNU Pascal Library; see the file COPYING.LIB.  If
not, write to the Free Software Foundation, Inc., 675 Mass Ave,
Cambridge, MA 02139, USA.
}

unit GPC;

interface

type
  AnyFile = Text; (*@@ create `AnyFile' parameters*)

const
  MaxLongInt = High (LongInt);

  { Maximum size of a variable }
  MaxVarSize = MaxInt;

{ ====================== RUNTIME ERROR HANDLING ETC. ====================== }

type
  PCStrings = ^TCStrings;
  TCStrings = array [0 .. MaxVarSize] of CString;

var
  ProcessID   : asmname '_p_pid'  Integer;
  CParamCount : asmname '_p_argc' Integer;
  CParameters : asmname '_p_argv' PCStrings;
  Environment : asmname '_p_envp' PCStrings;

{ Error handling functions, from error.pas }

var
  { BP compatible InOutRes variable }
  GPC_InOutRes : asmname '_p_inoutres' Integer;

  { CString parameter to some error messages, NOT the text of the error
    message (this can be obtained with GetErrorMessage) }
  GPC_InOutResStr : asmname '_p_inoutres_str' CString;

  { Error number (after runtime error) or exit status (after Halt) or
    0 (during program run and after succesful termination). }
  ExitCode : asmname '_p_exitcode' Integer;

  { Non-nil after runtime error, nil otherwise. Does not give the actual
    address of the code where the error occurred. }
  ErrorAddr : asmname '_p_erroraddr' Pointer;

function  GetErrorMessage                 (n : Integer) : CString;                   asmname '_p_errmsg';
function  GetCheckErrorMessage            (n : Integer; Format : CString) : CString; asmname '_p_check_errmsg';
procedure StartRuntimeMessage;                                                       asmname '_p_start_message';
procedure StartInternalError;                                                        asmname '_p_start_internal_error';
procedure EndRuntimeError                 (n : Integer);                             asmname '_p_end_error';
function  StartRuntimeWarning : Boolean;                                             asmname '_p_start_warning';
procedure EndRuntimeWarning;                                                         asmname '_p_end_warning';
procedure RuntimeError                    (n : Integer);                             asmname '_p_error';
procedure RuntimeErrorInteger             (n : Integer; i : MedInt);                 asmname '_p_error_integer';
procedure RuntimeErrorCString             (n : Integer; s : CString);                asmname '_p_error_string';
procedure InternalError                   (n : Integer);                             asmname '_p_internal_error';
procedure InternalErrorInteger            (n : Integer; i : MedInt);                 asmname '_p_internal_error_integer';
procedure RuntimeWarning                  (Message : CString);                       asmname '_p_warning';
procedure RuntimeWarningInteger           (Message : CString; i : MedInt);           asmname '_p_warning_integer';
procedure RuntimeWarningCString           (Message : CString; s : CString);          asmname '_p_warning_string';

{ Internal routine for `RunError' calls from the program }
procedure GPC_RunError                    (n : Integer);                             asmname '_p_runerror';

procedure StartTempIOError;                                                          asmname '_p_start_temp_io_error';
function  EndTempIOError : Integer;                                                  asmname '_p_end_temp_io_error';
procedure IOError                         (n : Integer);                             asmname '_p_io_error';
procedure IOErrorCString                  (n : Integer; s : CString);                asmname '_p_io_error_string';
function  GPC_IOResult : Integer;                                                    asmname '_p_ioresult';
function  GetIOErrorMessage : CString;                                               asmname '_p_get_io_error_message';
procedure CheckInOutRes;                                                             asmname '_p_check_inoutres';
procedure GPC_Halt (aExitCode : Integer);                                            asmname '_p_halt';

{ For GNU malloc }
procedure HeapWarning                     (s : CString);                             asmname '_p_heap_warning';

{ For signal handlers }
procedure PrintMessage (Message : CString; n : Integer; Warning : Boolean);          asmname '_p_prmessage';

{ =========================== MEMORY MANAGEMENT =========================== }

{ `MoveLeft', `MoveRight' and `Move' transfer procedures, from move.pas }

procedure GPC_MoveLeft  (const Source; var Dest; Count : SizeType); asmname '_p_moveleft';
procedure GPC_MoveRight (const Source; var Dest; Count : SizeType); asmname '_p_moveright';
procedure GPC_Move      (const Source; var Dest; Count : SizeType); asmname '_p_move';

function  MemCmp  (const s1, s2; Size : SizeType) : Integer; asmname 'memcmp';
function  MemComp (const s1, s2; Size : SizeType) : Integer; asmname 'memcmp';

{ Heap manipulation, from heap.c }

{ GPC implements both Mark/Release and Dispose. Both can be mixed freely
  in the same program. Dispose should be preferred, since it's faster. }

type
  GetMemType  = ^function (Size : SizeType) : Pointer;
  FreeMemType = ^procedure (aPointer : Pointer);

var
  GetMemPtr  : asmname '_p_getmem_ptr'  GetMemType;
  FreeMemPtr : asmname '_p_freemem_ptr' FreeMemType;

{ GetMemPtr and FreeMemPtr point to these routines by default }
function  Default_GetMem  (Size : SizeType) : Pointer; asmname '_p_default_malloc';
procedure Default_FreeMem (aPointer : Pointer);        asmname 'free';

{ GPC_GetMem and GPC_FreeMem call the actual routines through GetMemPtr and FreeMemPtr }
function  GPC_GetMem  (Size : SizeType) : Pointer;     asmname '_p_malloc';
procedure GPC_FreeMem (aPointer : Pointer);            asmname '_p_free';

{ GPC_New and GPC_Dispose call GPC_GetMem and GPC_FreeMem, but also do the
  stuff necessary for Mark and Release. Therefore, GPC_GetMem and GPC_FreeMem
  should not be called directly when using Mark and Release. GetMem and
  FreeMem in a Pascal program will call GPC_New and GPC_Dispose internally,
  not GPC_GetMem and GPC_FreeMem. }
function  GPC_New     (Size : SizeType) : Pointer;     asmname '_p_new';
procedure GPC_Dispose (aPointer : Pointer);            asmname '_p_dispose';

{ ReleaseF will return the number of pointers released. While Release
  discards this information, it can be used e.g. during debugging to
  verify that all allocated memory blocks are correctly disposed, and
  no memory leaks exist. }
procedure GPC_Mark    (var aPointer : Pointer);        asmname '_p_mark';
function  ReleaseFunc (aPointer : Pointer) : Integer;  asmname '_p_release';

{ C heap management routines. NOTE: when using Release, CFreeMem must not
  be used but for pointers allocated by CGetMem. }
function  CGetMem     (Size : SizeType) : Pointer;                     asmname 'malloc';
procedure CFreeMem    (aPointer : Pointer);                            asmname 'free';
function  ReAlloc     (aPointer : Pointer; Size : SizeType) : Pointer; asmname 'realloc';

{ Routines to handle endianness, from endian.pas }

{ Boolean constants about endianness and alignment }

const
  Bits_Big_Endian  = {$ifdef __BITS_LITTLE_ENDIAN__} False
                     {$else}{$ifdef __BITS_BIG_ENDIAN__} True
                     {$else}{$error Bit endianness is not defined!}
                     {$endif}{$endif};

  Bytes_Big_Endian = {$ifdef __BYTES_LITTLE_ENDIAN__} False
                     {$else}{$ifdef __BYTES_BIG_ENDIAN__} True
                     {$else}{$error Byte endianness is not defined!}
                     {$endif}{$endif};

  Words_Big_Endian = {$ifdef __WORDS_LITTLE_ENDIAN__} False
                     {$else}{$ifdef __WORDS_BIG_ENDIAN__} True
                     {$else}{$error Word endianness is not defined!}
                     {$endif}{$endif};

  Need_Alignment   = {$ifdef __NEED_ALIGNMENT__} True
                     {$else} False {$endif};

{ Convert single variables from or to little or big endian format.
  For structures, this has to be done for each component separately!
  Currently, ConvertFromFooEndian and ConvertToFooEndian are the same, but
  this might not be the case on middle-endian machines. So it's better to
  provide and use different names. }
procedure ReverseBytes            (var Buf; Count : SizeType);                                asmname '_p_reversebytes';
procedure ConvertFromLittleEndian (var Buf; Count : SizeType);                                asmname '_p_convertlittleendian';
procedure ConvertFromBigEndian    (var Buf; Count : SizeType);                                asmname '_p_convertbigendian';
procedure ConvertToLittleEndian   (var Buf; Count : SizeType);                                asmname '_p_convertlittleendian';
procedure ConvertToBigEndian      (var Buf; Count : SizeType);                                asmname '_p_convertbigendian';

{ Combine the conversion with file block routines }
function  BlockReadLittleEndian   (var aFile : File; var   Buf; Count : SizeType) : SizeType; asmname '_p_blockread_littleendian';
function  BlockReadBigEndian      (var aFile : File; var   Buf; Count : SizeType) : SizeType; asmname '_p_blockread_bigendian';
function  BlockWriteLittleEndian  (var aFile : File; const Buf; Count : SizeType) : SizeType; asmname '_p_blockwrite_littleendian';
function  BlockWriteBigEndian     (var aFile : File; const Buf; Count : SizeType) : SizeType; asmname '_p_blockwrite_bigendian';

{ ======================== STRING HANDLING ROUTINES ======================= }

{ String handling routines, from gstrings.pas }

type
  TChars = packed array [1 .. 1] of Char;
  PChars = ^TChars;

  GPC_PString = ^GPC_String;
  GPC_String (Capacity : Cardinal) = record
    Length : 0 .. Capacity;
    Chars  : packed array [1 .. Capacity + 1] of Char
  end;

  StringObject = object { Abstract }
    function  GetCapacity : Integer;           virtual;
    procedure SetLength (NewLength : Integer); virtual;
    function  GetLength : Integer;             virtual;
    function  GetFirstChar : PChars;           virtual;
  end;

  (*@@ AnyString parameters are not yet implemented, the following is only a draft*)
  AnyStringType = (AnyStringLong,
                   AnyStringUndiscriminated,
                   AnyStringShort,
                   AnyStringFixed,
                   AnyStringCString,
                   AnyStringObject);

  (*@@ only formally for now*)UndiscriminatedString = ^String;

  { When a const or var AnyString parameter is passed, internally these
    records are passed as const parameters. Value AnyString parameters are
    passed like value string parameters. }
  ConstAnyString = record
    Length : Integer;
    Chars  : PChars
  end;

  VarAnyString = record
    Capacity : Integer;
    Chars    : PChars;
    Truncate : Boolean;
  case StringType            : AnyStringType of
    AnyStringLong            : (PLongLength            : ^Integer);
    AnyStringUndiscriminated : (PUndiscriminatedString : ^UndiscriminatedString);
    AnyStringShort           : (PShortLength           : ^Byte);
    AnyStringFixed           : (CurrentLength          : Integer);
    AnyStringObject          : (PStringObject          : ^StringObject)
  end;

{ TString is a string type that is used for function results and local
  variables, as long as undiscriminated strings are not allowed there.
  The default size of 2048 characters should be enough for file names
  on any system, but can be changed when necessary. This should be at
  least as big as MAXPATHLEN. }

const
  TStringSize = 2048;
  SpaceCharacters = [' ', #9];

type
  TString    = String (TStringSize);
  TStringBuf = packed array [0 .. TStringSize] of Char;
  PString    = ^String;
  CharSet    = set of Char;

function  GPC_UpCase      (ch : Char) : Char;                                    asmname '_p_gpc_upcase';
function  GPC_LoCase      (ch : Char) : Char;                                    asmname '_p_gpc_locase';
function  BP_UpCase       (ch : Char) : Char;                                    asmname '_p_bp_upcase';
function  BP_LoCase       (ch : Char) : Char;                                    asmname '_p_bp_locase';

procedure UpCaseString    (var s : String);                                      asmname '_p_upcase_string';
procedure LoCaseString    (var s : String);                                      asmname '_p_locase_string';
function  UpCaseStr       (const s : String) : TString;                          asmname '_p_upcase_str';
function  LoCaseStr       (const s : String) : TString;                          asmname '_p_locase_str';

function  Pos             (const SubStr, Str : String) : Integer;                asmname '_p_pos';
function  LastPos         (const SubStr, Str : String) : Integer;                asmname '_p_lastpos';
function  CharPos         (const Chars : CharSet; const Str : String) : Integer; asmname '_p_charpos';
function  LastCharPos     (const Chars : CharSet; const Str : String) : Integer; asmname '_p_lastcharpos';

function  PosFrom         (const SubStr, Str : String; From : Integer) : Integer;                asmname '_p_posfrom';
function  LastPosTill     (const SubStr, Str : String; Till : Integer) : Integer;                asmname '_p_lastpostill';
function  CharPosFrom     (const Chars : CharSet; const Str : String; From : Integer) : Integer; asmname '_p_charposfrom';
function  LastCharPosTill (const Chars : CharSet; const Str : String; Till : Integer) : Integer; asmname '_p_lastcharpostill';

function  IsPrefix        (const Prefix, s : String) : Boolean;                  asmname '_p_isprefix';
function  IsSuffix        (const Suffix, s : String) : Boolean;                  asmname '_p_issuffix';

function  StrLen          (Src : CString) : SizeType;                            asmname '_p_strlen';
function  StrEnd          (Src : CString) : CString;                             asmname '_p_strend';
function  StrDup          (Src : CString) : CString;                             asmname '_p_strdup';
function  StrNew          (Src : CString) : CString;                             asmname '_p_strdup';
procedure StrDispose      (s : CString);                                         asmname '_p_dispose';
function  StrCmp          (s1, s2 : CString) : Integer;                          asmname '_p_strcmp';
function  StrComp         (s1, s2 : CString) : Integer;                          asmname '_p_strcmp';
function  StrLCmp         (s1, s2 : CString; MaxLen : SizeType) : Integer;       asmname '_p_strlcmp';
function  StrLComp        (s1, s2 : CString; MaxLen : SizeType) : Integer;       asmname '_p_strlcmp';
function  StrCaseCmp      (s1, s2 : CString) : Integer;                          asmname '_p_strcasecmp';
function  StrIComp        (s1, s2 : CString) : Integer;                          asmname '_p_strcasecmp';
function  StrLCaseCmp     (s1, s2 : CString; MaxLen : SizeType) : Integer;       asmname '_p_strlcasecmp';
function  StrLIComp       (s1, s2 : CString; MaxLen : SizeType) : Integer;       asmname '_p_strlcasecmp';
function  StrCpy          (Dest, Source : CString) : CString;                    asmname '_p_strcpy';
function  StrCopy         (Dest, Source : CString) : CString;                    asmname '_p_strcpy';
function  StrECpy         (Dest, Source : CString) : CString;                    asmname '_p_strecpy';
function  StrECopy        (Dest, Source : CString) : CString;                    asmname '_p_strecpy';
function  StrLCpy         (Dest, Source : CString; MaxLen : SizeType) : CString; asmname '_p_strlcpy';
function  StrLCopy        (Dest, Source : CString; MaxLen : SizeType) : CString; asmname '_p_strlcpy';
function  StrMove         (Dest, Source : CString; Count : SizeType) : CString;  asmname '_p_strmove';
function  StrCat          (Dest, Source : CString) : CString;                    asmname '_p_strcat';
function  StrLCat         (Dest, Source : CString; MaxLen : SizeType) : CString; asmname '_p_strlcat';
function  StrScan         (Src : CString; Ch : Char) : CString;                  asmname '_p_strscan';
function  StrRScan        (Src : CString; Ch : Char) : CString;                  asmname '_p_strrscan';
function  StrPos          (Str, SubStr : CString) : CString;                     asmname '_p_strpos';
function  StrRPos         (Str, SubStr : CString) : CString;                     asmname '_p_strrpos';
function  StrCasePos      (Str, SubStr : CString) : CString;                     asmname '_p_strcasepos';
function  StrRCasePos     (Str, SubStr : CString) : CString;                     asmname '_p_strrcasepos';
function  StrUpper        (s : CString) : CString;                               asmname '_p_strupper';
function  StrLower        (s : CString) : CString;                               asmname '_p_strlower';
function  StrEmpty        (s : CString) : Boolean;                               asmname '_p_strempty';

function  NewCString      (const Source : String) : CString;                     asmname '_p_newcstring';
function  StrPCopy        (Dest : CString; const Source : String) : CString;     asmname '_p_strpcopy';
procedure StrCCopy        (Source : CString; var Dest : String);                 asmname '_p_strccopy';

function  NewString       (const s : String) : PString;                          asmname '_p_newstring';
procedure DisposeString   (p : PString);                                         asmname '_p_dispose';

procedure SetString       (var s : String; Buffer : PChar; Count : Integer);     asmname '_p_set_string';
function  StringOfChar    (Ch : Char; Count : Integer) = s : TString;            asmname '_p_string_of_char';

procedure TrimLeft        (var s : String);                                      asmname '_p_trimleft';
procedure TrimRight       (var s : String);                                      asmname '_p_trimright';
procedure TrimBoth        (var s : String);                                      asmname '_p_trimboth';
function  TrimLeftStr     (const s : String) : TString;                          asmname '_p_trimleft_str';
function  TrimRightStr    (const s : String) : TString;                          asmname '_p_trimright_str';
function  TrimBothStr     (const s : String) : TString;                          asmname '_p_trimboth_str';

{ Internal routines for the string operators }

{ Compare strings for equality without padding }
function  StrEQ (s1 : PChar; Length1 : Integer; s2 : PChar; Length2 : Integer) : Boolean;    asmname '_p_eq';

{ Compare strings for `less-than' without padding }
function  StrLT (s1 : PChar; Length1 : Integer; s2 : PChar; Length2 : Integer) : Boolean;    asmname '_p_lt';

{ Compare strings for equality, padding the shorter string with spaces }
function  StrEQPad (s1 : PChar; Length1 : Integer; s2 : PChar; Length2 : Integer) : Boolean; asmname '_p_str_eq';

{ Compare strings for `less-than', padding the shorter string with spaces }
function  StrLTPad (s1 : PChar; Length1 : Integer; s2 : PChar; Length2 : Integer) : Boolean; asmname '_p_str_lt';

{ Internal routine for Index/Pos }
function  StrIndex (s1 : PChar; Length1 : Integer; s2 : PChar; Length2 : Integer) : Integer; asmname '_p_index';

{ Internal routine for Trim }
function  StrTrim (Src : PChar; SrcLength : Integer; Dest : PChar) : Integer; asmname '_p_trim';

{ Internal routine for SubStr/Copy }
function  SubStr (Src : PChar; SrcLength, Position, Count : Integer; Dest : PChar; Truncate : Boolean) : Integer; asmname '_p_substr';

function  GetStringCapacity (const s : String) : Integer;                        asmname '_p_get_string_capacity';

procedure GPC_Insert      (const Source: String; var Dest : String;
                           Index : Integer; Truncate : Boolean);                 asmname '_p_insert';
procedure GPC_Delete      (var s : String; Index, Count : Integer);              asmname '_p_delete';

{ Under development }
function VarAnyStringLength (var s : VarAnyString) : Integer; asmname '_p_var_anystring_length';
function VarAnyStringSetLength (var s : VarAnyString; NewLength : Integer) : Integer; asmname '_p_var_anystring_setlength';
procedure AnyStringTFDD_Reset (var f : AnyFile; var Buf : ConstAnyString); asmname '_p_anystring_tfdd_reset';
procedure AnyStringTFDD_Rewrite (var f : AnyFile; var Buf : VarAnyString); asmname '_p_anystring_tfdd_rewrite';
procedure StringTFDD_Reset (var f : AnyFile; var Buf : ConstAnyString; const s : String); asmname '_p_string_tfdd_reset';
procedure StringTFDD_Rewrite (var f : AnyFile; var Buf : VarAnyString; var s : String); asmname '_p_string_tfdd_rewrite';

{ ========================= FILE HANDLING ROUTINES ======================== }

{ Extended pascal binding routines, from bind.c }

const { from types.h }
  Binding_Name_Length = 255;

type
  CFilePtr = Pointer;

  UnixTimeType = LongInt; { This is hard-coded in the compiler. Do not change here. }

  { The standard fields are Bound and Name. The others are extensions. }
  GPC_BindingType = {@@packed} record
                      Bound             : Boolean;
                      Force             : Boolean;      { Can be set to allow binding to non-writable files or directories }
                      Extensions_Valid  : Boolean;
                      Readable          : Boolean;
                      Writable          : Boolean;
                      Executable        : Boolean;
                      Existing          : Boolean;      { Binding points to an existing file }
                      Directory         : Boolean;      { Binding points to an existing directory; Existing is False then }
                      Size              : LongInt;      { Number of elements or -1 }
                      AccessTime,                       { Time of last access }
                      ModificationTime,                 { Time of last modification }
                      ChangeTime        : UnixTimeType; { Time of last change }
                      Error             : Integer;      { Unused currently }
                      CFile             : CFilePtr;     { allows binding a Pascal file to a C file }
                      Name              : String (Binding_Name_Length)
                    end;

procedure GPC_Bind         (          var aFile : AnyFile; protected var aBinding : BindingType); asmname '_p_bind';
procedure GetBinding       (protected var aFile : AnyFile;           var aBinding : BindingType); asmname '_p_binding';
procedure GPC_Unbind       (          var aFile : AnyFile);                                       asmname '_p_unbind';
procedure ClearBinding     (          var aBinding : BindingType);                                asmname '_p_clearbinding';

{ TFDD interface, subject to change! }
type
  TOpenMode  = (foReset, foRewrite, foAppend, foSeekRead, foSeekWrite, foSeekUpdate);
  TOpenProc  = procedure (var PrivateData; Mode : TOpenMode);
  TReadFunc  = function  (var PrivateData; var   Buffer; Size : SizeType) : SizeType;
  TWriteFunc = function  (var PrivateData; const Buffer; Size : SizeType) : SizeType;
  TFileProc  = procedure (var PrivateData);
  TFlushProc = TFileProc;
  TCloseProc = TFileProc;
  TDoneProc  = TFileProc;

procedure AssignTFDD (var f : AnyFile;
                      OpenProc    : TOpenProc;
                      ReadFunc    : TReadFunc;
                      WriteFunc   : TWriteFunc;
                      FlushProc   : TFlushProc;
                      CloseProc   : TCloseProc;
                      DoneProc    : TDoneProc;
                      PrivateData : Pointer);       asmname '_p_assign_tfdd';

procedure GetTFDD    (var f : AnyFile;
                      var OpenProc    : TOpenProc;
                      var ReadFunc    : TReadFunc;
                      var WriteFunc   : TWriteFunc;
                      var FlushProc   : TFlushProc;
                      var CloseProc   : TCloseProc;
                      var DoneProc    : TDoneProc;
                      var PrivateData : Pointer);   asmname '_p_get_tfdd';

{ Default TFDD routines for files. @@ OpenProc and CloseProc still missing. }
function  F_Read     (var aFile; var   Buffer; Size : SizeType) : SizeType; asmname '_p_f_read';
function  F_Read_TTY (var aFile; var   Buffer; Size : SizeType) : SizeType; asmname '_p_f_read_tty';
function  F_Write    (var aFile; const Buffer; Size : SizeType) : SizeType; asmname '_p_f_write';
procedure F_Flush    (var aFile);                                           asmname '_p_f_flush';
const     F_Done : (*@@fjf258b*)TDoneProc=TDoneProc (nil);

{ bind a filename to an external file, from fassign.pas }

procedure GPC_Assign  (var T : AnyFile; const Name : String); asmname '_p_assign';
procedure AssignFile  (var T : AnyFile; const Name : String); asmname '_p_assign';
procedure AssignCFile (var T : AnyFile; CFile : CFilePtr);    asmname '_p_assign_cfile';

{ Generic file handling routines and their support, from file.c }

{ Flags that can be ORed into FileMode. The default value of FileMode is
  FileMode_Reset_ReadWrite. Sorry for the somewhat confusing values, they
  are meant to be compatible to BP (as far as BP supports this). }
const
  FileMode_Reset_ReadWrite   = 2; { Allow writing to files opened with Reset }
  FileMode_Rewrite_WriteOnly = 4; { Do not allow reading from files opened with Rewrite }
  FileMode_Extend_WriteOnly  = 8; { Do not allow reading from files opened with Extend }

type
  StatFSBuffer = record
    BlockSize, BlocksTotal, BlocksFree : LongestInt;
    FilesTotal, FilesFree : Integer
  end;

var
  GPC_FileMode : asmname '_p_filemode' Integer;

  { Open text files as binary. Only relevant on systems like Dos where
    this makes a difference. }
  TextFilesBinary : asmname '_p_textfiles_binary' Boolean;

procedure GPC_Flush    (          var aFile : AnyFile);                                        asmname '_p_flush';
function  GPC_GetFile  (protected var aFile : AnyFile) : CFilePtr;                             asmname '_p_getfile';

{ Various other versions of Reset, Rewrite and Extend are still overloaded magically }
procedure GPC_Rewrite  (          var aFile : AnyFile; FileName : CString; Length : Integer);  asmname '_p_rewrite';
procedure GPC_Extend   (          var aFile : AnyFile; FileName : CString; Length : Integer);  asmname '_p_extend';
procedure GPC_Reset    (          var aFile : AnyFile; FileName : CString; Length : Integer);  asmname '_p_reset';
procedure GPC_Close    (          var aFile : AnyFile);                                        asmname '_p_close';
procedure CloseFile    (          var aFile : (*@@Any*)File);                                        asmname '_p_close';
function  FileName_CString (protected var aFile : AnyFile) : CString;                          asmname '_p_filename';
procedure GPC_Erase    (          var aFile : AnyFile);                                        asmname '_p_erase';
procedure GPC_Rename   (          var aFile : AnyFile; NewName : CString);                     asmname '_p_rename';
Procedure GPC_ChDir    (Path : CString);                                                       asmname '_p_chdir';
Procedure GPC_MkDir    (Path : CString);                                                       asmname '_p_mkdir';
Procedure GPC_RmDir    (Path : CString);                                                       asmname '_p_rmdir';
procedure GPC_SetFileTime  (      var aFile : AnyFile; Time : UnixTimeType);                   asmname '_p_set_file_time';
(*@@IO critical*) procedure StatFS       (Path : CString; var Buf : StatFSBuffer);                               asmname '_p_statfs';
procedure GPC_Get        (var aFile : AnyFile);                    asmname '_p_get';
function  GPC_EOF        (var aFile : AnyFile) : Boolean;          asmname '_p_eof';
function  GPC_EOLn       (var aFile : Text)    : Boolean;          asmname '_p_eoln';

{ Checks if data are available to be read from aFile. This is similar to
  not EOF (aFile), but does not block on "files" that can grow, like TTYs
  or pipes. }
function  DataReady      (var aFile : AnyFile) : Boolean;          asmname '_p_data_ready';

{ Random access file routines, from randfile.c }

function  GPC_GetSize      (          var aFile : AnyFile) : Integer;           asmname '_p_getsize';
procedure GPC_Truncate     (          var aFile : AnyFile);                     asmname '_p_truncate';
procedure GPC_DefineSize   (          var aFile : AnyFile; NewSize :  Integer); asmname '_p_definesize';
procedure GPC_SeekAll      (          var aFile : AnyFile; NewPlace : Integer); asmname '_p_seekall';
procedure GPC_SeekRead     (          var aFile : AnyFile; NewPlace : Integer); asmname '_p_seekread';
procedure GPC_SeekWrite    (          var aFile : AnyFile; NewPlace : Integer); asmname '_p_seekwrite';
procedure GPC_SeekUpdate   (          var aFile : AnyFile; NewPlace : Integer); asmname '_p_seekupdate';
function  GPC_Empty        (protected var aFile : AnyFile) : Boolean;           asmname '_p_empty';
procedure GPC_Update       (          var aFile : AnyFile);                     asmname '_p_update';
function  GPC_LastPosition (          var aFile : AnyFile) : Integer;           asmname '_p_lastposition';
function  GPC_Position     (          var aFile : AnyFile) : Integer;           asmname '_p_position';

{ Versions with only 3 parameters are still overloaded magically }
(*@@*)procedure GPC_BlockRead  (var aFile : File;           var Buf : Void; Count : Cardinal; var Result : Cardinal); asmname '_p_blockread';
(*@@*)procedure GPC_BlockWrite (var aFile : File; protected var Buf : Void; Count : Cardinal; var Result : Cardinal); asmname '_p_blockwrite';

{ Routines to output various things, from write.c }

procedure GPC_Page      (var aFile : Text);                       asmname '_p_page';
procedure GPC_Put       (var aFile : AnyFile);                    asmname '_p_put';

{ File name routines, from filename.pas }

{
  Define constants for different systems:

  OSDosFlag:         flag to indicate whether the target system is Dos

  PathSeparator:     the separator of multiple paths, e.g. in the PATH
                     environment variable

  DirSeparator:      the separator of the directories within a full file name

  DirSeparators:     a set of all possible directory and drive name separators

  ExtSeparator:      the separator of a file name extension

  DirSelf:           the name of a directory in itself

  DirParent:         the name of the parent directory

  NullDeviceName:    the full file name of the null device

  TTYDeviceName:     the full file name of the current TTY

  ConsoleDeviceName: the full file name of the system console. On Dos systems,
                     this is the same as the TTY, but on systems that allow
                     remote login, this is a different thing and may reach a
                     completely different user than the one running the
                     program, so use with care.

  PathEnvVar:        the name of the environment variable which (usually)
                     contains the executable search path

  ShellEnvVar:       the name of the environment variable which (usually)
                     contains the path of the shell executable

  ConfigFileMask:    a mask for the option file name as returned by
                     ConfigFileName
}

{$ifdef __OS_DOS__}

const
  OSDosFlag         = True;
  PathSeparator     = ';';
  DirSeparator      = '\';
  DirSeparators     = [':', '\', '/'];
  ExtSeparator      = '.';
  DirSelf           = '.';
  DirParent         = '..';
  NullDeviceName    = 'nul';
  TTYDeviceName     = 'con';
  ConsoleDeviceName = 'con';
  PathEnvVar        = 'PATH';
  ShellEnvVar       = 'COMSPEC';
  ConfigFileMask    = '*.cfg';

{$else}

const
  OSDosFlag         = False;
  PathSeparator     = ':';
  DirSeparator      = '/';
  DirSeparators     = ['/'];
  ExtSeparator      = '.';
  DirSelf           = '.';
  DirParent         = '..';
  NullDeviceName    = '/dev/null';
  TTYDeviceName     = '/dev/tty';
  ConsoleDeviceName = '/dev/console';
  PathEnvVar        = 'PATH';
  ShellEnvVar       = 'SHELL';
  ConfigFileMask    = '.*';

{$endif}

const
  NewLine = {$char-escapes} "\n" {$no-char-escapes}; { the separator of lines within a file }
  WildCardChars = ['*', '?', '[', ']'];

type
  DirPtr = Pointer;

  GlobBuffer = record
    Count    : Integer;
    Result   : PCStrings;
    Internal : Pointer
  end;

{ libc system call }
function  CSystem (CmdLine : CString) : Integer;                           asmname 'system';

{ Like CSystem, but reports execution errors via the IOResult mechanism,
  returns only the exit status of the executed program. }
(*@@IO critical*)function  Execute (CmdLine : CString) : Integer;                           asmname '_p_execute';

{ Get an environment variable. If it does not exist, GetEnv returns the
  empty string, which can't be distinguished from a variable with an empty
  value, while CGetEnv returns nil then. }
function  GetEnv  (const EnvVar : String) : TString;                       asmname '_p_getenv';
function  CGetEnv (EnvVar : CString) : CString;                            asmname 'getenv';

{ If s is of the form var=value, the environment variable with the name var
  is set to value. A previous value, if any, is overwritten. The function
  returns 0 on success, and something else otherwise. NOTE: s is NOT copied
  automatically, so unless it's a constant, a copy of the actual variable
  made with StrDup/StrNew should be passed to this function. }
function  PutEnv  (s : CString) : Integer;                                 asmname 'putenv';

{ Change a file name to use the OS dependent directory separator }
function  Slash2OSDirSeparator (const s : String) : TString;               asmname '_p_slash2osdirseparator';

{ Change a file name to use '/' as directory separator }
function  OSDirSeparator2Slash (const s : String) : TString;               asmname '_p_osdirseparator2slash';

{ Like Slash2OSDirSeparator for CStrings -- NOTE: overwrites the CString }
function  Slash2OSDirSeparator_CString (s : CString) : CString;            asmname '_p_slash2osdirseparator_cstring';

{ Like OSDirSeparator2Slash for CStrings -- NOTE: overwrites the CString }
function  OSDirSeparator2Slash_CString (s : CString) : CString;            asmname '_p_osdirseparator2slash_cstring';

{ Add a DirSeparator to the end of s, if there is not already one and
  s denotes an existing directory }
function  AddDirSeparator (const s : String) : TString;                    asmname '_p_adddirseparator';

{ Remove a DirSeparator at the end of s, if there is one }
function  RemoveDirSeparator (const s : String) : TString;                 asmname '_p_removedirseparator';

{ Returns the current directory using OS dependent directory separators }
function  GetCurrentDirectory     : TString;                               asmname '_p_get_current_directory';

{ Returns a directory suitable for storing temporary files using OS dependent
  directory separators. If found, the result always ends in DirSeparator.
  If no directory is found, an empty string is returned. }
function  GetTempDirectory        : TString;                               asmname '_p_get_temp_directory';

{ Returns a non-existing file name in GetTempDirectory. If no temp
  directory is found, i.e. GetTempDirectory returns the empty string,
  a runtime error is raised. }
function  GetTempFileName         : TString;                               asmname '_p_get_temp_file_name';

{ The same as GetTempFileName, but returns a CString allocated from the heap. }
function  GetTempFileName_CString : CString;                               asmname '_p_get_temp_file_name_cstring';

{ Get the external file name }
function  FileName (var f : AnyFile) : TString;                            asmname '_p_file_name';

{ Returns true if the given file name is an existing file }
function  FileExists      (const FileName : String) : Boolean;             asmname '_p_file_exists';

{ Returns true if the given file name is an existing directory }
function  DirectoryExists (const FileName : String) : Boolean;             asmname '_p_directory_exists';

{ If a file of the given name exists in one of the directories given in
  DirList (separated by PathSeparator), returns the full path, otherwise
  returns an empty string. If FileName already contains an element of
  DirSeparators, returns Slash2OSDirSeparator (FileName) if it exists. }
function  FSearch (const FileName, DirList : String) : TString;            asmname '_p_fsearch';

{ Like FSearch, but only find executable files. Under Dos, if not found,
  the function tries appending '.com', '.exe' and '.bat', so you don't have
  to specify these extensions in FileName (and with respect to portability,
  it might be preferable to not do so). }
function  FSearchExecutable (const FileName, DirList : String) : TString;  asmname '_p_fsearch_executable';

{ Replaces all occurrences of `$FOO' and `~' in s by the value of the
  environment variables FOO or HOME, respectively. If a variable is not
  defined, the function returns False, and s contains the name of the
  undefined variable. Otherwise, if all variables are found, s contains
  the replaced string, and True is returned. }
function  ExpandEnvironment (var s : String) : Boolean;                    asmname '_p_expand_environment';

{ Expands the given path name to a full path name. Relative paths are
  expanded using the current directory, and occurrences of DirSelf and
  DirParent are removed. Under Dos, it removes a trailing ExtSeparator
  (except in a trailing DirSelf or DirParent), like Dos does.
  If the directory, i.e. the path without the file name, is invalid,
  the empty string is returned. }
function  FExpand (const Path : String) : TString;                         asmname '_p_fexpand';

{ FExpands Path, and then removes the current directory from it, if it
  is a prefix of it. If OnlyCurDir is set, the current directory will
  be removed only if Path denotes a file in, not below, it. }
function  RelativePath (const Path : String; OnlyCurDir : Boolean) : TString; asmname '_p_relative_path';

{ Is s a UNC filename? (Dos only, otherwise always returns False) }
function  isUNC (const s : String) : Boolean;                              asmname '_p_isunc';

{ Splits a file name into directory, name and extension }
procedure FSplit (const Path : String; var Dir, Name, Ext : String);       asmname '_p_fsplit';

{ Functions that extract one of the parts from FSplit }
function  DirFromPath  (const Path : String) : TString;                    asmname '_p_dir_from_path';
function  NameFromPath (const Path : String) : TString;                    asmname '_p_name_from_path';
function  ExtFromPath  (const Path : String) : TString;                    asmname '_p_ext_from_path';

{ Start reading a directory. If successful, a pointer is returned that can
  be used for subsequent calls to ReadDir and finally CloseDir. On failure,
  nil is returned. }
function  OpenDir  (Name : CString) : DirPtr;                              C;

{ Reads one entry from the directory Dir, and returns the file name. On
  errors or end of directory, the empty string is returned. }
function  ReadDir  (Dir : DirPtr) : TString;                               asmname '_p_readdir';

{ Closes a directory opened with OpenDir. Returns 0 on success, anything
  else on error. }
function  CloseDir (Dir : DirPtr) : Integer;                               asmname '_p_closedir';

{ Tests if a file name matches a shell wildcard pattern (?, *, []) }
function  FileNameMatch (const Pattern, Name : String) : Boolean;          asmname '_p_filenamematch';

{ File name globbing }
procedure Glob     (var Buf : GlobBuffer; Pattern : CString);              asmname '_p_glob';
procedure GlobFree (var Buf : GlobBuffer);                                 asmname '_p_globfree';

{ Return a file name suited for a configuration file.
  Under Unix, it is made up of '$HOME/.' plus the base name (without
  directory) of the running program (as returned by ParamStr (0)) plus 'rc',
  if Global is False (this is meant for user-specific configuration),
  and of '/etc/' plus base name plus 'rc' if Global is True (system-wide
  configuration).
  Under Dos, it is composed of the directory and base name of the running
  program plus '.cfg', regardless of Global.
  If Name is not empty, it will be used instead of the base name of the
  running program. }
function  ConfigFileName (Name : String; Global : Boolean) : TString;      asmname '_p_config_file_name';

{ ========================= MATHEMATICAL ROUTINES ========================= }

const
  Pi = 3.14159265358979323846264338327950288419716;

{ Int and Frac routines for real numbers, from math.pas }

function GPC_Frac (x : LongestReal) : LongestReal; asmname '_p_frac';
function GPC_Int  (x : LongestReal) : LongestReal; asmname '_p_int';

{ Transcendental functions for Reals and LongReals, from maths.c }

function Real_Arctan     (x : Double)                : Double;   asmname '_p_arctan';
function Arctan2         (y, x : Double)             : Double;   asmname '_p_arctan2';
function Real_Sqrt       (x : Double)                : Double;   asmname '_p_sqrt';
function Real_Ln         (x : Double)                : Double;   asmname '_p_ln';
function Real_Exp        (x : Double)                : Double;   asmname '_p_exp';
function Real_Sin        (x : Double)                : Double;   asmname '_p_sin';
function SinH            (x : Double)                : Double;   asmname '_p_sinh';
function Real_Cos        (x : Double)                : Double;   asmname '_p_cos';
function CosH            (x : Double)                : Double;   asmname '_p_cosh';
function LongReal_Arctan (x : LongReal)              : LongReal; asmname '_pp_arctan';
function LongReal_Sqrt   (x : LongReal)              : LongReal; asmname '_pp_sqrt';
function LongReal_Ln     (x : LongReal)              : LongReal; asmname '_pp_ln';
function LongReal_Exp    (x : LongReal)              : LongReal; asmname '_pp_exp';
function LongReal_Sin    (x : LongReal)              : LongReal; asmname '_pp_sin';
function LongReal_Cos    (x : LongReal)              : LongReal; asmname '_pp_cos';

{ Extended Pascal `**' operator }
function Real_Power      (x, y : Double)             : Double;   asmname '_p_expon';
function LongReal_Power  (x, y : LongReal)           : LongReal; asmname '_pp_expon';

{ from math.pas }

function Real_Pow        (x : Double; y : Integer)   : Double;   asmname '_p_pow';
function LongReal_Pow    (x : LongReal; y : Integer) : LongReal; asmname '_pp_pow';

{ Library functions for complex type arguments, from math.pas }

function Complex_Polar  (Length, Theta : Double)   : Complex; asmname '_p_polar';
function Complex_Arg    (z : Complex)              : Double;  asmname '_p_arg';
function Complex_Arctan (z : Complex)              : Complex; asmname '_p_z_arctan';
function Complex_Sqrt   (z : Complex)              : Complex; asmname '_p_z_sqrt';
function Complex_Ln     (z : Complex)              : Complex; asmname '_p_z_ln';
function Complex_Exp    (z : Complex)              : Complex; asmname '_p_z_exp';
function Complex_Sin    (z : Complex)              : Complex; asmname '_p_z_sin';
function Complex_Cos    (z : Complex)              : Complex; asmname '_p_z_cos';
function Complex_Pow    (z : Complex; y : Integer) : Complex; asmname '_p_z_pow';
function Complex_Power  (z : Complex; y : Double)  : Complex; asmname '_p_z_expon';

{ Random number routines, from random.c }

{ RandomizePtr, RandRealPtr and RandIntPtr point to these routines by default }
procedure Default_Randomize;                                 asmname '_p_default_randomize';
function  Default_RandReal : LongestReal;                    asmname '_p_default_randreal';
function  Default_RandInt (Max : LongestCard) : LongestCard; asmname '_p_default_randint';

{ GPC_Randomize, GPC_RandReal and GPC_RandInt call the actual routines through RandomizePtr, RandRealPtr and RandIntPtr }
procedure GPC_Randomize;                                 asmname '_p_randomize';
function  GPC_RandReal : LongestReal;                    asmname '_p_randreal';
function  GPC_RandInt (Max : LongestCard) : LongestCard; asmname '_p_randint';

type
  RandomizeType = ^procedure;
  RandRealType  = ^function : LongestReal;
  RandIntType   = ^function (Max : LongestCard) : LongestCard;

var
  RandomizePtr : asmname '_p_randomize_ptr' RandomizeType;
  RandRealPtr  : asmname '_p_randreal_ptr'  RandRealType;
  RandIntPtr   : asmname '_p_randint_ptr'   RandIntType;

{ ================== OPERATIONS ON CERTAIN BUILT-IN TYPES ================= }

{ Time and date routines for Extended Pascal, from times.c }

procedure UnixTimeToTime      (UnixTime : UnixTimeType; var Year, Month, Day, Hour, Minute, Second : Integer); asmname '_p_unix_time_to_time';
function  TimeToUnixTime      (Year, Month, Day, Hour, Minute, Second : Integer) : UnixTimeType;               asmname '_p_time_to_unix_time';
function  GetUnixTime         (var MicroSecond : Integer) : UnixTimeType;                                         asmname '_p_get_unix_time';

{ Time and date routines for Extended Pascal, from time.pas }

const { from types.h }
  GPC_Date_Length = 11;
  GPC_Time_Length = 8;

  MonthName : array [1 .. 12] of String [9] =
    ('January', 'February', 'March', 'April', 'May', 'June',
     'July', 'August', 'September', 'October', 'November', 'December');

  MonthLength : array [1 .. 12] of Integer =
    (31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);

type
  GPC_Date_String = packed array [1 .. GPC_Date_Length] of Char;
  GPC_Time_String = packed array [1 .. GPC_Time_Length] of Char;

  GPC_TimeStamp = {@@packed} record
                    DateValid,
                    TimeValid   : Boolean;
                    Year        : Integer;
                    Month       : 1 .. 12;
                    Day         : 1 .. 31;
                    DayOfWeek   : 0 .. 6;  { 0 means Sunday }
                    Hour        : 0 .. 23;
                    Minute      : 0 .. 59;
                    Second      : 0 .. 59;
                    MicroSecond : 0 .. 999999
                  end;

function  GetDayOfWeek        (Day, Month, Year : Integer) : Integer;                                          asmname '_p_dayofweek';
procedure UnixTimeToTimeStamp (UnixTime : UnixTimeType; var aTimeStamp : TimeStamp);                           asmname '_p_unix_time_to_time_stamp';
function  TimeStampToUnixTime (protected var aTimeStamp : TimeStamp) : UnixTimeType;                           asmname '_p_time_stamp_to_unix_time';
procedure GPC_GetTimeStamp    (var aTimeStamp : TimeStamp);                                                    asmname '_p_gettimestamp';

{ Is the year a leap year? }
function  IsLeapYear (Year : Integer) : Boolean; asmname '_p_is_leap_year';

procedure GPC_Date (protected var aTimeStamp : TimeStamp; var Result : GPC_Date_String); asmname '_p_date';
procedure GPC_Time (protected var aTimeStamp : TimeStamp; var Result : GPC_Time_String); asmname '_p_time';

implementation

end.
