/*
 *	cook - file construction tool
 *	Copyright (C) 1997, 1998 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to walk a single graph recipe node
 */

#include <ac/unistd.h> /* for getpid */

#include <cook.h>
#include <dir_part.h>
#include <error_intl.h>
#include <graph.h>
#include <graph/file.h>
#include <graph/file_list.h>
#include <graph/file_pair.h>
#include <graph/recipe.h>
#include <graph/run.h>
#include <id.h>
#include <id/variable.h>
#include <match.h>
#include <opcode/context.h>
#include <opcode/list.h>
#include <option.h>
#include <os.h>
#include <recipe.h>
#include <stmt.h>
#include <str_list.h>
#include <trace.h>


/*
 * NAME
 *	run_it
 *
 * SYNOPSIS
 *	opcode_status_ty run_it(opcode_list_ty *);
 *
 * DESCRIPTION
 *	The run_it function is used to execute a pre-compiled opcode
 *	stream.  Use to run recipe bodies.
 *
 * RETURNS
 *	opcode_status_ty - indicating success or failure
 */

static opcode_status_ty run_it _((opcode_list_ty *, const match_ty *));

static opcode_status_ty
run_it(olp, mp)
	opcode_list_ty	*olp;
	const match_ty	*mp;
{
	opcode_context_ty *ocp;
	opcode_status_ty status;

	ocp = opcode_context_new(olp, mp);
	status = opcode_context_execute_nowait(ocp);
	opcode_context_delete(ocp);
	return status;
}


static int make_target_directories _((graph_file_list_nrc_ty *));

static int
make_target_directories(gflp)
	graph_file_list_nrc_ty *gflp;
{
	size_t		k;
	int		status;
	int		echo;
	int		errok;

	status = 0;
	echo = !option_test(OPTION_SILENT);
	errok = option_test(OPTION_ERROK);
	for (k = 0; k < gflp->nfiles; ++k)
	{
		graph_file_ty	*gfp;
		string_ty	*s;
	
		gfp = gflp->file[k];
		s = dir_part(gfp->filename);
		if (s)
		{
			if (os_mkdir(s, echo) && !errok)
				status = -1;
			str_free(s);
		}
	}
	return status;
}


static string_ty *host_binding_round_robin _((string_list_ty  *));

static string_ty *
host_binding_round_robin(slp)
	string_list_ty	*slp;
{
	static int	j;

	if (!j)
		j = getpid();
	if (!slp || !slp->nstrings)
	{
		static string_ty *key;
		id_ty		*idp;

		if (!key)
			key = str_from_c("parallel_hosts");
		idp = id_search(key);
		if (!idp)
			return 0;
		slp = id_variable_query2(idp);
		if (!slp || !slp->nstrings)
			return 0;
	}
	return slp->string[j++ % slp->nstrings];
}


/*
 * NAME
 *	graph_recipe_run
 *
 * SYNOPSIS
 *	graph_walk_status_ty graph_recipe_run(graph_recipe_ty *);
 *
 * DESCRIPTION
 *	The graph_recipe_run function is used via graph_walk_inner to
 *	perform a recipe.  If the derived files are out-of-date, the
 *	recipe body will be run to bring them up-to-date.
 *
 * RETURNS
 *	graph_walk_status_ty
 *		error		something went wrong
 *		uptodate	no action required
 *		uptodate_done	fingerprints indicate the file did not change
 *		done		targets are out of date, because the
 *				recipe body was run
 */

graph_walk_status_ty
graph_recipe_run(grp, gp)
	graph_recipe_ty	*grp;
	graph_ty	*gp;
{
	graph_walk_status_ty status;
	time_t		target_age;
	long		target_depth;
	string_ty	*target_absent;
	int		forced;
	string_list_ty	wl;
	string_list_ty	younger;
	int		show_reasoning;
	string_ty	*target1;
	time_t		need_age;
	size_t		j, k;
	int		phony;
	sub_context_ty	*scp;

	trace(("graph_recipe_run(grp = %08lX)\n{\n"/*}*/, (long)grp));
	status = graph_walk_status_uptodate;
	if (grp->ocp)
	{
		need_age = grp->ocp->need_age;
		opcode_context_resume(grp->ocp);
		status = graph_walk_status_done;
		goto resume;
	}
	need_age = 0;
	phony = !grp->rp->out_of_date;

	/*
	 * Warn about essential information which is kept only in
	 * derived files.
	 */
	if (gp->file_pair)
	{
		for (j = 0; j < grp->output->nfiles; ++j)
		{
			target1 = grp->output->file[j]->filename;
			for (k = 0; k < grp->input->nfiles; ++k)
			{
				graph_file_pair_check
				(
					gp->file_pair,
					target1,
					grp->input->file[k]->filename,
					gp
				);
			}
		}
	}

	/*
	 * construct the ``target'' variable
	 */
	string_list_constructor(&wl);
	assert(grp->output);
	assert(grp->output->nfiles > 0);
	if (grp->output->nfiles > 0)
	{
		target1 = grp->output->file[0]->filename;
		string_list_append(&wl, target1);
	}
	else
		target1 = str_from_c("\7bogus\7"); /* mem leak */
	id_assign_push(id_target, id_variable_new(&wl));
	string_list_destructor(&wl);

	/*
	 * construct the ``targets'' variable
	 */
	string_list_constructor(&wl);
	assert(grp->input);
	for (j = 0; j < grp->output->nfiles; ++j)
		string_list_append(&wl, grp->output->file[j]->filename);
	id_assign_push(id_targets, id_variable_new(&wl));
	string_list_destructor(&wl);

	/*
	 * construct the ``need'' variable
	 */
	string_list_constructor(&wl);
	assert(grp->input);
	for (j = 0; j < grp->input->nfiles; ++j)
		string_list_append(&wl, grp->input->file[j]->filename);
	id_assign_push(id_need, id_variable_new(&wl));
	string_list_destructor(&wl);

	/*
	 * Constructing the ``younger'' variable takes quite a bit
	 * longer because we need to consult the file modification
	 * times.  [[Fake an assignment to avoid problems with errors.]]
	 */
	string_list_constructor(&younger);
	id_assign_push(id_younger, id_variable_new(&younger));

	/*
	 * Flags apply to the precondition and to the ingredients
	 * evaluation.  That is why the grammar puts them first.
	 */
	recipe_flags_set(grp->rp);
	show_reasoning = option_test(OPTION_REASON);

	/*
	 * see of the recipe is forced to activate
	 */
	forced = option_test(OPTION_FORCE);
	if (forced && show_reasoning)
	{
		scp = sub_context_new();
		sub_var_set(scp, "File_Name", "%S", target1);
		error_with_position
		(
			&grp->rp->pos,
			scp,
i18n("\"$filename\" is out of date because the \"forced\" flag is set (reason)")
		);
		sub_context_delete(scp);
	}

	/*
	 * age should be set to the worst case of all the targets
	 *
	 * The depth of the target search should be less than or equal
	 * to the depth of the worst (shallowest) ingredients search.
	 * This is to guarantee that when ingredients change they
	 * result in targets shallower in the path being updated.
	 */
	target_age = 0;
	target_absent = 0;
	target_depth = 32767;
	for (j = 0; j < grp->output->nfiles; ++j)
	{
		graph_file_ty	*gfp2;
		time_t		age2;
		long		depth2;

		gfp2 = grp->output->file[j];

		/*
		 * Remember the oldest mtime for later, when we compare
		 * them to see if it changed.  Only do this for
		 * fingerprints - they will stay the same even when the
		 * file is re-written.
		 */
		if (option_test(OPTION_FINGERPRINT))
		{
			depth2 = 32767;
			age2 = cook_mtime_oldest(gfp2->filename, &depth2);
			if (age2 < 0)
			{
				/* error message already printed */
				status = graph_walk_status_error;
				goto ret;
			}
			gfp2->mtime_oldest = age2;
		}
		
		depth2 = 32767;
		age2 = cook_mtime_newest(gfp2->filename, &depth2);
		if (age2 < 0)
		{
			/* error message already printed */
			status = graph_walk_status_error;
			goto ret;
		}
		if (age2 == 0)
			target_absent = gfp2->filename;
		else
		{
			if (depth2 < target_depth)
				target_depth = depth2;
			if (!target_age || age2 < target_age)
				target_age = age2;
		}
	}
	if (!forced && target_absent && !phony)
	{
		if (show_reasoning)
		{
			scp = sub_context_new();
			sub_var_set(scp, "File_Name", "%S", target_absent);
			error_with_position
			(
				&grp->rp->pos,
				scp,
	 i18n("\"$filename\" is out of date because it does not exist (reason)")
			);
			sub_context_delete(scp);
		}
		forced = 1;
	}
	if (!forced && target_depth > 0 && option_test(OPTION_SHALLOW))
	{
		if (show_reasoning)
		{
			scp = sub_context_new();
			sub_var_set(scp, "File_Name", "%S", target1);
			error_with_position
			(
				&grp->rp->pos,
				scp,
	    i18n("\"$filename\" is out of date because it is too deep (reason)")
			);
			sub_context_delete(scp);
		}
		forced = 1;
	}
	if (forced)
	{
		/* make sure ``younger'' contains all ingredients */
		target_age = 0;
		target_depth = 0;
	}
	trace(("forced = %d;\n", forced));
	trace(("target_depth = %d;\n", target_depth));
	trace(("target_age = %ld;\n", (long)target_age));

	/*
	 * Look at the mtimes for each of the ingredients.
	 */
	need_age = 0;
	for (j = 0; j < grp->input->nfiles; ++j)
	{
		graph_file_ty	*gfp2;
		time_t		age2;
		long		depth2;

		gfp2 = grp->input->file[j];
		depth2 = 32767;
		age2 = cook_mtime_oldest(gfp2->filename, &depth2);
		if (age2 < 0)
		{
			/* error message already printed */
			status = graph_walk_status_error;
			goto ret;
		}

		/*
		 * track the youngest ingredient,
		 * in case we need to adjust the targets' times
		 */
		if (age2 > need_age)
			need_age = age2;

		/*
		 * This function is only called AFTER an ingredient has
		 * been derived. It will exist (well, almost: it could
		 * be a phony) and so the mtime in the stat cache will
		 * have been set.
		 *
		assert(age2 != 0);
		 */

		/*
		 * Check to see if this ingredient invalidates the
		 * target, based on its age.  (Don't say anything if we
		 * already know it's out of date.)
		 */
		if (gfp2->done)
		{
			if (!forced)
			{
				if (show_reasoning)
				{
					scp = sub_context_new();
					sub_var_set
					(
						scp,
						"File_Name1",
						"%S",
						target1
					);
					sub_var_set
					(
						scp,
						"File_Name2",
						"%S",
						gfp2->filename
					);
					error_with_position
					(
						&grp->rp->pos,
						scp,
i18n("$filename1 is out of date because $filename2 was cooked and is \
now younger (reason)")
					);
					sub_context_delete(scp);
				}
				forced = 1;
			}
			string_list_append_unique(&younger, gfp2->filename);
		}

		/*
		 * Check to see if this ingredient invalidates the
		 * target, based on its age.  (Don't say anything if we
		 * already know it's out of date.)
		 */
		if (age2 >= target_age && !phony)
		{
			if (!forced)
			{
				if (show_reasoning)
				{
					scp = sub_context_new();
					sub_var_set
					(
						scp,
						"File_Name1",
						"%S",
						target1
					);
					sub_var_set
					(
						scp,
						"File_Name2",
						"%S",
						gfp2->filename
					);
					error_with_position
					(
						&grp->rp->pos,
						scp,
	i18n("$filename1 is out of date because $filename2 is younger (reason)")
					);
					sub_context_delete(scp);
				}
				forced = 1;
			}
			string_list_append_unique(&younger, gfp2->filename);
		}

		/*
		 * Check to see if this ingredient invalidates the
		 * target, based on its depth.  (Don't say anything if we
		 * already know its out of date.)
		 */
		if (depth2 < target_depth && !phony)
		{
			trace(("depth2 = %d;\n", depth2));
			if (!forced)
			{
				if (show_reasoning)
				{
					scp = sub_context_new();
					sub_var_set
					(
						scp,
						"File_Name1",
						"%S",
						target1
					);
					sub_var_set
					(
						scp,
						"File_Name2",
						"%S",
						gfp2->filename
					);
					error_with_position
					(
						&grp->rp->pos,
						scp,
      i18n("$filename1 is out of date because $filename2 is shallower (reason)")
					);
					sub_context_delete(scp);
				}
				forced = 1;
			}
			string_list_append_unique(&younger, gfp2->filename);
		}
	}
	if (grp->input->nfiles == 0)
	{
		/*
		 * If there are no input files, pretend that the
		 * youngest ingredient is ``now'' if the file does not
		 * exist, and a second older than the target if it does
		 * exist.
		 */
		if (forced)
			time(&need_age);
		else
			need_age = target_age - 1;
	}

	/*
	 * Assign the ``younger'' variable.
	 * (Use a normal assignment, we did the push already.)
	 */
	id_assign(id_younger, id_variable_new(&younger));
	string_list_destructor(&younger);

	/*
	 * See if we need to perform the actions attached to this recipe.
	 */
	if (forced)
	{
		/*
		 * Remember that we did something.
		 */
		status = graph_walk_status_done;

		if (grp->rp->out_of_date)
		{
			/*
			 * Make directories for the targets if asked to.
			 */
			trace(("do recipe body\n"));
			if
			(
				option_test(OPTION_MKDIR)
			&&
				make_target_directories(grp->output) < 0
			)
			{
				status = graph_walk_status_error;
				goto ret;
			}

			/*
			 * Unlink the targets if asked to.
			 */
			if (option_test(OPTION_UNLINK))
			{
				for (k = 0; k < grp->output->nfiles; ++k)
				{
					graph_file_ty	*gfp2;
	
					gfp2 = grp->output->file[k];
					if
					(
						os_delete
						(
							gfp2->filename,
						     !option_test(OPTION_SILENT)
						)
					&&
						!option_test(OPTION_ERROK)
					)
					{
						status =
							graph_walk_status_error;
						goto ret;
					}
				}
			}

			if (option_test(OPTION_TOUCH))
			{
				/*
				 * Touch the targets, if asked to touch
				 * rather than to build.
				 */
				for (k = 0; k < grp->output->nfiles; ++k)
				{
					graph_file_ty	*gfp2;
	
					gfp2 = grp->output->file[k];
					if (!option_test(OPTION_SILENT))
					{
						scp = sub_context_new();
						sub_var_set
						(
							scp,
							"File_Name",
							"%S",
							gfp2->filename
						);
						error_intl
						(
							scp,
							i18n("touch $filename")
						);
						sub_context_delete(scp);
					}
					if (os_touch(gfp2->filename))
						status =
							graph_walk_status_error;
				}
			}
			else
			{
				opcode_status_ty result;
				string_ty	*hostname;
	
				/*
				 * run the recipe body
				 */
				trace(("doing it now\n"));
				grp->ocp =
					opcode_context_new
					(
						grp->rp->out_of_date,
						grp->mp
					);
				hostname =
					host_binding_round_robin
					(
						grp->host_binding
					);
				if (hostname)
				{
					opcode_context_host_binding_set
					(
						grp->ocp,
						hostname
					);
				}
				resume:
				result = opcode_context_execute(grp->ocp);
				switch (result)
				{
				case opcode_status_wait:
					grp->ocp->need_age = need_age;
					opcode_context_suspend(grp->ocp);
					trace(("wait...\n"));
					trace((/*{*/"}\n"));
					return graph_walk_status_wait;

				case opcode_status_success:
					status = graph_walk_status_done;
					break;
	
				case opcode_status_error:
					if (option_test(OPTION_ERROK))
						status = graph_walk_status_done;
					else
						status =
							graph_walk_status_error;
					break;

				case opcode_status_interrupted:
					status = graph_walk_status_error;
					break;
				}
				opcode_context_delete(grp->ocp);
				grp->ocp = 0;

				/*
				 * Remove recipe targets on errors,
				 * unless asked to keep them around.
				 * This ensures they will be built
				 * again; hopefully without errors the
				 * next time.  (Perfect cookbooks with
				 * exact dependencies don't need this,
				 * but users often omit dependencies;
				 * removing the file forces a re-build.)
				 */
				if
				(
					status == graph_walk_status_error
				&&
					!option_test(OPTION_PRECIOUS)
				)
				{
					for
					(
						k = 0;
						k < grp->output->nfiles;
						++k
					)
					{
						graph_file_ty	*gfp2;
	
						gfp2 = grp->output->file[k];
						os_delete
						(
							gfp2->filename,
						     !option_test(OPTION_SILENT)
						);
					}
				}
			}
		}
		else
		{
			if (show_reasoning)
			{
				scp = sub_context_new();
				sub_var_set(scp, "File_Name", "%S", target1);
				error_intl(scp, i18n("$filename is phony (reason)"));
				sub_context_delete(scp);
			}

			/*
			 * remember that this ``file'' has been ``changed''
			 */
			for (j = 0; j < grp->output->nfiles; ++j)
			{
				graph_file_ty	*gfp;

				gfp = grp->output->file[j];
				gfp->done++;
			}
		}
	}
	else
	{
		if (show_reasoning)
		{
			scp = sub_context_new();
			sub_var_set(scp, "File_Name", "%S", target1);
			error_intl
			(
				scp,
				i18n("$filename is up to date (reason)")
			);
			sub_context_delete(scp);
		}
		if (grp->rp->up_to_date)
		{
			opcode_status_ty result;

			/*
			 * Make directories for the targets if asked to.
			 * Often redundant, but it wont hurt.
			 */
			if
			(
				option_test(OPTION_MKDIR)
			&&
				make_target_directories(grp->output) < 0
			)
			{
				status = graph_walk_status_error;
				goto ret;
			}

			/*
			 * This feature isn't used often, don't worry
			 * about making it parallel.
			 */
			trace(("perform ``use'' clause\n"));
			result = run_it(grp->rp->up_to_date, grp->mp);
			if (result != opcode_status_success)
				status = graph_walk_status_error;
		}
	}

	/*
	 * Delete the variables unique to recipe body execution.
	 */
ret:
	id_unassign(id_target);
	id_unassign(id_targets);
	id_unassign(id_need);
	id_unassign(id_younger);

	/*
	 * adjust file modification times
	 * (tracking fingerprints as we go)
	 */
	if (status == graph_walk_status_done && grp->rp->out_of_date)
	{
		time_t		mtime;

		/*
		 * The output files need to be at least this date
		 * stamp to be mtime-consistent with the inputs.
		 */
		mtime = need_age + 1;

		if (option_test(OPTION_FINGERPRINT))
		{
			/*
			 * Update the times, and see if the pringerprint
			 * changed.  If the fingerprint did not change
			 * on any target, use the uptodate_done result,
			 * otherwise use the done result.
			 */
			status = graph_walk_status_uptodate_done;
			for (j = 0; j < grp->output->nfiles; ++j)
			{
				graph_file_ty	*gfp;
				time_t		t;
				long		depth4;

				gfp = grp->output->file[j];
				os_clear_stat(gfp->filename);
				if (os_mtime_adjust(gfp->filename, mtime))
				{
					status = graph_walk_status_error;
					/*
					 * don't break here, need to
					 * update all of them
					 */
					continue;
				}
				t = cook_mtime_oldest(gfp->filename, &depth4);
				if (t < 0)
				{
					status = graph_walk_status_error;
					/*
					 * don't break here, need to
					 * update all of them
					 */
					continue;
				}
				if (t == gfp->mtime_oldest)
				{
					if (!option_test(OPTION_SILENT))
					{
						scp = sub_context_new();
						sub_var_set
						(
							scp,
							"File_Name",
							"%S",
							gfp->filename
						);
						error_intl
						(
							scp,
					 i18n("$filename fingerprint unchanged")
						);
					}
				}
				else if (status != graph_walk_status_error)
				{
					status = graph_walk_status_done;
					gfp->done++;
				}
			}
		}
		else
		{
			/*
			 * update the times if it worked
			 * and if it was not a phony recipe
			 */
			for (j = 0; j < grp->output->nfiles; ++j)
			{
				graph_file_ty	*gfp;

				gfp = grp->output->file[j];
				gfp->done++; /* for out-of-date calcs */
				if (os_mtime_adjust(gfp->filename, mtime))
				{
					/* error message already printed */
					status = graph_walk_status_error;
					/*
					 * don't break here, need to
					 * update all of them
					 */
				}
			}
		}
	}

	/*
	 * Make sure the target times are consistent with the
	 * ingredients times, when there was nothing to do.  This is
	 * usually only necessary when finger prints are in use, but can
	 * sometimes be necessary when file server times are out of
	 * sync.  This guarantees that a later fingerprint-less run will
	 * not find huge amounts of work to do.
	 */
	if
	(
		(
			status == graph_walk_status_uptodate
		||
			status == graph_walk_status_uptodate_done
		)
	&&
		grp->rp->out_of_date
	&&
		(
			option_test(OPTION_UPDATE)
		||
			option_test(OPTION_FINGERPRINT)
		)
	)
	{
		time_t		need_age_youngest;

		/*
		 * find the youngest ingredient's age
		 * (will be in the stat cache or the fingerprint cache)
		 */
		need_age_youngest = 0;
		for (j = 0; j < grp->input->nfiles; ++j)
		{
			graph_file_ty	*gfp2;
			time_t		age2;
			long		depth2;
	
			gfp2 = grp->input->file[j];
			age2 = cook_mtime_newest(gfp2->filename, &depth2);
			if (age2 > need_age_youngest)
				need_age_youngest = age2;
		}

		/*
		 * Advance one second younger.  This is the youngest a
		 * target may be to be mtime-consistent with the
		 * ingredients.
		 */
		need_age_youngest++;

		/*
		 * check each of the targets for consistency
		 */
		for (j = 0; j < grp->output->nfiles; ++j)
		{
			graph_file_ty	*gfp;
			time_t		age3;
			long		depth3;

			gfp = grp->output->file[j];
			age3 = cook_mtime_newest(gfp->filename, &depth3);
			if (age3 > 0 && depth3 == 0 && age3 < need_age_youngest)
				os_mtime_adjust(gfp->filename, need_age_youngest);
		}
	}

	/*
	 * cancel the recipe flags
	 */
	option_undo_level(OPTION_LEVEL_RECIPE);

	trace(("return %s;\n", graph_walk_status_name(status)));
	trace((/*{*/"}\n"));
	return status;
}
