/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <dirent.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/fcntl.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "loader.h"
#include "vmelilo.h"

/*--------------------------------------------------------------------------*/

#define MAXLINELEN			4096

#ifndef CL_SIZE
# ifndef COMPAT_CL_SIZE
#  define CL_SIZE			4096
# else
#  define CL_SIZE			COMPAT_CL_SIZE
# endif
#endif

/*--------------------------------------------------------------------------*/

static int 			line_number;			/* config file line number		*/
static BOOTRECORD	boot_record;

/*--------------------------------------------------------------------------*/

static
char *
strdupx
(	char	*str
)
{	char	*p;

	if ((p = malloc(strlen(str) + 1)) == NULL)
	{
		error_nomemory();
	}

	strcpy(p, str);

	return p;
}

/*--------------------------------------------------------------------------*/

char *
skip_white
(	char	*p
)
{
	while (*p && (*p < '!'))
		p++;

	return p;
}

/*--------------------------------------------------------------------------*/

static
void
trim_white
(	char	*p
)
{	char	*w;

	while (1)
	{
		/* say no white space */
		w = NULL;

		/* skip non-white space */
		while (*p >= '!')
			p++;

		/* if end of string */
		if (*p == '\0')
		{
			break;
		}

		/* remember where start of white space is */
		w = p;

		/* skip past white space */
		do	{ ++p; } while (*p && *p < '!');

		/* if end of string */
		if (*p == '\0')
		{
			break;
		}
	}

	/* if we ended up in white space, strip it */
	if (w) *w = '\0';
}

/*--------------------------------------------------------------------------*/
/* copy string 'p2' to string 'p1' until 1st whitespace or delimiter
 */

static
char *
extract
(	char	*p1,
	char	*p2,
	int		len,				/* size of 'p1' buffer						*/
	int		delim
)
{
	/* leave room for NULL terminator */
	len--;

	/* skip leading white space */
	p2 = skip_white(p2);

	/* until we hit a delimiter */
	while ((*p2 >= '!') && (*p2 != delim))
	{
		if (len)
		{
			*p1++ = *p2++;
			len--;
		}
		else
		{
			p2++;
		}
	}

	/* NULL terminate */
	*p1 = '\0';

	/* skip white space */
	p2 = skip_white(p2);

	/* skip delimiter and following white space */
	if (delim && (*p2 == delim))
	{
		p2 = skip_white(p2 + 1);
	}

	/* return pointer to next argument */
	return p2;
}

/*--------------------------------------------------------------------------*/

static
void
conferror
(	const char		*s
)
{
	die("%s:%d: %s\n", config_file, line_number, s);
}

/*--------------------------------------------------------------------------*/

static
void
error_spurious_data
(	void
)
{
	conferror("spurious data at end of line\n");
}

/*--------------------------------------------------------------------------*/

static
void
clear_boot_record
(	void
)
{
	memset(&boot_record, 0, sizeof(BOOTRECORD));
}

/*--------------------------------------------------------------------------*/

static
void
redefined_option
(	const char	*name
)
{
	printf("%s:%d: Redefinition of `%s'\n", config_file, line_number, name);
}

/*--------------------------------------------------------------------------*/

static
const u_long *
copy_long
(	u_long		data
)
{	u_long		*p;

	if ((p = malloc(sizeof(u_long))) == NULL)
	{
		error_nomemory();
	}

	*p = data;

	return p;
}

/*--------------------------------------------------------------------------*/

static
int
get_bool
(	const char	*text
)
{
	if (strcasecmp(text, "true") == 0)
	{
		return TRUE;
	}

	if (strcasecmp(text, "false") == 0)
	{
		return FALSE;
	}

	die("True or False argument expected on line %d of %s\n",
		line_number, config_file);
}

/*--------------------------------------------------------------------------*/

static
u_long
get_num
(	char			**str
)
{	char			c;
	int				base;
	int				digits;
	u_long			number;
	char			*p;

	p = skip_white(*str);
	trim_white(p);

	if (*p == '\0')
	{
		conferror("missing numeric constant\n");
	}

	/* determine number base from prefix */
	if (*p == '0')
	{
		if (p[1] == 'X' || p[1] == 'x')
		{
			p   += 2;
			base = 16;
		}
		else if (p[1] == 'B' || p[1] == 'b')
		{
			p   += 2;
			base = 2;
		}
		else
		{
			base = 8;
		}
	}
	else
	{
		base = 10;
	}

	number = 0;
	digits = 0;

	while (*p != '\0')
	{
		c = *p++;

		if (c < '!')
		{
			break;
		}

		/* look for 'M' and 'K' suffixes */
		if (digits && (*p < '!'))
		{
			if (c == 'M' || c == 'm')
			{
				number = number * 1024 * 1024;
				break;
			}

			if (c == 'K' || c == 'k')
			{
				number = number * 1024;
				break;
			}
		}

		if (isdigit(c))
		{
			c -= '0';
		}

		else if (islower(c))
		{
			c -= 'a' - 10;
		}

		else if (isupper(c))
		{
			c -= 'A' - 10;
		}

		else				/* unrecognised character	*/
		{
			goto fail;
		}

		if (c >= base)		/* not in number base		*/
		{
			goto fail;
		}

		digits++;
		number = number * base + c;
	}

	if (digits == 0)
	{
fail:
		conferror("invalid numeric constant\n");
	}

	*str = skip_white(p);
	return number;
}

/*--------------------------------------------------------------------------*/
/* Add a New Boot Record
 */

static
int
add_boot_record
(	const BOOTRECORD	*record
)
{	BOOTRECORD			**p;
	BOOTRECORD			*new;

	for (p = &config.records; *p; p = &(*p)->next)
	{
		if (equal_strings((*p)->label, record->label)
		||	equal_strings((*p)->alias, record->label)
		||	equal_strings((*p)->label, record->alias)
		||	equal_strings((*p)->alias, record->alias))
		{
			return FALSE;
		}
	}

	if ((new = malloc(sizeof(BOOTRECORD))) == NULL)
	{
		error_nomemory();
	}

	*new = *record;
	*p	 = new;

	return TRUE;
}

/*--------------------------------------------------------------------------*/
/* Add a New File Definition
 */

static
void
add_file_def
(	const char	*path
)
{	FILEDEF		**p;
	FILEDEF		*new;

	for (p = &config.files; *p; p = &(*p)->next)
	{
		if (case_equal_strings((*p)->path, path))
		{
			return;
		}
	}

	if ((new = malloc(sizeof(FILEDEF))) == NULL)
	{
		error_nomemory();
	}

	new->next = NULL;
	new->path = path;
	new->map  = NULL;
	*p		  = new;
}

/*--------------------------------------------------------------------------*/

static
int
parse_options
(	char	*name,
	char	*text
)
{
	if (strcasecmp(name, "default") == 0)
	{
		if (config.options.boot_default)
		{
			redefined_option(name);
		}
		config.options.boot_default = strdupx(text);
	}

	else if (strcasecmp(name, "message") == 0)
	{
		if (config.options.boot_message)
		{
			redefined_option(name);
		}
		add_file_def(config.options.boot_message = strdupx(text));
	}

	else if (strcasecmp(name, "boot") == 0)
	{
		if (config.boot_device_name)
		{
			redefined_option(name);
		}
		config.boot_device_name = strdupx(text);
	}

	else if (strcasecmp(name, "debug") == 0)
	{
		if (config.options.boot_debug)
		{
			redefined_option(name);
		}
		config.options.boot_debug = copy_long(get_bool(text));
	}

	else if (strcasecmp(name, "prompt") == 0)
	{
		if (config.options.boot_prompt)
		{
			redefined_option(name);
		}
		config.options.boot_prompt = copy_long(get_bool(text));
	}

	else if (strcasecmp(name, "delay") == 0)
	{
		if (config.options.boot_delay)
		{
			redefined_option(name);
		}

		config.options.boot_delay = copy_long(get_num(&text));
		if (*text)
		{
			error_spurious_data();
		}
	}

	else if (strcasecmp(name, "timeout") == 0)
	{
		if (config.options.boot_timeout)
		{
			redefined_option(name);
		}

		config.options.boot_timeout = copy_long(get_num(&text));
		if (*text)
		{
			error_spurious_data();
		}
	}

	else if (strcasecmp(name, "cmdline") == 0)
	{
		if (config.default_cmdline)
		{
			redefined_option(name);
		}

		if (strlen(text) >= CL_SIZE)
		{
			conferror("Boot argument is too long");
		}
		config.default_cmdline = strdupx(text);
	}

	else if (strcasecmp(name, "password") == 0)
	{
		if (config.default_password)
		{
			redefined_option(name);
		}
		config.default_password = strdupx(text);
	}

	else if (strcasecmp(name, "restricted") == 0)
	{
		if (config.default_restricted)
		{
			redefined_option(name);
		}

		config.default_restricted = copy_long(get_bool(text));
	}

	else if (strcasecmp(name, "image") == 0)
	{
		if (config.default_kernel)
		{
			redefined_option(name);
		}
		config.default_kernel = strdupx(text);
	}

	else
	{
		return FALSE;
	}

	return TRUE;
}

/*--------------------------------------------------------------------------*/

static
int
parse_boot
(	char	*name,
	char	*text
)
{
	if (strcasecmp(name, "label") == 0)
	{
		if (boot_record.label)
		{
			redefined_option(name);
		}
		boot_record.label = strdupx(text);
	}

	else if (strcasecmp(name, "alias") == 0)
	{
		if (boot_record.alias)
		{
			redefined_option(name);
		}
		boot_record.alias = strdupx(text);
	}

	else if (strcasecmp(name, "password") == 0)
	{
		if (boot_record.password)
		{
			redefined_option(name);
		}
		boot_record.password = strdupx(text);
	}

	else if (strcasecmp(name, "restricted") == 0)
	{
		if (boot_record.restricted)
		{
			redefined_option(name);
		}

		boot_record.restricted = copy_long(get_bool(text));
	}

	else if (strcasecmp(name, "cmdline") == 0)
	{
		if (boot_record.args)
		{
			redefined_option(name);
		}

		if (strlen(text) >= CL_SIZE)
		{
			conferror("Boot argument is too long");
		}
		boot_record.args = strdupx(text);
	}

	else if (strcasecmp(name, "append") == 0)
	{
		if (boot_record.append)
		{
			redefined_option(name);
		}

		if (strlen(text) >= CL_SIZE)
		{
			conferror("Boot argument is too long");
		}
		boot_record.append = strdupx(text);
	}

	else if (strcasecmp(name, "memsize") == 0)
	{
		if (boot_record.memsize)
		{
			redefined_option(name);
		}

		boot_record.memsize = copy_long(get_num(&text));
		if (*text)
		{
			error_spurious_data();
		}
	}

	else if (strcasecmp(name, "ramdisk") == 0)
	{
		if (boot_record.ramdisk)
		{
			redefined_option(name);
		}
		add_file_def(boot_record.ramdisk = strdupx(text));
	}

	else if (strcasecmp(name, "symbols") == 0)
	{
		if (boot_record.symtab)
		{
			redefined_option(name);
		}
		add_file_def(boot_record.symtab = strdupx(text));
	}

	else if (strcasecmp(name, "image") == 0)
	{
		if (boot_record.kernel)
		{
			redefined_option(name);
		}
		add_file_def(boot_record.kernel = strdupx(text));
	}

	else if (strcasecmp(name, "callmonitor") == 0)
	{
		if (boot_record.callmonitor)
		{
			redefined_option(name);
		}
		boot_record.callmonitor = copy_long(get_bool(text));
	}

	else
	{
		return FALSE;
	}

	return TRUE;
}

/*--------------------------------------------------------------------------*/

static
int
parse_files
(	char		*name,			/* always NULL for this section				*/
	char		*text
)
{
	add_file_def(strdupx(text));
	return TRUE;
}

/*--------------------------------------------------------------------------*/

static
void
boot_wrapup
(	void
)
{
	if (boot_record.label == NULL)
	{
		conferror("No label specified");
	}

	if (!add_boot_record(&boot_record))
	{
		conferror("Duplicate boot record label or alias");
	}

	clear_boot_record();
}

/*--------------------------------------------------------------------------*/

struct parser {
	char	*section;					/* section name						*/
	int		verbatim;					/* don't parse option name			*/
	void	(*wrapup)(void);			/* called at end of section			*/
	int		(*parse)(char *, char *);	/* parser function					*/
} parsers[] = {
	{"options",	FALSE,	NULL,		 parse_options	},
	{"boot",	FALSE,	boot_wrapup, parse_boot		},
	{"files",	TRUE,	NULL,		 parse_files	}
};

#define NUMPARSERS (sizeof(parsers) / sizeof(parsers[0]))

/*--------------------------------------------------------------------------*/

static struct parser *section_parse = NULL;

/*--------------------------------------------------------------------------*/

static
void
wrapup_section
(	void
)
{
	/* if there is a section handler */
	if (section_parse)
	{
		/* and a section wrap up handler */
		if (section_parse->wrapup)
		{
			/* call it */
			(*section_parse->wrapup)();
		}
	}
}

/*--------------------------------------------------------------------------*/

static
void
parse_line
(	char			*text
)
{	char			name[128];
	char			*ptr;
	int				j;

	/* if blank line or comment */
	if ((*text == '\0') || (*text == '#'))
	{
		/* discard it */
		return;
	}

	/* if start of a new section */
	if (*text == '[')
	{
		/* wrap up previous section */
		wrapup_section();

		/* get section name */
		extract(name, text + 1, sizeof(name), ']');

		/* find a parser for section */
		for (j = 0; j < NUMPARSERS; j++)
		{
			if (strcasecmp(parsers[j].section, name) == 0)
			{
				section_parse = &parsers[j];
				return;
			}
		}

		die("unrecognised section `%s' in %s line %d\n",
					name, config_file, line_number);
	}

	if (section_parse == NULL)
	{
		conferror("configuration item outside of file section\n");
	}

	if (section_parse->verbatim)
	{
		ptr = NULL;
	}
	else
	{
		/* get options name */
		text = extract(ptr = name, text, sizeof(name), '=');
	}

	if (!(*section_parse->parse)(ptr, text))
	{
		die("unrecognised option `%s' in %s line %d\n",
					name, config_file, line_number);
	}
}

/*--------------------------------------------------------------------------*/
/* Read the Configuration File
 */

void
read_config_file
(	void
)
{	FILE			*fp;
	char			*buff;

	if (strcmp(config_file, "-") == 0)
	{
		fp          = stdin;
		config_file = "standard input";
	}
	else
	{
		/* open input file */
		if ((fp = fopen(config_file, "r")) == NULL)
		{
			error_open(config_file);
		}
	}

	if (f_verbose)
	{
		printf("Reading configuration file `%s'\n", config_file);
	}

	/* allocate line buffer */
	if ((buff = malloc(MAXLINELEN)) == NULL)
	{
		error_nomemory();
	} 

	/* reset current configuration file line number */
	line_number = 0;

	/* read line of input until end of file */
	while (fgets(buff, MAXLINELEN, fp) != NULL)
	{
		/* update current line number */
		line_number++;

		/* check for long lines */
		if (strlen(buff) == (MAXLINELEN - 1))
		{
			die("line %d too long in file \"%s\"\n", line_number, config_file);
		}

		/* remove trailing white space */
		trim_white(buff);

		/* parse it */
		parse_line(buff);
	}

	/* wrap up final section */
	wrapup_section();

	/* check for errors */
	if (ferror(fp))
	{
		error_read(config_file);
	}

	/* finish with buffer */
	free(buff);

	/* finish with file */
	if (fp != stdin)
	{
		fclose(fp);
	}

	if (!config.options.boot_default)
	{
		config.options.boot_default = config.records->label;
	}
}

/*-----------------------------< end of file >------------------------------*/
