/*
 * Stack.cc : Method definitions for class dndStack
 * Copyright (C) 1996 Csar Crusius
 *
 * This file is part of the DND Library.  This library is free
 * software; you can redistribute it and/or modify it under the terms of
 * the GNU Library General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  This library is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */


//
// NOTE: The Top-of-stack is the first list element.
//

#include <OffiX/Stack.h>

/**************************************************************************
*
* FUNCTION:	dndStack.push
*
* SYNTAX:	dndPointer dndStack.push(dndPointer obj);
*
* DESCRIPTION:	Puts an object on the top of the stack.
*
* RETURNS:	NULL if could not add, obj otherwise
*
* LAST UPDATED
* ------------
*
* Jun 12, 1994	Implemented.
*
***************************************************************************/

dndPointer dndStack::push(dndPointer obj)
{
	getfirst();
	return addbefore(obj);
}

/**************************************************************************
*
* FUNCTION:	dndStack.pop
*
* SYNTAX:	dndPointer dndStack.pop(void);
*
* DESCRIPTION:	Removes an object from the top of the stack.
*
* RETURNS:	NULL if the stack is empty, top-of-stack otherwise
*
* LAST UPDATED
* ------------
*
* Jun 12, 1994	Implemented.
*
***************************************************************************/

dndPointer dndStack::pop(void)
{
	getfirst();
	return dndList::remove();
}

/**************************************************************************
*
* FUNCTION:	dndStack.drop
*
* SYNTAX:	dndPointer dndStack.drop(dndListIndex nelm=1);
*
* DESCRIPTION:	Removes nelm objects from the top of the stack.
*		Calls the internal user-defined delete routine
*		to free the memory.
*
*		nelm can be larger than the stack. In this case
*		the entire stack will be dropped.
*
* RETURNS:	NULL if the stack is empty, last deleted element otherwise.
*
* LAST UPDATED
* ------------
*
* Jun 12, 1994	Implemented.
*
***************************************************************************/

dndPointer dndStack::drop(dndListIndex NumberOfElements)
{
	getfirst();
	dndPointer LastObject=NULL;
	while(NumberOfElements--)
	{	LastObject=remove();
		if(DropFunc!=NULL && LastObject!=NULL) DropFunc(LastObject);
	}
	return LastObject;
}

/**************************************************************************
*
* FUNCTION:	dndStack.top
*
* SYNTAX:	dndPointer stack.top(void);
*
* DESCRIPTION:	Get the top-of-stack object.
*
* RETURNS:	NULL if the stack is empty, TOS otherwise.
*
* LAST UPDATED
* ------------
*
* Jun 12, 1994	Implemented.
*
***************************************************************************/

dndPointer dndStack::top(void)
{
	return getfirst();
}

/**************************************************************************
*
* FUNCTION:	dndStack.swap
*
* SYNTAX:	void dndStack.swap(dndListIndex elm=2);
*
* DESCRIPTION:	Exchange level elm and TOS objects.
*
* RETURNS:	None.
*
* LAST UPDATED
* ------------
*
* Jun 12, 1994	Implemented.
*
***************************************************************************/

void dndStack::swap(dndListIndex Element)
{
	dndList::swap(1L,Element);
}

/**************************************************************************
*
* FUNCTION:	dndStack.rot
*
* SYNTAX:	void dndStack.rot(dndListIndex elm=3);
*
* DESCRIPTION:	Rotate the stack. TOS->level 2 and elm->TOS.
*
* RETURNS:	None.
*
* LAST UPDATED
* ------------
*
* Jun 12, 1994	Implemented.
*
***************************************************************************/

void dndStack::rot(dndListIndex Element)
{
	if(Element<2L || Element>size()) return;

	dndPointer backup,object=go(Element);
	getfirst();

	while(Element--)
	{	backup=getthis();
		replace(object);
		object=backup;
		getnext();
	}
}

/**************************************************************************
*
* FUNCTION:	dndStack.unrot
*
* SYNTAX:	void dndStack.unrot(dndListIndex elm=3);
*
* DESCRIPTION:	Rotate the stack backwards. TOS->elm.
*
* RETURNS:	None.
*
* LAST UPDATED
* ------------
*
* Jun 12, 1994	Implemented.
*
***************************************************************************/

void dndStack::unrot(dndListIndex Element)
{
	if(Element<2L || Element>size()) return;

	dndPointer backup,object=getfirst();
	go(Element);

	while(Element--)
	{	backup=getthis();
		replace(object);
		object=backup;
		getprev();
	}

}
