/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: misc.c,v 1.10 94/07/13 14:10:24 nau Exp $";

/* additional functions used by parser
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <sys/stat.h>
#include <math.h>

#ifdef _POSIX_SOURCE
#include <unistd.h>
#endif

#include "global.h"

#include "cursor.h"
#include "control.h"
#include "data.h"
#include "draw.h"
#include "file.h"
#include "error.h"
#include "memory.h"
#include "misc.h"
#include "remove.h"
#include "transform.h"

#include <X11/cursorfont.h>
#include <X11/Shell.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Simple.h>

/* ---------------------------------------------------------------------------
 * some additional defines
 */
#define	M180	(M_PI/180.0)

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	long	AlarmTime = 0;		/* remaining alarm time */

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void CatchAlarmSignal(int);

/* ---------------------------------------------------------------------------
 * prints copyright information
 */
void Copyright(void)
{
	printf("\n"
		"                COPYRIGHT for %s version %s\n\n"
		"    PCB, interactive printed circuit board design\n"
		"    Copyright (C) 1994 Thomas Nau\n\n"
		"    This program is free software; you can redistribute it and/or modify\n"
		"    it under the terms of the GNU General Public License as published by\n"
		"    the Free Software Foundation; either version 2 of the License, or\n"
		"    (at your option) any later version.\n\n"
		"    This program is distributed in the hope that it will be useful,\n"
		"    but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
		"    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
		"    GNU General Public License for more details.\n\n"
		"    You should have received a copy of the GNU General Public License\n"
		"    along with this program; if not, write to the Free Software\n"
		"    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n\n",
		Progname, RELEASE);
	exit(0);
}

/* ---------------------------------------------------------------------------
 * prints usage message
 */
void Usage(void)
{
	fprintf(stderr, "\nUSAGE: %s [standard X options] [standard options] [layout]\n"
		"or   : %s [standard X options] <exactly one special option>\n\n"
		"standard options are:\n"
		"    -b:                 use backing store (takes some MB of your memory)\n"
		"    +b:                 don't use backing store (saves some MB of your memory)\n"
		"    -backup seconds:    time between two backups\n"
		"    -c number:          characters per output-line\n"
		"    -lelement command:  command to copy element files to stdout,\n"
		"                        %%f (%%s for compatibility too) is replaced by the filename\n"
		"                        %%p by the seachpath.\n"
		"    -lfile command:     command to copy layout files to stdout,\n"
		"                        %%f (%%s for compatibility too) is replaced by the filename\n"
		"                        %%p by the seachpath.\n"
		"    -lfont command:     command to copy font files to stdout,\n"
		"                        %%f (%%s for compatibility too) is replaced by the filename\n"
		"                        %%p by the seachpath.\n"
		"    -fontfile file:     read default font from this file\n"
		"    -lg layergroups:    set layergroups of new layouts to this\n"
		"    -mh value:          maximum height of layout in mil\n"
		"    -mw value:          maximum width of layout in mil\n"
		"    -p:                 use polygon lines (only for new layouts)\n"
		"    +p:                 use rounded X11 lines (only for new layouts)\n"
		"    -pnl value:         maximum display length of pin names\n"
		"    -pz value:          zoom factor for pinout windows\n"
		"    -reset:             reset connections after each element\n"
		"    +reset:             don't reset connections after each element\n"
		"    -ring:              ring bell when searching for connections is done\n"
		"    +ring:              don't ring bell when searching for connections is done\n"
		"    -s:                 save the last command entered by user\n"
		"    +s:                 don't save the last command entered by user\n\n"
		"    -save:              always save data before it is lost\n"
		"    +save:              override data if required by command\n"
		"    -sfile command:     command to copy stdin to layout file,\n"
		"                        %%f (%%s for compatibility too) is replaced by the filename\n"
		"special options are:\n"
		"    -copyright:         prints copyright information\n"
		"    -help:              prints this message\n"
		"    -version:           prints the current version number\n",
		Progname, Progname);
	exit(1);
}

/* ---------------------------------------------------------------------------
 * catches signals which abort the program
 */
void CatchSignal(int Signal)
{
	char	*s;

	switch(Signal)
	{
		case SIGHUP:	s = "SIGHUP"; break;
		case SIGINT:	s = "SIGINT"; break;
		case SIGQUIT:	s = "SIGQUIT"; break;
		case SIGABRT:	s = "SIGABRT"; break;
		case SIGTERM:	s = "SIGTERM"; break;
		case SIGSEGV:	s = "SIGSEGV"; break;
		default:		s = "unknown"; break;
	}
	EmergencySave();
	RemovePCB(PCB);
	MyFatal("aborted by %s signal\n", s);
}

/* ---------------------------------------------------------------------------
 * returns the pointer position relativ to the specified window 
 * funtion returns FALSE if everythin's OK
 */
Boolean PointerPosition(Drawable UseWindow, int *X, int *Y)
{
	int				root_x, root_y;
	unsigned int	mask;
	Window			root_window, child_window;

	return(!(XQueryPointer(Dpy, UseWindow, &root_window, &child_window,
		&root_x, &root_y, X, Y, &mask)));
}

/* ---------------------------------------------------------------------------
 * output of current elements name
 */
void SetElementnameField(void)
{
	char	*name;

	name = MyCursor.MarkedElement.Valid ? MyCursor.MarkedElement.Element.CanonicalName: "(none)";
	name = UNKNOWN(name);
	XtVaSetValues(Output.Elementname,
		XtNlabel, name,
		NULL);
}

/* ---------------------------------------------------------------------------
 * output of current layers name
 */
void SetNameField(void)
{
	XtVaSetValues(Output.PCBname,
		XtNlabel, UNKNOWN(PCB->Name),
		NULL);
}

/* ---------------------------------------------------------------------------
 * output of status line
 */
void SetCursorLine(void)
{
	char	text[20];

	sprintf(text, "%-i,%-i", MyCursor.X, MyCursor.Y);
	XtVaSetValues(Output.CursorPosition, XtNlabel, text, NULL);
}

/* ---------------------------------------------------------------------------
 * output of status line
 */
void SetStatusLine(void)
{
	char	text[80];
	char	*mode;

	switch (MyCursor.Mode)
	{
		case MODE_LINE:			mode = "line-mode"; break;
		case MODE_BLOCK:		mode = "block-mode"; break;
		case MODE_ELEMENT:		mode = "element-mode"; break;
		case MODE_LINESTACK:	mode = "linestack-mode"; break;
		default:				mode = "no mode selected"; break;
	}
	sprintf(text, "%c G=%i, Z=1:%-i, L=%-i, V=%-i, stack (%i) %s, %s",
		PCB->Changed ? '*' : ' ', PCB->Grid, (int) (TO_PCB(1)),
		(int) Settings.LineThickness,
		(int) Settings.ViaThickness,
		MyCursor.LineStack.LineN, MyCursor.LineStack.On ? "on" : "off",
		mode);
	XtVaSetValues(Output.StatusLine, XtNlabel, text, NULL);
}

/* ---------------------------------------------------------------------------
 * put some additional values into an elements structure
 */
void SetElementInfo(ElementTypePtr Element, Cardinal Count)
{
	Position	minx, miny,
				maxx, maxy;
	Cardinal	i,
				j;
	int			angle1,
				angle2,
				angle;
	float		fminx, fminy,
				fmaxx, fmaxy;
	LineTypePtr	line;
	PinTypePtr	pin;
	ArcTypePtr	arc;

		/* now scan for the lowest/highest X and Y coodinate and make the
		 * resulting point the upper left corner of the surrounding
		 * rectangle
		 */
	for (; Count; Count--, Element++)
	{
		minx = Element->TextX;
		miny = Element->TextY;
		maxx = Element->TextX;
		maxy = Element->TextY;
		for (i = Element->LineN, line = Element->Line; i; i--, line++)
		{
			minx = MIN(minx, line->X1);
			miny = MIN(miny, line->Y1);
			minx = MIN(minx, line->X2);
			miny = MIN(miny, line->Y2);
			maxx = MAX(maxx, line->X1);
			maxy = MAX(maxy, line->Y1);
			maxx = MAX(maxx, line->X2);
			maxy = MAX(maxy, line->Y2);
		}
		for (j = Element->PinN, pin = Element->Pin; j; j--, pin++)
		{
			minx = MIN(minx, pin->X);
			miny = MIN(miny, pin->Y);
			maxx = MAX(maxx, pin->X);
			maxy = MAX(maxy, pin->Y);
		}
		for (j = Element->ArcN, arc = Element->Arc; j; j--, arc++)
		{
				/* arc->StartAngle is in [0,360) and arc->Delta in (0,360] */
			angle1 = arc->StartAngle;
			angle2 = arc->StartAngle +arc->Delta;

				/* initialize limits */
			fminx = MIN(cos(M180 *(float) angle1), cos(M180 *(float) angle2));
			fmaxx = MAX(cos(M180 *(float) angle1), cos(M180 *(float) angle2));
			fminy = MIN(sin(M180 *(float) angle1), sin(M180 *(float) angle2));
			fmaxy = MAX(sin(M180 *(float) angle1), sin(M180 *(float) angle2));

				/* loop and check all angles n*180
				 * with angle1 <= a <= angle2
				 */
			for (angle = (angle1 /180 +1)*180; angle < angle2; angle += 180)
			{
				fminx = MIN(cos(M180 *(float) angle), fminx);
				fmaxx = MAX(cos(M180 *(float) angle), fmaxx);
			}

				/* loop and check all angles n*180+90
				 * with angle1 <= a <= angle2
				 */
			for (angle = ((angle1+90) /180)*180+90; angle < angle2; angle += 180)
			{
				fminy = MIN(sin(M180 *(float) angle), fminy);
				fmaxy = MAX(sin(M180 *(float) angle), fmaxy);
			}
			minx = MIN(minx, (int) (fminx *arc->Width) +arc->X);
			miny = MIN(miny, (int) (fminy *arc->Height) +arc->Y);
			maxx = MAX(maxx, (int) (fmaxx *arc->Width) +arc->X);
			maxy = MAX(maxy, (int) (fmaxy *arc->Height) +arc->Y);
		}
		Element->Rect.X = minx;
		Element->Rect.Y = miny;
		Element->Rect.Width = maxx -minx;
		Element->Rect.Height = maxy -miny;
	}
}

/* ---------------------------------------------------------------------------
 * gets minimum and maximum coordinates
 */
void GetMinMaxCoordinates(PCBTypePtr Ptr, Position *MinX, Position *MinY, Position *MaxX, Position *MaxY)
{
	Position		minx, miny,
					maxx, maxy;
	Cardinal		i,
					j;
	LineTypePtr		line;
	PinTypePtr		via;
	RectTypePtr		rect;
	ElementTypePtr	element;
	LayerTypePtr	layer;
	TextTypePtr		text;

		/* preset identifiers with highest and lowest possible values */
	minx = MAX_COORD;
	miny = MAX_COORD;
	maxx = MIN_COORD;
	maxy = MIN_COORD;

		/* now scan for the lowest/highest X and Y coodinate */
	for (j = Ptr->ViaN, via = Ptr->Via; j; j--, via++)
	{
		minx = MIN(minx, via->X);
		miny = MIN(miny, via->Y);
		maxx = MAX(maxx, via->X);
		maxy = MAX(maxy, via->Y);
	}
	for (j = Ptr->ElementN, element = Ptr->Element; j; j--, element++)
	{
		minx = MIN(minx, element->Rect.X);
		miny = MIN(miny, element->Rect.Y);
		maxx = MAX(maxx, element->Rect.X);
		maxy = MAX(maxy, element->Rect.Y);
		minx = MIN(minx, element->Rect.X +element->Rect.Width);
		miny = MIN(miny, element->Rect.Y +element->Rect.Height);
		maxx = MAX(maxx, element->Rect.X +element->Rect.Width);
		maxy = MAX(maxy, element->Rect.Y +element->Rect.Height);
	}

	for (i = 0; i < MAX_LAYER; i++)
	{
		layer = &Ptr->Layer[i];
		for (j = layer->LineN, line = layer->Line; j; j--, line++)
		{
			minx = MIN(minx, line->X1);
			miny = MIN(miny, line->Y1);
			minx = MIN(minx, line->X2);
			miny = MIN(miny, line->Y2);
			maxx = MAX(maxx, line->X1);
			maxy = MAX(maxy, line->Y1);
			maxx = MAX(maxx, line->X2);
			maxy = MAX(maxy, line->Y2);
		}
		for (j = layer->RectN, rect = layer->Rect; j; j--, rect++)
		{
			minx = MIN(minx, rect->X);
			miny = MIN(miny, rect->Y);
			maxx = MAX(maxx, rect->X);
			maxy = MAX(maxy, rect->Y);
			minx = MIN(minx, rect->X +rect->Width);
			miny = MIN(miny, rect->Y +rect->Height);
			maxx = MAX(maxx, rect->X +rect->Width);
			maxy = MAX(maxy, rect->Y +rect->Height);
		}
		for (j = layer->TextN, text = layer->Text; j; j--, text++)
		{
			minx = MIN(minx, text->Rect.X);
			miny = MIN(miny, text->Rect.Y);
			maxx = MAX(maxx, text->Rect.X);
			maxy = MAX(maxy, text->Rect.Y);
			minx = MIN(minx, text->Rect.X +text->Rect.Width);
			miny = MIN(miny, text->Rect.Y +text->Rect.Height);
			maxx = MAX(maxx, text->Rect.X +text->Rect.Width);
			maxy = MAX(maxy, text->Rect.Y +text->Rect.Height);
		}
	}
	*MinX = minx;
	*MinY = miny;
	*MaxX = maxx;
	*MaxY = maxy;
}

/* ---------------------------------------------------------------------------
 * sets cursor grid with respect to grid offset values
 */
void SetGrid(int Grid)
{
	if (Grid >= 1 && Grid <= MAX_GRID)
	{
		PCB->GridOffsetX = MyCursor.X % Grid;
		PCB->GridOffsetY = MyCursor.Y % Grid;
		PCB->Grid = Grid;
		SetStatusLine();
	}
} 

/* ---------------------------------------------------------------------------
 * centers the displayed PCB around the specified point
 * (X,Y) in screen coordinates
 */
void CenterDisplay(Position X, Position Y)
{
	Position	old_x, old_y,			/* output window in viewport */
				new_x, new_y;
	float		top;					/* top of thumb */

		/* save old values */
	XtVaGetValues(Output.Output, XtNx, &old_x, XtNy, &old_y, NULL);

		/* move origin half a screen width to get display centered */
	X -= (Output.Width /2);
	top = (float) X /(float) TO_SCREEN(Settings.Width);
	top = MIN(MAX(top, 0.0), 1.0);
	XtCallCallbacks(Output.ScrollbarBottom, XtNjumpProc, (XtPointer) &top);

	Y -= (Output.Height /2);
	top = (float) Y /(float) TO_SCREEN(Settings.Height);
	top = MIN(MAX(top, 0.0), 1.0);
	XtCallCallbacks(Output.ScrollbarLeft, XtNjumpProc, (XtPointer) &top);

		/* now get the new values (reread cause of rounding) */
	XtVaGetValues(Output.Output, XtNx, &new_x, XtNy, &new_y, NULL);

		/* the scrollbar widget triggers an expose event if (x,y)
		 * has changed. We have to do it if it hasn't changed
		 */
	if (new_x == old_x && new_y == old_y)
		RedrawOutput();
}

/* ---------------------------------------------------------------------------
* sets new zoom factor, adapts size of viewport and centers the cursor
*/
void SetZoom(int Zoom)
{
	Zoom = MAX(MIN_ZOOM, Zoom);
	Zoom = MIN(MAX_ZOOM, Zoom);

		/* redraw only if zoom has changed */
	if (PCB->Zoom != Zoom)
	{
		PCB->Zoom = Zoom;

			/* recalculate size of viewport */
		XtVaSetValues(Output.Output,
			XtNwidth, TO_SCREEN(Settings.Width),
			XtNheight, TO_SCREEN(Settings.Height),
			NULL);

		CenterDisplay(TO_SCREEN(MyCursor.X), TO_SCREEN(MyCursor.Y));
	}
		/* always redraw status line (used for init sequence) */
	SetStatusLine();
} 

/* ---------------------------------------------------------------------------
 * sets a new line thickness
 */
void SetLineSize(Dimension Size)
{
	if (Size >= MIN_LINESIZE && Size <= MAX_LINESIZE)
	{
		Settings.LineThickness = Size;
		SetStatusLine();
	}
}

/* ---------------------------------------------------------------------------
 * sets a new via thickness
 */
void SetViaSize(Dimension Size)
{
	if (Size >= MIN_PINORVIASIZE && Size <= MAX_PINORVIASIZE)
	{
		Settings.ViaThickness = Size;
		SetStatusLine();
	}
}

/* ---------------------------------------------------------------------------
 * sets or resets changed flag and redraws status
 */
void SetChangedFlag(Boolean New)
{
	PCB->Changed = New;
	SetStatusLine();
}

/* ---------------------------------------------------------------------------
 * strips leading and trailing blanks from the passed string and
 * returns a pointer to the new 'duped' one or NULL if the old one
 * holds only white space characters
 */
char *StripWhiteSpaceAndDup(char *S)
{
	char	*p1, *p2;
	size_t	length;

	if (!S || ! *S)
		return(NULL);

		/* strip leading blanks */
	for (p1 = S; *p1 && isspace(*p1); p1++);

		/* strip trailing blanksand get string length */
	length = strlen(p1);
	for (p2 = p1 +length -1; length && isspace(*p2); p2--, length--);

		/* string is not empty -> allocate memory */
	if (length)
	{
		p2 = MyRealloc(NULL, length+1, "StripWhiteSpace()");
		strncpy(p2, p1, length);
		*(p2 +length) = '\0';
		return(p2);
	}
	else
		return(NULL);
}

/* ---------------------------------------------------------------------------
 * transforms symbol coordinates to a (0,0) coord system
 * and calculates with and height of character cells
 */
void SetFontInfo(FontTypePtr Font)
{
	Cardinal		i, j;
	LineTypePtr		line;
	SymbolTypePtr	symbol;
	Dimension		totalmaxdelta = 0;
	Position		totalminx = MAX_COORD,
					totalminy = MAX_COORD,
					minx,
					miny,
					maxx,
					maxy;

		/* calculate cell with and height (is at least DEFAULT_CELLSIZE)
		 * maximum cell width and height
		 * minimum x and y position of all lines
		 */
	Font->MaxWidth = DEFAULT_CELLSIZE;
	Font->MaxHeight = DEFAULT_CELLSIZE;
	for (i = 0, symbol = Font->Symbol; i <= MAX_FONTPOSITION; i++, symbol++)
	{
		if (!symbol->LineN)
			continue;

		minx = miny = MAX_COORD;
		maxx = maxy = MIN_COORD;
		for (line = symbol->Line, j = symbol->LineN; j; j--, line++)
		{
			minx = MIN(minx, line->X1);
			miny = MIN(miny, line->Y1);
			minx = MIN(minx, line->X2);
			miny = MIN(miny, line->Y2);
			maxx = MAX(maxx, line->X1);
			maxy = MAX(maxy, line->Y1);
			maxx = MAX(maxx, line->X2);
			maxy = MAX(maxy, line->Y2);
		}
			/* set symbol size and save total min and max of coordinates */
		symbol->Width = maxx -minx;
		symbol->Height = maxy -miny;
		Font->MaxWidth = MAX(Font->MaxWidth, symbol->Width);
		Font->MaxHeight = MAX(Font->MaxHeight, symbol->Height);
		totalminx = MIN(totalminx, minx);
		totalminy = MIN(totalminy, miny);
		totalmaxdelta = MAX(totalmaxdelta, symbol->Delta);
	}

		/* move coordinate system and set size of unknown symbols 
		 * to maximum cell size
		 */
	for (i = 0, symbol = Font->Symbol; i <= MAX_FONTPOSITION; i++, symbol++)
	{
		if (symbol->LineN)
		{
			for (line = symbol->Line, j = symbol->LineN; j; j--, line++)
				MOVE_LINE(line, -totalminx, -totalminy);
		}
		else
		{
			symbol->Delta = totalmaxdelta;
			symbol->Width = Font->MaxWidth;
			symbol->Height = Font->MaxHeight;
		}
	}

		/* setup the rectangle for the default symbol */
	Font->DefaultSymbol.X = Font->DefaultSymbol.Y = 0;
	Font->DefaultSymbol.Width = Font->MaxWidth;
	Font->DefaultSymbol.Height = Font->MaxHeight;
} 

/* ---------------------------------------------------------------------------
 * updates all widgets like status, cursor position ... on screen
 */
void UpdateSettingsOnScreen(void)
{
	SetStatusLine();
	SetNameField();
	SetElementnameField();
	SetCursorLine();
	UpdateControlPanel();
}

/* ---------------------------------------------------------------------------
 * set a new mode
 */
void SetMode(int Mode)
{
		/* protect the cursor while changing the mode
		 * perform some additional stuff depending on the new mode
		 */
	HideCursor();
	MyCursor.Mode = Mode;
	switch(Mode)
	{
		case MODE_LINE:
			MyCursor.MarkedLine.State = MARKED_STATE_RESET;
			break;

		case MODE_BLOCK:
			MyCursor.MarkedBlock.State = MARKED_STATE_RESET;
			break;

		case MODE_ELEMENT:
			MyCursor.Mode = MyCursor.MarkedElement.Valid ? MODE_ELEMENT : MODE_NONE;
			break;
	}
	SetStatusLine();
	RestoreCursor();
}

/* ----------------------------------------------------------------------
 * parses the group definition string which is a colon seperated list of
 * comma seperated layer numbers (1,2:4,6,8)
 */
int ParseGroupString(char *s, LayerGroupTypePtr LG)
{
	int		group,
			member,
			layer;

		/* clear struct */
	memset(LG, 0, sizeof (LayerGroupType));

	for (group = 0; s && *s && group < MAX_LAYER; group++)
	{
		if (group >= MAX_LAYER)
			goto error;
		while (*s && isspace(*s))
			s++;
		for (member = 0; *s; s++)
		{
				/* ignore white spaces and get layernumber */
			while(*s && isspace(*s))
				s++;
			if (!isdigit(*s) ||
				(layer = atoi(s)) < 1 ||
				layer > MAX_LAYER ||
				member >= MAX_LAYER)
				goto error;
			LG->Entries[group][member++] = layer-1;
			while (*s && isdigit(*s))
				s++;

				/* ignore white spaces and check for seperator */
			while (*s && isspace(*s))
				s++;
			if (!*s || *s == ':')
				break;
			if (*s != ',')
				goto error;
		}
		LG->Number[group] = member;
		if (*s == ':')
			s++;
	}
	return(0);

	error:
			/* reset structure */
		memset(LG, 0, sizeof (LayerGroupType));
		return(1);
}

/* ---------------------------------------------------------------------------
 * quits application
 */
void QuitApplication(void)
{
		/* save data if necessary */
	if (PCB->Changed && Settings.SaveInTMP)
			EmergencySave();

		/* do some cleanup */
	ExitCursor();
	XtDestroyWidget(Output.Toplevel);
	XtDestroyApplicationContext(Context);
	XFreeGC(Dpy, Output.fgGC);
	XFreeGC(Dpy, Output.bgGC);
	RemovePCB(PCB);
	exit(0);
}

/* ---------------------------------------------------------------------------
 * catches alarm signal to call Backup()
 */
static void CatchAlarmSignal(int Signal)
{
	Backup();
}

/* ---------------------------------------------------------------------------
 * continues an interrupted backup alarm or scedules a new one
 */
void ContinueBackupAlarm(void)
{
	signal(SIGALRM, CatchAlarmSignal);
	alarm(AlarmTime ? AlarmTime : Settings.BackupInterval);
}

/* ---------------------------------------------------------------------------
 * interrupts the current alarm counter
 */
void StopBackupAlarm(void)
{
	AlarmTime = alarm(0);
} 
