/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: draw.c,v 1.2 94/06/19 11:22:12 nau Exp $";

/* drawing routines
 */

#include "global.h"

#include "create.h"
#include "cursor.h"
#include "data.h"
#include "draw.h"
#include "error.h"
#include "misc.h"
#include "transform.h"

/* ---------------------------------------------------------------------------
 * The type OutlineType describes the 8 corners of a polygon which
 * is used to draw lines, pins and vias
 */
typedef struct			/* dummy type to speed up drawing of pins and lines */
{
	Position	X,
				Y;
} PositionPolyType, *PositionPolyTypePtr;

typedef struct
{
	float	X,
			Y;
} FloatPolyType, *FloatPolyTypePtr;

/* ---------------------------------------------------------------------------
 * some local definitions and identifiers
 */
#define	TAN_CONST		0.207106781		/* 0.5*tan(22.5) */
#define	DOT_HOLE		0.35			/* inner/outer ration for pins */
#undef TO_SCREEN
#define TO_SCREEN(a)	((a) >> ZoomValue)

static	int				ZoomValue;		/* zoom value for drawing routines */
static	Window			DrawingWindow;	/* drawable used by drawing routines */

#if MAX_LINESIZE > MAX_PINORVIASIZE
static	PositionPolyType	Outline[MAX_LINESIZE+1][8];
static	PositionPolyType	Inline[MAX_LINESIZE+1][8];
#else
static	PositionPolyType	Outline[MAX_PINORVIASIZE+1][8];
static	PositionPolyType	Inline[MAX_PINORVIASIZE+1][8];
#endif

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void	InitPolygon(void);
static	void	Draw(void);
static	void	DrawPolygon(GC, Position, Position, PositionPolyTypePtr);
static	void	DrawElementPins(ElementType *);
static	void	EraseElementPins(ElementType *);
static	void	DrawElementLines(ElementType *);
static	void	EraseElementLines(ElementType *);
static	void	DrawElementArcs(ElementType *);
static	void	EraseElementArcs(ElementType *);
static	void	DrawVias(void);
static	void	DrawLayer(LayerTypePtr);
static	void	DrawText(TextTypePtr);
static	void	DrawFound(void);
static	void	DrawFoundLinesAndRectangles(void);
static	void	DrawFoundViasAndPins(void);
static	void	EraseArc(ArcTypePtr);

/* ----------------------------------------------------------------------
 * setup of zoom and output window for the next drawing operations
 */
void SwitchDrawingWindow(int Zoom, Window Drawable)
{
	ZoomValue = Zoom;
	DrawingWindow = Drawable;
	InitPolygon();
}

/* ---------------------------------------------------------------------------
 * initializes some zoom dependend information for pins and lines
 * just to speed up drawing a bit
 */
static void InitPolygon(void)
{
	int				i, j;
	FloatPolyType	p[8] = {{       0.5, -TAN_CONST},
							{ TAN_CONST,       -0.5},
							{-TAN_CONST,       -0.5},
							{      -0.5, -TAN_CONST},
							{      -0.5,  TAN_CONST},
							{-TAN_CONST,        0.5},
							{ TAN_CONST,        0.5},
							{       0.5,  TAN_CONST}};


		/* loop over maximum number of different sizes */
	for (i = 0; i <= MAX(MAX_PINORVIASIZE, MAX_LINESIZE); i++)
		for (j = 0; j < 8; j++)
		{
			Outline[i][j].X = (Position) (p[j].X * TO_SCREEN(i));
			Outline[i][j].Y = (Position) (p[j].Y * TO_SCREEN(i));
			Inline[i][j].X = (Position) (p[j].X * TO_SCREEN(i) *DOT_HOLE);
			Inline[i][j].Y = (Position) (p[j].Y * TO_SCREEN(i) *DOT_HOLE);
		}
}

/* ---------------------------------------------------------------------- 
 * redraws all the data
 * all necessary sizes are already set by the viewport widget and
 * by its callbackfunction CB_Viewport()
 */
void RedrawOutput(void)
{
	XEvent	event;

		/* prevent X errors during init phase */
	if (Output.OutputWindow)
	{
			/* remove all other pending expose events from queue */
		while (XtAppPending(Context))
		{
			XtAppNextEvent(Context, &event);
			if (event.type != Expose ||
				((XExposeEvent *) &event)->window != Output.OutputWindow)
				XtDispatchEvent(&event);
		}

		HideCursor();
		SwitchDrawingWindow(PCB->Zoom, Output.OutputWindow);
		Draw();
		DrawFound();

			/* redraw might have come from scolling the window
			 * so we do an update of the cursor position
			 */
		MoveCursorRelative(0, 0);
		SetStatusLine();
		SetNameField();
		SetElementnameField();
		RestoreCursor();
	}
}

/* ---------------------------------------------------------------------------
 * initializes some identifiers for a new zoom factor and redraws whole screen
 */
static void Draw(void)
{
	int	i;

		/* reset the cursor state because of clearing the background
		 * of the drawing area
		 */
	XFillRectangle(Dpy, DrawingWindow, Output.bgGC,
		0, 0, MAX_COORD, MAX_COORD);
	MyCursor.On = False;

		/* now draw all the stuff, except pins, vias and current layer */
	for (i = MAX_LAYER-1; i; i--)
		if ((LAYER_ON_STACK(i))->On)
			DrawLayer(LAYER_ON_STACK(i));

	if (PCB->ElementOn)
		for (i = 0; i < PCB->ElementN; i++)
		{
			DrawElementLines(PCB->Element +i);
			DrawElementArcs(PCB->Element +i);
			DrawElementName(PCB->Element +i);
		}	

	RedrawCurrentLayer();
}

/* ---------------------------------------------------------------------------
 * redraws the current layer as well as pins and vias
 * the later ones are drawn at last to make the drilling hole visible
 */
void RedrawCurrentLayer(void)
{
	int	i;

	HideCursor();
	SwitchDrawingWindow(PCB->Zoom, Output.OutputWindow);

		/* alway draw current layer */
	DrawLayer(CURRENT);

	if (PCB->PinOn)
		for (i = 0; i < PCB->ElementN; i++)
			DrawElementPins(PCB->Element +i);
	if (PCB->ViaOn)
		DrawVias();
	RestoreCursor();
}

/* ---------------------------------------------------------------------------
 * draws one polygon
 * x and y are already in display coordinates
 * the points are numbered:
 *
 *          5 --- 6
 *         /       \
 *        4         7
 *        |         |
 *        3         0
 *         \       /
 *          2 --- 1
 *
 */
static void DrawPolygon(GC DrawGC, Position X, Position Y, PositionPolyTypePtr PolyPtr)
{
	int					i;
	PositionPolyType	polygon[8];

		/* add line offset */
	for (i = 0; i < 8; i++)
	{
		polygon[i].X = X+ (PolyPtr +i)->X;
		polygon[i].Y = Y+ (PolyPtr +i)->Y;
	}
	XFillPolygon(Dpy, DrawingWindow, DrawGC,
		(XPoint *) &polygon, sizeof(polygon)/sizeof(polygon[0]),
		Convex, CoordModeOrigin);
}

/* ---------------------------------------------------------------------------
 * draws all vias
 */
static void DrawVias(void)
{
	PinTypePtr	via = PCB->Via;		/* pointer to pin data */
	Cardinal	n = PCB->ViaN;		/* number of pins */

		/* set colors */
	XSetForeground(Dpy, Output.fgGC, PCB->ViaColor);
	for (; n; n--, via++)
		if (!TEST_FLAG(FOUNDFLAG, via))
			DrawPinOrVia(via);
} 

/* ---------------------------------------------------------------------------
 * draw a new via
 */
void DrawNewVia(PinTypePtr Via)
{
	XSetForeground(Dpy, Output.fgGC, PCB->ViaColor);
	DrawPinOrVia(Via);
}

/* ---------------------------------------------------------------------------
 * draws one pin or via
 */
void DrawPinOrVia(PinTypePtr Ptr)
{
		/* transform X11 specific coord system */
	DrawPolygon(Output.fgGC, TO_SCREEN(Ptr->X), TO_SCREEN(Ptr->Y),
		&Outline[Ptr->Thickness][0]);
	DrawPolygon(Output.bgGC, TO_SCREEN(Ptr->X), TO_SCREEN(Ptr->Y),
		&Inline[Ptr->Thickness][0]);
}

/* ---------------------------------------------------------------------------
 * draw a new pin
 */
void DrawNewPin(PinTypePtr Pin)
{
	XSetForeground(Dpy, Output.fgGC, PCB->PinColor);
	DrawPinOrVia(Pin);
}

/* ---------------------------------------------------------------------------
 * erases one pin or via
 */
void ErasePinOrVia(PinTypePtr Ptr)
{
		/* transform X11 specific coord system */
	DrawPolygon(Output.bgGC, TO_SCREEN(Ptr->X), TO_SCREEN(Ptr->Y),
		&Outline[Ptr->Thickness][0]);
}

/* ---------------------------------------------------------------------------
 * draws a new created element
 */
void DrawNewElement(ElementTypePtr Element)
{
	DrawElementLines(Element);
	DrawElementArcs(Element);
	DrawElementName(Element);
	DrawElementPins(Element);
}

/* ---------------------------------------------------------------------------
 * draws pins of an element
 */
static void DrawElementPins(ElementTypePtr Element)
{
	PinTypePtr	pin = Element->Pin;		/* pointer to pin data */
	Cardinal	n = Element->PinN;		/* number of pins */

		/* set colors */
	XSetForeground(Dpy, Output.fgGC, PCB->PinColor);
	for (; n; n--, pin++)
		if (!TEST_FLAG(FOUNDFLAG, pin))
			DrawPinOrVia(pin);
} 

/* ---------------------------------------------------------------------------
 * draws lines of an element
 */
static void DrawElementLines(ElementTypePtr Element)
{
	LineTypePtr	line = Element->Line;		/* pointer to line data */
	Cardinal	n = Element->LineN;			/* number of lines */

		/* set color */
	XSetForeground(Dpy, Output.fgGC, PCB->ElementColor);
	for (; n; n--, line++)
		DrawLine(line);
} 

/* ---------------------------------------------------------------------------
 * draws arcs of an element
 */
static void DrawElementArcs(ElementTypePtr Element)
{
	ArcTypePtr	arc = Element->Arc;		/* pointer to line data */
	Cardinal	n = Element->ArcN;		/* number of lines */

	XSetForeground(Dpy, Output.fgGC, PCB->ElementColor);
	for (; n; n--, arc++)
		DrawArc(arc);
}

/* ---------------------------------------------------------------------------
 * draws the name of an element; used a faked text object to do it
 */
void DrawElementName(ElementTypePtr Element)
{
	static	TextType	text;		/* used to 'simulate' a text object */

	XSetForeground(Dpy, Output.fgGC, PCB->ElementColor);
	text.TextString = TEST_FLAG(CANONICALFLAG, PCB) ? Element->CanonicalName : Element->NameOnPCB;
	text.X = text.Rect.X = Element->TextX;
	text.Y = text.Rect.Y = Element->TextY;
	text.Direction = Element->Direction;
	text.Flags = 0;

		/* now set the surrounding box */
	SetSurroundingTextRectangle(PCB->Font.Symbol, &text);
	DrawText(&text);
}

/* ---------------------------------------------------------------------------
 * erases element from screen
 */
void EraseElement(ElementTypePtr Element)
{
	EraseElementLines(Element);
	EraseElementArcs(Element);
	EraseElementName(Element);
	EraseElementPins(Element);
}

/* ---------------------------------------------------------------------------
 * erases pins of an element
 */
static void EraseElementPins(ElementTypePtr Element)
{
	PinTypePtr	pin = Element->Pin;			/* pointer to pin data */
	Cardinal	n = Element->PinN;			/* number of pins */

	for (; n; n--, pin++)
		ErasePinOrVia(pin);
}

/* ---------------------------------------------------------------------------
 * erases lines of an element
 */
static void EraseElementLines(ElementTypePtr Element)
{
	LineTypePtr	line = Element->Line;		/* pointer to line data */
	Cardinal	n = Element->LineN;			/* number of lines */

	for (; n; n--, line++)
		EraseLine(line);
} 

/* ---------------------------------------------------------------------------
 * erases arcs of an element
 */
static void EraseElementArcs(ElementTypePtr Element)
{
	ArcTypePtr	arc = Element->Arc;		/* pointer to line data */
	Cardinal	n = Element->ArcN;		/* number of lines */

	for (; n; n--, arc++)
		EraseArc(arc);
}

/* ---------------------------------------------------------------------------
 * erases the name of an element; used a faked text object to do it
 */
void EraseElementName(ElementTypePtr Element)
{
	static	TextType	text;		/* used to 'simulate' a text object */

	text.TextString = TEST_FLAG(CANONICALFLAG, PCB) ? Element->CanonicalName : Element->NameOnPCB;
	text.X = text.Rect.X = Element->TextX;
	text.Y = text.Rect.Y = Element->TextY;
	text.Direction = Element->Direction;
	text.Flags = 0;

		/* now set the surrounding box */
	SetSurroundingTextRectangle(PCB->Font.Symbol, &text);
	EraseText(&text);
}

/* ---------------------------------------------------------------------------
 * draws a line on a layer
 */
void DrawLineOnLayer(LineTypePtr Line, LayerTypePtr Layer)
{
	XSetForeground(Dpy, Output.fgGC, Layer->Color);
	DrawLine(Line);
}

/* ---------------------------------------------------------------------------
 * draws a line
 * the points of a zero length line are numbered:
 *
 *     5 --- 6
 *    /       \
 *   4         7
 *   |         |
 *   3         0
 *    \       /
 *     2 --- 1
 *
 * valid directions are
 *
 *       0
 *     -   1
 *    -     2
 *     -   3
 *       4
 *
 */
void DrawLine(LineTypePtr Line)
{
	Position			x1, y1, x2, y2;
	int					i, j;
	PositionPolyTypePtr	polyptr = &Outline[Line->Thickness][0];
	PositionPolyType	polygon[8];

		/* convert to window coordinates */
	x1 = TO_SCREEN(Line->X1);
	y1 = TO_SCREEN(Line->Y1);
	x2 = TO_SCREEN(Line->X2);
	y2 = TO_SCREEN(Line->Y2);

	if (TEST_FLAG(POLYGONFLAG, PCB))
	{
			/* stretch the zero length polygon depending
			 * on the direction of the line
			 */
		for (i = 0, j = (-Line->Direction & 0x07); i < 8; i++, j = (++j & 0x07))
				/* add the the endpoint */
			if (j >= 4)
			{
				polygon[i].X = x2+ (polyptr +i)->X;
				polygon[i].Y = y2+ (polyptr +i)->Y;
			}
			else
				/* add the the startpoint */
			{
				polygon[i].X = x1+ (polyptr +i)->X;
				polygon[i].Y = y1+ (polyptr +i)->Y;
			}
		XFillPolygon(Dpy, DrawingWindow, Output.fgGC,
			(XPoint *) &polygon, 8, Convex, CoordModeOrigin);

			/* the following line is to make sure that everything is at least
			 * visible even when the lines are too thin to scale them right
			 */
		XSetLineAttributes(Dpy, Output.fgGC, 0, LineSolid, CapRound, JoinRound);
		XDrawLine(Dpy, DrawingWindow, Output.fgGC, x1, y1, x2, y2);
	}
	else
	{
		XSetLineAttributes(Dpy, Output.fgGC, TO_SCREEN(Line->Thickness), LineSolid, CapRound, JoinRound);
		XDrawLine(Dpy, DrawingWindow, Output.fgGC, x1, y1, x2, y2);
	}
}

/* ---------------------------------------------------------------------------
 * erase one line
 */
void EraseLine(LineTypePtr Line)
{
	GC	save;

		/* we just fake the gc */
	save = Output.fgGC;
	Output.fgGC = Output.bgGC;
	DrawLine(Line);
	Output.fgGC = save;
}

/* ---------------------------------------------------------------------------
 * draws filled rectangles from layer
 */
void DrawRect(RectTypePtr Rect)
{
	Position	x, y, width, height;

		/* convert x and y to window coordinates */
	x = TO_SCREEN(Rect->X);
	y = TO_SCREEN(Rect->Y);
	width = TO_SCREEN(Rect->Width);
	height = TO_SCREEN(Rect->Height);
	XFillRectangle(Dpy, DrawingWindow, Output.fgGC, x, y, width, height);
}

/* ---------------------------------------------------------------------------
 * draws a rectangle on a layer
 */
void DrawRectOnLayer(RectTypePtr Rect, LayerTypePtr Layer)
{
	XSetForeground(Dpy, Output.fgGC, Layer->Color);
	DrawRect(Rect);
}

/* ---------------------------------------------------------------------------
 * erases filled rectangles from layer
 */
void EraseRect(RectTypePtr Rect)
{
	Position	x, y, width, height;

		/* convert x and y to window coordinates */
	x = TO_SCREEN(Rect->X);
	y = TO_SCREEN(Rect->Y);
	width = TO_SCREEN(Rect->Width);
	height = TO_SCREEN(Rect->Height);
	XFillRectangle(Dpy, DrawingWindow, Output.bgGC, x, y, width, height);
}

/* ---------------------------------------------------------------------------
 * draws one layer
 */
static void DrawLayer(LayerTypePtr Layer)
{
	LineTypePtr	line = Layer->Line;		/* pointer to data */
	RectTypePtr	rect = Layer->Rect;
	TextTypePtr	text = Layer->Text;
	Cardinal	n;

	XSetForeground(Dpy, Output.fgGC, Layer->Color);
	for (n = Layer->LineN; n; n--, line++)
		if (!TEST_FLAG(FOUNDFLAG, line))
			DrawLine(line);
	for (n = Layer->RectN; n; n--, rect++)
		if (!TEST_FLAG(FOUNDFLAG, rect))
			DrawRect(rect);
	for (n = Layer->TextN; n; n--, text++)
		DrawText(text);
} 

/* ---------------------------------------------------------------------------
 * draws a string
 */
static void DrawText(TextTypePtr Text)
{
	Position			x = Text->X,
						y = Text->Y,
						x0 = 0;			/* initialize to get rid of compiler warnings */
	unsigned char		*string = (unsigned char *) Text->TextString;
	Cardinal			n;
	LineTypePtr			line;
	LineType			newline;
	FontTypePtr			font = &PCB->Font;
	RectType			defaultsymbol;

		/* if text has to be mirrored, get x0 with the unrotated
		 * width of the surrounding box (which is the length of the string)
		 */
	if (TEST_FLAG(MIRRORFLAG, Text))
	{
		if (Text->Direction == 0 || Text->Direction == 2)
			x0 = Text->X +Text->Rect.Width/2;
		else
			x0 = Text->X +Text->Rect.Height/2;
	}
	while (string && *string)
	{
			/* draw lines if symbol is valid and data is present */
		if (*string <= MAX_FONTPOSITION && font->Symbol[*string].Valid)
		{
			for (n = font->Symbol[*string].LineN, line = font->Symbol[*string].Line; n; n--, line++)
			{
					/* convert coordinates */
				newline = *line;
				newline.X1 += x;
				newline.Y1 += y;
				newline.X2 += x;
				newline.Y2 += y;

					/* do some mirroring and rotations */
				if (TEST_FLAG(MIRRORFLAG, Text))
				{
					newline.X1 = 2*x0 -newline.X1;
					newline.X2 = 2*x0 -newline.X2;
				}
				RotateLine(&newline, Text->X, Text->Y, Text->Direction);
				DrawLine(&newline);
			}
		}
		else
		{
				/* the default symbol is a rectangle */
			defaultsymbol = PCB->Font.DefaultSymbol;
			defaultsymbol.X += x;
			defaultsymbol.Y += y;

				/* do some mirroring and rotations */
			if (TEST_FLAG(MIRRORFLAG, Text))
				defaultsymbol.X = 2*x0 -defaultsymbol.X -defaultsymbol.Width;
			RotateRect(&defaultsymbol, Text->X, Text->Y, Text->Direction);
			DrawRect(&defaultsymbol);
		}
				/* move on to next cursor position */
		x += font->Symbol[*string].Width +font->Symbol[*string].Delta;
		string++;
	}
}

/* ---------------------------------------------------------------------------
 * erase a textline
 */
void EraseText(TextTypePtr Text)
{
	GC	save;

		/* we just fake the gc */
	save = Output.fgGC;
	Output.fgGC = Output.bgGC;
	DrawText(Text);
	Output.fgGC = save;
}

/* ---------------------------------------------------------------------------
 * draws a text on a layer
 */
void DrawTextOnLayer(TextTypePtr Text, LayerTypePtr Layer)
{
	XSetForeground(Dpy, Output.fgGC, Layer->Color);
	DrawText(Text);
}

/* ---------------------------------------------------------------------------
 * redraws all found objects
 */
static void DrawFound(void)
{
		/* set colors */
	XSetForeground(Dpy, Output.fgGC, PCB->ConnectedColor);
	DrawFoundLinesAndRectangles();
	DrawFoundViasAndPins();
}

/* ---------------------------------------------------------------------------
 * draws all found vias and pins
 */
static void DrawFoundViasAndPins(void)
{
	PinTypePtr		ptr;
	ElementTypePtr	element;
	Cardinal		i, j;

	if (PCB->ViaOn)
	{
		for (j = PCB->ViaN, ptr = PCB->Via; j; j--, ptr++)
			if (TEST_FLAG(FOUNDFLAG, ptr))
				DrawPinOrVia(ptr);
	}

	if (PCB->PinOn)
	{
		for (i = PCB->ElementN, element = PCB->Element; i; i--, element++)
			for (j = element->PinN, ptr = element->Pin; j; j--, ptr++)
				if (TEST_FLAG(FOUNDFLAG, ptr))
					DrawPinOrVia(ptr);
	}
} 

/* ---------------------------------------------------------------------------
 * draws all found lines and rectangles
 */
static void DrawFoundLinesAndRectangles(void)
{
	LineTypePtr		line;
	RectTypePtr		rect;
	Cardinal		layer, i;

	for (layer = 0; layer < MAX_LAYER; layer++)
		if (PCB->Layer[layer].On)
		{
			for (i = PCB->Layer[layer].LineN, line = PCB->Layer[layer].Line; i; i--, line++)
				if (TEST_FLAG(FOUNDFLAG, line))
					DrawLine(line);
			for (i = PCB->Layer[layer].RectN, rect = PCB->Layer[layer].Rect; i; i--, rect++)
				if (TEST_FLAG(FOUNDFLAG, rect))
					DrawRect(rect);
		}
} 

/* ---------------------------------------------------------------------------
 * draws an arc
 */
void DrawArc(ArcTypePtr Arc)
{
		/* angles have to be converted to X11 notation */
	XSetLineAttributes(Dpy, Output.fgGC, TO_SCREEN(Arc->Thickness), LineSolid, CapRound, JoinRound);
	XDrawArc(Dpy, DrawingWindow, Output.fgGC,
		TO_SCREEN(Arc->X -Arc->Width), TO_SCREEN(Arc->Y -Arc->Height),
		TO_SCREEN(2*Arc->Width), TO_SCREEN(2*Arc->Height),
		(Arc->StartAngle +180) *64, Arc->Delta *64);
}

/* ---------------------------------------------------------------------------
 * erase an arc
 */
static void EraseArc(ArcTypePtr Arc)
{
		/* angles have to be converted to X11 notation */
	XSetLineAttributes(Dpy, Output.bgGC, TO_SCREEN(Arc->Thickness), LineSolid, CapRound, JoinRound);
	XDrawArc(Dpy, DrawingWindow, Output.bgGC,
		TO_SCREEN(Arc->X -Arc->Width), TO_SCREEN(Arc->Y -Arc->Height),
		TO_SCREEN(2*Arc->Width), TO_SCREEN(2*Arc->Height),
		(Arc->StartAngle +180) *64, Arc->Delta *64);
}

