/* outelf.c	output routines for the Netwide Assembler to produce
 *		ELF32 (i386 of course) object file format
 *
 * The Netwide Assembler is copyright (C) 1996 Simon Tatham and
 * Julian Hall. All rights reserved. The software is
 * redistributable under the licence given in the file "Licence"
 * distributed in the NASM archive.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "nasm.h"
#include "nasmlib.h"

static FILE *elfp;		       /* the output file pointer */

typedef struct Symbol Symbol;
typedef struct Reloc Reloc;
typedef struct Section Section;

struct Symbol {
    Symbol *next;
    long name;
    long segment;
    long offset;
    int is_global;
};

struct Reloc {
    Reloc *next;
    unsigned long address;
    int symbol;
    int type;
};

struct Section {
    unsigned char *data;
    unsigned long datasize, datalen;
    long index;
    Reloc *rels, **reltail;
};

static void elf_deflabel (char *, long, long, int);
static void elf_write (void);
static void elf_section_header (int, int, int, void *, long,
				int, int, int, int);
static void elf_write_sections (void);
static void elf_build_symtab (char **, long *, int *);
static void elf_build_reltab (char **, long *, Reloc *);

#define SYMTAB_DELTA 512
#define STRTAB_DELTA 4096
#define DATA_DELTA 4096

#define REL_ABS 1		       /* these are the actual ELF */
#define REL_REL 2		       /* relocation type codes */

#define SHN_ABS 0xFFF1
#define SHN_COMMON 0xFFF2

#define SEG_ALIGN 16		       /* alignment of sections in file */
#define SEG_ALIGN_1 (SEG_ALIGN-1)

static efunc error;

/*
 * This is horrible, but until this entire module gets a going-over
 * and revamping, it'll stay. `symtab' maps NASM segment indices
 * (divided by two) to the string-table positions of their
 * external-symbol names. If an entry in `symtab' has SYMTAB_COMMON
 * set, however, the rest of the entry is in fact an index into
 * `commons', which _does_ contain the string table position.
 */
#define SYMTAB_COMMON 0x40000000
#define COMMON_DELTA 64
static struct Common {
    long strtabpos;
    long size;
} *commons;
static unsigned long common_size, common_pos;

static Section sections[3];
static long *symtab, *symnum;
static unsigned long symtab_size;
static char *symstrtab;
static unsigned long symstrtablen, symstrtab_size;

static Symbol *symbols, **symbols_tail;

static Section *curr_section;

#define ELF_MAX_SECTIONS 16	       /* really 10, but let's play safe */
static void *elf_section_data[ELF_MAX_SECTIONS];
static long elf_section_len[ELF_MAX_SECTIONS];
static int elf_sections;
static int elf_file_offset;

static void elf_init (FILE *fp, efunc errfunc, ldfunc ldef) {
    int i;

    error = errfunc;
    (void) ldef;		       /* placate optimisers */

    symtab = symnum = NULL;
    symtab_size = 0;
    commons = NULL;
    common_size = common_pos = 0;
    symstrtab = nasm_malloc(STRTAB_DELTA);
    symstrtab_size = STRTAB_DELTA;
    symstrtablen = 1;
    *symstrtab = '\0';
    for (i=0; i<3; i++) {
	sections[i].data = NULL;
	sections[i].datasize = sections[i].datalen = 0;
	sections[i].rels = NULL;
	sections[i].reltail = &sections[i].rels;
	sections[i].index = seg_alloc();
    }
    curr_section = sections;
    symbols = NULL;
    symbols_tail = &symbols;
    elfp = fp;
    elf_deflabel (".text", sections[0].index, 0L, -1);
    elf_deflabel (".data", sections[1].index, 0L, -1);
    elf_deflabel (".bss", sections[2].index, 0L, -1);
}

static void elf_cleanup (void) {
    int i;

    elf_write();

    nasm_free (symtab);
    nasm_free (symnum);
    nasm_free (commons);
    nasm_free (symstrtab);
    while (symbols) {
	Symbol *s = symbols->next;
	nasm_free (symbols);
	symbols = s;
    }

    for (i=0; i<3; i++) {
	Reloc *r;

	nasm_free (sections[i].data);
	for (r = sections[i].rels; r;) {
	    Reloc *s = r->next;
	    nasm_free (r);
	    r = s;
	}
    }
}

static void elf_out (long segto, void *data, unsigned long type,
		     long segment, long wrt) {
    long realbytes;
    long offset;
    char some_data[4], *p;

    if (wrt != NO_SEG) {
	wrt = NO_SEG;		       /* continue to do _something_ */
	error (ERR_NONFATAL, "WRT not supported by ELF output format");
    }

    /*
     * handle absolute-assembly (structure definitions)
     */
    if (segto == NO_SEG) {
	if ((type & OUT_TYPMASK) != OUT_RESERVE)
	    error (ERR_NONFATAL, "attempt to assemble code in [ABSOLUTE]"
		   " space");
	return;
    }

    if (segto == sections[0].index)
	curr_section = sections+0;
    else if (segto == sections[1].index)
	curr_section = sections+1;
    else if (segto == sections[2].index)
	curr_section = sections+2;
    else {
	error(ERR_WARNING, "attempt to assemble code in"
	      " segment %ld: defaulting to `.text'", segto);
	curr_section = sections;
    }

    realbytes = type & OUT_SIZMASK;
    if (type & OUT_TYPMASK == OUT_REL4ADR)
	realbytes = 4;

    if (segto != sections[2].index &&
	curr_section->datalen + realbytes > curr_section->datasize) {
	curr_section->datasize = curr_section->datalen+realbytes+DATA_DELTA;
	curr_section->data = nasm_realloc(curr_section->data,
					  curr_section->datasize);
    }

    if (segto == sections[2].index &&
	(type & OUT_TYPMASK) != OUT_RESERVE)
	error(ERR_WARNING, "attempt to initialise memory in the"
	      " BSS section: ignored");

    if ((type & OUT_TYPMASK) == OUT_REL2ADR) {
	error(ERR_NONFATAL, "ELF format does not allow 16-bit"
	      " relocations");
	return;
    } else if ((type & OUT_TYPMASK) == OUT_ADDRESS) {
	if (realbytes != 4) {
	    error(ERR_NONFATAL, "ELF format does not allow 16-bit"
		  " relocations");
	} else {
	    if (segment != NO_SEG) {
		Reloc *r;
		*curr_section->reltail = r = nasm_malloc(sizeof(Reloc));
		curr_section->reltail = &r->next;
		r->next = NULL;
		r->address = curr_section->datalen;
		r->symbol = segment;
		r->type = REL_ABS;
	    }
	}
	/*
	 * convert address to little-endian form
	 */
	p = some_data;
	WRITELONG (p, *(long *)data);
	data = some_data;
    } else if ((type & OUT_TYPMASK) == OUT_REL4ADR) {
	if (segment != segto) {
	    Reloc *r;
	    offset = *(long *)data-(type & OUT_SIZMASK);
	    *curr_section->reltail = r = nasm_malloc(sizeof(Reloc));
	    curr_section->reltail = &r->next;
	    r->next = NULL;
	    r->address = curr_section->datalen;
	    r->symbol = segment;
	    r->type = REL_REL;
	} else {
	    error(ERR_PANIC, "intrasegment REL4ADR");
	    exit(1);
	}
	/*
	 * convert address to little-endian form
	 */
	p = some_data;
	WRITELONG (p, offset);
	data = some_data;
    } else if ((type & OUT_TYPMASK) == OUT_RAWDATA) {
	if (segment != NO_SEG) {
	    error(ERR_PANIC, "ELF: segment in strange place");
	    exit(2);
	}
    } else if ((type & OUT_TYPMASK) == OUT_RESERVE) {
	if (segto != sections[2].index) {
	    error(ERR_WARNING, "uninitialised space declared in"
		  " %s section: zeroing",
		  (segto == sections[0].index ? "code" : "data"));
	}
	data = NULL;
    } else {
	error(ERR_PANIC, "bizarre output type received");
    }

    if (segto != sections[2].index) {
	if (data)
	    memcpy (curr_section->data + curr_section->datalen,
		    data, realbytes);
	else
	    memset (curr_section->data + curr_section->datalen,
		    0, realbytes);
    }
	
    curr_section->datalen += realbytes;
}

static void elf_deflabel (char *name, long segment, long offset,
			  int is_global) {
    int pos = symstrtablen;

    if (name[0] == '.' && name[1] == '.') {
	return;
    }

    if (symstrtablen + 1 + strlen(name) > symstrtab_size) {
	symstrtab_size = symstrtablen + 1 + strlen(name) + STRTAB_DELTA;
	symstrtab = nasm_realloc(symstrtab, symstrtab_size);
    }
    strcpy (symstrtab + pos, name);
    symstrtablen += strlen(name)+1;

    if ((segment == NO_SEG ||
	 segment == sections[0].index ||
	 segment == sections[1].index ||
	 segment == sections[2].index) &&
	is_global != -1) {
	Symbol *s;

	*symbols_tail = s = nasm_malloc(sizeof(Symbol));
	symbols_tail = &s->next;
	s->next = NULL;

	s->name = pos;
	s->segment = (segment == NO_SEG ? SHN_ABS :
		      segment == sections[0].index ? 1 :
		      segment == sections[1].index ? 2 :
		      segment == sections[2].index ? 3 : 0xFFFF);
	if (s->segment == 0xFFFF)
	    error(ERR_PANIC, "ELF: strange segment value in elf_deflabel");
	s->offset = offset;
	s->is_global = is_global;
    } else {
	if (segment % 2) {
	    error(ERR_NONFATAL, "ELF format does not support segment base"
		  " references");
	    return;
	}
	if (symtab_size <= segment/2) {
	    long newsize = segment/2+1;
	    symtab = nasm_realloc(symtab, newsize*sizeof(long));
	    while (symtab_size < newsize)
		symtab[symtab_size++] = 0;
	}
	if (!symtab[segment/2]) {
	    if (is_global != 2)
		symtab[segment/2] = pos;
	    else {
		symtab[segment/2] = SYMTAB_COMMON + common_pos;
		if (common_size <= common_pos) {
		    common_size += COMMON_DELTA;
		    commons = nasm_realloc(commons,
					   common_size*sizeof(*commons));
		}
		commons[common_pos].strtabpos = pos;
		commons[common_pos].size = offset;
		common_pos++;
	    }
	} else {
	    error(ERR_PANIC, "segment reused: redefining"
		  " symbol `%s' as `%s'",
		  symstrtab+symtab[segment/2], name);
	}
    }
}

static long elf_section_names (char *name, int pass) {
    if (!name)
	return sections[0].index;

    if (!strcmp(name, ".text"))
	return sections[0].index;
    else if (!strcmp(name, ".data"))
	return sections[1].index;
    else if (!strcmp(name, ".bss"))
	return sections[2].index;
    else
	return NO_SEG;
}

static int elf_directives (char *directive, char *value, int pass) {
    return 0;
}

static void elf_write (void) {
    int nsections, align;
    char shstrtab[80], *p;
    int shstrtablen, commlen;
    char comment[64];

    char *symtabdata, *reltextdata, *reldatadata;
    long symtabdatalen, reltextdatalen, reldatadatalen;
    int symtablocal;

    /*
     * Work out how many sections we will have.
     *
     * Fixed sections are:
     *    SHN_UNDEF .text .data .bss .comment .shstrtab .symtab .strtab
     *
     * Optional sections are:
     *    .rel.text .rel.data
     *
     * (.rel.bss makes very little sense;-)
     */
    nsections = 8;
    *shstrtab = '\0';
    shstrtablen = 1;
    shstrtablen += 1+sprintf(shstrtab+shstrtablen, ".text");
    shstrtablen += 1+sprintf(shstrtab+shstrtablen, ".data");
    shstrtablen += 1+sprintf(shstrtab+shstrtablen, ".bss");
    shstrtablen += 1+sprintf(shstrtab+shstrtablen, ".comment");
    shstrtablen += 1+sprintf(shstrtab+shstrtablen, ".shstrtab");
    shstrtablen += 1+sprintf(shstrtab+shstrtablen, ".symtab");
    shstrtablen += 1+sprintf(shstrtab+shstrtablen, ".strtab");
    if (sections[0].rels) {
	nsections++;
	shstrtablen += 1+sprintf(shstrtab+shstrtablen, ".rel.text");
    }
    if (sections[1].rels) {
	nsections++;
	shstrtablen += 1+sprintf(shstrtab+shstrtablen, ".rel.data");
    }

    /*
     * Do the comment.
     */
    *comment = '\0';
    commlen = 2+sprintf(comment+1, "The Netwide Assembler %s", NASM_VER);

    /*
     * Output the ELF header.
     */
    fwrite ("\177ELF\1\1\1\0\0\0\0\0\0\0\0\0", 16, 1, elfp);
    fwriteshort (1, elfp);	       /* ET_REL relocatable file */
    fwriteshort (3, elfp);	       /* EM_386 processor ID */
    fwritelong (1L, elfp);	       /* EV_CURRENT file format version */
    fwritelong (0L, elfp);	       /* no entry point */
    fwritelong (0L, elfp);	       /* no program header table */
    fwritelong (0x40L, elfp);	       /* section headers straight after
					* ELF header plus alignment */
    fwritelong (0L, elfp);	       /* 386 defines no special flags */
    fwriteshort (0x34, elfp);	       /* size of ELF header */
    fwriteshort (0, elfp);	       /* no program header table, again */
    fwriteshort (0, elfp);	       /* still no program header table */
    fwriteshort (0x28, elfp);	       /* size of section header */
    fwriteshort (nsections, elfp);     /* number of sections */
    fwriteshort (5, elfp);	       /* string table section index for
					* section header table */
    fwritelong (0L, elfp);	       /* align to 0x40 bytes */
    fwritelong (0L, elfp);
    fwritelong (0L, elfp);

    /*
     * Build the symbol table and relocation tables.
     */
    elf_build_symtab (&symtabdata, &symtabdatalen, &symtablocal);
    elf_build_reltab (&reltextdata, &reltextdatalen, sections[0].rels);
    elf_build_reltab (&reldatadata, &reldatadatalen, sections[1].rels);

    /*
     * Now output the section header table.
     */

    elf_file_offset = 0x40 + 0x28 * nsections;
    align = ((elf_file_offset+SEG_ALIGN_1) & ~SEG_ALIGN_1) - elf_file_offset;
    elf_file_offset += align;

    elf_section_header (0, 0, 0, NULL, 0L, 0, 0, 0, 0);   /* SHN_UNDEF */
    p = shstrtab+1;
    elf_section_header (p - shstrtab, 1, 6, sections[0].data,
			sections[0].datalen, 0, 0, 16, 0);   /* .text */
    p += strlen(p)+1;
    elf_section_header (p - shstrtab, 1, 3, sections[1].data,
			sections[1].datalen, 0, 0, 4, 0);   /* .data */
    p += strlen(p)+1;
    elf_section_header (p - shstrtab, 8, 3, NULL,
			sections[2].datalen, 0, 0, 4, 0);   /* .bss */
    p += strlen(p)+1;
    elf_section_header (p - shstrtab, 1, 0, comment,
			(long)commlen, 0, 0, 1, 0);/* .comment */
    p += strlen(p)+1;
    elf_section_header (p - shstrtab, 3, 0, shstrtab,
			(long)shstrtablen, 0, 0, 1, 0);/* .shstrtab */
    p += strlen(p)+1;
    elf_section_header (p - shstrtab, 2, 0, symtabdata,
			symtabdatalen, 7, symtablocal, 4, 16);/* .symtab */
    p += strlen(p)+1;
    elf_section_header (p - shstrtab, 3, 0, symstrtab,
			symstrtablen, 0, 0, 1, 0);	    /* .strtab */
    if (sections[0].rels) {
	p += strlen(p)+1;
	elf_section_header (p - shstrtab, 9, 0, reltextdata,
			    reltextdatalen, 6, 1, 4, 8);    /* .rel.text */
    }
    if (sections[1].rels) {
	p += strlen(p)+1;
	elf_section_header (p - shstrtab, 9, 0, reldatadata,
			    reldatadatalen, 6, 2, 4, 8);    /* .rel.data */
    }

    fwrite ("\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0", align, 1, elfp);

    /*
     * Now output the sections.
     */
    elf_write_sections();

    fclose (elfp);
}

static void elf_section_header (int name, int type, int flags,
				void *data, long datalen,
				int link, int info, int align, int eltsize) {
    elf_section_data[elf_sections] = data;
    elf_section_len[elf_sections] = datalen;
    elf_sections++;

    fwritelong ((long)name, elfp);
    fwritelong ((long)type, elfp);
    fwritelong ((long)flags, elfp);
    fwritelong (0L, elfp);	       /* no address, ever, in object files */
    fwritelong (type == 0 ? 0L : elf_file_offset, elfp);
    fwritelong (datalen, elfp);
    if (data)
	elf_file_offset += (datalen+SEG_ALIGN_1) & ~SEG_ALIGN_1;
    fwritelong ((long)link, elfp);
    fwritelong ((long)info, elfp);
    fwritelong ((long)align, elfp);
    fwritelong ((long)eltsize, elfp);
}

static void elf_write_sections (void) {
    int i;
    for (i = 0; i < elf_sections; i++)
	if (elf_section_data[i]) {
	    int len = elf_section_len[i];
	    int reallen = (len+SEG_ALIGN_1) & ~SEG_ALIGN_1;
	    int align = reallen - len;
	    fwrite (elf_section_data[i], len, 1, elfp);
	    fwrite ("\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0", align, 1, elfp);
	}
}

static void elf_build_symtab (char **table, long *datalen, int *localptr) {
    int i, j, l;
    Symbol *s;
    unsigned char *tab;

    symnum = nasm_malloc(symtab_size * sizeof(*symnum));

    i = 1;
    for (j=0; j<3; j++)		       /* count and number section symbols */
	symnum[sections[j].index/2] = i++;
    for (l=i, s=symbols; s; i++, s=s->next)
	if (!s->is_global) l++;	       /* count defined and local symbols */
    for (j=0; j<symtab_size; j++)
	if (symtab[j] &&
	    j != sections[0].index/2 &&
	    j != sections[1].index/2 &&
	    j != sections[2].index/2)
	    symnum[j] = i++;	       /* count and number undefined ones */

    *localptr = l;
    *datalen = 16*i;
    tab = (unsigned char *) (*table = nasm_malloc (16*i));

    WRITELONG (tab, 0);		       /* initial blank entry */
    WRITELONG (tab, 0);
    WRITELONG (tab, 0);
    *tab++ = 0;
    *tab++ = 0;
    WRITESHORT (tab, 0);

    for (i=0; i<3; i++) {	       /* process section symbols */
	WRITELONG (tab, symtab[sections[i].index/2]);
	WRITELONG (tab, 0);
	WRITELONG (tab, 0);
	*tab++ = 0x03;		       /* local, type SECTION */
	*tab++ = 0;
	WRITESHORT (tab, i+1);
    }

    for (s=symbols; s; s=s->next)      /* process local symbols */
	if (!s->is_global) {
	    WRITELONG (tab, s->name);
	    WRITELONG (tab, s->offset);
	    WRITELONG (tab, 0);
	    *tab++ = 0x00;	       /* local, no type */
	    *tab++ = 0;
	    WRITESHORT (tab, s->segment);
	}

    for (s=symbols; s; s=s->next)      /* process global defined symbols */
	if (s->is_global) {
	    WRITELONG (tab, s->name);
	    WRITELONG (tab, s->offset);
	    WRITELONG (tab, 0);
	    *tab++ = 0x10;	       /* global, no type */
	    *tab++ = 0;
	    WRITESHORT (tab, s->segment);
	}

    for (i=0; i<symtab_size; i++)      /* process external global symbols */
	if (2*i != sections[0].index &&
	    2*i != sections[1].index &&
	    2*i != sections[2].index &&
	    symtab[i]) {
	    if (symtab[i] >= SYMTAB_COMMON) {
		long j = symtab[i] - SYMTAB_COMMON;
		WRITELONG (tab, commons[j].strtabpos);
		WRITELONG (tab, 4);    /* alignment */
		WRITELONG (tab, commons[j].size);
		*tab++ = 0x10;
		*tab++ = 0;
		WRITESHORT (tab, 0xFFF2);
	    } else {
		WRITELONG (tab, symtab[i]);
		WRITELONG (tab, 0);
		WRITELONG (tab, 0);
		*tab++ = 0x10;
		*tab++ = 0;
		WRITESHORT (tab, 0);   /* SHN_UNDEF */
	    }
	}
}

static void elf_build_reltab (char **table, long *datalen, Reloc *rels) {
    int i;
    Reloc *r;
    char *tab;

    if (!rels) {
	*table = NULL;
	return;
    }

    for (i=0, r=rels; r; i++, r=r->next);   /* count relocations */

    *datalen = i*8;
    tab = *table = nasm_malloc(i*8);

    for (r = rels; r; r = r->next) {
	WRITELONG (tab, r->address);
	WRITELONG (tab, (symnum[r->symbol/2] << 8) + r->type);
    }
}

static long elf_segbase (long segment) {
    return segment;
}

static void elf_filename (char *inname, char *outname, efunc error) {
    standard_extension (inname, outname, ".o", error);
}

struct ofmt of_elf = {
    "ELF32 (i386) object files (e.g. Linux)",
    "elf",
    elf_init,
    elf_out,
    elf_deflabel,
    elf_section_names,
    elf_segbase,
    elf_directives,
    elf_filename,
    elf_cleanup
};
